/**
 * \file            UTIL_rspc.c
 *
 * \version
 *
 *
 * \Description     RSPC (Read Solomon Product like Correction)
 *                  This code was ported from UTIL gen2.
 *
 * \component       Utility
 *
 * \author          Kurt Gieske kgieske (at) de.adit-jv.com
 * \author          Simon Maleyka smaleyka (at) de.adit-jv.com
 * \author          Christoph Gellner cgellner (at) de.adit-jv.com
 * \author          sundhar.ashokan@in.bosch.com
		    saurabh.arora@in.bosch.com
 * Copyright (C) 2013 ADIT Corporation
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License ,  or
 * (at your option) any later version.
 *
 * \see
 *
 * \history
 *
 **/
#include    "masca_util_rspc.h"
#include    "masca_util_rspc_scramble_table.h"

/**
 Sector Mode 1

 <------------------------------------3234 bytes------------------------------->
 <------------------2352 bytes -------------------------><-------882 bytes----->
 ------------------------------------------------------------------------------|
 | SYNC | Header |     Data     |  EDC  | free |  ECC   | EDC/  | EDC/  | Ctrl |
 |      |        |              |       |      |        | ECC#2 | ECC#1 | bytes|
 |  12  |   4    |     2048     |   4   |  8   |  276   | 392   | 392   |  98  |
 | byte |  byte  |     byte     |  byte | byte |  byte  | byte  | byte  | byte |
 ------------------------------------------------------------------------------|
 <- fed into RS encoder(2064) ->
 <- fed into RS decoder(2340) ->                <------->

 Sector Mode 2

 <------------------------------------3234 bytes------------------------------->
 <------------------2352 bytes -------------------------><-------882 bytes----->
 ------------------------------------------------------------------------------|
 | SYNC | Header |                Data                  | EDC/  | EDC/  | Ctrl |
 |      |        |                                      | ECC#2 | ECC#1 | bytes|
 |  12  |   4    |                2336                  | 392   | 392   |  98  |
 | byte |  byte  |                byte                  | byte  | byte  | byte |
 ------------------------------------------------------------------------------|

SYNC		byte	size	content
		0	1	0000 0000
		1	10	1111 1111
		11	1	0000 0000
Header		12	1	minute
		13	1	second
		14	1	sector
		15	1	mode 1 or 2
Data		16	2048/2336  user data mode1 / user data mode2

 Mode 1 only:
 EDC          2064      4      EDC detection only
 Free         2068      8      free
 P-Parity     2076    172      P-Parity
 Q-Parity     2248    104      Q-Parity

 */

/* sizeof 2^m-1 where m = 8 define the Size of the Galois field */
#define GF	256
/* Prime Polynomial of this Galois field (x^8 + x^4 + x^3 + x^2 + 1) */
/* 256 + 29 */
#define PP	285

#define NUM_P_VECTORS		86  /* byte fields*/
#define NUM_Q_VECTORS		52  /* byte fields*/
#define NUM_P_VECTORS_16	(NUM_P_VECTORS >> 1)   /* 43 - 16bit fields */
#define NUM_Q_VECTORS_16	(NUM_Q_VECTORS >> 1)   /* 26 - 16bit fields */
#define LEN_P_VECTORS		(NUM_Q_VECTORS_16)     /* 26 - 16bit fields */
#define LEN_Q_VECTORS		(NUM_P_VECTORS_16 + 2) /* 45 - 16bit fields */
#define START_P_VECTOR		(GF - 1 - LEN_P_VECTORS)
						/* 255-26 - 16bit fields */
#define START_Q_VECTOR		(GF - 1 - LEN_Q_VECTORS)
						/* 255-45 - 16bit fields */
#define OFFSET_Q_VECTOR		(NUM_P_VECTORS_16 + 1)
						/* 44 - 16bit fields */
#define OFFSET_Q_VECTOR_8	(OFFSET_Q_VECTOR << 1)
						/* 88 - 16bit fields */
#define MODULO_Q_VECTOR		1118
#define BITS_PER_BYTE		8
#define MSB_LSB_BYTE_MAX_CNT	2

#define LEN_Q_VECTORS_MINUS_OFFSET	2
#define UTIL_RSPC_ROW_OFFSET_10		10u

#define RS_OPTIMIZED

#define RSPC_STOP_ON_P_ERROR	24
#define RSPC_STOP_ON_Q_ERROR	14

#define RSPC_DECODER_P_ERROR_THRESHHOLD		24
/* if remaining errors above this value after first iteration, break and give
								 error */
#define RSPC_DECODER_Q_ERROR_THRESHHOLD		14
/* if remaining errors above this value after first iteration, break and give
								 error */

#define NO_ERROR_FOUND	-1
#define TABLE_DESCRAMBLER
#ifdef TABLE_DESCRAMBLER
#define L2_RAW	(1024*2)
/* parity bytes for 16 bit units */
#define L2_Q	(26*2*2)
#define L2_P	(43*2*2)
#else
# define L2_SCRAMBLER_PRESET	0x1
#endif

enum rspc_iterations {
	NO_ITERATIONS = 0,
	LOW_ITERATIONS = 2,
	MID_ITERATIONS = 4,
	MAX_ITERATIONS = 15
};

enum paritytype_te {
	P_PARITY, Q_PARITY
};

/**
 * LOCAL functions
 */
static void util_rspc_correct_p_matrix(unsigned short *rspc_data,
						unsigned char iteration);
static void util_rspc_correct_q_matrix(unsigned short *rspc_data,
						unsigned char iteration);

static void util_rspc_correct_p_vector(unsigned short p_vector,
				unsigned short S0_16, unsigned short *rspc_data,
				unsigned char iteration);
static void util_rspc_correct_q_vector(unsigned short q_vector,
				unsigned short S0_16, unsigned short *rspc_data,
				unsigned char iteration);

/* Below functions calculating S0 compute the results of the xor for the lower
	and the upper byte at once, therefore the result is of type unsigned
	short */
static unsigned short UTIL_rspc_calc_p_s0(unsigned short row_16,
					unsigned short *rspc_data);
static unsigned short UTIL_rspc_calc_q_s0(unsigned short row_16,
					unsigned short *rspc_data);

static unsigned char UTIL_rspc_calc_p_s1(unsigned short row_16,
				unsigned int MLSB, unsigned short *rspc_data);
static unsigned char UTIL_rspc_calc_q_s1(unsigned short row_16,
				unsigned int MLSB, unsigned short *rspc_data);

static void util_rspc_init_decoder(void);

static void util_rspc_handle_error(unsigned char S0, unsigned short row_16,
				unsigned int MLSB,
				enum paritytype_te parity,
				unsigned short *rspc_data,
				unsigned char iteration);

static int util_rspc_decoder(unsigned short *rspc_data,
					enum rspc_iterations rspc_iterations);

static bool util_rspc_fill_galois_arrays(void);
static unsigned char UTIL_rspc_GF8_apow_product(unsigned char multiplicand,
					unsigned char multiplier);
static unsigned char UTIL_rspc_GF8_apow_quotient(unsigned char dividend,
					unsigned char divisor);

static void util_rspc_set_influence_matrix(enum paritytype_te parity,
					unsigned short row_16,
					unsigned char error_position);

#ifdef EXTENDED_GALOIS_ARITHMETIC      /*--- not relevant since not enabled */
static unsigned char UTIL_rspc_GF8_sum(unsigned char summanda,
						unsigned char summandb);
static unsigned char UTIL_rspc_GF8_diff(unsigned char minuend,
						unsigned char subtrahend);
#endif /* EXTENDED_GALOIS_ARITHMETIC */

/**
 * VARIABLES
 */
static short p_error_lines[NUM_P_VECTORS_16 + 1];
static short q_error_lines[NUM_Q_VECTORS_16 + 1];
static unsigned int p_error_line_index;
static unsigned int q_error_line_index;
static unsigned int remaining_p_errors;
static unsigned int remaining_q_errors;
static unsigned char _log[GF]; /* Log[0] is invalid and must not be used */
static unsigned char alog[GF]; /* establish global Log and Antilog arrays*/


/* CRC_TABLE_DATA */
const unsigned int aul_Crc32Tab[] = {
		0x00000000L, 0x90910101L, 0x91210201L, 0x01B00300L,
		0x92410401L, 0x02D00500L, 0x03600600L, 0x93F10701L,
		0x94810801L, 0x04100900L, 0x05A00A00L, 0x95310B01L,
		0x06C00C00L, 0x96510D01L, 0x97E10E01L, 0x07700F00L,
		0x99011001L, 0x09901100L, 0x08201200L, 0x98B11301L,
		0x0B401400L, 0x9BD11501L, 0x9A611601L, 0x0AF01700L,
		0x0D801800L, 0x9D111901L, 0x9CA11A01L, 0x0C301B00L,
		0x9FC11C01L, 0x0F501D00L, 0x0EE01E00L, 0x9E711F01L,
		0x82012001L, 0x12902100L, 0x13202200L, 0x83B12301L,
		0x10402400L, 0x80D12501L, 0x81612601L, 0x11F02700L,
		0x16802800L, 0x86112901L, 0x87A12A01L, 0x17302B00L,
		0x84C12C01L, 0x14502D00L, 0x15E02E00L, 0x85712F01L,
		0x1B003000L, 0x8B913101L, 0x8A213201L, 0x1AB03300L,
		0x89413401L, 0x19D03500L, 0x18603600L, 0x88F13701L,
		0x8F813801L, 0x1F103900L, 0x1EA03A00L, 0x8E313B01L,
		0x1DC03C00L, 0x8D513D01L, 0x8CE13E01L, 0x1C703F00L,
		0xB4014001L, 0x24904100L, 0x25204200L, 0xB5B14301L,
		0x26404400L, 0xB6D14501L, 0xB7614601L, 0x27F04700L,
		0x20804800L, 0xB0114901L, 0xB1A14A01L, 0x21304B00L,
		0xB2C14C01L, 0x22504D00L, 0x23E04E00L, 0xB3714F01L,
		0x2D005000L, 0xBD915101L, 0xBC215201L, 0x2CB05300L,
		0xBF415401L, 0x2FD05500L, 0x2E605600L, 0xBEF15701L,
		0xB9815801L, 0x29105900L, 0x28A05A00L, 0xB8315B01L,
		0x2BC05C00L, 0xBB515D01L, 0xBAE15E01L, 0x2A705F00L,
		0x36006000L, 0xA6916101L, 0xA7216201L, 0x37B06300L,
		0xA4416401L, 0x34D06500L, 0x35606600L, 0xA5F16701L,
		0xA2816801L, 0x32106900L, 0x33A06A00L, 0xA3316B01L,
		0x30C06C00L, 0xA0516D01L, 0xA1E16E01L, 0x31706F00L,
		0xAF017001L, 0x3F907100L, 0x3E207200L, 0xAEB17301L,
		0x3D407400L, 0xADD17501L, 0xAC617601L, 0x3CF07700L,
		0x3B807800L, 0xAB117901L, 0xAAA17A01L, 0x3A307B00L,
		0xA9C17C01L, 0x39507D00L, 0x38E07E00L, 0xA8717F01L,
		0xD8018001L, 0x48908100L, 0x49208200L, 0xD9B18301L,
		0x4A408400L, 0xDAD18501L, 0xDB618601L, 0x4BF08700L,
		0x4C808800L, 0xDC118901L, 0xDDA18A01L, 0x4D308B00L,
		0xDEC18C01L, 0x4E508D00L, 0x4FE08E00L, 0xDF718F01L,
		0x41009000L, 0xD1919101L, 0xD0219201L, 0x40B09300L,
		0xD3419401L, 0x43D09500L, 0x42609600L, 0xD2F19701L,
		0xD5819801L, 0x45109900L, 0x44A09A00L, 0xD4319B01L,
		0x47C09C00L, 0xD7519D01L, 0xD6E19E01L, 0x46709F00L,
		0x5A00A000L, 0xCA91A101L, 0xCB21A201L, 0x5BB0A300L,
		0xC841A401L, 0x58D0A500L, 0x5960A600L, 0xC9F1A701L,
		0xCE81A801L, 0x5E10A900L, 0x5FA0AA00L, 0xCF31AB01L,
		0x5CC0AC00L, 0xCC51AD01L, 0xCDE1AE01L, 0x5D70AF00L,
		0xC301B001L, 0x5390B100L, 0x5220B200L, 0xC2B1B301L,
		0x5140B400L, 0xC1D1B501L, 0xC061B601L, 0x50F0B700L,
		0x5780B800L, 0xC711B901L, 0xC6A1BA01L, 0x5630BB00L,
		0xC5C1BC01L, 0x5550BD00L, 0x54E0BE00L, 0xC471BF01L,
		0x6C00C000L, 0xFC91C101L, 0xFD21C201L, 0x6DB0C300L,
		0xFE41C401L, 0x6ED0C500L, 0x6F60C600L, 0xFFF1C701L,
		0xF881C801L, 0x6810C900L, 0x69A0CA00L, 0xF931CB01L,
		0x6AC0CC00L, 0xFA51CD01L, 0xFBE1CE01L, 0x6B70CF00L,
		0xF501D001L, 0x6590D100L, 0x6420D200L, 0xF4B1D301L,
		0x6740D400L, 0xF7D1D501L, 0xF661D601L, 0x66F0D700L,
		0x6180D800L, 0xF111D901L, 0xF0A1DA01L, 0x6030DB00L,
		0xF3C1DC01L, 0x6350DD00L, 0x62E0DE00L, 0xF271DF01L,
		0xEE01E001L, 0x7E90E100L, 0x7F20E200L, 0xEFB1E301L,
		0x7C40E400L, 0xECD1E501L, 0xED61E601L, 0x7DF0E700L,
		0x7A80E800L, 0xEA11E901L, 0xEBA1EA01L, 0x7B30EB00L,
		0xE8C1EC01L, 0x7850ED00L, 0x79E0EE00L, 0xE971EF01L,
		0x7700F000L, 0xE791F101L, 0xE621F201L, 0x76B0F300L,
		0xE541F401L, 0x75D0F500L, 0x7460F600L, 0xE4F1F701L,
		0xE381F801L, 0x7310F900L, 0x72A0FA00L, 0xE231FB01L,
		0x71C0FC00L, 0xE151FD01L, 0xE0E1FE01L, 0x7070FF00L
};
/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_descramble_data
 *
 * \see     UTIL_rspc.h
 *------------------------------------------------------------------------------
 */
void util_rspc_descramble_data(unsigned int *sector_data)
{
# ifdef TABLE_DESCRAMBLER

	/* 4 bytes descrambled at once */
	unsigned int *r = sector_data + (UTIL_RSPC_CDED_SYNCWORD_SIZE >> 2);
	unsigned int *s = (unsigned int *) util_rspc_yellowbook_scrambler;
	unsigned int i;

	for (i = (L2_RAW + L2_Q + L2_P + 16) >> 2; i; i--)
		*r++ ^= *s++;
# else

	unsigned int i, j;
	unsigned char *data = (unsigned char *)sector_data +
						UTIL_RSPC_CDED_SYNCWORD_SIZE;
	unsigned short scr; /* descrambler register */
	scr = L2_SCRAMBLER_PRESET;
	/* go for all bytes */
	for (i = 0; i < 2340; i++) {
		/* go for all bits of the byte */
		for (j = 0; j < 8; j++) {
			if (scr & 1) {
				/* descramble the bits */
				data[i] ^= (1 << j);
			}
			if ((scr & 1) ^ ((scr >> 1) & 1))
				scr |= 1 << 15;
			scr = scr >> 1;
		}
	}

# endif /* TABLE_DESCRAMBLER */
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_correction
 *
 * \see     UTIL_rspc.h
 *------------------------------------------------------------------------------
 */
int util_rspc_correction(unsigned short *sector_data, bool crc_check_enabled,
		enum cdedqoc_te iterations)
{
	int block_valid = -EIO;
	int result = E_OK;
	unsigned int crc32 = 0;
	enum rspc_iterations rspc_iterations = NO_ITERATIONS;
	enum cd_mode_te cd_mode = CD_MODE1;
	static bool b_galois_fld_init = FALSE;

	/* Mode2 XA Form 1 RSPC is detroying the header word    */
	/* So we have to back-it-up, and restore it afterwards. */
	unsigned int u32_backup_header = 0;

	unsigned char *u8_sector_data_pt = (unsigned char *) sector_data;

	/* get the maximum amount of iterations that could be done
	 * within the RSPC decoder */
	switch (iterations) {
	case CDED_QOC_LOW:
		rspc_iterations = LOW_ITERATIONS;
		break;
	case CDED_QOC_MID:
		rspc_iterations = MID_ITERATIONS;
		break;
	case CDED_QOC_HIGH:
		rspc_iterations = MAX_ITERATIONS;
		break;
	default:
		rspc_iterations = MID_ITERATIONS;
		break;
	}

	/* Initialize the Galois field */
	if (b_galois_fld_init != TRUE)
		b_galois_fld_init = util_rspc_fill_galois_arrays();

	/* check byte 15 (mode field) which mode is set */
	/*    if ( *(tmp_pt + UTIL_RSPC_CDED_MODE_POSITION) == 0x02 )*/

	/* Mode2 XA Form 1 RSPC is detroying the header word    */
	/* So we have to back-it-up, and restore it afterwards.  */
	/*if(*(sector_data+UTIL_RSPC_CDED_HEADER_MODE_POSITION_16)==CD_MODE2 )*/
	if (*(u8_sector_data_pt + UTIL_RSPC_CDED_MODE_POSITION) == CD_MODE2) {
		memcpy(&(u32_backup_header), sector_data,
				UTIL_RSPC_CD_DATA_SECTOR_HEADER_SIZE);
		memset(sector_data, 0, UTIL_RSPC_CD_DATA_SECTOR_HEADER_SIZE);
		cd_mode = CD_MODE2;
	}

	/* do the RSPC error correction for the complete sector data except the
	 * first 12 bytes ... these bytes only include the SYNC word */
	result = util_rspc_decoder(sector_data, rspc_iterations);

	/* Mode2 XA Form 1 RSPC is detroying the header word    */
	/* So we have to back-it-up, and restore it afterwards.  */
	if (cd_mode == CD_MODE2) {
		memcpy(sector_data, &(u32_backup_header),
				UTIL_RSPC_CD_DATA_SECTOR_HEADER_SIZE);
	}

	if (result != E_OK) {
		/* there are still errors */
		block_valid = -EIO;
	} else {
		/* all errors were corrected or there was no error in sector */
		block_valid = E_OK;
	}

	if (crc_check_enabled == TRUE) {
		/*calculate the CRC checksum after RSPC correction */
		/*calculate crc from byte 0 to 2063 ...has to be E_OK if valid*/
		crc32 = UTIL_rspc_calc_crc32(sector_data, cd_mode);

		if (crc32 != E_OK) {
			/* there are still errors */
			block_valid = -EIO;
		} else {
			/* set the block valid even if RSPC correction failed */
			/* because CRC check is valid --> Data is valid */
			/* That is meant with function is not assured in RSPC
								correction */
			block_valid = E_OK;
		}
	}
	/* take the result to the StreamWorkTask */
	return block_valid;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_calc_crc32
 *
 * \see     UTIL_rspc.h
 *------------------------------------------------------------------------------
 */
unsigned int UTIL_rspc_calc_crc32(unsigned short *data_pt,
						enum cd_mode_te sector_mode)
{
	unsigned int result = UTIL_RSPC_FULL_U32_MASK;
	unsigned short len = 0;
	unsigned short start_byte = 0;
	unsigned short i = 0;
	unsigned char *u8_data_pt = (unsigned char *) data_pt;

	switch (sector_mode) {
	case CD_MODE1: {
		/* this simulates the sync word result */
		result = UTIL_RSPC_SIM_SYNC_WORD_RESULT_MODE1;
		len = UTIL_RSPC_CRC32_MODE1_LEN << 1;
		start_byte = 0;
		break;
	}
	case CD_MODE2: {
		result = 0;
		/* CRC calc of a Mode2 sector officially starts after the header
		 word */
		/* 8 bytes more than for Mode 1 - 2064 */
		len = UTIL_RSPC_CRC32_MODE2_LEN << 1;
		start_byte = UTIL_RSPC_CRC32_MODE2_START_OFFSET;
		break;
	}
	default:
		break;
	}

	if (result != UTIL_RSPC_FULL_U32_MASK) {
		for (i = start_byte; i < len; i += (1 + 1)) {
			/* Take the BYTESWAP into account */
			result = aul_Crc32Tab[((result	^
					*(u8_data_pt + i +
					UTIL_RSPC_CDED_BYTE_SWAP_CORR_VAL))
					& UTIL_RSPC_LOWER_BYTE_MASK)]
					^ (result
	>> UTIL_RSPC_SHIFT_BITS_FOR_BYTE);
			result = aul_Crc32Tab[((result
					^ *(u8_data_pt + i + 1
					- UTIL_RSPC_CDED_BYTE_SWAP_CORR_VAL))
					& UTIL_RSPC_LOWER_BYTE_MASK)]
					^ (result
					>> UTIL_RSPC_SHIFT_BITS_FOR_BYTE);
		}
	}
	return result;
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_decoder
 *
 * Description: Execute the correction of errors in the given data until
 *              no more errors exist, the given maximum number of iterations
 *              has been exceeded or the number of detected errors is to
 *              high for any correction.
 *
 * \param   unsigned short *rspc_data - data block pointer
 * \param   unsigned int rspc_iterations - max number of iterations
 *
 * \return   int - E_OK in case of success; error code in case of failure
 E_IO - not all errors could be corrected
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static int util_rspc_decoder(unsigned short *rspc_data,
					enum rspc_iterations rspc_iterations)
{
	int rc = E_OK;
	bool continue_ = TRUE;
	unsigned char iteration = 0;

	/* start the RSPC decoder, first initialize some variables */
	util_rspc_init_decoder();

	/* first try to correct in both direction */
	while (((remaining_p_errors > 0) || (remaining_q_errors > 0))
		&& (continue_ == TRUE) && (iteration < rspc_iterations)) {
		p_error_line_index = 0;
		util_rspc_correct_p_matrix(rspc_data, iteration);
		remaining_q_errors = q_error_line_index;

		if (remaining_q_errors > RSPC_DECODER_Q_ERROR_THRESHHOLD) {
			/* to many errors for correction -> abort*/
			continue_ = FALSE;
		} else {
			q_error_line_index = 0;

			util_rspc_correct_q_matrix(rspc_data, iteration);

			remaining_p_errors = p_error_line_index;

			if (remaining_p_errors
					> RSPC_DECODER_P_ERROR_THRESHHOLD) {
				/* to many errors for correction -> abort*/
				continue_ = FALSE;
			}
			/* go for iterations */
			iteration++;
		}
	}

	if ((remaining_p_errors > 0) || (remaining_q_errors > 0)) {
		/* there are still errors, which could not be corrected */
		rc = -EIO;
	}

	return rc;
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_correct_p_matrix
 *
 * Description: Calculate the S0 checksum and execute P error correction if
 *              an error has been detected
 *              Mark the corresponding Q for each corrected error (
 *                  start Q error for the data again as now more errors
 *                  could be corrected )
 *
 *
 * \param   unsigned short *rspc_data - block data pointer
 * \param   unsigned int rspc_iterations - current iteration (for optimization)
 *
 * \return   None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_correct_p_matrix(unsigned short *rspc_data,
						unsigned char iteration)
{
	short p_vector = 0;
	unsigned int k = 0;
	unsigned short S0 = 0;

#ifdef RS_OPTIMIZED
	unsigned char i8P_vec = 0;
	unsigned short ierrp = 0;
	unsigned short uaS0_Vec[NUM_P_VECTORS_16] = { 0 };
	bool continue_ = TRUE;

	if (iteration == 0) {
		ierrp = 0;
		/* look for errors in all p-vectors, before correction*/
		for (i8P_vec = 0; i8P_vec < NUM_P_VECTORS_16; i8P_vec++) {
			/* reset the vector to No error found value -1 */
			p_error_lines[i8P_vec] = NO_ERROR_FOUND;

			/* check vector */
			S0 = UTIL_rspc_calc_p_s0(i8P_vec, rspc_data);

			if (S0 != 0) {
				/* save Syndrom 0 */
				uaS0_Vec[ierrp] = S0;
				/*all errors are in line in p_error_lines
									array*/
				p_error_lines[ierrp] = (signed char) i8P_vec;
				ierrp++;
			}
		}

		/* all vectors are defect */
		if (ierrp >= NUM_P_VECTORS_16) {
			p_error_line_index = NUM_P_VECTORS_16;
			return;
		}

		for (i8P_vec = 0; ((i8P_vec < ierrp) && (continue_ == TRUE));
				i8P_vec++) {
			/* testet vector */
			p_vector = p_error_lines[i8P_vec];
			/* reset line */
			p_error_lines[i8P_vec] = NO_ERROR_FOUND;

			/* an error was detected, try to correct it */
			/* p_vector is always greater -1 */
			util_rspc_correct_p_vector((unsigned short) p_vector,
					uaS0_Vec[i8P_vec], rspc_data,
					iteration);

			/* P error line index greater 24 */
			if (p_error_line_index > RSPC_STOP_ON_P_ERROR) {
				/* to many errors for correction -> abort*/
				continue_ = FALSE;
			}
		}
	} else {
		/* no need to calc the S0 for all rows
		 All rows which might be fixed are marked
		 by the previous Q run */

		/* Process marked rows in p_error_lines */
		k = 0;
		do {
			/* save the p_error_line_index entry */
			p_vector = p_error_lines[k];
			/* reset the p_error_line_index entry */
			p_error_lines[k] = NO_ERROR_FOUND;
			/* increment the index counter */
			k++;

			if (p_vector != NO_ERROR_FOUND) {
				/* Calculate Syndrom 0 on 16 Bit vectors, MSByte
						 and LSByte at the same time */
				/* we can cast the p_vector unsigned - because
					p_vector is not NO_Error_Found */
				S0 = UTIL_rspc_calc_p_s0((unsigned short)
						p_vector, rspc_data);

				/* if syndrom 0 is not equal zero, an error was
								detected */
				if (S0 != 0) {
					/* an error was detected,
							 try to correct it */
					util_rspc_correct_p_vector(
							(unsigned short)
							p_vector, S0,
							rspc_data, iteration);
				}
				if (p_error_line_index > RSPC_STOP_ON_P_ERROR) {
					/* to many errors for
							correction -> abort*/
					continue_ = FALSE;
				}
			}
		} while ((p_vector != NO_ERROR_FOUND) && (continue_ == TRUE));
	}
#else /* #ifdef RS_OPTIMIZED*/
	/* process all marked Ps to fix more errors (after init all marked)*/
	k = 0;
	do {
		/* save the p_error_line_index entry */
		p_vector = p_error_lines[k];
		/* reset the p_error_line_index entry */
		p_error_lines[k] = NO_ERROR_FOUND;
		/* increment the index counter */
		k++;

		if (p_vector != NO_ERROR_FOUND) {
			/* Calculate Syndrom 0 on 16 Bit vectros, MSByte and
						 LSByte at the same time */
			S0 = UTIL_rspc_calc_p_s0((unsigned short)p_vector,
							rspc_data);

			/*if syndrom 0 is not equal zero,an error was detected*/
			if (S0 != 0) {
				/* an error was detected, try to correct it */
				util_rspc_correct_p_vector((unsigned short)
					p_vector, S0, rspc_data, iteration);
			}
		}
	} while (p_vector != NO_ERROR_FOUND);
#endif /* #ifdef RS_OPTIMIZED*/
	return;
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_correct_q_matrix
 *
 * Description: calculate the S0 checksum and execute Q error correction if
 *              an error has been detected
 *              Mark the corresponding P for each corrected error (
 *                  start P error for the data again as now more errors
 *                  could be corrected )
 *
 * \param   unsigned short *rspc_data - block data pointer
 * \param   unsigned int rspc_iterations - current iteration (for optimization)
 *
 * \return   None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_correct_q_matrix(unsigned short *rspc_data,
						unsigned char iteration)
{
	short q_vector;
	unsigned int k;
	unsigned short S0;

	/* go for all Vectors of the Q-Matrix and calcularte syndrom 0 */
#ifdef RS_OPTIMIZED
	unsigned char i8Q_vec = 0;
	unsigned short uaS0_Vec[NUM_Q_VECTORS_16] = { 0 };
	unsigned short ierrq = 0;
	bool continue_ = TRUE;

	if (iteration == 0) {
		/* look for errors in all p-vectors, before correction */
		for (i8Q_vec = 0; i8Q_vec < NUM_Q_VECTORS_16; i8Q_vec++) {
			/* reset the vector */
			q_error_lines[i8Q_vec] = NO_ERROR_FOUND;

			/* check vector */
			S0 = UTIL_rspc_calc_q_s0(i8Q_vec, rspc_data);

			if (S0 != 0) {
				uaS0_Vec[ierrq] = S0; /*save syndrom*/
				q_error_lines[ierrq] = (signed char) i8Q_vec;
				ierrq++;
			}
		}

		if (ierrq >= NUM_Q_VECTORS_16) {
			/* to many errors for correction -> abort*/
			q_error_line_index = NUM_Q_VECTORS_16;
			return;
		}

		for (i8Q_vec = 0; ((i8Q_vec < ierrq) && (continue_ == TRUE));
								i8Q_vec++) {
			/* testet vector */
			q_vector = q_error_lines[i8Q_vec];
			/* reset line */
			q_error_lines[i8Q_vec] = NO_ERROR_FOUND;

			/* an error was detected, try to correct it */
			util_rspc_correct_q_vector((unsigned short) q_vector,
					uaS0_Vec[i8Q_vec], rspc_data,
					iteration);

			if (q_error_line_index > RSPC_STOP_ON_Q_ERROR) {
				/* to many errors for correction -> abort */
				continue_ = FALSE;
			}
		}
	} else {
		/* process all marked Qs to fix more errors */
		k = 0;
		do {
			/* save the q_error_line_index entry */
			q_vector = q_error_lines[k];
			/* reset the q_error_line_index entry */
			q_error_lines[k] = NO_ERROR_FOUND;
			/* increment the index counter */
			k++;

			if (q_vector != NO_ERROR_FOUND) {
				/* Calculate Syndrom 0 on 16 Bit vectros, MSByte
						 and LSByte at the same time */
				S0 = UTIL_rspc_calc_q_s0((unsigned short)
							q_vector, rspc_data);

				/* if syndrom 0 is not equal zero, an error was
								detected */
				if (S0 != 0) {
					/* an error was detected, try to correct
									 it */
					util_rspc_correct_q_vector(
							(unsigned short)
							q_vector, S0,
							rspc_data, iteration);
				}
				if (q_error_line_index > RSPC_STOP_ON_Q_ERROR) {
					/* to many errors for
							correction -> abort*/
					continue_ = FALSE;
				}
			}
		} while ((q_vector != NO_ERROR_FOUND) && (continue_ == TRUE));
	}
#else
	/* process all marked Qs to fix more errors  (after init all marked) */
	k = 0;
	do {
		/* save the q_error_line_index entry */
		q_vector = q_error_lines[k];
		/* reset the q_error_line_index entry */
		q_error_lines[k] = NO_ERROR_FOUND;
		/* increment the index counter */
		k++;

		if (q_vector != NO_ERROR_FOUND) {
			/* Calculate Syndrom 0 on 16 Bit vectros, MSByte and
						LSByte at the same time */
			S0 = UTIL_rspc_calc_q_s0((unsigned short)q_vector,
								rspc_data);

			/*if syndrom0 is not equal zero,an error was detected */
			if (S0 != 0) {
				/* an error was detected, try to correct it */
				util_rspc_correct_q_vector((unsigned short)
					q_vector, S0, rspc_data, iteration);
			}
		}
	} while (q_vector != NO_ERROR_FOUND);
#endif /* #if RS_OPTIMIZED*/
	return;
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_correct_p_vector
 *
 * Description: After an error has been detected using S0 checksum in a P row
 *              on 2 bytes this function is used to start error correction on
 *              the affected byte(s)
 *
 * \param   unsigned short p_vector - index of P which is processed
 * \param   unsigned short SO_16 - already computed S0 checksum
 * \param   unsigned short *rspc_data - block data pointer
 * \param   unsigned char iteration - current iteration (for optimization)
 *
 * \return   None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_correct_p_vector(unsigned short p_vector,
				unsigned short S0_16, unsigned short *rspc_data,
				unsigned char iteration)
{
	unsigned char S0 = 0;
	unsigned int k = 0;

	/* go for MSByte and LSByte */
	for (k = 0; k < MSB_LSB_BYTE_MAX_CNT; k++) {
		/* LSByte if k==0, MSByte if k==1 */
		S0 = (unsigned char)(S0_16 & BYTE_MASK);
		S0_16 >>= BITS_PER_BYTE;

		if (S0 != 0) {
			/*handle error, calculate the error position and value*/
			util_rspc_handle_error(S0, p_vector, k, P_PARITY,
					rspc_data, iteration);
		}
	}
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_correct_p_vector
 *
 * Description: After an error has been detected using S0 checksum in a P row
 *              on 2 bytes this function is used to start error correction on
 *              the affected byte(s)
 *
 * \param   unsigned short q_vector - index of Q which is processed
 * \param   unsigned short SO_16 - already computed S0 checksum
 * \param   unsigned short *rspc_data - block data pointer
 * \param   unsigned char  iteration - current iteration (for optimization)
 *
 * \return   None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_correct_q_vector(unsigned short q_vector,
				unsigned short S0_16, unsigned short *rspc_data,
				unsigned char iteration)
{
	unsigned char S0 = 0;
	unsigned int k = 0;

	/* go for MSByte and LSByte */
	for (k = 0; k < MSB_LSB_BYTE_MAX_CNT; k++) {
		S0 = (unsigned char)(S0_16 & BYTE_MASK);
		S0_16 >>= BITS_PER_BYTE;
		if (S0 != 0) {
			/*handle error, calculate the error position and value*/
			util_rspc_handle_error(S0, q_vector, k, Q_PARITY,
					rspc_data, iteration);
		}
	}
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_handle_error
 *
 * Description: Correct a detected error in the given P or Q row
 *              In case the error is corrected this function will mark
 *              the corresponding P (for parity Q) or Q (for parity P)
 *              as now more errors could be fixed
 *
 * \param   unsigned char S0 - S0 checksum of the given P or Q row
 * \param   unsigned short row_16 - index of the row which is processed
 * \param   paritytype_te parity - process Q or P
 * \param   unsigned short *rspc_data - block data pointer
 * \param   unsigned char  iteration - current iteration (for optimization)
 *
 * \return   None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_handle_error(unsigned char S0, unsigned short row_16,
				unsigned int MLSB,
				enum paritytype_te parity,
				unsigned short *rspc_data,
				unsigned char iteration)
{
	unsigned char *data8 = (unsigned char *) rspc_data;
	unsigned char error_value = 0;

	unsigned char error_position_u8 = 0;
	unsigned short error_position_u16 = 0;
	unsigned char S1 = 0;

	/* toggle upper and lower byte if BIG_ENDIAN is required*/
#ifdef UTIL_RSPC_CDED_BIG_ENDIAN
	MLSB ^= 1;
#endif

	/* calculate Syndrom 1 */
	if (parity == P_PARITY)
		S1 = UTIL_rspc_calc_p_s1(row_16, MLSB, rspc_data);
	else
		S1 = UTIL_rspc_calc_q_s1(row_16, MLSB, rspc_data);

	/* where is the position in the shortened reed solomon vector */
	/* in case S1 is 0 the result is invalid. This invalid value will
	 not lead to any problem as below if conditions are only
	 fullfilled if S1 is different from 0 */
	error_position_u8 = UTIL_rspc_GF8_apow_quotient(S0, S1);

	/* calculate the error from Syndrom 1, the error value from
							 Syndrom 0 is S0 */
	error_value = UTIL_rspc_GF8_apow_product(error_position_u8, S1);

	if (parity == P_PARITY) {
		if ((error_position_u8 >= START_P_VECTOR) && (S1 != 0)) {
			/* a valid error was found, so correct it */
			/* get the right position in datablock */
			/* it was a shortened Reed Solomon Code(leading zeros)*/
			error_position_u8 -= START_P_VECTOR;

			/* influence on Q-Matrix */
			if (iteration > 0) {
				/* in the first iteration, we don't have to
				influence the Q-Matrix,becaus we have to
				calculate all vectors of the Q-Matrix */
				util_rspc_set_influence_matrix(parity, row_16,
						error_position_u8);
			}

			/* calculate the error position in the block_data */
			error_position_u16 =
					(unsigned short)(((row_16 << 1) + MLSB)
						+ ((unsigned short)(
						error_position_u8)
						* NUM_P_VECTORS));

			/* do the error correction */
			data8[error_position_u16] ^= error_value;
		} else {
			/*if row is not already marked,mark it as not fixed*/
			if (p_error_line_index > 0) {
				if (p_error_lines[p_error_line_index - 1]
						!= (short) row_16) {
					p_error_lines[p_error_line_index++] =
							(short) row_16;
				}
			} else {
				p_error_lines[p_error_line_index++] =
						(short) row_16;
			}
		}
	} else {
		if ((error_position_u8 >= START_Q_VECTOR) && (S1 != 0)) {
			/* a valid error was found, so correct it */

			/* get the right position in datablock */
			/*it was a shortened Reed Solomon Code (leading zeros)*/
			error_position_u8 -= START_Q_VECTOR;

			if (error_position_u8
					< (LEN_Q_VECTORS
					- LEN_Q_VECTORS_MINUS_OFFSET)) {
					/* error is in data-field */

				/* influence on P-Matrix */
				util_rspc_set_influence_matrix(parity, row_16,
						error_position_u8);

				/* real index in the data block */
				error_position_u16 =
						(unsigned short)(
							((row_16
							* NUM_P_VECTORS_16)
							<< 1)
							+ MLSB)
							+ ((unsigned short)(
							error_position_u8)
							* OFFSET_Q_VECTOR_8);

				/* the real byte pos is calculated by doing the
							modulo operation */
				while (error_position_u16
						> (MODULO_Q_VECTOR << 1)) {
					error_position_u16 -= (MODULO_Q_VECTOR
							<< 1);
				}
			} else {
				/* error is in Q-parity field, no influence on
								P-Matrix */
				error_position_u16 =
						(error_position_u8
						- (LEN_Q_VECTORS
						- LEN_Q_VECTORS_MINUS_OFFSET));
				error_position_u16 =
						(unsigned short)(((((
						NUM_P_VECTORS_16
						+ error_position_u16)
						* NUM_Q_VECTORS_16)
						+ row_16)
						<< 1)
						+ MLSB);
			}
			/* do the error correction */
			data8[error_position_u16] ^= error_value;
		} else {
			/*if row is not already marked, mark it as not fixed */
			if (q_error_line_index > 0) {
				if (q_error_lines[q_error_line_index - 1]
						!= (short) row_16) {
					q_error_lines[q_error_line_index] =
							(short) row_16;
					q_error_line_index++;
				}
			} else {
				q_error_lines[q_error_line_index] =
						(short) row_16;
				q_error_line_index++;
			}
		}
	}
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_init_decoder
 *  Description : Init global variables for RSPC calculation
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_init_decoder(void)
{
	signed char k = 0;

	remaining_p_errors = NUM_P_VECTORS_16;
	remaining_q_errors = NUM_Q_VECTORS_16;
	p_error_line_index = 0;
	q_error_line_index = 0;

	/* at begin all lines may include errors */
	for (k = 0; k < NUM_Q_VECTORS_16; k++) {
		q_error_lines[k] = k;
		p_error_lines[k] = k;
	}

	/* mark end of Q error line */
	q_error_lines[k] = NO_ERROR_FOUND;

	/* reset rest of p_error_lines */
	for (; k < NUM_P_VECTORS_16; k++)
		p_error_lines[k] = k;

	/* mark end of P error lines */
	p_error_lines[k] = NO_ERROR_FOUND;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_calc_p_s0
 * Description : The S0 is only a simple XOR. Therefore it is possible to
 *               process the high and the low byte of a P row in parallel.
 *               This is done by this function.
 *
 * \param  unsigned short row_16 - index of the P row
 * \param  unsigned short * rspc_data - pointer to the data block
 *
 * \return   unsigned short - S0 of P row
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned short UTIL_rspc_calc_p_s0(unsigned short row_16,
						unsigned short *rspc_data)
{
	/* value of row is between 0 and NUM_P_VECTORS_16 (43) */

	unsigned short *data16 = rspc_data;
	unsigned int k = 0;
	/* reset Syndrom 0 */
	unsigned short S0 = 0;

	/* start with the right row */
	data16 += row_16;

	/* Syndrom 0 is just the XOR of all values of the vector */
	for (k = 0; k < LEN_P_VECTORS; k++) {
		S0 ^= *data16;
		data16 += NUM_P_VECTORS_16;
	}

	return S0;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_calc_q_s0
 * Description : The S0 is only a simple XOR. Therefore it is possible to
 *               process the high and the low byte of a Q row in parallel.
 *               This is done by this function.
 *
 * \param  unsigned short row_16 - index of the Q row
 * \param  unsigned short * rspc_data - pointer to the data block
 *
 * \return   unsigned short - S0 of Q row
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned short UTIL_rspc_calc_q_s0(unsigned short row_16,
						unsigned short *rspc_data)
{
	/* value of row is between 0 and NUM_Q_VECTORS_16 (26) */

	unsigned short *data16 = rspc_data;

	unsigned int k = 0;
	/* reset Syndrom 0 */
	unsigned short S0 = 0;

	/* start with the right row */
	data16 += (row_16 * NUM_P_VECTORS_16);

	/* Syndrom 0 is just the XOR of all values of the vector */
	for (k = 0; k < (NUM_Q_VECTORS_16 - row_16); k++) {
		S0 ^= *data16;
		data16 += OFFSET_Q_VECTOR;
	}

	/* do the modulo operation */
	data16 -= MODULO_Q_VECTOR;

	if (row_16 >= UTIL_RSPC_ROW_OFFSET_10) {
		/* after row 10 we have two times the modulo operation in
								adressing */
		for (; k < (NUM_Q_VECTORS - row_16); k++) {
			S0 ^= *data16;
			data16 += OFFSET_Q_VECTOR;
		}
		data16 -= MODULO_Q_VECTOR;
	}

	for (; k < (LEN_Q_VECTORS - LEN_Q_VECTORS_MINUS_OFFSET); k++) {
		S0 ^= *data16;
		data16 += OFFSET_Q_VECTOR;
	}

	/* add the parity words */
	data16 = rspc_data + MODULO_Q_VECTOR + row_16;

	S0 ^= *data16;
	data16 += NUM_Q_VECTORS_16;
	S0 ^= *data16;

	/* return Syndrom 0 of Q-Parity vector */
	return S0;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_calc_p_s1
 * Description : The S1 is based on multiplication in the galois field.
 *            Therefore only the high or the low byte can be calculated at once.
 *             This function will calculate the S1 for the high or the low byte
 *             of a given P row
 *
 * \param  unsigned short row_16 - index of the P row
 * \param  unsigned int MLSB - 0 or 1 -determine if S1 for high or low byte is
 *  calculated
 * \param  unsigned short * rspc_data - pointer to the data block
 *
 * \return   unsigned char - S1 of high or low byte of given P row
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_calc_p_s1(unsigned short row_16,
				unsigned int MLSB, unsigned short *rspc_data)
{
	/* value of row_16 is between 0 and NUM_Q_VECTORS_16 or NUM_P_VECTORS_16
									 (43) */

	unsigned short *data16 = rspc_data;
	unsigned char *data8;
	/* reset Syndrom 1 */
	unsigned char S1 = 0;
	unsigned char k = 0;

	/*start with the right row */
	data16 += row_16;

	/*S1 has to be calculated bytewise,because we are calculating in GF256*/
	data8 = (unsigned char *) data16;
	/* change to MSB or LSB */
	data8 += MLSB;

	for (k = LEN_P_VECTORS; k > 0; k--) {
		S1 ^= UTIL_rspc_GF8_apow_product(k, *data8);
		data8 += NUM_P_VECTORS;
	}

	return S1;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_calc_q_s1
 * Description :The S1 is based on multiplication in the galois field.
 *            Therefore only the high or the low byte can be calculated at once.
 *             This function will calculate the S1 for the high or the low byte
 *             of a given Q row
 *
 * \param  unsigned short row_16 - index of the Q row
 * \param  unsigned int MLSB - 0 or 1 -determine if S1 for high or low byte is
 *  calculated
 * \param  unsigned short * rspc_data - pointer to the data block
 *
 * \return   unsigned char - S1 of high or low byte of given Q row
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_calc_q_s1(unsigned short row_16,
				unsigned int MLSB, unsigned short *rspc_data)
{
	/* value of row16 is between 0 and NUM_Q_VECTORS_16 or
						 NUM_P_VECTORS_16 (43) */

	unsigned short *data16 = rspc_data;
	unsigned char *data8 = NULL;
	unsigned int k = 0;
	unsigned char S1 = 0; /* reset Syndrom 1 */

	/* the exponent for calculating the DFT - value of LEN_Q_VECTORS 45 */
	unsigned char exp_ = LEN_Q_VECTORS;

	/* start with the right row */
	data16 += (row_16 * NUM_P_VECTORS_16);

	/*S1 has to be calculated bytewise,because we are calculating in GF256*/
	data8 = (unsigned char *) data16;

	/* change to MSB or LSB */
	data8 += MLSB;

	/* start with the right row */
	data16 += (row_16 * NUM_P_VECTORS_16);

	/* in this for loop the exp_ will not decrement below 0 because */
	/* NUM_Q_VECTORS(26) - row) is smaller than the value of LEN_Q_VECTORS*/
	for (k = 0; k < (NUM_Q_VECTORS_16 - row_16); k++, exp_--) {
		S1 ^= UTIL_rspc_GF8_apow_product(exp_, *data8);
		data8 += OFFSET_Q_VECTOR_8;
	}

	/* do the first modulo operation */
	data8 -= (MODULO_Q_VECTOR << 1);

	if (row_16 >= UTIL_RSPC_ROW_OFFSET_10) {
		/* after row 10 we have two times the modulo operation in
								adressing */
		for (; k < (NUM_Q_VECTORS - row_16); k++, exp_--) {
			S1 ^= UTIL_rspc_GF8_apow_product(exp_, *data8);
			data8 += OFFSET_Q_VECTOR_8;
		}

		/* do the second modulo operation */
		data8 -= (MODULO_Q_VECTOR << 1);
	}

	for (; k < (LEN_Q_VECTORS - LEN_Q_VECTORS_MINUS_OFFSET); k++, exp_--) {
		S1 ^= UTIL_rspc_GF8_apow_product(exp_, *data8);
		data8 += OFFSET_Q_VECTOR_8;
	}

	/* add the parity bytes */
	data16 = (unsigned short *) (rspc_data + MODULO_Q_VECTOR + row_16);
	data8 = (unsigned char *) data16;

	/* change to MSB or LSB */
	data8 += MLSB;
	S1 ^= UTIL_rspc_GF8_apow_product(exp_, *data8);
	exp_--;
	data8 += NUM_Q_VECTORS;
	S1 ^= UTIL_rspc_GF8_apow_product(exp_, *data8);
	/* exp_--;*/

	/* return syndrom 1 of Q-Parity */
	return S1;
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_set_influence_matrix
 * Description : After an error for P or Q has been calculated, it might be
 *               possible to correct another error in the Q or P row of the
 *               error which could not be fixed before.
 *               If the error has been fixed while processing P the
 *               Q row of the corrected byte is marked to be rechecked in the
 *               next phase.
 *               If the error has been fixed while processing Q the
 *               P row of the corrected byte is marked to be rechecked in the
 *               next phase.
 *
 * \param paritytype_te parity - was the error corrected while processing P or
 *  Q
 * \param unsigned short row_16 - index of P or Q row
 * \param unsigned char error_position - position of the fixed byte in the P or
 *  Q row
 *
 * \result None
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static void util_rspc_set_influence_matrix(enum paritytype_te parity,
						unsigned short row_16,
						unsigned char error_position)
{
	unsigned int k = 0;
	short influenced_row = 0;

	if (parity == P_PARITY) {
		/* calculate the Q-row, which is was influenced by the error
								correction */
		influenced_row = (short)((unsigned short) error_position
								- row_16);

		while (influenced_row < 0)
			influenced_row += NUM_Q_VECTORS_16;

		/* mark the row, if it is not already marked */
		for (k = 0; k < q_error_line_index; k++) {
			if (q_error_lines[k] == influenced_row) {
				/* row already marked */
				return;
			}
		}

		q_error_lines[q_error_line_index] = influenced_row;
		q_error_line_index++;
	} else {
		/* calculate the Q-row, which is was influenced by the error
								correction */
		influenced_row = (short) error_position;

		/* mark the row, if it is not already marked */
		for (k = 0; k < p_error_line_index; k++) {
			if (p_error_lines[k] == influenced_row) {
				/* row already marked */
				return;
			}
		}

		p_error_lines[p_error_line_index++] = influenced_row;
	}
}

/**
 *------------------------------------------------------------------------------
 * \func    util_rspc_fill_galois_arrays
 *
 * Description: Fill with Galois field.
 *              Note that only Log[0] is invalid and must not be used
 *              All other values are positive and between 1 and 255.
 *
 * \return  bool  - True after init.
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static bool util_rspc_fill_galois_arrays(void)
{
	unsigned short i;
	unsigned short temp;
	/* fill the Log and AntiLog array for index 0 */
	_log[0] = 0; /* invalid - Log[0] must not be used */
	alog[0] = 1;

	/* fill the Log and AntiLog array from index 1 up to 255 */
	for (i = 1; i < GF; i++) {
		temp = alog[i - 1] * (1 + 1); /* 1,2,4,8,16.... */

		if (temp >= GF)
			temp ^= PP; /* PP=285... (256 ^= 285) = 29  */

		alog[i] = (unsigned char) temp;
		_log[alog[i]] = (unsigned char) i; /* Log[1,2,4,8] = 1,2,3,4 */
	}

	/* return true to indicate that the Log and AntiLog array are
								initialized */
	return TRUE;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_GF8_apow_product
 *
 * Description:
 *      The multiply of multiplicand*multiplier in GF256 is:
 *      alog[ (Log[multiplicand] + Log[multiplier]) % (GF-1)]);
 *      But we want to calculate alpha^multiplicand * multiplier,
 *      so we don't need to build the Log from multiplicand.
 *
 *
 * \param   unsigned char multiplicand
 * \param   unsigned char multiplier
 *
 * \return  unsigned char - the result
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_GF8_apow_product(unsigned char multiplicand,
						unsigned char multiplier)
{
	unsigned short sum = 0;
	unsigned char rc = 0;

	if ((multiplicand * multiplier) != 0) {
		sum = (multiplicand + _log[multiplier]);

		/* sum < 255 */
		if (sum < (GF - 1)) {
			/* Note : Log[0] is not accessed as multiplicand and
			 multiplier are > 0*/
			rc = (unsigned char)(alog[sum]);
		} else { /* sum > 255  - index of Alog is greater/equal 1 -
								so rc >= 0 */
			rc = (unsigned char)(alog[sum - (GF - 1)]);
		}
	}
	return rc;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_GF8_apow_quotient
 *
 * Description:
 *       This is the original quotient
 *       return ( alog[( ((GF-1) + Log[dividend]) - Log[divisor]) % (GF-1)] );
 *       but we need the Log of that, so Log(ALOG(x)) = x
 *
 * \param   unsigned char dividend
 * \param   unsigned char divisor - if divisor is 0 a invalid result is
 * returned.
 *                       The result has to be ignored in this case.
 *
 * \return  unsigned char - the result
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_GF8_apow_quotient(unsigned char dividend,
							unsigned char divisor)
{
	short sum = 0;
	unsigned char rc = 0;

	if (divisor == 0) {
		rc = 0; /* in this case the return value is invalid and has
								to be ignored*/
	} else if (dividend == 0) {
		rc = 0;
	} else {
		sum = (short)(_log[dividend]) - (short)(_log[divisor]);

		/* sum can be between -254 and +254 */
		if (sum > 0) {
			rc = (unsigned char)(sum);
		} else { /* rc = sum + 255 */
			rc = (unsigned char)(sum + (GF - 1));
		}
	}

	return rc;
}

#ifdef EXTENDED_GALOIS_ARITHMETIC
/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_GF8_sum
 *
 * Description: Calc the sum in GF (this is a simple XOR)
 *
 * \param   unsigned char summanda
 * \param   unsigned char summandb
 *
 * \return  unsigned char - the result of minuend xor subtrahend
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_GF8_sum(unsigned char summanda,
							unsigned char summandb)
{
	return summanda ^ summandb;
}

/**
 *------------------------------------------------------------------------------
 * \func    UTIL_rspc_GF8_diff
 *
 * Description: Calc the diff in GF (this is a simple XOR)
 *
 * \param   unsigned char minuend
 * \param   unsigned char subtrahend
 *
 * \return  unsigned char - the result of minuend xor subtrahend
 *
 * \see     None
 *------------------------------------------------------------------------------
 */
static unsigned char UTIL_rspc_GF8_diff(unsigned char minuend,
						unsigned char subtrahend)
{
	return minuend ^ subtrahend;
}
#endif /* EXTENDED_GALOIS_ARITHMETIC */

