/*
 * MASCA Sysfs Configuration Logic file
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/device.h>
#include "masca_common_types.h"
#include "masca_sysfs.h"

/*Show and Configure Auto Reinset Timeout in Sysfs*/
module_param_named(auto_reinsert_tmout_sys, auto_reinsert_tmout, int,
							S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(auto_reinsert_tmout_sys, "MASCA Auto reinsert timeout");

ssize_t auto_reinsert_tmout_show(struct device_driver *ddp, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", auto_reinsert_tmout/1000);
}

/* The auto_reinsert_tmout_sys variable can be configurable under below path
   /sys/bus/masca/drivers/MASCA/ */
ssize_t auto_reinsert_tmout_store(struct device_driver *ddp,
						const char *buf, size_t count)
{
	unsigned int value;

	sscanf(buf, "%d", &value);
	/*Accept parameter from 0 to 20 secs*/
	if (value >= 0 && value <= 20)
		auto_reinsert_tmout = value * 1000;

	return count;
}
DRIVER_ATTR(auto_reinsert_tmout_sys, S_IRUGO | S_IWUSR,
			auto_reinsert_tmout_show, auto_reinsert_tmout_store);

/*Show and Configure Lifetime Timeout in Sysfs*/
module_param_named(lifetime_tmout_sys, lifetime_tmout, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(lifetime_tmout_sys, "MASCA lifetime timeout");

ssize_t lifetime_tmout_show(struct device_driver *ddp, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", lifetime_tmout);
}
ssize_t lifetime_tmout_store(struct device_driver *ddp,
						const char *buf, size_t count)
{
	unsigned int value;

	sscanf(buf, "%d", &value);
	lifetime_tmout = value;

	return count;
}
DRIVER_ATTR(lifetime_tmout_sys, S_IRUGO | S_IWUSR,  lifetime_tmout_show,
							lifetime_tmout_store);

/*Show and Configure SPI history size in Sysfs*/
module_param_named(spi_hist_size_sys, spi_hist_size, int,
							S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(spi_hist_size_sys, "SPI History size");

ssize_t spi_hist_size_show(struct device_driver *ddp, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", spi_hist_size);
}

/* The spi_hist_size_sys variable can be configurable under below path
   /sys/bus/masca/drivers/MASCA/ */
ssize_t spi_hist_size_store(struct device_driver *ddp,
						const char *buf, size_t count)
{
	unsigned int value;

	sscanf(buf, "%d", &value);
	/*SPI_HIST_SZ_MAX -> max diagnostic parameter allocation length
	  in RECEVIE DIAGNOSTIC RESULTS command*/
	if (value > 0 && value <= SPI_HIST_SZ_MAX)
		/*spi history size in bytes*/
		spi_hist_size = value;

	return count;
}
DRIVER_ATTR(spi_hist_size_sys, S_IRUGO | S_IWUSR,
			spi_hist_size_show, spi_hist_size_store);

/* Show and Configure Drive Temperature limit is enabled or not in Sysfs.
 * By default it is disabled */
module_param_named(enable_temp_limit_sys, enable_temp_limit, bool,
							S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(enable_temp_limit_sys,
				"Enables Temperature limit of MACSA drive");

ssize_t temp_limit_show(struct device_driver *ddp, char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%d\n", enable_temp_limit);
}

ssize_t temp_limit_store(struct device_driver *ddp,
						const char *buf, size_t count)
{
	int value;

	sscanf(buf, "%d", &value);
	if (0 < value)
		enable_temp_limit = true;
	else if (0 == value)
		enable_temp_limit = false;

	return count;
}

DRIVER_ATTR(enable_temp_limit_sys, S_IRUGO | S_IWUSR,
		temp_limit_show, temp_limit_store);

int configure_sysfs_files(struct device_driver *ptr)
{
	int ret;
	ret = driver_create_file(ptr, &driver_attr_auto_reinsert_tmout_sys);
	if (!ret)
		ret = driver_create_file(ptr, &driver_attr_lifetime_tmout_sys);
	if (!ret)
		ret = driver_create_file(ptr, &driver_attr_spi_hist_size_sys);
	if (!ret)
		ret = driver_create_file(ptr,
					&driver_attr_enable_temp_limit_sys);
	if (ret)
		remove_sysfs_files(&masca_devdrv);

	return ret;
}

void remove_sysfs_files(struct device_driver *ptr)
{
	driver_remove_file(ptr, &driver_attr_auto_reinsert_tmout_sys);
	driver_remove_file(ptr, &driver_attr_lifetime_tmout_sys);
	driver_remove_file(ptr, &driver_attr_spi_hist_size_sys);
	driver_remove_file(ptr, &driver_attr_enable_temp_limit_sys);
}

