/*
* MASCA SPI Driver
*
* Copyright (C) 2013 ADIT Corporation
* Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
*          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
*          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
*
* Copyright (C) 2006 SWAPP
*      Andrea Paterniani <a.paterniani@swapp-eng.it>
* Copyright (C) 2007 David Brownell (simplification, cleanup)
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/spi/spi.h>
#include <linux/spi/spidev.h>
#include <linux/of_gpio.h>
#include <linux/pinctrl/consumer.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include "masca_trace.h"

 /* Bit masks for spi_device.mode management.  Note that incorrect
  * settings for some settings can cause *lots* of trouble for other
  * devices on a shared bus:
  *
  *  - CS_HIGH ... this device will be active when it shouldn't be
  *  - 3WIRE ... when active, it won't behave as it should
  *  - NO_CS ... there will be no explicit message boundaries; this
  *      is completely incompatible with the shared bus model
  *  - READY ... transfers may proceed when they shouldn't.
  *
  * REVISIT should changing those flags be privileged?
  */
 #define SPI_MODE_MASK		(SPI_CPHA | SPI_CPOL | SPI_CS_HIGH \
				| SPI_LSB_FIRST | SPI_3WIRE | SPI_LOOP \
				| SPI_NO_CS | SPI_READY)
struct spidev_data {
	dev_t                   devt;
	spinlock_t              spi_lock;
	struct spi_device       *spi;
	struct list_head        device_entry;

	/* buffer is NULL unless this device is open (users > 0) */
	struct mutex            buf_lock;
	unsigned                users;
	u8                      *buffer;
};

static LIST_HEAD(device_list);
static DEFINE_MUTEX(device_list_lock);
static unsigned bufsiz = 4096;
static int probestatus = -ENXIO;
struct spidev_data *global_spidev;
struct spi_device *ptr;

/*For SPI history*/
unsigned char *spi_hist_buff;
unsigned int spi_hist_size;
static unsigned int spi_hist_old;
static unsigned int spi_hist_idx;

/*GPIO pins default configuration*/
int masca_chipselect_num;
int masca_reqline_num;
int masca_reset_num;
/* currently gpio is not requested for lstart */
int masca_lstart_num;

/*low/high GPIO pins configuration*/
/*default configurations for GPIO's is low active*/
int cs_low_active;	/*low active logic for CS pin by default*/
int reset_low_active;	/*low active logic for RESET pin by default*/
int req_low_active;	/*low active logic for request line by default*/

 /*
  * We can't use the standard synchronous wrappers for file I/O; we
  * need to protect against async removal of the underlying spi_device.
  */
static void spidev_complete(void *arg)
{
	complete(arg);
}

static ssize_t
spidev_sync(struct spidev_data *spidev, struct spi_message *message)
{
	DECLARE_COMPLETION_ONSTACK(done);
	int status;

	message->complete = spidev_complete;
	message->context = &done;

	spin_lock_irq(&spidev->spi_lock);
	status = spi_async_locked(spidev->spi, message);
	spin_unlock_irq(&spidev->spi_lock);

	if (status == 0) {
		wait_for_completion(&done);
		status = message->status;
		if (status == 0)
			status = message->actual_length;
	}
	return status;
}

/*Below function updates the spi history size set by
 * user in sysfs.It also frees the old spi history buff
 * if there is change in spi history size in sysfs.
 * It allocates new spi buffer with updated history size.
 * */
unsigned int alloc_spi_hist_size(void)
{
	if (spi_hist_old != spi_hist_size) {
		if (spi_hist_buff != NULL)
			vfree(spi_hist_buff);
		spi_hist_buff = vmalloc(spi_hist_size);
		if (spi_hist_buff != NULL) {
			memset(spi_hist_buff, 0, spi_hist_size);
			spi_hist_idx = 0;
		} else {
			SPI_TR_D("SPI History buffer allocation failed\n");
		}
		spi_hist_old = spi_hist_size;
	}

	return spi_hist_old;
}

void masca_spi_hist_store(const char *spibuff, int len ,
					unsigned int spi_hist_arg)
{
	if ((spi_hist_idx + len) > spi_hist_arg)
		spi_hist_idx = 0;

	memcpy(&spi_hist_buff[spi_hist_idx] , spibuff, len);
	spi_hist_idx += len;
}

int masca_spi_read(char *mascabuff, int count)
{
	unsigned int spi_hist_size_sz = 0;
	ssize_t status = 0;
	struct spidev_data *spidev;
	struct spi_message      m;
	struct spi_transfer     t = {
		.rx_buf         = mascabuff,
		.len            = count,
	};

	spidev = global_spidev;
	/* chipselect only toggles at start or end of operation */
	if (count > bufsiz)
		return -EMSGSIZE;

	mutex_lock(&spidev->buf_lock);
	spi_message_init(&m);
	spi_message_add_tail(&t, &m);
	status = spidev_sync(spidev, &m);
	mutex_unlock(&spidev->buf_lock);

	spi_hist_size_sz = alloc_spi_hist_size();
	if (spi_hist_size_sz > 0)
		masca_spi_hist_store(mascabuff, count , spi_hist_size_sz);

	return status;
}

int masca_spi_write(const char *buf, int count)
{
	unsigned int spi_hist_size_sz = 0;
	ssize_t status = 0;
	struct spidev_data *spidev;
	struct spi_message      m;
	struct spi_transfer     t = {
		.tx_buf         = buf,
		.len            = count,
	};

	spidev = global_spidev;
	/* chipselect only toggles at start or end of operation */
	if (count > bufsiz)
		return -EMSGSIZE;
	mutex_lock(&spidev->buf_lock);
	spi_message_init(&m);
	spi_message_add_tail(&t, &m);

	status = spidev_sync(spidev, &m);
	mutex_unlock(&spidev->buf_lock);
	spi_hist_size_sz = alloc_spi_hist_size();
	if (spi_hist_size_sz > 0)
		masca_spi_hist_store(buf, count , spi_hist_size_sz);

	return status;
}

#define SET_SPI_MODE 1
#define SET_SPI_BITS_PER_WORD 2
#define SET_SPI_SPEED 3

long masca_spi_ioctl(unsigned int cmd, unsigned long *arg)
{
	int retval = 0;
	struct spidev_data *spidev;
	struct spi_device *spi;
	u32 tmp;
	spidev = global_spidev;
	spin_lock_irq(&spidev->spi_lock);
	spi = spi_dev_get(spidev->spi);
	spin_unlock_irq(&spidev->spi_lock);

	if (spi == NULL)
		return -ESHUTDOWN;

	/* use the buffer lock here for triple duty:
	 *  - prevent I/O (from us) so calling spi_setup() is safe;
	 *  - prevent concurrent SPI_IOC_WR_* from morphing
	 *    data fields while SPI_IOC_RD_* reads them;
	 *  - SPI_IOC_MESSAGE needs the buffer locked "normally".
	 */
	mutex_lock(&spidev->buf_lock);

	switch (cmd) {
	case SET_SPI_MODE:
		tmp = *arg;
		retval = 0;

		if (retval == 0) {
			u8      save = spi->mode;

			if (tmp & ~SPI_MODE_MASK) {
				retval = -EINVAL;
				break;
			}

			tmp |= spi->mode & ~SPI_MODE_MASK;
			spi->mode = (u8)tmp;
			retval = spi_setup(spi);
			if (retval < 0)
				spi->mode = save;
			else
				dev_dbg(&spi->dev, "spi mode %02x\n", tmp);
		}
		break;

	case SET_SPI_BITS_PER_WORD:
		tmp = *arg;
		retval = 0;
		if (retval == 0) {
			u8      save = spi->bits_per_word;

			spi->bits_per_word = tmp;
			retval = spi_setup(spi);
			if (retval < 0)
				spi->bits_per_word = save;
			else
				dev_dbg(&spi->dev, "%d bits per word\n", tmp);
		}
		break;

	case SET_SPI_SPEED:
		tmp = *arg;
		retval = 0;
		if (retval == 0) {
			u32     save = spi->max_speed_hz;

			spi->max_speed_hz = tmp;
			retval = spi_setup(spi);
			if (retval < 0)
				spi->max_speed_hz = save;
			else
				dev_dbg(&spi->dev, "%d Hz (max)\n", tmp);
		}
		break;
	}

	mutex_unlock(&spidev->buf_lock);
	spi_dev_put(spi);
	return retval;
}

int masca_spi_open(void)
{
	/* SPI configuration */
	unsigned long mode = 0x03; /*SPI mode 3*/
	unsigned long word = 8;
	unsigned long speed = 100000;
	struct spidev_data *spidev;
	int status = -ENXIO;
	spidev = global_spidev;
	mutex_lock(&device_list_lock);
	spidev->buffer = kmalloc(bufsiz, GFP_KERNEL);
	status = 0;
	if (!spidev->buffer) {
		dev_dbg(&spidev->spi->dev, "open/ENOMEM\n");
		status = -ENOMEM;
	}
	mutex_unlock(&device_list_lock);

	/* SPI configuration */
	masca_spi_ioctl(SET_SPI_MODE, &mode);
	msleep(20);
	masca_spi_ioctl(SET_SPI_BITS_PER_WORD, &word);
	msleep(20);
	masca_spi_ioctl(SET_SPI_SPEED, &speed);
	msleep(20);
	return status;
}


int masca_spi_release(void)
{
	int status = 0;
	struct spidev_data *spidev;

	spidev = global_spidev;
	mutex_lock(&device_list_lock);
	kfree(spidev->buffer);
	spidev->buffer = NULL;
	mutex_unlock(&device_list_lock);

	return status;
}

static const struct file_operations spidev_fops = {
	.owner =        THIS_MODULE,
};

static int masca_read_gpio_from_dt(struct spi_device *spi)
{
	int ret = 0;
	enum of_gpio_flags flags;
	/*Reading GPIO pin values from Device Tree*/
	struct device *dev = &spi->dev;
	struct device_node *np = dev->of_node;
	struct pinctrl *pinctrl;

	if (!of_device_is_available(np))
		return -ENODEV;

	/* Enabling pinmux explicitly */
	pinctrl = devm_pinctrl_get_select_default(dev);
	if (IS_ERR(pinctrl)) {
		dev_err(dev, "setup pinctrl failed!");
		return PTR_ERR(pinctrl);
	}

	ret = of_get_named_gpio_flags(np, "reset-gpio", 0, &flags);
	if (ret < 0)
		return -ENXIO;
	masca_reset_num	= ret;
	if ((flags & OF_GPIO_ACTIVE_LOW) == 1)
		reset_low_active = 0;
	else
		reset_low_active = 1;

	ret = of_get_named_gpio_flags(np, "cs-gpio", 0, &flags);
	if (ret < 0)
		return -ENXIO;
	masca_chipselect_num = ret;
	if ((flags & OF_GPIO_ACTIVE_LOW) == 1)
		cs_low_active = 0;
	else
		cs_low_active = 1;

	ret = of_get_named_gpio_flags(np, "req-gpio", 0, &flags);
	if (ret < 0)
		return -ENXIO;
	masca_reqline_num = ret;
	if ((flags & OF_GPIO_ACTIVE_LOW) == 1)
		req_low_active = 0;
	else
		req_low_active = 1;

	ret = of_get_named_gpio_flags(np, "lstart-gpio", 0, &flags);
	if (ret < 0)
		return -ENXIO;
	masca_lstart_num = ret;
	ret = 0;
	return ret;
}

void masca_spi_lock_bus(void)
{
	struct masca_spi_str {
		struct spi_transfer xfer;
		struct spi_message  msg;
	} spimasca;

	char tx_buf[1];
	char rx_buf[1];
	/*lock bus prevents other users from adding new messages */
	spi_bus_lock(ptr->master);

	/*add dummy message and wait for completion
	to make sure no more SPI messages are pending*/
	spi_message_init(&spimasca.msg);
	spimasca.xfer.len = 0;/*len 0 will result in skipping real xfer*/
	spimasca.xfer.tx_buf = tx_buf;

	spimasca.xfer.rx_buf = rx_buf;
	spimasca.xfer.bits_per_word = 8;
	spimasca.xfer.cs_change = 0; /*no cs and waitstates between transfers*/
	spimasca.xfer.speed_hz = 100000;
	spimasca.xfer.delay_usecs = 0;

	spi_message_add_tail(&spimasca.xfer, &spimasca.msg);
	spi_sync_locked(ptr, &spimasca.msg);
}

static int masca_spi_probe(struct spi_device *spi)
{
	struct spidev_data      *spidev;
	int                     status;
	probestatus = 0;
	/* Allocate driver data */
	spidev = kzalloc(sizeof(*spidev), GFP_KERNEL);
	if (!spidev || !spi)
		return -ENXIO;

	global_spidev = spidev;
	ptr = spi;
	/* Initialize the driver data */
	spidev->spi = spi;
	spin_lock_init(&spidev->spi_lock);
	mutex_init(&spidev->buf_lock);
	INIT_LIST_HEAD(&spidev->device_entry);
	status = 0;
	status = masca_read_gpio_from_dt(spi); /*Reading GPIO values from DT*/
	if (0 != status)
		probestatus = status;
	return status;
}

static int masca_spi_remove(struct spi_device *spi)
{
	struct spidev_data      *spidev ;
	spidev = global_spidev;
	/* make sure ops on existing fds can abort cleanly */
	spin_lock_irq(&spidev->spi_lock);
	spidev->spi = NULL;
	spi_set_drvdata(spi, NULL);
	spin_unlock_irq(&spidev->spi_lock);
	/* prevent new opens */
	mutex_lock(&device_list_lock);
	list_del(&spidev->device_entry);
	if (spidev->users == 0)
		kfree(spidev);
	mutex_unlock(&device_list_lock);

	return 0;
}

static const struct of_device_id spidev_dt_ids[] = {
	{ .compatible = "rohm,dh2228fv" },
	{},
};

MODULE_DEVICE_TABLE(of, spidev_dt_ids);

static struct spi_driver masca_spi_driver = {
	.driver = {
	.name =         "masca_spidev",
	.owner =        THIS_MODULE,
	.of_match_table = of_match_ptr(spidev_dt_ids),
	},
	.probe =        masca_spi_probe,
	.remove =       masca_spi_remove,

	/* NOTE:  suspend/resume methods are not necessary here.
	* We don't do anything except pass the requests to/from
	* the underlying controller.  The refrigerator handles
	* most issues; the controller driver handles the rest.
	*/
};

int masca_spi_init(void)
{
	int status;

	status = spi_register_driver(&masca_spi_driver);
	if (0 != probestatus) {
		spi_unregister_driver(&masca_spi_driver);
		return probestatus;
	}

	return status;
}

void masca_spi_exit(void)
{
	if (spi_hist_buff != NULL)
			vfree(spi_hist_buff);
	spi_unregister_driver(&masca_spi_driver);
}
