/*
 * MASCA SCSI Handler File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <scsi/scsi_eh.h>
#include <linux/kthread.h>  /*for threads*/
#include <linux/vmalloc.h>

#include "masca_interface.h"
#include "masca_dispatch.h"
#include "masca_hal.h"
#include "masca_interpreter.h"
#include "masca_sm.h"
#include "masca_cmd_hndlr.h"
#include "masca_diag_hndlr.h"
#include "masca_atapi_bridge.h"
#include "masca_scsi_handler.h"
#include "masca_event_config.h"
#include "masca_blkdev.h"
#include "masca_block_reader.h"

#define MASCA_STATMSG		0x02
#define MASCA_DEINIT		0x04
#define DATA_PRESENT		0x07 /* (0x01 & 0x02 & 0x04) */
#define MAX_MASCA_CLASSES	5
#define MAX_CONV_CMD		40

DECLARE_WAIT_QUEUE_HEAD(waitq);
unsigned char waitcond = 0x00;
struct task_struct *recvthrd;
static DEFINE_SPINLOCK(receiver_lock);
atomic_t event_pattern = ATOMIC_INIT(0);
static DEFINE_SPINLOCK(rcvrq_lock);
struct masca_rcvr masca_rcvr_queue;
static unsigned int scmd_repid = 0x00;
struct masca_sense_buf sense_buffer;
unsigned char *g_app_buf;

struct masca_event_info out_event_global;

struct message_decoder_type {
	enum masca_cmd conv_cmd;
	unsigned char    size;
	enum masca_error
	(*cmd_decode_function)
	(const struct masca_output * const pp_output,
	unsigned char * const pp_buffer,
	const unsigned int pbuf_length);
};

static const struct message_decoder_type atapi_conv_decoder[MAX_CONV_CMD] = {
	{GET_TOC,                           0, masca_atapi_convert_toc },
	{GET_SESSION_INFO,                  0,
					 masca_atapi_convert_session_info },
	{GET_TITLE_ALBUM_AUTHOR,            0, masca_atapi_convert_cd_text },
	{GET_VERSION_INFO,                  36, masca_atapi_convert_inquiry },
	{GET_CD_CAPACITY,                   8,
					masca_atapi_convert_read_capacity },
	{GET_MEDIA_STAT,                    0, masca_atapi_convert_media_stat },
	{DIAG_GET_TEMPERATURE,              0, masca_log_sense_get_temp },
	{CMD_READ_DISK_INFO,                0, masca_read_disk_info },
	{CMD_SEND_DIAGNOSTIC,               0, masca_send_diagnostic },
	{CMD_RECEIVE_DIAGNOSTIC,            0, masca_receive_diagnostic },
	{CMD_GET_EVT_STAT,		    0, masca_get_evt_notification },
	{READ_SUBCHANNEL,                   0, masca_read_subchannel },
	{DIAG_GET_LASER_CURRENT,            MAX_LASER_CUR_SIZE, NULL },
	{DIAG_GET_SWITCH_STATUS,            MAX_SWITCH_STATUS_SIZE, NULL },
	{DIAG_CMD_ADJUSTMENT,               MAX_DIAG_ADJUST_SIZE, NULL },
	{DIAG_CMD_CD_TEXT_OFF,              MAX_CMD_RECEIVE_SIZE, NULL },
	{DIAG_CMD_CD_TEXT_MODE2,            MAX_CMD_RECEIVE_SIZE, NULL },
	{DIAG_CMD_DATE_INSTALL_WRITE,       MAX_CMD_RECEIVE_SIZE, NULL },
	{DIAG_CMD_C1C2_ERR_COUNT_ON,        MAX_C1C2_ERR_CNT_SIZE, NULL },
	{DIAG_CMD_TIME_READOUT,             MAX_DIAG_TIME_READOUT_SIZE, NULL },
	{DIAG_CMD_EJECT_LOAD_CYCLES,        MAX_EJT_LD_CYC_SIZE, NULL },
	{DIAG_CMD_JUMP_COUNTER,             MAX_JUMP_CNT_SIZE, NULL },
	{DIAG_CMD_EEPROM_READOUT,           MAX_EEPROM_READOUT_SIZE, NULL },
	{DIAG_CMD_WRITE_STICKER,            MAX_WRITE_STICKER_SIZE, NULL },
	{DIAG_CMD_READ_STICKER,             MAX_READ_STICKER_SIZE, NULL },
	{DIAG_CMD_JITTER_MEASURE,           MAX_JITTER_MEASURE_SIZE, NULL },
	{DIAG_CMD_VERTICAL_DEV_MEASURE,     MAX_VER_DEV_SIZE, NULL },
	{DIAG_CMD_ECCENTRIC_MEASURE,        MAX_ECCENTRICITY_SIZE, NULL },
	{DIAG_CMD_REFLECTIVE_MEASURE,       MAX_REFLECTIVE_SIZE, NULL },
	{DIAG_CMD_INTERNAL_SELF_TEST,       MAX_INTER_SELF_SIZE, NULL },
	{DIAG_CMD_EXTERNAL_SDRAM_TEST,      MAX_EXTER_SDRAM_TEST_SIZE, NULL },
	{DIAG_CMD_INIT_LASER_CURRENT,       MAX_INTI_LASER_CUR_SIZE, NULL },
	{DIAG_CMD_NUM_READ_ERRS,            MAX_NO_READ_ERRS_SIZE, NULL },
	{DIAG_CMD_DATE_PRODUCTION_READ,     MAX_DATE_PROD_INSTALL_SIZE, NULL },
	{DIAG_CMD_DATE_INSTALL_READ,        MAX_DATE_PROD_INSTALL_SIZE, NULL },
	{DIAG_CMD_SERIAL_NO_READ,           MAX_SERIAL_NO_SIZE, NULL },
	{DIAG_CMD_STATUS_MSG_ERR_HISTROY,   MAX_MSG_ERR_HIS_SIZE, NULL },
	{DIAG_CMD_STATUS_MSG_LAST_ERRS,     MAX_MSG_LAST_ERRS_SIZE, NULL },
	{DIAG_CMD_USED_CD_TYPE_COUNT,       MAX_CD_TYPE_CNT_SIZE, NULL },
	{DIAG_CMD_USER_PROFILE_USED_CDS,    MAX_PROFIL_USED_CD_SIZE, NULL }
};

/* For Sense key implementation */
struct error_sense {
	enum masca_error err;
	struct masca_sense_info si;
};

/*!!!!!!!!!!!!!!!!!!!IMPORTANT!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
/*The order of the array has dependency in masca_common_types.h.   */
/*Refer to enum masca_error and adjust it accordingly.             */
/*!!!!!!!!!!!!!!!!!!!IMPORTANT!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/
struct error_sense err_sen_map[] = {
	{ MASCA_OK,			{NO_SENSE, 0x00, 0x00} },
	{ MASCA_SE_OVR_TEMP,		{RECOVERED_ERROR, 0x0B, 0x01} },
	{ MASCA_NR_UNREPORTABLE,	{NOT_READY, 0x04, 0x00} },
	{ MASCA_NR_INPROGRESS_OF_RDY,	{NOT_READY, 0x04, 0x01} },
	{ MASCA_NR_OPR_IN_PROGRESS,	{NOT_READY, 0x04, 0x07} },
	{ MASCA_NR_MEDIUM_INCOMPATIBLE,	{NOT_READY, 0x30, 0x00} },
	{ MASCA_NR_UNKNOWN_FORMAT,	{NOT_READY, 0x30, 0x01} },
	{ MASCA_NR_NO_MEDIUM,		{NOT_READY, 0x3A, 0x00} },
	{ MASCA_ME_IDCRC_CDROM,		{MEDIUM_ERROR, 0x10, 0x00} },
	{ MASCA_ME_READ_ERR_CDROM,	{MEDIUM_ERROR, 0x11, 0x00} },
	{ MASCA_ME_READ_RETRY_EXHAUST,	{MEDIUM_ERROR, 0x11, 0x01} },
	{ MASCA_ME_TOC_ERROR,		{MEDIUM_ERROR, 0x57, 0x00} },
	{ MASCA_HE_COMM_FAIL,		{HARDWARE_ERROR, 0x08, 0x00 } },
	{ MASCA_HE_DIAG_FAIL,		{HARDWARE_ERROR, 0x40, 0x00 } },
	{ MASCA_HE_UNSUCCESS_SFT_RST,	{HARDWARE_ERROR, 0x46, 0x00 } },
	{ MASCA_HE_LOAD_EJECT_FAIL,	{HARDWARE_ERROR, 0x53, 0x00 } },
	{ MASCA_HE_VOLTAGE_FAULT,	{HARDWARE_ERROR, 0x65, 0x00 } },
	{ MASCA_HE_SELF_TEST_FAIL,	{HARDWARE_ERROR, 0x3E, 0x03 } },
	{ MASCA_HE_TIMEOUT_ON_LOGUNIT,	{HARDWARE_ERROR, 0x3E, 0x02 } },
	{ MASCA_IR_PLY_PROGRESS,	{ILLEGAL_REQUEST, 0x00, 0x11} },
	{ MASCA_IR_PARAM_LISTLEN_ERR,	{ILLEGAL_REQUEST, 0x1A, 0x00} },
	{ MASCA_IR_CMD_NOT_SUPP,	{ILLEGAL_REQUEST, 0x20, 0x00} },
	{ MASCA_IR_LBA_OUT_OF_RANGE,	{ILLEGAL_REQUEST, 0x21, 0x00} },
	{ MASCA_IR_INVALID_FLD_IN_CDB,	{ILLEGAL_REQUEST, 0x24, 0x00} },
	{ MASCA_IR_INVALID_FLD_IN_LIST,	{ILLEGAL_REQUEST, 0x26, 0x00} },
	{ MASCA_IR_PARAM_NOT_SUPP,	{ILLEGAL_REQUEST, 0x26, 0x01} },
	{ MASCA_IR_PARAM_VAL_INVALID,	{ILLEGAL_REQUEST, 0x26, 0x02} },
	{ MASCA_IR_CMD_SEQ_ERR,		{ILLEGAL_REQUEST, 0x2C, 0x00} },
	{ MASCA_IR_NO_MEDIA_REMOVAL,	{ILLEGAL_REQUEST, 0x53, 0x02} },
	{ MASCA_IR_TRK_END_MSG,		{ILLEGAL_REQUEST, 0x63, 0x00} },
	{ MASCA_IR_ILLEGAL_TRK_MOD,	{ILLEGAL_REQUEST, 0x64, 0x00} },
	{ MASCA_UA_ERR_LOG_OVRFLW,	{UNIT_ATTENTION, 0x0A, 0x00} },
	{ MASCA_UA_POR,			{UNIT_ATTENTION, 0x29, 0x00} },
	{ MASCA_UA_DEV_RST,		{UNIT_ATTENTION, 0x29, 0x04} },
	{ MASCA_UA_PARAM_CNG,		{UNIT_ATTENTION, 0x2A, 0x00} },
	{ MASCA_UA_MODE_PARAM_CNG,	{UNIT_ATTENTION, 0x2A, 0x01} },
	{ MASCA_UA_DISK_END,		{UNIT_ATTENTION, 0x3B, 0x0f} },
	{ MASCA_UA_INQ_DATA_CNG,	{UNIT_ATTENTION, 0x3F, 0x03} },
	{ MASCA_UA_MED_REM_REQ,		{UNIT_ATTENTION, 0x5A, 0x01} },
	{ MASCA_UA_LOG_CTR_MAX,		{UNIT_ATTENTION, 0x5B, 0x02} },
	{ MASCA_UA_LOW_PWR_ON,		{UNIT_ATTENTION, 0x5E, 0x00} },
	{ MASCA_UA_IDLE_TMR,		{UNIT_ATTENTION, 0x5E, 0x01} },
	{ MASCA_UA_STDBY_TMR,		{UNIT_ATTENTION, 0x5E, 0x02} },
	{ MASCA_UA_IDLE_CMD,		{UNIT_ATTENTION, 0x5E, 0x03} },
	{ MASCA_UA_STDBY_CMD,		{UNIT_ATTENTION, 0x5E, 0x04} },
	{ MASCA_AC_READ_ERR_CDDA,	{ABORTED_COMMAND, 0x11, 0x11} }
};

/* Local function declaration */
static enum masca_error initialize_classes(const struct masca_configuration
						 *const p_config);
static void deinitialize_classes(void);
static void masca_ack_response(struct masca_output * const p_output);

void masca_set_recvr_waitcondition(unsigned char evt)
{
	unsigned long flags;

	spin_lock_irqsave(&receiver_lock, flags);
	waitcond |= evt;
	wake_up_interruptible(&waitq);
	spin_unlock_irqrestore(&receiver_lock, flags);
}

void masca_clr_recvr_waitcondition(unsigned char evt)
{
	unsigned long flags;

	spin_lock_irqsave(&receiver_lock, flags);
	waitcond &= ~evt;
	spin_unlock_irqrestore(&receiver_lock, flags);
}

unsigned char masca_get_recvr_waitcondition(void)
{
	unsigned long flags;
	unsigned char evt;

	spin_lock_irqsave(&receiver_lock, flags);
	evt = waitcond;
	spin_unlock_irqrestore(&receiver_lock, flags);

	return evt;
}

void masca_init_rcvr_queue(void)
{
	unsigned long flags;
	spin_lock_irqsave(&rcvrq_lock, flags);
	masca_rcvr_queue.evtflag = -1;
	INIT_LIST_HEAD(&(masca_rcvr_queue.list));
	spin_unlock_irqrestore(&rcvrq_lock, flags);
}

void masca_util_set_event(const unsigned int evt_pattern)
{
	int localevtpattern;
	/* Same function is used to add commands and events to
	 * processing queue, To satisfy parameter 1 NULL was passed
	 * in case of events. */
	masca_add_rcvr_que(NULL, true);
	localevtpattern = atomic_read(&event_pattern);
	atomic_set(&event_pattern, localevtpattern | evt_pattern);
	masca_set_recvr_waitcondition(MASCA_STATMSG);
}

void masca_util_clr_event(const unsigned int evt_pattern)
{
	int localevtpattern;
	localevtpattern = atomic_read(&event_pattern);
	atomic_set(&event_pattern, localevtpattern & (~evt_pattern));
}

int masca_util_get_event(void)
{
	return atomic_read(&event_pattern);
}

void masca_add_rcvr_que(struct scsi_cmnd *scmdp, bool evtflag)
{
	unsigned long flags;
	struct masca_rcvr *cmd_evt = kzalloc(sizeof(struct masca_rcvr),
								GFP_ATOMIC);
	if (NULL != cmd_evt) {
		cmd_evt->evtflag = evtflag;
		cmd_evt->a_cmnd  = scmdp;
		cmd_evt->is_cmd_done = false;
		spin_lock_irqsave(&rcvrq_lock, flags);
		if (false == evtflag)
			cmd_evt->replyid = scmd_repid++;
		list_add_tail(&(cmd_evt->list), &(masca_rcvr_queue.list));
		spin_unlock_irqrestore(&rcvrq_lock, flags);
		SCSI_HND_TD_TRACE("%s: New Entry to process\n", __func__);
	} else {
		SCSI_HND_TD_TRACE(
			"Failed : SCSI command not added in queue\n");
	}
}

void masca_rmv_rcvr_que(struct masca_rcvr *cmd_evt)
{
	unsigned long flags;
	spin_lock_irqsave(&rcvrq_lock, flags);
	if ((NULL != cmd_evt) && (!list_empty(&(cmd_evt->list)))) {
		list_del(&(cmd_evt->list));
		kfree(cmd_evt);
		cmd_evt = NULL;
	}
	spin_unlock_irqrestore(&rcvrq_lock, flags);
}

bool masca_is_rcvrqempty(void)
{
	unsigned long flags;
	bool isempty = false;
	spin_lock_irqsave(&rcvrq_lock, flags);
	isempty = list_empty(&masca_rcvr_queue.list);
	spin_unlock_irqrestore(&rcvrq_lock, flags);
	return isempty;
}

void masca_clear_rcvrq(void)
{
	unsigned long flags;
	struct masca_rcvr *rcvr;
	struct list_head *pos, *temp;
	if (!masca_is_rcvrqempty()) {
		spin_lock_irqsave(&rcvrq_lock, flags);
		list_for_each_safe(pos, temp, &(masca_rcvr_queue.list)) {
		rcvr = list_entry(pos, struct masca_rcvr, list);
		list_del(&(rcvr->list));
		kfree(rcvr);
		rcvr = NULL;
		}
		spin_unlock_irqrestore(&rcvrq_lock, flags);
	}
}
struct masca_rcvr *masca_get_aborted_cmd_from_que(struct scsi_cmnd *scpnt)
{
	struct masca_rcvr *cmd_msg = NULL;
	struct list_head *pos, *temp;
	unsigned long flags;
	if (!masca_is_rcvrqempty()) {
		spin_lock_irqsave(&rcvrq_lock, flags);
		list_for_each_safe(pos, temp, &(masca_rcvr_queue.list)) {
			cmd_msg = list_entry(pos, struct masca_rcvr, list);
			if (cmd_msg->a_cmnd == scpnt)
				break;
			cmd_msg = NULL;
		}
		spin_unlock_irqrestore(&rcvrq_lock, flags);
	}
	return cmd_msg;
}

/* This function will always return NULL if
   Queue is empty or if all entries are processed */
struct masca_rcvr *masca_get_rcvr_que(void)
{
	struct masca_rcvr *cmd_msg = NULL;
	struct list_head *pos, *temp;
	unsigned long flags;
	if (!masca_is_rcvrqempty()) {
		spin_lock_irqsave(&rcvrq_lock, flags);
		list_for_each_safe(pos, temp, &(masca_rcvr_queue.list)) {
			cmd_msg = list_entry(pos, struct masca_rcvr, list);
			if (false == cmd_msg->is_cmd_done)
				break;
			cmd_msg = NULL;
		}
		spin_unlock_irqrestore(&rcvrq_lock, flags);
	}

	return cmd_msg;
}

struct masca_rcvr *masca_get_rcvr_que_indx(int repid)
{
	struct masca_rcvr *cmd_msg = NULL;
	struct list_head *pos, *temp;
	unsigned long flags;
	if (!masca_is_rcvrqempty()) {
		spin_lock_irqsave(&rcvrq_lock, flags);
		list_for_each_safe(pos, temp, &(masca_rcvr_queue.list)) {
			cmd_msg = list_entry(pos, struct masca_rcvr, list);
			if (repid == cmd_msg->replyid)
				break;
			cmd_msg = NULL;
		}
		spin_unlock_irqrestore(&rcvrq_lock, flags);
	}

	return cmd_msg;
}

static enum masca_error initialize_classes(const struct masca_configuration *
							const p_config)
{
	enum masca_error error = MASCA_OK;
	signed char count;
	void (*const init_container[MAX_MASCA_CLASSES])
	(const struct masca_configuration * const p_config)  = {
	masca_intrptr_init, masca_drv_mngr_init,
	masca_sm_init, masca_cmd_hndlr_init, masca_diag_hndlr_init };

	masca_helper_init();
	error = masca_blkrdr_init(p_config);
	if (MASCA_OK == error) {
		for (count = 0; count < MAX_MASCA_CLASSES; count++)
			init_container[count](p_config);
	}
	return error;
}

static void deinitialize_classes(void)
{
	masca_helper_deinit();
	masca_blkrdr_deinit();
	masca_intrptr_deinit();
	masca_drv_mngr_deinit();
	masca_cmd_hndlr_deinit();
}

int masca_core_init(void)
{
	enum masca_error error = MASCA_IO_ERR;

	masca_read_configuration(&masca_config);
	masca_config.p_cmd_hndlr_lst = kcalloc((unsigned int)
					masca_config.pllel_maxreq,
					sizeof(struct masca_cmd_lst_elem),
					GFP_KERNEL);
	if (NULL == masca_config.p_cmd_hndlr_lst)
		goto cmd_hndlr_list_error;

	masca_config.p_diag_hndlr_lst = kcalloc((unsigned int)
					masca_config.pllel_maxreq,
					sizeof(struct masca_cmd_lst_elem),
					GFP_KERNEL);
	if (NULL == masca_config.p_diag_hndlr_lst)
		goto diag_hndlr_list_error;

	g_app_buf = vzalloc(APP_BUF_SZ);
	if (NULL == g_app_buf)
		goto g_app_buf_init_error;

	masca_init_rcvr_queue();

	error = initialize_classes(&masca_config);
	if ((initialize_masca_receiver()) && (MASCA_OK != error)) {
		SCSI_HND_TA_TRACE("Error: masca receiver init failed\n");
		error = MASCA_IO_ERR;
		goto masca_receiver_init_error;
	}
	masca_hal_rst_pulse();
	/* Manually checking the drive state */
	if ((TRUE == masca_check_req()) && (1 != masca_util_get_event()))
		masca_util_set_event(DRV_MNGR_RCV_MESSAGE);
	return error;

masca_receiver_init_error:
		vfree(g_app_buf);
g_app_buf_init_error:
		kfree(masca_config.p_diag_hndlr_lst);
diag_hndlr_list_error:
		kfree(masca_config.p_cmd_hndlr_lst);
cmd_hndlr_list_error:
		return error;
}

int masca_core_deinit(void)
{
	deinitialize_masca_receiver();
	deinitialize_classes();
	masca_clear_rcvrq();
	vfree(g_app_buf);
	kfree(masca_config.p_cmd_hndlr_lst);
	kfree(masca_config.p_diag_hndlr_lst);

	return 0;
}

int initialize_masca_receiver(void)
{
	recvthrd = kthread_run(masca_recvthrd, NULL, "masca_msghdlr");
	/* Thread Creation Error handling need to be changed */
	if (recvthrd == NULL) {
		SCSI_HND_TA_TRACE("MASCA Processor Thread Creation failed\n");
		return -ENOMEM;
	}
	return MASCA_OK;
}

int deinitialize_masca_receiver(void)
{
	masca_set_recvr_waitcondition(MASCA_DEINIT);
	kthread_stop(recvthrd);

	return MASCA_OK;
}

int masca_recvthrd(void *thr_data)
{
	struct masca_rcvr *rcv = NULL;

	while (!kthread_should_stop()) {
		if ((!masca_is_rcvrqempty()) || (0 == (wait_event_interruptible
					(waitq, (masca_get_recvr_waitcondition()
							& DATA_PRESENT))))) {
			SCSI_HND_TD_TRACE("MASCA: Got the next event ...\n");
			if (MASCA_DEINIT & masca_get_recvr_waitcondition()) {
				break;
			} else {
				rcv = masca_get_rcvr_que();
				if (NULL != rcv) {
					if (true == rcv->evtflag) {
						SCSI_HND_TD_TRACE(
						"masca status mesg received\n");
						masca_clr_recvr_waitcondition
								(MASCA_STATMSG);
						process_statmsg();
						rcv->is_cmd_done = true;
						masca_rmv_rcvr_que(rcv);
					} else {
						SCSI_HND_TD_TRACE(
						"masca user cmd received\n");
						masca_handle_scsi_cmd(rcv);
						masca_clr_recvr_waitcondition
								(MASCA_COMMAND);
					}
				} else {
					/* This means we have allprocessed
					 * entry */
					wait_event_interruptible(waitq,
					(DATA_PRESENT &
					masca_get_recvr_waitcondition()));
				}
			}
		}
	}
	SCSI_HND_TD_TRACE("%s exited\n", __func__);
	return 0;
}

void masca_handle_scsi_cmd(struct masca_rcvr *rcvr)
{
	struct masca_output output;
	struct scsi_cmnd *scsicmd = NULL;
	enum masca_error ret_err;
	struct masca_cmd_params cmd;

	memset(&cmd, 0, sizeof(struct masca_cmd_params));
	scsicmd = rcvr->a_cmnd;
	rcvr->is_cmd_done = true;
	cmd.reply_id = rcvr->replyid;
	output.cmd_len = scsicmd->sdb.length;
	output.cmdbuf = (unsigned char *) scsicmd->cmnd;
	ret_err = masca_atapi_decode(&output, &cmd);
	if (MASCA_OK == ret_err) {
		masca_dispatch_cmd(&cmd, &output);
	} else {
		SCSI_HND_TE_TRACE(
		"%s: MASCA_atapi_decode failed: %d\n", __func__, ret_err);
		output.reply_status = ret_err;
		output.replied_command = INVALID_CMD;
		output.response_id = rcvr->replyid;
	}
	masca_ack_response(&output);
}

void process_statmsg(void)
{
	int flg_pattern = 0;
	int single_pattern = 0;
	char shift_count = 0;

	flg_pattern =  masca_util_get_event();
	masca_util_clr_event(flg_pattern);

	/* process the priority based flags */
	if ((flg_pattern & SM_EVENT_SUSPEND) != 0) {
		process_event(SM_EVENT_SUSPEND);
		flg_pattern &= ~(SM_EVENT_SUSPEND);
	}
	if ((flg_pattern & SM_EVENT_RESUME) != 0) {
		process_event(SM_EVENT_RESUME);
		flg_pattern &= ~SM_EVENT_RESUME;
	}
	/* Process still all events get over */
	while (0 != flg_pattern) {
		single_pattern = (flg_pattern & (1 << shift_count));
		if (0 != single_pattern) {
			process_event(single_pattern);
			flg_pattern &= ~single_pattern;
		}
		shift_count++;
	}
}

enum masca_error func_conv(unsigned char cmd_replied, const struct masca_output
				 *const p_output, unsigned char *p_buf,
				 unsigned int buf_size)
{
	unsigned char count = 0;
	unsigned char *p_out;
	enum masca_error ret_error = MASCA_OK;
	bool is_cmd_found = FALSE;
	/*now search for the command from the configured table*/
	while ((count < MAX_CONV_CMD) && (FALSE == is_cmd_found)) {
		if (cmd_replied == atapi_conv_decoder[count].conv_cmd) {
			if (count <= 11) {
				ret_error =
				atapi_conv_decoder[count].cmd_decode_function
						(p_output, p_buf, buf_size);
			} else {
				if (buf_size >=
					atapi_conv_decoder[count].size) {
					p_out = (unsigned char *)&p_output->
						output_params.cd_diag_params;
					memcpy(p_buf, p_out,
					atapi_conv_decoder[count].size);
				} else {
					ret_error = MASCA_OUT_OF_BOUND;
				}
			}
			is_cmd_found = TRUE;
		}
		count++;
	}

	return ret_error;
}

#define ALLOC_LEN_IN_3RD_BYTE	3	/* Allocation length byte in cbd[3] */
#define ALLOC_LEN_IN_7TH_BYTE	7	/* Allocation length byte in cbd[7] */
/* calculate allocated buffer size from cbd */
int masca_calculate_allocation_length(
				const struct masca_output * const p_output,
				const struct scsi_cmnd *scsicmd)
{
	int buf_size = 0;
	unsigned short i = 0;
	char *cmdbuf = scsicmd->cmnd;

	if ((p_output->replied_command >= PLAY_LBA) &&
		(p_output->replied_command <= DIAG_GET_TEMPERATURE)) {
		if ((p_output->replied_command == CMD_SEND_DIAGNOSTIC) ||
		(p_output->replied_command == CMD_RECEIVE_DIAGNOSTIC) ||
		(p_output->replied_command == GET_VERSION_INFO)) {
			buf_size = (cmdbuf[ALLOC_LEN_IN_3RD_BYTE]
							<< BYTE_2_START) +
				cmdbuf[ALLOC_LEN_IN_3RD_BYTE + 0x01];
		} else if ((p_output->replied_command == GET_CD_CAPACITY) ||
			(p_output->replied_command == READ_SECTORS)) {
			buf_size = scsicmd->sdb.length;
		} else if (!(((p_output->replied_command >= PLAY_LBA) &&
			(p_output->replied_command <= RESET_DRIVE)) ||
			((p_output->replied_command >= READ_SECTORS_HASH) &&
			(p_output->replied_command <= GET_TEST_UNIT_READY)))) {
			buf_size = (cmdbuf[ALLOC_LEN_IN_7TH_BYTE]
							<< BYTE_2_START) +
					cmdbuf[ALLOC_LEN_IN_7TH_BYTE + 0x01];
		} else {
			/* For all other commands buf_size will be zero */
		}
	}

	for (i = 0; (i < MAX_CONV_CMD) && (buf_size > 0); i++) {
		if (atapi_conv_decoder[i].conv_cmd == p_output->replied_command)
			if ((buf_size > atapi_conv_decoder[i].size)
					&& (atapi_conv_decoder[i].size != 0)) {
				buf_size = atapi_conv_decoder[i].size;
				break;
			}
	}

	return buf_size;
}
enum masca_error masca_reply_cmddata(const struct masca_output * const p_output)
{
	enum masca_cmd		command_replied;
	enum masca_error	error = MASCA_CHECK_CONDITION;
	unsigned int		buf_size = 0;
	unsigned char		*p_buf = NULL;
	char			*cmd = NULL;
	struct scsi_cmnd	*scsicmd = NULL;
	struct masca_rcvr	*qcmd = masca_get_rcvr_que_indx(
							p_output->response_id);

	if (NULL != qcmd) {
		scsicmd = qcmd->a_cmnd;
		command_replied = p_output->replied_command;
		cmd = scsicmd->cmnd;
		buf_size = masca_calculate_allocation_length(p_output,
								scsicmd);
		SCSI_HND_TD_TRACE("masca_reply_cmddata: buf_size: %d\n",
				buf_size);

		/* TODO: READ_SECTORS reply handling */
		if (command_replied == READ_SECTORS) {
			SCSI_HND_TD_TRACE("rd sect resp buf_size=%d\n",
								buf_size);
			masca_fill_scsi_buffer(scsicmd, g_app_buf, buf_size);
			scsicmd->result = 0; /*cmd execution success*/
			error = MASCA_OK;
		} else {
			p_buf = kzalloc(buf_size, GFP_KERNEL);
			if (NULL != p_buf) {
				error = func_conv(command_replied, p_output,
						p_buf, buf_size);
				if (MASCA_OK == error) {
					masca_fill_scsi_buffer(scsicmd, p_buf,
							buf_size);
					/*cmd execution success*/
					scsicmd->result = 0;
				}
				kfree(p_buf);
			} else {
				/* further error handling need to be added */
				error = MASCA_IR_INVALID_FLD_IN_CDB;
				SCSI_HND_TA_TRACE("failed alloc :size=0x%x\n",
							buf_size);
			}
		}
	}
	return error;
}

struct masca_sense_info masca_get_sense_value(int err)
{
	struct masca_sense_info seninfo = sense_buffer.sense_info;

	SCSI_HND_TD_TRACE(
		"Sen_err INDEX : %d\t key :0x%x\t asc: 0x%x\t ascq: 0x%x\n",
		err, err_sen_map[err].si.key, err_sen_map[err].si.asc,
		err_sen_map[err].si.ascq);
	/* If with in range retrive from mapped table otherwise return
	 * global previous error */
	if ((err >= MASCA_OK) && (err <= MASCA_AC_READ_ERR_CDDA))
		seninfo = err_sen_map[err].si;
	return seninfo;
}

#define FIXED_FORMAT	0
#define MASCA_SENSE_LEN 32
void masca_fill_sense_buffer(struct scsi_cmnd *scp, struct masca_sense_info si)
{
	unsigned char sbuff[MASCA_SENSE_LEN];
	memset(sbuff, 0, MASCA_SENSE_LEN);
	scsi_build_sense_buffer(FIXED_FORMAT, sbuff, si.key, si.asc, si.ascq);
	memcpy(scp->sense_buffer, sbuff,
	       (SCSI_SENSE_BUFFERSIZE > MASCA_SENSE_LEN) ?
	       MASCA_SENSE_LEN : SCSI_SENSE_BUFFERSIZE);
	/* No response data transfer as this is error case */
	scsi_set_resid(scp, scsi_bufflen(scp));
}

void masca_update_global_sense_buffer(char key, char asc, char ascq)
{
	sense_buffer.sense_info.key = key;
	sense_buffer.sense_info.asc = asc;
	sense_buffer.sense_info.ascq = ascq;
	sense_buffer.is_sense_buf_free = false;
}

static void masca_reply_cmdsense(const struct masca_output *const p_output)
{
	struct masca_sense_info sense_buf;
	struct scsi_cmnd *scsicmd = NULL;
	struct masca_rcvr *qcmd = masca_get_rcvr_que_indx(
							p_output->response_id);
	if (NULL != qcmd) {
		scsicmd = qcmd->a_cmnd;
		sense_buf = masca_get_sense_value(p_output->reply_status);
		masca_update_global_sense_buffer(sense_buf.key,
			sense_buf.asc, sense_buf.ascq);
		masca_fill_sense_buffer(scsicmd, sense_buf);
		scsicmd->result = SAM_STAT_CHECK_CONDITION;
	}
}

static void masca_ack_response(struct masca_output * const p_output)
{
	enum masca_error     error = MASCA_OK;
	struct scsi_cmnd *scsicmd = NULL;
	struct masca_rcvr *qcmd = NULL;

	if (MAX_COMMAND != p_output->replied_command) {
		error = p_output->reply_status;
		if ((MASCA_OK == error) || (MASCA_PROCESSED == error)) {
			error = masca_reply_cmddata(p_output);
			p_output->reply_status = error;
		}
		if (MASCA_OK != error)
			masca_reply_cmdsense(p_output);
		qcmd = masca_get_rcvr_que_indx(p_output->response_id);
		if (NULL != qcmd) {
			scsicmd = qcmd->a_cmnd;
			scsicmd->scsi_done(scsicmd);
			masca_rmv_rcvr_que(qcmd);
		}
	}
}

void process_event(int single_event_pattern)
{
	struct masca_event_info out_event;
	enum masca_media_type disc_type;
	struct masca_output     command_output;
	struct masca_evt_info   event_to_send;
	enum masca_error      error;
	bool            is_auto_reinsert;
	bool            is_same_cd;
	bool            is_dev_mounted;

	command_output.replied_command = MAX_COMMAND;
	out_event.evt = NO_EVENT;
	error = masca_dispatch_event(single_event_pattern, &out_event,
							 &command_output);

	if (error >= MASCA_OK) {
		event_to_send.cd_hash_id = 0;
		event_to_send.media_type = MASCA_MAX_MEDIA;
		event_to_send.info = 0;
		event_to_send.masca_event = MASCA_NO_EVT;
		switch (out_event.evt) {
		case CD_ABSENT:
			event_to_send.masca_event = MASCA_NO_CD;
			masca_update_evt_stat(MASCA_MEDIA_REMOVAL);
			masca_clr_async_sense(SENSE_LOAD_ERR);
			memset(&out_event_global, 0,
					sizeof(struct masca_event_info));
		break;

		case CD_INSERTING:
		case CD_AUTO_REINSERTING:
			event_to_send.masca_event = MASCA_CD_INSERTING;
			masca_update_evt_stat(MASCA_MEDIA_CHANGED);
			masca_clr_async_sense(SENSE_LOAD_ERR);
		break;
		case CD_INSERTED:
			/*currently nothing needs to be done over here*/
			event_to_send.masca_event = MASCA_CD_INSERTED;
			masca_clr_async_sense(SENSE_LOAD_ERR);
			masca_update_evt_stat(MASCA_MEDCHG_INSERTED);
		break;

		case CD_LOADING:
			/*currently nothing needs to be done over here*/
			event_to_send.masca_event = MASCA_CD_LOADING;
		break;

		case CD_LOAD_ERROR:
			disc_type = out_event.event_info.cd_identify.cd_type;
			if (DISC_BAD_MEDIA == disc_type) {
				event_to_send.media_type = MASCA_CD_BAD_MEDIA;
				event_to_send.masca_event = MASCA_CD_TOC_ERROR;
			}
			masca_set_async_sense(SENSE_TOC_ERR);
		break;

		case CD_ACCESSIBLE:
			is_auto_reinsert = out_event.event_info.cd_identify.
								auto_reinsert;
			is_same_cd = out_event.event_info.cd_identify.
							same_as_previous;
			if (TRUE == is_auto_reinsert) {
				event_to_send.masca_event =
							 MASCA_TDE_AUTO_MOUNT;
			}
			masca_update_evt_stat(MASCA_NEW_MEDIA);
		break;

		case CD_EJECTING:
			event_to_send.masca_event = MASCA_CD_EJECTING;
			is_dev_mounted = out_event.event_info.cd_identify.
			was_cd_accessible;
			/* reset audio ignore bit */
			masca_set_last_cmd_ign_bit(FALSE);
			if (TRUE == is_dev_mounted) {
				/* TODO: handle if(TRUE == is_dev_open) and
				 * corresponding else also.add the below line
				 * before using is_dev_open.
				 * static bool is_dev_open = FALSE; */
			}
			masca_update_evt_stat(MASCA_EJECT_REQ);
			masca_clr_async_sense(SENSE_TOC_ERR
					|SENSE_UNKNOWNFORM_ERR);
		break;

		case CD_IN_SLOT:
			event_to_send.masca_event = MASCA_CD_IN_SLOT;
			masca_update_evt_stat(MASCA_MEDIA_REM_SLT);
			masca_clr_async_sense(SENSE_LOAD_ERR);
			memset(&out_event_global, 0,
					sizeof(struct masca_event_info));
		break;

		case CD_DEFECTIVE:
			event_to_send.masca_event = MASCA_CD_DEFECTIVE;
			out_event_global.evt = CD_DEFECTIVE;
			masca_set_async_sense(SENSE_UNKNOWNFORM_ERR);
		break;

		case CD_MECHA_LOAD_ERR:
			event_to_send.masca_event = MASCA_MECHA_LOAD_ERR;
			masca_set_async_sense(SENSE_LOAD_ERR);
		break;

		case CD_PLAY_INFO:
			event_to_send.masca_event = MASCA_CD_PLAY_POS;
			out_event_global.evt = NO_EVENT;
			if (out_event.event_info.play_info.abs_sec !=
				out_event_global.event_info.play_info.abs_sec) {
				memcpy(&out_event_global.event_info.play_info,
					&out_event.event_info.play_info,
					 sizeof(struct masca_drv_play_info));
				out_event_global.evt = CD_PLAY_INFO;
			}
		break;

		case CD_TRACK_END:
			event_to_send.masca_event = MASCA_CD_TRACK_END;
		break;

		case CD_END:
			event_to_send.masca_event = MASCA_CD_END;
			out_event_global.evt = CD_PLAY_COMPLETE;
		break;

		case CD_PLAY_COMPLETE:
			event_to_send.masca_event = MASCA_CD_PLAY_COMPLETE;
			out_event_global.evt = CD_PLAY_COMPLETE;
		break;

		case DRIVE_OVER_TEMPERATURE:
			event_to_send.masca_event = MASCA_DRIVE_OVER_TEMP;
			masca_set_async_sense(SENSE_TEMP_ERR);
		break;

		case DRIVE_NORMAL_TEMPERATURE:
			event_to_send.masca_event = MASCA_DRIVE_NORMAL_TEMP;
			masca_clr_async_sense(SENSE_TEMP_ERR);
		break;

		case DRIVE_UNDER_VOLT:
			disc_type = out_event.event_info.cd_identify.cd_type;
			event_to_send.masca_event = MASCA_DRIVE_UNDER_VOLT;
			masca_set_async_sense(SENSE_VOLT_ERR);
		break;

		case DRIVE_NORMAL_VOLT:
		disc_type = out_event.event_info.cd_identify.cd_type;
		event_to_send.masca_event = MASCA_DRIVE_NORMAL_VOLT;
			masca_clr_async_sense(SENSE_VOLT_ERR);
		break;
		default:
		break;
		}

		if (event_to_send.masca_event != MASCA_NO_EVT) {
			/*Update these informations always as they might
			be needed by application*/
			event_to_send.media_type = (enum masca_media)
			out_event.event_info.cd_identify.cd_type;
			event_to_send.cd_hash_id =
			out_event.event_info.cd_identify.unique_id;
			/* TODO: POST EVENT TO USER HERE */
		}
		/*look up device request using reponse int
		from the queue and provide GDI_reply*/
		masca_ack_response(&command_output);
	}
}
