/*
 * MASCA Transmission and Reception Messages Handler
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"
#include "masca_hal.h"
#include "masca_sm.h"

struct masca_tx_rx_unit {
	enum masca_intrptr_cmd	cmd_from_app;
	struct masca_intrptr_cmd_info	cmd_sent;
	unsigned int			rty_cnt;
	unsigned int			current_rty_cnt;
	unsigned int			rst_cnt;
	unsigned int			current_rst_cnt;
};

#define IGN_AUDIO_BIT		8

static enum masca_error
masca_chk_unit_response(
		const struct masca_tx_rx_unit * const p_unit,
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds);

static struct masca_tx_rx_unit masca_cmd_unit[SLOT_MAX_NUM];


void masca_init_tx_rx(const enum masca_slot_num slot_no,
			const unsigned int retry_cnt,
			const unsigned int reset_cnt)
{
	struct masca_tx_rx_unit *p_unit;

	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		p_unit->cmd_sent.command	= CMD_MAX;
		p_unit->cmd_sent.dir_number	= 0;
		p_unit->cmd_sent.block_count	= 0;
		p_unit->cmd_sent.track_number	= 0;
		p_unit->cmd_from_app		= CMD_MAX;
		p_unit->rst_cnt			= reset_cnt;
		p_unit->current_rst_cnt		= reset_cnt;
		p_unit->rty_cnt			= retry_cnt;
		p_unit->current_rty_cnt		= retry_cnt;
	}
}

extern void masca_send_cmd(const enum masca_slot_num slot_no,
				const enum masca_intrptr_cmd cmd_to_drive,
				const unsigned int play_pos,
				const unsigned char trk_no,
				const unsigned char blk_cnt)
{
	struct masca_tx_rx_unit	*p_unit;
	struct masca_last_evt		last_evt;
	struct masca_intrptr_cmd_info	status_req;

	DRV_MNGR_TD_TRACE("%s\n", __func__);

	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		p_unit->cmd_from_app = cmd_to_drive;
		p_unit->cmd_sent.track_number = trk_no;
		p_unit->cmd_sent.block_count = blk_cnt;
		if ((CMD_SEARCH_FWD <= cmd_to_drive)
			&& (CMD_FAST_SEARCH_BWD >= cmd_to_drive))
			/*MASCA hardware does not support scanning from
			a specified position, so do a play to that position
			and then do scan after receiving response to play*/
			p_unit->cmd_sent.command = CMD_PLAY_CONTINUOUS;
		else
			p_unit->cmd_sent.command = p_unit->cmd_from_app;

		masca_get_msf(play_pos, &p_unit->cmd_sent.minute,
			&p_unit->cmd_sent.second, &p_unit->cmd_sent.frame);
		masca_tx_cmd(&p_unit->cmd_sent);

		masca_get_last_event(&last_evt);

		/*NOTES on hardware: As per masca specifications, the drive does
		* not respond for every drive control command unless it causes
		* a state change. So if we see disc is already playing and
		* issuing one more play command will not result in a response.
		* In this case issue a status command to get the current status
		* of the drive so that response is located accordingly*/

		if ((drv_responses[p_unit->cmd_sent.command]
				.drive_message[0] == last_evt.disc_event)
			|| (drv_responses[p_unit->cmd_sent.command]
				.drive_message[1] == last_evt.disc_event)) {
			status_req.command = CMD_SYS_STAT;
			masca_tx_cmd(&status_req);
		}
	}
}

extern void masca_send_diag_cmd(
				const enum masca_slot_num slot_no,
				const enum masca_intrptr_cmd cmd_to_drive,
				const unsigned int inst_date_write,
				unsigned char * const p_write_sticker)
{
	struct masca_tx_rx_unit	*p_unit;
	struct masca_last_evt		last_evt;

	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		p_unit->cmd_from_app = cmd_to_drive;

		if (CMD_WRITE_STICKER == cmd_to_drive)
			memcpy(p_unit->cmd_sent.date_write_sticker,
				p_write_sticker, DIAG_CMD_WIRTE_STICKER);
		else if (cmd_to_drive == CMD_DATE_INSTALL_WRITE)
			p_unit->cmd_sent.instll_dat_write = inst_date_write;

		p_unit->cmd_sent.command = p_unit->cmd_from_app;

		masca_tx_cmd(&p_unit->cmd_sent);

		masca_get_last_event(&last_evt);

	}
}

enum masca_error
masca_rcv_response(
		const enum masca_slot_num slot_no,
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds)
{
	struct masca_tx_rx_unit	*p_unit;
	enum masca_error		ret_err = MASCA_OK;
	enum masca_intrptr_cmd	cmd_to_drive;
	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		cmd_to_drive = p_unit->cmd_sent.command;
		if (CMD_MAX > cmd_to_drive) {
			ret_err = masca_chk_unit_response(p_unit, p_list,
								p_rjt_cmds);
			if (MASCA_PROCESSED == ret_err) {
				if ((CMD_SEARCH_FWD <= p_unit->cmd_from_app)
				&& (CMD_FAST_SEARCH_BWD >= p_unit->cmd_from_app)
				&& (CMD_PLAY_CONTINUOUS == cmd_to_drive)) {
					/*MASCA hardware does not support
					scanning from a specified position,
					so do a play to that position
					and then do scan after receiving
					response to play*/
					p_unit->cmd_sent.command =
							p_unit->cmd_from_app;
					masca_tx_cmd(&p_unit->cmd_sent);
					ret_err = MASCA_OK;
				} else {
					p_unit->cmd_sent.command = CMD_MAX;
					p_unit->cmd_from_app = CMD_MAX;
				}
			} else if (MASCA_REJECTED == ret_err) {
				p_unit->cmd_sent.command = CMD_MAX;
				p_unit->cmd_from_app = CMD_MAX;
			}
		}
	}
	return ret_err;
}

enum masca_error masca_evt_tmout(const enum masca_slot_num slot_no,
				bool * const p_is_rst_cnt_change)
{
	struct masca_tx_rx_unit	*p_unit;
	enum masca_error		ret_err = MASCA_OK;
	bool			is_rst_cnt_change = FALSE;
	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		if (CMD_MAX > p_unit->cmd_from_app) {
			if (p_unit->current_rty_cnt > 0) {
				p_unit->current_rty_cnt--;
				/*Resend the command again*/
				masca_tx_cmd(&p_unit->cmd_sent);
			} else if (p_unit->current_rst_cnt > 0) {
				p_unit->current_rst_cnt--;
				p_unit->current_rty_cnt = p_unit->rty_cnt;
				/*rst will be done based on this return value*/
				is_rst_cnt_change = TRUE;
			} else {
				/*We have done enough tries, now we exit as
				* hardware is faulty*/
				ret_err = MASCA_HE_TIMEOUT_ON_LOGUNIT;
			}
		}
	}
		*p_is_rst_cnt_change = is_rst_cnt_change;

	return ret_err;
}


void masca_abort_unit(const enum masca_slot_num slot_no)
{
	struct masca_tx_rx_unit *p_unit;
	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		p_unit->cmd_sent.command = CMD_MAX;
		p_unit->cmd_from_app = CMD_MAX;
	}
}

void masca_reload_rty_cnt(const enum masca_slot_num slot_no)
{
	struct masca_tx_rx_unit *p_unit;
	if (SLOT_MAX_NUM > slot_no) {
		p_unit = &masca_cmd_unit[slot_no];
		p_unit->current_rst_cnt         = p_unit->rst_cnt;
		p_unit->current_rty_cnt         = p_unit->rty_cnt;
	}
}

static enum masca_error
masca_chk_unit_response(
		const struct masca_tx_rx_unit * const p_unit,
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds)
{
	unsigned char cmd_count;
	unsigned char reject_cmd_no;
	struct masca_msg_search	msg_search;
	bool msg_end = FALSE;
	enum masca_intrptr_cmd cmmd;
	enum masca_intrptr_decode_msg found_msg;
	enum masca_error ret_err = MASCA_OK;

		cmmd = p_unit->cmd_sent.command;
		if (NULL != p_rjt_cmds) {
			reject_cmd_no = p_rjt_cmds->num_of_rejected_cmds;
			for (cmd_count = 0; cmd_count < reject_cmd_no;
			     cmd_count++) {
				if (p_rjt_cmds->reject_cmds[cmd_count] == cmmd)
					ret_err = MASCA_REJECTED;
			}
		}
		if (MASCA_OK == ret_err) {
			/*now check for response to the command sent*/
			for (cmd_count = 0;
			    (cmd_count < MAX_MSG_PER_CMD) && (FALSE == msg_end);
			    cmd_count++) {
				msg_search.search_msg[cmd_count] =
				   drv_responses[cmmd].drive_message[cmd_count];
				if (MESSAGE_MAX == drv_responses[cmmd]
						.drive_message[cmd_count])
					msg_end = TRUE;
			}
			msg_search.num_messages = cmd_count;
			if (TRUE == masca_is_msg_prsnt(&msg_search, p_list,
				       &found_msg, drv_responses[cmmd].msg_rel))
				ret_err = MASCA_PROCESSED;
		}

	return ret_err;
}

void masca_tx_cmd(const struct masca_intrptr_cmd_info * const p_cmd)
{
	unsigned char cmd_stream[COMMAND_MAX_LENGTH];
	unsigned char stream_lng;
	enum masca_error err;
	enum masca_media_type cd_type;
	masca_sm_get_media_type(&cd_type);

	if (NULL != p_cmd)
		err = masca_intrptr_encode_cmd(p_cmd, cmd_stream,
							&stream_lng);
	else
		err = MASCA_PTR_NULL;

	if (cd_type == DISC_MIXED_MODE) {
		/* Special handling for mixed mode CD, read command will be
		 * returned with MASCA_BUSY without sending to drive
		 * during mode switching is in progress */
		if ((cmd_stream[COMMAND_ID_POS] == COMMAND_PLAY_ID) &&
			(p_cmd->block_count != 0) &&
			(IGN_AUD_ENABLE_TOC_READY !=
					masca_sm_get_ign_aud_status())) {
			err = MASCA_BUSY;
			DRV_MNGR_TD_TRACE(
				"DrvMngrTx:drive-busy on mixed mode switching");
		}
	}

	if (MASCA_OK == err) {
		if ((TRUE == masca_get_last_cmd_ign_bit()) &&
			(COMMAND_STATUS_ID == cmd_stream[COMMAND_ID_POS]))
			/* add ignore_audio_bit for CDROM in mixed mode CD*/
			cmd_stream[COMMAND_IGN_AUDIO_POS] |= IGN_AUDIO_BIT;

		err = masca_prepare_sndbuf(cmd_stream, stream_lng);
		if (MASCA_OK > err) {
			/*trace out an error here as it should never happen*/
			/*possible a reset of drive is required*/
			DRV_MNGR_TE_TRACE(
			"DrvMngrTx:Msg handler snd cmd failed,err = %d\n", err);
		}
	} else {
		DRV_MNGR_TE_TRACE
		("DrvMngrTx: Intrptr encode cmd failed,err = %d\n", err);
	}
}


bool masca_is_msg_prsnt(const struct masca_msg_search * const p_msg_search,
			const struct masca_intrptr_msg_info_list * const p_list,
			enum masca_intrptr_decode_msg * const p_msg_intrptr,
			const enum masca_msg_relation msg_relation)
{
	bool is_msg_prsnt = FALSE;
	unsigned char count_msg;
	enum masca_intrptr_decode_msg low_msg;
	enum masca_intrptr_decode_msg high_msg;
	if ((p_msg_search->num_messages < MAX_SEARCH_MSGS)
		&& (p_msg_search->num_messages > 0)) {
		if (MASCA_AND == msg_relation) {
			is_msg_prsnt = TRUE;
			for (count_msg = 0;
				((count_msg < p_msg_search->num_messages) &&
				(TRUE == is_msg_prsnt)); count_msg++) {
				/*search for one message*/
				is_msg_prsnt = masca_chk_msg(
				p_list, p_msg_search->search_msg[count_msg]);
				if ((TRUE == is_msg_prsnt) &&
						(NULL != p_msg_intrptr))
					/*update the last found message*/
					*p_msg_intrptr =
					    p_msg_search->search_msg[count_msg];
			}
		} else if (MASCA_OR == msg_relation) {
			is_msg_prsnt = FALSE;
			for (count_msg = 0;
				((count_msg < p_msg_search->num_messages) &&
				(FALSE == is_msg_prsnt)); count_msg++) {
				is_msg_prsnt = masca_chk_msg(
				p_list, p_msg_search->search_msg[count_msg]);
				if ((TRUE == is_msg_prsnt) &&
						(NULL != p_msg_intrptr))
					/*update the first found message*/
					*p_msg_intrptr =
					    p_msg_search->search_msg[count_msg];
			}
		} else if (MASCA_RANGE == msg_relation) {
			/*valid only when two messages are passed and
			any of the messages in between are asked for presence*/
			if ((RANGE_COUNT == p_msg_search->num_messages)
					&& (NULL != p_list)) {
				is_msg_prsnt = FALSE;
				low_msg = p_msg_search->search_msg[0];
				high_msg = p_msg_search->search_msg[1];
				for (count_msg = 0;
					((count_msg < p_list->num_of_msgs) &&
					(FALSE == is_msg_prsnt)); count_msg++) {
					if ((low_msg <=
					    p_list->decoded_messages[count_msg])
					    && (high_msg >=
					    p_list->decoded_messages[count_msg])
									) {
						is_msg_prsnt = TRUE;
						/*update the 1st found message*/
						*p_msg_intrptr = p_list->
						    decoded_messages[count_msg];
					}
				}
			}
		}
	}
	return is_msg_prsnt;
}


bool masca_chk_msg(const struct masca_intrptr_msg_info_list * const p_list,
			const enum masca_intrptr_decode_msg msg_intrptr)
{
	bool is_msg_prsnt = FALSE;
	unsigned char count_msg;
		for (count_msg = 0;
			((count_msg < p_list->num_of_msgs) &&
			(FALSE == is_msg_prsnt)); count_msg++) {
			if (msg_intrptr == p_list->decoded_messages[count_msg])
				is_msg_prsnt = TRUE;
		}
	return is_msg_prsnt;
}
