/*
 * MASCA TOC and CD Text Read File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"

static enum masca_error
masca_toctexthash_response(const struct masca_intrptr_msg_info * const p_msg,
					const enum masca_error	response);
static void
masca_fill_repsonse(struct masca_drv_response_info * const p_response_info);
static void
masca_exec_toctexthash(struct masca_drv_response_info * const p_response_info);

static struct masca_cmd_slot		toc_text_hash;
static struct masca_intrptr_sys_toc	cd_global_toc;
static unsigned char			cur_trk_num;
static struct masca_cd_toc		*p_cd_toc_info;
static enum masca_intrptr_cmd	last_cmd_to_drive;
static enum masca_intrptr_decode_msg	expected_response;

void masca_init_toc_text_hash(const unsigned int retry_cnt,
				const unsigned int reset_cnt,
				const unsigned int response_tmout)
{
	toc_text_hash.slot_stat = SLOT_PAUSED;
	toc_text_hash.app_cmd_info.command = MAX_COMMAND;
	toc_text_hash.hw_response = MASCA_OK;
	toc_text_hash.slot_tmout = response_tmout;
	last_cmd_to_drive = CMD_MAX;
	expected_response = MESSAGE_MAX;
	masca_init_tx_rx(SLOT_TOC_TEXT_HASH, retry_cnt, reset_cnt);
}

enum masca_cmd masca_get_toctexthash_cmd(void)
{
	return toc_text_hash.app_cmd_info.command;
}

void masca_toctexthash_alm_cbk(void)
{
	masca_util_set_event(SLOT_TOC_TEXT_HASH_TMOUT);
}

enum masca_error
masca_tx_toctexthash(const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	bool	exec_slot;
	enum masca_error ret_err = masca_fill_slot(&toc_text_hash, p_cmd_info,
								&exec_slot);

	if (MASCA_OK == ret_err) {
		/*reload the retry counters for the slot*/
		masca_reload_rty_cnt(SLOT_TOC_TEXT_HASH);
		masca_exec_toctexthash(p_response_info);
	}
	return ret_err;
}


void masca_rx_toctexthash(
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_response_info * const p_response_info)
{
	/*In the paused state also the responses from the hardware
	* are respected*/
	enum masca_error ret_err;
	if (SLOT_PAUSED != toc_text_hash.slot_stat) {
		/*Finding out a response is received or not is a generic
		* functionality. Hence it is implemented in a common place*/
		ret_err = masca_rcv_response(SLOT_TOC_TEXT_HASH, p_list,
								   p_rjt_cmds);
		if (MASCA_OK != ret_err) {
			ret_err = masca_toctexthash_response(p_msg, ret_err);
			if (MASCA_OK != ret_err) {
				toc_text_hash.hw_response = ret_err;
				/*Now we need to provide a reply back*/
				if (MAX_COMMAND ==
					   p_response_info->drv_cmd_response) {
					masca_fill_repsonse(p_response_info);
				} else {
					/*We got response to the slot, so clear
					  the timeout*/
					masca_util_stp_alm(ALM_TOCTEXTHASH);
					masca_util_clr_event(
						SLOT_TOC_TEXT_HASH_TMOUT);
					masca_util_set_event(
							SLOT_RESPONSE_UPDATE);
				}
			}
		}
	}
}


void masca_pause_toctexthash(void)
{
	toc_text_hash.slot_stat = SLOT_PAUSED;
	/*we are pausing so don't allow timeouts*/
	if (MAX_COMMAND > toc_text_hash.app_cmd_info.command) {
		masca_util_stp_alm(ALM_TOCTEXTHASH);
		masca_util_clr_event(SLOT_TOC_TEXT_HASH_TMOUT);
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_TOC_TEXT_HASH);
	}
}

void
masca_resume_toctexthash(struct masca_drv_response_info * const p_response_info)
{
	toc_text_hash.slot_stat = SLOT_IDLE;
	if (MAX_COMMAND > toc_text_hash.app_cmd_info.command) {
		/*start from the point where we left for cd info read*/
		toc_text_hash.slot_stat = SLOT_EXECUTING;
		masca_send_cmd(SLOT_TOC_TEXT_HASH, last_cmd_to_drive, 0,
								cur_trk_num, 0);
		masca_util_sta_alm(toc_text_hash.slot_tmout, ALM_TOCTEXTHASH);
	}
}

void
masca_abort_toctexthash(struct masca_drv_response_info * const p_response_info,
					enum masca_error	abort_reason)
{
	if (MAX_COMMAND > toc_text_hash.app_cmd_info.command) {
		toc_text_hash.hw_response = abort_reason;
		if (MAX_COMMAND == p_response_info->drv_cmd_response) {
			toc_text_hash.abort_slot = FALSE;
			masca_fill_repsonse(p_response_info);
		} else {
			/* set an abort update so that through event
			* interface of this slot we can abort the command*/
			toc_text_hash.abort_slot = TRUE;
			masca_util_set_event(SLOT_ABORT_UPDATE);
			masca_util_stp_alm(ALM_TOCTEXTHASH);
			masca_util_clr_event(SLOT_TOC_TEXT_HASH_TMOUT);
		}
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_TOC_TEXT_HASH);
	}
}


bool
masca_evt_toctexthash(unsigned int evt_ptn_slot,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_required = FALSE;
	enum masca_error  ret_err;
	if (SLOT_RESPONSE_UPDATE == evt_ptn_slot) {
		if (MASCA_OK != toc_text_hash.hw_response) {
			if (MAX_COMMAND == p_response_info->drv_cmd_response)
				masca_fill_repsonse(p_response_info);
			else
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
		}
	} else if (SLOT_ABORT_UPDATE == evt_ptn_slot) {
		if (TRUE == toc_text_hash.abort_slot)
			masca_abort_toctexthash(p_response_info,
						toc_text_hash.hw_response);
	} else if (SLOT_PAUSED != toc_text_hash.slot_stat) {
		if (SLOT_TOC_TEXT_HASH_TMOUT == evt_ptn_slot) {
			/*Now we need to do retries*/
			ret_err = masca_evt_tmout(SLOT_TOC_TEXT_HASH,
							&is_rst_required);
			if (MASCA_OK != ret_err) {
				toc_text_hash.hw_response = ret_err;
				masca_fill_repsonse(p_response_info);
			} else {
				masca_util_sta_alm(toc_text_hash.slot_tmout,
							ALM_TOCTEXTHASH);
			}
		}
	} else {
		/*In paused state no other events are entertained*/
	}
	return is_rst_required;
}

void masca_get_cd_toc(struct masca_cd_toc **const pp_toc)
{
	*pp_toc = p_cd_toc_info;
}

/* Calculate track numbers for audio and data tracks */
static void masca_track_nos_calc(struct masca_cd_toc *p_toc)
{
	unsigned char	trk_num = 0;
	unsigned char	trk_addr_ctrl = 0;
	/* MASCA drive specification */
	const unsigned char	data_track_bit = 0x40;

	p_toc->first_audio_trk_num = 0;
	p_toc->last_audio_trk_num = 0;
	/* Extract first and last audio track number's
	* form complete TOC info */
	for (trk_num = cd_global_toc.minimum_track_num;
		trk_num <= cd_global_toc.maximum_track_num;
		trk_num++) {
		trk_addr_ctrl = p_toc->track_info[trk_num].adr_ctrl;

		if ((trk_addr_ctrl & data_track_bit) != data_track_bit) {
			p_toc->last_audio_trk_num = trk_num;
			if (p_toc->first_audio_trk_num == 0)
				p_toc->first_audio_trk_num = trk_num;
		}
	}

	p_toc->first_data_trk_num = 0;
	p_toc->last_data_trk_num = 0;
	/* Extract first and last data track number's from complete
	* TOC info */
	for (trk_num = cd_global_toc.minimum_track_num;
			trk_num <= cd_global_toc.maximum_track_num;
			trk_num++) {
		trk_addr_ctrl = p_toc->track_info[trk_num].adr_ctrl;

		if ((trk_addr_ctrl & data_track_bit) == data_track_bit) {
			p_toc->last_data_trk_num = trk_num;
			if (p_toc->first_data_trk_num == 0)
				p_toc->first_data_trk_num = trk_num;
		}
	}
}

static enum masca_error
masca_toctexthash_response(const struct masca_intrptr_msg_info * const p_msg,
				const enum masca_error	response)
{
	enum masca_error			err = MASCA_OK;
	union masca_intrptr_decode_info	msg_detail;
	struct masca_cd_toc			*p_app_toc = NULL;
	struct masca_cd_text			*p_app_cd_text = NULL;
	bool				snd_cmd = FALSE;
	bool				move_to_next_track = FALSE;

	/*Now we need to check whether we received any response for the
	command we sent*/
	p_app_toc = toc_text_hash.app_cmd_info.command_param.p_cd_toc;
	p_app_cd_text = toc_text_hash.app_cmd_info.command_param.p_cd_text;
	/*The command is accepted and executed by hardware
	successfully*/
	if (MASCA_PROCESSED == response) {
		err = masca_intrptr_decode_msg_info(expected_response,
							&msg_detail, p_msg);
		if (MASCA_OK != err) {
			/*This should never happen*/
			DRV_MNGR_TA_TRACE
				("DrvMngTOC : Err frm decode msg %d\n", err);
		} else if (SYS_TOC == expected_response) {
			/*store the TOC information locally*/
			memcpy(&cd_global_toc, &(msg_detail.sys_toc),
					sizeof(struct masca_intrptr_sys_toc));

			p_app_toc->cd_max_min =
					cd_global_toc.max_cd_min;
			p_app_toc->cd_max_sec =
					cd_global_toc.max_cd_sec;
			p_app_toc->first_track_number =
				cd_global_toc.minimum_track_num;
			p_app_toc->last_track_number =
				cd_global_toc.maximum_track_num;
			p_app_toc->last_min =
					cd_global_toc.last_toc_min;
			p_app_toc->last_sec =
					cd_global_toc.last_toc_sec;
			p_app_toc->last_frm =
					cd_global_toc.last_toc_frame;

			last_cmd_to_drive = CMD_TRACK_START;
			expected_response = TRACK_START;
			cur_trk_num = cd_global_toc.minimum_track_num;
			snd_cmd = TRUE;
		} else if (TRACK_START == expected_response) {
			/*send out next command to read the toc*/
			/*The command is accepted and executed by hardware
			successfully*/
			p_app_toc->track_info[cur_trk_num].
				track_number =
				    msg_detail.track_start.track_number;
			p_app_toc->track_info[cur_trk_num].track_start.min =
					msg_detail.track_start.start_min;
			p_app_toc->track_info[cur_trk_num].track_start.sec =
					msg_detail.track_start.start_sec;
			p_app_toc->track_info[cur_trk_num].track_start.frm = 0;
			p_app_toc->track_info[cur_trk_num].
			    track_start_lba =
			      GET_LBA(msg_detail.track_start.start_min,
				   msg_detail.track_start.start_sec, 0);
			/*we do not expect any error here as TRACK_START
			* is already successful and TRACK_AUDIO is part
			* of same message*/
			masca_intrptr_decode_msg_info(TRACK_AUDIO, &msg_detail,
									p_msg);
			p_app_toc->track_info[cur_trk_num].adr_ctrl =
					msg_detail.track_info.adr_control;

			if (cur_trk_num == cd_global_toc.maximum_track_num)
				masca_track_nos_calc(p_app_toc);

			cur_trk_num++;

			if (cur_trk_num <= cd_global_toc.maximum_track_num)
				snd_cmd = TRUE;
			else
				/*TOC information is completely read*/
				err = MASCA_PROCESSED;
		} else if (TRACK_NAME == expected_response) {
			p_app_cd_text->track_text[cur_trk_num].
					track_num = cur_trk_num;
			p_app_cd_text->track_text[cur_trk_num].
					length_title = msg_detail.
						track_text.name_length;
			memcpy(p_app_cd_text->track_text[cur_trk_num].
					track_title,
					msg_detail.track_text.p_name,
					p_app_cd_text->track_text[cur_trk_num].
							length_title);
			last_cmd_to_drive = CMD_AUTHOR_NAME;
			expected_response = AUTHOR_NAME;
			snd_cmd = TRUE;
		} else if (AUTHOR_NAME == expected_response) {
			p_app_cd_text->track_text[cur_trk_num].
				length_author = msg_detail.
						track_text.name_length;
			memcpy(p_app_cd_text->track_text[cur_trk_num].
				track_author,
				msg_detail.track_text.p_name,
				p_app_cd_text->track_text[cur_trk_num]
							.length_author);
			last_cmd_to_drive = CMD_ALBUM_NAME;
			expected_response = ALBUM_NAME;
			snd_cmd = TRUE;
		} else if (ALBUM_NAME == expected_response) {
			p_app_cd_text->track_text[cur_trk_num].
				length_album = msg_detail.
						track_text.name_length;
			memcpy(p_app_cd_text->track_text[cur_trk_num].
				track_album,
				msg_detail.track_text.p_name,
				p_app_cd_text->track_text[cur_trk_num]
							.length_album);
			move_to_next_track = TRUE;
		} else {
			DRV_MNGR_TA_TRACE(
			"DrvMngrTOC:unknown err frm decode msg info %d\n", err);
		}
	} else if (MASCA_REJECTED == response) {
		/*if cd text is not available the drive rejects
		the commands*/
		if (CMD_TRACK_NAME == last_cmd_to_drive) {
			p_app_cd_text->track_text[cur_trk_num].length_title = 0;
			last_cmd_to_drive = CMD_AUTHOR_NAME;
			expected_response = AUTHOR_NAME;
			snd_cmd = TRUE;
		} else if (CMD_AUTHOR_NAME == last_cmd_to_drive) {
			p_app_cd_text->track_text[cur_trk_num].
							length_author = 0;
			last_cmd_to_drive = CMD_ALBUM_NAME;
			expected_response = ALBUM_NAME;
			snd_cmd = TRUE;
		} else if (CMD_ALBUM_NAME == last_cmd_to_drive) {
			p_app_cd_text->track_text[cur_trk_num].length_album = 0;
			move_to_next_track = TRUE;
		} else {
			/*The command is rejected by the drive. Should be due
			to command sent in invalid state*/
			err = MASCA_REJECTED;
		}
	} else {
		/*nothing to do*/
	}

	if (TRUE == move_to_next_track) {
		if (0 == cur_trk_num)
			/*We fetched the whole CD info, now we will fetch the
			* info of all tracks*/
			cur_trk_num = toc_text_hash.app_cmd_info.command_param.
						p_cd_toc->first_track_number;
		else
			cur_trk_num++;

		if (NULL != p_app_toc) {
			if (cur_trk_num <= p_app_toc->last_track_number) {
				/*fetch name of next track*/
				last_cmd_to_drive = CMD_TRACK_NAME;
				expected_response = TRACK_NAME;
				snd_cmd = TRUE;
			} else {
				/*We are done with reading of complete cd text
				now give response to application*/
				err = MASCA_PROCESSED;
			}
		}
	}
	if (TRUE == snd_cmd)
		masca_send_cmd(SLOT_TOC_TEXT_HASH, last_cmd_to_drive, 0,
								cur_trk_num, 0);

	return err;
}

static void
masca_fill_repsonse(struct masca_drv_response_info * const p_response_info)
{
	p_response_info->drv_cmd_response =
			toc_text_hash.app_cmd_info.command;
	p_response_info->response_error = toc_text_hash.hw_response;
	masca_util_stp_alm(ALM_TOCTEXTHASH);
	masca_util_clr_event(SLOT_TOC_TEXT_HASH_TMOUT);
	toc_text_hash.app_cmd_info.command = MAX_COMMAND;
	if (SLOT_PAUSED != toc_text_hash.slot_stat)
		toc_text_hash.slot_stat = SLOT_IDLE;

	toc_text_hash.hw_response = MASCA_OK;
}

static void
masca_exec_toctexthash(struct masca_drv_response_info * const p_response_info)
{
	enum masca_cmd               app_cmd;
	bool                    snd_cmd = FALSE;
	enum masca_intrptr_cmd       cmd_to_drive = CMD_MAX;
	unsigned char                      trk_no = 0;

	app_cmd = toc_text_hash.app_cmd_info.command;
	toc_text_hash.slot_stat = SLOT_EXECUTING;
	toc_text_hash.hw_response = MASCA_OK;
	if (GET_TOC == app_cmd) {
		/*We will request the Global TOC first to know what are all the
		tracks that are present*/
		p_cd_toc_info =
			toc_text_hash.app_cmd_info.command_param.p_cd_toc;
		cmd_to_drive = CMD_SYS_TOC;
		expected_response = SYS_TOC;
		snd_cmd = TRUE;
	} else if (GET_SESSION_INFO == app_cmd) {
		/*The hardware does not support session info. This should be
		computed masca_sm from the TOC and cd type*/
		p_response_info->drv_cmd_response = app_cmd;
		p_response_info->response_error = MASCA_PROCESSED;
		toc_text_hash.app_cmd_info.command = MAX_COMMAND;
		if (SLOT_PAUSED != toc_text_hash.slot_stat)
			toc_text_hash.slot_stat = SLOT_IDLE;
	} else if (GET_TITLE_ALBUM_AUTHOR == app_cmd) {
		/*Use the systoc to get each of the present track's info*/
		/*we will take the TOC information from application as it can
		* tailor TOC for multisession CDs*/
		cmd_to_drive = CMD_TRACK_NAME;
		expected_response = TRACK_NAME;
		/*Fetch info of whole CD first*/
		cur_trk_num = 0;
		toc_text_hash.app_cmd_info.command_param.
			p_cd_text->track_number_first = toc_text_hash.
			app_cmd_info.command_param.p_cd_toc->first_track_number;
		toc_text_hash.app_cmd_info.command_param.
			p_cd_text->track_number_last = toc_text_hash.
			app_cmd_info.command_param.p_cd_toc->last_track_number;
		trk_no = cur_trk_num;
		snd_cmd = TRUE;
	} else {
		/*Nothing to do*/
	}

	last_cmd_to_drive = cmd_to_drive;
	if ((TRUE == snd_cmd) && (SLOT_PAUSED != toc_text_hash.slot_stat)) {
		toc_text_hash.hw_response = MASCA_OK;
		masca_send_cmd(SLOT_TOC_TEXT_HASH, cmd_to_drive, 0, trk_no, 0);
		masca_util_sta_alm(toc_text_hash.slot_tmout, ALM_TOCTEXTHASH);
	}
}
