/*
 * MASCA Drive Manager Eject Response File
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_interpreter.h"
#include "masca_event_config.h"
#include "masca_drv_mngr.h"
#include "masca_drv_mngr_slot.h"
#include "masca_drv_mngr_cnfg.h"
#include "masca_drv_mngr_internal.h"

static void
masca_eject_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info);

static void masca_exec_eject(const struct masca_drv_response_info *
							const p_response_info);

static struct masca_cmd_slot eject_slot;


void masca_init_eject(const unsigned int retry_cnt,
			const unsigned int reset_cnt,
			const unsigned int response_tmout)
{
	eject_slot.slot_stat = SLOT_PAUSED;
	eject_slot.app_cmd_info.command = MAX_COMMAND;
	eject_slot.hw_response = MASCA_OK;
	eject_slot.slot_tmout = response_tmout;
	masca_init_tx_rx(SLOT_EJECT, retry_cnt, reset_cnt);
}

enum masca_cmd masca_get_eject_cmd(void)
{
	return eject_slot.app_cmd_info.command;
}

void masca_eject_alm_cbk(void)
{
	masca_util_set_event(SLOT_EJECT_TMOUT);
}

enum masca_error
masca_tx_eject(const struct masca_drv_cmd_info * const p_cmd_info,
			struct masca_drv_response_info * const p_response_info)
{
	enum masca_error	ret_err;
	bool		exec_slot = FALSE;

	ret_err = masca_fill_slot(&eject_slot, p_cmd_info, &exec_slot);
	DRV_MNGR_TD_TRACE(
	"tx_eject fill_slot ret_err = %d,exec_slot = %d\n", ret_err, exec_slot);
	if ((MASCA_OK == ret_err) && (TRUE == exec_slot)) {
		/*reload the retry counters for the slot*/
		masca_reload_rty_cnt(SLOT_EJECT);
		masca_exec_eject(p_response_info);
	}
	return ret_err;
}

void masca_rx_eject(
		const struct masca_intrptr_msg_info_list * const p_list,
		const struct masca_intrptr_rjt_cmd_list * const p_rjt_cmds,
		const struct masca_intrptr_msg_info * const p_msg,
		struct masca_drv_response_info * const p_response_info)
{
	/*In the paused state also the responses from the hardware
	* are respected*/
	enum masca_error ret_err;
	if (SLOT_PAUSED != eject_slot.slot_stat) {
		/*Finding out a response is received or not is a generic
		* functionality. Hence it is implemented in a common place*/
		ret_err = masca_rcv_response(SLOT_EJECT, p_list, p_rjt_cmds);
		if ((NULL != p_response_info) && (MASCA_OK != ret_err)) {
			eject_slot.hw_response = ret_err;
			/*Now we need to provide a reply back*/
			if (MAX_COMMAND == p_response_info->drv_cmd_response) {
				masca_eject_response(p_msg, p_response_info);
			} else {
				/*We got response to the slot, so clear
				the timeout*/
				masca_util_stp_alm(ALM_EJECT);
				masca_util_clr_event(SLOT_EJECT_TMOUT);
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
			}
		}
	}
}

void masca_pause_eject(void)
{
	eject_slot.slot_stat = SLOT_PAUSED;
	/*we are pausing so don't allow timeouts*/
	if (MAX_COMMAND > eject_slot.app_cmd_info.command) {
		masca_util_stp_alm(ALM_EJECT);
		masca_util_clr_event(SLOT_EJECT_TMOUT);
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_EJECT);
	}
}

void masca_resume_eject(struct masca_drv_response_info * const p_response_info)
{
	eject_slot.slot_stat = SLOT_IDLE;
	if (MAX_COMMAND > eject_slot.app_cmd_info.command)
		masca_exec_eject(p_response_info);
}


void masca_abort_eject(struct masca_drv_response_info * const p_response_info,
			enum masca_error              abort_reason)
{
	if (MAX_COMMAND > eject_slot.app_cmd_info.command) {
		eject_slot.hw_response = abort_reason;
		if (MAX_COMMAND == p_response_info->drv_cmd_response) {
			eject_slot.abort_slot = FALSE;
			masca_eject_response(NULL, p_response_info);
		} else {
			/* set an abort update so that through event
			* interface of this slot we can abort the command*/
			eject_slot.abort_slot = TRUE;
			masca_util_set_event(SLOT_ABORT_UPDATE);
			masca_util_stp_alm(ALM_EJECT);
			masca_util_clr_event(SLOT_EJECT_TMOUT);
		}
		/*stop the respective unit so that no retry happens*/
		masca_abort_unit(SLOT_EJECT);
	}
}

bool masca_evt_eject(unsigned int evt_ptn_slot,
			struct masca_drv_response_info * const p_response_info)
{
	bool is_rst_required = FALSE;
	enum masca_error            ret_err;
	if (SLOT_RESPONSE_UPDATE == evt_ptn_slot) {
		if (MASCA_OK != eject_slot.hw_response) {
			if (MAX_COMMAND == p_response_info->drv_cmd_response)
				masca_eject_response(NULL, p_response_info);
			else
				masca_util_set_event(SLOT_RESPONSE_UPDATE);
		}
	} else if (SLOT_ABORT_UPDATE == evt_ptn_slot) {
		if (TRUE == eject_slot.abort_slot)
			masca_abort_eject(p_response_info,
							eject_slot.hw_response);
	} else if (SLOT_PAUSED != eject_slot.slot_stat) {
		if (SLOT_EJECT_TMOUT == evt_ptn_slot) {
			/*Now we need to do retries*/
			ret_err = masca_evt_tmout(SLOT_EJECT, &is_rst_required);
			if (MASCA_OK != ret_err) {
				eject_slot.hw_response = ret_err;
				masca_eject_response(NULL, p_response_info);
			} else {
			   masca_util_sta_alm(eject_slot.slot_tmout, ALM_EJECT);
			}
		}
	} else {
		/*In paused state no other events are entertained*/
	}
	return is_rst_required;
}

static void
masca_eject_response(const struct masca_intrptr_msg_info * const p_msg,
			struct masca_drv_response_info * const p_response_info)
{
	if (MASCA_OK != eject_slot.hw_response) {
		p_response_info->drv_cmd_response =
						eject_slot.app_cmd_info.command;
		p_response_info->response_error = eject_slot.hw_response;
		masca_util_stp_alm(ALM_EJECT);
		masca_util_clr_event(SLOT_EJECT_TMOUT);
		eject_slot.app_cmd_info.command    = MAX_COMMAND;

		if (SLOT_PAUSED != eject_slot.slot_stat)
			eject_slot.slot_stat = SLOT_IDLE;

		eject_slot.hw_response = MASCA_OK;
	}
}

static void masca_exec_eject(const struct masca_drv_response_info *
					const p_response_info)
{
	enum masca_intrptr_cmd       cmd_to_drive;

	eject_slot.slot_stat = SLOT_EXECUTING;
	cmd_to_drive =
		app_cmd_mapping[eject_slot.app_cmd_info.command].mapped_cmd;
	eject_slot.hw_response = MASCA_OK;
	masca_send_cmd(SLOT_EJECT, cmd_to_drive, 0, 0, 0);
	masca_util_sta_alm(eject_slot.slot_tmout, ALM_EJECT);
}
