/*
 * Handles all diagnostic commands from dispatcher. It gets the state
 * from state machine and then handles the commands accordingly.
 * Each of these commands is valid in only particular states.
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_dispatch.h"
#include "masca_event_config.h"
#include "masca_diag_hndlr.h"
#include "masca_cmd_hndlr.h"
#include "masca_sm.h"

/**
 * \masca_dispatch_cmd
 *
 * It routes the command to various classes i.e. all diagnostic commands
 * are routed to diag_handler. Commands other than diagnostic are
 * routed to cmd_handler.
 *
 * \param p_cmd     : pointer to the memory location containing
 *                    command structure.
 * \param p_output  : pointer to the memory location containing
 *                    output structure.
 */
void masca_dispatch_cmd(const struct masca_cmd_params * const p_cmd,
				struct masca_output * const p_output)
{
	if (p_cmd->command >= DIAG_GET_TEMPERATURE)
		masca_diag_hndlr_cmd(p_cmd, p_output);
	else
		masca_cmd_hndlr_cmd(p_cmd, p_output);
}

/**
 * \masca_dispatch_event
 *
 * The evt_pattern is specific to each of the classes.
 * This evt_pattern is analyzed and accordingly the event is routed
 * to either drive manager, state machine, command handler or
 * diagnostic handler
 *
 * \param evt_pattern   : event pattern
 * \param p_evt_info    : pointer to event information structure to be filled.
 * \param p_output      : pointer to the memory location containing
 *                        output structure.
 *
 * \return
 *      MASCA_PROCESSED : when the command is successfully processed and
 *                        reply is also given out.
 *      MASCA_PTR_NULL  : If any of the parameters passed is NULL.
 *      MASCA_ABORT     : If the particular command is aborted
 *                        e.g. sector read is going on and in this time
 *                        an eject command is given.
 *      MASCA_IO_ERR    : If the command ends with an IO error.
 *
 */
extern enum masca_error masca_dispatch_event(
				const unsigned int evt_pattern,
				struct masca_event_info * const p_evt_info,
				struct masca_output * const p_output)
{
	struct masca_drv_response_info	response_info;
	struct masca_drv_event_info	drv_evt_info;
	enum masca_error		ret_err = MASCA_PTR_NULL;

	/* Initialize structure to NULL */
	memset((void *)&response_info, 0, sizeof(response_info));
	memset((void *)&drv_evt_info, 0, sizeof(drv_evt_info));

	/*check whether the evt_pattern is enabled for
	***************CD_Drive_Manager event**************/
	if ((evt_pattern & DRIVE_MANAGER_EVENT_MASK) != 0) {
		/*masca drive manager event*/
		ret_err = masca_drive_mngr_event(evt_pattern, &drv_evt_info,
							&response_info);
		/*In masca_sm_event, evt_pattern = 0 */
		if (drv_evt_info.drv_event != DRV_NO_EVENT) {
			ret_err = masca_sm_event(0, &drv_evt_info, p_evt_info);

			/* Checking the critical events with the help updated
			* SM event info */
			if (DRV_DISC_PAUSED == drv_evt_info.drv_event)
				masca_cmd_hndlr_evt(CMD_HNDLR_DSK_PAUSE, NULL);

			if ((p_evt_info->evt >= DRIVE_OVER_TEMPERATURE)
				&& (p_evt_info->evt <= DRIVE_NORMAL_VOLT)) {
				/* update event */
				ret_err = masca_cmd_hndlr_evt(0, NULL);
				ret_err = masca_diag_hndlr_evt(0, NULL);
			}
		}
		/* If the response info is other than temperature, TOC,
		 * session info and title album author */
		if ((response_info.drv_cmd_response <= GET_VERSION_INFO) ||
			((response_info.drv_cmd_response >= RESUME) &&
			(response_info.drv_cmd_response <
						DIAG_GET_TEMPERATURE))) {
			masca_cmd_hndlr_cmd_reply(&response_info, p_output);
		} else if ((response_info.drv_cmd_response >=
						DIAG_GET_TEMPERATURE) &&
			(response_info.drv_cmd_response < MAX_COMMAND)) {
			/* If the response info is a temperature */
			ret_err = masca_diag_hndlr_cmd_reply(&response_info,
								p_output);

			if (ret_err >= MASCA_OK) {
				ret_err = masca_sm_cmd_reply(&response_info,
							      p_evt_info);

				/* Checking the critical events with the help
				 * updated SM event info */
				if ((p_evt_info->evt >= DRIVE_OVER_TEMPERATURE)
				    && (p_evt_info->evt <= DRIVE_NORMAL_VOLT)) {
					/* update temperature */
					ret_err = masca_cmd_hndlr_evt(0, NULL);
					ret_err = masca_diag_hndlr_evt(0, NULL);
				}
			}
		} else if ((GET_TOC <= response_info.drv_cmd_response) &&
			(READ_SECTORS_HASH >= response_info.drv_cmd_response)) {
			/* In masca_sm_event, evt_pattern = 0.
			* If the response is TOC,session info and
			* title album author */
			ret_err = masca_sm_cmd_reply(&response_info,
								p_evt_info);
			/* Checking the critical events with the help updated
			* SM event info */
			if ((p_evt_info->evt >= DRIVE_OVER_TEMPERATURE) &&
				(p_evt_info->evt <= DRIVE_NORMAL_VOLT)) {
				/* update event */
				ret_err = masca_cmd_hndlr_evt(0, NULL);
				ret_err = masca_diag_hndlr_evt(0, NULL);
			}
		}
	}

	/*check whether the evt_pattern is enabled for
	************state machine event*************/
	if ((evt_pattern & SM_EVENT_MASK) != 0) {
		ret_err = masca_sm_event(evt_pattern, NULL, p_evt_info);

		/* Checking the critical events with the help updated
		 * SM event info */
		if (((p_evt_info->evt >= DRIVE_OVER_TEMPERATURE) &&
			(p_evt_info->evt <= DRIVE_NORMAL_VOLT)) ||
			(SM_EVENT_RESUME == (evt_pattern & SM_EVENT_RESUME)) ||
			(SM_EVENT_SUSPEND ==
					(evt_pattern & SM_EVENT_SUSPEND))) {
			ret_err = masca_cmd_hndlr_evt(0, NULL);
			ret_err = masca_diag_hndlr_evt(0, NULL);
		}
	}

	/*check whether the evt_pattern is enabled for
	************command handler event************/
	if ((evt_pattern & CMD_HANDLER_EVENT_MASK) != 0)
		ret_err = masca_cmd_hndlr_evt(evt_pattern, p_output);

	/*check whether the evt_pattern is enabled for
	***********diagnostic handler event*********/
	if ((evt_pattern & DIAG_HANDLER_EVENT_MASK) != 0)
		ret_err = masca_diag_hndlr_evt(evt_pattern, p_output);

	return ret_err;
}

/**
 * \masca_dispatch_abort_req
 *
 * The abort request is routed to both cmd_handler and diag_handler.
 * If the request ID is applicable the respective class picks it up.
 *
 * \param abort_req     : The request to be aborted
 *
 * \return
 *      MASCA_PROCESSED    : when the command is successfully processed
 *      MASCA_INVALID_PARAM  : If the particular request ID is not pending
 *                        or executing.
 *
 */
extern enum masca_error masca_dispatch_abort_req(const unsigned int abort_req)
{
	enum masca_error ret_err = MASCA_INVALID_PARAM;
	/* Dispatch abort request*/
	ret_err = masca_cmd_hndlr_abort(abort_req);
	if (ret_err < MASCA_OK)
		ret_err = masca_diag_hndlr_abort(abort_req);

	return ret_err;
}
