/*
 * MASCA Handler File for Diagnostic Messages from Dispatcher
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "masca_common_types.h"
#include "masca_sm.h"
#include "masca_cmd_hndlr.h"
#include "masca_diag_hndlr.h"
#include "masca_event_config.h"
#include "masca_blkdev.h"

/* diag command handler elements */
static struct masca_cmd_lst_elem *p_frst_diag_elem;
static struct masca_cmd_lst_elem *p_last_diag_elem;
static bool               abort_executing_diag_cmd;
static bool               abort_pending_diag_cmd;
static bool               abort_diag_cmd_search;

/* As per above array table between command and different driver state */
/* We have only 30 Diagnostics related entries */
static bool diag_cmd_driver_state[MAX_COMMAND - DIAG_GET_SWITCH_STATUS] = {
	1, /* DIAG_GET_TEMPERATURE */
	1, /* DIAG_GET_LASER_CURRENT */
	1, /* DIAG_GET_SWITCH_STATUS */
	1, /* DIAG_CMD_ADJUSTMENT */
	1, /* DIAG_CMD_CD_TEXT_OFF */
	1, /* DIAG_CMD_CD_TEXT_MODE2 */
	0, /* DIAG_CMD_C1C2_ERR_COUNT_ON */
	0, /* DIAG_CMD_TIME_READOUT */
	0, /* DIAG_CMD_EJECT_LOAD_CYCLES */
	0, /* DIAG_CMD_JUMP_COUNTER */
	0, /* DIAG_CMD_EEPROM_READOUT */
	0, /* DIAG_CMD_WRITE_STICKER */
	0, /* DIAG_CMD_READ_STICKER */
	0, /* DIAG_CMD_JITTER_MEASURE */
	0, /* DIAG_CMD_VERTICAL_DEV_MEASURE */
	0, /* DIAG_CMD_ECCENTRIC_MEASURE */
	0, /* DIAG_CMD_REFLECTIVE_MEASURE */
	0, /* DIAG_CMD_INTERNAL_SELF_TEST */
	0, /* DIAG_CMD_EXTERNAL_SDRAM_TEST */
	0, /* DIAG_CMD_INIT_LASER_CURRENT */
	0, /* DIAG_CMD_NUM_READ_ERRS */
	0, /* DIAG_CMD_DATE_PRODUCTION_READ */
	0, /* DIAG_CMD_DATE_INSTALL_WRITE */
	0, /* DIAG_CMD_DATE_INSTALL_READ */
	0, /* DIAG_CMD_SERIAL_NO_READ */
	0, /* DIAG_CMD_STATUS_MSG_ERR_HISTROY */
	0, /* DIAG_CMD_STATUS_MSG_LAST_ERRS */
	0, /* DIAG_CMD_USED_CD_TYPE_COUNT */
	0, /* DIAG_CMD_USER_PROFILE_USED_CDS */
	1, /* DIAG_CACHE_DISABLE */
};

/* As per above array table between command and different drive state */
static bool diag_cmd_drive_state[MAX_COMMAND - DIAG_GET_SWITCH_STATUS]
						[SM_MAX_DRV_STATE] = {
	{1, 1,  1,  1,  1,  1}, /* DIAG_GET_TEMPERATURE */
	{1, 1,  1,  1,  1,  1}, /* DIAG_GET_LASER_CURRENT */
	{1, 1,  1,  1,  1,  1}, /* DIAG_GET_SWITCH_STATUS */
	{1, 1,  1,  1,  1,  1}, /* DIAG_CMD_ADJUSTMENT */
	{1, 1,  1,  1,  1,  1}, /* DIAG_CMD_CD_TEXT_OFF */
	{1, 1,  1,  1,  1,  1}, /* DIAG_CMD_CD_TEXT_MODE2 */
	{0, 0,  0,  0,  1,  0}, /* DIAG_CMD_C1C2_ERR_COUNT_ON */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_TIME_READOUT */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_EJECT_LOAD_CYCLES */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_JUMP_COUNTER */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_EEPROM_READOUT */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_WRITE_STICKER */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_READ_STICKER */
	{0, 0,  0,  0,  1,  0}, /* DIAG_CMD_JITTER_MEASURE */
	{0, 0,  0,  0,  1,  0}, /* DIAG_CMD_VERTICAL_DEV_MEASURE */
	{0, 0,  0,  0,  1,  0}, /* DIAG_CMD_ECCENTRIC_MEASURE */
	{0, 0,  0,  0,  1,  0}, /* DIAG_CMD_REFLECTIVE_MEASURE */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_INTERNAL_SELF_TEST */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_EXTERNAL_SDRAM_TEST */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_INIT_LASER_CURRENT */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_NUM_READ_ERRS */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_DATE_PRODUCTION_READ */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_DATE_INSTALL_WRITE */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_DATE_INSTALL_READ */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_SERIAL_NO_READ */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_STATUS_MSG_ERR_HISTROY */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_STATUS_MSG_LAST_ERRS */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_USED_CD_TYPE_COUNT */
	{1, 0,  1,  0,  1,  0}, /* DIAG_CMD_USER_PROFILE_USED_CDS */
	{1, 1,  1,  1,  1,  1}, /* DIAG_CACHE_DISABLE */
};

/* As per above array table between command and different disc state */
static bool diag_cmd_disc_state[MAX_COMMAND - DIAG_GET_SWITCH_STATUS]
						[SM_MAX_DISC_STATE] = {
	{1, 1,  1}, /* DIAG_GET_TEMPERATURE */
	{1, 1,  1}, /* DIAG_GET_LASER_CURRENT */
	{1, 1,  1}, /* DIAG_GET_SWITCH_STATUS */
	{1, 1,  1}, /* DIAG_CMD_ADJUSTMENT */
	{1, 1,  1}, /* DIAG_CMD_CD_TEXT_OFF */
	{1, 1,  1}, /* DIAG_CMD_CD_TEXT_MODE2 */
	{1, 1,  0}, /* DIAG_CMD_C1C2_ERR_COUNT_ON */
	{0, 0,  1}, /* DIAG_CMD_TIME_READOUT */
	{0, 0,  1}, /* DIAG_CMD_EJECT_LOAD_CYCLES */
	{0, 0,  1}, /* DIAG_CMD_JUMP_COUNTER */
	{0, 0,  1}, /* DIAG_CMD_EEPROM_READOUT */
	{0, 0,  1}, /* DIAG_CMD_WRITE_STICKER */
	{0, 0,  1}, /* DIAG_CMD_READ_STICKER */
	{1, 0,  0}, /* DIAG_CMD_JITTER_MEASURE */
	{1, 0,  0}, /* DIAG_CMD_VERTICAL_DEV_MEASURE */
	{1, 0,  0}, /* DIAG_CMD_ECCENTRIC_MEASURE */
	{1, 0,  0}, /* DIAG_CMD_REFLECTIVE_MEASURE */
	{0, 0,  1}, /* DIAG_CMD_INTERNAL_SELF_TEST */
	{0, 0,  1}, /* DIAG_CMD_EXTERNAL_SDRAM_TEST */
	{0, 0,  1}, /* DIAG_CMD_INIT_LASER_CURRENT */
	{0, 0,  1}, /* DIAG_CMD_NUM_READ_ERRS */
	{0, 0,  1}, /* DIAG_CMD_DATE_PRODUCTION_READ */
	{0, 0,  1}, /* DIAG_CMD_DATE_INSTALL_WRITE */
	{0, 0,  1}, /* DIAG_CMD_DATE_INSTALL_READ */
	{0, 0,  1}, /* DIAG_CMD_SERIAL_NO_READ */
	{0, 0,  1}, /* DIAG_CMD_STATUS_MSG_ERR_HISTROY */
	{0, 0,  1}, /* DIAG_CMD_STATUS_MSG_LAST_ERRS */
	{0, 0,  1}, /* DIAG_CMD_USED_CD_TYPE_COUNT */
	{0, 0,  1}, /* DIAG_CMD_USER_PROFILE_USED_CDS */
	{1, 1,  1}, /* DIAG_CACHE_DISABLE */
};

/* As per above array table between command and different voltage state */
static bool diag_cmd_volt_state[MAX_COMMAND - DIAG_GET_SWITCH_STATUS]
						[MAX_VOLT_STATE] = {
	{1, 1,  1,  1}, /* DIAG_GET_TEMPERATURE */
	{1, 1,  1,  1}, /* DIAG_GET_LASER_CURRENT */
	{1, 1,  1,  1}, /* DIAG_GET_SWITCH_STATUS */
	{1, 1,  1,  1}, /* DIAG_CMD_ADJUSTMENT */
	{1, 1,  1,  1}, /* DIAG_CMD_CD_TEXT_OFF */
	{1, 1,  1,  1}, /* DIAG_CMD_CD_TEXT_MODE2 */
	{0, 0,  1,  0}, /* DIAG_CMD_C1C2_ERR_COUNT_ON */
	{0, 0,  1,  0}, /* DIAG_CMD_TIME_READOUT */
	{0, 0,  1,  0}, /* DIAG_CMD_EJECT_LOAD_CYCLES */
	{0, 0,  1,  0}, /* DIAG_CMD_JUMP_COUNTER */
	{0, 0,  1,  0}, /* DIAG_CMD_EEPROM_READOUT */
	{0, 0,  1,  0}, /* DIAG_CMD_WRITE_STICKER */
	{0, 0,  1,  0}, /* DIAG_CMD_READ_STICKER */
	{0, 0,  1,  0}, /* DIAG_CMD_JITTER_MEASURE */
	{0, 0,  1,  0}, /* DIAG_CMD_VERTICAL_DEV_MEASURE */
	{0, 0,  1,  0}, /* DIAG_CMD_ECCENTRIC_MEASURE */
	{0, 0,  1,  0}, /* DIAG_CMD_REFLECTIVE_MEASURE */
	{0, 0,  1,  0}, /* DIAG_CMD_INTERNAL_SELF_TEST */
	{0, 0,  1,  0}, /* DIAG_CMD_EXTERNAL_SDRAM_TEST */
	{0, 0,  1,  0}, /* DIAG_CMD_INIT_LASER_CURRENT */
	{0, 0,  1,  0}, /* DIAG_CMD_NUM_READ_ERRS */
	{0, 0,  1,  0}, /* DIAG_CMD_DATE_PRODUCTION_READ */
	{0, 0,  1,  0}, /* DIAG_CMD_DATE_INSTALL_WRITE */
	{0, 0,  1,  0}, /* DIAG_CMD_DATE_INSTALL_READ */
	{0, 0,  1,  0}, /* DIAG_CMD_SERIAL_NO_READ */
	{0, 0,  1,  0}, /* DIAG_CMD_STATUS_MSG_ERR_HISTROY */
	{0, 0,  1,  0}, /* DIAG_CMD_STATUS_MSG_LAST_ERRS */
	{0, 0,  1,  0}, /* DIAG_CMD_USED_CD_TYPE_COUNT */
	{0, 0,  1,  0}, /* DIAG_CMD_USER_PROFILE_USED_CDS */
	{1, 1,  1,  1}, /* DIAG_CACHE_DISABLE */
};

/* Below diagnostic commands denoted with 0 aren't supported by CD drive
 * Ref: BE0111D_20120810_S11 5_ALL_MASCA 3 2 Deviation List.pdf */
static bool diag_cmd_supportable[MAX_COMMAND - DIAG_GET_SWITCH_STATUS] = {
	1, /* DIAG_GET_TEMPERATURE */
	1, /* DIAG_GET_LASER_CURRENT */
	1, /* DIAG_GET_SWITCH_STATUS */
	1, /* DIAG_CMD_ADJUSTMENT */
	0, /* DIAG_CMD_CD_TEXT_OFF */
	0, /* DIAG_CMD_CD_TEXT_MODE2 */
	1, /* DIAG_CMD_C1C2_ERR_COUNT_ON */
	1, /* DIAG_CMD_TIME_READOUT */
	1, /* DIAG_CMD_EJECT_LOAD_CYCLES */
	1, /* DIAG_CMD_JUMP_COUNTER */
	1, /* DIAG_CMD_EEPROM_READOUT */
	1, /* DIAG_CMD_WRITE_STICKER */
	1, /* DIAG_CMD_READ_STICKER */
	0, /* DIAG_CMD_JITTER_MEASURE */
	0, /* DIAG_CMD_VERTICAL_DEV_MEASURE */
	0, /* DIAG_CMD_ECCENTRIC_MEASURE */
	0, /* DIAG_CMD_REFLECTIVE_MEASURE */
	0, /* DIAG_CMD_INTERNAL_SELF_TEST */
	0, /* DIAG_CMD_EXTERNAL_SDRAM_TEST */
	0, /* DIAG_CMD_INIT_LASER_CURRENT */
	0, /* DIAG_CMD_NUM_READ_ERRS */
	1, /* DIAG_CMD_DATE_PRODUCTION_READ */
	0, /* DIAG_CMD_DATE_INSTALL_WRITE */
	0, /* DIAG_CMD_DATE_INSTALL_READ */
	0, /* DIAG_CMD_SERIAL_NO_READ */
	0, /* DIAG_CMD_STATUS_MSG_ERR_HISTROY */
	0, /* DIAG_CMD_STATUS_MSG_LAST_ERRS */
	0, /* DIAG_CMD_USED_CD_TYPE_COUNT */
	0, /* DIAG_CMD_USER_PROFILE_USED_CDS */
	1, /* DIAG_CACHE_DISABLE */
};

/**
 * \func masca_is_diag_cmd_allowed
 *
 * This function is used to check the possible commands with respect
 * to SM states with the help of array table.It will update is_diag_cmd_allowed
 * # if is_diag_cmd_allowed = TRUE, commands is allowed with respect to SM state
 * # if is_diag_cmd_allowed = FALSE,commands is not allowed with respect to
 *                            SM state
 *
 */
static bool masca_is_diag_cmd_allowed(const enum masca_cmd diag_cmd)
{
	bool is_diag_cmd_allowed = TRUE;
	struct masca_sm_state sm_state;

	if ((diag_cmd >= DIAG_GET_TEMPERATURE) && (diag_cmd < MAX_COMMAND)) {
		masca_sm_get_state(&sm_state);
		/* Since initial value is TRUE else is not handled */
		if (NORMAL != sm_state.sm_driver_state) {
			is_diag_cmd_allowed &= diag_cmd_driver_state
						[diag_cmd -
						 DIAG_GET_TEMPERATURE];
		}

		is_diag_cmd_allowed &= (diag_cmd_drive_state
					[diag_cmd - DIAG_GET_TEMPERATURE]
					[sm_state.sm_drive_state]);

		if (SM_ACCESSIBLE == sm_state.sm_drive_state) {
			is_diag_cmd_allowed &= (diag_cmd_disc_state
					       [diag_cmd - DIAG_GET_TEMPERATURE]
					       [sm_state.sm_disc_state]);
		}

		is_diag_cmd_allowed &= (diag_cmd_volt_state
					[diag_cmd - DIAG_GET_TEMPERATURE]
					[sm_state.sm_volt_state]);
	} else {
		is_diag_cmd_allowed = false;
	}
	return is_diag_cmd_allowed;
}

static enum masca_error
masca_add_diag_to_lst(const struct masca_cmd_params * const p_cmd_info,
					const enum masca_cmd_lst_stat elem_stat)
{
	enum masca_error         ret_err = MASCA_Q_FULL;
	struct masca_cmd_lst_elem *p_diag_elem = p_last_diag_elem;
	if (NULL != p_diag_elem) {
		if (EMPTY == p_diag_elem->elem_stat) {
			ret_err = MASCA_OK;
			memcpy((void *)&(p_diag_elem->acpptd_cmd),
					(void *)p_cmd_info,
					sizeof(struct masca_cmd_params));
			p_diag_elem->elem_stat = elem_stat;
		}
		if (NULL != p_diag_elem->p_nxt) {
			/*point to next free element*/
			p_last_diag_elem = p_diag_elem->p_nxt;
		}
	}
	return ret_err;
}


static void masca_empty_diag_lst_elem(const struct masca_cmd_lst_elem *
						const p_diag_lst_to_empty)
{
	struct masca_cmd_lst_elem *p_diag_elem = p_frst_diag_elem;
	struct masca_cmd_lst_elem *p_diag_prev = NULL;
	while ((NULL != p_diag_elem) && (NULL != p_diag_lst_to_empty)) {
		if (p_diag_elem == p_diag_lst_to_empty) {
			p_diag_elem->elem_stat = EMPTY;
			if (p_diag_elem == p_last_diag_elem) {
				/*nothing to do as it is end of the list*/
			} else if (NULL != p_diag_prev) {
				/*element in the middle of the list to be
				  emptied*/
				/*move this to end of the list*/
				p_diag_prev->p_nxt = p_diag_elem->p_nxt;
				if (NULL != p_last_diag_elem) {
					p_diag_elem->p_nxt =
							p_last_diag_elem->p_nxt;
					p_last_diag_elem->p_nxt = p_diag_elem;
				}
			} else {
			/*This is the first element in the list to be emptied*/
				p_frst_diag_elem = p_diag_elem->p_nxt;
				if (NULL != p_last_diag_elem) {
					p_diag_elem->p_nxt =
							p_last_diag_elem->p_nxt;
					p_last_diag_elem->p_nxt = p_diag_elem;
				}
			}
			/*terminate the loop*/
			p_diag_elem = NULL;
		} else {
			p_diag_prev = p_diag_elem;
			p_diag_elem = p_diag_elem->p_nxt;
		}
	}
}

static struct masca_cmd_lst_elem*
masca_get_frst_diag_elem(const enum masca_cmd_lst_stat elem_stat)
{
	bool                is_diag_cmd_allowed = FALSE;
	struct masca_cmd_lst_elem *p_diag_elem = p_frst_diag_elem;
	struct masca_cmd_lst_elem *p_found_diag_elem = NULL;

	while (NULL != p_diag_elem) {
		if (abort_diag_cmd_search != FALSE)
			is_diag_cmd_allowed =
			masca_is_diag_cmd_allowed(p_diag_elem->
							acpptd_cmd.command);

		if (is_diag_cmd_allowed == TRUE) {
			p_diag_elem = p_diag_elem->p_nxt;
		} else {
			if (elem_stat == p_diag_elem->elem_stat) {
				p_found_diag_elem = p_diag_elem;
				p_diag_elem = NULL;
			} else {
				p_diag_elem = p_diag_elem->p_nxt;
			}
		}
	}
	return p_found_diag_elem;
}


static struct masca_cmd_lst_elem
			*masca_get_diag_elem_by_id(const unsigned int hash)
{
	struct masca_cmd_lst_elem *p_diag_elem = p_frst_diag_elem;
	struct masca_cmd_lst_elem *p_found_diag_elem = NULL;

	while (NULL != p_diag_elem) {
		if ((EMPTY != p_diag_elem->elem_stat)
			&& (hash == p_diag_elem->acpptd_cmd.reply_id)) {
			p_found_diag_elem = p_diag_elem;
			p_diag_elem = NULL;
		} else {
			p_diag_elem = p_diag_elem->p_nxt;
		}
	}
	return p_found_diag_elem;
}

/**
 * \func    masca_diag_hndlr_init
 *
 */
void masca_diag_hndlr_init(const struct masca_configuration *const p_config)
{
	struct masca_cmd_lst_elem *p_diag_elem;
	unsigned int         diag_elem_count;
	unsigned int         lists_to_diag_construct;

	abort_executing_diag_cmd = FALSE;
	abort_pending_diag_cmd = FALSE;
	abort_diag_cmd_search = FALSE;
	p_diag_elem = p_config->p_diag_hndlr_lst;
	/*Remember the first element of linked list*/
	p_frst_diag_elem = p_diag_elem;
	p_last_diag_elem = p_diag_elem;
	if (NULL != p_diag_elem) {
		lists_to_diag_construct =
				(unsigned int)(p_config
					->pllel_maxreq - 1);
		/*create singly linked list*/
		for (diag_elem_count = 0;
		     diag_elem_count < lists_to_diag_construct;
		     diag_elem_count++) {
			p_diag_elem[diag_elem_count].p_nxt =
			&p_diag_elem[diag_elem_count + 1];
			p_diag_elem[diag_elem_count].elem_stat = EMPTY;
		}
		p_diag_elem[diag_elem_count].p_nxt = NULL;
		p_diag_elem[diag_elem_count].elem_stat = EMPTY;
	}
}



static struct masca_cmd_lst_elem *
masca_get_diag_elem_by_cmd(const enum masca_cmd cmd,
					const enum masca_cmd_lst_stat elem_stat)
{
	struct masca_cmd_lst_elem *p_diag_elem = p_frst_diag_elem;
	struct masca_cmd_lst_elem *p_found_diag_elem = NULL;

	while (NULL != p_diag_elem) {
		if ((elem_stat == p_diag_elem->elem_stat)
			&& (p_diag_elem->acpptd_cmd.command == cmd)) {
			p_found_diag_elem = p_diag_elem;
			p_diag_elem = NULL;
		} else {
			p_diag_elem = p_diag_elem->p_nxt;
		}
	}
	return p_found_diag_elem;
}

static void masca_prepare_diag_response(
			const struct masca_drv_response_info * const p_res_info,
			struct masca_output * const p_output)
{
	struct masca_cmd_lst_elem *p_diag_elem;
	enum masca_cmd           rep_cmd;
	if ((NULL != p_res_info) && (NULL != p_output)) {
		p_output->replied_command = MAX_COMMAND;
		rep_cmd = p_res_info->drv_cmd_response;
		if (rep_cmd != MAX_COMMAND) {
			p_diag_elem = masca_get_diag_elem_by_cmd(rep_cmd,
								    EXECUTING);
			if (NULL != p_diag_elem) {
				/*We got response to the command we sent*/
				p_output->replied_command = rep_cmd;
				p_output->response_id = p_diag_elem->acpptd_cmd
								      .reply_id;
				p_output->reply_status = p_res_info
							       ->response_error;
				if (p_res_info->response_error ==
					MASCA_REJECTED) {
					if (diag_cmd_supportable
						[rep_cmd-DIAG_GET_TEMPERATURE]
								!= 1) {
						p_output->reply_status =
							     MASCA_CMD_NOT_SUPP;
					}
				}
				p_output->output_params.cd_version_info =
				     p_res_info->response_param.cd_version_info;
				p_output->output_params.cd_diag_params =
				      p_res_info->response_param.cd_diag_params;
				/*clear out the entry in our list*/
				masca_empty_diag_lst_elem(p_diag_elem);
			}
		}
	}
}


static void masca_diag_cmd_drive_mngr(const struct masca_cmd_params *
			const p_cmd, struct masca_output * const p_output)
{
	struct masca_drv_cmd_info      manager_cmd;
	struct masca_drv_response_info manager_response;
	enum masca_error             ret_err;

	memcpy((void *)&manager_cmd.command_param.drv_param,
	(void *)&p_cmd->cmd_params, sizeof(p_cmd->cmd_params));
	manager_cmd.command = p_cmd->command;
	manager_response.drv_cmd_response = MAX_COMMAND;

	ret_err = masca_drive_mngr_cmd(&manager_cmd, &manager_response);
	if (MASCA_BUSY == ret_err) {
		/*Add this command to the list*/
		masca_add_diag_to_lst(p_cmd, PENDING);
		masca_prepare_diag_response(&manager_response, p_output);
	} else if ((MASCA_OK == ret_err) || (MASCA_ACCEPTED == ret_err) ||
			(MASCA_PROCESSED == ret_err)) {
		/*command is accepted by drive manager*/
		if (manager_response.drv_cmd_response != p_cmd->command) {
			/*add command to list*/
			masca_add_diag_to_lst(p_cmd, EXECUTING);
			masca_prepare_diag_response(&manager_response,
								      p_output);
		} else {
			/*no need to add to the list as command is already
			* processed. Just reply over here*/
			if (NULL != p_output) {
				p_output->replied_command = manager_cmd.command;
				p_output->response_id = p_cmd->reply_id;
				p_output->reply_status = manager_response
								.response_error;
				p_output->output_params.cd_version_info =
				manager_response.response_param.cd_version_info;
			}
		}
	} else {
		DIAG_TE_TRACE("DIAG_CMD_HND: drive mngr return error = %d",
								ret_err);
	}
}


/**
 * \func    masca_diag_hndlr_cmd
 *
 * This API is used to get a temperature of the drive.
 *
 * \param p_cmd     : pointer to the memory location holding
 *                    command information structure.
 *
 * \param p_output  : pointer to the output structure holding
 *                    command response information.
 *
 * \return
 *  MASCA_PROCESSED : when the command is successfully processed and reply is
 *                    also given out.
 *  MASCA_ACCEPTED  : when the command is accepted for processing.
 *  MASCA_REJECTED  : Command is rejected in the case the queue is full.
 *  MASCA_Q_FULL    : command is accepted but no further commands can be
 *                    accepted unless some of the commands are replied.
 *  MASCA_PTR_NULL  : If any of the parameters passed is NULL.
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_OK        : If message receiving is successful.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 *  MASCA_INVALID_STATE     : To process the command particular state is
 *                            not valid.
 *  MASCA_UNKNOWN_MSG_CMD   : If the command sent is an invalid command.
 */
extern enum masca_error masca_diag_hndlr_cmd(
				const struct masca_cmd_params * const p_cmd,
				struct masca_output * const p_output)
{
	bool                    diag_disable_cache = FALSE;
	enum masca_error             ret_err = MASCA_PTR_NULL;
	enum masca_cmd               diag_cmd;

	diag_cmd = p_cmd->command;
	p_output->replied_command = diag_cmd;
	p_output->response_id = p_cmd->reply_id;

	ret_err = masca_check_async_sense();

	if ((ret_err != MASCA_OK)  && ((ret_err == MASCA_HE_COMM_FAIL)
				|| (ret_err == MASCA_HE_VOLTAGE_FAULT))) {
		p_output->reply_status = ret_err;
		return ret_err;
	}
	ret_err = MASCA_OK;

	if (masca_is_diag_cmd_allowed(p_cmd->command)) {
		if (diag_cmd == DIAG_CACHE_DISABLE) {
			diag_disable_cache =
			p_cmd->cmd_params.diag_cmd_params.cache_disable;
			/* DIAG_CACHE_DISABLE command is used to set
			 * a variable forenable/disable cache.
			 * So, No need to send this command to CD drive
			 * and reply back with success. */
			masca_drv_mngr_disable_cache
						(diag_disable_cache);
			p_output->reply_status = ret_err;
		} else if ((diag_cmd >= DIAG_GET_TEMPERATURE)
				&& (diag_cmd < MAX_COMMAND)) {
			/* If the command is between Temperature
			   and diag commands */
			masca_diag_cmd_drive_mngr(p_cmd,
							 p_output);
		} else {
			DIAG_TE_TRACE(
			"DIAG:SM allowed,cmd(%d) isn't exe", diag_cmd);
		}

	} else {
		p_output->reply_status = MASCA_INVALID_STATE;
		DIAG_TE_TRACE(
		"Diag cmd is not allowed-CD drive due to drive states");
	}

	return ret_err;
}


/**
 * \func    masca_diag_hndlr_cmd_reply
 *
 * This API is called when reply to a command is available.
 * If the CD_Drv_Manager replies to a command
 * then it is routed to diag_handler.
 *
 * \param p_response_info   : pointer to the memory location holding
 *                            response information from CD_Drv_Manager.
 *
 * \param p_output          : pointer to the output structure holding
 *                            command response information.
 *
 * \return
 *  MASCA_PROCESSED : The command is processed successfully.
 *  MASCA_ABORT     : If the particular command is aborted.
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_PTR_NULL  : If any of the parameters are NULL.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 */

extern enum masca_error masca_diag_hndlr_cmd_reply(
		const struct masca_drv_response_info * const p_response_info,
		struct masca_output * const p_output)
{
	enum masca_error         ret_err = MASCA_OK;
	struct masca_cmd_lst_elem  *p_diag_elem;

	/*Check if this particular command exists in the queue
	* and fill out the response structure*/
	masca_prepare_diag_response(p_response_info, p_output);
	/*Check if there are any commands pending*/
	p_diag_elem = masca_get_frst_diag_elem(PENDING);
	if (NULL != p_diag_elem)
		/*Handle this request in event function*/
		masca_util_set_event(DIAG_CMD_FROM_QUEUE);

	return ret_err;
}

void masca_diag_extn(struct masca_output * const p_output,
				struct masca_drv_cmd_info manager_cmd,
				struct masca_cmd_lst_elem *p_diag_elem,
				struct masca_drv_response_info manager_response)
{
	if (manager_response.drv_cmd_response != p_diag_elem->acpptd_cmd.
		command) {
		/*Chk whether we got response to any othercommand & update*/
		masca_prepare_diag_response(&manager_response, p_output);
	} else { /*The command is successfully processed.remove from list*/
		/* This NULL check needed,called with NULL param in one place.*/
		if (NULL != p_output) {
			p_output->replied_command = manager_cmd.command;
			p_output->response_id =	p_diag_elem->acpptd_cmd.
						reply_id;
			p_output->reply_status = manager_response.
						response_error;
			p_output->output_params.cd_version_info =
				manager_response.response_param.cd_version_info;
		}
		/*clear out the entry in our list*/
		masca_empty_diag_lst_elem(p_diag_elem);
	}
}

/**
 * \func    masca_diag_hndlr_evt
 *
 * This API is called in case of an event. When cmd_handler has events
 * then it posts to GDI handler's function which is ultimately routed
 * through this API.
 *
 * \param evt_pattern   : event pattern of command handler
 *
 * \param p_output      : pointer to the output structure holding
 *                            command response information.
 *
 * \return
 *  MASCA_PROCESSED : The command is processed successfully.
 *  MASCA_ABORT     : If the particular command is aborted.
 *  MASCA_IO_ERR    : If the command ends with an IO error.
 *  MASCA_PTR_NULL  : If any of the parameters are NULL.
 *  MASCA_BUSY      : If drive manager is not able handle command,
 *                    then drive manager return an error of MASCA busy
 */
extern enum masca_error masca_diag_hndlr_evt(const unsigned int evt_pattern,
					struct masca_output * const p_output)
{
	struct masca_sm_state          sm_state;
	struct masca_drv_cmd_info      manager_cmd;
	struct masca_drv_response_info manager_response;
	struct masca_cmd_lst_elem      *p_diag_elem;
	enum masca_error             ret_err = MASCA_OK;

	/* Initialize structure to NULL */
	memset(&sm_state, 0, sizeof(sm_state));

	masca_sm_get_state(&sm_state);

	switch (evt_pattern) {
	case 0:
		/* Update state machine */
		masca_sm_get_state(&sm_state);

		if ((sm_state.sm_temp_state == OVER_TEMPERATURE)
			|| (sm_state.sm_volt_state == UNDER_VOLTAGE)
			|| (sm_state.sm_driver_state == SUSPENDED)) {
			/*setting both of these to FALSE will result in pause
			of the commands to drive manager i.e the commands will
			just be aborted in drive manager and are kept in the
			queue as pending*/
			abort_executing_diag_cmd = TRUE;
			abort_pending_diag_cmd = TRUE;
			masca_util_set_event(DIAG_CMD_ABORT);
		} else if ((sm_state.sm_temp_state == NORMAL_TEMPERATURE)
				&& (sm_state.sm_volt_state == NORMAL_VOLTAGE)
				&& (sm_state.sm_driver_state == NORMAL)) {
			masca_util_set_event(DIAG_CMD_FROM_QUEUE);
		} else {
		/*nothing to do*/
		}
		break;

	case DIAG_CMD_FROM_QUEUE:
		/*Get the next pending element from the list*/
		/*Only diag commands sent to drive manager need to be queued*/
		p_diag_elem = masca_get_frst_diag_elem(PENDING);
		if (NULL != p_diag_elem) {
			if (masca_is_diag_cmd_allowed(
				p_diag_elem->acpptd_cmd.command)) {
				memcpy((void *)&manager_cmd.command_param.
				drv_param,
				(void *)&p_diag_elem->acpptd_cmd.cmd_params,
				sizeof(p_diag_elem->acpptd_cmd.cmd_params));
				manager_cmd.command = p_diag_elem->acpptd_cmd
								.command;
				manager_response.drv_cmd_response = MAX_COMMAND;

				ret_err = masca_drive_mngr_cmd(&manager_cmd,
							&manager_response);
				if (MASCA_BUSY == ret_err) {
					/*Check whether we got response to any
					  other command and update*/
					masca_prepare_diag_response(
						&manager_response, p_output);
				} else if ((MASCA_OK == ret_err) ||
						(MASCA_ACCEPTED == ret_err) ||
						(MASCA_PROCESSED == ret_err)) {
					/*command is accepted by drive manager*/
					p_diag_elem->elem_stat = EXECUTING;
					masca_diag_extn(p_output, manager_cmd,
					p_diag_elem, manager_response);
					/*Check for any next pending command*/
					p_diag_elem = masca_get_frst_diag_elem(
								       PENDING);
					if (NULL != p_diag_elem) {
						masca_util_set_event(
							DIAG_CMD_FROM_QUEUE);
					}
				} else {
					DIAG_TE_TRACE(
					"DiagCmd drv mngr ret_err:%d", ret_err);
				}
			} else {
				p_output->replied_command =
						p_diag_elem->acpptd_cmd.command;
				p_output->response_id =
					       p_diag_elem->acpptd_cmd.reply_id;
				p_output->reply_status = MASCA_INVALID_STATE;
				/*clear out the entry in our list*/
				masca_empty_diag_lst_elem(p_diag_elem);
			}
		}
		break;
	case DIAG_CMD_ABORT:
		abort_diag_cmd_search = TRUE;
		if (TRUE == abort_executing_diag_cmd) {
			p_diag_elem = masca_get_frst_diag_elem(EXECUTING);
			if (NULL != p_diag_elem) {
				masca_drive_mngr_abort(
					       p_diag_elem->acpptd_cmd.command);
				p_output->replied_command =
						p_diag_elem->acpptd_cmd.command;
				p_output->response_id =
					       p_diag_elem->acpptd_cmd.reply_id;
				p_output->reply_status = MASCA_ABORT;
				masca_empty_diag_lst_elem(p_diag_elem);
			} else {
				/*all executing commands aborted*/
				abort_executing_diag_cmd = FALSE;
			}
			masca_util_set_event(DIAG_CMD_ABORT);
		} else if (TRUE == abort_pending_diag_cmd) {
			p_diag_elem = masca_get_frst_diag_elem(PENDING);
			if (NULL != p_diag_elem) {
				masca_drive_mngr_abort(
					       p_diag_elem->acpptd_cmd.command);
				p_output->replied_command =
						p_diag_elem->acpptd_cmd.command;
				p_output->response_id =
					       p_diag_elem->acpptd_cmd.reply_id;
				p_output->reply_status = MASCA_ABORT;
				masca_empty_diag_lst_elem(p_diag_elem);
				masca_util_set_event(DIAG_CMD_ABORT);
			} else {
				/*all executing commands aborted*/
				abort_pending_diag_cmd = FALSE;
			}
		} else {
			/*Just abort the executing commands from drive manager
			* and keep them in the list*/
			p_diag_elem = masca_get_frst_diag_elem(EXECUTING);
			while (NULL != p_diag_elem) {
				p_diag_elem->elem_stat = PENDING;
				masca_drive_mngr_abort(
					       p_diag_elem->acpptd_cmd.command);
				p_diag_elem =
					    masca_get_frst_diag_elem(EXECUTING);
			}
		}
		abort_diag_cmd_search = FALSE;
		break;
	default:
		break;
	}
	return ret_err;
}


/**
 * \func    masca_diag_hndlr_abort
 *
 * This API aborts the execution of particular command corresponding
 * to the request int. The request is a unique int for each of the commands.
 * It is given when the command was sent to cmd_handler.
 *
 * \param req_id    : Request/command int which needs to be aborted
 *
 * \return
 *  MASCA_PROCESSED     : If the API is successfully executed.
 *  MASCA_INVALID_PARAM : If the particular request int is not pending or
 *                        executing.
 *
 */
extern enum masca_error masca_diag_hndlr_abort(const unsigned int req_id)
{
	enum masca_error ret_err = MASCA_INVALID_PARAM;
	struct masca_cmd_lst_elem *p_diag_elem;
	p_diag_elem = masca_get_diag_elem_by_id(req_id);
	if (NULL != p_diag_elem) {
		ret_err = MASCA_OK;
		if (EXECUTING == p_diag_elem->elem_stat)
			masca_drive_mngr_abort(p_diag_elem->
							acpptd_cmd.command);

		masca_empty_diag_lst_elem(p_diag_elem);
	}
	return ret_err;
}
