/*
 * MASCA Block Reader, handles the Data read request(READ(10))
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Dhanasekaran D <dhanasekaran.d@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/slab.h>
#include "masca_common_types.h"
#include "masca_helper.h"
#include "masca_event_config.h"
#include "masca_block_reader.h"
#include "masca_util_rspc.h"
#include "masca_ssi.h"
#include "masca_dma.h"

#define MODE2_CD		0x02
#define DATA_POS_MODE1_CD	4
#define DATA_POS_MODE2_CD	12

/* currently 6 sectors is defined to be the pick up error.
 * Any value below 5 has many number of failures in
 * repositioning. So this value has been considered as
 * optimum value */
#define CD_PICKUP_ERROR		6

#define IS_RNG_VALID(sect_strt, sect_cnt) \
			(((sect_strt) >= blk_rdr.cd_start_address)  \
			&& (((sect_strt) \
			+ (sect_cnt)) < blk_rdr.cd_end_address))

#define update_idx(idx, offset)			\
	do {					\
		idx += (offset);		\
		if (idx < 0)			\
			idx += DMA_BUF_SZ;	\
		else if (idx >= DMA_BUF_SZ)	\
			idx -= DMA_BUF_SZ;	\
	} while (0)

#define BLKRDR_TMOUT		5000
#define MASCA_BLKRD_MAX_RETRY	64
#define HEAD_REPOS_TH		(DMA_MAX_FRAMES / 2)

/* Force request threshold : sectors/s * 0.2s * drive speed
 * FORCE_REQ_TH = 75 sectors * 0.2s * 1 (1x speed) = 15 */
#define FORCE_REQ_TH 15
unsigned char drive_speed;/* drive speed : 1x, 2x and 4x speed */

/* CD sector header */
enum sector_header {
	SCT_MIN_POS = 0,
	SCT_SEC_POS,
	SCT_FRM_POS,
	SCT_MODE_POS
};

/* Requested sector in processed/unprocessed/partially present/new request
 * needs to be issues to the drive to fetch the blocks */
enum req_range {
	REQ_RANGE_NEW = 0,	/* New read request */
	REQ_RANGE_PRO,		/* In processed region */
	REQ_RANGE_PRO_UNPRO,	/* In both processed and unprocessed region */
	REQ_RANGE_PRO_NEW,	/* Partly in processed and further new req*/
	REQ_RANGE_UNPRO,	/* In unprocessed region */
	REQ_RANGE_UNPRO_NEW,	/* Partly in unprocessed and remain new req */
	REQ_RANGE_FORCE		/* Send a read request to CD drive compulsory */
};

struct masca_blk_req {
	unsigned int blks_fetched;
	unsigned char rty_count;
	unsigned int sct_start;
	unsigned int sct_end;
	unsigned int sct_cnt;
	unsigned int prev_sct_start;
	unsigned int prev_sct_cnt;
	unsigned int prev_sct_end;
	unsigned char state;
	unsigned char prev_state;
};

struct masca_blkrdr_info {
	struct mutex mtx_id;
	unsigned int cd_start_address;
	unsigned int cd_end_address;
	unsigned int blkrd_retry_cnt;
	int blkrd_retry_tmout;
	unsigned int rcv_status;
	unsigned int req_sect_start;
	unsigned int req_sect_end;
	unsigned int blks_copied;
	unsigned char  *p_appbuf;
};

struct masca_cded_info {
	unsigned int current_sector_start;
	unsigned char sect_to_fetch;
};

static bool masca_handle_cded_data(unsigned char * const p_data,
				bool dma_valid_data, unsigned int *sector);
static bool masca_store_sct_payload(const unsigned int sect_addr,
					unsigned char * const p_data);
static bool process_data(struct masca_blk_req * const p_sct_rd);
static unsigned char *detect_fsync(unsigned char *buf, unsigned int *rd_idx,
				const unsigned int wr_idx);
static unsigned int pending_data(unsigned int rd_idx, unsigned int wr_idx);
static inline bool is_wrap_around(unsigned int rd_idx, unsigned int wr_idx);
static bool masca_copy_to_appbuffer(const unsigned int block_address,
					const unsigned char * const p_buffer);
static int masca_blkrdr_task(void *thrd_data);

static struct masca_blkrdr_info blk_rdr;
static struct masca_cded_info cded_unit;
static struct task_struct *blkrdrthrd;
static unsigned char *frm_buf;

/* Accessed from 3 threads, masca_recvthrd, masca_blkrdr_task
 * and masca_dma_period_callback. so protected using
 * blkrdrcond_lock */
static unsigned char blkrdrcond;
static DEFINE_SPINLOCK(blkrdrcond_lock);
DECLARE_WAIT_QUEUE_HEAD(blkrdrwaitq);

static struct dma_buffer *pdma_buf;
static unsigned char sync_ptn[] = {	0x00, 0xFF, 0xFF, 0xFF,
					0xFF, 0xFF, 0xFF, 0xFF,
					0xFF, 0xFF, 0xFF, 0x00	};

typedef void (*blkrdr_state_handler)(struct masca_blk_req *);

/*block reader task state handlers*/
static void blkrdr_deinit_hdlr(struct masca_blk_req *rq);
static void blkrdr_abort_hdlr(struct masca_blk_req *rq);
static void blkrdr_blk_req(struct masca_blk_req *rq);
static void blkrdr_prepare(struct masca_blk_req *rq);
static void blkrdr_retry_fetch(struct masca_blk_req *rq);
static void blkrdr_read_data(struct masca_blk_req *rq);
static void blkrdr_idle(struct masca_blk_req *rq);
static void blkrdr_invalid(struct masca_blk_req *rq);

/*state table*/
blkrdr_state_handler blkrdr_hdlrs[] = {
		blkrdr_deinit_hdlr,
		blkrdr_abort_hdlr,
		blkrdr_idle,
		blkrdr_blk_req,
		blkrdr_prepare,
		blkrdr_retry_fetch,
		blkrdr_read_data,
		blkrdr_invalid
};

void masca_util_set_blkevent(const unsigned char evt_ptn)
{
	unsigned long flags;
	spin_lock_irqsave(&blkrdrcond_lock, flags);
	blkrdrcond |= evt_ptn;
	spin_unlock_irqrestore(&blkrdrcond_lock, flags);
	wake_up_interruptible(&blkrdrwaitq);
}

void masca_util_clr_blkevent(const unsigned char evt_ptn)
{
	unsigned long flags;
	spin_lock_irqsave(&blkrdrcond_lock, flags);
	blkrdrcond &= (~evt_ptn);
	spin_unlock_irqrestore(&blkrdrcond_lock, flags);
}

unsigned char masca_util_get_blkevent(void)
{
	unsigned long flags;
	unsigned char blkevt = 0x00;
	spin_lock_irqsave(&blkrdrcond_lock, flags);
	blkevt = blkrdrcond;
	spin_unlock_irqrestore(&blkrdrcond_lock, flags);
	return blkevt;
}

enum masca_error
masca_blkrdr_init(const struct masca_configuration * const p_config)
{
	enum masca_error ret_err = MASCA_OK;
	mutex_init(&blk_rdr.mtx_id);

	pdma_buf = masca_dma_get_buffer();
	if (pdma_buf) {
		masca_dma_set_clbks(&masca_util_set_blkevent);
		blk_rdr.blkrd_retry_tmout = p_config->blkrd_tmout;
		blk_rdr.blkrd_retry_cnt = (unsigned int)p_config->blkrd_rty_cnt;
		blkrdrthrd = kthread_run(masca_blkrdr_task, NULL,
				"masca_blkrdr");
		if (blkrdrthrd == NULL)
			ret_err = MASCA_NO_RESOURCE;
	} else {
		ret_err = MASCA_NO_RESOURCE;
		BLKRDR_TR_A("MASCA Block-Reader failed to get DMA buf!!");
	}
	/* Whenever an index has overlapped in DMA buffer,then
	 * copy the particular sector information to internal
	 * buffer */
	if (MASCA_OK == ret_err) {
		frm_buf = kzalloc((FRAME_SZ + sizeof(sync_ptn)),
					GFP_KERNEL);
		if (NULL == frm_buf) {
			ret_err = MASCA_NO_RESOURCE;
			BLKRDR_TR_A("memory failed to get DMA overlap buf!!");
		}
	}

	cded_unit.current_sector_start = 0;
	cded_unit.sect_to_fetch = 0;
	return ret_err;
}

void masca_blkrdr_deinit(void)
{
	masca_util_set_blkevent(BLKRDR_DEINIT);
	kthread_stop(blkrdrthrd);
	mutex_destroy(&blk_rdr.mtx_id);
	if (NULL != frm_buf)
		kfree(frm_buf);
}

void masca_blkrdr_abort(void)
{
	BLKRDR_TR_I("READ_10 aborted in blockreader\n");
	masca_util_set_blkevent(BLKRDR_ABORT);
}

/*
 * masca_blkrdr_cache_disable
 *
 * Bocke read cache enable/diaable
 *
 * blkrdr_cache_disable = true : Disable cache
 * blkrdr_cache_disable = false : Enable cache
 *
 */
extern void masca_blkrdr_cache_disable(bool blkrdr_cache_disable)
{
	/* TODO: Place holder for FUA bit impl. in READ_10 command */
	masca_blkrdr_flush();
}

/* Copy the processed data to the application buffer */
static bool copy_processed_data_to_app(struct masca_blk_req * const rq,
				const struct buf_water_mark wmark)
{
	int blk_cnt = 0;
	bool req_compl = false;
	int sct_idx = wmark.pro_end_idx -
		((wmark.pro_end_sct - blk_rdr.req_sect_start) * FRAME_SZ);

	if (sct_idx < 0) {/*Sector data overlapped*/
		sct_idx += DMA_BUF_SZ;
		BLKRDR_TR_D("Overlapped processed sct_idx:0x%x\n", sct_idx);
	}

	for (; (blk_cnt <= (blk_rdr.req_sect_end - blk_rdr.req_sect_start))
			&& (req_compl == false); blk_cnt++) {
		if ((sct_idx + FRAME_SZ) > DMA_BUF_SZ) {
			memcpy(frm_buf, pdma_buf->buf + sct_idx,
					DMA_BUF_SZ - sct_idx);
			memcpy(frm_buf + DMA_BUF_SZ - sct_idx, pdma_buf->buf,
					(FRAME_SZ - (DMA_BUF_SZ - sct_idx)));
			util_rspc_descramble_data((unsigned int *)frm_buf);
			req_compl = masca_handle_cded_data(frm_buf, true, NULL);
			BLKRDR_TR_D("Overlapped frm_buf copy\n");
		} else {
			req_compl = masca_handle_cded_data(
						pdma_buf->buf + sct_idx,
						true, NULL);
		}
		sct_idx += FRAME_SZ;

		if (sct_idx >= DMA_BUF_SZ)
			sct_idx -= DMA_BUF_SZ;
	}

	return req_compl;
}

/* Find the requested sector in dma buffer region */
static enum req_range
get_req_range_in_dma(struct masca_blk_req * const rq,
			const struct buf_water_mark wmark,
			unsigned int free_sz)
{
	enum req_range range = REQ_RANGE_NEW;
	enum req_range st_sct_range = REQ_RANGE_NEW;
	enum req_range end_sct_range = REQ_RANGE_NEW;

	BLKRDR_TR_D("Sector::req_start:%d pro_st:%d pro_end:%d unpro_end:%d\n",
		rq->sct_start, wmark.pro_st_sct, wmark.pro_end_sct,
		wmark.unpro_end_sct);

	/* Check whether the requested sector in DMA buf*/
	/* 1. Requested start sector in region */
	if ((rq->sct_start >
			(wmark.pro_end_sct + (FORCE_REQ_TH * drive_speed)))
		|| (rq->sct_start < wmark.pro_st_sct)
		|| (0 == wmark.pro_st_sct)
		|| (free_sz < (rq->sct_cnt * FRAME_SZ)))
		range = REQ_RANGE_FORCE;
	else if ((wmark.pro_st_sct <= rq->sct_start)
			&& (wmark.pro_end_sct >= rq->sct_start))
		st_sct_range = REQ_RANGE_PRO;
	else if ((wmark.pro_end_sct < rq->sct_start)
			&& (wmark.unpro_end_sct >= rq->sct_start))
		st_sct_range = REQ_RANGE_UNPRO;
	else
		if ((masca_ssi_is_on() == false))
			range = REQ_RANGE_FORCE;

	/* 2. Requested end sector in region */
	if ((wmark.pro_st_sct <= rq->sct_end)
			&& (wmark.pro_end_sct >= rq->sct_end))
		end_sct_range = REQ_RANGE_PRO;
	else if ((wmark.pro_end_sct < rq->sct_end)
			&& (wmark.unpro_end_sct >= rq->sct_end))
		end_sct_range = REQ_RANGE_UNPRO;
	else
		if (REQ_RANGE_FORCE != range)
			end_sct_range = REQ_RANGE_NEW;

	/* 3. Requested blocks region */
	if (st_sct_range == REQ_RANGE_PRO) {
		if (end_sct_range == REQ_RANGE_PRO)
			range = REQ_RANGE_PRO;
		else if (end_sct_range == REQ_RANGE_UNPRO)
			range = REQ_RANGE_PRO_UNPRO;
		else
			range = REQ_RANGE_PRO_NEW;
	} else if (st_sct_range == REQ_RANGE_UNPRO) {
		if (end_sct_range == REQ_RANGE_UNPRO) {
			range = REQ_RANGE_UNPRO;
			if ((rq->sct_end > wmark.unpro_end_sct)
					&& (masca_ssi_is_on() == false))
				range = REQ_RANGE_FORCE;
		} else if (end_sct_range == REQ_RANGE_NEW) {
			range = REQ_RANGE_UNPRO_NEW;
			if (masca_ssi_is_on() == false)
				range = REQ_RANGE_FORCE;
		} else {
			range = REQ_RANGE_NEW;
		}
	}

	BLKRDR_TR_I("%s = %d\n", __func__, range);

	return range;
}

/**
 * masca_blkrdr_read_blocks() - block reader request
 * @return error code
 * @param sector_address - start sector address to fetch
 * @param block_count - sector count to fetch
 * @param p_buffer - application buffer to store the sector data
 *
 * Entry point for Block-READ request from the main task. Updates the read
 * request details and triggers the blkrdr_task with BLK_RQ state.
 */
enum masca_error masca_blkrdr_read_blocks(const unsigned int sector_address,
		const unsigned int block_count, unsigned char * const p_buffer)
{
	BLKRDR_TR_I("In %s sector_address:0x%x, block_count:%d\n", __func__,
			sector_address, block_count);

	mutex_lock(&blk_rdr.mtx_id);
	blk_rdr.req_sect_start = sector_address;
	blk_rdr.req_sect_end = sector_address + block_count;
	blk_rdr.blks_copied = 0;
	blk_rdr.p_appbuf = p_buffer;
	mutex_unlock(&blk_rdr.mtx_id);

	masca_util_set_blkevent(BLKRDR_BLK_RQ);

	return MASCA_ACCEPTED;
}

void masca_blkrdr_flush(void)
{
	mascassi_dma_init_buf_idx();
}

void masca_blkrdr_get_play_pos(unsigned int * const p_play_pos,
		unsigned char * const p_block_count)
{
	*p_play_pos = cded_unit.current_sector_start;
	*p_block_count = cded_unit.sect_to_fetch;
}

void masca_blkrdr_set_cd_bounds(const unsigned int sect_start_addr,
		const unsigned int end_sect_addr)
{
	blk_rdr.cd_start_address = sect_start_addr;
	blk_rdr.cd_end_address = end_sect_addr;
}

static void masca_compute_sect_fetch(const unsigned int sect_start,
		const unsigned char sect_cnt)
{
	unsigned int calibrated_start;
	if ((sect_start >= blk_rdr.cd_start_address)
		&& (sect_start <= blk_rdr.cd_end_address)) {
		calibrated_start = sect_start;
		if (sect_start >= CD_PICKUP_ERROR)
			calibrated_start = sect_start - CD_PICKUP_ERROR;
		/*The minimum address of fetch is 150 and the pickup error is
		 far smaller so no need to check with cd start address*/
		cded_unit.current_sector_start = calibrated_start;
		cded_unit.sect_to_fetch = sect_cnt + CD_PICKUP_ERROR;
		/*Check for upper bound of CD*/
		if ((cded_unit.current_sector_start + cded_unit.sect_to_fetch)
				> blk_rdr.cd_end_address) {
			cded_unit.sect_to_fetch
					= blk_rdr.cd_end_address
					- cded_unit.current_sector_start;
		}
	}
}

static void masca_register_blkread_sector(const unsigned int sect_addr)
{
	unsigned int bit_to_set;
	if ((sect_addr >= blk_rdr.req_sect_start)
		&& (sect_addr <= blk_rdr.req_sect_end)) {
		/*set the corresponding bit to indicate successful
		 * reception of the sector data*/
		bit_to_set = sect_addr - blk_rdr.req_sect_start;
		blk_rdr.rcv_status |= BIT(bit_to_set);
	}
}

static bool masca_copy_to_appbuffer(const unsigned int block_address,
					const unsigned char * const p_buffer)
{
	bool is_buf_full = false;
	unsigned int end_addr;
	unsigned int start_addr;
	unsigned int sect_offset;
	start_addr = blk_rdr.req_sect_start;
	end_addr = blk_rdr.req_sect_end;
	if ((block_address >= start_addr) && (block_address < end_addr)) {
		sect_offset = block_address - start_addr;
		sect_offset *= SECTOR_SIZE_CD; /*Block Size*/

		memcpy((void *)(&blk_rdr.p_appbuf[sect_offset]),
				(void *)p_buffer, SECTOR_SIZE_CD);

		blk_rdr.blks_copied++;
		if (blk_rdr.blks_copied >=
			(blk_rdr.req_sect_end - blk_rdr.req_sect_start)) {
			is_buf_full = true;
			mutex_lock(&blk_rdr.mtx_id);
			blk_rdr.req_sect_start = 0;
			blk_rdr.req_sect_end = 0;
			mutex_unlock(&blk_rdr.mtx_id);
		}
	}
	return is_buf_full;
}

static bool masca_store_sct_payload(const unsigned int sect_addr,
						unsigned char * const p_data)
{
	masca_register_blkread_sector(sect_addr);
	return masca_copy_to_appbuffer(sect_addr, p_data);
}

static void masca_get_rty_blocks(unsigned int * const p_sct_strt,
						unsigned int * const p_sct_cnt)
{
	unsigned int min_sector_not_rcvd = 0xFF;
	unsigned int max_sector_not_rcvd = 0;
	unsigned int offset = 0;
	unsigned int bit;
	unsigned int bit_pattern;
	unsigned int bit_count;

	bit_count = (blk_rdr.req_sect_end - blk_rdr.req_sect_start);

	/*Compute the range over here*/
	bit_pattern = blk_rdr.rcv_status;

	for (bit = 0; bit < bit_count; bit++) {
		if ((bit_pattern & BIT(bit)) == 0) {
			if (bit < min_sector_not_rcvd)
				min_sector_not_rcvd = bit;
			if (bit > max_sector_not_rcvd)
				max_sector_not_rcvd = bit;
		}
	}

	if (min_sector_not_rcvd <= max_sector_not_rcvd) {
		min_sector_not_rcvd = blk_rdr.req_sect_start
				+ min_sector_not_rcvd;
		max_sector_not_rcvd = blk_rdr.req_sect_start
				+ max_sector_not_rcvd;
		offset = max_sector_not_rcvd - min_sector_not_rcvd + 1;
	} else {
		min_sector_not_rcvd = 0;
		max_sector_not_rcvd = 0;
		offset = 0;
	}
	*p_sct_strt = min_sector_not_rcvd;
	*p_sct_cnt = offset;
}

static bool masca_is_blkread_sector_rcvd(const unsigned int sect_addr)
{
	unsigned int bit_to_check;
	bool is_sector_prsnt = true;
	if ((sect_addr >= blk_rdr.req_sect_start) && (sect_addr
			<= blk_rdr.req_sect_end)) {
		bit_to_check = sect_addr - blk_rdr.req_sect_start;
		if (((blk_rdr.rcv_status) & BIT(bit_to_check)) == 0)
			is_sector_prsnt = false;
	}
	return is_sector_prsnt;
}

void masca_get_read_sct_info(struct masca_blk_req *rq)
{
	mutex_lock(&blk_rdr.mtx_id);
	rq->sct_start = blk_rdr.req_sect_start;
	rq->sct_end = blk_rdr.req_sect_end - 1;
	rq->sct_cnt = blk_rdr.req_sect_end - blk_rdr.req_sect_start;
	blk_rdr.rcv_status = 0;
	mutex_unlock(&blk_rdr.mtx_id);
}

void masca_init_rq(struct masca_blk_req *rq)
{
	memset(rq, 0, sizeof(struct masca_blk_req));
}

/* Fill the valid received sectors information */
static void masca_update_sector_info(unsigned int sct_addr, unsigned int rd_idx)
{
	static unsigned int prev_sct_addr;
	struct buf_water_mark wmark;
	unsigned long flags;
	unsigned int data_sz;

	spin_lock_irqsave(&pdma_buf->lock, flags);
	wmark = pdma_buf->wmark;
	wmark.pro_end_sct = sct_addr;
	wmark.pro_end_idx = rd_idx;

	/*HEAD reposition handling*/
	if ((sct_addr < wmark.pro_st_sct)
		|| (wmark.unpro_end_sct < wmark.pro_end_sct)
		|| ((prev_sct_addr + 1) != sct_addr)
		|| (0 == wmark.pro_st_sct)) {
		wmark.pro_st_sct = sct_addr;
		wmark.pro_st_idx = rd_idx;
		data_sz = pending_data(wmark.pro_end_idx,
					atomic_read(&pdma_buf->wr_idx));
		wmark.unpro_end_sct = wmark.pro_end_sct + data_sz/FRAME_SZ;
		BLKRDR_TR_D("HEAD repos: psi=%u, pss=%d upes=%d\n",
				wmark.pro_st_idx, wmark.pro_st_sct,
				wmark.unpro_end_sct);
	}
	prev_sct_addr = sct_addr;
	pdma_buf->wmark = wmark;
	spin_unlock_irqrestore(&pdma_buf->lock, flags);
}

/**
 * masca_handle_cded_data() - processes the raw cded data
 * @param p_data - DMA pointer to process the data
 * @param dma_valid_data - valid sector, de-scrambling not required
 * @param sector[OUT] - handled sector address
 * @return read request is complete/in progress
 *
 * Descrambles the data and if its a desired one then applies CRC32 calculation.
 * RSPC correction algorithm is applied when data is corrupt, then stores the
 * sector into the application buffer.
 */
bool masca_handle_cded_data(unsigned char * const p_data, bool dma_valid_data,
				unsigned int *sector)
{
	unsigned int sect_addr;
	unsigned int sect_offset;
	unsigned int crc_result;
	unsigned char *p_sector;
	bool rc = false;
	bool do_store = true;

	p_sector = p_data;

	if (!dma_valid_data)
		util_rspc_descramble_data((unsigned int *)p_sector);

	p_sector +=  sizeof(sync_ptn);

	sect_addr = GET_LBA(BCD_TO_HEX(p_sector[SCT_MIN_POS]),
				BCD_TO_HEX(p_sector[SCT_SEC_POS]),
				BCD_TO_HEX(p_sector[SCT_FRM_POS]));
	if (sector)
		*sector = sect_addr;

	/* Error handling:
	 * A standard CD_ROM disc contain a start address of 150th sector,
	 * but drive can able to fetch from 144th sector for 150th sector
	 * request. Hence, the sector address starts from 144 to 149 are
	 * CD pickup error sector and no need to give RETRY for these sectors.
	 * Check whether the received sector is in the range of disc capacity,
	 * if not, then trigger a retry event */
	if (!(((blk_rdr.cd_start_address - CD_PICKUP_ERROR) <= sect_addr) &&
		(blk_rdr.cd_end_address >= sect_addr))) {
		BLKRDR_TR_I("OutofRange sct=0x%x CDstart=0x%x CDend=0x%x\n",
				sect_addr, blk_rdr.cd_start_address,
				blk_rdr.cd_end_address);
		if (sector)
			*sector = 0;
		return false;
	}

	if (false != masca_is_blkread_sector_rcvd(sect_addr))
		return rc; /*requested sector already received or not in range*/

	if (MODE2_CD == p_sector[SCT_MODE_POS]) {
		crc_result = UTIL_rspc_calc_crc32(
				(unsigned short *)p_sector, CD_MODE2);
		sect_offset = DATA_POS_MODE2_CD;
	} else {
		crc_result = UTIL_rspc_calc_crc32(
				(unsigned short *)p_sector, CD_MODE1);
		sect_offset = DATA_POS_MODE1_CD;
	}

	/*perform error correction*/
	if (crc_result != 0) {
		if (MASCA_OK != util_rspc_correction((unsigned short *)p_sector,
					true, CDED_QOC_HIGH)) {
			do_store = false;
			BLKRDR_TR_I("rspc correction failed sect_addr:0x%x!!\n",
					sect_addr);
			/*Set a retry event to fetch the failed sector*/
			if ((blk_rdr.req_sect_start <= sect_addr)
				&& (blk_rdr.req_sect_end >= sect_addr))
				masca_util_set_blkevent(BLKRDR_RETRY_FETCH);
		}
	}

	if (do_store == true) {
		rc = masca_store_sct_payload(sect_addr, &p_sector[sect_offset]);
		BLKRDR_TR_D("sector received: 0x%x\n", sect_addr);
	}

	return rc;
}

/**
 * detect_fsync() - detects the frame sync pattern in the SSI raw data
 * @return - NULL when no SYNC detected otherwise SYNC start pointer
 * @param buf - SYNC to be identified in this buffer
 * @param read_idx - Start index in pdma_buf when entered [IN/OUT]
 * @param wr_idx  - End index to identify the SYNC pattern
 *
 * identifies the frame sync pattern "00 FF FF FF FF FF FF FF FF FF FF 00"
 */
static unsigned char *detect_fsync(unsigned char *buf, unsigned int *read_idx,
					const unsigned int wr_idx)
{
	unsigned int next_idx;
	unsigned int rd_idx = *read_idx;
	unsigned char *p_sync = NULL;
	unsigned char *p_buf = NULL;

	if (!is_wrap_around(rd_idx, rd_idx + FRAME_SZ))
		if (0 == memcmp(sync_ptn, buf + rd_idx, sizeof(sync_ptn)))
			p_sync = buf + rd_idx;

	while (NULL == p_sync && pending_data(rd_idx, wr_idx) >= FRAME_SZ) {
		next_idx = rd_idx + 1;
		if (is_wrap_around(rd_idx, next_idx))
			next_idx -= DMA_BUF_SZ;

		if ((buf[rd_idx] == 0x00) && (buf[next_idx] == 0xFF)) {
			if (is_wrap_around(rd_idx, rd_idx + FRAME_SZ)) {
				/*probable frame is overlapped*/
				memcpy(frm_buf, buf + rd_idx,
						DMA_BUF_SZ - rd_idx);
				memcpy(frm_buf + (DMA_BUF_SZ - rd_idx),	buf,
						((FRAME_SZ) -
						(DMA_BUF_SZ - rd_idx)));
				p_buf = frm_buf;
				BLKRDR_TR_I("Probable Frame is overlapped\n");
			} else
				p_buf = buf + rd_idx;

			if (0 == memcmp(sync_ptn, p_buf, sizeof(sync_ptn)))
				p_sync = p_buf;
		}
		if (p_sync == NULL)
			update_idx(rd_idx, 1);
	}
	*read_idx = rd_idx;
	return p_sync;
}

static inline bool is_wrap_around(unsigned int rd_idx, unsigned int wr_idx)
{
	return (wr_idx < rd_idx) || (wr_idx >= DMA_BUF_SZ);
}

static unsigned int pending_data(unsigned int rd_idx, unsigned int wr_idx)
{
	return ((wr_idx >= rd_idx) ?
			(wr_idx - rd_idx) : (DMA_BUF_SZ - rd_idx + wr_idx));
}

/**
 * process_data() - Processes one DMA period of data
 * @return read is success or in progress
 * @param rq - read request details
 *
 * Checks for the valid frames, process the data and stores the expected
 * sector into the application buffer.
 */
static bool process_data(struct masca_blk_req * const rq)
{
	unsigned int wr_idx = atomic_read(&pdma_buf->wr_idx);
	unsigned int rd_idx = atomic_read(&pdma_buf->rd_idx);
	bool req_compl = false;
	unsigned char *p_frame;
	unsigned int sector;

	/*process only one period of data per call*/
	if (pending_data(rd_idx, wr_idx) >= DMA_PERIOD_SZ) {
		if (!is_wrap_around(rd_idx, rd_idx + DMA_PERIOD_SZ))
			wr_idx = rd_idx + DMA_PERIOD_SZ;
		else
			wr_idx = rd_idx + DMA_PERIOD_SZ - DMA_BUF_SZ;
	}

	while ((false == req_compl) &&
		(pending_data(rd_idx, wr_idx) >= FRAME_SZ)) {
		p_frame = detect_fsync(pdma_buf->buf, &rd_idx, wr_idx);
		if (p_frame) {
			req_compl = masca_handle_cded_data(p_frame, false,
								&sector);
			if (sector) {
				masca_update_sector_info(sector, rd_idx);
				rd_idx += FRAME_SZ;
				rq->blks_fetched++;
			} else {
				rd_idx += sizeof(sync_ptn);
			}
		}
		atomic_set(&pdma_buf->rd_idx, rd_idx);
	}
	return req_compl;
}

/* Get the free buffer size from dma buffer
 * with respect to start and end index */
static unsigned int get_free_buf_sz(unsigned int start_idx,
					unsigned int end_idx)
{
	unsigned int free_sz;
	if (start_idx <= end_idx)
		free_sz = DMA_BUF_SZ - (end_idx - start_idx);
	else /*wrap around*/
		free_sz = start_idx - end_idx;
	return free_sz;
}

/**
 * blkrdr_deinit_hdlr() - deinit handler
 * @return none
 * @param rq - unused
 *
 */
static void blkrdr_deinit_hdlr(struct masca_blk_req *rq)
{
	masca_ssi_off();
}

static void req_completed(struct masca_blk_req *rq)
{
	rq->prev_sct_end = rq->sct_start + rq->sct_cnt - 1;
	rq->prev_sct_start = rq->sct_start;
	rq->prev_sct_cnt = rq->sct_cnt;
	rq->blks_fetched = 0;
	rq->sct_cnt = 0;
}

/**
 * blkrdr_abort_hdlr() - read command abort handler
 * @return none
 * @param rq - read request details
 *
 * Abort request will be triggered by the SCSI FW (e.g. Timeout) and the same is
 * forwarded by the masca main task
 */
static void blkrdr_abort_hdlr(struct masca_blk_req *rq)
{
	masca_util_clr_blkevent(BLKRDR_ABORT);
	masca_util_clr_event(BLOCK_READ_REQ);
	masca_util_set_blkevent(BLKRDR_IDLE);
	rq->sct_start = 0;
	rq->sct_cnt = 0;
}

/**
 * blkrdr_blk_req() - new read request
 * @return none
 * @param rq - read request details
 *
 * READ_10 request is received by MASCA main task from SCSI handler.
 * This state is the starting point for fetching the data. This triggers
 * PREPARE state.
 */
static void blkrdr_blk_req(struct masca_blk_req *rq)
{
	unsigned long flags;
	int free_sz;
	struct buf_water_mark wmark;
	bool req_sucess = false;
	enum req_range range = REQ_RANGE_NEW;

	masca_util_clr_blkevent(BLKRDR_BLK_RQ | BLKRDR_IDLE);
	masca_init_rq(rq);
	masca_get_read_sct_info(rq);

	spin_lock_irqsave(&pdma_buf->lock, flags);
	wmark = pdma_buf->wmark;
	free_sz = get_free_buf_sz(wmark.pro_st_idx,
			atomic_read(&pdma_buf->wr_idx));
	spin_unlock_irqrestore(&pdma_buf->lock, flags);

	range = get_req_range_in_dma(rq, wmark, free_sz);
	if (range == REQ_RANGE_PRO) {
		req_sucess = copy_processed_data_to_app(rq, wmark);
		if (false == req_sucess)
			range = REQ_RANGE_FORCE;
	} else if ((range == REQ_RANGE_PRO_UNPRO)
			|| (range == REQ_RANGE_PRO_NEW)) {
		req_sucess = copy_processed_data_to_app(rq, wmark);
	}

	if (req_sucess) {
		req_completed(rq);
		masca_util_set_event(BLOCK_READ_COMPLETE);
		BLKRDR_TR_I("BLOCK_READ_COMPLETE without drive access\n");
	} else {
		pdma_buf->prefetch = false;
		if (REQ_RANGE_FORCE == range) {
			rq->rty_count++; /*force HEAD reposition*/
			masca_util_set_blkevent(BLKRDR_PREPARE);
		} else {
			masca_util_set_blkevent(BLKRDR_READ_DATA);
		}
	}
}

/**
 * blkrdr_prepare() - prepare for read
 * @return none
 * @param rq - read request details
 *
 * 1. Computes the sectors to be fetched considering the drive behaviour.
 * 2. Issues Play command to the MASCA drive
 * 3. Switch on SSI and DMA for fetching data
 */
static void blkrdr_prepare(struct masca_blk_req *rq)
{
	masca_util_clr_blkevent(BLKRDR_PREPARE);
	masca_compute_sect_fetch(rq->sct_start, rq->sct_cnt);

	if ((masca_ssi_is_on() == false) || (rq->rty_count > 0)) {
		masca_util_set_event(BLOCK_READ_REQ);
		masca_ssi_off();
		masca_ssi_on();
	}
}

static bool is_retry_required(struct masca_blk_req const *rq)
{
	struct buf_water_mark wmark;
	unsigned long flags;
	bool retry = false;

	spin_lock_irqsave(&pdma_buf->lock, flags);
	wmark = pdma_buf->wmark;

	/* HEAD reposition takes time when the drive is already rotating,
	 * even though SSI & DMA off & on done.To avoid frequent RETRY request
	 * HEAD_REPOS_TH sectors are processed before the next RETRY is issued*/
	if (wmark.pro_end_sct > (rq->sct_end + HEAD_REPOS_TH)
		&& (blk_rdr.blks_copied < rq->sct_cnt)
		&& ((wmark.pro_end_sct - wmark.pro_st_sct) > PERIOD_SZ)
		&& (wmark.pro_end_sct > (rq->prev_sct_start + HEAD_REPOS_TH))) {
		/* once the valid received sector crossed
		 * the requested sectors, then go for RETRY */
		BLKRDR_TR_I("BLKRDR_RETRY_FETCH - crossed vaild sect\n");
		retry = true;
	} else if ((wmark.pro_end_sct < (rq->sct_start - CD_PICKUP_ERROR))
		&& ((wmark.pro_end_sct + HEAD_REPOS_TH) < rq->sct_start)
		&& (wmark.pro_end_sct > (rq->prev_sct_start + HEAD_REPOS_TH))) {
		/* Checking whether received sector is out of DMA buffer
		 * from the requested sector */
		BLKRDR_TR_I("BLKRDR_RETRY_FETCH - recv sect out of buf\n");
		retry = true;
	} else if (rq->blks_fetched > HEAD_REPOS_TH) {
		retry = true;
	}

	spin_unlock_irqrestore(&pdma_buf->lock, flags);
	return retry;
}

static bool is_continue_processing(struct masca_blk_req const *rq)
{
	return (((rq->sct_end >= pdma_buf->wmark.pro_end_sct)
			|| ((rq->prev_sct_end+1) == rq->sct_start))
		&& (masca_ssi_is_on() == false)
		&& (blk_rdr.blks_copied < rq->sct_cnt));
}

/**
 * blkrdr_read_data() - Read & process the data
 * @return none
 * @param rq - read request details
 *
 * Triggered by the I2S module when DMA period callback is called. Read data
 * will be processed and based on the state different state change triggered.
 */
static void blkrdr_read_data(struct masca_blk_req *rq)
{
	bool is_req_compl;
	static unsigned int retry_cnt;
	unsigned int fetch_sct_cnt = rq->blks_fetched;

	masca_util_clr_blkevent(BLKRDR_READ_DATA);
	is_req_compl = process_data(rq);

	if (is_req_compl) {
		BLKRDR_TR_I("BLOCK_READ_COMPLETE\n");
		req_completed(rq);
		pdma_buf->prefetch = true;
		retry_cnt = 0;
		masca_util_set_event(BLOCK_READ_COMPLETE);
		masca_util_clr_blkevent(BLKRDR_READ_DATA); /*clear again*/
	} else if (fetch_sct_cnt == rq->blks_fetched)/*Invalid data*/ {
		retry_cnt++;
		if (retry_cnt >= MASCA_BLKRD_MAX_RETRY) {
			BLKRDR_TR_I("BLKRDR_RETRY_FETCH - MAX retry\n");
			masca_util_set_blkevent(BLKRDR_RETRY_FETCH);
			retry_cnt = 0;
		}
	} else if (is_retry_required(rq)) {
		masca_util_set_blkevent(BLKRDR_RETRY_FETCH);
	} else if (is_continue_processing(rq)) {
		masca_util_set_blkevent(BLKRDR_READ_DATA);
	} else {
		BLKRDR_TR_I("Wait for DMA event\n");
	}
}

/**
 * blkrdr_retry_fetch() - Retry the read request (complete/partial)
 * @return none
 * @param rq - read request details
 *
 * Retry will be done when timeout happens or no SYNC detected for predefined
 * amount of data received. This state will reposition the drive and try to
 * refetch the missing sectors.
 */
static void blkrdr_retry_fetch(struct masca_blk_req *rq)
{
	masca_util_clr_blkevent(BLKRDR_RETRY_FETCH);
	if (rq->rty_count < blk_rdr.blkrd_retry_cnt) {
		rq->rty_count++;
		rq->blks_fetched = 0;
		masca_get_rty_blocks(&rq->sct_start, &rq->sct_cnt);
		if (IS_RNG_VALID(rq->sct_start, rq->sct_cnt))
			masca_util_set_blkevent(BLKRDR_PREPARE);

		BLKRDR_TR_I("retry blocks start:%d cnt:%d",
				rq->sct_start, rq->sct_cnt);
	} else {
		pdma_buf->prefetch = true;
		masca_util_set_event(BLOCK_READ_FAILED);
		masca_util_set_blkevent(BLKRDR_IDLE);
		BLKRDR_TR_E("MASCA: BLOCK_READ_FAILED\n");
	}
}

/**
 * blkrdr_idle() - No further read request
 * @return none
 * @param rq - read request details
 *
 * Read request completed/aborted and state changed to idle, which switch off
 * the I2S & DMA. This state will be triggered by either self or I2S module.
 * I2S module will trigger this state when pre-fetching is complete.
 */
static void blkrdr_idle(struct masca_blk_req *rq)
{
	masca_ssi_off();
	masca_util_clr_blkevent(BLKRDR_IDLE);
	masca_util_set_event(BLOCK_READER_IDLE);
	/*continue processing the data*/
	if (false == pdma_buf->prefetch && rq->blks_fetched < rq->sct_cnt)
		masca_util_set_blkevent(BLKRDR_READ_DATA);
}

/**
 * blkrdr_invalid() - No valid state triggered
 * @return none
 * @param rq - read request details
 *
 * No valid state set in the block reader. This function is called when timeout
 * occurs. PH for timeout handling!!
 */
static void blkrdr_invalid(struct masca_blk_req *rq)
{
	BLKRDR_TR_D("Timeout/Invalid block reader state!!\n");
}

/*Get the block reader state based on priority*/
inline int get_state(unsigned char state)
{
	int bit;
	int cnt = sizeof(blkrdr_hdlrs)/sizeof(blkrdr_state_handler) - 1;
	/*exclude invalid state from the handler count*/
	for (bit = 0; bit < cnt; bit++) {
		if ((BIT(bit) & state) != 0)
			break;
	}
	BLKRDR_TR_I("state:%d\n", bit);
	return bit;
}

/**
 * masca_blkrdr_task() - block reader task
 * @return error code
 * @param thrd_data - unused
 *
 * Complete DATA fetching is handled as a state machine. Task waits for the
 * data request from main task and switch-on the SSI & SDMA for fetching data.
 * SSI module triggers when data is available and the state machine works
 * accordingly.
 */
int masca_blkrdr_task(void *thrd_data)
{
	struct masca_blk_req rq;
	long rc;
	long timeout = msecs_to_jiffies(BLKRDR_TMOUT);
	masca_init_rq(&rq);

	while (!kthread_should_stop()) {
		rc = wait_event_interruptible_timeout(blkrdrwaitq,
				(masca_util_get_blkevent()) & 0xFF, timeout);
		/*external interrupt to terminate the task*/
		if (-ERESTARTSYS == rc)
			break;
		rq.state = masca_util_get_blkevent();
		/*DMA timeout occurred*/
		if ((0 == rc)
			&& (rq.prev_state & (BLKRDR_PREPARE|BLKRDR_READ_DATA)))
			rq.state |= BLKRDR_RETRY_FETCH;

		blkrdr_hdlrs[get_state(rq.state)](&rq);
		if (rq.state & (~BLKRDR_IDLE)) /*other than idle state*/
			rq.prev_state = rq.state;
	}
	return 0;
}
