/*
 * MASCA ATAPI Bridge
 *
 * Copyright (C) 2013 ADIT Corporation
 * Authors: Saurabh Arora <saurabh.arora@in.bosch.com>
 *          Ramesh Ramachandran <ramesh.ramachandran@in.bosch.com>
 *          Mahendran Kuppusamy <mahendran.kuppusamy@in.bosch.com>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <scsi/scsi_cmnd.h>
#include <scsi/scsi.h>
#include <linux/vmalloc.h>
#include <linux/cdrom.h>
#include "masca_common_types.h"
#include "masca_dispatch.h"
#include "masca_atapi_bridge.h"
#include "masca_scsi_handler.h"
#include "masca_sm.h"
#include "masca_hal.h"
#include "masca_blkdev.h"
#include "masca_sysfs.h"
#include "masca_interpreter.h"

/* ATA/ATAPI event commands */
#define ATAPI_DEVICE_DIAG	0x90  /* Execute device diagnostics */

#define SESSION_INFO_LNG	0x0A
#define SESSION_HEADER_LNG	0x02
#define ATAPI_SCAN_LBA		0x00
#define ATAPI_SCAN_MSF		0x40
#define ATAPI_SCAN_TRACK	0x80
#define ATAPI_SCAN_TYPECAST	0xc0
#define READ_CAPACITY_RESP_LEN	8

/* Direction byte for scan */
#define FWD			0x00
#define BWD			0x10
#define FFWD			0x20
#define FBWD			0x30
#define ATAPI_SCAN_DIR_TYPECAST	0x10

#define ATAPI_PAUSE_TYPECAST	0x01
#define ATAPI_TOC_MSF_TYPECAST	0x02
#define TOC_FORMAT_TRACK_TOC	0x00
#define TOC_FORMAT_SESSION_INFO	0x01
#define TOC_FORMAT_FULL_TOC	0x02
#define TOC_FORMAT_CD_TEXT	0x05
#define TOC_FORMAT_TYPECAST	0x0F
#define TOC_CTRL_BYTE_REQ	0x40

#define SIZE_OF_TOC_HDR		4
#define SIZE_OF_CDTEXT_HDR	4
#define ATAPI_RESERVED		0
#define ATAPI_PACK_SIZE		18
#define ATAPI_PACK_PAYLOAD_SIZE	12

#define CHK_SUM_BYTE		0x00

#define ATAPI_TITLE		0x80
#define ATAPI_AUTHOR		0x81
#define ATAPI_ALBUM		0x83

/* Media status */
#define ATAPI_NO_MEDIA			0x42
#define ATAPI_MEDIA_INSERTED		0x60
#define ATAPI_MEDIA_EJECT_PROCESSED	0x48
#define ATAPI_NO_UPDATE			0x40

/* Full TOC variables */
#define FULL_TOC_TRK_SZ		11
#define SESS_HEADER		3
#define SESS_TAIL		1
#define HYBRID_DISC_HEADER	1
#define TOC_REQUEST_SIZE	12
#define FULL_TOC_REQ		0x02
#define FULL_TOC_SUB_REQ	0x80
#define TOC_LEAD_OUT_LBA	0xAA
#define TOC_START_LBA_OFFSET	150

/*LOG SENSE - TEMPERATURE LOG PAGE*/
#define LOG_PAGE_TEMPERATURE_PAGE	0x0D
#define REMAINING_TEMP_PAGE_LENGTH	0xC

#define LOG_TEMPERATURE_OFFSET	100 /* Log temperature offset for application */
#define LOG_SENSE_SP_MASK	0x01 /* Parameter Pointer Control (PPC) bit */
#define LOG_SENSE_PPC_MASK	0x02 /* Save Parameters (SP) bit */
#define LOG_SENSE_PC_MASK	0x0f /* Page code field */
#define LOG_SENSE_PC_CHECK	0x0d /* Log sense page code check */
#define LOG_SENSE_RESP_FORMAT	0x01
#define LOG_SENSE_RESP_PARA_CODE	0x01 /* log sense parameter code resp */
#define LOG_SENSE_RESP_LENGTH	0x02
#define RECEIVE_DIAG_PAGE_LENGTH	0x02 /* Default page length */
#define RECEIVE_DIAG_PAGE_CODE	0x80
#define READ_DISC_INFO_LENGTH	0x0a
#define READ_DISC_COMPLETE_SESSION	0x0e
#define READ_DISC_INCOMPLETE_SESSION	0x01
#define READ_DISC_UNRESTRICTED_USE	0x00
#define READ_DISC_RESP_CDROM_XA_DISC	0x20	/*CD-ROM XA Disc*/
#define READ_DISC_RESP_CDDA_DISC	0x00	/*CD-DA or MIXED mode Disc*/
#define MSF_RESP_LAST_TRK_SECOND_OFFSET 2
#define RESUME_PLAYBACK_MSF	0xffffff

enum scsi_command_cbd { /* command byte descriptor */
	LOG_SENSE_PPC_SP_BYTE = 1,/* PPC and SP bits*/
	LOG_SENSE_PAGE_CODE = 2,
	LOG_SENSE_LENGTH = 7,
	PLAY_MSF_START_MIN = 3,
	PLAY_MSF_START_SEC,
	PLAY_MSF_START_FRM,
	PLAY_MSF_END_MIN,
	PLAY_MSF_END_SEC,
	PLAY_MSF_END_FRM,
	PLAY_AUDIO_LENGHT = 7,
	PLAY_START_TRK_NO = 4,
	PLAY_END_TRK_NO = 7,
	READ_TOC_FORMAT = 2,
	READ_TOC_TRACK_NO = 6,
	READ_TOC_CTRL = 9,
	DIAG_COMMAND_SELECT = 1
};

static bool masca_updt_track_text(unsigned char * const p_cd_text_buf,
					unsigned char * const p_trk_text_buf,
					unsigned int * const p_buf_length,
					unsigned short trk_text_length,
					const unsigned char trk_num);

static unsigned short masca_updt_txt_lng(unsigned short cd_text_lng,
					unsigned short text_length);

/*used for cd text writing*/
static unsigned short   cursor_pos = (0);
static unsigned char    sequence_no;
static unsigned char    bytes_left_over;
static unsigned char    pack_type;
static bool  updt_header = TRUE;

static bool g_masca_toc_msf_flag = FALSE;

static inline unsigned int masca_create_lba_from_cdb(unsigned char *cdb)
{
	return (cdb[READ10_REQ_LBA+0x00] << BYTE_4_START) |
		(cdb[READ10_REQ_LBA+0x01] << BYTE_3_START) |
		(cdb[READ10_REQ_LBA+0x02] << BYTE_2_START) |
		cdb[READ10_REQ_LBA+0x03];
}

static void masca_get_scan_command(const unsigned char * const p_buf,
		struct masca_cmd_params * const p_decode_scan_params)
{
	/* Direction byte1 */
	switch (p_buf[1] & ATAPI_SCAN_DIR_TYPECAST) {
	case FWD:
		p_decode_scan_params->command = FWD_MSF;
		break;
	case BWD:
		p_decode_scan_params->command = BWD_MSF;
		break;
	/* The below two options are not used, to use change the
	 * ATAPI_SCAN_DIR_TYPECAST flag accordingly
	 */
	case FFWD:
		p_decode_scan_params->command = FFWD_MSF;
		break;
	/* Fast backward : scan backwards and stops once the starting
	 * point of current scanning track is reached. This is MASCA
	 * drive behaviour and its a limitation in MASCA drive.
	 */
	case FBWD:
		p_decode_scan_params->command = FBWD_MSF;
		break;
	default:
		break;
	}

}

#define	DIAG_DATE	2
#define DIAG_MONTH	3
#define DIAG_YEAR	4
#define DIAG_WRITE_STICKER 2
#define DIAG_CACHE_CONTROL 2
/* fill_diagnostic_info_from_cdb */
static void
fill_diagnostic_info_from_cdb(const unsigned char * const p_buffer,
				struct masca_cmd_params * const p_decode_params)
{
	if (p_decode_params->command == DIAG_CMD_DATE_INSTALL_WRITE) {
		/* Installation date */
		p_decode_params->cmd_params.diag_cmd_params.install_date_write =
			(((unsigned int)p_buffer[DIAG_YEAR] << BYTE_3_START) |
			((unsigned int)p_buffer[DIAG_MONTH] << BYTE_2_START) |
			((unsigned int)p_buffer[DIAG_DATE]));
	} else if (p_decode_params->command == DIAG_CMD_WRITE_STICKER) {
		/* Write sticker */
		memcpy(p_decode_params->cmd_params.diag_cmd_params.
								write_sticker,
			&p_buffer[DIAG_WRITE_STICKER], DIAG_CMD_WIRTE_STICKER);
	} else if (p_decode_params->command == DIAG_CACHE_DISABLE) {
		/* Block read cache enable/disable */
		p_decode_params->cmd_params.diag_cmd_params.cache_disable =
					(bool)p_buffer[DIAG_CACHE_CONTROL];
	}
}

/**
 * \func masca_atapi_scan
 *
 * This function decodes the ATAPI scan command to the enumeration masca_cmd.
 *
 * \param p_buf                  : pointer to the ATAPI scan command.
 *
 * \param p_decode_scan_params   : pointer to the structure holding decoding
 *                                 information. This will be filled by
 *                                 ATAPI_bridge.
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_INVALID_PARAM : If the ATAPI command is not supported
 */
static enum masca_error masca_atapi_scan(const unsigned char * const p_buf,
			struct masca_cmd_params * const p_decode_scan_params)
{

	enum masca_error ret_err = MASCA_OK;
	switch (p_buf[SCAN_REQ_TYPE] & ATAPI_SCAN_TYPECAST) {
	case ATAPI_SCAN_LBA:                      /*LBA*/
		p_decode_scan_params->cmd_params.play_info_lba.
		start_sector_address =
			masca_create_lba_from_cdb((unsigned char *)p_buf);

		p_decode_scan_params->cmd_params.play_info_lba.
						trk_num_playlba = 0;
		/* Direction byte1 */
		switch (p_buf[SCAN_REQ_DIR_RELADR] & ATAPI_SCAN_DIR_TYPECAST) {
		case FWD:
			p_decode_scan_params->command = FWD_LBA;
			break;
		case BWD:
			p_decode_scan_params->command = BWD_LBA;
			break;
		case FFWD:
			p_decode_scan_params->command = FFWD_LBA;
			break;
		case FBWD:
			p_decode_scan_params->command = FBWD_LBA;
			break;
		default:
			break;
		}
		break;
	case ATAPI_SCAN_MSF:                      /*MSF*/
		p_decode_scan_params->cmd_params.play_info_msf.start_min
					= p_buf[SCAN_REQ_SCAN_ADDR+0x01];
		p_decode_scan_params->cmd_params.play_info_msf.start_sec
					= p_buf[SCAN_REQ_SCAN_ADDR+0x02];
		p_decode_scan_params->cmd_params.play_info_msf.start_frame
					= p_buf[SCAN_REQ_SCAN_ADDR+0x03];
		p_decode_scan_params->cmd_params.play_info_msf.
							trk_num_playmsf = 0;
		masca_get_scan_command(p_buf, p_decode_scan_params);
		break;
	case ATAPI_SCAN_TRACK:                      /*Track number*/
		/*Track number between 1 to 99 tracks*/
		if ((p_buf[SCAN_REQ_SCAN_ADDR+0x03] >= MIN_TRACK) &&
				(p_buf[SCAN_REQ_SCAN_ADDR+0x03] <= MAX_TRACK)) {
			p_decode_scan_params->cmd_params.play_info_msf
			.trk_num_playmsf = p_buf[SCAN_REQ_SCAN_ADDR+0x03];
			p_decode_scan_params->cmd_params.play_info_msf
							.start_min = 0;
			p_decode_scan_params->cmd_params.play_info_msf
							.start_sec = 0;
			p_decode_scan_params->cmd_params.play_info_msf
							.start_frame = 0;
			masca_get_scan_command(p_buf, p_decode_scan_params);
		} else {
		ret_err = MASCA_INVALID_PARAM;
		}
		break;
	default:
		break;
	} /* switch(p_buf[9] & 0xc0) */

	return ret_err;
}

#define	START_STOP_EJECT	0x02
#define START_STOP_INSERT	0x03
#define POWER_ACTIVE		0x10
#define POWER_STANDBY		0x30
#define POWER_SLEEP		0x50

static enum masca_error masca_start_stop_command(const unsigned char * const
				p_buffer,
				struct masca_cmd_params * const p_decode_params)
{
	enum masca_error ret_err = MASCA_OK;
	struct masca_sm_state	sm_state;
	masca_sm_get_state(&sm_state);

	switch (p_buffer[START_STOP_REQ_PWR_LOAD_EJCT]) {
	case START_STOP_EJECT:
		p_decode_params->command = EJECT;
		usr_pause_cmd = FALSE;
		break;
	case START_STOP_INSERT:
		p_decode_params->command = INSERT;
		break;
	case POWER_ACTIVE:
		p_decode_params->command = PWR_ACTIVE;
		masca_drv_reset(FALSE);
		break;
	case POWER_STANDBY:
		if (sm_state.sm_drive_state == SM_NO_CD) {
			p_decode_params->command = PWR_STANDBY_NO_CD;
			if (sm_state.sm_pwr_state == SM_PWR_SLEEP)
				masca_drv_reset(FALSE);
		} else
			p_decode_params->command = PWR_STANDBY;
		break;
	case POWER_SLEEP:
		masca_drv_reset(TRUE);
		p_decode_params->command = PWR_SLEEP;
		break;
	default:
		ret_err = MASCA_IR_PARAM_NOT_SUPP;
		break;
	}
	return ret_err;
}


#define MAX_SECOND 59
#define MIN_2_SECOND 60 /* Minute to second */
/* Fill play by track audio information */
enum masca_error
fill_play_audio_trk_info(unsigned char * const p_buf,
			struct masca_cd_toc *p_cd_toc,
			struct masca_cmd_params * const p_cmd)
{
	enum masca_media_type cd_type;
	usr_pause_cmd = USR_PLAY_CMD;
	out_event_global.evt = NO_EVENT;

	masca_sm_get_media_type(&cd_type);
	if (DISC_CDROM == cd_type)
		return MASCA_NR_MEDIUM_INCOMPATIBLE;

	/* Start & End track should be in range 1-99 */
	/* Start track number should be less than End
	 * track number.
	 * Also start track number < max track in CD */
	if (!((p_buf[PLAY_START_TRK_NO] >= MIN_TRACK) &&
			(p_buf[PLAY_START_TRK_NO] <= MAX_TRACK)) ||
		(p_buf[PLAY_START_TRK_NO] > p_buf[PLAY_END_TRK_NO]) ||
		(p_buf[PLAY_START_TRK_NO] > p_cd_toc->last_audio_trk_num) ||
		(!((p_buf[PLAY_END_TRK_NO] >= MIN_TRACK) &&
				(p_buf[PLAY_END_TRK_NO] <= MAX_TRACK))))
		return MASCA_IR_INVALID_FLD_IN_CDB;

	/*If End track number > max track in CD
	 * then end track number = max track*/
	if (p_buf[PLAY_END_TRK_NO] > p_cd_toc->last_audio_trk_num)
		p_buf[PLAY_END_TRK_NO] = p_cd_toc->last_audio_trk_num;

	/* Starting track number field for play by track
	 * command(12h) */
	p_cmd->cmd_params.play_info_msf.trk_num_playmsf =
						p_buf[PLAY_START_TRK_NO];
	p_cmd->cmd_params.play_info_msf.trk_num_last =
						p_buf[PLAY_END_TRK_NO];
	/*Start MSF calculation for start track*/
	p_cmd->cmd_params.play_info_msf.start_min =
		p_cd_toc->track_info[p_buf[PLAY_START_TRK_NO]].track_start.min;
	p_cmd->cmd_params.play_info_msf.start_sec =
		p_cd_toc->track_info[p_buf[PLAY_START_TRK_NO]].track_start.sec;
	p_cmd->cmd_params.play_info_msf.start_frame = 0;
	if (p_buf[PLAY_END_TRK_NO] == p_cd_toc->last_audio_trk_num) {
		/*Play upto lead out MSF*/
		p_cmd->cmd_params.play_info_msf.end_min = p_cd_toc->cd_max_min;
		p_cmd->cmd_params.play_info_msf.end_sec = p_cd_toc->cd_max_sec;
		p_cmd->cmd_params.play_info_msf.end_frame = 0;
	} else {
		/*End MSF for end track*/
		p_cmd->cmd_params.play_info_msf.end_min =
			p_cd_toc->track_info[p_buf[PLAY_END_TRK_NO] + 1].
							track_start.min;

		if (!p_cd_toc->track_info[p_buf[PLAY_END_TRK_NO] + 1].
						track_start.sec) {
			p_cmd->cmd_params.play_info_msf.end_min =
			p_cd_toc->track_info[p_buf[PLAY_END_TRK_NO] + 1].
							track_start.min - 1;
			p_cmd->cmd_params.play_info_msf.end_sec = MAX_SECOND;
		} else {
			p_cmd->cmd_params.play_info_msf.end_sec =
			p_cd_toc->track_info[p_buf[PLAY_END_TRK_NO] + 1].
							track_start.sec - 1;
		}
		p_cmd->cmd_params.play_info_msf.end_frame = 0;
	}

	if (p_buf[PLAY_END_TRK_NO] == p_buf[PLAY_START_TRK_NO])
		p_cmd->command = PLAY_MSF;
	else
		p_cmd->command = PLAY_MSF_CONTINUOUS;

	return MASCA_OK;
}

#define SEC_TO_FRAME 75
/**
 * \func masca_atapi_decode
 *
 * This function decodes the ATAPI command to the enumeration masca_cmd.
 *

 * \param p_buf          : pointer to the ATAPI command.
 *
 * \param p_decode_params   : pointer to the structure holding decoding
 *                            information. This will be filled by
 *                            ATAPI_bridge.
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *  MASCA_INVALID_PARAM : If the ATAPI command is not supported
 */
enum masca_error masca_atapi_decode(
				const struct masca_output * const output,
				struct masca_cmd_params * const p_cmd)
{
	enum masca_error ret_err = MASCA_OK;
	unsigned short block_length    = 0;
	unsigned int start_msf = 0;
	unsigned int end_msf = 0;
	short spin_speed = SPIN_SPEED_NORMAL;
	unsigned char *p_buf = output->cmdbuf;
	struct masca_cd_toc *p_cd_toc = &cd_toc;

	masca_sm_get_toc(&p_cd_toc);
	if (p_buf == NULL)
		return MASCA_PTR_NULL;

	switch (*p_buf) {
	case TEST_UNIT_READY:
		p_cmd->command = GET_TEST_UNIT_READY;
		break;
	case REQUEST_SENSE:
		p_cmd->command = GET_REQUEST_SENSE;
		break;
	case RESERVE:
		p_cmd->command = CMD_RESERVE;
		break;
	case RELEASE:
		p_cmd->command = CMD_RELEASE;
		break;
	case SEND_DIAGNOSTIC:
		p_cmd->command = CMD_SEND_DIAGNOSTIC;
		break;
	case RECEIVE_DIAGNOSTIC:
		p_cmd->command = CMD_RECEIVE_DIAGNOSTIC;
		break;
	case LOG_SENSE:
		p_cmd->command = DIAG_GET_TEMPERATURE;
		block_length = (p_buf[LOG_SENSE_LENGTH] << BYTE_2_START) |
				p_buf[LOG_SENSE_LENGTH+0x01];
		/* Third check is for Temperature log page */
		if ((p_buf[LOG_SENSE_PPC_SP_BYTE] & LOG_SENSE_PPC_MASK) ||
			(p_buf[LOG_SENSE_PPC_SP_BYTE] & LOG_SENSE_SP_MASK) ||
			((p_buf[LOG_SENSE_PAGE_CODE] & LOG_SENSE_PC_MASK) !=
							LOG_SENSE_PC_CHECK) ||
			(0 == block_length) || (0 == output->cmd_len))
			ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
		break;
	case GPCMD_PLAY_AUDIO_MSF:	/* starts playing at MSF */
		usr_pause_cmd = USR_PLAY_CMD;
		out_event_global.evt = NO_EVENT;
		p_cmd->cmd_params.play_info_msf.start_min =
						p_buf[PLAY_MSF_START_MIN];
		p_cmd->cmd_params.play_info_msf.start_sec =
						p_buf[PLAY_MSF_START_SEC];
		p_cmd->cmd_params.play_info_msf.start_frame =
						p_buf[PLAY_MSF_START_FRM];
		p_cmd->cmd_params.play_info_msf.end_min =
						p_buf[PLAY_MSF_END_MIN];
		p_cmd->cmd_params.play_info_msf.end_sec =
						p_buf[PLAY_MSF_END_SEC];
		p_cmd->cmd_params.play_info_msf.end_frame =
						p_buf[PLAY_MSF_END_FRM];
		p_cmd->cmd_params.play_info_msf.trk_num_playmsf = 0;
		p_cmd->command = PLAY_MSF_CONTINUOUS;
		start_msf = p_buf[PLAYMSF_REQ_START_FRAME] +
			(p_buf[PLAYMSF_REQ_START_SEC] << BYTE_2_START) +
			(p_buf[PLAYMSF_REQ_START_MIN] << BYTE_3_START);
		end_msf = p_buf[PLAYMSF_REQ_END_FRAME] +
			(p_buf[PLAYMSF_REQ_END_SEC] << BYTE_2_START) +
			(p_buf[PLAYMSF_REQ_END_MIN] << BYTE_3_START);

		if ((start_msf != RESUME_PLAYBACK_MSF) &&
				(start_msf > end_msf) && (start_msf == 0))
			ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
		break;
	case GPCMD_PLAY_AUDIO_10:	/* starts playing at LBA */
		usr_pause_cmd = USR_PLAY_CMD;
		out_event_global.evt = NO_EVENT;

		block_length = ((p_buf[PLAY_AUDIO_LENGHT] << BYTE_2_START) |
			((unsigned short)p_buf[PLAY_AUDIO_LENGHT+0x01]));
		p_cmd->cmd_params.play_info_lba.start_sector_address =
					masca_create_lba_from_cdb(p_buf);
		p_cmd->cmd_params.play_info_lba.play_lng = block_length;
		p_cmd->cmd_params.play_info_lba.trk_num_playlba = 0;
		p_cmd->command = PLAY_LBA_CONTINUOUS;
		break;
	case GPCMD_PLAY_AUDIO_TI:
		ret_err = fill_play_audio_trk_info(p_buf, p_cd_toc, p_cmd);
		break;
	case GPCMD_PAUSE_RESUME:	/* starts or ends pause mode */

		if ((p_buf[PAUSE_RESUME_REQ_RESUME] &
			ATAPI_PAUSE_TYPECAST) != 0x01) {
			p_cmd->command = PAUSE;
			out_event_global.evt = NO_EVENT;
			if (usr_pause_cmd & USR_PLAY_CMD)
				usr_pause_cmd |= USR_PAUSE_CMD;
		} else {
			p_cmd->command = RESUME;
			usr_pause_cmd |= FALSE;
		}
		break;
	case GPCMD_STOP_PLAY_SCAN:	/* STOP */
		p_cmd->command = STOP;
		if (usr_pause_cmd & USR_PLAY_CMD) {
			usr_pause_cmd = FALSE;
			out_event_global.evt = CD_PLAY_COMPLETE;
		}
		break;
	case GPCMD_SCAN:	/* SCAN DATA */
		/* scan wrapper function */
		ret_err = masca_atapi_scan(p_buf, p_cmd);
		break;
	case READ_10:
		p_cmd->cmd_params.block_read_info.sector_start_address =
					masca_create_lba_from_cdb(p_buf);
		/* TODO: Local application buffer passed for getting
		 * sector data. This buffer is copied to SCSI buffer
		 * in the response handling in masca_reply_cmddata
		 * and its callees.
		 */
		p_cmd->cmd_params.block_read_info.p_buffer = g_app_buf;

		block_length = (((unsigned short)
				p_buf[READ10_REQ_TXLEN+0x00] << BYTE_2_START) |
				((unsigned short)p_buf[READ10_REQ_TXLEN+0x01]));
		ATAPI_TD_TRACE("MASCA: ******block add:%x len:%d\n",
		masca_create_lba_from_cdb(p_buf), block_length);
		p_cmd->cmd_params.block_read_info.number_of_sectors =
								block_length;
		p_cmd->command = READ_SECTORS;
		break;
	case READ_CAPACITY:	/* Returns the information about
		block size and maximum block
		count of the currently inserted
		CD media */
		p_cmd->command = GET_CD_CAPACITY;
		break;
	case GPCMD_GET_MEDIA_STATUS:	/* Returns media status */
		p_cmd->command = GET_MEDIA_STAT;
		break;
	case GPCMD_READ_SUBCHANNEL:
		p_cmd->command = READ_SUBCHANNEL;
		break;
	case INQUIRY:		/* Software and hardware version */
		p_cmd->command = GET_VERSION_INFO;
		/*If EVPD bit is zero,PAGE code not equal
		 * to zero, return INVALID FIELD IN CDB*/
		if (((0x01 & p_buf[INQ_REQ_EVPD]) == 0) &&
				((0xff & p_buf[INQ_REQ_PAGE_CODE]) != 0))
			ret_err = MASCA_IR_INVALID_FLD_IN_CDB;

		/*If EVPD bit is set then only
		 * supported VPD pages are 0x00 and 0x80 */
		else if (((0x01 & p_buf[INQ_REQ_EVPD]) == 1) &&
			!(p_buf[INQ_REQ_PAGE_CODE] == 0x00 ||
				p_buf[INQ_REQ_PAGE_CODE] == 0x80))
			ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
		break;
	case READ_TOC:			/* READ TOC */
		if (0 == p_cd_toc->last_track_number)
			break;
		if (!((p_buf[TOC_REQ_TRK_SESSION_NUM] == TOC_LEAD_OUT_LBA) ||
			((p_buf[TOC_REQ_TRK_SESSION_NUM] >= 0)  &&
			(p_buf[TOC_REQ_TRK_SESSION_NUM] <=
					p_cd_toc->last_track_number)))) {
			ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
			break;
		}
		if ((p_buf[TOC_REQ_TIME] & ATAPI_TOC_MSF_TYPECAST) != 0x00)
			g_masca_toc_msf_flag = TRUE;/* TOC reply - MSF format*/
		else
			g_masca_toc_msf_flag = FALSE;/* TOC reply - LBA format*/

		switch (p_buf[TOC_REQ_FORMAT] & TOC_FORMAT_TYPECAST) {
		case TOC_FORMAT_TRACK_TOC:
			p_cmd->command = GET_TOC;
			break;
		case TOC_FORMAT_SESSION_INFO:
			p_cmd->command = GET_SESSION_INFO;
			break;
		case TOC_FORMAT_FULL_TOC:
			p_cmd->command = GET_TOC;
			break;
		case TOC_FORMAT_CD_TEXT:
			p_cmd->command = GET_TITLE_ALBUM_AUTHOR;
			break;
		default:
			/* Unknown ATAPI command */
			ret_err = MASCA_INVALID_PARAM;
			break;
		}
		break;
	case START_STOP:
		ret_err = masca_start_stop_command(p_buf, p_cmd);
		break;
	case ATAPI_DEVICE_RESET:	/* Device reset */
		p_cmd->command = RESET_DRIVE;
		break;
	case GPCMD_SET_SPEED:
		p_cmd->command = CMD_SET_CD_SPEED;
		spin_speed = (p_buf[SET_SPEED_REQ_READ_SPEED+0x00]
							<< BYTE_2_START) +
				p_buf[SET_SPEED_REQ_READ_SPEED+0x01];
		p_cmd->cmd_params.cd_speed = spin_speed;
		break;
	case ATAPI_DEVICE_DIAG:		/* Execute device diagnostics */
		if ((p_buf[DIAG_COMMAND_SELECT] > 0) &&
			(p_buf[DIAG_COMMAND_SELECT] <=
					(MAX_COMMAND - DIAG_GET_TEMPERATURE))) {
			p_cmd->command = p_buf[1] + DIAG_GET_TEMPERATURE - 1;
			fill_diagnostic_info_from_cdb(p_buf, p_cmd);
		} else {
			ret_err = MASCA_INVALID_PARAM;
		}
		break;
	case GPCMD_READ_DISC_INFO:
		p_cmd->command = CMD_READ_DISK_INFO;
		break;
	case GET_EVENT_STATUS_NOTIFICATION:
		p_cmd->command = CMD_GET_EVT_STAT;
		break;
	default:
		ret_err = MASCA_INVALID_PARAM;/* Unknown ATAPI command*/
		break;
	} /* switch(*p_buf)*/

	return ret_err;
}

/**
 * \func masca_log_sense_get_temp
 *
 * This function gets the drive temperature.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *
 */
enum masca_error masca_log_sense_get_temp(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				unsigned int buf_length)
{
	p_buffer[LOGSENSE_RESP_PG_CODE] = LOG_PAGE_TEMPERATURE_PAGE;
	/* Subpage=0x00, Page Length MSB=0 */
	p_buffer[LOGSENSE_RESP_PG_LEN+0x01] = REMAINING_TEMP_PAGE_LENGTH;
	/* Temperature Log Parameter 1 (Temperature) Start */
	/* Parameter Code = 0x0000 */
	/* Format and Linking = 01b */
	p_buffer[LOGSENSE_RESP_TEMP_FRMT] = LOG_SENSE_RESP_FORMAT;
	/* Parameter Length */
	p_buffer[LOGSENSE_RESP_TEMP_LEN] = LOG_SENSE_RESP_LENGTH;
	/*LOG_TEMPERATURE_OFFSET(100) + RealTemperature In Degree Celsius*/
	p_buffer[LOGSENSE_RESP_TEMP_VAL+0x01] = LOG_TEMPERATURE_OFFSET +
			p_output->output_params.cd_diag_params.temperature[1];
	/* Temperature Log Parameter 2 (Reference Temperature) Start */
	/* Parameter Code = 0x0001 */
	p_buffer[LOGSENSE_RESP_REF_TEMP_CODE+0x01] = LOG_SENSE_RESP_PARA_CODE;
	/* Format and Linking = 01b */
	p_buffer[LOGSENSE_RESP_REF_TEMP_FRMT] = LOG_SENSE_RESP_FORMAT;
	/* Parameter Length */
	p_buffer[LOGSENSE_RESP_REF_TEMP_LEN] = LOG_SENSE_RESP_LENGTH;
	p_buffer[LOGSENSE_RESP_REF_TEMP_VAL] = 0;

	return MASCA_OK;
}
/**
 * \func masca_atapi_convert_cd_text
 *
 * This function converts the CD text information to the ATAPI format.
 *
 * \param p_cd_txt      : pointer to structure holding CD text information.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *
 */
enum masca_error masca_atapi_convert_cd_text(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				unsigned int buf_length)
{
	bool    is_mem_available = TRUE;
	unsigned char      trk_cnt     = 0;
	unsigned char      frst_trk    = 0;
	unsigned char      lst_trk     = 0;
	unsigned char	*p_text_buffer;
	unsigned short     cd_text_lng     = 0;
	unsigned short     num_of_packs    = 0;
	unsigned short     text_length     = 0;
	struct masca_cd_text *p_cd_txt = NULL;
	enum masca_error ret_err     = MASCA_INVALID_PARAM;

	p_cd_txt = p_output->output_params.p_cd_text;

	if ((p_cd_txt != NULL) && (buf_length >= SIZE_OF_CDTEXT_HDR)) {
		ret_err = MASCA_OK;
		frst_trk = p_cd_txt->track_number_first;
		lst_trk = p_cd_txt->track_number_last;

		/*Compute the whole CD info text first*/
		text_length = p_cd_txt->track_text[CD_DISC_INFO].length_title;
		cd_text_lng = masca_updt_txt_lng(cd_text_lng, text_length);
		for (trk_cnt = frst_trk; trk_cnt <= lst_trk; trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_title;
			cd_text_lng = masca_updt_txt_lng(cd_text_lng,
							 text_length);
		}

		text_length = p_cd_txt->track_text[CD_DISC_INFO].length_author;
		cd_text_lng = masca_updt_txt_lng(cd_text_lng, text_length);
		for (trk_cnt = frst_trk; trk_cnt <= lst_trk; trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_author;
			cd_text_lng = masca_updt_txt_lng(cd_text_lng,
							 text_length);
		}

		text_length = p_cd_txt->track_text[CD_DISC_INFO].length_album;
		cd_text_lng = masca_updt_txt_lng(cd_text_lng, text_length);
		for (trk_cnt = frst_trk; trk_cnt <= lst_trk; trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_album;
			cd_text_lng = masca_updt_txt_lng(cd_text_lng,
							 text_length);
		}
		/*From total pay load size compute number of packs*/
		num_of_packs = (unsigned short)(cd_text_lng /
						(ATAPI_PACK_PAYLOAD_SIZE));
		if (0 != (cd_text_lng % ATAPI_PACK_PAYLOAD_SIZE))
			num_of_packs++;
		cd_text_lng = (num_of_packs * ATAPI_PACK_SIZE) + 2;
		cursor_pos = 0;
		p_buffer[cursor_pos] = (unsigned char) ((cd_text_lng &
						0xFF00)  >> BYTE_2_START);
		cursor_pos++;
		p_buffer[cursor_pos] = (unsigned char) (cd_text_lng & 0x00FF);
		cursor_pos++;
		p_buffer[cursor_pos] = p_cd_txt->track_number_first;
		cursor_pos++;
		p_buffer[cursor_pos] = p_cd_txt->track_number_last;
		cursor_pos++;
		buf_length -= SIZE_OF_CDTEXT_HDR;
		sequence_no = 0;
		bytes_left_over = 0;
		updt_header = TRUE;

		/*We will first mention all the titles*/
		pack_type = ATAPI_TITLE;
		/*Add title of whole CD*/
		text_length = p_cd_txt->track_text[CD_DISC_INFO].length_title;
		if (text_length > 0) {
			/*This increment is for separator 0*/
			text_length++;
		}
		p_text_buffer = p_cd_txt->track_text[CD_DISC_INFO].track_title;
		is_mem_available = masca_updt_track_text(p_buffer,
				    p_text_buffer, &buf_length, text_length, 0);
		for (trk_cnt = frst_trk;
			((trk_cnt <= lst_trk) && (TRUE == is_mem_available));
			trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_title;
			if (text_length > 0) {
				/*This increment is for separator 0*/
				text_length++;
			}
			p_text_buffer = p_cd_txt->track_text[trk_cnt]
								.track_title;
			is_mem_available = masca_updt_track_text(p_buffer,
								 p_text_buffer,
								 &buf_length,
								 text_length,
								 trk_cnt);
		}

		pack_type = ATAPI_AUTHOR;
		/*Add author information of whole CD*/
		if (TRUE == is_mem_available) {
			text_length = p_cd_txt->track_text[CD_DISC_INFO].
								length_author;
			if (text_length > 0) {
				/*This increment is for separator 0*/
				text_length++;
			}
			p_text_buffer = p_cd_txt->track_text[CD_DISC_INFO].
								track_author;
			is_mem_available = masca_updt_track_text(p_buffer,
								 p_text_buffer,
								 &buf_length,
								text_length, 0);
		}

		for (trk_cnt = frst_trk;
			(trk_cnt <= lst_trk) && (TRUE == is_mem_available);
			trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_author;
			if (text_length > 0) {
				/*This increment is for separator 0*/
				text_length++;
			}
			p_text_buffer = p_cd_txt->track_text[trk_cnt]
								.track_author;
			is_mem_available = masca_updt_track_text(p_buffer,
								 p_text_buffer,
								 &buf_length,
								 text_length,
								 trk_cnt);
		}

		pack_type = ATAPI_ALBUM;
		/*Add album information of whole CD*/
		if (TRUE == is_mem_available) {
			text_length = p_cd_txt->track_text[CD_DISC_INFO].
								length_album;
			if (text_length > 0) {
				/*This increment is for separator 0*/
				text_length++;
			}
			p_text_buffer = p_cd_txt->track_text[CD_DISC_INFO].
								track_album;
			is_mem_available = masca_updt_track_text(p_buffer,
								 p_text_buffer,
								 &buf_length,
								text_length, 0);
		}
		for (trk_cnt = frst_trk;
			(trk_cnt <= lst_trk) && (TRUE == is_mem_available);
			trk_cnt++) {
			text_length = p_cd_txt->track_text[trk_cnt]
								.length_album;
			if (text_length > 0) {
				/*This increment is for separator 0*/
				text_length++;
			}
			p_text_buffer = p_cd_txt->track_text[trk_cnt]
								.track_album;
			is_mem_available = masca_updt_track_text(p_buffer,
								 p_text_buffer,
								 &buf_length,
								 text_length,
								 trk_cnt);
		}
	}
	return ret_err;
}

struct full_toc_header {
	unsigned short data_len;
	unsigned char first_comp_sess_num;
	unsigned char last_comp_sess_num;
};

struct full_toc_descriptor {
	unsigned char sess_num;
	unsigned char adr_control;
	unsigned char tno;
	unsigned char point_field;
	unsigned char min;
	unsigned char sec;
	unsigned char frame;
	unsigned char zero_field;
	unsigned char pmin;
	unsigned char psec;
	unsigned char pframe;
};

#define FIRST_TRK_NO_PRG_AREA	0xA0
#define LAST_TRK_NO_PRG_AREA	0xA1
#define START_LOC_LEAD_OUT_AREA	0xA2
#define START_TIME_NEXT_PRG_AREA	0xB0
#define START_TIME_FIRST_LEAD_IN_AREA	0xC0

static void fill_session_info(bool multi_sess, unsigned char sess_no,
				unsigned short index,
				unsigned int lba, struct masca_cd_toc *toc,
				unsigned char * const buf)
{
	unsigned char  session_header = 3;
	unsigned char  track_no = 0;
	unsigned char  start_trk = 0;
	unsigned char  end_trk = 0;
	unsigned short idx = 0;

	struct full_toc_descriptor full_toc_desc = {0};
	idx += sizeof(struct full_toc_header);

	/* Session header */
	for (track_no = 1; track_no <= session_header; track_no++) {
		full_toc_desc.sess_num = sess_no;
		full_toc_desc.adr_control =
			((multi_sess != TRUE) ? ADR_CTRL_AUDIO : ADR_CTRL_DATA);

		if (track_no == 1) {
			full_toc_desc.point_field = FIRST_TRK_NO_PRG_AREA;
			full_toc_desc.pmin = ((multi_sess != TRUE) ?
					toc->first_track_number : sess_no);
		} else if (track_no == 2) {
			full_toc_desc.point_field = LAST_TRK_NO_PRG_AREA;
			full_toc_desc.pmin = ((multi_sess != TRUE) ?
					toc->last_track_number : sess_no);
		} else if (track_no == 3) {
			full_toc_desc.point_field = START_LOC_LEAD_OUT_AREA;
			full_toc_desc.pmin = lba / FRAMES_PER_MIN;
			full_toc_desc.psec = (lba % FRAMES_PER_MIN)
							/ FRAMES_PER_SEC;
			full_toc_desc.pframe = lba % FRAMES_PER_SEC;
		}

		memcpy(buf + idx, &full_toc_desc,
					sizeof(struct full_toc_descriptor));
		idx += sizeof(struct full_toc_descriptor);
	}

	if (multi_sess != TRUE) {
		start_trk = toc->first_track_number;
		end_trk = toc->last_track_number;
	} else {
		start_trk = sess_no;
		end_trk = sess_no;
	}

	/* Track details of each session */
	for (track_no = start_trk; track_no <= end_trk; track_no++) {

		full_toc_desc.sess_num = sess_no;

		full_toc_desc.adr_control = SWAP_NIBBLE(toc->track_info
							[track_no].adr_ctrl);

		full_toc_desc.point_field = track_no;

		full_toc_desc.pmin = toc->track_info[track_no].track_start.min;

		full_toc_desc.psec = toc->track_info[track_no].track_start.sec;

		full_toc_desc.pframe = toc->track_info[track_no].
								track_start.frm;

		memcpy(buf + idx, &full_toc_desc,
					sizeof(struct full_toc_descriptor));
		idx += sizeof(struct full_toc_descriptor);
	}

	/* Recordable area of hybrid disc */
	if (multi_sess == TRUE) {

		full_toc_desc.sess_num = sess_no;

		/* A/C = 0x45 for next recordable area
		 address of hybrid disc */
		full_toc_desc.adr_control = ADR_CTRL_HYBRID;
		/* Pnt = 0xB0 for next recordable area
		 address of hybrid disc */
		full_toc_desc.point_field = START_TIME_NEXT_PRG_AREA;

		memcpy(buf + idx, &full_toc_desc,
					sizeof(struct full_toc_descriptor));

		idx += sizeof(struct full_toc_descriptor);
	}

	/* Hybrid disc */
	if ((multi_sess == TRUE) && (sess_no == 1)) {
		full_toc_desc.sess_num = sess_no;
		/* A/C = 0x45 for hybrid disc */
		full_toc_desc.adr_control = ADR_CTRL_HYBRID;
		/* Pnt = 0xC0 for hybrid disc */
		full_toc_desc.point_field = START_TIME_FIRST_LEAD_IN_AREA;
		full_toc_desc.min = 0xC0;/* Min = 0xC0 for hybrid disc */
		memcpy(buf + idx, &full_toc_desc,
					sizeof(struct full_toc_descriptor));
	}

}

/* Full TOC */
static enum masca_error  masca_atapi_convert_full_toc(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				unsigned int buf_length)
{
	bool multi_sess = FALSE;
	unsigned char  start_no = 0;
	unsigned char  end_no = 0;
	unsigned char  sess_cnt = 0;
	unsigned int toc_length;
	unsigned short index = 0;
	struct masca_cd_toc *p_toc = NULL;
	enum masca_error ret_err = MASCA_OK;
	unsigned int lba = 0;
	struct masca_session_info	sess_info;
	struct masca_session_info	*p_sess_info = &sess_info;

	struct full_toc_header full_toc_head = {0};

	p_toc = p_output->output_params.p_cd_toc;
	masca_sm_get_session_info(&p_sess_info);

	if (p_sess_info->last_complete_session > 1) {
		/* Multi-session */
		multi_sess = TRUE;
		start_no = p_sess_info->first_complete_session;
		end_no = p_sess_info->last_complete_session;

	} else {
		/* Single-session */
		multi_sess = FALSE;
		start_no = p_toc->first_track_number;
		end_no = p_toc->last_track_number;
	}

	if ((p_toc != NULL) && (p_buffer != NULL)) {
		if (multi_sess != TRUE)
			toc_length = (SESS_HEADER + end_no - start_no + 1)
						* FULL_TOC_TRK_SZ;
		else
			toc_length = (((SESS_HEADER + SESS_TAIL) *
					(end_no - start_no + 1)) +
					HYBRID_DISC_HEADER) * FULL_TOC_TRK_SZ;

			memset(p_buffer, 0, toc_length + SIZE_OF_TOC_HDR);
			full_toc_head.data_len = cpu_to_be16(toc_length);
			full_toc_head.first_comp_sess_num = p_sess_info->
							first_complete_session;
			full_toc_head.last_comp_sess_num = p_sess_info->
							last_complete_session;
			memcpy(p_buffer, &full_toc_head,
						sizeof(struct full_toc_header));

			for (sess_cnt = p_sess_info->first_complete_session;
				sess_cnt <= p_sess_info->last_complete_session;
				sess_cnt++) {
				if (sess_cnt ==
					p_sess_info->last_complete_session)
					masca_sm_get_cd_capacity(&lba);
				else
					lba = p_toc->track_info[sess_cnt + 1]
					     .track_start_lba - FRAMES_PER_SEC;

				fill_session_info(multi_sess, sess_cnt,
						index, lba, p_toc, p_buffer);
			}
	} else {
		ret_err = MASCA_PTR_NULL;
	}
	return ret_err;
}

struct toc_header {
	unsigned short data_len;
	unsigned char first_trk_num;
	unsigned char last_trk_num;
};

union toc_addr {
	struct {
		unsigned char reserved;
		unsigned char m;
		unsigned char s;
		unsigned char f;
	};
	unsigned int lba;
};

struct toc_descriptor {
	unsigned char reserved0;
	unsigned char adr_control;
	unsigned char trk_num;
	unsigned char reserved3;
	union toc_addr toc_start_addr;
};


/* masca_fill_toc_lead_out */
static void masca_fill_toc_lead_out(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				unsigned int buf_length)
{
	struct toc_header toc_head = {0};
	struct toc_descriptor toc_desc = {0};
	struct masca_cd_toc *p_toc = p_output->output_params.p_cd_toc;

	/* Fill TOC Header */
	toc_head.data_len = cpu_to_be16(
				FIRST_TRK_LAST_TRK_NUM + TRACK_TOC_SZ);
	toc_head.first_trk_num = p_toc->first_track_number;
	toc_head.last_trk_num = p_toc->last_track_number;

	memcpy(p_buffer, &toc_head, sizeof(struct toc_header));

	/* Fill Track TOC Descriptor*/
	toc_desc.adr_control = SWAP_NIBBLE(p_toc->
				track_info[toc_head.first_trk_num].adr_ctrl);

	toc_desc.trk_num = TOC_LEAD_OUT_LBA;

	if (g_masca_toc_msf_flag != 0) {/* MSF TOC */
		toc_desc.toc_start_addr.m = p_toc->cd_max_min;
		toc_desc.toc_start_addr.s = p_toc->cd_max_sec;
	} else {
		toc_desc.toc_start_addr.lba =
			cpu_to_be32(GET_LBA(p_toc->cd_max_min,
				p_toc->cd_max_sec, 0) - TOC_START_LBA_OFFSET);
	}
	memcpy(p_buffer + sizeof(struct toc_header), &toc_desc,
					sizeof(struct toc_descriptor));
}

/* Function to identify GET_TOC request is for single track or not */
void masca_get_toc_track(const struct masca_output * const p_output,
			unsigned char *track_start, unsigned char *track_end,
			bool *is_full_toc)
{
	struct scsi_cmnd *scsicmd = NULL;
	struct masca_rcvr *rcvr = NULL;
	enum masca_media_type cd_type;

	rcvr = masca_get_rcvr_que_indx(p_output->response_id);
	masca_sm_get_media_type(&cd_type);

	if (NULL != rcvr) {
		scsicmd = rcvr->a_cmnd;
		if (TOC_FORMAT_TRACK_TOC == scsicmd->cmnd[READ_TOC_FORMAT]) {
			*track_start = scsicmd->cmnd[READ_TOC_TRACK_NO];
			*track_end   = cd_toc.last_track_number;

			if ((scsicmd->cmnd[READ_TOC_CTRL] == TOC_CTRL_BYTE_REQ)
					&& (cd_type == DISC_CDROM)) {
				*track_start = cd_toc.last_track_number;
			} else if (scsicmd->cmnd[READ_TOC_TRACK_NO] == 0) {
				*track_start = 1;
			} else if (scsicmd->cmnd[READ_TOC_TRACK_NO] ==
							TOC_LEAD_OUT_LBA) {
				*track_start = TOC_LEAD_OUT_LBA;
			}
		} else if ((FULL_TOC_REQ == scsicmd->cmnd[READ_TOC_FORMAT]) ||
			((scsicmd->cmnd[READ_TOC_FORMAT] ==
							TOC_FORMAT_TRACK_TOC)
			&& (scsicmd->cmnd[READ_TOC_CTRL] ==
							FULL_TOC_SUB_REQ))) {
			*is_full_toc = TRUE;
		}
	}
}

/**
 * \func masca_atapi_convert_toc
 *
 * This function converts the TOC information to the ATAPI format.
 *
 * \param p_toc         : pointer to structure holding TOC information.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *
 */
extern enum masca_error
masca_atapi_convert_toc(const struct masca_output * const p_output,
				unsigned char * const p_buff,
				unsigned int buf_length)
{
	bool is_full_toc = FALSE;
	unsigned char  track_start = 0;
	unsigned char  track_end = 0;
	unsigned char  track_no = 0;
	unsigned char  minute = 0;
	unsigned char  second = 0;
	unsigned char  frame = 0;
	unsigned short toc_length = 0;
	unsigned char lead_out_buffer[LEAD_OUT_BUFF_LEN] = {0};
	char *p_buffer = NULL;
	unsigned short idx = 0;
	unsigned int toc_lba_start_addr = 0;
	enum masca_media_type cd_type;
	unsigned int req_toclen = 0;

	struct toc_header toc_head = {0};
	struct toc_descriptor toc_desc = {0};

	struct masca_cd_toc *p_toc = NULL;
	enum masca_error ret_err = MASCA_OK;
	p_toc = p_output->output_params.p_cd_toc;
	masca_sm_get_media_type(&cd_type);

	if (p_toc != NULL) {
		masca_get_toc_track(p_output, &track_start, &track_end,
								&is_full_toc);
		/*TOC header = TOC Data length (2 bytes) +
		 * First track number (1 byte) +
		 * Last track number (1 byte) */

		toc_length = FIRST_TRK_LAST_TRK_NUM +
				((track_end - track_start + 1) * TRACK_TOC_SZ) +
				LEAD_OUT_SZ;

		p_buffer = vzalloc(toc_length + TOC_DATA_LENGTH);

		if (NULL == p_buffer)
			return MASCA_PTR_NULL;

		if (is_full_toc != FALSE) {
			ret_err = masca_atapi_convert_full_toc(p_output,
							p_buffer, buf_length);
			if (ret_err == MASCA_OK)
				memcpy(p_buff, p_buffer, buf_length);
			if (p_buffer != NULL)
				vfree(p_buffer);
			return ret_err;
		}


		if (track_start == TOC_LEAD_OUT_LBA) {

			masca_fill_toc_lead_out(p_output, p_buffer, buf_length);
			memcpy(p_buff, p_buffer, buf_length);
			vfree(p_buffer);
			return ret_err;
		}

		/* always update the exact toc length so that
		 * application comes to know about it even if the size
		 * passed by application is less*/
		toc_head.data_len = cpu_to_be16(toc_length);
		toc_head.first_trk_num = p_toc->first_track_number;
		toc_head.last_trk_num = p_toc->last_track_number;

		memcpy(p_buffer, &toc_head, sizeof(struct toc_header));

		idx += sizeof(struct toc_header);

		for (track_no = track_start;
			(track_no <= track_end); track_no++) {
			toc_desc.adr_control = SWAP_NIBBLE(
					p_toc->track_info[track_no].adr_ctrl);

			toc_desc.trk_num = p_toc->track_info[track_no].
								track_number;
		if (g_masca_toc_msf_flag != 0) {         /* MSF TOC */
			if ((p_toc->last_track_number == track_no)
					&& (DISC_CDROM == cd_type)) {
				toc_desc.toc_start_addr.m = p_toc->last_min;
				toc_desc.toc_start_addr.s = p_toc->last_sec +
						MSF_RESP_LAST_TRK_SECOND_OFFSET;
				if (toc_desc.toc_start_addr.s > MAX_SECOND) {
					/*If incoming seconds was 59s (lets say)
					 * and since offset of 2 seconds has
					 * been added in previous step so it
					 * will exceed MAX_SECOND.
					 * So below we subtract 60s and
					 * increment a minute*/
					toc_desc.toc_start_addr.s -=
								MIN_2_SECOND;
					toc_desc.toc_start_addr.m += 1;
				}
				toc_desc.toc_start_addr.f = p_toc->last_frm;
			} else {
				toc_desc.toc_start_addr.m =
				p_toc->track_info[track_no].track_start.min;
				toc_desc.toc_start_addr.s =
				p_toc->track_info[track_no].track_start.sec;
				toc_desc.toc_start_addr.f =
				p_toc->track_info[track_no].track_start.frm;
			}

			minute = toc_desc.toc_start_addr.m;
			second = toc_desc.toc_start_addr.s;
			frame = toc_desc.toc_start_addr.f;

			ATAPI_TD_TOC(
			"TrkNo = %d, minute = %d,second = %d,frm = %d\n",
				track_no, minute, second, frame);

		} else {
			if ((p_toc->last_track_number == track_no)
					&& (DISC_CDROM == cd_type)) {
				toc_lba_start_addr = GET_LBA(p_toc->last_min,
							p_toc->last_sec,
							p_toc->last_frm);
			} else {
				toc_lba_start_addr =
					p_toc->track_info[track_no].
					track_start_lba - TOC_START_LBA_OFFSET;
			}
			toc_desc.toc_start_addr.lba =
					cpu_to_be32(toc_lba_start_addr);
		}

		memcpy(p_buffer + idx,
			&toc_desc, sizeof(struct toc_descriptor));

		idx += sizeof(struct toc_descriptor);

		if (TOC_REQUEST_SIZE == buf_length)
			break;
		} /* for loop */

		memset(lead_out_buffer, 0, LEAD_OUT_BUFF_LEN);
		masca_fill_toc_lead_out(p_output, lead_out_buffer,
							TRACK_TOC_SZ);
		memcpy(p_buffer+idx, lead_out_buffer + SIZE_OF_TOC_HDR,
							TRACK_TOC_SZ);
		req_toclen = ((buf_length <= toc_length) ? buf_length :
								toc_length);
		memcpy(p_buff, p_buffer, req_toclen);

		if (p_buffer != NULL)
			vfree(p_buffer);

	} else { /* if(p_toc != NULL) */
		ret_err = MASCA_PTR_NULL;
	}

	return ret_err;
}

struct toc_sess_info_header {
	unsigned char toc_data_len_msb;
	unsigned char toc_data_len_lsb;
	unsigned char first_comp_sess_num;
	unsigned char last_comp_sess_num;
};

struct toc_session_info_descriptor {
	unsigned char reserved0;
	unsigned char adr_control;
	unsigned char first_trk_last_comp_sess;
	unsigned char reserved3;
	union toc_addr toc_start_addr;
};

/**
 * \func masca_atapi_convert_session_info
 *
 * This function converts the session information to the ATAPI format.
 *
 * \param p_session_info: pointer to structure holding TOC information.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *
 */
extern enum masca_error
masca_atapi_convert_session_info(const struct masca_output * const p_output,
					unsigned char * const p_buffer,
					unsigned int buf_length)
{
	struct masca_session_info *p_session_info = NULL;
	enum masca_error ret_err = MASCA_OK;
	struct masca_cd_toc *p_toc = NULL;
	struct toc_sess_info_header sess_info_head = {0};
	struct toc_session_info_descriptor  sess_info_desc = {0};

	p_session_info = p_output->output_params.p_cd_session_info;
	(void)masca_sm_get_toc(&p_toc);

	if ((p_session_info == NULL) || (p_toc == NULL))
		return MASCA_PTR_NULL;

	if (buf_length >= (SESSION_INFO_LNG + SESSION_HEADER_LNG)) {
		/* session info length is fixed size of 10(0x0a)bytes*/

		sess_info_head.toc_data_len_msb = 0x00;
		sess_info_head.toc_data_len_lsb = SESSION_INFO_LNG;
		sess_info_head.first_comp_sess_num =
					p_session_info->first_complete_session;
		sess_info_head.last_comp_sess_num =
					p_session_info->last_complete_session;
		memcpy(p_buffer, &sess_info_head,
				sizeof(struct toc_sess_info_header));
		/*ADR = 0001b and Control = 0000b*/
		sess_info_desc.adr_control = SWAP_NIBBLE(p_session_info->
						last_sess_trk_info.adr_ctrl);

		/* Byte6 : First track number in last session*/
		sess_info_desc.first_trk_last_comp_sess =
				p_session_info->last_sess_trk_info.track_number;
		if (g_masca_toc_msf_flag != 0) {         /* MSF TOC */
			sess_info_desc.toc_start_addr.m = p_toc->last_min;
			sess_info_desc.toc_start_addr.s = p_toc->last_sec +
						MSF_RESP_LAST_TRK_SECOND_OFFSET;
			if (sess_info_desc.toc_start_addr.s > MAX_SECOND) {
				sess_info_desc.toc_start_addr.s -= MIN_2_SECOND;
				sess_info_desc.toc_start_addr.m += 1;
			}

			sess_info_desc.toc_start_addr.f = p_toc->last_frm;
		} else {
			sess_info_desc.toc_start_addr.lba =
					cpu_to_be32(GET_LBA(p_toc->last_min,
							p_toc->last_sec,
							p_toc->last_frm));
		}

		memcpy(p_buffer + sizeof(struct toc_sess_info_header),
					&sess_info_desc, sizeof(struct
					toc_session_info_descriptor));
	} else {/* if(buf_length >= MAX_SESSION_DATA) */
		ret_err = MASCA_INVALID_PARAM;
	}

	return ret_err;
}

/**
 * \func masca_atapi_convert_inquiry
 *
 * This function converts the version information to the ATAPI format.
 *
 * \param p_version_info: pointer to structure holding version information.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *
 */

#define VPD_PAGE_CODE_80	0x80
#define VPD_PAGE_CODE_00	0x00
#define VPD_PAGE_CODE_80_LEN	0x0a
#define VPD_PAGE_CODE_00_LEN	0x02
#define MASCA_SERIAL_NO_PREFIX	"MASCA_"

void masca_fill_vpd_page(struct scsi_cmnd *scsi_cmd,
			unsigned char * const p_buffer,
			struct masca_version_info *p_version_info)
{
	/*Page Code = 0x80: Unit Serial Number*/
	if (scsi_cmd->cmnd[INQ_REQ_PAGE_CODE] == VPD_PAGE_CODE_80) {

		p_buffer[INQ_VPD_RESP_DEVICE_TYPE] = PHERIPHERAL_TYPE_CDROM;
		p_buffer[INQ_VPD_RESP_PAGE_CODE] = VPD_PAGE_CODE_80;
		p_buffer[INQ_VPD_RESP_PAGE_LEN] = VPD_PAGE_CODE_80_LEN;

		memcpy(&p_buffer[INQ_VPD80_RESP_SRL_NUM],
				MASCA_SERIAL_NO_PREFIX,
				strlen(MASCA_SERIAL_NO_PREFIX));

		sprintf(&p_buffer[INQ_VPD80_RESP_HW_VER], "%02x",
			p_version_info->software_version);
		sprintf(&p_buffer[INQ_VPD80_RESP_SW_VER], "%02x",
			p_version_info->hardware_version);

	} else { /*Page Code = 0x00: Supported VPD Pages*/

		p_buffer[INQ_VPD_RESP_DEVICE_TYPE] = PHERIPHERAL_TYPE_CDROM;
		p_buffer[INQ_VPD_RESP_PAGE_CODE] = VPD_PAGE_CODE_00;
		p_buffer[INQ_VPD_RESP_PAGE_LEN] = VPD_PAGE_CODE_00_LEN;
		p_buffer[INQ_VPD00_RESP_PAGE_LIST+0x00] = VPD_PAGE_CODE_00;
		p_buffer[INQ_VPD00_RESP_PAGE_LIST+0x01] = VPD_PAGE_CODE_80;
	}
}

enum masca_error
masca_atapi_convert_inquiry(const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				unsigned int buf_length)
{
	unsigned char *vendor_name[3] = {"TANASHIN", "KENWOOD", "PIONEER"};
	struct masca_version_info *p_version_info = NULL;
	enum masca_error         ret_err = MASCA_OK;
	struct scsi_cmnd *scsi_cmd = NULL;
	struct masca_rcvr *qcmnd = NULL;

	p_version_info = (struct masca_version_info *)&p_output->output_params
							.cd_version_info;
	qcmnd = masca_get_rcvr_que_indx(p_output->response_id);


	if (qcmnd != NULL) {

		scsi_cmd = qcmnd->a_cmnd;

		if ((0x01 & scsi_cmd->cmnd[INQ_REQ_EVPD]) == 0) {

			/*Regular inquiry*/
			/* Vendor ID can be max of 8 bytes acc. to SPC-3 spec.*/
			/* vendor_id  string having less than 8 characters have
			 * to be filled with spaces hence initializing the same
			 * with spaces to have flexibility in vendor_name. */

			memset(version_info.vendor_id, ' ', VENDOR_ID_SIZE);

			memset(version_info.product_id, ' ', PRODUCT_ID_SIZE);

			memset(version_info.prod_rev_level, ' ',
								PROD_REV_LEVEL);

			strncpy(version_info.vendor_id,
			vendor_name[p_version_info->vendor_identification],
					strlen(vendor_name[p_version_info->
						vendor_identification]));

			strncpy(version_info.product_id, COMPONENT_NAME_MASCA,
						strlen(COMPONENT_NAME_MASCA));

			sprintf(version_info.prod_rev_level, "%02x"
				, p_version_info->software_version);

			sprintf(&version_info.prod_rev_level[2], "%02x"
				, p_version_info->hardware_version);

			memcpy(p_buffer, &version_info, buf_length);

		} else {
			/*Inquiry asking for VPD information*/
			masca_fill_vpd_page(scsi_cmd, p_buffer, p_version_info);
		}

	} else {/* (p_version_info != NULL ) */
		ret_err = MASCA_PTR_NULL;
	}

	return ret_err;
}

/**
 * \func masca_atapi_convert_read_capacity
 *
 * This function converts the version information to the ATAPI format.
 *
 * \param cd_capacity   : cd read capacity.
 *
 * \param p_buffer      : pointer to buffer which holds ATAPI reply.
 *
 * \param buf_length    : length of the buffer which holds ATAPI reply
 *
 * \return
 *  MASCA_OK            : Decode of ATAPI command is successful
 *  MASCA_PTR_NULL      : Any of the parameters passed are NULL
 *	MASCA_INVALID_PARAM : Invalid message length
 *
 */
enum masca_error masca_atapi_convert_read_capacity(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	unsigned int cd_capacity = 0;
	enum masca_error         ret_err = MASCA_OK;

	cd_capacity = p_output->output_params.cd_capacity;
	if (buf_length >= READ_CAPACITY_RESP_LEN) {
		p_buffer[READCAP_RESP_LBA+0x00] = (unsigned char) ((cd_capacity
						>> BYTE_4_START)
							& BYTE_MASK);
		p_buffer[READCAP_RESP_LBA+0x01] = (unsigned char) ((cd_capacity
						>> BYTE_3_START)
							& BYTE_MASK);
		p_buffer[READCAP_RESP_LBA+0x02] = (unsigned char) ((cd_capacity
						>> BYTE_2_START)
							& BYTE_MASK);
		p_buffer[READCAP_RESP_LBA+0x03] = (unsigned char) ((cd_capacity
						>> BYTE_1_START)
							& BYTE_MASK);

		/* Block length is set to 2048 (0x00000800)
		for byte4(MSB) to byte7(LSB) */
		p_buffer[READCAP_RESP_LEN+0x00] = 0x00;
		p_buffer[READCAP_RESP_LEN+0x01] = 0x00;
		p_buffer[READCAP_RESP_LEN+0x02] = 0x08;
		p_buffer[READCAP_RESP_LEN+0x03] = 0x00;
	} else {
		ret_err = MASCA_INVALID_PARAM;
	}

	return ret_err;
}

enum masca_error masca_send_diagnostic(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	enum masca_error ret_err = MASCA_OK;
	unsigned int param_list_len =
			(p_output->cmdbuf[SEND_DIAG_REQ_LIST_LEN]
					  << BYTE_2_START)
			+ (p_output->cmdbuf[SEND_DIAG_REQ_LIST_LEN + 0x01]);

	if (((test_bit(SEND_DIAG_REQ_SELFTEST,
			(unsigned long *) p_output->cmdbuf)))
			&& (test_bit(SEND_DIAG_REQ_PAGE_FORMAT,
					(unsigned long *) p_output->cmdbuf))
			&& (!test_bit(SEND_DIAG_REQ_DEVOFF,
					(unsigned long *) p_output->cmdbuf))
			&& (!test_bit(SEND_DIAG_REQ_UNITOFF,
					(unsigned long *) p_output->cmdbuf))
			&& (param_list_len == 0)) {
		ret_err = masca_check_async_sense();
		if (MASCA_OK != ret_err)
			ret_err = MASCA_HE_SELF_TEST_FAIL;
	} else {
		ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
	}
	return ret_err;
}

enum masca_error masca_receive_diagnostic(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	enum masca_error         ret_err = MASCA_OK;
	if (p_output->cmdbuf[RCV_DIAG_REQ_PG_CODE] == 0x00) {
		/*Provide page codes supported by device*/
		/*Page code 00*/
		p_buffer[RCV_DIAG_RESP_PG_CODE] = 0x00;
		/*Two supported pages*/
		p_buffer[RCV_DIAG_RESP_PG_LEN+0x01] = RECEIVE_DIAG_PAGE_LENGTH;
		/*Page code 00 supported*/
		p_buffer[RCV_DIAG_RESP_PG_LIST+0x00] = 0x00;
		/*Vendor Page 80 supported*/
		p_buffer[RCV_DIAG_RESP_PG_LIST+0x01] = RECEIVE_DIAG_PAGE_CODE;
	} else if (p_output->cmdbuf[RCV_DIAG_REQ_PG_CODE] ==
						RECEIVE_DIAG_PAGE_CODE) {
		p_buffer[RCV_DIAG_RESP_PG_CODE] = RECEIVE_DIAG_PAGE_CODE;
		/*Page length MSB*/
		p_buffer[RCV_DIAG_RESP_PG_LEN+0x00] =
				p_output->cmdbuf[RCV_DIAG_REQ_ALLOC_LEN+0x00];
		/*Page length LSB*/
		p_buffer[RCV_DIAG_RESP_PG_LEN+0x01] =
				p_output->cmdbuf[RCV_DIAG_REQ_ALLOC_LEN+0x01];
		if (spi_hist_buff != NULL)
			memcpy(&p_buffer[RCV_DIAG_RESP_PG_LIST],
				spi_hist_buff,
				(p_buffer[RCV_DIAG_RESP_PG_LEN+0x00]  <<
				BYTE_2_START) +
				p_buffer[RCV_DIAG_RESP_PG_LEN+0x01]);
	} else {
		ret_err = MASCA_IR_INVALID_FLD_IN_CDB;
	}

	return ret_err;
}

enum masca_error masca_read_disk_info(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	char last_open_session = 0;
	enum masca_error         ret_err = MASCA_OK;
	enum masca_media_type cd_type;
	struct masca_session_info	*p_sess_info;
	struct masca_cd_toc		*p_toc_info;

	masca_sm_get_session_info(&p_sess_info);
	masca_sm_get_toc(&p_toc_info);
	masca_sm_get_media_type(&cd_type);
	p_buffer[READ_DISC_RESP_DATA_LEN] = 0;
	p_buffer[READ_DISC_RESP_DATA_LEN+0x01] = READ_DISC_INFO_LENGTH;
	if (masca_sm_get_last_sess_state()) {
		p_buffer[READ_DISC_RESP_STATUS] = READ_DISC_COMPLETE_SESSION;
	} else {
		p_buffer[READ_DISC_RESP_STATUS] = READ_DISC_INCOMPLETE_SESSION;
		last_open_session = 1;
	}

	p_buffer[READ_DISC_RESP_FIRST_TRK_NUM] = 1;
	p_buffer[READ_DISC_RESP_SESS_CNT_LSB] =
			p_sess_info->last_complete_session + last_open_session;

	/*Behaviour replicated from USB CDROM response to READ DISK INFO cmd*/
	if (TRUE == is_multisession_on)
		p_buffer[READ_DISC_RESP_FIRST_TRK_LSB] =
						p_toc_info->last_track_number;
	else
		p_buffer[READ_DISC_RESP_FIRST_TRK_LSB] =
						p_toc_info->first_track_number;

	p_buffer[READ_DISC_RESP_LAST_TRK_LSB] = p_toc_info->last_track_number;

	p_buffer[READ_DISC_RESP_URU_BIT] = READ_DISC_UNRESTRICTED_USE;

	/* Disc Type */
	if (DISC_CDROM == cd_type)
		p_buffer[READ_DISC_RESP_CD_TYPE] = READ_DISC_RESP_CDROM_XA_DISC;
	else
		p_buffer[READ_DISC_RESP_CD_TYPE] = READ_DISC_RESP_CDDA_DISC;

	/*No.of Sessions(MSB)*/
	p_buffer[READ_DISC_RESP_SESS_CNT_MSB] = 0;
	/*First track number in Last Session(MSB)*/
	p_buffer[READ_DISC_RESP_FIRST_TRK_MSB] = 0;
	/*Last track number in Last Session(MSB)*/
	p_buffer[READ_DISC_RESP_LAST_TRK_MSB] = 0;

	return ret_err;
}

enum masca_error masca_atapi_convert_media_stat(
				const struct masca_output * const p_output,
				unsigned char * const p_buffer,
				const unsigned int buf_length)
{
	enum masca_media_status media_stat = MASCA_NO_MEDIA;
	enum masca_error         ret_err = MASCA_OK;

	media_stat = p_output->output_params.mda_stat;

	if (buf_length >= 1) {
		if (MASCA_NO_MEDIA == media_stat)
			*p_buffer = ATAPI_NO_MEDIA;
		else if (MASCA_MEDIA_INSERTED == media_stat)
			*p_buffer = ATAPI_MEDIA_INSERTED;
		else if (MASCA_MEDIA_EJECT_PROCESSED == media_stat)
			*p_buffer = ATAPI_MEDIA_EJECT_PROCESSED;
		else
			*p_buffer = ATAPI_NO_UPDATE;
	} else {
		ret_err = MASCA_INVALID_PARAM;
	}

	return ret_err;
}

static unsigned short masca_updt_txt_lng(unsigned short cd_text_lng,
				unsigned short text_length)
{
	unsigned short  rem_pack_bytes;
	if (text_length > 0) {
		/*We will closely pack CD text so that memory is efficiently
		* utilised, hence we can calculate total pay load size,excluding
		* 4 byte header and 2 bytes checksum*/
		/*additional one byte is for separator byte 0*/
		/*each track text information for a particular pack type
		* is terminated with a zero*/
		text_length++;
		rem_pack_bytes = (cd_text_lng % ATAPI_PACK_PAYLOAD_SIZE);
		cd_text_lng += text_length;
		if (0 != rem_pack_bytes) {
			/*We were in the middle of a pack*/
			if ((rem_pack_bytes + text_length) <=
						ATAPI_PACK_PAYLOAD_SIZE) {
				/*The current track data is not big enough.
				*As track number is represented as part of a
				*pack at least one pack is required for the
				*track. Hence we pad additional bytes with 0 */
				cd_text_lng += (ATAPI_PACK_PAYLOAD_SIZE -
							rem_pack_bytes);
			}
		}
	}
	return cd_text_lng;
}

static bool masca_updt_track_text(unsigned char * const p_cd_text_buf,
					unsigned char * const p_trk_text_buf,
					unsigned int * const p_buf_length,
					unsigned short trk_text_length,
					const unsigned char trk_num)
{
	unsigned int         buf_length;
	bool        is_mem_available = TRUE;
	unsigned short         copy_index = 0;
	unsigned char          bytes_in_prev_pack = 0;
	buf_length = *p_buf_length;

	while ((trk_text_length > 0) && (TRUE == is_mem_available)) {
		if (TRUE == updt_header) {
			if (buf_length >= ATAPI_PACK_SIZE) {
				is_mem_available = TRUE;
				buf_length -= ATAPI_PACK_SIZE;
				p_cd_text_buf[cursor_pos] = pack_type;
				cursor_pos++;
				p_cd_text_buf[cursor_pos] = trk_num;
				cursor_pos++;
				p_cd_text_buf[cursor_pos] = sequence_no;
				cursor_pos++;
				p_cd_text_buf[cursor_pos] = bytes_in_prev_pack;
				cursor_pos++;
				sequence_no++;
			} else {
				is_mem_available = FALSE;
			}
		}

		if (TRUE == is_mem_available) {
			if (bytes_left_over > 0) {
				/* we are starting a track text in the middle of
				 * a pack*/
				if (trk_text_length > bytes_left_over) {
					memcpy(
					    (void *)&p_cd_text_buf[cursor_pos],
					    (void *)&p_trk_text_buf[copy_index],
						bytes_left_over);
					trk_text_length -= bytes_left_over;
					copy_index += bytes_left_over;
					bytes_in_prev_pack = bytes_left_over;
				} else {
					/*pad rest of the bytes with 0*/
					memset(&p_cd_text_buf[cursor_pos], 0,
							bytes_left_over);
					copy_index = 0;
					bytes_in_prev_pack = 0;
				}
				cursor_pos += bytes_left_over;
				p_cd_text_buf[cursor_pos] = CHK_SUM_BYTE;
				cursor_pos++;
				p_cd_text_buf[cursor_pos] = CHK_SUM_BYTE;
				cursor_pos++;
				bytes_left_over = 0;
				updt_header = TRUE;
			} else if (trk_text_length > ATAPI_PACK_PAYLOAD_SIZE) {
				memcpy((void *)&p_cd_text_buf[cursor_pos],
					(void *)&p_trk_text_buf[copy_index],
					ATAPI_PACK_PAYLOAD_SIZE);
				copy_index += ATAPI_PACK_PAYLOAD_SIZE;
				cursor_pos += ATAPI_PACK_PAYLOAD_SIZE;
				p_cd_text_buf[cursor_pos] = CHK_SUM_BYTE;
				cursor_pos++;
				p_cd_text_buf[cursor_pos] = CHK_SUM_BYTE;
				cursor_pos++;
				trk_text_length -= ATAPI_PACK_PAYLOAD_SIZE;
				bytes_in_prev_pack = 0x0f;
				updt_header = TRUE;
			} else {
				/* We will start update of the next track
				 * somewhere in the middle*/
				memcpy((void *)&p_cd_text_buf[cursor_pos],
					(void *)&p_trk_text_buf[copy_index],
					(trk_text_length - 1));
				cursor_pos += (trk_text_length - 1);
				copy_index += trk_text_length - 1;
				bytes_left_over = ATAPI_PACK_PAYLOAD_SIZE -
							trk_text_length;
				trk_text_length = 0;
				p_cd_text_buf[cursor_pos] = 0;
				cursor_pos++;

				if (0 == bytes_left_over) {
					/*The payload is full*/
					p_cd_text_buf[cursor_pos] =
								CHK_SUM_BYTE;
					cursor_pos++;
					p_cd_text_buf[cursor_pos] =
								CHK_SUM_BYTE;
					cursor_pos++;
					updt_header = TRUE;
				} else {
					updt_header = FALSE;
				}
			}
		}
	}
	*p_buf_length = buf_length;

	return is_mem_available;
}
