/*
 * linux/drivers/char/exchnd/exchnd_watchdog.c
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Watchdog - Zombies hunter
 *
 * This watchdog purpose is to find and kick any process that was stopped by
 * the driver but not restarted by the daemon. The reason might be because
 * the daemon is not there anymore, is getting stuck or is really, really too
 * slow.
 */
#define pr_fmt(fmt) "exchnd: " fmt

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/workqueue.h>
#include <linux/slab.h>
#include <linux/kallsyms.h>

#include <linux/exchnd.h>
#include "exchnd_internal.h"

static void (*exch_ptrace_unlink)(struct task_struct *child);

static unsigned long exchnd_wd_delay;

static struct workqueue_struct *exchnd_wd_wq;

/**
 * exchnd_init_wd - Initialize the watchdog work queue
 *
 * Return: 0 on success, 1 otherwise.
 */
int exchnd_init_wd(void)
{
	exchnd_wd_wq = create_singlethread_workqueue("exchnd_wd");
	exchnd_wd_delay = msecs_to_jiffies(EXCHND_WD_MSDELAY);

	exch_ptrace_unlink = (void *) kallsyms_lookup_name("__ptrace_unlink");

	if (exchnd_get_debug() & EXCHND_DEBUG_WD)
		pr_info("Watchdog initialized (delay: %d ms).\n",
				EXCHND_WD_MSDELAY);

	return exchnd_wd_wq == NULL;
}

/**
 * exchnd_deinit_wd - Deinitialization of the watchdog work queue
 *
 * Flush and clean-up the work queue dedicated to the watchdog.
 */
void exchnd_deinit_wd(void)
{
	flush_workqueue(exchnd_wd_wq);
	destroy_workqueue(exchnd_wd_wq);

	if (exchnd_get_debug() & EXCHND_DEBUG_WD)
		pr_info("Watchdog deinitialized.\n");
}

/**
 * exchnd_wd_kick - Wakeup a stopped task
 * @pid pid of the task to wake up
 *
 * Retrieve the task struct form PID and wake it up if it exists.
 */
static void exchnd_wd_kick(struct task_struct *task)
{
	unsigned long flags;
	if (!task)
		/* else, maybe the daemon was there finally ?*/
		goto exit;

	pr_err("Continuing task with pid %d.\n", task->pid);
	if (!task->ptrace) {
		/* Task may already be exiting. */
		if (unlikely(task->flags & PF_EXITING))
			goto exit;

		if (exchnd_get_debug() & EXCHND_DEBUG_WD)
			pr_info("Task was not traced.\n");

		spin_lock_irqsave(&task->sighand->siglock, flags);
		exchnd_wakeup_pid_wd(task->pid);
		spin_unlock_irqrestore(&task->sighand->siglock, flags);
	} else {
		if (exchnd_get_debug() & EXCHND_DEBUG_WD)
			pr_info("Task was traced.\n");

		spin_lock_irqsave(&task->sighand->siglock, flags);
		exchnd_wakeup_pid_wd(task->pid);
		spin_unlock_irqrestore(&task->sighand->siglock, flags);
		exch_ptrace_unlink(task);
	}

exit:
	return;
}

/**
 * exchnd_wd_exec - Watchdog execution function
 * @work the work_struct with which we can retrieve our stuff
 *
 * This function will first check if the daemon is around, if not or
 * if it's too late, try to kick the task from pid.
 */
static void exchnd_wd_exec(struct work_struct *work)
{
	struct exchnd_wd_worker *wd_work = (struct exchnd_wd_worker *)work;

	struct task_struct *task =
		pid_task(find_vpid(wd_work->pid), PIDTYPE_PID);

	if (!task) {
		pr_err("Task %d already out but watchdog still running.\n",
				wd_work->pid);
		/* That shall not be possible as exchnd_watch_pid and
		 * exchnd_unwatch are called in the same function.
		 * If the task is not found, that would probably mean that
		 * the PID is not anymore valid due to some memory corruption.
		 */
		BUG();
	}

	if (!wd_work->wakeup_count &&
			waitqueue_active(rb_get_read_wait_queue())) {

		if (exchnd_get_debug() & EXCHND_DEBUG_WD)
			pr_info("WD gets up for %d.\n", wd_work->pid);

		/* Give the daemon the benefit of the doubt. */
		/*TODO: Relaunch delay once and stop task in case daemon
		 * is still attached but stuck
		 */
		wd_work->wakeup_count++;
		queue_delayed_work(exchnd_wd_wq,
				(struct delayed_work *)work,
				exchnd_wd_delay);

		/* Try to wake-it up. */
		wake_up(rb_get_read_wait_queue());
		goto exit;
	}

	if (exchnd_get_debug() & EXCHND_DEBUG_WD)
		pr_info("WD kicks %d.\n", wd_work->pid);

	/* Daemon is still not there or can be stuck.
	 * Kick the task if it still exists.
	 */
	wd_work->wakeup_count = EXCHND_WD_KICKED;
	exchnd_wd_kick(task);

	/* If the daemon is not there we need to wake up
	 * the daemon_wait_queue. A restart may be pending.
	 */
	eq_wake_daemon_wait_queue();

exit:
	return;
}

/**
 * exchnd_watch_pid - Add new entry in the watchdog work queue
 * @pid pid of the task to watch
 *
 * If the driver needs to stop a process, this function takes care on adding
 * the task to the list of task that should be woken up if the daemon is not
 * there anymore after few seconds.
 */
struct exchnd_wd_worker *exchnd_watch_pid(pid_t pid)
{
	struct exchnd_wd_worker *work;

	work = (struct exchnd_wd_worker *)
		kzalloc(sizeof(struct exchnd_wd_worker), GFP_ATOMIC);

	if (!work) {
		if (exchnd_get_debug() & EXCHND_DEBUG_WD)
			pr_warn("No memory for watchdog.\n");
		return NULL;
	}

	if (exchnd_get_pid() && !waitqueue_active(rb_get_read_wait_queue()))
		exchnd_unset_pid();

	INIT_DELAYED_WORK(&work->dwork, exchnd_wd_exec);
	work->pid = pid;
	work->wakeup_count = 0;

	if (exchnd_get_debug() & EXCHND_DEBUG_WD)
		pr_info("Now queuing work for %d.\n", pid);

	queue_delayed_work(exchnd_wd_wq, &work->dwork, exchnd_wd_delay);

	return work;
}

void exchnd_unwatch(struct exchnd_wd_worker *work)
{
	if (!work)
		return;

	if (work->wakeup_count != EXCHND_WD_KICKED) {
		if (exchnd_get_debug() & EXCHND_DEBUG_WD)
			pr_info("Cancel work for %d.\n", current->pid);

		cancel_delayed_work_sync(&work->dwork);
	}

	/* If exchnd_unwatch is called, work can't have been freed.*/
	kfree(work);
}
