/*
 * linux/drivers/char/exchnd/exchnd_triggers_arm.c
 *
 * Arm specific code for exception handler triggers.
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * exception trigger PROCESS_EXIT
 *
 * The exception hooks on a probe on do_exit to trigger when a process ends.
 * Aim is to temporarily modify the call stack, adding a function that will
 * stop the task we want to watch.
 */

#include <linux/list.h>
#include <linux/ptrace.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/exchnd.h>

#include "exchnd_internal.h"

static LIST_HEAD(exit_regs);
static DEFINE_SPINLOCK(exit_regs_lck);

struct exchnd_exit_list {
	struct list_head list;
	unsigned long task;
	struct pt_regs regs;
};

/* probe set to do exit */
static struct kprobe kp_exit;

/**
 * exchnd_stop_exit - Effectively stop task on do_exit
 *
 * This function is added in call stack by exchnd_do_exit. The function triggers
 * the analysis of the process exit.
 */
static void exchnd_stop_exit(void)
{
	unsigned long flags;
	int is_filtered = 0;
	struct exchnd_exit_list *el, *n;
	struct pt_regs save_regs;
	struct pt_regs *regs = NULL;
	struct exception_info *info = NULL;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT] = NULL;
	struct exchnd_wd_worker *work;

	/* First of all recover registers. */
	spin_lock_irqsave(&exit_regs_lck, flags);
	list_for_each_entry_safe(el, n, &exit_regs, list) {
		if (el->task != (unsigned long) current)
			continue;
		memcpy(&save_regs,
			&el->regs,
			sizeof(struct pt_regs));
		regs = &save_regs;
		list_del(&el->list);
		kfree(el);
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("Registers loaded for process %d.\n",
					current->pid);
	};
	spin_unlock_irqrestore(&exit_regs_lck, flags);

	BUG_ON(!regs);

	/* No-one is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		exchnd_unset_pid();
		goto exit;
	}

	/* Ignore exchndd.
	 * We don't want to stop it !
	 */
	if (exchnd_get_pid() == current->group_leader->pid) {
		if (((exchnd_get_debug() & EXCHND_DEBUG_PEXIT) ||
				(exchnd_get_debug() & EXCHND_DEBUG_DAEMON)) &&
				(current->group_leader->pid == current->pid))
			pr_info("%s: Daemon (%d) exiting.\n",
					__func__,
					current->pid);
		goto exit;
	}

	is_filtered =  exchnd_is_filtered(current);
	if ((exchnd_get_debug() & EXCHND_DEBUG_PEXIT) ||
			(exchnd_get_debug() & EXCHND_DEBUG_FILTER))
		pr_info("%s: Filter status for process %d: %d.\n",
				__func__,
				current->pid,
				is_filtered);

	/* If process is not in white list, use default. */
	if (!is_filtered)
		modules = exchnd_pexit_default;
	else
		modules = exchnd_trigger_list[ET_PROCESS_EXIT].modules;

	/* Early exit ...*/
	if ((*modules)[0] == EHM_NONE) {
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("%s: No modules for exit of %d.\n",
					__func__,
					current->pid);
		goto exit;
	}

	info = eq_get_info(ET_PROCESS_EXIT);
	info->task = current;
	info->modules = modules;

	preempt_disable();
	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Preparing to stop task on exit.\n", __func__);

	set_current_state(TASK_TRACED);

	/* Add JOBCTL_STOP_SIGMASK so that ptrace kick this task. */
	current->jobctl |= JOBCTL_STOP_SIGMASK;

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Starting handling exit.\n", __func__);

	info->ready = 1;
	eq_start_handling(ET_PROCESS_EXIT);

	work = exchnd_watch_pid(current->pid);
	if (!work)
		pr_warn("Unable to watch pid: %d\n", current->pid);

	preempt_enable();
	schedule();

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Back from schedule.\n", __func__);

	/* Back from schedule, disable watchdog for this thread. */
	exchnd_unwatch(work);
exit:

	if (test_thread_flag(TIF_MEMDIE))
		exchnd_oom_end();

	__asm__ __volatile__ (
		/*
		 * Return to the context saved on stack.
		 */
		"mov    sp, %0\n\t"

#ifdef CONFIG_THUMB2_KERNEL
		"ldr	lr, [sp, %1]\n\t"	/* lr = saved sp */
		"ldrd	r0, r1, [sp, %4]\n\t"	/* r0,r1 = saved lr,pc */
		"ldr	r2, [sp, %3]\n\t"	/* r2 = saved psr */
		"stmdb	lr!, {r0, r1, r2}\n\t"	/* push saved lr and */
						/* rfe context */
		"ldmia	sp, {r0 - r12}\n\t"
		"mov	sp, lr\n\t"
		"ldr	lr, [sp], #4\n\t"
		"rfeia	sp!\n\t"
#else
		"ldr	r0, [sp, %3]\n\t"
		"msr	cpsr_cxsf, r0\n\t"
		"ldmia	sp, {r0 - pc}\n\t"
#endif
		:
		: "p" (regs),
		  "J" (offsetof(struct pt_regs, ARM_sp)),
		  "J" (offsetof(struct pt_regs, ARM_pc)),
		  "J" (offsetof(struct pt_regs, ARM_cpsr)),
		  "J" (offsetof(struct pt_regs, ARM_lr))
		: "memory", "cc");
}

#ifdef CONFIG_THUMB2_KERNEL
static inline void exchnd_prepare_cpsr(struct pt_regs *regs)
{
	long cpsr;
	cpsr = regs->ARM_cpsr;
	/* Set correct Thumb state in cpsr */
	if (regs->ARM_pc & 1)
		cpsr |= PSR_T_BIT;
	else
		cpsr &= ~PSR_T_BIT;
	regs->ARM_cpsr = cpsr;
}
#else
static inline void exchnd_prepare_cpsr(struct pt_regs *regs) {}
#endif

/**
 * exchnd_do_exit - Function notified by the probe for the process exit
 * @p: probe information
 * @regs: registers at the time of the call
 *
 * This function is called by the probe for "do_exit". The function triggers
 * the analysis of the process exit.
 * We save regs on a list as we are not able to do it on stack on every cases.
 */
void exchnd_do_exit(struct kprobe *p, struct pt_regs *regs,
		unsigned long flags)
{
	struct exchnd_exit_list *new;

	/* No-one is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("Daemon is not active while %d exits.\n",
					current->pid);
		exchnd_unset_pid();

		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		goto exit;
	}

	/* Just ignore kernel threads. No chance for us to trace them. */
	if (unlikely(current->flags & PF_KTHREAD)) {
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("Ignoring kernel thread.\n");

		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		goto exit;
	}

	new = kzalloc(sizeof(struct exchnd_exit_list), GFP_ATOMIC);

	if (!new) {
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_warn("No memory available to handle exit.\n");

		if (test_thread_flag(TIF_MEMDIE))
			exchnd_oom_end();

		goto exit;
	}

	new->task = (unsigned long) current;

	memcpy(&new->regs,
			regs,
			sizeof(struct pt_regs));

	spin_lock_irqsave(&exit_regs_lck, flags);
	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("Saving registers on list for process %d.\n",
				current->pid);
	list_add_tail(&new->list, &exit_regs);
	spin_unlock_irqrestore(&exit_regs_lck, flags);

	regs->ARM_pc = (long)exchnd_stop_exit;

	exchnd_prepare_cpsr(regs);

exit:
	return;
}

/**
 * exchnd_exit_init - Initializes out of process exit trigger
 * @trigger: a pointer to the process exit trigger structure
 *
 * This function will will register a probe with the "do_exit" kernel
 * function.
 *
 * Return: probe structure
 */
void *exchnd_exit_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	kp_exit.symbol_name = "do_exit";
	kp_exit.post_handler = exchnd_do_exit;

	if (register_kprobe(&kp_exit) == 0)
		trigger->opaque = &kp_exit;

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("Process exit initialized.\n");

	return trigger->opaque;
}

/**
 * exchnd_exit_deinit - Deinitializes process exit trigger
 * @trigger: a pointer to the process exit structure
 *
 * This function will deinitialize the process exit trigger by registering
 * the registered probe.
 */
void exchnd_exit_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		unregister_kprobe(trigger->opaque);

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("Process exit deinitialized.\n");
}
