/*
 * linux/drivers/char/exchnd/exchnd_triggers_arm.c
 *
 * Arm specific code for exception handler triggers.
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Frederic Berat (fberat@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

#include <linux/thread_info.h>
#include <linux/notifier.h>
#include <linux/profile.h>
#include <linux/sched.h>

#include "exchnd_internal.h"
/**
 * exchnd_do_exit - Stop task on do_exit
 *
 * This function is called early in do_exit once the PROFILING is enabled.
 * That's the simplest way to monitor tasks exit.
 */
int exchnd_do_exit(struct notifier_block *nb,
		unsigned long e,
		void *data)
{
	int is_filtered = 0;
	struct exception_info *info = NULL;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT] = NULL;
	struct exchnd_wd_worker *work = NULL;
	struct task_struct *task = data;

	/* No-one is there to handle this event */
	/* TODO: Mechanism to handle this event internally anyway ? */
	if (!waitqueue_active(rb_get_read_wait_queue())) {
		exchnd_unset_pid();
		goto exit;
	}

	/* Ignore exchndd.
	 * We don't want to stop it !
	 */
	if (exchnd_get_pid() == task->group_leader->pid) {
		if (((exchnd_get_debug() & EXCHND_DEBUG_PEXIT) ||
				(exchnd_get_debug() & EXCHND_DEBUG_DAEMON)) &&
				(task->group_leader->pid == current->pid))
			pr_info("%s: Daemon (%d) exiting.\n",
					__func__,
					task->pid);
		goto exit;
	}

	is_filtered =  exchnd_is_filtered(task);
	if ((exchnd_get_debug() & EXCHND_DEBUG_PEXIT) ||
			(exchnd_get_debug() & EXCHND_DEBUG_FILTER))
		pr_info("%s: Filter status for process %d: %d.\n",
				__func__,
				task->pid,
				is_filtered);

	/* If process is not in white list, use default. */
	if (!is_filtered)
		modules = exchnd_pexit_default;
	else
		modules = exchnd_trigger_list[ET_PROCESS_EXIT].modules;

	/* Early exit ...*/
	if ((*modules)[0] == EHM_NONE) {
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("%s: No modules for exit of %d.\n",
					__func__,
					task->pid);
		goto exit;
	}

	info = eq_get_info(ET_PROCESS_EXIT);
	info->task = task;
	info->modules = modules;

	preempt_disable();
	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Preparing to stop task on exit.\n", __func__);

	set_task_state(task, TASK_TRACED);

	/* Add JOBCTL_STOP_SIGMASK so that ptrace kick this task. */
	task->jobctl |= JOBCTL_STOP_SIGMASK;

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Starting handling exit.\n", __func__);

	info->ready = 1;
	eq_start_handling(ET_PROCESS_EXIT);

	work = exchnd_watch_pid(task->pid);
	if (!work)
		pr_warn("Unable to watch pid: %d\n", task->pid);

	preempt_enable();
	schedule();

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("%s: Back from schedule.\n", __func__);

	/* Back from schedule, disable watchdog for this thread. */
	exchnd_unwatch(work);

exit:
	if (test_thread_flag(TIF_MEMDIE))
		exchnd_oom_end();

	return NOTIFY_OK;
}

static struct notifier_block nb_exit = {
	.notifier_call = exchnd_do_exit
};

/**
 * exchnd_exit_init - Initializes out of process exit trigger
 * @trigger: a pointer to the process exit trigger structure
 *
 * This function will will register a callback for the "do_exit" profiling.
 *
 * Return: pro structure
 */
void *exchnd_exit_init(struct exchnd_trigger *trigger)
{
	trigger->opaque = NULL;

	if (!profile_event_register(PROFILE_TASK_EXIT, &nb_exit)) {
		trigger->opaque = &nb_exit;
		if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
			pr_info("Process exit initialized.\n");
	} else if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("Process exit initialization failed.\n");

	return trigger->opaque;
}

/**
 * exchnd_exit_deinit - Deinitializes process exit trigger
 * @trigger: a pointer to the process exit structure
 *
 * This function will deinitialize the process exit trigger by deregistering
 * the registered probe.
 */
void exchnd_exit_deinit(struct exchnd_trigger *trigger)
{
	if (trigger->opaque)
		profile_event_unregister(PROFILE_TASK_EXIT, trigger->opaque);

	if (exchnd_get_debug() & EXCHND_DEBUG_PEXIT)
		pr_info("Process exit deinitialized.\n");

}
