/*
 * linux/drivers/char/exchnd/exchnd_modules_x86.c
 *
 * Arch specific exception handler internal module handling.
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by:
 *      Frederic Berat (fberat@de.adit-jv.com)
 *      Shoji Morita (smorita@jp.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception Handler Modules
 *
 * The exception handler modules collect data of the exception or trigger
 * specific action. They allow to configured what data is collected and what
 * actions are done per exception type.
 */

#include <linux/stacktrace.h>
#include <asm/stacktrace.h>
#include <linux/exchnd.h>
#include "exchnd_internal.h"

#define EHM_BACKTRACE_HEADER "====== backtrace:"

/*
 * Exception handler kernel backtrace collector.
 *
 * Walk and record stack.
 */
int exchnd_kbacktrace(struct pt_regs *regs,
		struct task_struct *tsk,
		unsigned char *data)
{
	unsigned int len = 0;
	struct stack_trace trace;
	unsigned long entries[EXCHND_MAX_TRACE_ENTRIES];
	int i = 0;

	/* Init trace */
	trace.nr_entries = 0;
	trace.max_entries = ARRAY_SIZE(entries);
	trace.entries = entries;
	trace.skip = 0;

	if (!tsk)
		tsk = current;

	save_stack_trace_tsk(tsk, &trace);

	/* Now trace is stored, dump it into data. */
	for (i = 0; i < trace.nr_entries; i++) {
		len += snprintf(data + len,
				EXCHND_MAX_TRACE_LEN - len,
				"\n [<%p>] %pS",
				(void *) trace.entries[i],
				(void *) trace.entries[i]);
		if (len >= EXCHND_MAX_TRACE_LEN) {
			len--;
			break;
		}
	}

	data[len] = '\0';

	return len;
}

unsigned long exchnd_get_fault(struct task_struct *task)
{
	return task->thread.cr2;
}

#define EHM_PROCESSOR_REGISTERS_HEADER "====== processor registers:\n"

#	ifdef __i386__
/*
 * exception handler register dumper for X86
 *
 * Architecture dependent.
 */
static int exchnd_dump_regs_i386(unsigned char *data, struct pt_regs *regs)
{
	return snprintf(data,
			EXCHND_MAX_TRACE_LEN,
			EHM_PROCESSOR_REGISTERS_HEADER
			"bx:  %08lx "
			"cx:  %08lx "
			"dx:  %08lx "
			"si:  %08lx "
			"di:  %08lx "
			"bp:  %08lx\n"
			"ax:  %08lx "
			"ds:  %08lx "
			"es:  %08lx "
			"fs:  %08lx "
			"gs:  %08lx\n"
			"orig_ax:  %08lx "
			"ip:  %08lx "
			"cs:  %08lx "
			"flags:  %08lx "
			"sp:  %08lx "
			"ss: %08lx ",
			regs->bx,
			regs->cx,
			regs->dx,
			regs->si,
			regs->di,
			regs->bp,
			regs->ax,
			regs->ds,
			regs->es,
			regs->fs,
			regs->gs,
			regs->orig_ax,
			regs->ip,
			regs->cs,
			regs->flags,
			regs->sp,
			regs->ss);
}
#	else /* then !__i386__ */
/*
 * exception handler register dumper for i386
 *
 * Architecture dependent.
 */
static int exchnd_dump_regs_x86(unsigned char *data, struct pt_regs *regs)
{
	return snprintf(data,
			EXCHND_MAX_TRACE_LEN,
			EHM_PROCESSOR_REGISTERS_HEADER
			"r15:  %08lx "
			"r14:  %08lx "
			"r13:  %08lx "
			"r12:  %08lx\n"
			"r11:  %08lx "
			"r10:  %08lx "
			"r9:   %08lx "
			"r8:   %08lx\n"
			"bx:  %08lx "
			"cx:  %08lx "
			"dx:  %08lx "
			"si:  %08lx "
			"di:  %08lx "
			"bp:  %08lx\n"
			"ax:  %08lx "
			"ds:  %08lx "
			"es:  %08lx "
			"fs:  %08lx "
			"gs:  %08lx\n"
			"orig_ax:  %08lx "
			"ip:  %08lx "
			"cs:  %08lx "
			"flags:  %08lx "
			"sp:  %08lx "
			"ss: %08lx ",
			regs->r15,
			regs->r14,
			regs->r13,
			regs->r12,
			regs->r11,
			regs->r10,
			regs->r9,
			regs->r8,
			regs->bx,
			regs->cx,
			regs->dx,
			regs->si,
			regs->di,
			regs->bp,
			regs->ax,
			regs->ds,
			regs->es,
			regs->fs,
			regs->gs,
			regs->orig_ax,
			regs->ip,
			regs->cs,
			regs->flags,
			regs->sp,
			regs->ss);
}
#	endif /* !__i386__ */

#ifdef __i386__
int exchnd_dump_regs(unsigned char *data,
		struct pt_regs *regs,
		__attribute__ ((unused))struct task_struct *task)
{
	return exchnd_dump_regs_i386(data, regs);
}
#else
int exchnd_dump_regs(unsigned char *data,
		struct pt_regs *regs,
		__attribute__ ((unused))struct task_struct *task)
{
	return exchnd_dump_regs_x86(data, regs);
}
#endif /* __i386__ */

unsigned long *exchnd_get_sp(struct task_struct *task)
{
	return (unsigned long *)KSTK_ESP(task);
}
