/*
 * linux/drivers/char/exchnd/exchnd_ioctl.c
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Exception I/O Control
 *
 * The exception handler is controlled via I(O controls of the exception
 * handler character device. The implementation behind the I/O controls
 * is found here.
 */
#define pr_fmt(fmt) "exchnd: " fmt

#include <linux/signal.h>
#include <linux/slab.h>
#include <linux/exchnd.h>
#include <linux/sched.h>
#include <linux/uaccess.h>

#include "exchnd_internal.h"

/**
 * modify_trigger_config - Sets EHM per trigger in the configuration
 * @param: ioctl parameter - contains trigger and config
 *
 * This function modify the configured modules for a trigger. It is a helper
 * for the ioctl function. It retrieves an array containing the trigger id and
 * the module ids to modify_trigger.
 *
 * Return: 0 for success, error code otherwise
 */
static long modify_trigger_config(unsigned long param)
{
	enum exchnd_triggers req_trig;
	unsigned int i;

	/* First element contains requested trigger */
	if (copy_from_user(&req_trig, (void __user *) param,
	  sizeof(enum exchnd_triggers)))
		return -EACCES;
	if ((req_trig < ET_NONE) || (req_trig >= ET_LAST_ELEMENT))
		return -EINVAL;
	/* Empty configuration array */
	for (i = 0; i < EHM_LAST_ELEMENT; i++)
		(*exchnd_trigger_list[req_trig].modules)[i] = EHM_NONE;
	/* Skip trigger in array */
	param += sizeof(enum exchnd_modules);
	if (copy_from_user(exchnd_trigger_list[req_trig].modules,
	  (void __user *) param,
	  sizeof(*exchnd_trigger_list[req_trig].modules)))
		return -EACCES;

	return 0;
}

/**
 * modify_trigger_config_set - Sets default trigger's configuration set.
 * @param: ioctl parameter - contains configuration set to select.
 *
 * This function modify_triggers the configured modules for a trigger.
 * It is a helper for the ioctl function. It retrieves an array containing
 * the trigger id and the module ids to modify_trigger.
 *
 * Return: 0 for success, error code otherwise
 */
static long modify_trigger_config_set(unsigned long param)
{
	enum exchnd_conf req_conf;
	unsigned int i;

	/* First element contains requested trigger */
	if (copy_from_user(&req_conf, (void __user *) param,
	  sizeof(enum exchnd_conf)))
		return -EACCES;

	if ((req_conf < EC_NO_MODULES) || (req_conf >= EC_LAST_ELEMENT))
		return -EINVAL;

	/* Modify default conf */
	default_trigger_set = req_conf;

	/* Setup exception triggers */
	for (i = 0; i < ET_LAST_ELEMENT; i++) {
		/* Attach modules */
		exchnd_module_attach(&exchnd_trigger_list[i].modules,
				&trigger_conf[default_trigger_set][i]);
	}

	return 0;
}

/**
 * modify_signal_config - Select EHMs for a specific signal.
 * @param: ioctl parameter - contains signal and config
 *
 * This function sets the configured modules for a signal. It is a helper
 * for the ioctl function. It retrieves an array containing the signal id and
 * the module ids to set. If the specified signal doesn't already have
 * a specific configuration it creates it. It overrides the existing one
 * otherwise. This doesn't affect the default configuration set.
 *
 * Return: 0 for success, error code otherwise
 */
static long modify_signal_config(unsigned long param)
{
	int req_sig;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT];
	unsigned int i;

	/* First element contains requested signal */
	if (copy_from_user(&req_sig, (void __user *) param,
	  sizeof(enum exchnd_modules)))
		return -EACCES;
	if ((req_sig < 1) || (req_sig >= SIGUNUSED))
		return -EINVAL;
	/* If specific configuration doesn't exist, create a new one */
	modules = exchnd_signal_list[req_sig];

	/* Init module list */
	for (i = 0; i < EHM_LAST_ELEMENT; i++)
		(*modules)[i] = EHM_NONE;

	/* Skip signal ID in array */
	param += sizeof(enum exchnd_modules);

	/* Copy list form user */
	if (copy_from_user(modules, (void __user *) param, sizeof(*modules)))
		return -EACCES;

	return 0;
}

/**
 * modify_signal_config_set - Sets default signal's configuration set.
 * @param: ioctl parameter - contains configuration set to select.
 *
 * This function modify_signals the configured modules for a signal.
 * It is a helper for the ioctl function. It retrieves an array containing
 * the signal id and the module ids to modify_signal.
 *
 * Return: 0 for success, error code otherwise
 */
static long modify_signal_config_set(unsigned long param)
{
	enum exchnd_signal_conf req_conf;
	unsigned int i;

	/* First element contains requested signal */
	if (copy_from_user(&req_conf, (void __user *) param,
	  sizeof(enum exchnd_conf)))
		return -EACCES;

	if ((req_conf < ESC_NO_MODULES) || (req_conf >= ESC_LAST_ELEMENT))
		return -EINVAL;

	/* Modify default conf */
	default_signal_set = req_conf;

	/* Add module list to signals */
	for (i = 1; i < SIGUNUSED; i++) {
		/* Attach modules */
		exchnd_module_attach(&exchnd_signal_list[i],
				&signal_conf[default_signal_set][i]);
	}

	return 0;
}

/**
 * modify_signal_handled_mask - Modifies the signal handled mask
 *
 * @param: ioctl parameter - container for new mask
 *
 * This function gets the new mask to use for signal handling from user.
 *
 * Return: 0 for success, error code otherwise.
 */
static long modify_signal_handled_mask(unsigned long param)
{
	if (copy_from_user(&exchnd_sighdl_mask,
				(void __user *) param,
				sizeof(unsigned long)))
		return -EACCES;

	return 0;
}

/**
 * modify_config - Sets the configured modules per trigger in the configuration
 * @param: ioctl parameter - contains trigger and config
 *
 * This function sets the configured modules for a trigger. It is a helper
 * for the ioctl function. It retrieves an array containing the trigger id and
 * the module ids to set.
 *
 * Return: 0 for success, error code otherwise
 */
static long modify_config(unsigned long param)
{
	enum exchnd_conf_type conf_type = 0;

	/* First element contains configuration type */
	if (copy_from_user(&conf_type,
				(void __user *) param,
				sizeof(enum exchnd_conf_type)))
		return -EACCES;

	/* Skipping consumed conf_type */
	param += sizeof(enum exchnd_conf_type);

	switch (conf_type) {
	case EXCHND_CONF_TRIGGER:
		return modify_trigger_config(param);
	case EXCHND_CONF_TRIGGER_SET:
		return modify_trigger_config_set(param);
	case EXCHND_CONF_SIGNAL:
		return modify_signal_config(param);
	case EXCHND_CONF_SIGNAL_SET:
		return modify_signal_config_set(param);
	case EXCHND_CONF_SIGHDL_MASK:
		return modify_signal_handled_mask(param);

	default:
		return -EINVAL;
	}

	return 0;

}

/**
 * retrieve_trigger_config - Retrieves the configured modules per trigger from config
 * @param: ioctl parameter - contains the trigger as first element of the array
 *
 * This function reads and returns the configuration of the modules of the
 * given trigger. The configuration is returned in the ioctl parameter.
 *
 * Return: 0 for success, error code otherwise
 */
static long retrieve_trigger_config(unsigned long param)
{
	enum exchnd_triggers req_trig;

	/* First element contains requested trigger */
	if (copy_from_user(&req_trig, (void __user *) param,
	  sizeof(enum exchnd_triggers)))
		return -EACCES;
	if ((req_trig < ET_NONE) || (req_trig >= ET_LAST_ELEMENT))
		return -EINVAL;

	if (copy_to_user((void __user *) param,
	  exchnd_trigger_list[req_trig].modules,
	  sizeof(*exchnd_trigger_list[req_trig].modules)))
		return -EACCES;

	return 0;
}

/**
 * retrieve_trigger_config_set - Retrieves EHMs for a trigger's configuration set.
 * @param: ioctl parameter - contains the trigger as first element of the array
 *
 * This function reads and returns the configuration of the modules of the
 * given trigger's set. The configuration is returned in the ioctl parameter.
 * If no signal set is given, then return currently selected set ID.
 *
 * Return: 0 for success, error code otherwise
 */
static long retrieve_trigger_config_set(unsigned long param)
{
	enum exchnd_conf req_conf;

	/* First element contains requested trigger */
	if (copy_from_user(&req_conf, (void __user *) param,
	  sizeof(enum exchnd_conf)))
		return -EACCES;
	if ((req_conf < EC_NO_MODULES - 1) || (req_conf >= EC_LAST_ELEMENT))
		return -EINVAL;

	if (!req_conf) {
		if (copy_to_user((void __user *) param,
					&default_trigger_set,
					sizeof(default_trigger_set)))
			return -EACCES;
	} else if (copy_to_user((void __user *) param,
					trigger_conf[req_conf],
					sizeof(trigger_conf[req_conf])))
			return -EACCES;

	return 0;
}

/**
 * retrieve_signal_config - Retrieves EHMs for a specific signal.
 *                          Can be different from default configuration.
 * @param: ioctl parameter - contains the signal as first element of the array
 *
 * This function reads and returns the configuration of the modules of the
 * given signal. The configuration is returned in the ioctl parameter.
 *
 * Return: 0 for success, error code otherwise
 */
static long retrieve_signal_config(unsigned long param)
{
	int req_sig;

	/* First element contains requested signal */
	if (copy_from_user(&req_sig, (void __user *) param,
	  sizeof(enum exchnd_modules)))
		return -EACCES;
	if ((req_sig < 1) || (req_sig >= SIGUNUSED))
		return -EINVAL;

	if (copy_to_user((void __user *) param,
				exchnd_signal_list[req_sig],
				sizeof(*exchnd_signal_list[req_sig])))
		return -EACCES;

	return 0;
}

/**
 * retrieve_signal_config_set - Retrieves signal's EHM from a configuration set.
 * @param: ioctl parameter - contains the signal as first element of the array
 *
 * This function reads and returns the configuration of the modules of the
 * given signal's set. The configuration is returned in the ioctl parameter.
 * If no signal set is given, then return currently selected set ID.
 *
 * Return: 0 for success, error code otherwise
 */
static long retrieve_signal_config_set(unsigned long param)
{
	enum exchnd_signal_conf req_conf;

	/* First element contains requested signal */
	if (copy_from_user(&req_conf, (void __user *) param,
	  sizeof(enum exchnd_signal_conf)))
		return -EACCES;
	if ((req_conf < ESC_NO_MODULES - 1) || (req_conf >= ESC_LAST_ELEMENT))
		return -EINVAL;

	if (!req_conf) {
		if (copy_to_user((void __user *) param,
					&default_signal_set,
					sizeof(default_signal_set)))
			return -EACCES;
	} else if (copy_to_user((void __user *) param,
				signal_conf[req_conf],
				sizeof(signal_conf[req_conf])))
		return -EACCES;

	return 0;
}

/**
 * retrieve_signal_handled_mask - Retrieves the signal handled mask
 *
 * @param: ioctl parameter - container for answer
 *
 * This function provides to user the actual mask used for signal handling.
 *
 * Return: 0 for success, error code otherwise.
 */
static long retrieve_signal_handled_mask(unsigned long param)
{
	if (copy_to_user((void __user *) param,
				&exchnd_sighdl_mask,
				sizeof(unsigned long)))
		return -EACCES;

	return 0;
}

/**
 * retrieve_config - Retrieves the configured modules per trigger/signal
 * @param: ioctl parameter - Contains configuration type to retrieve as the
 *                           first parameter.
 *                           Contains the trigger, the signal, the trigger set's
 *                           ID or the signal set's ID as second element
 *                           of the array.
 *
 * This function reads and returns the configuration of the modules of the
 * given value. The configuration is returned in the ioctl parameter.
 *
 * Return: 0 for success, error code otherwise
 */
static long retrieve_config(unsigned long param)
{
	enum exchnd_conf_type conf_type = 0;

	/* First element contains configuration type */
	if (copy_from_user(&conf_type,
				(void __user *) param,
				sizeof(enum exchnd_conf_type)))
		return -EACCES;

	/* Skipping consumed conf_type */
	param += sizeof(enum exchnd_conf_type);

	switch (conf_type) {
	case EXCHND_CONF_TRIGGER:
		return retrieve_trigger_config(param);
	case EXCHND_CONF_TRIGGER_SET:
		return retrieve_trigger_config_set(param);
	case EXCHND_CONF_SIGNAL:
		return retrieve_signal_config(param);
	case EXCHND_CONF_SIGNAL_SET:
		return retrieve_signal_config_set(param);
	case EXCHND_CONF_SIGHDL_MASK:
		return retrieve_signal_handled_mask(param);
	default:
		return -EINVAL;
	}

	return 0;
}

static long modify_module_config(unsigned long in)
{
	enum exchnd_conf_type conf_type;
	struct exchnd_module *module = NULL;
	enum exchnd_modules module_id;
	int param = 0;

	if (copy_from_user(&conf_type,
				(void __user *) in,
				sizeof(enum exchnd_conf_type)))
		return -EACCES;

	in += sizeof(enum exchnd_conf_type);

	if ((conf_type == EXCHND_CONF_MODULE_ENABLE) ||
			(conf_type == EXCHND_CONF_MODULE_DISABLE)) {

		if (copy_from_user(&module_id,
					(void __user *) in,
					sizeof(enum exchnd_modules)))
			return -EACCES;

		if ((module_id <= EHM_NONE) ||
				(module_id >= EHM_LAST_ELEMENT))
			return -EINVAL;

	} else {
		if (copy_from_user(&param, (void __user *) in, sizeof(int)))
			return -EACCES;
	}

	module = &exchnd_module_list[module_id];

	switch (conf_type) {
	case EXCHND_CONF_MODULE_ENABLE:
		pr_err("Enabling %s module.\n", exchnd_mod_names[module_id]);
		disabled_modules &= ~(1 << module_id);
		if (module->init != NULL)
			module->init(module);
		break;
	case EXCHND_CONF_MODULE_HIST_SIZE:
	{
		struct exchnd_module *msyscalls =
			&exchnd_module_list[EHM_HIST_SYSCALLS];
		struct exchnd_module *mtswitch =
			&exchnd_module_list[EHM_HIST_TASKSWITCHES];

		if (!(disabled_modules & (1 << EHM_HIST_SYSCALLS)))
			msyscalls->deinit(msyscalls);
		if (!(disabled_modules & (1 << EHM_HIST_TASKSWITCHES)))
			mtswitch->deinit(mtswitch);

		exchnd_set_hist_size(param);

		if (!(disabled_modules & (1 << EHM_HIST_SYSCALLS)))
			msyscalls->init(msyscalls);
		if (!(disabled_modules & (1 << EHM_HIST_TASKSWITCHES)))
			mtswitch->init(mtswitch);

		break;
	}
	case EXCHND_CONF_MODULE_HIST_PID:
		exchnd_set_trace_pid(param);
		break;
	default:
		pr_err("Disabling %s module.\n", exchnd_mod_names[module_id]);
		disabled_modules |= (1 << module_id);
		if (module->deinit != NULL)
			module->deinit(module);
		break;
	}
	return 0;
}

/**
 * exchnd_fop_ioctl - ioctl function for exception handler driver
 * @filp: file handle of the requesting process
 * @ioctl: number of the requested ioctl
 * @in: ioctl parameter
 *
 * Controls the exception handler modules and configures them. The I/O controls
 * allow to configure the handling of the exception and may trigger specific
 * action, e.g. an exception or are used for synchronization.
 *
 * Return: 0 for success, error code otherwise
 */
long exchnd_fop_ioctl(struct file *filp, unsigned int ioctl, unsigned long in)
{
	switch (ioctl) {
	case IOCTL_EXCHND_VERSION: {
		unsigned long out = EXCHND_VERSION;
		if (copy_to_user((void __user *)in, &out, sizeof(out)))
			return -EACCES;
		return 0;
	}

	case IOCTL_EXCHND_ON_DEMAND: {
		struct exchnd_on_demand param;
		if (copy_from_user(&param,
					(void __user *) in,
					sizeof(param)))
			return -EINVAL;
		return exchnd_on_demand(&param);
	}

	case IOCTL_EXCHND_CONFIGURATION_GET:
		return retrieve_config(in);

	case IOCTL_EXCHND_CONFIGURATION_SET:
		return modify_config(in);

	case IOCTL_EXCHND_DAEMON_READY: {
		pid_t process = 0;
		if (copy_from_user(&process,
					(void __user *) in,
					sizeof(pid_t)))
			return -EACCES;

		if (process > 0) {
			if (exchnd_get_debug() & EXCHND_DEBUG_DAEMON)
				pr_info("Daemon wakes up %d.\n", process);
			exchnd_wakeup_pid(process);
		} else {
			if (exchnd_get_debug() & EXCHND_DEBUG_DAEMON)
				pr_info("Daemon is ready.\n");
			eq_sync_wake_up();
		}
		return 0;
	}

	case IOCTL_EXCHND_MODIFY_FILTER: {
		static DEFINE_SPINLOCK(mod_filter_lock);
		struct exchnd_conf_filter process;
		long rc;
		if (copy_from_user(&process,
					(void __user *) in,
					sizeof(struct exchnd_conf_filter)))
			return -EACCES;

		spin_lock(&mod_filter_lock);
		if (process.add)
			rc = exchnd_process_filter_add(
					process.id,
					process.is_kernel);
		else
			rc = exchnd_process_filter_remove(
					process.id,
					process.is_kernel);
		spin_unlock(&mod_filter_lock);

		return rc;
	}

	case IOCTL_EXCHND_APP_SPECIFIC: {
		struct exchnd_conf_app_spec *conf;

		conf = kzalloc(sizeof(struct exchnd_conf_app_spec), GFP_KERNEL);
		if (!conf)
			return -ENOMEM;

		if (copy_from_user(conf,
					(void __user *) in,
					sizeof(struct exchnd_conf_app_spec)))
			return -EACCES;

		if (exchnd_set_lib_path(conf))
			return -EINVAL;

		kfree(conf);
		return 0;
	}

	case IOCTL_EXCHND_RECOVERY: {
		struct exchnd_file_private *private =
		    (struct exchnd_file_private *)(filp->private_data);

		if (!private || !private->read_permit)
			return -EACCES;

		rb_read_recover();
		return 0;
	}

	case IOCTL_EXCHND_MODULES:
		return modify_module_config(in);

	default:
		/* invalid ioctl */
		return -ENOSYS;
	}
}
