/*
 * linux/include/linux/exchnd_internal.h
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Matthias Weise (mweise@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 *
 */

#ifndef __EXCHND_INTERNAL_H__
#define __EXCHND_INTERNAL_H__
#include <linux/kprobes.h>
#include <linux/platform_device.h>
#include <linux/sched.h>
#include <linux/exchnd.h>

#define EXCHND_MINOR	0xec
#define EXCHND_MAX_TRACE_ENTRIES 25
/* Let limit each entry to 80 chars.*/
#define EXCHND_MAX_ENTRY_LEN (80*sizeof(char))
#define EXCHND_MAX_TRACE_LEN (EXCHND_MAX_TRACE_ENTRIES*EXCHND_MAX_ENTRY_LEN)

/* Exchnd private structure accessors */
atomic_t *exchnd_get_locker(void);
void exchnd_set_pid(unsigned long pid);
unsigned long exchnd_get_pid(void);
void exchnd_unset_pid(void);
void exchnd_oom_start(void);
void exchnd_oom_end(void);
int exchnd_oom_ongoing(void);
int exchnd_get_debug(void);
int exchnd_set_lib_path(struct exchnd_conf_app_spec *conf);

/* Private data per file handle of user space character device access */
struct exchnd_file_private {
	bool read_permit;
};

/* Debug levels */
#define EXCHND_DEBUG_SIGNAL	0x01 /* Signal specific */
#define EXCHND_DEBUG_PEXIT	0x02 /* Process exit specific */
#define EXCHND_DEBUG_TRIGGER	0x04 /* Any other trigger */
#define EXCHND_DEBUG_HEADER	0x08 /* Dump header information */
#define EXCHND_DEBUG_WD		0x10 /* Watchdog specific */
#define EXCHND_DEBUG_RB		0x20 /* Ring buffer: Indexes and data written */
#define EXCHND_DEBUG_DAEMON	0x40 /* Daemon events (poll and IOCTL) */
#define EXCHND_DEBUG_FILTER	0x80 /* Dump filter data (add/remove/tests) */
#define EXCHND_DEBUG_MODULE	0x100 /* Provide module specific information */

/* Prototype of function to write exception data */
typedef void (*exc_write_func)(struct exchnd_message_header *header,
	unsigned char *data);

/* Structure containing information of an exception */
struct exception_info {
	enum exchnd_triggers trigger;
	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT];
	exc_write_func write_func;
	struct task_struct *task;
	/* SP to be used only in case task is not available directly */
	struct pt_regs *regs;
	int sig;
	int ready;
	/* Allows to know whether the *module has been allocated dynamically */
	int mod_type;
	/* Message to add instead of EXCEPTION in the header */
	char msg[256];
	unsigned int syscall_index;
	unsigned int tswitch_index;
};
/* mod_type available */
#define EXCHND_MOD_STAT	0
#define EXCHND_MOD_DYN	1

/* List of all triggers supported by the exception handler */
/* Structure containing the information of one exception handler module */
struct exchnd_module {
	void *opaque;
	void* (*init)(struct exchnd_module *);
	int (*execute)(struct exception_info *, enum exchnd_modules);
	void (*deinit)(struct exchnd_module *);
};

/* Structure containing the information of one exception handler trigger */
struct exchnd_trigger {
	void *opaque;
	enum exchand_fatality fatality;

	enum exchnd_modules (*modules)[EHM_LAST_ELEMENT];

	void* (*init)(struct exchnd_trigger *);
	void (*deinit)(struct exchnd_trigger *);
};

/* Helper to attach module list */
void exchnd_module_attach(enum exchnd_modules (**dest)[EHM_LAST_ELEMENT],
		enum exchnd_modules (*src)[EHM_LAST_ELEMENT]);

/* Helper to restart a task */
void exchnd_wakeup_pid(pid_t);
void exchnd_wakeup_pid_wd(pid_t);

/* Check if PC is a kernel text address */
extern int (*exchnd_ktext_add)(unsigned long);

/* List of default trigger's configurations */
extern enum exchnd_modules
	trigger_conf[EC_LAST_ELEMENT][ET_LAST_ELEMENT][EHM_LAST_ELEMENT];
extern int default_trigger_set;

extern enum exchnd_modules
	signal_conf[ESC_LAST_ELEMENT][SIGUNUSED][EHM_LAST_ELEMENT];
extern int default_signal_set;

/* Handle unhandled signals ? */
extern unsigned int exchnd_sighdl_mask;

extern enum exchnd_modules pexit_conf[EPEC_LAST_ELEMENT][EHM_LAST_ELEMENT];

extern struct exchnd_trigger exchnd_trigger_list[ET_LAST_ELEMENT];
extern enum exchnd_modules (*exchnd_signal_list[SIGUNUSED])[EHM_LAST_ELEMENT];
extern enum exchnd_modules (*exchnd_pexit_default)[EHM_LAST_ELEMENT];

/* List of all available data collection and action modules */
extern struct exchnd_module exchnd_module_list[EHM_LAST_ELEMENT];

/* I/O control function which is implemented in separate file */
extern long exchnd_fop_ioctl(struct file *filp, unsigned int ioctl,
	unsigned long in);

/* Function to initialize the ring buffer */
int rb_init(struct platform_device *pdev);
/* Function to check if buffer is empty */
bool rb_empty(void);
/* Function to recover read pointer */
void rb_read_recover(void);
/* Function to read from the buffer to the  user space */
int rb_read(unsigned char *data, unsigned int size);
/* Function to write to the buffer of the user space device */
void rb_write(struct exchnd_message_header *header, unsigned char *data);
/* Function to wake up waiting writes */
void rb_wakeup_write_wait_queue(void);
/* Function to wait for data in ring buffer */
int rb_wait_read_wait_queue(void);
/* Function to get the read queue handle */
wait_queue_head_t *rb_get_read_wait_queue(void);
/* Initialize exception queue */
int eq_init(void);
/* Deinitialize exception queue */
void eq_deinit(void);
/* Wake up exception queue after synchronization */
void eq_sync_wake_up(void);
/* Function to wake daemon queue if the daemon is dead */
void eq_wake_daemon_wait_queue(void);
/* Get free slot from exception queue */
struct exception_info *eq_get_info(enum exchnd_triggers trigger);
/* Notify worker thread of new exception in queue */
void eq_start_handling(enum exchnd_triggers trigger);
/* Safely add module list on info */
void eq_add_modules(struct exception_info *info,
		enum exchnd_modules (*modules)[EHM_LAST_ELEMENT],
		int mode);
/* Function to write to the buffer directly to the error memory */
void em_write(struct exchnd_message_header *header, unsigned char *data);
/* Simplified function to write a string as exception message */
void exc_write_string(unsigned char *string, struct exception_info *info);
/* Write default header for an exception */
void write_exc_header(struct exception_info *info);
/* Adds task to the filtered list */
int exchnd_process_filter_add(const char *id, unsigned int kernel);
/* Remove task from the filtered list */
int exchnd_process_filter_remove(const char *id, unsigned int kernel);
/* Checks if task shall be handled or not */
int exchnd_is_filtered(struct task_struct *task);

/* Process exit init/deinit */
void *exchnd_exit_init(struct exchnd_trigger *);
void exchnd_exit_deinit(struct exchnd_trigger *);

/* External triggers */
int exchnd_on_demand(struct exchnd_on_demand *);
int exchnd_external(pid_t,
		enum exchnd_modules (*)[EHM_LAST_ELEMENT],
		char[256]);

/* Arch specific modules functions */
int exchnd_kbacktrace(struct pt_regs *, struct task_struct *, unsigned char *);
unsigned long exchnd_get_fault(struct task_struct *);
int exchnd_dump_regs(unsigned char *, struct pt_regs *, struct task_struct *);
unsigned long *exchnd_get_sp(struct task_struct *);

/* Modules accessors */
extern int disabled_modules;
extern const char *exchnd_mod_names[];
void exchnd_set_trace_pid(int);
void exchnd_set_hist_size(int);
unsigned int get_syscall_index(void);
unsigned int get_tswitch_index(void);

/* Watchdog */
#define EXCHND_WD_MSDELAY 5000
struct exchnd_wd_worker {
	/* work_struct on top so we don't need container_of */
	struct delayed_work dwork;
	/* We'll retrieve task from pid */
	pid_t pid;
	int wakeup_count;
};
#define EXCHND_WD_KICKED 2

/* Init/Deinit watchdog */
int exchnd_init_wd(void);
void exchnd_deinit_wd(void);
/* Add new pid to watchout */
struct exchnd_wd_worker *exchnd_watch_pid(pid_t pid);
void exchnd_unwatch(struct exchnd_wd_worker *);
#endif
