/*
 * linux/drivers/char/exchnd/exchnd_device_buffer.c
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Matthias Weise (mweise@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 */

/*
 * Buffer of the exception device
 *
 * This file contains the implementation of the buffer that is provided by
 * the exception handler device and contains the exception messages to be
 * handled by the exception handler daemon. The buffer is organized as
 * ring buffer which blocks writing if it is full.
 */
#define pr_fmt(fmt) "exchnd: " fmt

#include <linux/exchnd.h>
#include "linux/device.h"
#include <linux/platform_device.h>
#include <linux/sched.h>
#include <asm-generic/uaccess.h>

#include "exchnd_internal.h"

#define EXCHND_HEADER_MAGIC 0xDEADBEEF

#if (CONFIG_EXCHND_SIZE  & (CONFIG_EXCHND_SIZE - 1)) != 0
	#error Ring buffer size must be a power of 2
#endif
#define OFFSET(x) (((unsigned long) (CONFIG_EXCHND_SIZE - 1)) & x)

static struct {
	unsigned char *buffer;	/* Ring buffer containing the exception data
				   for the user land */
	unsigned int size;	/* Size of the ring buffer (retrieved by Kernel
				   config) */
	unsigned long rd_idx;	/* Continuous write index for ring buffer */
	unsigned long wr_idx;	/* Continuous read index for ring buffer */
	spinlock_t write_lock;
				/* Spinlock locks ring buffer against parallel
				   writes */
	wait_queue_head_t read_wait_queue;
				/* Queue for blocking reads to the device
				   without data */
	wait_queue_head_t write_wait_queue;
				/* Queue for blocking write if ring buffer is
				   full */
} rb = { .write_lock = __SPIN_LOCK_UNLOCKED(rb.write_lock) };

/**
 * rb_init - initializes the ring buffer
 * @pdev Platform device use to bind allocated memory to
 *
 * This function initializes the ring buffer. It reads the buffer size from
 * configuration and allocates the memory for the buffer.
 *
 * Return: 0 if successful, error code otherwise
 */
int rb_init(struct platform_device *pdev)
{
	rb.size = CONFIG_EXCHND_SIZE;
	rb.buffer = devm_kzalloc(&pdev->dev, rb.size, GFP_KERNEL);

	if (exchnd_get_debug() & EXCHND_DEBUG_RB)
		pr_info("RB size=0x%x\n", rb.size);

	if (!rb.buffer) {
		pr_err("no memory\n");
		return -ENOMEM;
	}

	init_waitqueue_head(&rb.read_wait_queue);
	init_waitqueue_head(&rb.write_wait_queue);

	return 0;
}

/**
 * rb_empty - checks if buffer is empty
 *
 * This function checks if the ring buffer is empty
 *
 * Return: true if buffer is empty, false otherwise
 */
bool rb_empty(void)
{
	return rb.rd_idx == rb.wr_idx;
}

/**
 * rb_read_recover - Find next valid header to realign read index
 *
 * This function search for next magic number in the ring buffer and validates
 * it. Any data skipped this way is printed out as ASCII.
 * rb_read and rb_read_recover are not expected to be called in parallel, which
 * explains why there is no lock around rd_idx.
 * There are 2 possibility or the recovery call:
 *  - By the driver once the daemon is not there anymore (so there is no reader)
 *  - By the daemon before reading
 */
void rb_read_recover(void)
{
	const int header_size = sizeof(struct exchnd_message_header);
	unsigned long start = 0;
	unsigned int rem_sz = 0;
	start = rb.rd_idx;
	while (rb.rd_idx < rb.wr_idx) {
		struct exchnd_message_header *header;
		struct exchnd_message_header local;
		rem_sz = rb.size - OFFSET(rb.rd_idx);
		if (header_size > rem_sz) {
			memcpy(&local, rb.buffer + OFFSET(rb.rd_idx), rem_sz);
			memcpy(((unsigned char *) &local) + rem_sz, rb.buffer,
					header_size - rem_sz);
			header = &local;
		} else {
			header = (struct exchnd_message_header *)
					(rb.buffer + OFFSET(rb.rd_idx));
		}

		if ((header->magic == EXCHND_HEADER_MAGIC) &&
				(header->type < EHM_LAST_ELEMENT) &&
				(header->type >= EHM_NONE) &&
				(header->trigger < ET_LAST_ELEMENT) &&
				(header->trigger > ET_NONE) &&
				(header->length <= 5*EXCHND_MAX_TRACE_LEN))
			break;

		rb.rd_idx++;
	}
	/* Now dumping what we missed */
	if (start != rb.rd_idx) {
		unsigned long total = rb.rd_idx - start;
		rem_sz = rb.size - OFFSET(start);
		if (total > rem_sz) {
			print_hex_dump(KERN_ERR,
					"EXH recovery:",
					DUMP_PREFIX_OFFSET,
					16,
					4,
					rb.buffer + OFFSET(start),
					rem_sz,
					true);
			print_hex_dump(KERN_ERR,
					"EXH recovery:",
					DUMP_PREFIX_OFFSET,
					16,
					4,
					rb.buffer,
					total - rem_sz,
					true);
		} else {
			print_hex_dump(KERN_ERR,
					"EXH recovery:",
					DUMP_PREFIX_OFFSET,
					16,
					4,
					rb.buffer + OFFSET(start),
					total,
					true);
		}
	}
}

/**
 * rb_read - write exception message to ring buffer
 * @data: User space buffer the data shall be copied to
 * @size: Number of bytes to read and copy to user space
 *
 * This function reads from the ring buffer and copies it to the given user
 * space buffer. If the buffer does not contain enough bytes to read it
 * return EAGAIN.
 *
 * Return: Number of bytes read, or EAGAIN if there was not enough data
 */
int rb_read(unsigned char *data, unsigned int size)
{
	unsigned int rem_sz = 0;

	/* check if reading would block */
	if (rb.rd_idx > rb.wr_idx)
		return -EAGAIN;

	/* Check if data is already overwritten */
	if (rb.wr_idx - rb.rd_idx > rb.size) {
		/* This should never happen. Overwriting are
		 * normally blocked ! */
		pr_err("Ring buffer overwritten !\n");
		return -EIO;
	}
	/* Check if there is enough data for requested size */
	if (rb.wr_idx - rb.rd_idx < size)
		return -EAGAIN;

	rem_sz = rb.size - OFFSET(rb.rd_idx);
	/* Copy data up to end of ring buffer to user space */
	if (copy_to_user((void __user *) data,
				rb.buffer + OFFSET(rb.rd_idx),
				min(size, rem_sz)))
		return -EFAULT;

	/* Copy data from start of ring buffer to user space */
	if (size > rb.size - OFFSET(rb.rd_idx))
		if (copy_to_user((void __user *) data + rem_sz,
					rb.buffer, size - rem_sz))
			return -EFAULT;

	rb.rd_idx += size;

	if (exchnd_get_debug() & EXCHND_DEBUG_RB)
		pr_info("Read index is now at %lu (<%p>+%lu).\n",
				rb.rd_idx,
				rb.buffer,
				OFFSET(rb.rd_idx));

	return size;
}

/**
 * rb_write - write exception message to ring buffer
 * @header: header of the exception contains metadata such a length, etc.
 * @data: the data of the exception message
 *
 * Writes one exception message to the ring buffer. The function expects that
 * the message header is correctly filled. The write function block in case the
 * ring buffer is full. It waits until enough buffer is read/emptied to write
 * the current message
 */
void rb_write(struct exchnd_message_header *header, unsigned char *data)
{
	unsigned long flags;
	unsigned long tmp_idx;

	/* Lock write access */
	spin_lock_irqsave(&rb.write_lock, flags);
	header->magic = EXCHND_HEADER_MAGIC;
	/* Force the following values to 0 to avoid mistake during recovery */
	header->reserved3 = 0;
	header->reserved4 = 0;

	if (sizeof(struct exchnd_message_header) + header->length >
	  rb.size - (rb.wr_idx - rb.rd_idx)) {
		/* Do not wait in spinlock */
		spin_unlock_irqrestore(&rb.write_lock, flags);
		pr_err("Ring buffer full!\n");
		/* Message does not fit into ring buffer -> wait */
		wait_event_interruptible(rb.write_wait_queue,
		  sizeof(struct exchnd_message_header) + header->length <=
		  rb.size - (rb.wr_idx - rb.rd_idx));
		/* Now write access has to locked again */
		spin_lock_irqsave(&rb.write_lock, flags);
	}

	tmp_idx = rb.wr_idx;
	/* Write header */
	if (sizeof(struct exchnd_message_header) <= rb.size - OFFSET(tmp_idx)) {
		/* No wrap around necessary */
		memcpy(rb.buffer + OFFSET(tmp_idx), header,
		  sizeof(struct exchnd_message_header));

		if (exchnd_get_debug() & EXCHND_DEBUG_RB)
			print_hex_dump_bytes("exchnd: to RB: ",
					DUMP_PREFIX_OFFSET,
					rb.buffer + OFFSET(tmp_idx),
					sizeof(struct exchnd_message_header));
	} else {
		/* Wrap around necessary */
		unsigned int rem_sz = rb.size - OFFSET(tmp_idx);
		memcpy(rb.buffer + OFFSET(tmp_idx), header, rem_sz);

		if (exchnd_get_debug() & EXCHND_DEBUG_RB)
			print_hex_dump_bytes("exchnd: to RB: ",
					DUMP_PREFIX_OFFSET,
					rb.buffer + OFFSET(tmp_idx),
					sizeof(struct exchnd_message_header));

		memcpy(rb.buffer, ((unsigned char *) header) + rem_sz,
		  sizeof(struct exchnd_message_header) - rem_sz);

		if (exchnd_get_debug() & EXCHND_DEBUG_RB)
			print_hex_dump_bytes("exchnd: to RB: ",
					DUMP_PREFIX_OFFSET,
					rb.buffer,
					sizeof(struct exchnd_message_header) -
						rem_sz);

	}
	tmp_idx += sizeof(struct exchnd_message_header);
	if ((header->length != 0) && (header->flags.collected == 1)) {
		/* Write data */
		if (header->length <= rb.size - OFFSET(tmp_idx)) {
			/* No wrap around necessary */
			memcpy(rb.buffer + OFFSET(tmp_idx),
			  data, header->length);

		if (exchnd_get_debug() & EXCHND_DEBUG_RB)
			print_hex_dump_bytes("exchnd: to RB: ",
					DUMP_PREFIX_OFFSET,
					rb.buffer + OFFSET(tmp_idx),
					header->length);

		} else {
			/* Wrap around necessary */
			unsigned int rem_sz = rb.size - OFFSET(tmp_idx);
			memcpy(rb.buffer + OFFSET(tmp_idx), data, rem_sz);

			if (exchnd_get_debug() & EXCHND_DEBUG_RB)
				print_hex_dump_bytes("exchnd: to RB: ",
						DUMP_PREFIX_OFFSET,
						rb.buffer + OFFSET(tmp_idx),
						rem_sz);

			memcpy(rb.buffer, data + rem_sz,
			  header->length - rem_sz);

			if (exchnd_get_debug() & EXCHND_DEBUG_RB)
				print_hex_dump_bytes("exchnd: to RB: ",
						DUMP_PREFIX_OFFSET,
						rb.buffer,
						header->length - rem_sz);

		}
		tmp_idx += header->length;
	}
	rb.wr_idx = tmp_idx;

	if (exchnd_get_debug() & EXCHND_DEBUG_RB)
		pr_info("Write index is now at %lu (<%p>+%lu).\n",
				rb.wr_idx,
				rb.buffer,
				OFFSET(rb.wr_idx));

	/* Unlock write access */
	spin_unlock_irqrestore(&rb.write_lock, flags);
	/* Wake up poll requests */
	wake_up(&rb.read_wait_queue);
}

/*
 * rb_wakeup_write_wait_queue - wake up the waiting writes
 *
 * Sends a wake up signal to all threads waiting to write to the ring buffer
 */
void rb_wakeup_write_wait_queue(void)
{
	wake_up(&rb.write_wait_queue);
}

/*
 * rb_wait_read_wait_queue - wait for data in ring buffer
 *
 * Sleeps until data is written to ring buffer. It waits for a signal from
 * the wait queue and checks if data is in the buffer.
 *
 * Return: 0 if waited successfully, error code otherwise
 */
int rb_wait_read_wait_queue(void)
{
	return wait_event_interruptible(rb.read_wait_queue, !rb_empty());
}

/*
 * rb_get_read_wait_queue - retrieve the read wait queue handle
 *
 * Returns internal read queue handle
 *
 * Return: Pointer to wait queue handle
 */
wait_queue_head_t *rb_get_read_wait_queue(void)
{
	return &rb.read_wait_queue;
}
