/*
 * (c) 2013 Advanced Driver Information Technology GmbH
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * llrb_data - data to store in a lock-less ring buffer
 *
 * Data in the ring buffer is either binary or ASCII. The maximum data
 * size is LLRB_DATA_SIZE.
 *
 * By convention, binary data starts with \0. Binary data is stored as
 * is, without adding or removing any bytes, one entry per chunk of
 * data. When the maximum data size is reached, remaining data is put
 * into another entry.
 *
 * ASCII data is split into seperate entries at newlines, either at
 * the first newline, or at the last newline fitting into a ring buffer
 * slot if LLRB_DATA_ASCII_AGGREGATE == 1.
 *
 * Entries not ending with a newline are merged with successive
 * entries. When the maximum data size is reached without a newline,
 * an entry without a newline is stored and remaining data put into
 * another entry. Entries are \0 terminated.
 */

#ifndef _LLRB_DATA_H
#define _LLRB_DATA_H

#include <linux/types.h>
#include <linux/string.h>

/* maximum console message level */
#define CONSOLEMAXLEVEL            10
/* maximum size of data in the ring buffer */
#define LLRB_DATA_SIZE             256

/* kind of data to be stored */
enum llrb_data_kind {
	LLRB_DATA_UNKNOWN = ~0,
	LLRB_DATA_BINARY = 0,
	LLRB_DATA_ASCII
};

/* iterator for external data */
struct llrb_iterator {
	unsigned char *data;   /* data */
	unsigned len;          /* length of data */
	unsigned pos;          /* position of data processing */
	unsigned complete;     /* current message is complete */
	enum llrb_data_kind kind; /* LLRB_DATA_UNKNOWN or kind of data */
	int userspace;         /* if set, copy from/to user space */
};

/* llrb_iterator_init - initialize an iterator */
static inline struct llrb_iterator *llrb_iterator_init(
	struct llrb_iterator *it,   /* the iterator */
	unsigned char *data,   /* data */
	unsigned len,          /* length of data */
	enum llrb_data_kind kind, /* LLRB_DATA_UNKNOWN or kind of data */
	int userspace)
{
	it->data = data;
	it->len = len ? len : strlen((char *) data);
	it->pos = 0;
	it->complete = 0;
	it->kind = kind;
	it->userspace = userspace;

	return it;
}

/* llrb_iterator_complete - indicate if all data has been copied */
static inline int llrb_iterator_complete(struct llrb_iterator *it)
{
	return !it || it->len == it->pos;
}

/* llrb_iterator_set_msg_complete - indicates a message is completed */
static inline void llrb_iterator_set_msg_complete(struct llrb_iterator *it)
{
	if (it)
		it->complete = 1;
}

/* llrb_iterator_msg_complete - indicates a message is not completed */
static inline void llrb_iterator_set_msg_incomplete(struct llrb_iterator *it)
{
	if (it)
		it->complete = 0;
}

/* llrb_iterator_msg_complete - return if a message is completed or not */
/* In case iterator is null the message is to set complete */
static inline int llrb_iterator_msg_complete(struct llrb_iterator *it)
{
	return !it || it->complete;
}

/* llrb_iterator_get_position - give the corrected iterator position field */
static inline unsigned llrb_iterator_get_length(struct llrb_iterator *it)
{
	return it ? it->len : 0;
}

/* llrb_iterator_get_kind - give the iterator kind field */
static inline enum llrb_data_kind llrb_iterator_get_kind(
	struct llrb_iterator *it)
{
	return it ? it->kind : LLRB_DATA_UNKNOWN;
}

/* data to store in the ring buffer */
struct llrb_data {
	enum llrb_data_kind kind; /* kind of data */
	unsigned len;          /* length of data in buffer */
	char buffer[LLRB_DATA_SIZE];
};

/* llrb_data_init - initialize empty ring buffer data */
struct llrb_data *llrb_data_init(struct llrb_data *);

/* llrb_data_get - copy ring buffer data to an outside buffer */
/* returns the amount copied or ~0 if an error occurs */
unsigned llrb_data_get(
	struct llrb_data *src,     /* the location to copy data from */
	struct llrb_iterator *dst); /* the location to copy data to */

/* llrb_data_put - copy ring buffer data from an outside buffer */
/* returns the amount copied or ~0 if an error occurs */
unsigned llrb_data_put(
	struct llrb_data *dst,      /* in: the location to copy data to */
	struct llrb_iterator *src); /* in: the location to copy data from */

#endif /* ifndef _LLRB_DATA_H */
