/*
 * (c) 2013 Advanced Driver Information Technology GmbH
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/uaccess.h>

#include "llrb_data.h"

#ifndef min
#define min(a, b) ((a) <= (b) ? (a) : (b))
#endif /* ifndef min */

#ifndef max
#define max(a, b) ((a) >= (b) ? (a) : (b))
#endif /* ifndef max */

/* llrb_data_init - initialize ring buffer data */
struct llrb_data *llrb_data_init(struct llrb_data *data)
{
	data->kind = LLRB_DATA_UNKNOWN;
	data->len = 0;
	data->buffer[0] = 0;

	return data;
}

/* llrb_data_get - copy ring buffer data to an outside buffer */
/* returns the amount copied or ~0 if an error occurs */
unsigned llrb_data_get(
	struct llrb_data *src,     /* the location to copy data from */
	struct llrb_iterator *dst) /* the location to copy data to */
{
	unsigned len = 0;
	if (!src || !dst)
		return ~len;
	/* Ensure that src->len does not exceed the slot size */
	if (src->len > sizeof(src->buffer))
		src->len = sizeof(src->buffer);
	if (dst->len >= src->len)
		/* copy all data if there is enough space */
		len = src->len;
	else {
		/* copy as much data as possible and set message incomplete */
		len = dst->len;
	}
	if (len) {
		if (dst->userspace) {
			/* copy to userspace */
			if (copy_to_user((void __user *)
					 dst->data,
					 src->buffer, len))
				return ~0;
		} else
			/* inside the Linux kernel */
			memcpy(dst->data, src->buffer, len);

		/* Check in case of ASCII data whether the data is correctly
		 * terminated. If the slot became valid during validation
		 * the data might not be newline and null byte terminated.
		 */
		if (src->kind == LLRB_DATA_ASCII) {
			/* The message is completed - always add null byte */
			if ((len > 1) && (dst->data[len-2] == '\n'))
				dst->data[len-1] = '\0';
			else {
				/* guarantee newline and nullbyte */

				/* len-2 is not newline, len-1 is null byte
				 * ==> null byte will be overwritten with
				 *     newline and added behind. We need an
				 *     additional byte in the destination
				 *     buffer (len++). */
				if (!dst->data[len-1])
					len++;
				/* newline and null byte have to be appended
				 * ==> we need two additional bytes in the
				 *     destination buffer (len+=2) */
				else if (dst->data[len-1] != '\n')
					len += 2;
				/* newline is the last byte in destination
				 * ==> we need an additional byte in the
				 *     buffer to add null byte (len++) */
				else
					len++;
				/* resulting len exceeds the size of destination
				 * buffer
				 * ==> limit message to dst->len */
				if (len > dst->len)
					len = dst->len;
				/* write newline and null byte on calculated
				 * positions */
				dst->data[len-2] = '\n';
				dst->data[len-1] = '\0';
			}
		}
	}
	dst->kind = src->kind;
	dst->pos  = len;
	dst->len = len;
	return len;
}

/* llrb_data_put - copies the data of the current data stream into the slots
 * of the ring buffer.
 * One write request of binary data always represents one binary message. This
 * data is copied one by one into one slot. A binary message has at
 * a maximum a length of 256 bytes.
 * ===========================================================================
 * An ASCII message is always seen as terminated with the appearance of a
 * newline ("\n") character. A null byte ("\0") character does not finish
 * an ASCII message.
 * An incoming data stream has to be parsed for newline characters.
 * An incoming data stream can have different characteristics.
 * - it can contain a single message  ("data\n")
 * - it can contain serveral terminated messages ("data1\ndata2\ndatax\n")
 * - it can contain terminated and not terminated messages ("data1\ndatax")
 * - it can contain not terminated data ("data")
 * A message can have a maximum length of 256 bytes including ("\n\0").
 * If no newline character appears in the data stream, the message is not be
 * seen as completed. The driver expects more data which will be appended to
 * this already stored data.
 * ASCII messages will be splitted or concatenated.
 * Messages are created from incoming data streams using these principles:
 * Message size is restricted to 256 bytes. If it exceeds it will be cut-off.
 * Remaining part will be transferred into a next new message.
 * If necessary this will be repeated until the data is finally terminated.
 * ===========================================================================
 * Content of incoming data streams ==> Resulting message(s)
 * ===========================================================================
 * Split data from one data stream into independent messages:
 * DATA1\n DATA2\n DATA3\n          ==> Message: DATA1\n\0 (max. 256 bytes)
 *                                  ==> Message: DATA2\n\0 (max. 256 bytes)
 *                                  ==> Message: DATA3\n\0 (max. 256 bytes)
 * ===========================================================================
 * Concatenate data from several data streams into one message:
 * DATA1 incoming not terminated data, will be written to slot wr.
 * DATA2 incoming not terminated data, will be concatenated to DATA1 in slot wr
 * DATA3 incoming not terminated data, will be concatenated to DATA2 in slot wr
 * \n    termination data comes in ==> Message: DATA1DATA2DATA3\n\0
 * ===========================================================================
 * Null bytes in data streams will be discarded
 * DATA1\0    (incoming not terminated data - only null byte)
 * DATA2\0    (incoming not terminated data - only null byte)
 * DATA3\0    (incoming not terminated data - only null byte)
 * \n         (termination data comes in) ==> Message: DATA1DATA2DATA3\n\0
 * ===========================================================================
 * In case a message exceeds 256 bytes during concatenation it will be
 * splitted in seperated messages each terminated by \n\0
 * ===========================================================================
 * This function checks all above mentioned principles and returns the amount
 * of data copied from source (incoming data stream to the current slot (RAM)
 */
unsigned llrb_data_put(
	struct llrb_data *dst,     /* in: the location to copy data to */
	struct llrb_iterator *src) /* in: the location to copy data from */
{
	if (src && dst) {
		/* len is the minimum between remaining data to copy from data
		 * stream or remaining space in current slot (dst->buffer) */
		int len = (int)min(src->len - src->pos,
				   sizeof(dst->buffer)-dst->len);

		if (len < 0)
			return ~0;
		if (src->userspace) {
			/* inside the Linux kernel, copy from userspace */
			if (copy_from_user(dst->buffer + dst->len,
					   (void __user *)
					   (src->data + src->pos), len))
				return ~0;
		} else
			memcpy(dst->buffer + dst->len,
			       src->data + src->pos, len);

		/* Only a slot containing ASCII data can have already a length
		 * here. This prevents from wrong determination of data type.
		 * If, for what reason ever, dst->buffer[0] contains a null byte
		 * data type will become binary instead of ASCII. That only can
		 * happens in case llrb_data_put is called repeatedly for
		 * processing the same write request.
		 */
		if (dst->len)
			dst->kind = LLRB_DATA_ASCII;

		/* if kind of data is yet not known */
		if (dst->kind == LLRB_DATA_UNKNOWN) {
			if (src->kind != LLRB_DATA_UNKNOWN)
				dst->kind = src->kind;
			else
				dst->kind = dst->buffer[0] ?
					LLRB_DATA_ASCII : LLRB_DATA_BINARY;
		}
		if (dst->kind == LLRB_DATA_ASCII) {
			char     *newline = NULL;
			unsigned cb       = 0;
			/* Only in case the message is terminated the null
			 * byte is considered by length value (dst->len).
			 * len always stores the number of bytes copied for
			 * current processing (execution of this function)
			 * from the src buffer into destination buffer.
			 * len determines the new start position in src->data
			 * when copying remaing data from src->data to dst.
			 * len does not necessarily correspond one to one
			 * with the new dst->len value!
			 * Now search for newline from the beginning.
			 * Keep in mind that the data in dst->buffer,
			 * starting at position dst->buffer[dst->len] up to
			 * dst->buffer[dst->len+len-1] represents the identical
			 * data contained in src->data[src->pos+len-1]. So if
			 * we look at dst->buffer we look at the same data in
			 * src->data. */
			newline = memchr(dst->buffer + dst->len, '\n', len);
			if (newline) {
				/* set input position behind newline (+1) */
				len = newline-(dst->buffer + dst->len) + 1;
				/* cb is required length with null byte */
				cb = dst->len + len + 1;
				/* dst->buffer contains data from src->data.
				 * If data from dst->buffer is accessed, this
				 * data represents the data from src-> data! */
				if (cb <= sizeof(dst->buffer)) {
					/* Size of destination buffer is
					 * sufficient. dst->len includes
					 * necessary space for null byte. */
					dst->len = cb;
					if (dst->buffer[cb-1]) {
						/* Write null byte if not
						 * already at this position in
						 * destination buffer.
						 * len points to the next byte
						 * following the copied newline
						 * byte. len keeps its value for
						 * next iteration */
						dst->buffer[cb-1] = '\0';
					} else if (src->len >= cb)
						/* len will be moved over the
						 * null byte copied from src,
						 * if this null byte is really
						 * copied from src->data and not
						 * randomly in dst->buffer at
						 * this position. */
						len++;
				} else {
					/* Only to be save. cb should and
					 * can never exceed the value
					 * sizeof(dst->buffer) + 1 */
					if (cb > sizeof(dst->buffer)+1)
						cb = sizeof(dst->buffer)+1;
					/* Adding null byte will exceed the size
					 * of dest->buffer by one.
					 * Newline is the last byte fitting into
					 * dst->buffer. Therefore we here
					 * have to skip the byte directly
					 * preceeding the current newline and
					 * copy this byte into the next slot as
					 * an independent message. ...happens.
					 */
					dst->buffer[cb-3] = '\n'; /*skip byte */
					dst->buffer[cb-2] = '\0';
					dst->len = cb-1;
					/* move len to skipped byte */
					len -= 2;
				}
				llrb_iterator_set_msg_complete(src);
			} else {
				/* Eat up multiple null bytes from the end of
				 * destination.
				 */
				while ((len > 0) &&
				       (!dst->buffer[dst->len+len-1]))
					len--;
				cb = dst->len + len;
				if (cb == sizeof(dst->buffer)-2) {
					/* Message will be completed.
					 * Remember we don't have a newline
					 * and we have to complete. We need
					 * two bytes left in the buffer */
					dst->len = sizeof(dst->buffer);
				} else if (cb == sizeof(dst->buffer)-1) {
					/* The second to last byte is going to
					 * become a newline and the last byte
					 * is going to become the null byte.
					 * Therefore the copied length has to
					 * be reduced by one to not lose this
					 * byte which will be overwritten with
					 * the newline */
					dst->len = sizeof(dst->buffer);
					len--;
				} else if (cb == sizeof(dst->buffer)) {
					/* Here we have to compensate 2 bytes */
					dst->len = sizeof(dst->buffer);
					len -= 2;
				} else {
					dst->len = cb;
				}
				if (dst->len == sizeof(dst->buffer)) {
					dst->buffer[dst->len-2] = '\n';
					dst->buffer[dst->len-1] = '\0';
					llrb_iterator_set_msg_complete(src);
				}
			}
		} else {
			/* binary data */
			llrb_iterator_set_msg_complete(src);
			dst->len = len;
		}
		/* len can become negative in case the creation of a
		 * message needs to overwrite bytes in the destination buffer
		 */
		src->pos += len;
		/* Just to be sure that src->pos never overtakes src->len */
		if (src->pos > src->len)
			src->pos = src->len;

		/* Remove null bytes starting with byte src->pos + 1 that is
		 * at index src->pos from the remaining data. This will also
		 * delete the null bytes ate up from the end of the destination
		 * buffer above because len was reduced accordingly and
		 * therefore src->pos points to the last byte of payload in
		 * the destination buffer. */
		while ((dst->kind == LLRB_DATA_ASCII) &&
		       (src->pos < src->len) &&
		       (src->data[src->pos] == '\0'))
			src->pos++;

		if (len < 0)
			len = 0;
		return len;
	} else if (!src) {
		/* finish this slot */
		if (dst->kind == LLRB_DATA_ASCII) {
			/* The slot is already completed. Not expected here */
			if ((dst->len > 1) &&
			    (dst->buffer[dst->len-2] == '\n')) {
				dst->buffer[dst->len-1] = '\0';
				return dst->len;
			}
			/* for what reason ever the message length is 0 */
			if (!dst->len)
				dst->len += 2;
			/* newline and null byte have to be appended
			 * but null byte will be overwritten with newline
			 */
			else if (!dst->buffer[dst->len-1])
				dst->len++;
			/* newline and null byte have to be appended */
			else if (dst->buffer[dst->len-1] != '\n')
				dst->len += 2;
			else
				dst->len++;
			if (dst->len > sizeof(dst->buffer))
				dst->len = sizeof(dst->buffer);
			dst->buffer[dst->len - 2] = '\n';
			dst->buffer[dst->len - 1] = '\0';
		}
		return 0;
	}
	return ~0;
}
