/*
 * (c) 2013 Advanced Driver Information Technology GmbH
 *          Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/crc16.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>

#include "llrb.h"

/* atomic states of slots in the ring buffer */
#define LLRB_FREE                0
#define LLRB_WRITING             1
#define LLRB_WRITTEN             2
#define LLRB_READING             3

/* initial maximum threshold calculated as size devided by this (25%) */
#define INITIAL_THRES            4

/* access slots */
#define slot(llrb, seqnum)			\
	((llrb)->slots[(seqnum)%(llrb)->size])

/* access slot state */
#define state_init(llrb, seqnum, value)					\
	(atomic_set(&slot(llrb, seqnum).state, value))

#define state_get(llrb, seqnum)				\
	(atomic_read(&slot(llrb, seqnum).state))

#define state_transition(llrb, seqnum, old, new)			\
	(atomic_cmpxchg(&slot(llrb, seqnum).state, (old), (new)) == (old))

/* variable used to store  RWT (real_world_time) in nanoseconds*/
static unsigned long long real_world_time;

/* access llrb flags */
static inline int flags_test(struct llrb *llrb, unsigned mask)
{
	return
		((mask & LLRB_SLOT_INITIALIZED) &&
		 test_bit(LLRB_SLOT_INITIALIZED_, &llrb->flags)) ||

		((mask & LLRB_SLOT_RESTART) &&
		 test_bit(LLRB_SLOT_RESTART_, &llrb->flags)) ||

		((mask & LLRB_SLOT_DROPPED) &&
		 test_bit(LLRB_SLOT_DROPPED_, &llrb->flags));
}

static inline unsigned flags_get(struct llrb *llrb)
{
	return
		(test_bit(LLRB_SLOT_INITIALIZED_, &llrb->flags) ?
		 LLRB_SLOT_INITIALIZED : 0) |

		(test_bit(LLRB_SLOT_RESTART_, &llrb->flags) ?
		 LLRB_SLOT_RESTART : 0) |

		(test_bit(LLRB_SLOT_DROPPED_, &llrb->flags) ?
		 LLRB_SLOT_DROPPED : 0);
}

static inline void flags_set(struct llrb *llrb, unsigned mask)
{
	if (mask & LLRB_SLOT_INITIALIZED)
		test_and_set_bit(LLRB_SLOT_INITIALIZED_, &llrb->flags);

	if (mask & LLRB_SLOT_RESTART)
		test_and_set_bit(LLRB_SLOT_RESTART_, &llrb->flags);

	if (mask & LLRB_SLOT_DROPPED)
		test_and_set_bit(LLRB_SLOT_DROPPED_, &llrb->flags);
}

static inline void flags_clear(struct llrb *llrb, unsigned mask)
{
	if (mask & LLRB_SLOT_INITIALIZED)
		test_and_clear_bit(LLRB_SLOT_INITIALIZED_, &llrb->flags);

	if (mask & LLRB_SLOT_RESTART)
		test_and_clear_bit(LLRB_SLOT_RESTART_, &llrb->flags);

	if (mask & LLRB_SLOT_DROPPED)
		test_and_clear_bit(LLRB_SLOT_DROPPED_, &llrb->flags);
}

/* llrb_validate_slot - validate the state of a ring buffer slot */
static void llrb_validate_slot(struct llrb *llrb, unsigned n)
{
	if (slot(llrb, n).seqnum < atomic_read(&llrb->wr) &&
	    !(slot(llrb, n).flags & ~LLRB_SLOT_ALL_FLAGS)) {
		switch (state_get(llrb, n)) {
		case LLRB_FREE:
			break;
		case LLRB_WRITING:
			/* slot is incomplete */
			llrb->slots[n].flags |= LLRB_SLOT_INCOMPLETE;
			(void) state_transition(llrb, n,
						LLRB_WRITING,
						LLRB_WRITTEN);
			break;
		case LLRB_READING:
			/* slot has not been read yet */
			llrb->slots[n].flags &= ~LLRB_SLOT_READ;
			(void) state_transition(llrb, n,
						LLRB_READING,
						LLRB_WRITTEN);
			break;
		case LLRB_WRITTEN:
			/* slot has not been read yet */
			llrb->slots[n].flags &= ~LLRB_SLOT_READ;
			break;
		default:
			/* something seems to be wrong here
			 * unknown state, re-initialize */
			goto init_slot;
		}
	} else
		goto init_slot;

	return;

init_slot:
	state_init(llrb, n, LLRB_FREE);
	llrb->slots[n].seqnum = ~0;
	llrb->slots[n].flags = LLRB_SLOT_OK;

	return;
}

/* llrb_init_mem - initialize the ring buffer in a piece of memory */
struct llrb *llrb_init_mem(void *mem, unsigned amount, int reuse)
{
	/* calculate the size of the ring buffer from the */
	/* aligned size of the ring buffer slots fitting */
	/* into the given piece of memory */
	struct llrb *llrb = (struct llrb *) mem;

	/* size of one slot including alignment/padding */
	int single = sizeof(((struct llrb *) NULL)->slots)/LLRB_SIZE;
	int total = sizeof(struct llrb);
	/* number slots fitting into the actual memory*/

	int slots = (int) LLRB_SIZE + ((int) amount - total)/single;

	if (amount < total && (total - amount)%single)
		/* compensate rounding errors when there is little memory */
		slots--;

	if (reuse) {
		/* look for the magic numbers to find the end of the */
		/* ring buffer in memory */
		if (llrb->magic_start == LLRB_MAGIC_START &&
		    llrb->size <= slots &&
		    *((unsigned *) (llrb->slots + llrb->size)) ==
		    LLRB_MAGIC_END)
			slots = llrb->size;
	}

	if (slots > 0 && reuse) {
		/* there might be a valid ring buffer in memory */
		/* check magic numbers and globals */
		if (llrb->magic_start == LLRB_MAGIC_START &&
		    llrb->size == slots &&
		    atomic_read(&llrb->rd) &&  /* must not be 0 */
		    ~atomic_read(&llrb->rd) && /* must not be 0xffffff */
		    atomic_read(&llrb->rd) <= atomic_read(&llrb->wr) &&
		    atomic_read(&llrb->wr)-atomic_read(&llrb->rd) <= slots &&
		    atomic_read(&llrb->free) < atomic_read(&llrb->wr) &&
		    llrb->low <= slots &&
		    llrb->high <= slots &&
		    !flags_test(llrb, ~LLRB_SLOT_ALL_FLAGS) &&
		    *((unsigned *) (llrb->slots + slots)) == LLRB_MAGIC_END) {
			/* check all slots */
			unsigned i = 0;
			for (; i < (unsigned) slots; i++)
				llrb_validate_slot(llrb, i);

			/* the ring buffer appears to be intact */
			flags_set(llrb, LLRB_SLOT_RESTART);
			return llrb;
		}
	}
	/* initialize  real_world_time to Zero */
	real_world_time = 0;

	/* initialize ring buffer */
	return slots > 0 && reuse >= 0 ? llrb_init(llrb, slots) : NULL;
}

/* llrb_init - initialize the ring buffer */
struct llrb *llrb_init(struct llrb *llrb, unsigned size)
{
	unsigned i;

	/* initialize magic numbers and globals */
	llrb->magic_start = LLRB_MAGIC_START;
	llrb->size = size;

	/* ring buffer indexes: free < rd <= wr */
	atomic_set(&llrb->rd, 1);   /* next index to read */
	atomic_set(&llrb->wr, 1);   /* next index to write */
	atomic_set(&llrb->free, 0); /* last index free'd */

	llrb->low = 0;
	llrb->high = size / INITIAL_THRES;
	llrb->flags = LLRB_SLOT_INITIALIZED;

	*((unsigned *) (llrb->slots + size)) = LLRB_MAGIC_END;

	/* initialize all slots */
	for (i = 0; i < llrb->size; i++) {
		state_init(llrb, i, LLRB_FREE);
		llrb->slots[i].seqnum = ~0;
		llrb->slots[i].flags = LLRB_SLOT_OK;
	}

	return llrb;
}

/* llrb_calculate_crc - calculate a slot's crc */
static u16 llrb_calculate_crc(struct llrb *llrb, unsigned seqnum)
{
	/* crc over seqnum,time, and data (but not the whole struct) */
	u16 crc = crc16(0,
			(unsigned char *) &slot(llrb, seqnum).seqnum,
			(unsigned char *) &slot(llrb, seqnum).crc-
			(unsigned char *) &slot(llrb, seqnum).seqnum);

	return crc;
}

/* llrb_get_timestamp - gives time stamp of (local clock + RWT) */
static unsigned long long llrb_get_timestamp(void)
{
	return real_world_time + local_clock();
}

/* llrb_allocate - allocate a ring buffer slot */
static unsigned llrb_allocate(struct llrb *llrb)
{
	unsigned flags = LLRB_SLOT_OK;
	unsigned paranoid = llrb->size;

	/* avoid dead-locks */
	while (paranoid--) {
		unsigned rd = atomic_read(&llrb->rd);
		unsigned wr = atomic_read(&llrb->wr);

		/* check for free slots */
		if (wr - rd < llrb->size) {
			/* allocate a slot */
			if (atomic_cmpxchg(&llrb->wr, wr, wr+1) == wr &&
			    state_transition(llrb, wr,
					     LLRB_FREE, LLRB_WRITING)) {

				/* initialize */
				slot(llrb, wr).seqnum = wr;
				slot(llrb, wr).flags = flags;
				slot(llrb, wr).time = llrb_get_timestamp();

				if (flags_test(llrb, LLRB_SLOT_RESTART)) {
					/* deliver the restart flag */
					slot(llrb, wr).flags =
						flags | LLRB_SLOT_RESTART;
				}

				llrb_data_init(&slot(llrb, wr).data);
				return wr;
			}
		} else {
#if !LLRB_DROP_OLD
			/* note some data has been dropped and bail out */
			flags_set(llrb, LLRB_SLOT_DROPPED);
			return ~0;
#else
			/* try to discard an older, probably the oldest */
			/* slot that hasn't been acknowledged */
			if (atomic_cmpxchg(&llrb->rd, rd, rd+1) == rd &&
			    state_transition(llrb, rd,
					     LLRB_WRITTEN, LLRB_READING)) {
				/* restore global flags of dropped slot */
				flags |= slot(llrb, rd).flags &
					(LLRB_SLOT_RESTART |
					 LLRB_SLOT_INITIALIZED);

				/* success, mark the slot as unused */
				(void) state_transition(llrb, rd,
							LLRB_READING,
							LLRB_FREE);
				flags |= LLRB_SLOT_DROPPED;
			}
#endif /* ifdef LLRB_DROP_OLD */
		}
	}

	/* failure */
	flags_set(llrb, LLRB_SLOT_DROPPED);
	return ~0;
}

/* llrb_written - change slots from writing to written */
static void llrb_written(struct llrb *llrb, unsigned wr)
{
	if (~wr && wr) {
		/* mark the slot as used now */
		slot(llrb, wr).crc = llrb_calculate_crc(llrb, wr);
		(void) state_transition(llrb, wr,
					LLRB_WRITING,
					LLRB_WRITTEN);
	}
}

/* llrb_append - append data to the ring buffer according to specification.
 * ==> see documentation for function llrb_data_put in file llrb_data.c
 * For binary data a maximum of 256 bytes (slot message size) payload is
 * allowed. No concatenation or splitting is done at all.
 * In case of ASCII data:
 * Data in slot wr (parameter wr > 0) is not finished yet. Incoming data
 * shall be appended to the already stored data in slot wr.
 * If parameter wr equals 0, incoming data will be written into a new slot.
 * =========================================================================
 * Returns 0 if all data of this incoming data stream are completely stored
 * as one or more completed messages in one or more slots. Slots will not be
 * chained anymore. One slot one message and if necessary the creation of
 * a completed message will be forced if the length would exceed slot message.
 * size (256 bytes).
 * Returns wr > 0, as the current slot number, if data in this slot is not
 * yet completed. Next incoming data from a write request of this file handle
 * will be appended to the data stored in this slot (return value wr which
 * is going to be stored with the file handle initiating the current write
 * request). In the next write request wr will be passed to this function as
 * parameter indicating the slot currently in use by this file handle.
 * =========================================================================
 * Parameter: llrb : ring buffer to store messages in
 *            wr   : number of slot in use to handle this write request.
 *	             wr comes with the file handle which initiates the
 *                   write request.
 *	             wr > 0 ==> data will be appended to already stored data
 *                              in slot wr.
 *                   wr = 0 ==> data will be stored in a new slot
 *            it   : the incoming data stream
 * Return   : wr>0 : the slot currently in use for this file handle
 *            wr=0 : the data is completely stored as finished messages. All
 *                   slots occupied by data of this processed data stream
 *                   contain finished (terminated) messages which are ready
 *                   to be read.
 *            wr=~0: error
 * =========================================================================
 */
static unsigned llrb_append(
	struct llrb *llrb, unsigned wr,
	struct llrb_iterator *it)
{
	unsigned  stored;
	/* allocate a new slot if wr is 0 */
	while (wr || ~(wr = llrb_allocate(llrb))) {
		/* put data into the slot */
		stored = llrb_data_put(&slot(llrb, wr).data, it);
		if (!~stored)
			break;
		/* Check whether the message is complete to set slot to state
		 * LLRB_WRITTEN.
		 * In case of ASCII data iterator it can be NULL. This is
		 * when the corresponding file handle is released. The
		 * return value of llrb_data_put in such cases is 0. This
		 * causes the setting of flag LLRB_SLOT_INCOMPLETE.
		 */
		if (llrb_iterator_msg_complete(it)) {
			/* assign global flags to slot */
			slot(llrb, wr).flags |= flags_get(llrb);
			flags_clear(llrb, slot(llrb, wr).flags);
			if (!stored ||
			     ((llrb_iterator_get_kind(it) == LLRB_DATA_BINARY)
			      &&  stored != llrb_iterator_get_length(it)))
				slot(llrb, wr).flags |= LLRB_SLOT_INCOMPLETE;
			llrb_written(llrb, wr);
			wr = 0;
			/* for binary message max allowed size 256 bytes
			 * if size is more than 256 bytes discard message
			 * after 256 bytes
			*/
			if (llrb_iterator_get_kind(it)
					== LLRB_DATA_BINARY)
				break;
		}
		/* until everything is copied */
		if (llrb_iterator_complete(it))
			break;
		/* process next bytes from write buffer */
		llrb_iterator_set_msg_incomplete(it);
	}
	return wr;
}

/* llrb_update_flags - update flags after reading data */
static inline void llrb_update_flags(
	struct llrb *llrb, unsigned rd, unsigned len, unsigned *flags)
{
	/* store global flags in slot in */
	/* order to acknowledge them with */
	/* the slot */
	slot(llrb, rd).flags |= flags_get(llrb);

	*flags |= slot(llrb, rd).flags &
		LLRB_SLOT_STATUS;

	if (slot(llrb, rd).seqnum < rd)
		*flags |= LLRB_SLOT_OUTDATED;

	if (slot(llrb, rd).crc !=
	    llrb_calculate_crc(llrb, rd))
		*flags |= LLRB_SLOT_CRC_ERROR;

	if (len < slot(llrb, rd).data.len)
		/* slot has not been fully */
		/* read */
		*flags |=
			LLRB_SLOT_INCOMPLETE;
	/* mark slot as read */
	slot(llrb, rd).flags |=
		LLRB_SLOT_READ;
}

/* llrb_read - read data from the ring buffer */
static unsigned llrb_read(
	struct llrb *llrb, unsigned rd, struct llrb_iterator *it,
	unsigned long long *time, unsigned *flags)
{
	/* check for the allocated slot rd */
	if (rd < atomic_read(&llrb->wr)) {
		if (state_transition(llrb, rd, LLRB_WRITTEN, LLRB_READING)) {
			/* check that data has not been overwritten */
			/* with newer data */
			if (slot(llrb, rd).seqnum <= rd) {
				/* either it's the sequence number wanted */
				/* or some older data occupying the slot */
				/* retrieve data */
				unsigned len = llrb_data_get(
					&slot(llrb, rd).data, it);

				if (time)
					*time = slot(llrb, rd).time;

				if (flags)
					llrb_update_flags(llrb, rd, len, flags);

				/* success, slot is still in use */
				(void) state_transition(llrb, rd,
							LLRB_READING,
							LLRB_WRITTEN);

				/* reading of one slot done */
				return 0;
			}

			/* data has been overwritten with newer data */
			if (flags)
				*flags = LLRB_SLOT_DROPPED;

			/* slot is still in use */
			(void) state_transition(llrb, rd,
						LLRB_READING, LLRB_WRITTEN);
		} else if (state_get(llrb, rd) != LLRB_FREE) {
			/* slot is busy */
			if (flags)
				*flags = LLRB_SLOT_BUSY;
		}
	}

	/* something went wrong during read */
	return ~0;
}

/* llrb_free - free ring buffer slots */
unsigned llrb_free(struct llrb *llrb, unsigned rd, unsigned wr)
{
	/* start at the first not yet free'd slot at least */
	rd = max(rd, (unsigned) atomic_read(&llrb->free)+1);

	/* up to the last allocated slot at most */
	wr = min(wr, (unsigned) atomic_read(&llrb->wr)-1);

	/* check for allocated slots within rd...wr */
	while (rd <= wr) {
		if (state_transition(llrb, rd, LLRB_WRITTEN, LLRB_READING)) {
			if (slot(llrb, rd).seqnum <= rd &&
			    (slot(llrb, rd).flags & LLRB_SLOT_READ)) {
				/* the slot has been read */
				/* acknowledge global flags */
				flags_clear(llrb, slot(llrb, rd).flags);

				/* mark the slot as unused */
				(void) state_transition(llrb, rd,
							LLRB_READING,
							LLRB_FREE);
				/* only one reader is allowed */
				/* slot has been freed */
				(void) atomic_set(&llrb->free, rd);
				/* slot has been read - increase read pointer */
				(void) atomic_set(&llrb->rd, rd+1);

				rd++;
				continue;
			}

			/* enqueue it again */
			(void) state_transition(llrb, rd,
						LLRB_READING, LLRB_WRITTEN);
		} else if (state_get(llrb, rd) == LLRB_FREE) {
			/* the slot has been free'd */
			(void) atomic_cmpxchg(&llrb->free, rd-1, rd);
			(void) atomic_cmpxchg(&llrb->rd, rd, rd+1);
		}

		rd++;
	}

	return atomic_read(&llrb->free);
}

/* llrb_can_read - determine if any and which slot can be read */
unsigned llrb_can_read(
	struct llrb *llrb, unsigned rd, int skip)
{
	unsigned wr = atomic_read(&llrb->wr);

	/* start at the first not yet free'd slot */
	rd = max(rd, (unsigned) atomic_read(&llrb->free)+1);

	/* check for used slots */
	/* obeye the low watermark only if skip is not set */
	while (rd < wr && (skip || wr - rd >= llrb->low)) {
		/* determine the state of the oldest slot to be read */
		switch (state_get(llrb, rd)) {
		case LLRB_FREE:
			/* the oldest slot is free, skip it */
			break;
		case LLRB_READING:
			if (wr - rd < llrb->high)
				return 0;
			break;
		case LLRB_WRITING:
			/* the oldest slot can't be read but in case a
			 * threshold of 25 percent of size is reached
			 * we start to deliver messages even if some may
			 * be not completeley processed. This is in order
			 * to avoid overwriting messages because of one yet
			 * not finished message.
			 */
			if (wr - rd < llrb->high)
				/* give time to complete message */
				return 0;
			else
				/* threshold reached, deliver content of slot */
				(void) state_transition(llrb, rd,
							LLRB_WRITING,
							LLRB_WRITTEN);
		case LLRB_WRITTEN:
			/* the oldest slot can be read */
			if (slot(llrb, rd).seqnum <= rd)
				/* ... and it's really the slot expected */
				if (state_transition(llrb, rd,
						     LLRB_WRITTEN,
						     LLRB_READING)) {
					/* check if it has been read already */
					if (!(slot(llrb, rd).flags &
					      LLRB_SLOT_READ)) {
						(void) state_transition(
							llrb, rd,
							LLRB_READING,
							LLRB_WRITTEN);

						return rd;
					}

					(void) state_transition(llrb, rd,
								LLRB_READING,
								LLRB_WRITTEN);
				}
			break;
		}

		/* look at the next slot */
		rd++;
	}

	/* no slot to be read now */
	return skip ?
		rd /* skip already read slots, read here later */ :
		0 /* the ring buffer is (almost) empty */;
}

/* llrb_can_write - determine if and how many slots can be written */
unsigned llrb_can_write(struct llrb *llrb)
{
	unsigned rd = atomic_read(&llrb->rd);
	unsigned wr = atomic_read(&llrb->wr);

	if (wr - rd < llrb->size)
		/* some slots are available */
		return llrb->size-(wr - rd);

	/* the ring buffer is full */
	return 0;
}

/* llrb_get_state - get the state of a slot */
unsigned llrb_get_state(struct llrb *llrb, unsigned seqnum)
{
	return state_get(llrb, seqnum);
}

/* llrb_store - store an arbitrary large block of data in the ring buffer */
unsigned llrb_store(
	struct llrb *llrb, unsigned wr, struct llrb_iterator *it)
{
	if (~wr && wr && (!state_get(llrb, wr)))
		wr = 0;
	if (~wr)
		/* append to the last slot */
		wr = llrb_append(llrb, wr, it);
	return wr;
}

/* llrb_retrieve - retrieve a block of data from the ring buffer */
unsigned llrb_retrieve(
	struct llrb *llrb, unsigned rd, struct llrb_iterator *it,
	unsigned long long *time, unsigned *flags)
{
	if (~rd && rd)
		return llrb_read(llrb, rd, it, time, flags);

	return ~0;  /* rd is invalid */
}

/* llrb_set_epoch - set the epoch value in ns */
long llrb_set_real_world_time(unsigned long long epoch_time)
{
	/* Get elapsed time by local_clock() */
	unsigned long long elapsed_time = local_clock();
	/* check elapsed is less than or equal to epoch time */
	if (time_before64(elapsed_time, epoch_time)) {
		real_world_time = epoch_time - elapsed_time;
		return 0;
	}
	/* error epoch time is less than elapsed_time */
	return -EINVAL;

}
