/*
 * linux/drivers/char/errmem/errmem.c
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/console.h>
#include <linux/errmem.h>
#include <linux/kernel.h>
#include <linux/memblock.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/io.h>

#include "llrb.h"

#ifndef CONFIG_OF
struct platform_device *errmem_device;
#endif

struct errmem_private_data {
	unsigned rd;    /* next message to read */
	unsigned wr;    /* message to write (append) */
	unsigned ack;   /* next message to acknowledge */
	unsigned read_size;  /* buffer size for IOCTL_ERRMEM_READ */
};

static int errmem_fatal;
static atomic_t nr_read_session; /* no of read sessions, must not exceed 1 */
static struct llrb *llrb; /* lock-less ring buffer */
static unsigned console_wr;
static unsigned kernel_wr;
static wait_queue_head_t q;
static struct console _errmem_console;

static int errmem_fop_open(struct inode *inode, struct file *filp)
{
	struct errmem_private_data *private = NULL;

	if (errmem_fatal)
		return -ENOEXEC;

	/* only one session with read access will be accepted in order to keep
	 * provided data content consistent and complete during this session
	 */
	if (filp->f_mode & FMODE_READ) {
		if (atomic_cmpxchg(&nr_read_session, 0, 1))
			return -EMFILE;
	}

	private = kmalloc(sizeof(struct errmem_private_data), GFP_KERNEL);
	if (!private)
		return -ENOMEM;

	private->ack = llrb_get_free(llrb);
	private->wr = 0;
	private->read_size = 1<<10;

	/* Initialize read pointer only in case of FMODE_READ (should requested
	 * from errmemd only).
	 * Initialization is time consuming and not needed for WRITE access
	 * which is the most used case.
	 */
	if (filp->f_mode & FMODE_READ)
		private->rd = llrb_can_read(llrb, private->ack, 0);

	filp->private_data = (void *)private;

	return 0;
}

static int errmem_fop_release(struct inode *inode, struct file *file)
{
	struct errmem_private_data *private =
		(struct errmem_private_data *)(file->private_data);

	if (private) {
		if (private->wr) {
			/* force writing any remaining data */
			llrb_store(llrb, private->wr, NULL);

			/* wake up readers */
			wake_up(&q);
		}

		kfree(private);
	}

	if (file->f_mode & FMODE_READ)
		atomic_set(&nr_read_session, 0);

	return 0;
}

static ssize_t errmem_write(
	struct file *filp, struct errmem_message *msg,
	const char __user *data);

static ssize_t errmem_read(
	struct file *filp, struct errmem_message *msg,
	char __user *data, size_t size);

static long errmem_fop_ioctl(
	struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct errmem_private_data *private =
		(struct errmem_private_data *) (filp->private_data);

	switch (cmd) {
	case IOCTL_ERRMEM_ACKNOWLEDGE: {
		unsigned ack = 0;
		if (!(filp->f_mode & FMODE_READ))
			return -EACCES;
		if (!private)
			return -EIO;
		if (get_user(ack, (unsigned __user *)arg))
			return -EFAULT;
		if (~ack) {
			/* acknowledge a range of message */
			private->ack = llrb_free(llrb, private->ack, ack) + 1;
		} else {
			/* re-initialize the ring buffer */
			llrb_init(llrb, llrb_get_size(llrb));
			private->rd = 1;
			private->wr = 0;
			private->ack = 1;
		}
		return 0;
	}
	case IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL: {
		return put_user(_errmem_console.level_filter,
				(int __user *)arg);
	}
	case IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL: {
		int kernel_msg_level = 0;
		if (get_user(kernel_msg_level, (int __user *)arg))
			return -EFAULT;
		if (kernel_msg_level > CONSOLEMAXLEVEL)
			return -EINVAL;
		_errmem_console.level_filter = kernel_msg_level;
		return 0;
	}
	case IOCTL_ERRMEM_GET_WATERMARK_LOW: {
		unsigned watermark = 0;
		watermark = llrb_get_watermark_low(llrb);
		return put_user(watermark, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_SET_WATERMARK_LOW: {
		unsigned watermark = 0;
		if (get_user(watermark, (unsigned __user *)arg))
			return -EFAULT;
		if (((long) watermark) < 0)
			/* in is negative, count backwards from the end */
			watermark += llrb_get_size(llrb);
		if (watermark > llrb_get_size(llrb))
			return -EINVAL;
		llrb_set_watermark_low(llrb, watermark);
		return 0;
	}
	case IOCTL_ERRMEM_GET_WATERMARK_HIGH: {
		unsigned watermark = 0;
		watermark = llrb_get_watermark_high(llrb);
		return put_user(watermark, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_SET_WATERMARK_HIGH: {
		unsigned watermark = 0;
		if (get_user(watermark, (unsigned __user *)arg))
			return -EFAULT;
		if (((long) watermark) < 0)
			/* in is negative, count backwards from the end */
			watermark += llrb_get_size(llrb);
		if (watermark > llrb_get_size(llrb))
			return -EINVAL;
		llrb_set_watermark_high(llrb, watermark);
		return 0;
	}
	case IOCTL_ERRMEM_FLUSH: {
		if (console_wr) {
			llrb_store(llrb, console_wr, NULL);
			console_wr = 0;

			/* wake up readers */
			wake_up(&q);
		}
		return 0;
	}
	case IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS: {
		unsigned slots = 0;
		slots = llrb_get_size(llrb);
		return put_user(slots, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_GET_SLOTSIZE: {
		unsigned slots_size = 0;
		slots_size = llrb_get_slotsize(llrb);
		return put_user(slots_size, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_GET_USED_SLOTS: {
		unsigned slots_used = 0;
		slots_used = llrb_get_wr(llrb) - llrb_get_rd(llrb);
		return put_user(slots_used, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_VERSION: {
		unsigned long version = ERRMEM_VERSION;
		return put_user(version, (unsigned long __user *)arg);
	}
	case IOCTL_ERRMEM_WRITE: {
		struct errmem_message msg;
		unsigned header = (unsigned)
			((struct errmem_message *) NULL)->message;

		/* copy user space message header */
		if (copy_from_user(&msg, (const char __user *) arg, header + 1))
			return -EFAULT;

		return errmem_write(
			filp, &msg,
			(const char __user *)
			((struct errmem_message *) arg)->message);
	}
	case IOCTL_ERRMEM_READ: {
		struct errmem_message msg;
		unsigned header = (unsigned)
			((struct errmem_message *) NULL)->message;
		ssize_t rc;

		if (!access_ok(VERIFY_READ, (char __user *) arg + header,
			private->read_size - header))
			return -EINVAL;

		rc = errmem_read(
			filp, &msg,
			(char __user *) arg + header,
			private->read_size - header);

		if (rc >= 0) {
			msg.length = rc;
			if (copy_to_user((void __user *) arg, &msg, header))
				return -EFAULT;
		}

		return rc;
	}
	case IOCTL_ERRMEM_SET_READ_SIZE: {
		return get_user(private->read_size, (unsigned __user *)arg);
	}
	case IOCTL_ERRMEM_SET_EPOCH_MS: {
		unsigned long long epoch_time = 0;
		if (copy_from_user(&epoch_time, (void __user *) arg,
				sizeof(unsigned long long)))
			return -EFAULT;
		epoch_time *= MILLISECONDS_TO_NANOSECONDS;
		return llrb_set_real_world_time(epoch_time);
	}
	default:
		return -ENOSYS;

	}
}

static unsigned int errmem_fop_poll(
	struct file *filp, poll_table *wait) {
	unsigned rc = 0;
	struct errmem_private_data *private =
		(struct errmem_private_data *)(filp->private_data);

	if (private) {
		poll_wait(filp, &q,  wait);

		if (filp->f_mode & FMODE_READ) {
			/* check if there is something to be read */
			unsigned rd = llrb_can_read(
				llrb, private->rd, 0);
			if (rd) {
				/* read the indicated slot next */
				private->rd = rd;

				/* can read */
				rc |= POLLIN | POLLRDNORM;
			}
		}

		if (filp->f_mode & FMODE_WRITE) {
			/* check if there is space to be write */
			if (llrb_can_write(llrb))
				/* can write */
				rc |= POLLOUT | POLLWRNORM;
		}
	}

	return rc;
}

static ssize_t errmem_fop_read(
	struct file *filp, char __user *data, size_t size, loff_t *offset)
{
	struct errmem_private_data *private =
		(struct errmem_private_data *)(filp->private_data);
	struct errmem_message msg;

	return
		!llrb_blocking_read(llrb, private->rd) ?
		errmem_read(filp, &msg, data, size) : 0;
}

static ssize_t errmem_read(
	struct file *filp, struct errmem_message *msg,
	char __user *data, size_t size)
{
	struct errmem_private_data *private =
		(struct errmem_private_data *)(filp->private_data);
	unsigned rd;
	unsigned long long time;
	unsigned flags;
	struct llrb_iterator it;

	if (!data)
		return -EINVAL;

	if (errmem_fatal)
		return -ENOEXEC;

	if (!(filp->f_mode & FMODE_READ))
		return -EACCES;

	if (!private)
		return -EIO;

	/* retrieve a message from the ring buffer */
	flags = 0;
	llrb_iterator_init(
		&it,
		(unsigned char *) data, size,
		LLRB_DATA_UNKNOWN, 1);

	rd = llrb_retrieve(
		llrb, private->rd,
		&it,
		&time, &flags);

	if (~rd) {
		/* a message has been retrieved */
		/* use fields of errmem_message to return */
		/* message attributes */

		/* sequence number */
		memcpy(&msg->internal.seqnum, &private->rd,
		       sizeof(private->rd));

		/* kind */
		msg->type = (enum errmem_entry_type)
			llrb_iterator_get_kind(&it);

		/* time */
		memcpy(&msg->internal.local_clock,
		       &time, sizeof(time));

		/* offset no longer needed */
		msg->internal.offset = 0;

		/* length */
		msg->length = (unsigned short)
			llrb_iterator_get_length(&it);

		/* flags ERRMEM_FLAG_... */
		msg->internal.flags = (unsigned short) flags;

		/* read the next message */
		private->rd = llrb_can_read(llrb, private->rd+1, 1);
		return llrb_iterator_get_length(&it);
	}

	/* the ring buffer might be temporarily busy */
	if (flags & LLRB_SLOT_BUSY)
		return -EAGAIN;

	if (flags & LLRB_SLOT_DROPPED) {
		/* some data has been lost */
		/* read the next message */
		private->rd = llrb_can_read(
			llrb, private->rd+1, 1);
		return -ENOENT;
	}

	/* the sequence number must have been acknowledged */
	memcpy(&msg->internal.seqnum, &private->rd,
	       sizeof(private->rd));
	msg->type = (enum errmem_entry_type)
		(unsigned) LLRB_DATA_UNKNOWN;
	msg->length = (unsigned short) 0;
	msg->internal.flags = (unsigned short) flags;

	private->rd = llrb_can_read(llrb, private->rd, 1);
	return 0;
}

unsigned errmem_store(int fatal, unsigned length, unsigned char *msg)
{
	struct llrb_iterator it;

	if (!llrb)
		return 0;

	errmem_fatal |= fatal;

	if (!length)
		return 0;

	/* store message in ring buffer */
	llrb_iterator_init(
		&it,
		msg,
		length, LLRB_DATA_UNKNOWN, 0);

	kernel_wr = llrb_store(llrb, kernel_wr, &it);

	/* continue a partial message later */
	if (!kernel_wr)
		/* wake up readers if the message is complete */
		wake_up(&q);

	return kernel_wr;
}
EXPORT_SYMBOL(errmem_store);

static ssize_t errmem_fop_write(struct file *filp, const char __user *data,
				size_t size, loff_t *offset) {
	struct errmem_message msg;
	ssize_t rc;
	unsigned char type;

	/* check the first byte to determine the message type */
	if (copy_from_user(&type, data, 1))
		return -EFAULT;

	msg.type = type ? ERRMEM_TYPE_ASCII : ERRMEM_TYPE_TRACE;
	msg.length = size;
	rc = errmem_write(filp, &msg, data);
	return rc < 0 ? rc : size;
}

static ssize_t errmem_write(
	struct file *filp, struct errmem_message *msg,
	const char __user *data) {
	struct errmem_private_data *private =
		(struct errmem_private_data *)(filp->private_data);
	unsigned wr;
	struct llrb_iterator it;

	if (!data)
		return -EINVAL;

	if (!(filp->f_mode & FMODE_WRITE))
		return -EACCES;

	if (!private)
		return -EIO;

	/* store user space message in ring buffer */
	llrb_iterator_init(
		&it,
		(unsigned char *) data,
		msg->length, (enum llrb_data_kind) msg->type, 1);

	wr = llrb_store(llrb, private->wr, &it);

	if (!~wr)
		return -EFAULT;

	/* continue a partial message later */
	private->wr = wr;

	/* if the message is complete */
	if (!wr)
		/* wake up readers */
		wake_up(&q);

	return 0;
}

static const struct file_operations errmem_fops = {
	.owner = THIS_MODULE,
	.open = errmem_fop_open,
	.release = errmem_fop_release,
	.unlocked_ioctl = errmem_fop_ioctl,
	.poll = errmem_fop_poll,
	.read = errmem_fop_read,
	.write = errmem_fop_write,
};

static void errmem_console_write(
	struct console *cons, const char *str, unsigned size)
{
	struct llrb_iterator it;
	llrb_iterator_init(&it, (unsigned char *) str, size,
			   LLRB_DATA_ASCII, 0);
	console_wr = llrb_store(llrb, console_wr, &it);

	/* if the message is complete */
	if (!console_wr)
		/* wake up readers */
		wake_up(&q);
}

static int errmem_console_setup(struct console *cons, char *str)
{
	return 0;
}

static struct console _errmem_console = {
	.name = "errmem",
	.write = errmem_console_write,
	.setup = errmem_console_setup,
	.flags = CON_ENABLED | CON_CONSDEV | CON_ANYTIME,
	.level_filter = 2,
};

static struct miscdevice errmem_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "errmem",
	.fops = &errmem_fops
};

static int errmem_probe(struct platform_device *pdev)
{
	phys_addr_t start = 0;
	unsigned size = 0x100000;
	void *addr;

#ifdef CONFIG_ERRMEM_ADDRESS
	start = CONFIG_ERRMEM_ADDRESS;
#endif

#ifdef CONFIG_ERRMEM_SIZE
	size = CONFIG_ERRMEM_SIZE;
#endif

#ifdef CONFIG_OF
	struct resource *res;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		pr_err("errmem: not found in device tree\n");
		return -ENODEV;
	}

	start = res->start;
	size = (unsigned) resource_size(res);
#endif

	/* try to reserve physical memory */
	if (start &&
	    (memblock_reserve(start, size) ||
	     memblock_is_memory(start))) {
		pr_warn("errmem: memblock address=0x%x size=0x%x failed\n",
			(unsigned) start, size);
		start = 0;
	}

	if (start) {
		/* use reserved physical memory*/
		addr = (void *) devm_ioremap(&pdev->dev, start, size);
		if (!addr) {
			pr_warn("errmem: ioremap failed on address=0x%x size=0x%x\n",
				(unsigned) start, size);
			return -ENOMEM;
		}
		pr_info("errmem: ioremap on address=0x%x size=0x%x\n",
			(unsigned) start, size);
	} else {
		/*
		 * reserved physical memory is not available;
		 * allocate with kmalloc anyway to have a
		 * somewhat functional error memory if even it
		 * doesn't survive a reboot
		 */
		addr = devm_kzalloc(&pdev->dev, size, GFP_ATOMIC);
		if (!addr) {
			pr_warn("errmem: devm_kzalloc failed\n");
			return -ENOMEM;
		}
		pr_info("errmem: kmalloc address=0x%p size=0x%x\n",
			addr, size);
	}

	init_waitqueue_head(&q);

	llrb = llrb_init_mem(addr, size, 1);

	if (!llrb)
		return -ENODEV;

	if (misc_register(&errmem_miscdev)) {
		pr_err("errmem: failed to register device\n");
		return -ENODEV;
	}

	pr_info("errmem: version %x.%02x, console loglevel %d\n",
		ERRMEM_VERSION>>8, ERRMEM_VERSION&0xff, console_printk[0]);

	register_console(&_errmem_console);

	return 0;
}

static int errmem_remove(struct platform_device *pdev)
{
	misc_deregister(&errmem_miscdev);
	unregister_console(&_errmem_console);

	pr_info("errmem: removed\n");
	return 0;
}

#ifdef CONFIG_OF
static const struct of_device_id errmem_ids[] = {
	{ .compatible = "ADIT,errmem" },
	{ }
};
#endif

static struct platform_driver errmem_driver = {
	.driver = {
		.name   = "errmem",
#ifdef CONFIG_OF
		.of_match_table = errmem_ids,
#endif
		.owner  = THIS_MODULE,
	},
	.probe = errmem_probe,
	.remove = errmem_remove
};

static int __init errmem_init(void)
{
	int ret = platform_driver_register(&errmem_driver);
	if (ret)
		goto fail_platform_device1;

#ifndef CONFIG_OF
	ret = -ENOMEM;
	errmem_device = platform_device_alloc("errmem", -1);
	if (!errmem_device)
		goto fail_platform_device2;

	ret = platform_device_add(errmem_device);
	if (ret)
		goto fail_platform_device3;
#endif
	return 0;

#ifndef CONFIG_OF
fail_platform_device3:
	platform_device_put(errmem_device);

fail_platform_device2:
	platform_driver_unregister(&errmem_driver);
#endif

fail_platform_device1:
	pr_info("errmem: failed ret=%d\n", ret);
	return ret;
}

static void __exit errmem_exit(void)
{
#ifndef CONFIG_OF
	if (errmem_device)
		platform_device_unregister(errmem_device);
#endif

	platform_driver_unregister(&errmem_driver);
}

subsys_initcall(errmem_init);
module_exit(errmem_exit);

MODULE_AUTHOR("Kai Tomerius <ktomerius@de.adit-jv.com>");
MODULE_DESCRIPTION("Error memory");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("2.00");
