/*
 * boottime_impl.c
 *
 * Copyright (C) Robert Bosch Car Multimedia GmbH Hildesheim 2013
 * Written by Jonas Oester <jonas.oester@de.bosch.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/init.h>
#include <linux/module.h>
#include <linux/debugfs.h>
#include "boottime_internal.h"

static struct dentry *my_dir;

struct timestamp {
	unsigned int tag;
	char *info;
};

static const struct timestamp timestamp_table[] = {
	{TIMESTAMP_TABLE_TAG_INIT1, "bl_init1"},
	{TIMESTAMP_TABLE_TAG_INIT2, "bl_init2"},
	{TIMESTAMP_TABLE_TAG_AFTER_HW_INIT, "bl_after_hw_init"},
	{TIMESTAMP_TABLE_TAG_AFTER_PROMPT, "bl_after_prompt"},
	{TIMESTAMP_TABLE_TAG_AFTER_BTM_CHECK, "bl_after_btm_check"},
	{TIMESTAMP_TABLE_TAG_AFTER_KERNEL, "bl_after_kernel"},
	{TIMESTAMP_TABLE_TAG_AFTER_DTB, "bl_after_dtb"},
	{TIMESTAMP_TABLE_TAG_AFTER_DTB_PARSE, "bl_after_dtb_parse"},
	{TIMESTAMP_TABLE_TAG_AFTER_EMMC_WAKEUP, "bl_after_emmc_wakeup"},
} ;

/*
  expr `cat /sys/kernel/debug/rbcm_boottime/kernel_first_timestamp` / \
  "(" `cat /sys/kernel/debug/rbcm_boottime/timestamp_rate` / 1000 ")"
*/

static int __init boottime_init_module(void)
{
	int err = 0;
#ifdef CONFIG_RBCM_BL_TIMESTAMP
	size_t size;
	u32 *ts_ptr;
	int i, j;
#endif
	u32 mxc_boottime_rate;
	u32 mxc_boottime_first_ts;

	mxc_get_boottime(&mxc_boottime_first_ts, &mxc_boottime_rate);
	my_dir = debugfs_create_dir("rbcm_boottime", NULL);
	if (IS_ERR(my_dir)) {
		pr_err("Unable to create rbcm_boottime directory in debugfs\n");
		return -ENOENT;
	}
	debugfs_create_u32("timestamp_rate", 0444, my_dir, &mxc_boottime_rate);
	pr_info("timestamp_rate: %d\n", mxc_boottime_rate);
	debugfs_create_u32("kernel_first_timestamp",
			   0444, my_dir, &mxc_boottime_first_ts);
	pr_info("kernel_first_timestamp: %d\n",
	       mxc_boottime_first_ts);
#ifdef CONFIG_RBCM_BL_TIMESTAMP
	ts_ptr = imx6q_get_timestamp_table(&size);
	if (IS_ERR(ts_ptr))
		return (int)ts_ptr;

	for (i = 0; i < size/sizeof(u32); ++i) {
		int found = 0;
		for (j = 0; j < sizeof(timestamp_table)/
				sizeof(struct timestamp); ++j) {
			if (ts_ptr[i] == timestamp_table[j].tag) {
				debugfs_create_u32(timestamp_table[j].info,
						   0444, my_dir, &ts_ptr[i+1]);
				pr_info("%s: %d\n", timestamp_table[j].info,
					ts_ptr[i+1]);
				found = 1;
			}
		}

		if ((!found) &&
		    (ts_ptr[i] >= TIMESTAMP_TABLE_TAG_MIN) &&
		    (ts_ptr[i] <= TIMESTAMP_TABLE_TAG_MAX)) {
			pr_info("unknown timestamp tag 0x%08x: %d\n",
				ts_ptr[i], ts_ptr[i+1]);
		}

		++i;
	}
#endif	/* CONFIG_RBCM_BL_TIMESTAMP */

	return err;
}

static void boottime_exit_module(void)
{
	debugfs_remove_recursive(my_dir);
	my_dir = NULL;
}

module_init(boottime_init_module);
module_exit(boottime_exit_module);

MODULE_AUTHOR("Jonas Oester <jonas.oester@de.bosch.com>");
MODULE_DESCRIPTION("boottime module");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("0001");
