/*
 * boardcfg_impl.c
 *
 * Copyright (C) Robert Bosch Car Multimedia GmbH Hildesheim 2012
 * Written by Jonas Oester <jonas.oester@de.bosch.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/module.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/highmem.h>
#include <linux/string.h>
#include <linux/semaphore.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/device.h>
#include <linux/mutex.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>

#define MAX_BOARDCFG_CHAR_DEVICES	1
#define BOARDCFG_MAX_EXPORTED_GPIOS	64

static struct class *boardcfg_class;
static struct device *boardcfg_device;

struct boardcfg_data {
	struct mutex lock;
	int initialized;
	int exported_gpios[BOARDCFG_MAX_EXPORTED_GPIOS];
	int num_gpios;
};

static const char *gpio_dt_node_path = "/board-configuration/exported-gpios";

static int trigger_exports(struct device *dev,
			   struct boardcfg_data *bc)
{
	struct device_node *dn;
	struct device_node *dn_child;
	struct property *prop;
	int index;

	dn = of_find_node_by_path(gpio_dt_node_path);
	if (!dn) {
		pr_warn("can't find %s node in device tree\n",
		       gpio_dt_node_path);
		return -ENOENT;
	}

	mutex_lock(&bc->lock);

	index = 0;
	for_each_child_of_node(dn, dn_child) {
		int gpio;
		int gpio_flags = 0;
		int err = 0;
		int active_low = 0;
		u32 value;

		if (bc->initialized && (bc->exported_gpios[index] >= 0)) {
			index++;
			continue;
		}

		gpio = of_get_gpio(dn_child, 0);
		bc->exported_gpios[index++] = gpio;
		if (gpio < 0) {
			pr_warn("of_get_gpio returned %d for %s",
			       gpio, dn_child->name);
			continue;
		}

		if (index > BOARDCFG_MAX_EXPORTED_GPIOS - 1) {
			pr_warn("exceeded maximum number of exported GPIOs\n");
			break;
		}

		prop = of_find_property(dn_child, "input", NULL);
		if (prop != NULL)
			gpio_flags |= GPIOF_DIR_IN;

		prop = of_find_property(dn_child, "output", NULL);
		if (prop != NULL)
			gpio_flags |= GPIOF_DIR_OUT;

		err = of_property_read_u32_array(dn_child, "value", &value, 1);
		if ((err < 0) && !(gpio_flags & GPIOF_DIR_IN)) {
			pr_warn("No value given for output GPIO %s\n",
			       dn_child->name);
			continue;
		}

		prop = of_find_property(dn_child, "active-low", NULL);
		if (prop != NULL)
			active_low = 1;

		if (!(gpio_flags & GPIOF_DIR_IN))
			gpio_flags |= value ? GPIOF_INIT_HIGH : GPIOF_INIT_LOW;

		gpio_flags |= GPIOF_EXPORT_DIR_CHANGEABLE;
		err = gpio_request_one(gpio, gpio_flags, dn_child->name);
		if (err < 0) {
			pr_warn("gpio_request_one returned %d for %s (%d)\n",
			       err, dn_child->name, gpio);
			continue;
		}

		err = gpio_export_link(dev, dn_child->name, gpio);
		if (err < 0) {
			pr_warn("gpio_export_link returned %d for %s (%d)\n",
			       err, dn_child->name, gpio);
			continue;
		}

		err = gpio_sysfs_set_active_low(gpio, active_low);
		if (err < 0) {
			pr_warn("gpio_sysfs_set active_low returned %d for %d (%d)\n",
				err, active_low, gpio);
			continue;
		}

		pr_debug("%s (%d) exported\n", dn_child->name, gpio);
	}

	if (!bc->initialized) {
		bc->initialized = 1;
		bc->num_gpios = index;
	}

	mutex_unlock(&bc->lock);

	of_node_put(dn);

	return 0;
}

#ifdef CONFIG_BOARDCFG_DEBUG
static ssize_t retrigger_store(struct device *dev,
			       struct device_attribute *attr,
			       const char *buf, size_t count)
{
	struct boardcfg_data *bc = (struct boardcfg_data *)
		dev_get_drvdata(boardcfg_device);

	trigger_exports(dev, bc);

	return count;
}

DEVICE_ATTR(retrigger, S_IWUSR, NULL, retrigger_store);

static void boardcfg_create_debug_files(void)
{
	device_create_file(boardcfg_device, &dev_attr_retrigger);
}

static void boardcfg_remove_debug_files(void)
{
	device_remove_file(boardcfg_device, &dev_attr_retrigger);
}

#else

static void boardcfg_create_debug_files(void)
{
}

static void boardcfg_remove_debug_files(void)
{
}

#endif /* #ifdef CONFIG_BOARDCFG_DEBUG */

static int __init boardcfg_init_module(void)
{
	int err = 0;
	struct boardcfg_data *private_data;

	boardcfg_class = class_create(THIS_MODULE, "boardcfg");
	if (!boardcfg_class) {
		pr_warn("can't create device class\n");
		err = -ENOMEM;
		goto error;
	}

	boardcfg_device = device_create(boardcfg_class, NULL, 0, NULL,
				"boardcfg");
	if (!boardcfg_device) {
		pr_warn("can't create device\n");
		err = -ENOMEM;
		goto error_after_class;
	}

	private_data = kzalloc(sizeof(struct boardcfg_data), GFP_KERNEL);
	if (private_data == NULL) {
		pr_warn("unable to allocate private data\n");
		err = -ENOMEM;
		goto error_after_device;
	}

	mutex_init(&private_data->lock);

	dev_set_drvdata(boardcfg_device, private_data);

	boardcfg_create_debug_files();

	trigger_exports(boardcfg_device, private_data);

	pr_info("boardcfg_init_module: done\n");
	return err;

error_after_device:
	device_destroy(boardcfg_class, 0);
error_after_class:
	class_destroy(boardcfg_class);
error:
	pr_info("boardcfg_init_module returns %d\n", err);
	return err;
}

static void boardcfg_exit_module(void)
{
	struct boardcfg_data *private_data = (struct boardcfg_data *)
		dev_get_drvdata(boardcfg_device);
	int i;

	for (i = 0; i < private_data->num_gpios; ++i) {
		if (private_data->exported_gpios[i] < 0)
			continue;

		/* sysfs symbolic link is removed when the
		   entire device goes away */
		gpio_unexport(private_data->exported_gpios[i]);
		gpio_free(private_data->exported_gpios[i]);
	}

	boardcfg_remove_debug_files();
	kfree(private_data);
	device_destroy(boardcfg_class, 0);
	class_destroy(boardcfg_class);

	pr_info("boardcfg_exit_module: done\n");
}

module_init(boardcfg_init_module);
module_exit(boardcfg_exit_module);

MODULE_AUTHOR("Jonas Oester <jonas.oester@de.bosch.com>");
MODULE_DESCRIPTION("boardcfg module");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("0001");
