/*
 * adc-inc.c - ADC port extender over INC
 *
 *   Copyright (c) 2013 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
#include <linux/version.h>
#include <linux/module.h>  /* Needed by all modules */
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/kernel.h>  /* Needed for KERN_ALERT */
#include <linux/cdev.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/gfp.h>
#include <linux/err.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/mutex.h>
#include <linux/wait.h>
#include <linux/in.h>
#include <linux/inc.h>
#include <linux/inc_ports.h>
#include <linux/inc_scc_port_extender_adc.h>
#include <linux/net.h>
#include <linux/adc_inc_ioctl.h>
#include <linux/uaccess.h>
#include <linux/capability.h>
#include <linux/kthread.h>
#include <linux/poll.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/dgram_service.h>
#include <linux/sched.h>

MODULE_DESCRIPTION("ADC Port Extender");
MODULE_AUTHOR("Basavaraj Nagar <basavaraj.rayappanagar@in.bosch.com>");
MODULE_LICENSE("GPL");

/**
 * external configuration of remote port may be required
 * for tesing purpose
 */

static ushort remote_port = 0x0000;
module_param(remote_port, ushort, 0);
MODULE_PARM_DESC(remote_port, "INC remote_port");

#define ADC_INC_DRV_NAME				"adc-inc"
#define ADC_INC_STATUS_ACTIVE				0x01
#define ADC_INC_STATUS_INACTIVE				0x02
#define ADC_INC_MAX_CHANNELS				256
#define ADC_INC_ADC_RESOLUTION_10BIT			0x0A
#define ADC_INC_ADC_RESOLUTION_12BIT			0x0c
#define ADC_INC_THRESHOLD_UPPER_LIMIT			0x01
#define ADC_INC_THRESHOLD_LOWER_LIMIT			0x02
#define ADC_INC_GETSAMPLE_BYTEPER_SAMPLE		0x02
#define ADC_INC_THRESHOLD_HIT				0x01
#define ADC_INC_THRESHOLD_NOHIT				0x00
#define ADC_INC_R_GET_SAMPLE_MIN_SIZE			0x02
#define ADC_INC_R_COMP_STAT_MIN_SIZE			0x02
#define ADC_INC_R_SET_THRESHOLD_SIZE			0x05
#define ADC_INC_R_GET_CONFIG_MIN_SIZE			0x01
#define ADC_INC_R_THRESHOLD_HIT_MIN_SIZE		0X04
#define ADC_INC_R_REJECT_MIN_SIZE			0X03
#define FALSE						0x00
#define TRUE						0x01
#define ADC_INC_PEER_RESPONSE_TIMEOUT	(1*(HZ)) /* 1 second */

#define ADC_RX_BUFFER_SIZE	((ADC_INC_MAX_CHANNELS * \
			ADC_INC_GETSAMPLE_BYTEPER_SAMPLE) + 2)
#define ADC_INC_DGRAM_MAX				ADC_RX_BUFFER_SIZE

#ifdef DEBUG
	#define dbg_msg(string, args...) \
		pr_debug("%s:%s:%d:" string, \
			ADC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define err_msg(string, args...) \
		pr_err("%s:%s:%d:" string, \
			ADC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define alert_msg(string, args...) \
		pr_alert("%s:%s:%d:" string, \
			ADC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define warn_msg(string, args...) \
		pr_warn("%s:%s:%d:" string, \
			ADC_INC_DRV_NAME, __func__, __LINE__, ##args)
	#define info_msg(string, args...) \
		pr_info("%s:%s:%d:" string, \
			ADC_INC_DRV_NAME, __func__, __LINE__, ##args)
#else
	#define dbg_msg(string, args...) \
		pr_debug("%s:" string, \
				__func__, ##args)
	#define err_msg(string, args...) \
		pr_err("%s:" string, \
				__func__, ##args)
	#define alert_msg(string, args...) \
		pr_alert("%s:" string, \
				__func__, ##args)
	#define warn_msg(string, args...) \
		pr_warn("%s:" string, \
				__func__, ##args)
	#define info_msg(string, args...) \
		pr_info("%s:" string, \
				__func__, ##args)
#endif

/* ================================================================ */

/* Device specific data */

/* ================================================================ */
struct adc_inc_device {
	 struct __wait_queue_head adc_channel_data;
	 struct __wait_queue_head adc_ch_thrld_hit;
	 struct __wait_queue_head adc_channel_set_threshold;
	 struct __wait_queue_head adc_waitq;
	 struct socket		*sock;
	 struct sk_dgram	*dgram;
	 struct sockaddr_in local;
	 struct sockaddr_in remote;
	 struct mutex lock;
	 struct task_struct *rx_task;
	 int upper_threshold_hit;
	 int lower_threshold_hit;
	 /*Pointer to Buffer for storing ADC sample value*/
	 u_int8_t *buffer;
	 /*Pointer to array of flags for waking up the read function*/
	 u_int8_t *data_received;
	 /*Pointer to array of flags for waking up the poll function*/
	 u_int8_t *thrld_hit ;
	 /*Flag indicating if set threshold for a channel is requested or not*/
	 u_int8_t *set_threshold;
	 /*Pointer to buffer which holds the threshold info per channel*/
	 struct dev_threshold *thrld_buffer;
	 /*Array of flags for locking a channel*/
	 pid_t *openlock ;
	 /*holds the resolution data of the device*/
	 u_int8_t resolution;
	 int	adc_rsp_status_active;
};

struct adc_inc_device_info {
	struct device *dev;
	struct cdev cdev;
};

static unsigned int adc_inc_major;
static struct adc_inc_device_info *adc_inc_dev_info;
static struct class *adc_inc_class;
static struct adc_inc_device *adc_inc_dev;
static unsigned int adc_inc_n_channels;
static unsigned char adc_inc_resolution;
static unsigned char adc_inc_init_pending;
/* ================================================================ */

static int adc_inc_late_init(void);

static u32 inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a;
	arr[1] = b;
	arr[2] = c;
	arr[3] = d;
	return *(u32 *)arr;

}

static int adc_inc_get_dt_properties(void)
{
	const char *property_string = NULL, *inc_node_path = NULL;
	const  void *prop = NULL;
	struct device_node *inc_node = NULL, *adcincdev = NULL;

	adcincdev = of_find_node_by_path("/adc_inc");
	if (adcincdev == NULL) {
		err_msg("Couldnot find root node\n");
		return -ENODEV;
	}

	property_string = "node";
	prop = of_get_property(adcincdev, property_string, NULL);
	if (prop != NULL)
		inc_node_path = (char *) prop;
	else
		goto err;

	inc_node = of_find_node_by_path(inc_node_path);

	if (inc_node == NULL) {
		err_msg("can't find %s node in device tree\n", inc_node_path);
		return -ENODEV;
	}

	dbg_msg("\n");

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	adc_inc_dev->local.sin_family = AF_INC;
	adc_inc_dev->remote.sin_family = AF_INC;

	property_string = "local-addr";

	prop = of_get_property(inc_node, property_string, NULL);

	if (prop) {
		adc_inc_dev->local.sin_addr.s_addr = inet_addr((char *)prop);
	} else {
		err_msg("Couldnt find local-addr in device tree\n");
		goto err;
	}

	property_string = "remote-addr";
	prop = of_get_property(inc_node, property_string, NULL);

	if (prop) {
		adc_inc_dev->remote.sin_addr.s_addr = inet_addr((char *)prop);
	} else {
		err_msg("Couldnt find remote-addr in device tree\n");
		goto err;
	}

	if (remote_port) {
		adc_inc_dev->local.sin_port = htons(PORT_EXTENDER_ADC_PORT);
		adc_inc_dev->remote.sin_port = htons(remote_port);
	} else {
		adc_inc_dev->local.sin_port = htons(PORT_EXTENDER_ADC_PORT);
		adc_inc_dev->remote.sin_port = adc_inc_dev->local.sin_port;
		info_msg("remote_port read from device-tree\n");
	}
	return 0;

err:
	err_msg("error in parsing device tree\n");
	return -1;
}


static int adc_inc_recvmsg(void *data, int len)
{
	int ret = 0, i = 0;

	if (adc_inc_dev == NULL) {
		err_msg("Null pointer passed\n");
		ret  = -1;
		goto err;
	}

	dbg_msg("inside adc_inc_recvmsg len:%d\n", len);

	do {
		ret = dgram_recv(adc_inc_dev->dgram, data, len);

		if ((ret < 0) && (ret != -EAGAIN))
			alert_msg("recvmsg failed ret: %d\n", ret);

		for (i = 0; i < ret; i++)
			dbg_msg("data[%d]=0x%02X\n", i, ((u_int8_t *)data)[i]);

	} while (ret == -EAGAIN);

err:
	return ret;
}

static int adc_inc_sendmsg(u_int8_t *data, int len)
{
	int ret = 0 , i = 0;

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	dbg_msg("dgram_send to %pI4:%d\n",
			&adc_inc_dev->remote.sin_addr.s_addr,
			ntohs(adc_inc_dev->remote.sin_port));

	ret = dgram_send(adc_inc_dev->dgram, data, len);

	if (ret < 0)
		alert_msg("Couldnot not send message: %d\n", ret);
	else
		dbg_msg("ADC send message done return value : %d\n", ret);


	for (i = 0; i < ret; i++)
		dbg_msg("data[%d]=0x%02X\n", i, ((u_int8_t *)data)[i]);

	return ret;
}

static int  adc_inc_sock_open(void)
{
	int ret = 0;
	struct socket *socket = NULL;
	struct sk_dgram *dgram = NULL;

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	dbg_msg("local %pI4:%d remote %pI4:%d\n",
			&adc_inc_dev->local.sin_addr.s_addr,
			ntohs(adc_inc_dev->local.sin_port),
			&adc_inc_dev->remote.sin_addr.s_addr,
			ntohs(adc_inc_dev->remote.sin_port));

	ret = sock_create_kern(AF_INC, SOCK_STREAM, 0, &socket);

	if (ret < 0) {
		err_msg("failed to initialize socket\n");
		goto err;
	} else {
		dbg_msg("done creating socket\n");
	}

	dgram = dgram_init(socket, ADC_INC_DGRAM_MAX, NULL);

	if (dgram == NULL) {
		err_msg("dgram_init failed\n");
		goto err;
	}

	ret = kernel_bind(socket, (struct sockaddr *)&adc_inc_dev->local,
			sizeof(adc_inc_dev->local));

	if (ret < 0) {
		err_msg("binding to socket failed ret %d\n", ret);
		goto sock_release;
	} else {
		dbg_msg("done binding to socket\n");
	}

	ret = kernel_connect(socket, (struct sockaddr *)&adc_inc_dev->remote,
			sizeof(adc_inc_dev->remote), 0);

	if (ret < 0) {
		err_msg("connecting to socket failed ret %d\n", ret);
		goto sock_release;
	} else {
		dbg_msg("done connecting to socket\n");
	}

	adc_inc_dev->sock = socket;
	adc_inc_dev->dgram = dgram;

	return ret;

sock_release:
	sock_release(socket);
err:
	return ret;

}

static void adc_inc_sock_close(void)
{

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return;
	}

	dbg_msg("local %pI4:%d remote %pI4:%d\n",
			&adc_inc_dev->local.sin_addr.s_addr,
			ntohs(adc_inc_dev->local.sin_port),
			&adc_inc_dev->remote.sin_addr.s_addr,
			ntohs(adc_inc_dev->local.sin_port));

	if (adc_inc_dev->dgram) {
		dbg_msg("dgram_exit\n");
		dgram_exit(adc_inc_dev->dgram);
	} else {
		err_msg("dgram is null\n");
	}

	if (adc_inc_dev->sock) {
		dbg_msg("sock_shutdown\n");
		kernel_sock_shutdown(adc_inc_dev->sock, SHUT_RDWR);
	} else {
		err_msg("sock is null\n");
	}

	if (adc_inc_dev->sock) {
		dbg_msg("sock_release\n");
		sock_release(adc_inc_dev->sock);
	}
	return;
}
static int adc_inc_handle_threshold_set_response(u_int8_t channel,
		u_int8_t *data)
{
	u8 encomparision = data[0];
	u16 threshold = data[2];
	char thlevel[10] = {0};
	threshold = (threshold << 8) | data[1];

	if (adc_inc_dev == NULL) {
		alert_msg("Device structure pointer is NULL\n");
		return -1;
	}

	if (encomparision == ADC_INC_THRESHOLD_UPPER_LIMIT)
		strcpy(thlevel, "Upper");
	else if (encomparision == ADC_INC_THRESHOLD_LOWER_LIMIT)
		strcpy(thlevel, "Lower");
	else
		strcpy(thlevel, "Unknown");

	dbg_msg("%s Threshold is set with value %d, data[1]=%d,data[2]=%d\n",
			thlevel, threshold, data[1], data[2]);

	mutex_lock(&adc_inc_dev->lock);

	adc_inc_dev->thrld_buffer[channel].u16threshold = threshold;
	adc_inc_dev->thrld_buffer[channel].encomparision = encomparision;
	adc_inc_dev->set_threshold[channel] = TRUE ;

	mutex_unlock(&adc_inc_dev->lock);

	wake_up_interruptible(&adc_inc_dev->adc_channel_set_threshold);

	return 0;
}

static int adc_inc_handle_threshold_hit(u_int8_t channel, u_int8_t upperhit,
		u_int8_t lowerhit)
{
	int ret = 0;

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	mutex_lock(&adc_inc_dev->lock);

	if (upperhit == ADC_INC_THRESHOLD_HIT &&\
			lowerhit == ADC_INC_THRESHOLD_NOHIT) {
		adc_inc_dev->thrld_hit[channel] =\
			ADC_INC_THRESHOLD_UPPER_LIMIT;

	} else if (lowerhit == ADC_INC_THRESHOLD_HIT &&\
			upperhit == ADC_INC_THRESHOLD_NOHIT) {
		adc_inc_dev->thrld_hit[channel] =\
			ADC_INC_THRESHOLD_LOWER_LIMIT;
	} else {
		alert_msg("Ivalid msg from remote adc\n");
		alert_msg("upperhit :%d lowerhit : %d\n",
				upperhit, lowerhit);
		ret = -EINVAL;
	}
	mutex_unlock(&adc_inc_dev->lock);

	return ret;
}

static int adc_inc_cmd_status_active(void)
{
	u_int8_t data[] = { SCC_PORT_EXTENDER_ADC_C_COMPONENT_STATUS_MSGID,
			ADC_INC_STATUS_ACTIVE};

	int ret = 0;

	ret = adc_inc_sendmsg(data, sizeof(data));
	if (ret < 0) {
		err_msg("Error in sending message\n");
		return -1;
	}

	return ret;
}


static int adc_inc_buffer_update(u_int8_t nadc, u_int8_t *data, int msg_size)
{

	u_int8_t no_of_adc_bytes = (nadc * ADC_INC_GETSAMPLE_BYTEPER_SAMPLE);
	int ret = 0 , channelcount = 0;

	if (nadc > adc_inc_n_channels) {
		alert_msg("configured channel not same as received channel\n");
		alert_msg("configured channel %d\n", adc_inc_n_channels);
		alert_msg("received channel %d\n", nadc);
		return -EINVAL;
	}

	if (msg_size < (ADC_INC_R_GET_SAMPLE_MIN_SIZE + no_of_adc_bytes)) {
		alert_msg("Message incomplete\n");
		return -EINVAL;
	}

	if (adc_inc_dev == NULL) {
		alert_msg("Device structure pointer is NULL\n");
		return -1;
	}

	mutex_lock(&adc_inc_dev->lock);

	memcpy(adc_inc_dev->buffer, data, no_of_adc_bytes);

	for (; channelcount < nadc; channelcount++) {

		dbg_msg("channelcount:%d\n", channelcount);

		adc_inc_dev->data_received[channelcount] = TRUE;
	}

	dbg_msg("Unlocking mutex in ret %d\n", ret);

	mutex_unlock(&adc_inc_dev->lock);

	return ret;
}


static int adc_inc_handle_msg(u_int8_t *data, int size)
{

	int ret = 0;

	/* before activation discard all non-activation messages */
	if (data[0] != SCC_PORT_EXTENDER_ADC_R_COMPONENT_STATUS_MSGID &&
	    !adc_inc_dev->adc_rsp_status_active) {
		dbg_msg("ADC-INC:%s: discarding msgid: 0x%02X\n",
			__func__, data[0]);
		return 0;
	}

	switch (data[0]) {

	case SCC_PORT_EXTENDER_ADC_R_COMPONENT_STATUS_MSGID:

		if (size != ADC_INC_R_COMP_STAT_MIN_SIZE) {
			alert_msg("R_COMPONENT_STATUS:Invalid msg size\n");
			return -EINVAL;
		}

		if (data[1] != ADC_INC_STATUS_ACTIVE) {
			alert_msg("Component state not Active\n");
			return -EINVAL;
		}

		/*This is done once at startup*/
		if (TRUE == adc_inc_init_pending) {
			ret = adc_inc_late_init();
			if (ret < 0) {
				alert_msg("adc_inc_late_init failed\n");
				return -EINVAL;
			} else {
				info_msg("adc_inc_late_init done\n");
				adc_inc_init_pending = FALSE;
			}
		}
		adc_inc_dev->adc_rsp_status_active = TRUE;
		wake_up_interruptible(&adc_inc_dev->adc_waitq);

		break;
	case SCC_PORT_EXTENDER_ADC_R_GET_SAMPLE_MSGID:

		dbg_msg("R_GET_SAMPLE\n");
		/*At least one channel value is expected*/
		if (size <= ADC_INC_R_GET_SAMPLE_MIN_SIZE) {
			alert_msg("R_GET_SAMPLE:Invalid msg size\n");
			return -EINVAL;
		}

		ret = adc_inc_buffer_update(data[1], &data[2], size);

		if (ret < 0)
			return ret;
		else
			wake_up_interruptible(&adc_inc_dev->adc_channel_data);

		break;
	case SCC_PORT_EXTENDER_ADC_R_SET_THRESHOLD_MSGID:

		if (size < ADC_INC_R_SET_THRESHOLD_SIZE) {
			alert_msg("R_SET_THRESHOLD response size = %d\n", size);
			return -EINVAL;
		}

		ret = adc_inc_handle_threshold_set_response(data[1]/*ADCID*/\
				, &data[2]/*data*/);
		break;
	case SCC_PORT_EXTENDER_ADC_R_GET_CONFIG_MSGID:

		if (size < ADC_INC_R_GET_CONFIG_MIN_SIZE) {
			alert_msg("R_SET_THRESHOLD response size = %d\n", size);
			return -EINVAL;
		}
		dbg_msg("get config ret = %d\n", ret);

		break;
	case SCC_PORT_EXTENDER_ADC_R_THRESHOLD_HIT_MSGID:

		if (size != ADC_INC_R_THRESHOLD_HIT_MIN_SIZE) {
			alert_msg("R_THRESHOLD_HIT Invalid message size: %d\n",
					size);
			return -EINVAL;
		}
		ret = adc_inc_handle_threshold_hit(data[1], data[2], data[3]);

		wake_up_interruptible(&adc_inc_dev->adc_ch_thrld_hit);

		dbg_msg("Threshold hit return value %d\n", ret);

		break;
	case SCC_PORT_EXTENDER_ADC_R_REJECT_MSGID:

		if (size != ADC_INC_R_REJECT_MIN_SIZE) {
			alert_msg("invalid msg size: 0x%02X %d\n",
					data[0], size);
			return -EINVAL;
		}
		alert_msg("msg rejected: id 0x%02X reason 0x%02X\n",
				data[2], data[1]);
		return -EINVAL;
	default:
		alert_msg("Invalid msgid, %d\n", data[0]);

		return -EINVAL;
		break;
	}
	return ret;
}

static int inc_handler_thread_function(void *arg)
{
	u_int8_t data[ADC_RX_BUFFER_SIZE] = {0};
	int ret = 0;

	do {
		ret = adc_inc_recvmsg(data,	ADC_RX_BUFFER_SIZE);
		dbg_msg("Received msg size %d\n", ret);

		if (ret >= 0)
			ret = adc_inc_handle_msg(data, ret);

	} while (ret >= 0);

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	mutex_lock(&adc_inc_dev->lock);
	adc_inc_dev->rx_task = NULL;
	mutex_unlock(&adc_inc_dev->lock);

	dbg_msg("Exiting\n");

	return 0;
}

static int drv_adc_open(struct inode *inode, struct file *filp)
{
	unsigned int mj = 0, mn = 0;

	mj = imajor(inode);
	mn = iminor(inode);

	if (mj != adc_inc_major || mn < 0 || mn >= adc_inc_n_channels) {
		warn_msg("No device found with minor = %d and"\
				"major = %d\n", mn, mj);
		return -ENODEV; /* No such device */
	}

	dbg_msg("open at major = %d, minor = %d\n", mj, mn);

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	/**Locking the adc channel so that one adc channel
      is used by one process at a time*/

	mutex_lock(&adc_inc_dev->lock);

	if (0 == adc_inc_dev->openlock[mn])	{
		adc_inc_dev->openlock[mn] = current->tgid;
		dbg_msg("Dev %d opend by proc tgid %d\n", mn, current->tgid);
	} else {
		err_msg("Error Proc %d is using dev %d,current %d\n",
			 adc_inc_dev->openlock[mn], mn, current->tgid);
		mutex_unlock(&adc_inc_dev->lock);
		return -EBUSY;
	}

	mutex_unlock(&adc_inc_dev->lock);

	return 0;
}

static int drv_adc_close(struct inode *inode, struct file *filp)
{
	unsigned int mj = 0, mn = 0;

	mn = iminor(inode);
	mj = imajor(inode);
	dbg_msg("close at major=%d, minor=%d\n", mj, mn);

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	mutex_lock(&adc_inc_dev->lock);

	if (current->tgid == adc_inc_dev->openlock[mn]) {
		adc_inc_dev->openlock[mn] = 0;
		dbg_msg("closing device current tgid %d\n", current->tgid);
	} else {
		err_msg("Error Dev %d busy n used by proc :%d cur-tgid %d\n",
			 mn, adc_inc_dev->openlock[mn], current->tgid);

		mutex_unlock(&adc_inc_dev->lock);
		return -EBUSY;
	}
	mutex_unlock(&adc_inc_dev->lock);

	return 0;
}

static int drv_adc_read(struct file *filp, char __user *buffer\
		, size_t count,	loff_t *f_pos)
{
	int mn = 0, retval = 0, ch_data_offset = 0;
	u_int8_t lc = 0;
	u_int8_t data[1] = {SCC_PORT_EXTENDER_ADC_C_GET_SAMPLE_MSGID};
	u_int8_t channel_buffer[ADC_INC_GETSAMPLE_BYTEPER_SAMPLE] = {0};
	struct inode *inode = filp->f_path.dentry->d_inode;
	u_int8_t bytepersmpl = ADC_INC_GETSAMPLE_BYTEPER_SAMPLE;
	int ret = 0;

	mn = iminor(inode);

	dbg_msg("minor value is %d\n", mn);

	if (count < bytepersmpl) {
		err_msg("allocate more than %d bytes\n", bytepersmpl);
		return -EINVAL;
	}

	ch_data_offset = mn * ADC_INC_GETSAMPLE_BYTEPER_SAMPLE;

	mutex_lock(&adc_inc_dev->lock);
	adc_inc_dev->data_received[mn] = FALSE;
	mutex_unlock(&adc_inc_dev->lock);

	retval = adc_inc_sendmsg(data, sizeof(data));
	if (retval < 0) {
		err_msg("send msg failed\n");
		return retval;
	}
	dbg_msg("Send msg done\n");

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	dbg_msg("putting process on wait queue\n");

	ret = wait_event_interruptible_timeout(adc_inc_dev->adc_channel_data,
		 (adc_inc_dev->data_received[mn] == TRUE),
		  ADC_INC_PEER_RESPONSE_TIMEOUT);

	adc_inc_dev->data_received[mn] = FALSE;

	if (ret < 0)
		return ret;
	if (ret == 0) {
		dev_alert(adc_inc_dev_info->dev,
			  "%s: peer not responding to ADC-Read",
			  __func__);
		ret = -ETIME;
		return ret;
	}

	dbg_msg("Out of wait queue\n");

	mutex_lock(&adc_inc_dev->lock);

	for (lc = 0; lc < ADC_INC_GETSAMPLE_BYTEPER_SAMPLE; lc++) {
		channel_buffer[lc] = adc_inc_dev->buffer[ch_data_offset+lc];
		dbg_msg("Channel_buffer[%d]:%d\n", lc, channel_buffer[lc]);
	}

	mutex_unlock(&adc_inc_dev->lock);

	retval = copy_to_user(buffer, channel_buffer, sizeof(channel_buffer));
	if (retval != 0) {
		err_msg("data copying not completed\n");
		return -EFAULT;
	}
	retval = sizeof(channel_buffer);

	return retval;

}

/**Function handles threshold setting for adc specific
 * channels over INC to remote processor*/
static int  drv_adc_set_threshold(struct dev_threshold *threshold_data\
		, int channel_id)
{
	int ret = 0;
	u_int8_t th_lowbyte = (0x00ff & threshold_data->u16threshold);
	u_int8_t th_highbyte = ((0xff00 & threshold_data->u16threshold) >> 8);
	u_int8_t u8comparision = 0;
	u_int8_t data[5] = {0};
	u_int8_t ch = channel_id;

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return -1;
	}

	/**This required because of OSAL and remote processor has different
	 * values of for Upper and lower limit defines.
	 */
	dbg_msg("encomparison flag value :%d lt: %d gt : %d\n",
			 threshold_data->encomparision, devadc_threshold_lt,
			 devadc_threshold_gt);

	if (threshold_data->encomparision == devadc_threshold_lt)
		u8comparision = ADC_INC_THRESHOLD_LOWER_LIMIT;
	else if (threshold_data->encomparision == devadc_threshold_gt)
		u8comparision = ADC_INC_THRESHOLD_UPPER_LIMIT;
	else {
		err_msg("Invalid threshold comparision argument\n");
		return -EINVAL;
	}

	data[0] = SCC_PORT_EXTENDER_ADC_C_SET_THRESHOLD_MSGID;
	data[1] = (u_int8_t)channel_id;
	data[2] = u8comparision;
	data[3] = th_lowbyte;
	data[4] = th_highbyte;

	dbg_msg("lowbyte= %d highbyte= %d\n", th_lowbyte, th_highbyte);
	dbg_msg("Sending SET_THRESHOLD msg datasize %d\n", sizeof(data));

	mutex_lock(&adc_inc_dev->lock);
	ret = adc_inc_sendmsg(data, sizeof(data));
	mutex_unlock(&adc_inc_dev->lock);

	if (ret < 0)
		goto err;
	else
		dbg_msg("Sending SET_THRESHOLD msg sent\n");

	/*Make the process wait on wait till response
	 * for SET_THRESHOLD is received*/
	ret = wait_event_interruptible_timeout(
			adc_inc_dev->adc_channel_set_threshold,
			(adc_inc_dev->set_threshold[ch] == TRUE),
			ADC_INC_PEER_RESPONSE_TIMEOUT);

	if (ret < 0)
		return ret;
	if (ret == 0) {
		dev_alert(adc_inc_dev_info->dev,
			  "%s: peer not responding to SET_THRESHOLD command",
			  __func__);
		ret = -ETIME;
		return ret;
	}

	if (adc_inc_dev->thrld_buffer[ch].u16threshold != \
			threshold_data->u16threshold ||\
			adc_inc_dev->thrld_buffer[ch].encomparision != \
			u8comparision) {

		/*Return error to application*/
		ret = -1;

		err_msg("Error threshold value at remote processor is\n");
		err_msg("ch:%d, u8com:%d,u16threshold:%d, encomparision:%d\n",
			 ch, u8comparision,
			 adc_inc_dev->thrld_buffer[ch].u16threshold,
			 adc_inc_dev->thrld_buffer[ch].encomparision);
	}
	mutex_lock(&adc_inc_dev->lock);
	/*Clear condition flag*/
	adc_inc_dev->set_threshold[ch] = FALSE;

	mutex_unlock(&adc_inc_dev->lock);

err:
	return ret;

}


static long drv_adc_ioctl(struct file *filp, unsigned int cmd\
		, unsigned long arg)
{
	struct inode *inode = filp->f_path.dentry->d_inode;
	unsigned int mn = 0;
	long ret = 0;
	void  *argp = NULL;
	struct dev_threshold devadc_setth;
	struct dev_resolution adc_config;

	argp = (void *)arg;
	mn = iminor(inode);

	dbg_msg("minor value is %d ioctl %d\n", mn, cmd);

	switch (cmd) {

	case PE_DEVADC_IOCSET_THRESHOLD:

		dbg_msg("ioctl:SET_THRESHOLD\n");
		if (_IOC_TYPE(cmd) != \
			_IOC_TYPE(PE_DEVADC_IOCSET_THRESHOLD)) {
				err_msg("IOCTL type Error..\n");
				return -ENOTTY;
		}

		ret = copy_from_user(&devadc_setth, (const void __user *)argp\
				, sizeof(struct dev_threshold));

		if (ret) {
			err_msg("Error while copy from user ret = %ld\n", ret);
			ret = -EFAULT;
		} else {
			dbg_msg("copied threshold from user\n");
			ret = drv_adc_set_threshold(&devadc_setth, mn);
		}
		break;
	case PE_DEVADC_IOCGET_CONFIG:
		/**ADC resolution is get at the time of
		 * init and stored in the Global variable */
		adc_config.adc_resolution = adc_inc_resolution;

		dbg_msg("ioctl:GET_CONFIG\n");

		ret = copy_to_user((void __user *)argp, &adc_config\
					, sizeof(struct dev_resolution));

		if (ret != 0) {
			err_msg("GET_CONFIG:Err in copytouser ret:%ld\n", ret);
			return -1;
		}
		return 0;
		break;
	default:
		err_msg("Invalid ioctl option 0x%x\n", cmd);
		return -EINVAL;
		break;
	}

	return ret;
}

static unsigned int drv_adc_poll(struct file *filp, poll_table *wait)
{

	struct inode *inode = filp->f_path.dentry->d_inode;
	unsigned int mn = iminor(inode);
	unsigned int mask = 0;

	dbg_msg("Enterinig poll mn = %d\n", mn);

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return POLLERR;
	}

	poll_wait(filp, &adc_inc_dev->adc_ch_thrld_hit, wait);

	dbg_msg("Out of threshold wait queue mn = %d\n", mn);

	mutex_lock(&adc_inc_dev->lock);

	if (adc_inc_dev->thrld_hit[mn] == ADC_INC_THRESHOLD_UPPER_LIMIT)
		mask |= POLLWRBAND;
	if (adc_inc_dev->thrld_hit[mn] == ADC_INC_THRESHOLD_LOWER_LIMIT)
		mask |= POLLRDBAND;

	adc_inc_dev->thrld_hit[mn] = 0;

	mutex_unlock(&adc_inc_dev->lock);

	dbg_msg("-----threshold mask return value = %d-----\n", mask);

	return mask;
}

static const struct file_operations adc_inc_fops = {

		.owner		=	THIS_MODULE,
		.open		=	drv_adc_open,
		.read		=	drv_adc_read,
		.release	=	drv_adc_close,
		.poll		=	drv_adc_poll,
		.unlocked_ioctl	= drv_adc_ioctl,
};

static int adc_inc_getconfig(void)
{
	u_int8_t data[1] = {SCC_PORT_EXTENDER_ADC_C_GET_SAMPLE_MSGID};
	u_int8_t recvdata[ADC_RX_BUFFER_SIZE] = {0};
	int ret = 0, ret1 = 0;

	ret = adc_inc_sendmsg(data, 1);
	if (ret < 0)
		return -1;

	ret = adc_inc_recvmsg(recvdata, ADC_RX_BUFFER_SIZE);

	if (ret <= 0)
		return -1;

	if (recvdata[0] != SCC_PORT_EXTENDER_ADC_R_GET_SAMPLE_MSGID) {
		alert_msg("Invalid msg received %d\n", recvdata[0]);
		return -EINVAL;
	}

	if (recvdata[1] > 0) {
		/*Setting global variable is safe here because
		 * this function is called during init */
		adc_inc_n_channels = recvdata[1];
		ret = recvdata[1];
		dbg_msg("Configured ADC channels are %d\n", recvdata[1]);
	} else {
		alert_msg("Invalid value of configured adc channels\n");
		return -EINVAL;
	}

	data[0] = SCC_PORT_EXTENDER_ADC_C_GET_CONFIG_MSGID;

	ret1 = adc_inc_sendmsg(data, 1);

	if (ret1 < 0)
		return -1;

	ret1 = adc_inc_recvmsg(recvdata, ADC_RX_BUFFER_SIZE);

	if (ret1 <= 0)
		return -1;

	dbg_msg("recvmsg ret1 = %d\n", ret1);

	if (recvdata[0] != SCC_PORT_EXTENDER_ADC_R_GET_CONFIG_MSGID) {
		alert_msg("Invalid msg received %d\n", recvdata[0]);
		return -1;
	}

	if ((ADC_INC_ADC_RESOLUTION_10BIT == recvdata[1]) ||
			(ADC_INC_ADC_RESOLUTION_12BIT == recvdata[1])) {
		adc_inc_resolution = recvdata[1];
		dbg_msg("adc resolution is %d\n", adc_inc_resolution);
	} else {
		alert_msg("Invalid resolution received from Remote adc\n");
		return -1;
	}

	return ret;
}

static int adc_inc_allocate_buffer(void)
{
	int ret = 0;
	u_int8_t chs = adc_inc_n_channels;
	u_int8_t bytes_per_sample = ADC_INC_GETSAMPLE_BYTEPER_SAMPLE;

	dbg_msg("\n");

	if (adc_inc_dev == NULL) {
		alert_msg("Device structure pointer is NULL\n");
		return -1;
	}

	adc_inc_dev->buffer = NULL;
	adc_inc_dev->buffer = kzalloc(chs * bytes_per_sample, GFP_KERNEL);

	if (NULL == adc_inc_dev->buffer) {
		ret = -ENOMEM;
		goto err;
	}

	adc_inc_dev->data_received = NULL;
	adc_inc_dev->data_received = kzalloc(sizeof(u_int8_t) * chs\
					, GFP_KERNEL);

	if (NULL == adc_inc_dev->data_received) {
		ret = -ENOMEM;
		goto err_freeup_buffer;	}

	adc_inc_dev->thrld_hit = NULL;
	adc_inc_dev->thrld_hit = kzalloc(sizeof(u_int8_t) * chs, GFP_KERNEL);

	if (NULL == adc_inc_dev->thrld_hit) {
		ret = -ENOMEM;
		goto err_freeup_datareceived;
	}

	adc_inc_dev->openlock = kzalloc(sizeof(u_int8_t) * chs, GFP_KERNEL);
	if (NULL == adc_inc_dev->openlock) {
		ret = -ENOMEM;
		goto err_freeup_thresholdhit;
	}

	adc_inc_dev->set_threshold = kzalloc(sizeof(u_int8_t) * chs,
			GFP_KERNEL);
	if (NULL == adc_inc_dev->set_threshold) {
		ret = -ENOMEM;
		goto err_freeup_openlock;
	}

	adc_inc_dev->thrld_buffer = kzalloc(sizeof(struct dev_threshold) * chs\
			, GFP_KERNEL);
	if (NULL == adc_inc_dev->thrld_buffer) {
		ret = -ENOMEM;
		goto err_freeup_setthreshold;
	}

	dbg_msg("Buffer memory allocated\n");

	return ret;

err_freeup_setthreshold:
	kfree(adc_inc_dev->set_threshold);
	adc_inc_dev->set_threshold = NULL;
err_freeup_openlock:
	kfree(adc_inc_dev->openlock);
	adc_inc_dev->openlock = NULL;
err_freeup_thresholdhit:
	kfree(adc_inc_dev->thrld_hit);
	adc_inc_dev->thrld_hit = NULL;
err_freeup_datareceived:
	kfree(adc_inc_dev->data_received);
	adc_inc_dev->data_received = NULL;
err_freeup_buffer:
	kfree(adc_inc_dev->buffer);
	adc_inc_dev->buffer = NULL;
err:
	alert_msg("Memory allocation error\n");
	return ret;
}


static void adc_inc_deallocate_buffer(void)
{
	dbg_msg("\n");

	if (adc_inc_dev == NULL) {
		err_msg("Device structure pointer is NULL\n");
		return;
	}

	kfree(adc_inc_dev->buffer);
	adc_inc_dev->buffer = NULL;

	kfree(adc_inc_dev->data_received);
	adc_inc_dev->data_received = NULL;

	kfree(adc_inc_dev->thrld_hit);
	adc_inc_dev->thrld_hit = NULL;

	kfree(adc_inc_dev->openlock);
	adc_inc_dev->openlock = NULL;

	kfree(adc_inc_dev->set_threshold);
	adc_inc_dev->set_threshold = NULL;

	kfree(adc_inc_dev->thrld_buffer);
	adc_inc_dev->thrld_buffer = NULL;

}

static int adc_construct_device(int minor, struct class *class)
{
	int err = 0;
	dev_t devno = MKDEV(adc_inc_major, minor);
	struct device *device = NULL;

	if (adc_inc_dev_info == NULL) {
		alert_msg("adc_inc_dev_info is NULL\n");
		return -1;
	}

	cdev_init(&adc_inc_dev_info[minor].cdev, &adc_inc_fops);
	adc_inc_dev_info[minor].cdev.owner = THIS_MODULE;

	err = cdev_add(&adc_inc_dev_info[minor].cdev, devno, 1);

	if (err) {
		alert_msg("Err %d while trying to create device %d",
				err, minor);
		return err;
	}

	/*Parameters for device_create: class, no parent class,
	 * device number, no additional data, format string, minor
	 */
	device = device_create(class, NULL, devno, NULL, "adc/%d", minor);
	adc_inc_dev_info[minor].dev = device;

	if (IS_ERR(adc_inc_dev_info[minor].dev)) {
		err = PTR_ERR(device);
		alert_msg("Error %d while trying to create %d\n", err, minor);
		cdev_del(&adc_inc_dev_info[minor].cdev);
		return err;
	}
	return 0;
}

/* Destroy the device and free its buffer */
static void adc_destroy_device(int minor, struct class *class)
{
	dbg_msg("begin\n");
	if (adc_inc_dev == NULL || adc_inc_dev_info == NULL) {
		err_msg("adc_inc structure pointer is NULL\n");
		return;
	}
	device_destroy(class, MKDEV(adc_inc_major, minor));

	mutex_lock(&adc_inc_dev->lock);
		cdev_del(&adc_inc_dev_info[minor].cdev);
	mutex_unlock(&adc_inc_dev->lock);

	return;
}

static void adc_cleanup_module(int devices_to_destroy)
{

	int i = 0;

	dbg_msg("Begin\n");

	if (adc_inc_dev) {
		/**Protecting with lock to protect against race condition
		 * that rx_task is set to NULL in kernel thread*/
		mutex_lock(&adc_inc_dev->lock);

		if (adc_inc_dev->rx_task)
			force_sig(SIGKILL, adc_inc_dev->rx_task);

		mutex_unlock(&adc_inc_dev->lock);

		if (NULL != adc_inc_dev->rx_task)
			kthread_stop(adc_inc_dev->rx_task);
		else
			err_msg("kthread pointer is NULL\n");

		/*Close inc socket*/
		adc_inc_sock_close();
		dbg_msg("Thread stopped,sock closed\n");

		/* Get rid of character devices (if any exist) */
		if (adc_inc_dev_info) {
			for (i = 0; i < devices_to_destroy; ++i)
					adc_destroy_device(i, adc_inc_class);

			kfree(adc_inc_dev_info);
		}

		dbg_msg("devices destroyed\n");

		if (adc_inc_class)
			class_destroy(adc_inc_class);

		adc_inc_deallocate_buffer();

		kfree(adc_inc_dev);
		/**Remove dangling pointer*/
		adc_inc_dev = NULL;
	}

	unregister_chrdev_region(MKDEV(adc_inc_major, 0), adc_inc_n_channels);
	return;
}

static int adc_inc_late_init(void)
{
	int ret = 0, i = 0, devices_to_destroy = 0;
	dev_t dev = 0;
	/** At this point GET_SAMPLE and GET_CONFIG message is
	* sent to remote processor to get number of ADC
	* channels configured from remote port extender  */

	ret = adc_inc_getconfig();

	if (ret <= 0) {
		alert_msg("unable to get number of ADC channels\n");
		goto remote_adc_err;
	}

	ret = adc_inc_allocate_buffer();
	if (ret < 0) {
		alert_msg("Enough memory not available\n");
		goto memory_err;
	}

	/* Create device class (before allocation of the
	 * array of devices */
	adc_inc_class = class_create(THIS_MODULE, "adc");

	if (IS_ERR(adc_inc_class)) {
		ret = PTR_ERR(adc_inc_class);
		alert_msg("Error in creating class\n");
		goto memory_err;
	}

	/* Get a range of minor numbers (starting with 0) to work with*/
	ret = alloc_chrdev_region(&dev, 0, adc_inc_n_channels,
			ADC_INC_DRV_NAME);

	if (ret < 0) {
		alert_msg("alloc_chrdev_region() failed\n");
		goto registration_failed;
	}

	adc_inc_major = MAJOR(dev);

	/* Allocate the array of devices */
	adc_inc_dev_info = kzalloc((adc_inc_n_channels *
			sizeof(struct adc_inc_device_info)), GFP_KERNEL);

	if (adc_inc_dev_info == NULL) {
		alert_msg("adc_inc_devinfo is NULL\n");
		ret = -ENOMEM;
		goto fail;
	}

	/* Construct devices */
	for (i = 0; i < adc_inc_n_channels; ++i) {
		ret = adc_construct_device(i, adc_inc_class);

		if (ret) {
			devices_to_destroy = i;
			goto fail;
		}
	}

	return ret;

registration_failed:
	class_destroy(adc_inc_class);
memory_err:
	adc_inc_deallocate_buffer();
remote_adc_err:
	adc_inc_sock_close();
	kfree(adc_inc_dev);
/*removing dangling pointer*/
	adc_inc_dev = NULL;

	return ret;
fail:
	adc_cleanup_module(devices_to_destroy);
	return ret;
}

static int __init adc_inc_init(void)
{

	int ret = 0;
	int devices_to_destroy = 0;

	/**Global data initialisation starts here */
	adc_inc_dev = kzalloc(sizeof(struct adc_inc_device), GFP_KERNEL);
	if (NULL == adc_inc_dev)
		goto fail;

	mutex_init(&adc_inc_dev->lock);

	ret = adc_inc_get_dt_properties();
	if (ret < 0)
		goto out;

	ret = adc_inc_sock_open();
	if (ret < 0)
		goto out;

	/**Initialize wait queues*/
	init_waitqueue_head(&adc_inc_dev->adc_channel_data);
	init_waitqueue_head(&adc_inc_dev->adc_ch_thrld_hit);
	init_waitqueue_head(&adc_inc_dev->adc_channel_set_threshold);
	init_waitqueue_head(&adc_inc_dev->adc_waitq);

	dbg_msg("Wait queues initialized\n");
	adc_inc_dev->adc_rsp_status_active = FALSE;
	/*Preparation for late init*/
	adc_inc_init_pending = TRUE;

	adc_inc_dev->rx_task = kthread_run(inc_handler_thread_function\
			, NULL, ADC_INC_DRV_NAME);

	if (IS_ERR(adc_inc_dev->rx_task)) {
		err_msg("failed to create inc handler thread\n");
		goto fail;
	} else {
		dbg_msg("Kthread created\n");
	}

	ret = adc_inc_cmd_status_active();
	if (ret < 0) {
		err_msg("Remote ADC not ready\n");
		goto remote_adc_err;
	}
	ret = wait_event_interruptible_timeout(adc_inc_dev->adc_waitq,
			(adc_inc_dev->adc_rsp_status_active == TRUE),
			ADC_INC_PEER_RESPONSE_TIMEOUT);

	if (ret < 0)
		goto fail;
	if (ret == 0) {
		pr_alert("ADC-INC: Remote ADC not responding");
		ret = -ETIME;
		goto fail;
	}


	return 0;


remote_adc_err:
	adc_inc_sock_close();
out:
	kfree(adc_inc_dev);
	/*removing dangling pointer*/
	adc_inc_dev = NULL;
	return ret;
fail:
	adc_cleanup_module(devices_to_destroy);
	return ret;
}

/**
 * INC communication must be initialized before
 * adc-inc initialisation.
 */
late_initcall(adc_inc_init);

static void __exit adc_exit_module(void)
{
	adc_cleanup_module(adc_inc_n_channels);
}

module_exit(adc_exit_module);

/* ================================================================ */
