/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @defgroup MXC_V4L2_CAPTURE MXC V4L2 Video Capture Driver
 */
/*!
 * @file mxc_v4l2_capture.h
 *
 * @brief mxc V4L2 capture device API  Header file
 *
 * It include all the defines for frame operations, also three structure defines
 * use case ops structure, common v4l2 driver structure and frame structure.
 *
 * @ingroup MXC_V4L2_CAPTURE
 */
#ifndef __MXC_V4L2_CAPTURE_H__
#define __MXC_V4L2_CAPTURE_H__

#include <linux/uaccess.h>
#include <linux/list.h>
#include <linux/delay.h>
#include <drm/drmP.h>
#include <drm/drm.h>
#include <drm/drm_gem_cma_helper.h>
#include <drm/drm_fb_cma_helper.h>
#include <drm/drm_crtc_helper.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <media/v4l2-int-device.h>
#include <media/v4l2-dev.h>
#include <media/v4l2-of.h>
#include <media/imx6.h>

#define FRAME_NUM 10

/*
 * There can be a maximum of 5 endpoints (and 5 sensors attached to those
 * endpoints): 1 parallel endpoints, and 4 MIPI-CSI2 endpoints for each
 * virtual channel.
 */
#define MXC_MAX_ENDPOINTS 5

/*
 * How long before no EOF interrupts cause a stream/preview
 * restart, or a buffer dequeue timeout, in msec. The dequeue
 * timeout should be longer than the EOF timeout.
 */
#define MXC_EOF_TIMEOUT       1000
#define MXC_DQ_TIMEOUT        5000

/*
 * How long to delay a restart on signal lock change or NFB4EOF, in msec.
 */
#define MXC_RESTART_DELAY      200

/*!
 * v4l2 frame structure.
 */
struct mxc_v4l_frame {
	u32 paddress;
	void *vaddress;
	int count;
	int width;
	int height;

	struct v4l2_buffer buffer;
	struct list_head queue;
	int index;
	int ipu_buf_num;
};

/*
 * The first and last input indexes of a sensor.
 */
struct mxc_sensor_input {
	int num;
	int first;
	int last;
};

/*
 * Everything to describe a V4L2 endpoint. Endpoints are handled by
 * one of the two CSI's, and connect to exactly one remote sensor.
 */
struct mxc_v4l2_endpoint {
	struct v4l2_of_endpoint ep;      /* the parsed DT endpoint info */
	struct device_node *remote_node; /* the remote sensor device node */
	struct v4l2_int_device *sensor;  /* the remote sensor when attached
					    to this endpoint */
	struct mxc_sensor_input sensor_input;
	struct ipu_csi_signal_cfg csi_sig_cfg;
};

/*!
 * common v4l2 driver structure.
 */
struct mx6_camera_dev {
	struct video_device *video_dev;
	struct v4l2_int_device this;  /* this internal device (master) */
	struct device *dev;

	/* mutex guard against SMP multithreading */
	struct mutex busy_lock;

	int open_count;

	/* params lock for this camera */
	struct mutex param_lock;

	/* Encoder */
	struct list_head ready_q;
	struct list_head done_q;
	struct list_head working_q;
	int current_buf_num;
	struct mxc_v4l_frame frame[FRAME_NUM];
	struct mxc_v4l_frame dummy_frame;
	int skip_frame;
	wait_queue_head_t enc_queue;
	int enc_counter;
	enum v4l2_buf_type type;
	struct mutex irq_lock; /* protects the above queues and counters
				  from EOF threaded interrupt handler */

	/* overlay */
	struct v4l2_window win;
	struct v4l2_framebuffer v4l2_fb;
	bool overlay_active;

	/* v4l2 format */
	struct v4l2_format user_fmt;
	struct v4l2_format sensor_fmt;

	/* Controls */
	int rotation;     /* degrees */
	int vf_rotation;  /* degrees */
	bool hflip;
	bool vflip;

	/* the rotation modes are derived from above controls */
	enum ipu_rotate_mode rot_mode;
	enum ipu_rotate_mode vf_rot_mode;

	struct v4l2_mxc_offset offset;

	/* standard */
	struct v4l2_streamparm streamparm;
	struct v4l2_standard standard;
	v4l2_std_id current_std;

	/* crop */
	struct v4l2_rect crop_bounds;
	struct v4l2_rect crop_defrect;
	struct v4l2_rect crop_current;

	/* Private data for the IPU encoder submodules */
	void *prp_still_priv;
	void *csi_enc_priv;
	void *prp_enc_priv;
	void *prp_vf_sdc_priv;

	int (*enc_update_eba) (struct mx6_camera_dev *this, dma_addr_t eba,
			       int bufferNum);
	int (*enc_enable) (struct mx6_camera_dev *this,
			   dma_addr_t eba0, dma_addr_t eba1);
	int (*enc_disable) (struct mx6_camera_dev *this);
	int (*enc_enable_csi) (struct mx6_camera_dev *this);
	int (*enc_disable_csi) (struct mx6_camera_dev *this);
	void (*enc_callback) (struct mx6_camera_dev *this, u32 mask);
	int (*vf_start_adc) (struct mx6_camera_dev *this);
	int (*vf_stop_adc) (struct mx6_camera_dev *this);
	int (*vf_start_sdc) (struct mx6_camera_dev *this);
	int (*vf_stop_sdc) (struct mx6_camera_dev *this);
	int (*vf_enable_csi) (struct mx6_camera_dev *this);
	int (*vf_disable_csi) (struct mx6_camera_dev *this);
	int (*csi_start) (struct mx6_camera_dev *this, char __user *buf);
	int (*csi_stop) (struct mx6_camera_dev *this);

	struct work_struct restart_work;
	struct work_struct stop_work;

	struct timer_list restart_timer;
	struct timer_list eof_timer;
	struct timer_list eof_vf_timer;

	/* misc status flag */
	int current_input; /* the current input */
	bool overlay_on;
	bool streaming_on;
	bool restarting;    /* streaming restart in progress */
	bool signal_locked;
	bool status_change; /* sensor status change */
	bool encoder_on;
	int overlay_pid;
	int capture_pid;
	bool low_power;
	wait_queue_head_t power_queue;
	bool using_ic; /* IC is being used in capture path */

	/* endpoints */
	struct mxc_v4l2_endpoint all_eps[MXC_MAX_ENDPOINTS];
	struct mxc_v4l2_endpoint *ep; /* the current active endpoint */
	int num_eps;

	/* the MIPI CSI-2 receiver int device */
	struct v4l2_int_device *csi2;

	/* our parent IPU */
	struct ipu_soc *ipu;
};

/* Global function prototypes */
void mxc_v4l2_poll_std_and_lock_status(struct mx6_camera_dev *cam);

#endif				/* __MXC_V4L2_CAPTURE_H__ */
