/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file ipu_prp_vf_sdc.c
 *
 * @brief IPU Use case for PRP-VF
 *
 * @ingroup IPU
 */

#include <linux/dma-mapping.h>
#include <linux/console.h>
#include <media/v4l2-int-device.h>
#include "mxc_v4l2_capture.h"
#include "ipu_prp_sw.h"


struct prpvf_priv {
	struct ipuv3_channel *prpvf_ch;
	struct ipuv3_channel *prpvf_rot_in_ch;
	struct ipuv3_channel *prpvf_rot_out_ch;
	struct ipu_ic *ic_vf;
	struct ipu_irt *irt;
	struct ipu_csi *csi;

	int eof_irq;
	int nfb4eof_irq;

	dma_addr_t rot_vf_buf[2];
	void *rot_vf_buf_vaddr[2];
	int rot_vf_buf_size[2];
	int buf_num;

	bool last_eof;  /* waiting for last EOF at preview off */
	struct completion last_eof_comp;
};

static inline struct prpvf_priv *get_priv(struct mx6_camera_dev *cam)
{
	return cam->prp_vf_sdc_priv;
}

/*
 * Update the CSI whole sensor and active windows, and initialize
 * the CSI interface and muxes.
 */
static void prpvf_setup_csi(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);
	int csi_id = ipu_csi_get_num(priv->csi);

	ipu_csi_set_window_size(priv->csi, cam->crop_current.width,
				cam->crop_current.height);
	ipu_csi_set_window_pos(priv->csi, cam->crop_current.left,
			       cam->crop_current.top);
	ipu_csi_init_interface(priv->csi, cam->crop_bounds.width,
			       cam->crop_bounds.height,
			       &cam->ep->csi_sig_cfg);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_csi_set_mipi_datatype(priv->csi, cam->ep->ep.id,
					  &cam->ep->csi_sig_cfg);

	/* select either parallel or MIPI-CSI2 as input to our CSI */
	ipu_csi_set_src(priv->csi, cam->ep->ep.id,
			cam->ep->ep.bus_type == V4L2_MBUS_CSI2);
	/* set CSI destination to IC */
	ipu_csi_set_dest(priv->csi, IPU_CSI_DEST_IC);
	/* set IC to receive from CSI */
	ipu_ic_set_src(priv->ic_vf, csi_id, false);
}

static void prpvf_put_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);

	if (!IS_ERR_OR_NULL(priv->irt))
		ipu_irt_put(priv->irt);
	priv->irt = NULL;

	if (!IS_ERR_OR_NULL(priv->ic_vf))
		ipu_ic_put(priv->ic_vf);
	priv->ic_vf = NULL;

	if (!IS_ERR_OR_NULL(priv->prpvf_ch))
		ipu_idmac_put(priv->prpvf_ch);
	priv->prpvf_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->prpvf_rot_in_ch))
		ipu_idmac_put(priv->prpvf_rot_in_ch);
	priv->prpvf_rot_in_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->prpvf_rot_out_ch))
		ipu_idmac_put(priv->prpvf_rot_out_ch);
	priv->prpvf_rot_out_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->csi))
		ipu_csi_put(priv->csi);
	priv->csi = NULL;
}

static int prpvf_get_ipu_resources(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);
	int csi_id, err;

	csi_id = cam->ep->ep.port;
	priv->csi = ipu_csi_get(cam->ipu, csi_id);
	if (IS_ERR(priv->csi)) {
		dev_err(cam->dev, "failed to get CSI %d\n", csi_id);
		return PTR_ERR(priv->csi);
	}

	priv->ic_vf = ipu_ic_get(cam->ipu, IC_TASK_VIEWFINDER);
	if (IS_ERR(priv->ic_vf)) {
		dev_err(cam->dev, "failed to get IC VF\n");
		err = PTR_ERR(priv->ic_vf);
		goto out;
	}

	priv->irt = ipu_irt_get(cam->ipu);
	if (IS_ERR(priv->irt)) {
		dev_err(cam->dev, "failed to get IRT\n");
		err = PTR_ERR(priv->irt);
		goto out;
	}

	priv->prpvf_ch = ipu_idmac_get(cam->ipu,
				       IPUV3_CHANNEL_IC_PRP_VF_MEM,
				       false);
	if (IS_ERR(priv->prpvf_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_IC_PRP_VF_MEM);
		err = PTR_ERR(priv->prpvf_ch);
		goto out;
	}

	priv->prpvf_rot_in_ch = ipu_idmac_get(cam->ipu,
					      IPUV3_CHANNEL_MEM_ROT_VF,
					      false);
	if (IS_ERR(priv->prpvf_rot_in_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_MEM_ROT_ENC);
		err = PTR_ERR(priv->prpvf_rot_in_ch);
		goto out;
	}

	priv->prpvf_rot_out_ch = ipu_idmac_get(cam->ipu,
					       IPUV3_CHANNEL_ROT_VF_MEM,
					       false);
	if (IS_ERR(priv->prpvf_rot_out_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			IPUV3_CHANNEL_ROT_ENC_MEM);
		err = PTR_ERR(priv->prpvf_rot_out_ch);
		goto out;
	}

	return 0;
out:
	prpvf_put_ipu_resources(cam);
	return err;
}

static irqreturn_t prpvf_eof_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = dev_id;
	struct prpvf_priv *priv = get_priv(cam);

	if (priv->last_eof) {
		complete(&priv->last_eof_comp);
		priv->last_eof = false;
		return IRQ_HANDLED;
	}

	mutex_lock(&cam->irq_lock);

	mxc_v4l2_poll_std_and_lock_status(cam);

	/* bump the VF EOF timeout timer */
	mod_timer(&cam->eof_vf_timer,
		  jiffies + msecs_to_jiffies(MXC_EOF_TIMEOUT));

	mutex_unlock(&cam->irq_lock);

	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_idmac_select_buffer(priv->prpvf_rot_out_ch, priv->buf_num);
	else
		ipu_idmac_select_buffer(priv->prpvf_ch, priv->buf_num);

	priv->buf_num ^= 1;

	return IRQ_HANDLED;
}

static irqreturn_t prpvf_nfb4eof_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;

	dev_err(cam->dev, "ERROR: NFB4EOF\n");

	/*
	 * It has been discovered that with rotation, prpvf disable
	 * creates a single NFB4EOF event which is 100% repeatable. So
	 * scheduling a restart here causes an endless NFB4EOF-->restart
	 * cycle. The error itself seems innocuous, capture is not adversely
	 * affected.
	 *
	 * So don't schedule a restart on NFB4EOF error. If the source
	 * of the NFB4EOF event on prpvf disable is ever found, it can
	 * be re-enabled, but is probably not necessary. Detecting the
	 * interrupt (and clearing the irq status in the IPU) seems to
	 * be enough.
	 */
#if 0
	mod_timer(&cam->restart_timer,
		  jiffies + msecs_to_jiffies(MXC_RESTART_DELAY));
#endif

	return IRQ_HANDLED;
}

/*
 * Function definitions
 */

static void prpvf_free_rot_bufs(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);

	if (priv->rot_vf_buf_vaddr[0]) {
		dma_free_coherent(cam->dev, priv->rot_vf_buf_size[0],
				  priv->rot_vf_buf_vaddr[0],
				  priv->rot_vf_buf[0]);
	}
	if (priv->rot_vf_buf_vaddr[1]) {
		dma_free_coherent(cam->dev, priv->rot_vf_buf_size[1],
				  priv->rot_vf_buf_vaddr[1],
				  priv->rot_vf_buf[1]);
	}

	priv->rot_vf_buf_vaddr[0] = NULL;
	priv->rot_vf_buf[0] = 0;
	priv->rot_vf_buf_vaddr[1] = NULL;
	priv->rot_vf_buf[1] = 0;
}

static int prpvf_alloc_rot_bufs(struct mx6_camera_dev *cam,
				struct v4l2_pix_format *outf)
{
	struct prpvf_priv *priv = get_priv(cam);
	int size = (outf->width * outf->height *
		    ipu_bits_per_pixel(outf->pixelformat)) >> 3;

	prpvf_free_rot_bufs(cam);

	priv->rot_vf_buf_size[0] = PAGE_ALIGN(size);
	priv->rot_vf_buf_vaddr[0] =
		dma_alloc_coherent(cam->dev, priv->rot_vf_buf_size[0],
				   &priv->rot_vf_buf[0], GFP_DMA | GFP_KERNEL);
	if (!priv->rot_vf_buf_vaddr[0]) {
		dev_err(cam->dev, "alloc vf_bufs0 err\n");
		return -ENOMEM;
	}

	priv->rot_vf_buf_size[1] = PAGE_ALIGN(size);
	priv->rot_vf_buf_vaddr[1] =
		dma_alloc_coherent(cam->dev, priv->rot_vf_buf_size[1],
				   &priv->rot_vf_buf[1], GFP_DMA | GFP_KERNEL);
	if (!priv->rot_vf_buf_vaddr[1]) {
		dev_err(cam->dev, "alloc vf_bufs1 err\n");
		prpvf_free_rot_bufs(cam);
		return -ENOMEM;
	}

	return 0;
}

static void prpvf_setup_channel(struct mx6_camera_dev *cam,
				struct ipuv3_channel *channel,
				struct v4l2_pix_format *f,
				enum ipu_rotate_mode rot_mode,
				dma_addr_t addr0, dma_addr_t addr1,
				bool rot_swap_width_height)
{
	struct prpvf_priv *priv = get_priv(cam);
	unsigned int burst_size;
	u32 width, height, stride;

	if (rot_swap_width_height) {
		width = f->height;
		height = f->width;
		stride = f->height;
	} else {
		width = f->width;
		height = f->height;
		stride = f->width;
	}

	stride = ipu_stride_to_bytes(stride, f->pixelformat);

	ipu_cpmem_channel_init(channel, f->pixelformat, width, height, stride,
			       0, 0, 0, addr0, addr1, 0);

	if (rot_mode)
		ipu_cpmem_set_rotation(channel, rot_mode);

	if (channel == priv->prpvf_rot_in_ch ||
	    channel == priv->prpvf_rot_out_ch) {
		burst_size = 8;
		ipu_cpmem_set_block_mode(channel);
	} else
		burst_size = (width % 16) ? 8 : 16;

	ipu_cpmem_set_burst_size(channel, burst_size);

	if (ipu_csi_is_interlaced(priv->csi) && channel == priv->prpvf_ch)
		ipu_cpmem_set_interlaced_scan(channel, 0);

	ipu_ic_task_idma_init(priv->ic_vf, channel, width, height,
			      burst_size, rot_mode);

	ipu_cpmem_set_axi_id(channel, 1);

	ipu_idmac_set_double_buffer(channel, true);
	ipu_idmac_set_triple_buffer(channel, false);
}

static int prpvf_setup_rotation(struct mx6_camera_dev *cam,
				dma_addr_t eba0, dma_addr_t eba1,
				struct v4l2_pix_format *inf,
				struct v4l2_pix_format *outf)
{
	struct prpvf_priv *priv = get_priv(cam);
	enum ipu_color_space in_cs, out_cs;
	int err;

	err = prpvf_alloc_rot_bufs(cam, outf);
	if (err) {
		dev_err(cam->dev, "prpvf_alloc_rot_bufs failed, %d\n", err);
		return err;
	}

	in_cs = ipu_mbus_code_to_colorspace(inf->pixelformat);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	err = ipu_ic_task_init(priv->ic_vf,
			       inf->width, inf->height,
			       outf->height, outf->width,
			       in_cs, out_cs);
	if (err) {
		dev_err(cam->dev, "ipu_ic_task_init failed, %d\n", err);
		prpvf_free_rot_bufs(cam);
		return err;
	}

	/* init the IC PRPVF-->MEM IDMAC channel */
	prpvf_setup_channel(cam, priv->prpvf_ch, outf,
			      IPU_ROTATE_NONE,
			      priv->rot_vf_buf[0], priv->rot_vf_buf[1],
			      true);

	/* init the MEM-->IC PRPVF ROT IDMAC channel */
	prpvf_setup_channel(cam, priv->prpvf_rot_in_ch, outf,
			      cam->vf_rot_mode,
			      priv->rot_vf_buf[0], priv->rot_vf_buf[1],
			      true);

	/* init the destination IC PRPVF ROT-->MEM IDMAC channel */
	prpvf_setup_channel(cam, priv->prpvf_rot_out_ch, outf,
			      IPU_ROTATE_NONE,
			      eba0, eba1,
			      false);

	/* now link IC PRPVF-->MEM to MEM-->IC PRPVF ROT */
	ipu_link_prpvf_rot_prpvf(cam->ipu);

	/* enable the IC and IRT */
	ipu_ic_enable(priv->ic_vf);
	ipu_irt_enable(priv->irt);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->prpvf_ch, 0);
	ipu_idmac_select_buffer(priv->prpvf_ch, 1);
	ipu_idmac_select_buffer(priv->prpvf_rot_out_ch, 0);
	ipu_idmac_select_buffer(priv->prpvf_rot_out_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->prpvf_ch);
	ipu_idmac_enable_channel(priv->prpvf_rot_in_ch);
	ipu_idmac_enable_channel(priv->prpvf_rot_out_ch);

	/* and finally enable the IC PRPVF task */
	ipu_ic_task_enable(priv->ic_vf);

	return 0;
}

static int prpvf_setup_norotation(struct mx6_camera_dev *cam,
				  dma_addr_t eba0, dma_addr_t eba1,
				  struct v4l2_pix_format *inf,
				  struct v4l2_pix_format *outf)
{
	struct prpvf_priv *priv = get_priv(cam);
	enum ipu_color_space in_cs, out_cs;
	int err;

	in_cs = ipu_mbus_code_to_colorspace(inf->pixelformat);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	err = ipu_ic_task_init(priv->ic_vf,
			       inf->width, inf->height,
			       outf->width, outf->height,
			       in_cs, out_cs);
	if (err) {
		dev_err(cam->dev, "ipu_ic_task_init failed, %d\n", err);
		return err;
	}

	/* init the IC PRPVF-->MEM IDMAC channel */
	prpvf_setup_channel(cam, priv->prpvf_ch, outf,
			    cam->vf_rot_mode, eba0, eba1, false);

	ipu_ic_enable(priv->ic_vf);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->prpvf_ch, 0);
	ipu_idmac_select_buffer(priv->prpvf_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->prpvf_ch);

	/* and finally enable the IC PRPVF task */
	ipu_ic_task_enable(priv->ic_vf);

	return 0;
}

/*!
 * prpvf_start - start the vf task
 *
 * @param cam    struct mx6_camera_dev * mxc v4l2 main structure
 *
 */
static int prpvf_start(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);
	struct v4l2_pix_format inf, outf;
	dma_addr_t fb_buf;
	int err = 0;

	if (cam->overlay_active == true) {
		dev_dbg(cam->dev, "already started.\n");
		return 0;
	}

	err = prpvf_get_ipu_resources(cam);
	if (err)
		return err;

	/* if encoder is enabled it has already setup the CSI */
	if (!cam->encoder_on)
		prpvf_setup_csi(cam);

	priv->nfb4eof_irq = ipu_idmac_channel_irq(cam->ipu,
						  priv->prpvf_ch,
						  IPU_IRQ_NFB4EOF);
	err = devm_request_threaded_irq(cam->dev, priv->nfb4eof_irq, NULL,
					prpvf_nfb4eof_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-prpvf-nfb4eof", cam);
	if (err) {
		dev_err(cam->dev,
			"Error registering prp_vf NFB4EOF irq: %d\n", err);
		goto out_put_ipu;
	}

	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT)
		priv->eof_irq = ipu_idmac_channel_irq(cam->ipu,
						      priv->prpvf_rot_out_ch,
						      IPU_IRQ_EOF);
	else
		priv->eof_irq = ipu_idmac_channel_irq(cam->ipu,
						      priv->prpvf_ch,
						      IPU_IRQ_EOF);

	err = devm_request_threaded_irq(cam->dev, priv->eof_irq, NULL,
					prpvf_eof_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-prp-vf", cam);
	if (err) {
		dev_err(cam->dev, "Error registering prp_vf irq: %d\n", err);
		goto out_free_nfb4eof_irq;
	}

	inf = cam->sensor_fmt.fmt.pix;
	inf.width = cam->crop_current.width;
	inf.height = cam->crop_current.height;
	outf.width = cam->win.w.width;
	outf.height = cam->win.w.height;
	outf.pixelformat = cam->v4l2_fb.fmt.pixelformat;

	fb_buf = (dma_addr_t)cam->v4l2_fb.base;

	priv->buf_num = 0;

	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT)
		err = prpvf_setup_rotation(cam, fb_buf, fb_buf,
					   &inf, &outf);
	else
		err = prpvf_setup_norotation(cam, fb_buf, fb_buf,
					     &inf, &outf);
	if (err)
		goto out_free_eof_irq;

	cam->overlay_active = true;

	return 0;

out_free_eof_irq:
	devm_free_irq(cam->dev, priv->eof_irq, cam);
out_free_nfb4eof_irq:
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);
out_put_ipu:
	prpvf_put_ipu_resources(cam);
	return err;
}

/*!
 * prpvf_stop - stop the vf task
 *
 * @param cam    struct mx6_camera_dev * mxc v4l2 main structure
 *
 */
static int prpvf_stop(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);

	if (cam->overlay_active == false)
		return 0;

	devm_free_irq(cam->dev, priv->eof_irq, cam);
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);

	/* disable IC tasks and the channels */
	ipu_ic_task_disable(priv->ic_vf);

	ipu_idmac_disable_channel(priv->prpvf_ch);
	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT) {
		ipu_idmac_disable_channel(priv->prpvf_rot_in_ch);
		ipu_idmac_disable_channel(priv->prpvf_rot_out_ch);
	}

	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_unlink_prpvf_rot_prpvf(cam->ipu);

	ipu_ic_disable(priv->ic_vf);
	if (cam->vf_rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_irt_disable(priv->irt);

	prpvf_free_rot_bufs(cam);

	prpvf_put_ipu_resources(cam);

	cam->overlay_active = false;
	return 0;
}

/*!
 * Enable csi
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_vf_enable_csi(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);

	return ipu_csi_enable(priv->csi);
}

/*!
 * Disable csi
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int prp_vf_disable_csi(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv = get_priv(cam);
	int ret;

	/*
	 * Mark next EOF interrupt as the last before preview off,
	 * and then wait for interrupt handler to mark completion.
	 */
	init_completion(&priv->last_eof_comp);
	priv->last_eof = true;
	ret = wait_for_completion_timeout(&priv->last_eof_comp,
					  msecs_to_jiffies(MXC_EOF_TIMEOUT));
	if (ret == 0)
		dev_warn(cam->dev, "%s: wait last EOF timeout\n", __func__);

	return ipu_csi_disable(priv->csi);
}

/*!
 * function to select PRP-VF as the working path
 *
 * @param cam    struct mx6_camera_dev * mxc v4l2 main structure
 *
 * @return  status
 */
int prp_vf_sdc_select(struct mx6_camera_dev *cam)
{
	cam->vf_start_sdc = prpvf_start;
	cam->vf_stop_sdc = prpvf_stop;
	cam->vf_enable_csi = prp_vf_enable_csi;
	cam->vf_disable_csi = prp_vf_disable_csi;
	cam->overlay_active = false;

	return 0;
}

/*!
 * function to de-select PRP-VF as the working path
 *
 * @param cam    struct mx6_camera_dev * mxc v4l2 main structure
 *
 * @return  int
 */
int prp_vf_sdc_deselect(struct mx6_camera_dev *cam)
{
	cam->vf_start_sdc = NULL;
	cam->vf_stop_sdc = NULL;
	cam->vf_enable_csi = NULL;
	cam->vf_disable_csi = NULL;

	return 0;
}


int prp_vf_sdc_init(struct mx6_camera_dev *cam)
{
	struct prpvf_priv *priv;

	priv = devm_kzalloc(cam->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	cam->prp_vf_sdc_priv = priv;

	return 0;
}

void prp_vf_sdc_exit(struct mx6_camera_dev *cam)
{
}
