/*
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2011 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file ipu_csi_enc.c
 *
 * @brief CSI Use case for video capture
 *
 * @ingroup IPU
 */

#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/dma-mapping.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-int-device.h>
#include "mxc_v4l2_capture.h"
#include "ipu_prp_sw.h"

struct csi_enc_priv {
	struct ipuv3_channel *csi_ch;
	struct ipu_smfc *smfc;
	struct ipu_csi *csi;

	int eof_irq;
	int nfb4eof_irq;

	bool last_eof;     /* waiting for last EOF at stream off */
	struct completion last_eof_comp;
};

static inline struct csi_enc_priv *get_priv(struct mx6_camera_dev *cam)
{
	return cam->csi_enc_priv;
}

/*
 * Function definitions
 */

/*
 * Update the CSI whole sensor and active windows, and initialize
 * the CSI interface and muxes.
 */
static void csi_enc_setup_csi(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);

	ipu_csi_set_window_size(priv->csi, cam->crop_current.width,
				cam->crop_current.height);
	ipu_csi_set_window_pos(priv->csi, cam->crop_current.left,
			       cam->crop_current.top);
	ipu_csi_init_interface(priv->csi, cam->crop_bounds.width,
			       cam->crop_bounds.height,
			       &cam->ep->csi_sig_cfg);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_csi_set_mipi_datatype(priv->csi, cam->ep->ep.id,
					  &cam->ep->csi_sig_cfg);

	/* select either parallel or MIPI-CSI2 as input to our CSI */
	ipu_csi_set_src(priv->csi, cam->ep->ep.id,
			cam->ep->ep.bus_type == V4L2_MBUS_CSI2);
	/* set CSI destination to memory via SMFC */
	ipu_csi_set_dest(priv->csi, IPU_CSI_DEST_IDMAC);

}

static void csi_enc_put_ipu_resources(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);

	if (!IS_ERR_OR_NULL(priv->smfc))
		ipu_smfc_put(priv->smfc);
	priv->smfc = NULL;

	if (!IS_ERR_OR_NULL(priv->csi_ch))
		ipu_idmac_put(priv->csi_ch);
	priv->csi_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->csi))
		ipu_csi_put(priv->csi);
	priv->csi = NULL;
}

static int csi_enc_get_ipu_resources(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);
	unsigned csi_ch_num;
	int csi_id, err;

	csi_id = cam->ep->ep.port;
	priv->csi = ipu_csi_get(cam->ipu, csi_id);
	if (IS_ERR(priv->csi)) {
		dev_err(cam->dev, "failed to get CSI %d\n", csi_id);
		return PTR_ERR(priv->csi);
	}

	priv->smfc = ipu_smfc_get(cam->ipu);
	if (IS_ERR(priv->smfc)) {
		dev_err(cam->dev, "failed to get SMFC\n");
		err = PTR_ERR(priv->smfc);
		goto out;
	}

	/*
	 * Choose the direct CSI-->SMFC-->MEM channel corresponding
	 * to the IPU and CSI IDs.
	 */
	csi_ch_num = IPUV3_CHANNEL_CSI0 + (ipu_get_num(cam->ipu) << 1) + csi_id;

	priv->csi_ch = ipu_idmac_get(cam->ipu, csi_ch_num, false);
	if (IS_ERR(priv->csi_ch)) {
		dev_err(cam->dev, "could not get IDMAC channel %u\n",
			csi_ch_num);
		err = PTR_ERR(priv->csi_ch);
		goto out;
	}

	return 0;
out:
	csi_enc_put_ipu_resources(cam);
	return err;
}

/*!
 * csi ENC callback function.
 *
 * @param irq       int irq line
 * @param dev_id    void * device id
 *
 * @return status   IRQ_HANDLED for handled
 */
static irqreturn_t csi_enc_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;
	struct csi_enc_priv *priv = get_priv(cam);

	if (cam->enc_callback == NULL)
		return IRQ_HANDLED;

	if (priv->last_eof) {
		complete(&priv->last_eof_comp);
		priv->last_eof = false;
		return IRQ_HANDLED;
	}

	cam->enc_callback(cam, irq);
	return IRQ_HANDLED;
}

static irqreturn_t csi_nfb4eof_callback(int irq, void *dev_id)
{
	struct mx6_camera_dev *cam = (struct mx6_camera_dev *) dev_id;

	dev_err(cam->dev, "ERROR: NFB4EOF\n");

	mod_timer(&cam->restart_timer,
		  jiffies + msecs_to_jiffies(MXC_RESTART_DELAY));

	return IRQ_HANDLED;
}

/*!
 * CSI ENC channel setup function
 *
 * @param cam           struct struct mx6_camera_dev * mxc capture instance
 * @param eba0, eba1    physaddr's of first two queued v4l2 buffers
 *
 * @return  status
 */
static void csi_enc_setup_channel(struct mx6_camera_dev *cam,
				  dma_addr_t eba0, dma_addr_t eba1)
{
	struct csi_enc_priv *priv = get_priv(cam);
	u32 pixel_fmt = cam->user_fmt.fmt.pix.pixelformat;
	unsigned int burst_size;
	bool passthrough;
	u32 stride;

	/*
	 * Set the channel for the direct CSI-->memory via SMFC
	 * use-case to very high priority, by enabling the watermark
	 * signal in the SMFC, enabling WM in the channel, and setting the
	 * channel priority to high.
	 *
	 * Refer to the iMx6 rev. D TRM Table 36-8: Calculated priority
	 * value.
	 *
	 * The WM's are set very low by intention here to ensure that
	 * the SMFC FIFOs do not overflow.
	 */
	ipu_smfc_set_wmc(priv->smfc, priv->csi_ch, false, 0x01);
	ipu_smfc_set_wmc(priv->smfc, priv->csi_ch, true, 0x02);
	ipu_cpmem_set_high_priority(priv->csi_ch);
	ipu_idmac_enable_watermark(priv->csi_ch, true);
	ipu_cpmem_set_axi_id(priv->csi_ch, 0);
	ipu_idmac_lock_enable(priv->csi_ch, 8);

	/*
	 * If the sensor uses 16-bit parallel CSI bus, we must handle
	 * the data internally in the IPU as 16-bit generic, aka
	 * passthrough mode.
	 */
	passthrough = (cam->ep->ep.bus_type != V4L2_MBUS_CSI2 &&
		       cam->ep->csi_sig_cfg.data_width ==
		       IPU_CSI_DATA_WIDTH_16);

	/* convert pixel stride to bytes */
	stride = ipu_stride_to_bytes(cam->user_fmt.fmt.pix.width, pixel_fmt);

	ipu_cpmem_channel_init(priv->csi_ch, pixel_fmt,
			       cam->user_fmt.fmt.pix.width,
			       cam->user_fmt.fmt.pix.height, stride,
			       cam->offset.u_offset, cam->offset.v_offset, 0,
			       eba0, eba1, 0);

	if (passthrough)
		ipu_cpmem_set_format_passthrough(priv->csi_ch, 16);

	if (ipu_csi_is_interlaced(priv->csi))
		ipu_cpmem_set_interlaced_scan(priv->csi_ch, 0);

	burst_size = ipu_cpmem_get_burst_size(priv->csi_ch);
	ipu_smfc_set_burst_size(priv->smfc, priv->csi_ch,
				burst_size, passthrough);

	ipu_idmac_set_double_buffer(priv->csi_ch, true);
}

/*!
 * CSI ENC setup function
 *
 * @param cam           struct struct mx6_camera_dev * mxc capture instance
 * @param eba0, eba1    physaddr's of first two queued v4l2 buffers
 *
 * @return  status
 */
static int csi_enc_setup(struct mx6_camera_dev *cam,
			 dma_addr_t eba0, dma_addr_t eba1)
{
	struct csi_enc_priv *priv = get_priv(cam);
	int csi_id = ipu_csi_get_num(priv->csi);

	csi_enc_setup_csi(cam);

	if (cam->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_smfc_map(priv->smfc, priv->csi_ch, csi_id, cam->ep->ep.id);
	else
		ipu_smfc_map(priv->smfc, priv->csi_ch, csi_id, 0);

	csi_enc_setup_channel(cam, eba0, eba1);

	ipu_smfc_enable(priv->smfc);

	ipu_idmac_select_buffer(priv->csi_ch, 0);
	ipu_idmac_select_buffer(priv->csi_ch, 1);

	ipu_idmac_enable_channel(priv->csi_ch);

	return 0;
}

/*!
 * function to update physical buffer address for encoder IDMA channel
 *
 * @param eba         physical buffer address for encoder IDMA channel
 * @param buffer_num  int buffer 0 or buffer 1
 *
 * @return  status
 */
static int csi_enc_update_eba(struct mx6_camera_dev *cam, dma_addr_t eba,
			      int buffer_num)
{
	struct csi_enc_priv *priv = get_priv(cam);

	dev_dbg(cam->dev, "%s: eba = %x\n", __func__, eba);

	if (ipu_idmac_buffer_is_ready(priv->csi_ch, buffer_num))
		ipu_idmac_clear_buffer_ready(priv->csi_ch, buffer_num);

	ipu_cpmem_set_buffer(priv->csi_ch, buffer_num, eba);
	ipu_idmac_select_buffer(priv->csi_ch, buffer_num);

	return 0;
}

/*!
 * Enable encoder task
 * @param cam      struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int csi_enc_enabling_tasks(struct mx6_camera_dev *cam,
				  dma_addr_t eba0, dma_addr_t eba1)
{
	struct csi_enc_priv *priv = get_priv(cam);
	int err = 0;

	dev_dbg(cam->dev, "%s: enter\n", __func__);

	err = csi_enc_get_ipu_resources(cam);
	if (err)
		return err;

	priv->eof_irq = ipu_idmac_channel_irq(cam->ipu,
					      priv->csi_ch,
					      IPU_IRQ_EOF);
	err = devm_request_threaded_irq(cam->dev, priv->eof_irq,
					NULL, csi_enc_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-csi-enc", cam);
	if (err) {
		dev_err(cam->dev, "Error registering EOF irq\n");
		goto out_put_ipu;
	}

	priv->nfb4eof_irq = ipu_idmac_channel_irq(cam->ipu,
						 priv->csi_ch,
						 IPU_IRQ_NFB4EOF);
	err = devm_request_threaded_irq(cam->dev, priv->nfb4eof_irq, NULL,
					csi_nfb4eof_callback, IRQF_ONESHOT,
					"imx6-v4l2-cap-csi-nfb4eof", cam);
	if (err) {
		dev_err(cam->dev, "Error registering NFB4EOF irq\n");
		goto out_free_eof_irq;
	}

	err = csi_enc_setup(cam, eba0, eba1);
	if (err) {
		dev_err(cam->dev, "csi_enc_setup %d\n", err);
		goto out_free_nfb4eof_irq;
	}

	return 0;

out_free_nfb4eof_irq:
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);
out_free_eof_irq:
	devm_free_irq(cam->dev, priv->eof_irq, cam);
out_put_ipu:
	csi_enc_put_ipu_resources(cam);
	return err;
}

/*!
 * Disable encoder task
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
static int csi_enc_disabling_tasks(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);

	devm_free_irq(cam->dev, priv->eof_irq, cam);
	devm_free_irq(cam->dev, priv->nfb4eof_irq, cam);

	ipu_idmac_disable_channel(priv->csi_ch);

	ipu_smfc_disable(priv->smfc);

	csi_enc_put_ipu_resources(cam);

	return 0;
}

/*!
 * Enable csi
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int csi_enc_enable_csi(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);

	return ipu_csi_enable(priv->csi);
}

/*!
 * Disable csi
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  status
 */
static int csi_enc_disable_csi(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv = get_priv(cam);
	int ret;

	/*
	 * Mark next EOF interrupt as the last before streamoff,
	 * and then wait for interrupt handler to mark completion.
	 */
	init_completion(&priv->last_eof_comp);
	priv->last_eof = true;
	ret = wait_for_completion_timeout(&priv->last_eof_comp,
					  msecs_to_jiffies(MXC_EOF_TIMEOUT));
	if (ret == 0)
		dev_warn(cam->dev, "%s: wait last EOF timeout\n", __func__);

	return ipu_csi_disable(priv->csi);
}

/*!
 * function to select CSI ENC as the working path
 *
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
int csi_enc_select(struct mx6_camera_dev *cam)
{
	cam->enc_update_eba = csi_enc_update_eba;
	cam->enc_enable = csi_enc_enabling_tasks;
	cam->enc_disable = csi_enc_disabling_tasks;
	cam->enc_enable_csi = csi_enc_enable_csi;
	cam->enc_disable_csi = csi_enc_disable_csi;

	return 0;
}

/*!
 * function to de-select CSI ENC as the working path
 *
 * @param cam       struct struct mx6_camera_dev * mxc capture instance
 *
 * @return  int
 */
int csi_enc_deselect(struct mx6_camera_dev *cam)
{
	cam->enc_update_eba = NULL;
	cam->enc_enable = NULL;
	cam->enc_disable = NULL;
	cam->enc_enable_csi = NULL;
	cam->enc_disable_csi = NULL;

	return 0;
}

int csi_enc_init(struct mx6_camera_dev *cam)
{
	struct csi_enc_priv *priv;

	priv = devm_kzalloc(cam->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	cam->csi_enc_priv = priv;

	return 0;
}

void csi_enc_exit(struct mx6_camera_dev *cam)
{
}

