/*
 * Copyright 2013-2014 Robert Bosch Engineering and Business solutions Ltd.,
 *			Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 *
 */

/*!
 * @file adv7182.c
 *
 * @brief Analog Device adv7182 video decoder functions
 *
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/semaphore.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/of_gpio.h>
#include <linux/wait.h>
#include <linux/videodev2.h>
#include <linux/workqueue.h>
#include <linux/regulator/consumer.h>
#include <linux/fsl_devices.h>
#include <media/v4l2-chip-ident.h>
#include <media/v4l2-int-device.h>
#include "mxc_v4l2_capture.h"
#include "adv7182.h"


/*! Description of video formats supported.
 *
 *  PAL: raw=720x625, active=720x576.
 *  NTSC: raw=720x525, active=720x480.
 */
static struct video_fmt_t video_fmts[] = {
	{			/*! NTSC */
	 .v4l2_id = V4L2_STD_NTSC,
	 .name = "NTSC",
	 .raw_width = 720,	/* SENS_FRM_WIDTH */
	 .raw_height = 525,	/* SENS_FRM_HEIGHT */
	 .active_width = 720,	/* ACT_FRM_WIDTH plus 1 */
	 .active_height = 480,	/* ACT_FRM_WIDTH plus 1 */
	 },
	{			/*! (B, G, H, I, N) PAL */
	 .v4l2_id = V4L2_STD_PAL,
	 .name = "PAL",
	 .raw_width = 720,
	 .raw_height = 625,
	 .active_width = 720,
	 .active_height = 576,
	 },
};

/* supported controls */
/* This hasn't been fully implemented yet.
 * This is how it should work, though. */
static struct v4l2_queryctrl adv7182_qctrl[] = {
	{
	.id = V4L2_CID_BRIGHTNESS,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Brightness",
	.minimum = 0,		/* check this value */
	.maximum = 255,		/* check this value */
	.step = 1,		/* check this value */
	.default_value = 0x00,	/* check this value */
	.flags = 0,
	}, {
	.id = V4L2_CID_SATURATION,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Saturation",
	.minimum = 0,		/* check this value */
	.maximum = 255,		/* check this value */
	.step = 0x1,		/* check this value */
	.default_value = 128,	/* check this value */
	.flags = 0,
	}, {
	.id = V4L2_CID_CONTRAST,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Contrast",
	.minimum = 0,
	.maximum = 255,
	.step = 0x1,
	.default_value = 0x80,
	.flags = 0,
	}, {
	.id = V4L2_CID_HUE,
	.type = V4L2_CTRL_TYPE_INTEGER,
	.name = "Hue",
	.minimum = -127,
	.maximum = 128,
	.step = 0x1,
	.default_value = 0,
	.flags = 0,
	}
};

/* Analog Inputs on 32-Lead LFCSP */
static const struct adv7182_inputs_t adv7182_inputs_32[] = {
	{ .insel = 0x00, .desc = "ADV7182 Composite on Ain1" },
	{ .insel = 0x01, .desc = "ADV7182 Composite on Ain2" },
	{ .insel = 0x02, .desc = "ADV7182 Composite on Ain3" },
	{ .insel = 0x03, .desc = "ADV7182 Composite on Ain4" },
	{ .insel = 0x08, .desc = "ADV7182 Y/C on AIN1/2" },
	{ .insel = 0x09, .desc = "ADV7182 Y/C on AIN3/4" },
	{ .insel = 0x0C, .desc = "ADV7182 YPbPr on AIN1/2/3" },
	{ .insel = 0x0E, .desc = "ADV7182 Differential +/- on AIN1/2" },
	{ .insel = 0x0F, .desc = "ADV7182 Differential +/- on AIN3/4" },
};
#define NUM_INPUTS_32 ARRAY_SIZE(adv7182_inputs_32)

/* Default Register value table */
const u8 adv7182_def_reg_val[ADV7182_MAX_REG][2] = {
/* Register , values */
{ADV_REG_ADDR_PWR, ADV_PWR_ON},
{ADV_REG_ADDR_INSEL, ADV_INSEL_DEFAULT},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_RST_CLAMPDEFAULT1},
{ADV_REG_ADI_RST_CLAMP, ADV_ADI_RST_CLAMPDEFAULT2},
{ADV_REG_ADI_RST_CLAMP, ADV_ADI_RST_CLAMPDEFAULT3},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
{ADV_REG_ADDR_ADC_VID_SEL, ADV_ADC_VID_SEL_AUTO},
{ADV_REG_ADDR_ADC_ADI_CTRL2, ADV_ADC_ADI_CTRL2_VAL},
{ADV_REG_ADDR_ADC_OUT_CTRL_EXTD, ADV_ADC_OUT_CTRL_EXTD_DEFAULT},
{ADV_REG_ADDR_ADC_OUT_CTRL, ADV_ADC_OUT_CTRL_DEFAULT},
{ADV_REG_ADDR_LOCK_CNT, ADV_LOCK_CNT_VAL},
{ADV_REG_ADDR_DRIVE_STRENGTH, ADV_DRIVE_STRENGTH_VAL},
{ADV_REG_ADI_IBIAS_AFE, ADV_ADI_IBIAS_AFE_DEFAULT},
{ADV_REG_ADI_RES_CIR, ADV_ADI_RES_CIR_DEFAULT},
{ADV_REG_ADI_CLAMP_ADJ, ADV_ADI_CLAMP_ADJ_DEFAULT},
{ADV_REG_ADI_DIFF_MODE, ADV_ADI_DIFF_MODE_DEFAULT},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_RST_CLAMPDEFAULT1},
{ADV_REG_ADI_AGC_ADJ1, ADV_ADI_AGC_ADJ1_DEFAULT},
{ADV_REG_ADI_AGC_ADJ2, ADV_ADI_AGC_ADJ2_DEFAULT},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
{ADV_REG_ADDR_STATUS1, ADV_STATUS1_DEFAULT},
{ADV_REG_ADDR_VID_SEL1, ADV_VID_SEL1_DEFAULT},
{ADV_REG_ADDR_VID_SEL2, ADV_VID_SEL2_DEFAULT},
{ADV_REG_ADDR_CONTRAST, ADV_CONTRAST_DEFAULT},
{ADV_REG_ADDR_BRIGHTNESS, ADV_BRIGHTNESS_DEFAULT},
{ADV_REG_ADDR_HUE, ADV_HUE_DEFAULT},
{ADV_REG_ADDR_DEF_COLOR_Y, ADV_COLOR_Y_DEFAULT},
{ADV_REG_ADDR_DEF_COLOR_CR_CB, ADV_COLOR_CR_CB_DEFAULT},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
{ADV_REG_ADDR_ANA_CLAMP_CTRL, ADV_ANA_CLAMP_CTRL_DEFAULT},
{ADV_REG_ADDR_DIG_CLAMP_CTRL, ADV_DIG_CLAMP_CTRL_DEFAULT},
{ADV_REG_ADDR_SHAP_FIL_CTRL1, ADV_SHAP_FIL_CTRL1_VAL},
{ADV_REG_ADDR_SHAP_FIL_CTRL2, ADV_SHAP_FIL_CTRL2_DEFAULT},
{ADV_REG_ADDR_COMB_FIL_CTRL, ADV_COMB_FIL_CTRL_DEFAULT},
{ADV_REG_ADDR_PIX_DELAY, ADV_PIX_DELAY_DEFAULT},
{ADV_REG_ADDR_MISC_GAIN_CTRL, ADV_MISC_GAIN_CTRL_DEFAULT},
{ADV_REG_ADDR_AGC_MOD_CTRL, ADV_AGC_MOD_CTRL_DEFAULT},
{ADV_REG_ADDR_CHR_GAIN_CTRL1, ADV_CHR_GAIN_CTRL1_DEFAULT},
{ADV_REG_ADDR_CHR_GAIN_CTRL2, ADV_CHR_GAIN_CTRL2_DEFAULT},
{ADV_REG_ADDR_LUM_GAIN_CTRL1, ADV_LUM_GAIN_CTRL1_DEFAULT},
{ADV_REG_ADDR_LUM_GAIN_CTRL2, ADV_LUM_GAIN_CTRL2_DEFAULT},
{ADV_REG_ADDR_VS_CTRL1, ADV_VS_CTRL1_DEFAULT},
{ADV_REG_ADDR_VS_CTRL2, ADV_VS_CTRL2_DEFAULT},
{ADV_REG_ADDR_VS_CTRL3, ADV_VS_CTRL3_DEFAULT},
{ADV_REG_ADDR_HS_CTRL1, ADV_HS_CTRL1_DEFAULT},
{ADV_REG_ADDR_HS_CTRL2, ADV_HS_CTRL2_DEFAULT},
{ADV_REG_ADDR_HS_CTRL3, ADV_HS_CTRL3_DEFAULT},
{ADV_REG_ADDR_POL, ADV_POL_DEFAULT},
{ADV_REG_ADDR_NTSC_CMB_CTRL, ADV_NTSC_CMB_CTRL_DEFAULT},
{ADV_REG_ADDR_PAL_CMB_CTRL, ADV_PAL_CMB_CTRL_DEFAULT},
{ADV_REG_ADDR_WIN_CTRL, ADV_WIN_CTRL_DEFAULT},
{ADV_REG_ADDR_RESAMPLE_CTRL, ADV_RESAMPLE_CTRL_DEFAULT},
{ADV_REG_ADDR_CTI_DNR_CTRL1, ADV_CTI_DNR_CTRL1_DEFAULT},
{ADV_REG_ADDR_CTI_DNR_CTRL2, ADV_CTI_DNR_CTRL2_DEFAULT},
{ADV_REG_ADDR_DNR_NOISE_THR1, ADV_DNR_NOISE_THR1_DEFAULT},
{ADV_REG_ADDR_OUT_SYNC_SEL1, ADV_OUT_SYNC_SEL1_DEFAULT},
{ADV_REG_ADDR_OUT_SYNC_SEL2, ADV_OUT_SYNC_SEL2_DEFAULT},
{ADV_REG_ADDR_FREE_RUN_LEN1, ADV_FREE_RUN_LEN1_DEFAULT},
{ADV_REG_ADDR_CRC_ENABLE, ADV_CRC_ENABLE_DEFAULT},
{ADV_REG_ADDR_LETBOX1_CTRL, ADV_LETBOX1_CTRL_DEFAULT},
{ADV_REG_ADDR_LETBOX2_CTRL, ADV_LETBOX2_CTRL_DEFAULT},
{ADV_REG_ADDR_SD_CB_CHN_OFFSET, ADV_SD_CB_CHN_OFFSET_DEFAULT},
{ADV_REG_ADDR_SD_CR_CHN_OFFSET, ADV_SD_CR_CHN_OFFSET_DEFAULT},
{ADV_REG_ADDR_SD_CB_CHN_SAT, ADV_SD_CB_CHN_SAT_DEFAULT},
{ADV_REG_ADDR_SD_CR_CHN_SAT, ADV_SD_CR_CHN_SAT_DEFAULT},
{ADV_REG_ADDR_NTSC_VSB, ADV_NTSC_VSB_DEFAULT},
{ADV_REG_ADDR_NTSC_VSE, ADV_NTSC_VSE_VAL},
{ADV_REG_ADDR_NTSC_FLD_TGLE, ADV_NTSC_FLD_TGLE_DEFAULT},
{ADV_REG_ADDR_PAL_VSB, ADV_PAL_VSB_DEFAULT},
{ADV_REG_ADDR_PAL_VSE, ADV_PAL_VSE_VAL},
{ADV_REG_ADDR_PAL_FLD_TGLE, ADV_PAL_FLD_TGLE_DEFAULT},
{ADV_REG_ADDR_VBLK_CTRL1, ADV_VBLK_CTRL1_DEFAULT},
{ADV_REG_ADDR_VBLK_CTRL2, ADV_VBLK_CTRL2_DEFAULT},
{ADV_REG_ADDR_AFE_CTRL1, ADV_AFE_CTRL1_DEFAULT},
{ADV_REG_ADDR_IF_CLMP_CTRL1, ADV_IF_CLMP_CTRL1_DEFAULT},
{ADV_REG_ADDR_VS_MODE_CTRL, ADV_VS_MODE_CTRL_DEFAULT},
{ADV_REG_ADDR_SHARPNESS, ADV_SHARPNESS_DEFAULT},
{ADV_REG_ADDR_DNR_NOISE_THR2, ADV_DNR_NOISE_THR2_DEFAULT},
{ADV_REG_ADDR_CSI_TX_SLAVE, ADV_CSI_TX_SLAVE_DEFAULT},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_USR_SUB_MAP_40},
{ADV_REG_ADI_ACE_CTRL5, ADV_ADI_ACE_CHROMA_GAIN_OFF},
{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT}
};

/***********************************************************************
 * I2C transfer.
 ***********************************************************************/

/*! Read one register from a adv7182 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7182_read(u8 reg)
{
	int ret;
	ret = i2c_smbus_read_byte_data(adv7182_data.i2c_client, reg);
	if (ret < 0) {
		dev_dbg(&adv7182_data.i2c_client->dev,
			"%s:read reg error: reg=%2x\n", __func__, reg);
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182 I2C communication read failed\n");
	}
	return ret;
}

/*! Write one register of a adv7182 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7182_write_reg(u8 reg, u8 val)
{
	int ret;
	ret = i2c_smbus_write_byte_data(adv7182_data.i2c_client, reg, val);
	if (ret < 0) {
		dev_dbg(&adv7182_data.i2c_client->dev,
			"%s:write reg error:reg=%2x,val=%2x\n", __func__,
			reg, val);
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182 I2C communication write failed\n");
	}
	return ret;
}

/* Update lock status */
static void adv7182_update_lock_status(struct adv7182_dev *sensor)
{
	int stat1, int_stat1, int_stat3, int_raw_stat3;

	stat1 = adv7182_read(ADV_REG_STATUS_1);

	/* Switch to interrupt register map */
	adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_INT_REG_MAP);

	int_stat1 = adv7182_read(ADV7182_INT_STATUS_1);
	int_stat3 = adv7182_read(ADV7182_INT_STATUS_3);
	/* clear the interrupts */
	adv7182_write_reg(ADV7182_INT_CLEAR_1, int_stat1);
	adv7182_write_reg(ADV7182_INT_CLEAR_3, int_stat3);

	int_raw_stat3 = adv7182_read(ADV7182_INT_RAW_STATUS_3);

	/* Switch back to normal register map */
	adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT);

	dev_dbg(&sensor->i2c_client->dev,
		"stat1=0x%02x, int_stat 1/3/raw3 = 0x%02x/0x%02x/0x%02x\n",
		stat1, int_stat1, int_stat3, int_raw_stat3);

	if ((int_stat1 & ADV7182_INT_SD_LOCK) ||
	    (int_stat1 & ADV7182_INT_SD_UNLOCK) ||
	    (int_stat3 & ADV7182_INT_SD_V_LOCK_CHNG))
		sensor->lock_status_change = true;

	sensor->locked = ((stat1 & ADV7182_IN_LOCK) &&
			  (stat1 & ADV7182_FSC_LOCK) &&
			  (int_raw_stat3 & ADV7182_INT_SD_V_LOCK));
	dev_dbg(&sensor->i2c_client->dev, "lsc %s: %s\n",
		sensor->lock_status_change ? "true" : "false",
		sensor->locked ? "locked" : "unlocked");
}

/* Read AD_RESULT to get the autodetected video standard */
static void adv7182_get_autodetect_std(struct adv7182_dev *sensor)
{
	int stat1, ad_result, idx = adv7182_PAL;
	v4l2_std_id std = V4L2_STD_PAL;

	/*
	 * When the chip loses lock, it continues to send data at whatever
	 * standard was detected before, so leave the standard at the last
	 * detected standard.
	 */
	if (!sensor->locked) {
		dev_dbg(&sensor->i2c_client->dev, "No signal lock\n");
		return;
	}

	stat1 = adv7182_read(ADV_REG_STATUS_1);
	ad_result = (stat1 & ADV7182_AD_RESULT_MASK) >> ADV7182_AD_RESULT_BIT;

	switch (ad_result) {
	case ADV7182_AD_PAL:
		std = V4L2_STD_PAL;
		idx = adv7182_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL standard\n");
		break;
	case ADV7182_AD_PAL_M:
		std = V4L2_STD_PAL_M;
		/* PAL M is very similar to NTSC (same lines/field) */
		idx = adv7182_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL M standard\n");
		break;
	case ADV7182_AD_PAL_N:
		std = V4L2_STD_PAL_N;
		idx = adv7182_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL N standard\n");
		break;
	case ADV7182_AD_PAL_60:
		std = V4L2_STD_PAL_60;
		/* PAL 60 has same lines as NTSC */
		idx = adv7182_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected PAL 60 standard\n");
		break;
	case ADV7182_AD_NTSC:
		std = V4L2_STD_NTSC;
		idx = adv7182_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected NTSC standard\n");
		break;
	case ADV7182_AD_NTSC_4_43:
		std = V4L2_STD_NTSC_443;
		idx = adv7182_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected NTSC 4.43 standard\n");
		break;
	case ADV7182_AD_SECAM:
		std = V4L2_STD_SECAM;
		idx = adv7182_PAL;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected SECAM standard\n");
		break;
	case ADV7182_AD_SECAM_525:
		/*
		 * FIXME: could not find any info on "SECAM 525", assume
		 * it is SECAM but with NTSC line standard.
		 */
		std = V4L2_STD_SECAM;
		idx = adv7182_NTSC;
		dev_dbg(&sensor->i2c_client->dev,
			"Detected SECAM 525 standard\n");
		break;
	}

	if (std != sensor->std_id) {
		dev_dbg(&sensor->i2c_client->dev,
			"changing std: 0x%08x --> 0x%08x\n",
			(u32)sensor->std_id, (u32)std);
		sensor->video_idx = idx;
		sensor->std_id = std;
		sensor->pix.width = video_fmts[sensor->video_idx].raw_width;
		sensor->pix.height = video_fmts[sensor->video_idx].raw_height;
	}
}

/*!
 * Sets the camera power.
 *
 * sensor_priv -  pointer to the camera device
 * enable if 1, power is to be turned on.  0 means power is to be turned off
 *
 * ioctl_s_power - V4L2 sensor interface handler for vidioc_int_s_power_num
 * @sensor_priv: pointer to sensor device structure
 * @enable: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 */
static int adv7182_power(struct adv7182_dev *sensor_priv, bool enable)
{
	if (enable && !sensor_priv->on) {
		if (gpio_is_valid(sensor_priv->gpio_pwrdn))
			gpio_set_value_cansleep(sensor_priv->gpio_pwrdn, 1);

		usleep_range(5000, 5001);
		if (adv7182_write_reg(ADV_REG_ADDR_PWR, ADV_PWR_ON) != 0)
			return -EIO;
	} else if (!enable && sensor_priv->on) {
		if (adv7182_write_reg(ADV_REG_ADDR_PWR, ADV_PWR_OFF) != 0)
			return -EIO;

		if (gpio_is_valid(sensor_priv->gpio_pwrdn))
			gpio_set_value_cansleep(sensor_priv->gpio_pwrdn, 0);
	}

	sensor_priv->on = enable;
	return ADV7182_SUCCESS;
}

/*
 * Enable the SD_UNLOCK and SD_AD_CHNG interrupts.
 */
static void adv7182_enable_interrupts(struct adv7182_dev *sensor)
{
	mutex_lock(&sensor->mutex);

	/* Switch to interrupt register map */
	adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_INT_REG_MAP);
	/* INTRQ active low, active until cleared */
	adv7182_write_reg(ADV7182_INT_CONFIG_1, ADV_INTR_CONFIG_1);
	/* unmask SD_UNLOCK and SD_LOCK */
	adv7182_write_reg(ADV7182_INT_MASK_1,
			  ADV7182_INT_SD_UNLOCK | ADV7182_INT_SD_LOCK);
	/* unmask SD_AD_CHNG and SD_V_LOCK_CHNG */
	adv7182_write_reg(ADV7182_INT_MASK_3,
			  ADV7182_INT_SD_AD_CHNG | ADV7182_INT_SD_V_LOCK_CHNG);
	/* Switch back to normal register map */
	adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT);

	mutex_unlock(&sensor->mutex);
}

/* threaded irq handler */
static irqreturn_t adv7182_interrupt(int irq, void *dev_id)
{
	struct adv7182_dev *sensor = dev_id;

	mutex_lock(&sensor->mutex);

	adv7182_update_lock_status(sensor);
	adv7182_get_autodetect_std(sensor);

	mutex_unlock(&sensor->mutex);

	return IRQ_HANDLED;
}

/*!
 * adv7182_gpio_initialise - initialize the GPIO pin with direction set.
 *
 * @param gpio_pin: points to GPIO pin number
 * @param direction: points to direction(IN or OUT)
 * @param high_low: high-1,low-0 is used if direction is set as "OUT".
 * @param gpio_name: Has GPIO name for the provided pin.
 *
 * @return		0 if success, an error code otherwise.
 *
 */

static s32 adv7182_gpio_initialise(u32 gpio_pin, u32 direction,
u32 high_low, const char *gpio_name)
{
	s32 ret;

	/*ckeck for valid GPIO pin*/
	if (!gpio_is_valid(gpio_pin)) {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182: **GPIO invalid pin**\n");
		return ADV7182_GPIO_ERR_INVALID_PIN;
	}

	/*Allocates memory for requested pin*/
	if (gpio_request(gpio_pin, gpio_name)) {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182: **GPIO request is failed**\n");
		return ADV7182_GPIO_ERR_REQ;
	}
	switch (direction) {
	case ADV7182_GPIO_DIR_OUT:
	{
		/* set direction as output for the gpio pin and set
		   the value which will be initialised */
		if (gpio_direction_output(gpio_pin, high_low)
						== ADV7182_SUCCESS) {
			ret = ADV7182_SUCCESS;
		} else {
			dev_err(&adv7182_data.i2c_client->dev,
				"ADV7182: gpio Dir set OUT failed\n");
			gpio_free(gpio_pin);
			ret = ADV7182_GPIO_ERR_DIR_OUT;
		}
	}
	break;
	case ADV7182_GPIO_DIR_IN:
	{
		/*set the direction as input for the gpio pin*/
		if (gpio_direction_input(gpio_pin) == ADV7182_SUCCESS) {
			ret = ADV7182_SUCCESS;
		} else {
			dev_err(&adv7182_data.i2c_client->dev,
				"ADV7182: gpio Dir set IN failed\n");
			gpio_free(gpio_pin);
			ret = ADV7182_GPIO_ERR_DIR_IN;
		}
	}
	break;
	default:
	{
		/*deallocate the memory for the gpio pin*/
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:GPIO dir is not known**\n");
		gpio_free(gpio_pin);
		ret = ADV7182_GPIO_ERR_INVALID_DIR;
	}
	break;
	}

	return ret;
}

/*! adv7182 ACE feature enable/disable function.
 *
 *  @param enable	enable/disable - 1/0 ACE feature.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7182_ace(u8 enable)
{
	int ret = 0;
	dev_dbg(&adv7182_data.i2c_client->dev,
		"In adv7182:adv7182_ace:enable:%x\n", enable);

	if (adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_USR_SUB_MAP_40) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADI_ENA_ACE,
			      enable) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_CTRL1_DEFAULT) != 0) {
		ret = -EIO;
		goto error;
	}
error:
	return ret;
}

/***********************************************************************
 * mxc_v4l2_capture interface.
 ***********************************************************************/


/*!
 * ioctl_querystd - V4L2 sensor interface handler for vidioc_int_querystd_num.
 * Return current video standard. This device autodetects the current
 * standard, so this function also sets the values that need to be changed
 * if the standard changes. There is no set std equivalent function.
 */
static int ioctl_querystd(struct v4l2_int_device *s, v4l2_std_id *std)
{
	struct adv7182_dev *sensor = s->priv;

	mutex_lock(&sensor->mutex);

	/*
	 * If we have the ADV7182 irq, we can just return the currently
	 * detected standard. Otherwise we have to poll the AD_RESULT
	 * bits every time ioctl_querystd() is called.
	 */
	if (!sensor->i2c_client->irq) {
		adv7182_update_lock_status(sensor);
		adv7182_get_autodetect_std(sensor);
	}

	*std = sensor->std_id;

	mutex_unlock(&sensor->mutex);

	return ADV7182_SUCCESS;
}


/***********************************************************************
 * IOCTL Functions from v4l2_int_ioctl_desc.
 ***********************************************************************/

/*!
 * ioctl_g_ifparm - V4L2 sensor interface handler for vidioc_int_g_ifparm_num
 * s: pointer to standard V4L2 device structure
 * p: pointer to standard V4L2 vidioc_int_g_ifparm_num ioctl structure
 *
 * Gets slave interface parameters.
 * Calculates the required xclk value to support the requested
 * clock parameters in p.  This value is returned in the p
 * parameter.
 *
 * vidioc_int_g_ifparm returns platform-specific information about the
 * interface settings used by the sensor.
 *
 * Called on open.
 */
static int ioctl_g_ifparm(struct v4l2_int_device *s, struct v4l2_ifparm *p)
{
	struct adv7182_dev *sensor_priv = s->priv;
	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_g_ifparm\n");

	if (s == NULL) {
		dev_err(&sensor_priv->i2c_client->dev,
			"ERROR!! no slave device set!\n");
		return -ENODEV;
	}

	/* Initialize structure to 0s then set any non-0 values. */
	memset(p, 0, sizeof(*p));
	p->if_type = V4L2_IF_TYPE_BT656; /* This is the only possibility. */
	p->u.bt656.mode = V4L2_IF_TYPE_BT656_MODE_BT_8BIT;
	p->u.bt656.bt_sync_correct = 1; /* use bt656 sync codes */
	p->u.bt656.clock_curr = 0;

	/* adv7182 has a dedicated clock so no clock settings needed. */

	return ADV7182_SUCCESS;
}

/*!
 * Sets the camera power.
 *
 * s  pointer to the camera device
 * on if 1, power is to be turned on.  0 means power is to be turned off
 *
 * ioctl_s_power - V4L2 sensor interface handler for vidioc_int_s_power_num
 * @s: pointer to standard V4L2 device structure
 * @on: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 * This is called on open, close, suspend and resume.
 */
static int ioctl_s_power(struct v4l2_int_device *s, int on)
{
	struct adv7182_dev *sensor_priv = s->priv;
	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_s_power\n");

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int ioctl_g_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	struct adv7182_dev *sensor_priv = s->priv;
	struct v4l2_captureparm *cparm = &a->parm.capture;

	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_g_parm\n");

	switch (a->type) {
	/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"type is V4L2_BUF_TYPE_VIDEO_CAPTURE\n");
		memset(a, 0, sizeof(*a));
		a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		cparm->capability = sensor_priv->streamcap.capability;
		cparm->timeperframe = sensor_priv->streamcap.timeperframe;
		cparm->capturemode = sensor_priv->streamcap.capturemode;
		break;

	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;

	default:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"ioctl_g_parm:type is unknown %d\n", a->type);
		break;
	}

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 *
 * This driver cannot change these settings.
 */
static int ioctl_s_parm(struct v4l2_int_device *s, struct v4l2_streamparm *a)
{
	struct adv7182_dev *sensor_priv = s->priv;
	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_s_parm\n");

	switch (a->type) {
	/* These are all the possible cases. */
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	case V4L2_BUF_TYPE_VIDEO_OUTPUT:
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	case V4L2_BUF_TYPE_VBI_CAPTURE:
	case V4L2_BUF_TYPE_VBI_OUTPUT:
	case V4L2_BUF_TYPE_SLICED_VBI_CAPTURE:
	case V4L2_BUF_TYPE_SLICED_VBI_OUTPUT:
		break;

	default:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"type is unknown - %d\n", a->type);
		break;
	}

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_g_fmt_cap - V4L2 sensor interface handler for ioctl_g_fmt_cap
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 v4l2_format structure
 *
 * Returns the sensor's current pixel format in the v4l2_format
 * parameter.
 */
static int ioctl_g_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct adv7182_dev *sensor_priv = s->priv;

	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_g_fmt_cap\n");

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"Returning size of %dx%d\n",
			sensor_priv->pix.width, sensor_priv->pix.height);
		f->fmt.pix = sensor_priv->pix;
		break;
	default:
		f->fmt.pix = sensor_priv->pix;
		break;
	}

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_try_fmt_cap - V4L2 sensor interface handler for VIDIOC_TRY_FMT
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 format description structure
 *
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int ioctl_try_fmt_cap(struct v4l2_int_device *s,
			     struct v4l2_format *f)
{
	struct adv7182_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return ADV7182_SUCCESS;
}

/*!
 * ioctl_s_fmt_cap - V4L2 sensor interface handler for VIDIOC_S_FMT
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 format description structure
 *
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int ioctl_s_fmt_cap(struct v4l2_int_device *s, struct v4l2_format *f)
{
	struct adv7182_dev *sensor = s->priv;

	f->fmt.pix = sensor->pix;
	return ADV7182_SUCCESS;
}

/*!
 * ioctl_queryctrl - V4L2 sensor interface handler for VIDIOC_QUERYCTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @qc: standard V4L2 VIDIOC_QUERYCTRL ioctl structure
 *
 * If the requested control is supported, returns the control information
 * from the video_control[] array.  Otherwise, returns -EINVAL if the
 * control is not supported.
 */
static int ioctl_queryctrl(struct v4l2_int_device *s,
			   struct v4l2_queryctrl *qc)
{
	int i;
	struct adv7182_dev *sensor_priv = s->priv;

	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_queryctrl\n");

	for (i = 0; i < ARRAY_SIZE(adv7182_qctrl); i++)
		if (qc->id && qc->id == adv7182_qctrl[i].id) {
			memcpy(qc, &(adv7182_qctrl[i]),
			       sizeof(*qc));
			return ADV7182_SUCCESS;
		}

	return -EINVAL;
}

/*!
 * ioctl_g_ctrl - V4L2 sensor interface handler for VIDIOC_G_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_G_CTRL ioctl structure
 *
 * If the requested control is supported, returns the control's current
 * value from the video_control[] array.  Otherwise, returns -EINVAL
 * if the control is not supported.
 */
static int ioctl_g_ctrl(struct v4l2_int_device *s, struct v4l2_control *vc)
{
	int ret = 0;
	struct adv7182_dev *sensor_priv = s->priv;

	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_g_ctrl\n");

	switch (vc->id) {
	case V4L2_CID_BRIGHTNESS:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_BRIGHTNESS\n");
		sensor_priv->brightness = adv7182_read(ADV_REG_ADDR_BRIGHTNESS);
		vc->value = sensor_priv->brightness;
		break;
	case V4L2_CID_CONTRAST:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_CONTRAST\n");
		sensor_priv->contrast = adv7182_read(ADV_REG_ADDR_CONTRAST);
		vc->value = sensor_priv->contrast;
		break;
	case V4L2_CID_SATURATION:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_SATURATION\n");
		sensor_priv->saturation =
			adv7182_read(ADV_REG_ADDR_SD_CB_CHN_SAT);
		vc->value = sensor_priv->saturation;
		break;
	case V4L2_CID_HUE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_HUE\n");
		vc->value = sensor_priv->hue;
		break;
	case V4L2_CID_AUTO_WHITE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_AUTO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_DO_WHITE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_DO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_RED_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_RED_BALANCE\n");
		vc->value = sensor_priv->red;
		break;
	case V4L2_CID_BLUE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_BLUE_BALANCE\n");
		vc->value = sensor_priv->blue;
		break;
	case V4L2_CID_GAMMA:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_GAMMA\n");
		break;
	case V4L2_CID_EXPOSURE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_EXPOSURE\n");
		vc->value = sensor_priv->ae_mode;
		break;
	case V4L2_CID_AUTOGAIN:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_AUTOGAIN\n");
		break;
	case V4L2_CID_GAIN:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_GAIN\n");
		break;
	case V4L2_CID_HFLIP:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_HFLIP\n");
		break;
	case V4L2_CID_VFLIP:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_VFLIP\n");
		break;
	default:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   Default case\n");
		vc->value = 0;
		ret = -EPERM;
		break;
	}

	return ret;
}

/*!
 * ioctl_s_ctrl - V4L2 sensor interface handler for VIDIOC_S_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_S_CTRL ioctl structure
 *
 * If the requested control is supported, sets the control's current
 * value in HW (and updates the video_control[] array).  Otherwise,
 * returns -EINVAL if the control is not supported.
 */
static int ioctl_s_ctrl(struct v4l2_int_device *s, struct v4l2_control *vc)
{
	int retval = 0;
	u8 tmp;
	struct adv7182_dev *sensor_priv = s->priv;

	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_s_ctrl\n");

	switch (vc->id) {
	case V4L2_CID_BRIGHTNESS:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_BRIGHTNESS\n");
		tmp = vc->value;
		adv7182_write_reg(ADV_REG_ADDR_BRIGHTNESS, tmp);
		sensor_priv->brightness = vc->value;
		break;
	case V4L2_CID_CONTRAST:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_CONTRAST\n");
		tmp = vc->value;
		adv7182_write_reg(ADV_REG_ADDR_CONTRAST, tmp);
		sensor_priv->contrast = vc->value;
		break;
	case V4L2_CID_SATURATION:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_SATURATION\n");
		tmp = vc->value;
		adv7182_write_reg(ADV_REG_ADDR_SD_CB_CHN_SAT, tmp);
		adv7182_write_reg(ADV_REG_ADDR_SD_CR_CHN_SAT, tmp);
		sensor_priv->saturation = vc->value;
		break;
	case V4L2_CID_HUE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_HUE\n");
		tmp = vc->value;
		/* Hue is inverted according to HSL chart */
		adv7182_write_reg(ADV_REG_ADDR_HUE, -tmp);
		sensor_priv->hue = vc->value;
		break;
	case V4L2_CID_AUTO_WHITE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_AUTO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_DO_WHITE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_DO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_RED_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_RED_BALANCE\n");
		break;
	case V4L2_CID_BLUE_BALANCE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_BLUE_BALANCE\n");
		break;
	case V4L2_CID_GAMMA:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_GAMMA\n");
		break;
	case V4L2_CID_EXPOSURE:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_EXPOSURE\n");
		break;
	case V4L2_CID_AUTOGAIN:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_AUTOGAIN\n");
		break;
	case V4L2_CID_GAIN:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_GAIN\n");
		break;
	case V4L2_CID_HFLIP:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_HFLIP\n");
		break;
	case V4L2_CID_VFLIP:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   V4L2_CID_VFLIP\n");
		break;
	default:
		dev_dbg(&sensor_priv->i2c_client->dev,
			"   Default case\n");
		retval = -EPERM;
		break;
	}

	return retval;
}

/*!
 * ioctl_enum_framesizes - V4L2 sensor interface handler for
 *			   VIDIOC_ENUM_FRAMESIZES ioctl
 * @s: pointer to standard V4L2 device structure
 * @fsize: standard V4L2 VIDIOC_ENUM_FRAMESIZES ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_framesizes(struct v4l2_int_device *s,
				 struct v4l2_frmsizeenum *fsize)
{
	struct adv7182_dev *sensor_priv = s->priv;

	if (fsize->index > 0)
		return -EINVAL;

	fsize->pixel_format = sensor_priv->pix.pixelformat;
	fsize->discrete.width =
			video_fmts[sensor_priv->video_idx].active_width;
	fsize->discrete.height =
			video_fmts[sensor_priv->video_idx].active_height;
	dev_dbg(&sensor_priv->i2c_client->dev, "Framesize %d %d\n",
		fsize->discrete.width, fsize->discrete.height);
	return ADV7182_SUCCESS;
}

/*!
 * ioctl_enum_input - V4L2 sensor interface handler for VIDIOC_ENUMINPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @input: standard V4L2 VIDIOC_ENUMINPUT ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_enum_input(struct v4l2_int_device *s,
			    struct v4l2_input *input)
{
	struct adv7182_dev *sensor_priv = s->priv;
	const struct adv7182_inputs_t *advinput;

	if (input->index >= NUM_INPUTS_32)
		return -EINVAL;

	advinput = &adv7182_inputs_32[input->index];

	input->type = V4L2_INPUT_TYPE_CAMERA;
	input->std = V4L2_STD_NTSC | V4L2_STD_PAL | V4L2_STD_SECAM;
	input->status = 0;

	mutex_lock(&sensor_priv->mutex);

	if (input->index == sensor_priv->current_input) {
		if (sensor_priv->on) {
			if (!sensor_priv->locked)
				input->status = V4L2_IN_ST_NO_SIGNAL |
						V4L2_IN_ST_NO_SYNC;
		} else {
			input->status = V4L2_IN_ST_NO_POWER;
		}
	}

	mutex_unlock(&sensor_priv->mutex);

	input->capabilities = 0;
	strcpy(input->name, advinput->desc);

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_g_signal_lsc - V4L2 sensor interface handler for
 * vidioc_int_g_signal_lsc ioctl
 * @s: pointer to standard V4L2 device structure
 *
 * Returns -EIO if locked, else 0.
 */
static int ioctl_g_signal_lsc(struct v4l2_int_device *s)
{
	struct adv7182_dev *sensor = s->priv;
	int ret = 0;

	mutex_lock(&sensor->mutex);

	if (sensor->lock_status_change) {
		sensor->lock_status_change = false;
		ret = -EIO;
	}

	mutex_unlock(&sensor->mutex);

	return ret;
}

/*!
 * ioctl_g_input - V4L2 sensor interface handler for VIDIOC_G_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to return index number
 *
 * Returns 0.
 */
static int ioctl_g_input(struct v4l2_int_device *s, int *index)
{
	struct adv7182_dev *sensor_priv = s->priv;

	*index = sensor_priv->current_input;
	return ADV7182_SUCCESS;
}

/*!
 * ioctl_s_input - V4L2 sensor interface handler for VIDIOC_S_INPUT ioctl
 * @s: pointer to standard V4L2 device structure
 * @index: pointer to index number to set
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int ioctl_s_input(struct v4l2_int_device *s, int *index)
{
	struct adv7182_dev *sensor_priv = s->priv;
	int ret = 0;
	const struct adv7182_inputs_t *advinput;

	if (*index >= NUM_INPUTS_32)
		return -EINVAL;

	advinput = &adv7182_inputs_32[*index];

	mutex_lock(&sensor_priv->mutex);
	if (adv7182_write_reg(ADV_REG_ADDR_INSEL,
			      advinput->insel) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_RST_CLAMPDEFAULT1) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADI_RST_CLAMP,
			      ADV_ADI_RST_CLAMPDEFAULT2) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADI_RST_CLAMP,
			      ADV_ADI_RST_CLAMPDEFAULT3) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_CTRL1_DEFAULT) != 0) {
		ret = -EIO;
		goto error;
	}
	ret = adv7182_ace(adv7182_data.chn_ace[*index]);
	if (ret)
		goto error;

	sensor_priv->current_input = *index;
error:
	mutex_unlock(&sensor_priv->mutex);

	return ret;
}

/*!
 * ioctl_g_chip_ident - V4L2 sensor interface handler for
 *			VIDIOC_DBG_G_CHIP_IDENT ioctl
 * @s: pointer to standard V4L2 device structure
 * @id: pointer to int
 *
 * Return 0.
 */
static int ioctl_g_chip_ident(struct v4l2_int_device *s, int *id)
{
	((struct v4l2_dbg_chip_ident *)id)->match.type =
					V4L2_CHIP_MATCH_I2C_DRIVER;
	strcpy(((struct v4l2_dbg_chip_ident *)id)->match.name,
	       "adv7182_decoder");
	((struct v4l2_dbg_chip_ident *)id)->ident = V4L2_IDENT_ADV7182;

	return ADV7182_SUCCESS;
}

/*!
 * ioctl_init - V4L2 sensor interface handler for VIDIOC_INT_INIT
 * @s: pointer to standard V4L2 device structure
 */
static int ioctl_init(struct v4l2_int_device *s)
{
	struct adv7182_dev *sensor_priv = s->priv;
	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_init\n");
	return ADV7182_SUCCESS;
}

/*!
 * ioctl_enum_fmt_cap - V4L2 sensor interface handler for VIDIOC_ENUM_FMT
 * @s: pointer to standard V4L2 device structure
 * @fmt: pointer to standard V4L2 fmt description structure
 *
 * Return 0.
 */
static int ioctl_enum_fmt_cap(struct v4l2_int_device *s,
			      struct v4l2_fmtdesc *fmt)
{
	struct adv7182_dev *sensor_priv = s->priv;
	if (fmt->index > 0)
		return -EINVAL;

	fmt->pixelformat = sensor_priv->pix.pixelformat;
	return ADV7182_SUCCESS;
}


/*!
 * ioctl_dev_init - V4L2 sensor interface handler for vidioc_int_dev_init_num
 * @s: pointer to standard V4L2 device structure
 *
 * Initialise the device when slave attaches to the master.
 */
static int ioctl_dev_init(struct v4l2_int_device *s)
{
	struct adv7182_dev *sensor_priv = s->priv;
	dev_dbg(&sensor_priv->i2c_client->dev, "In adv7182:ioctl_dev_init\n");
	return ADV7182_SUCCESS;
}

static int ioctl_g_dev(struct v4l2_int_device *s, struct device **dev)
{
	struct adv7182_dev *sensor_priv = s->priv;

	*dev = &sensor_priv->i2c_client->dev;
	return 0;
}

/*!
 * This structure defines all the ioctls for this module.
 */
static struct v4l2_int_ioctl_desc adv7182_ioctl_desc[] = {
	{vidioc_int_dev_init_num,
	 (v4l2_int_ioctl_func *)ioctl_dev_init},
	{vidioc_int_g_dev_num,
	 (v4l2_int_ioctl_func *)ioctl_g_dev},
	{vidioc_int_s_power_num,
	 (v4l2_int_ioctl_func *)ioctl_s_power},
	{vidioc_int_g_ifparm_num,
	 (v4l2_int_ioctl_func *)ioctl_g_ifparm},
	{vidioc_int_init_num,
	 (v4l2_int_ioctl_func *)ioctl_init},
	{vidioc_int_enum_fmt_cap_num,
	 (v4l2_int_ioctl_func *)ioctl_enum_fmt_cap},
	{vidioc_int_try_fmt_cap_num,
	 (v4l2_int_ioctl_func *)ioctl_try_fmt_cap},
	{vidioc_int_g_fmt_cap_num,
	 (v4l2_int_ioctl_func *)ioctl_g_fmt_cap},
	{vidioc_int_s_fmt_cap_num,
	 (v4l2_int_ioctl_func *)ioctl_s_fmt_cap},
	{vidioc_int_g_parm_num,
	 (v4l2_int_ioctl_func *)ioctl_g_parm},
	{vidioc_int_s_parm_num,
	 (v4l2_int_ioctl_func *)ioctl_s_parm},
	{vidioc_int_queryctrl_num,
	 (v4l2_int_ioctl_func *)ioctl_queryctrl},
	{vidioc_int_g_ctrl_num,
	 (v4l2_int_ioctl_func *)ioctl_g_ctrl},
	{vidioc_int_s_ctrl_num,
	 (v4l2_int_ioctl_func *)ioctl_s_ctrl},
	{vidioc_int_enum_framesizes_num,
	 (v4l2_int_ioctl_func *)ioctl_enum_framesizes},
	{vidioc_int_enum_input_num,
	 (v4l2_int_ioctl_func *)ioctl_enum_input},
	{vidioc_int_g_input_num,
	 (v4l2_int_ioctl_func *)ioctl_g_input},
	{vidioc_int_s_input_num,
	 (v4l2_int_ioctl_func *)ioctl_s_input},
	{vidioc_int_g_chip_ident_num,
	 (v4l2_int_ioctl_func *)ioctl_g_chip_ident},
	{vidioc_int_querystd_num,
	 (v4l2_int_ioctl_func *)ioctl_querystd},
	{vidioc_int_g_signal_lsc_num,
	 (v4l2_int_ioctl_func *)ioctl_g_signal_lsc},
};

static struct v4l2_int_slave adv7182_slave = {
	.ioctls = adv7182_ioctl_desc,
	.num_ioctls = ARRAY_SIZE(adv7182_ioctl_desc),
};

static struct v4l2_int_device adv7182_int_device = {
	.module = THIS_MODULE,
	.name = "adv7182",
	.type = v4l2_int_type_slave,
	.u = {
		.slave = &adv7182_slave,
	},
};


/***********************************************************************
 * I2C client and driver.
 ***********************************************************************/

/*! adv7182 Reset function.
 *
 *  @return		None.
 */
static int adv7182_hard_reset(void)
{
	int count = 0;
	dev_dbg(&adv7182_data.i2c_client->dev,
		"In adv7182:adv7182_hard_reset\n");

	while (count < ADV7182_MAX_REG) {
		if (adv7182_write_reg(adv7182_def_reg_val[count][0],
				      adv7182_def_reg_val[count][1]) != 0)
			return -EIO;
		count++;
	}
	adv7182_data.current_input = ADV7182_INPUT_INDEX_DEF;

	return adv7182_ace(adv7182_data.chn_ace[ADV7182_INPUT_INDEX_DEF]);
}

static int setup_irq_adv7182(void)
{
	int ret = ADV7182_SUCCESS;
	if (adv7182_data.i2c_client->irq) {
		ret = request_threaded_irq(adv7182_data.i2c_client->irq,
					   NULL, adv7182_interrupt,
					   IRQF_TRIGGER_LOW | IRQF_ONESHOT,
					   IF_NAME, &adv7182_data);
		if (ret < 0) {
			dev_err(&adv7182_data.i2c_client->dev,
				"Failed to register irq %d\n",
				adv7182_data.i2c_client->irq);
			return ret;
		}

		adv7182_enable_interrupts(&adv7182_data);

		dev_info(&adv7182_data.i2c_client->dev,
			 "Registered irq %d\n",
			 adv7182_data.i2c_client->irq);
	}
	return ret;
}
/*!
 * adv7182 I2C probe function.
 * Function set in i2c_driver struct.
 * Called by insmod.
 *
 *  @param *client	I2C client descriptor.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7182_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct v4l2_of_endpoint bus_cfg;
	struct device_node *endpoint;
	int ret = 0;
	const char *norm = "pal";
	s32 gpio_rst;
	char *rst_gpio_name = ADV7182_GPIO_RST_NAME;
	char *pwrdn_gpio_name = ADV7182_GPIO_PWR_DN_NAME;

	dev_dbg(&client->dev, "In adv7182_probe\n");

	/* Set initial values for the sensor struct. */
	memset(&adv7182_data, 0, sizeof(adv7182_data));

	ret = of_property_read_string(client->dev.of_node, "default-std",
			&norm);
	if (ret < 0 && ret != -EINVAL) {
		dev_err(&client->dev, "error reading default-std property!\n");
		return ret;
	}
	if (!strcasecmp(norm, "pal")) {
		adv7182_data.std_id = V4L2_STD_PAL;
		adv7182_data.video_idx = adv7182_PAL;
		dev_info(&client->dev, "defaulting to PAL!\n");
	} else if (!strcasecmp(norm, "ntsc")) {
		adv7182_data.std_id = V4L2_STD_NTSC;
		adv7182_data.video_idx = adv7182_NTSC;
		dev_info(&client->dev, "defaulting to NTSC!\n");
	} else {
		dev_err(&client->dev, "invalid default-std value: '%s'!\n",
			norm);
		return -EINVAL;
	}

	adv7182_data.i2c_client = client;
	adv7182_data.streamcap.timeperframe.denominator = 30;
	adv7182_data.streamcap.timeperframe.numerator = 1;
	adv7182_data.pix.width = video_fmts[adv7182_data.video_idx].raw_width;
	adv7182_data.pix.height = video_fmts[adv7182_data.video_idx].raw_height;
	adv7182_data.pix.pixelformat = V4L2_MBUS_FMT_UYVY8_2X8;
	adv7182_data.pix.field = V4L2_FIELD_INTERLACED;
	adv7182_data.chn_ace[7] = ADV_ADI_ENA_ACE;
	adv7182_data.chn_ace[8] = ADV_ADI_ENA_ACE;

	endpoint = v4l2_of_get_next_endpoint(client->dev.of_node, NULL);
	if (!endpoint) {
		dev_err(&client->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	v4l2_of_parse_endpoint(endpoint, &bus_cfg);
	if (bus_cfg.bus_type != V4L2_MBUS_BT656) {
		dev_err(&client->dev, "invalid bus type, must be bt.656\n");
		return -EINVAL;
	}
	of_node_put(endpoint);

	adv7182_data.bus_cfg = bus_cfg.bus.parallel;

	dev_dbg(&adv7182_data.i2c_client->dev,
		"%s:adv7182 probe i2c address is 0x%02X\n",
		__func__, adv7182_data.i2c_client->addr);
	gpio_rst = of_get_named_gpio(client->dev.of_node, "rst-gpios", 0);
	adv7182_data.gpio_pwrdn = of_get_named_gpio(client->dev.of_node,
						    "pwr-gpios", 0);
	if (of_find_property(client->dev.of_node, "CHN7_ACE_DIS", NULL))
		adv7182_data.chn_ace[7] = ADV_ADI_DIS_ACE;
	if (of_find_property(client->dev.of_node, "CHN8_ACE_DIS", NULL))
		adv7182_data.chn_ace[8] = ADV_ADI_DIS_ACE;
	if (gpio_rst >= 0) {
		ret = adv7182_gpio_initialise(
						gpio_rst,
						(u32)ADV7182_GPIO_DIR_OUT,
						(u32)ADV7182_RESET_ACTIVE,
						(const char *)rst_gpio_name);
		if (ret != ADV7182_SUCCESS) {
			dev_err(&adv7182_data.i2c_client->dev,
				"ADV7182:reset pin initialization fail\n");
			goto err;
		}
	} else {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:GPIO reset pin property not exist\n");
		ret = gpio_rst;
		goto err;
	}
	if (adv7182_data.gpio_pwrdn >= 0) {
		ret = adv7182_gpio_initialise(
					adv7182_data.gpio_pwrdn,
					(u32)ADV7182_GPIO_DIR_OUT,
					(u32)ADV7182_PWRDN_ACTIVE,
					(const char *)pwrdn_gpio_name);
		if (ret != ADV7182_SUCCESS) {
			gpio_free(gpio_rst);
			dev_err(&adv7182_data.i2c_client->dev,
				"ADV7182:powerdown pin initialization fail\n");
			goto err;
		}
		/*Reset the device ADV7182*/
		gpio_set_value(gpio_rst, ADV7182_RESET_ACTIVE);
		gpio_set_value(adv7182_data.gpio_pwrdn,	ADV7182_PWRDN_ACTIVE);
		usleep_range(5000, 10000);
		gpio_set_value(gpio_rst, ADV7182_RESET_DEACTIVE);
		usleep_range(5000, 10000);
		gpio_set_value(adv7182_data.gpio_pwrdn,	ADV7182_PWRDN_DEACTIVE);
		gpio_free(gpio_rst);
	} else {
		gpio_free(gpio_rst);
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:GPIO powerdown pin property not exist\n");
		ret = adv7182_data.gpio_pwrdn;
		goto err;
	}
	/* Power on the adv7182 chip */
	if ((adv7182_power(&adv7182_data, true) != ADV7182_SUCCESS)) {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:Power ON failed\n");
		ret = -EIO;
		goto cleanup;
	}
	/*! Read the revision ID of the ADV7182 chip */
	adv7182_data.rev_id = adv7182_read(ADV_REG_IDENT);
	if (adv7182_data.rev_id < 0) {
		ret = -ENODEV;
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182: Revision ID read failed\n");
		goto cleanup;
	}
	dev_dbg(&adv7182_data.i2c_client->dev,
		"%s:Analog Device adv7%2X0 detected!\n", __func__,
		adv7182_data.rev_id);

	/*! adv7182 initialization. */
	ret = adv7182_hard_reset();
	if (ret != ADV7182_SUCCESS) {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:chip initialization failed\n");
		goto cleanup;
	}
	mutex_init(&adv7182_data.mutex);
	adv7182_data.on = true;
	dev_dbg(&adv7182_data.i2c_client->dev, "type is %d (expect %d)\n",
		adv7182_int_device.type,
		v4l2_int_type_slave);
	dev_dbg(&adv7182_data.i2c_client->dev, "   num ioctls is %d\n",
		adv7182_int_device.u.slave->num_ioctls);
	/* This function attaches this structure
	 *  to the /dev/video0 device.*/
	adv7182_int_device.priv = &adv7182_data;
	ret = v4l2_int_device_register(&adv7182_int_device);
	if (ret != ADV7182_SUCCESS) {
		dev_err(&adv7182_data.i2c_client->dev,
			"ADV7182:v4l2 registration failed\n");
		goto cleanup;
	}
	/* see if there is a signal lock already */
	adv7182_update_lock_status(&adv7182_data);
	adv7182_get_autodetect_std(&adv7182_data);
	ret = setup_irq_adv7182();
	if (!ret)
		return ret;

	v4l2_int_device_unregister(&adv7182_int_device);
cleanup:
	gpio_free(adv7182_data.gpio_pwrdn);

err:
	return ret;
}

/*!
 * adv7182 I2C detach function.
 * Called on rmmod.
 *
 *  @param *client	struct i2c_client*.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7182_detach(struct i2c_client *client)
{
	struct adv7182_dev *sensor_priv = &adv7182_data;
	dev_dbg(&sensor_priv->i2c_client->dev,
		"%s:Removing %s video decoder @ 0x%02X from adapter %s\n",
		__func__, IF_NAME, client->addr << 1, client->adapter->name);

	if (sensor_priv->i2c_client->irq)
		free_irq(sensor_priv->i2c_client->irq, sensor_priv);

	/* Power off the adv7182 chip */
	if ((adv7182_power(sensor_priv, false) != ADV7182_SUCCESS))
		dev_err(&sensor_priv->i2c_client->dev,
			"ADV7182:Power OFF failed\n");

	if (gpio_is_valid(sensor_priv->gpio_pwrdn))
		gpio_free(sensor_priv->gpio_pwrdn);

	v4l2_int_device_unregister(&adv7182_int_device);

	return 0;
}

static const struct i2c_device_id adv7182_id[] = {
	{ "adv7182", 0 },
	{}
};
MODULE_DEVICE_TABLE(i2c, adv7182_id);

static struct of_device_id adv7182_dt_ids[] = {
	{ .compatible = "adi,adv7182" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, adv7182_dt_ids);

static struct i2c_driver adv7182_driver = {
	.driver = {
		.name	= "adv7182",
		.owner	= THIS_MODULE,
		.of_match_table	= adv7182_dt_ids,
	},
	.id_table	= adv7182_id,
	.probe		= adv7182_probe,
	.remove		= adv7182_detach,
};

module_i2c_driver(adv7182_driver);

MODULE_AUTHOR("Robert Bosch Engineering and Business solutions Ltd");
MODULE_DESCRIPTION("Analog Device adv7182 video decoder driver");
MODULE_LICENSE("GPL");
