/*
 * V4L2 Capture Encoder Subdev for Freescale i.MX6 SOC
 *
 * Copyright (c) 2012-2014 Mentor Graphics Inc.
 * Copyright 2004-2012 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/pinctrl/consumer.h>
#include <media/v4l2-device.h>
#include <media/v4l2-ioctl.h>
#include <media/videobuf2-dma-contig.h>
#include <media/v4l2-subdev.h>
#include <media/v4l2-of.h>
#include <media/v4l2-ctrls.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <media/imx6.h>
#include "mx6-camif.h"

struct encoder_priv {
	struct mx6cam_dev    *dev;
	struct v4l2_subdev    sd;

	struct ipuv3_channel *enc_ch;
	struct ipuv3_channel *enc_rot_in_ch;
	struct ipuv3_channel *enc_rot_out_ch;
	struct ipu_ic *ic_enc;
	struct ipu_irt *irt;
	struct ipu_smfc *smfc;
	struct ipu_csi *csi;

	/* active (undergoing DMA) buffers, one for each IPU buffer */
	struct mx6cam_buffer *active_frame[2];

	struct mx6cam_dma_buf rot_buf[2];
	struct mx6cam_dma_buf underrun_buf;
	int buf_num;

	struct timer_list eof_timeout_timer;
	int eof_irq;
	int nfb4eof_irq;

	bool last_eof;  /* waiting for last EOF at encoder off */
	struct completion last_eof_comp;
};

/*
 * Update the CSI whole sensor and active windows, and initialize
 * the CSI interface and muxes.
 */
static void encoder_setup_csi(struct encoder_priv *priv)
{
	struct mx6cam_dev *dev = priv->dev;
	int csi_id = ipu_csi_get_num(priv->csi);

	ipu_csi_set_window_size(priv->csi, dev->crop.width, dev->crop.height);
	ipu_csi_set_window_pos(priv->csi, dev->crop.left, dev->crop.top);
	ipu_csi_init_interface(priv->csi, dev->crop_bounds.width,
			       dev->crop_bounds.height, &dev->ep->csi_sig_cfg);

	if (dev->ep->ep.bus_type == V4L2_MBUS_CSI2)
		ipu_csi_set_mipi_datatype(priv->csi, dev->ep->ep.id,
					  &dev->ep->csi_sig_cfg);

	/* select either parallel or MIPI-CSI2 as input to our CSI */
	ipu_csi_set_src(priv->csi, dev->ep->ep.id,
			dev->ep->ep.bus_type == V4L2_MBUS_CSI2);
	/* set CSI destination to IC or direct to mem via SMFC */
	ipu_csi_set_dest(priv->csi, dev->using_ic ?
			 IPU_CSI_DEST_IC : IPU_CSI_DEST_IDMAC);
	if (dev->using_ic) {
		/* set IC to receive from CSI */
		ipu_ic_set_src(priv->ic_enc, csi_id, false);
	}
}

static void encoder_put_ipu_resources(struct encoder_priv *priv)
{
	if (!IS_ERR_OR_NULL(priv->irt))
		ipu_irt_put(priv->irt);
	priv->irt = NULL;

	if (!IS_ERR_OR_NULL(priv->ic_enc))
		ipu_ic_put(priv->ic_enc);
	priv->ic_enc = NULL;

	if (!IS_ERR_OR_NULL(priv->enc_ch))
		ipu_idmac_put(priv->enc_ch);
	priv->enc_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->enc_rot_in_ch))
		ipu_idmac_put(priv->enc_rot_in_ch);
	priv->enc_rot_in_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->enc_rot_out_ch))
		ipu_idmac_put(priv->enc_rot_out_ch);
	priv->enc_rot_out_ch = NULL;

	if (!IS_ERR_OR_NULL(priv->smfc))
		ipu_smfc_put(priv->smfc);
	priv->smfc = NULL;

	if (!IS_ERR_OR_NULL(priv->csi))
		ipu_csi_put(priv->csi);
	priv->csi = NULL;
}

static int encoder_get_ipu_resources(struct encoder_priv *priv)
{
	struct mx6cam_dev *dev = priv->dev;
	int csi_id, csi_ch_num, err;

	csi_id = dev->ep->ep.port;
	priv->csi = ipu_csi_get(dev->ipu, csi_id);
	if (IS_ERR(priv->csi)) {
		v4l2_err(&priv->sd, "failed to get CSI %d\n", csi_id);
		return PTR_ERR(priv->csi);
	}

	if (dev->using_ic) {
		priv->ic_enc = ipu_ic_get(dev->ipu, IC_TASK_ENCODER);
		if (IS_ERR(priv->ic_enc)) {
			v4l2_err(&priv->sd, "failed to get IC ENC\n");
			err = PTR_ERR(priv->ic_enc);
			goto out;
		}

		priv->irt = ipu_irt_get(dev->ipu);
		if (IS_ERR(priv->irt)) {
			v4l2_err(&priv->sd, "failed to get IRT\n");
			err = PTR_ERR(priv->irt);
			goto out;
		}

		priv->enc_ch = ipu_idmac_get(dev->ipu,
					     IPUV3_CHANNEL_IC_PRP_ENC_MEM,
					     false);
		if (IS_ERR(priv->enc_ch)) {
			v4l2_err(&priv->sd, "could not get IDMAC channel %u\n",
				 IPUV3_CHANNEL_IC_PRP_ENC_MEM);
			err = PTR_ERR(priv->enc_ch);
			goto out;
		}

		priv->enc_rot_in_ch = ipu_idmac_get(dev->ipu,
						    IPUV3_CHANNEL_MEM_ROT_ENC,
						    false);
		if (IS_ERR(priv->enc_rot_in_ch)) {
			v4l2_err(&priv->sd, "could not get IDMAC channel %u\n",
				 IPUV3_CHANNEL_MEM_ROT_ENC);
			err = PTR_ERR(priv->enc_rot_in_ch);
			goto out;
		}

		priv->enc_rot_out_ch = ipu_idmac_get(dev->ipu,
						     IPUV3_CHANNEL_ROT_ENC_MEM,
						     false);
		if (IS_ERR(priv->enc_rot_out_ch)) {
			v4l2_err(&priv->sd, "could not get IDMAC channel %u\n",
				 IPUV3_CHANNEL_ROT_ENC_MEM);
			err = PTR_ERR(priv->enc_rot_out_ch);
			goto out;
		}
	} else {
		priv->smfc = ipu_smfc_get(dev->ipu);
		if (IS_ERR(priv->smfc)) {
			v4l2_err(&priv->sd, "failed to get SMFC\n");
			err = PTR_ERR(priv->smfc);
			goto out;
		}

		/*
		 * Choose the direct CSI-->SMFC-->MEM channel corresponding
		 * to the IPU and CSI IDs.
		 */
		csi_ch_num = IPUV3_CHANNEL_CSI0 +
			(ipu_get_num(dev->ipu) << 1) + csi_id;

		priv->enc_ch = ipu_idmac_get(dev->ipu, csi_ch_num, false);
		if (IS_ERR(priv->enc_ch)) {
			v4l2_err(&priv->sd, "could not get IDMAC channel %u\n",
				 csi_ch_num);
			err = PTR_ERR(priv->enc_ch);
			goto out;
		}
	}

	return 0;
out:
	encoder_put_ipu_resources(priv);
	return err;
}

static irqreturn_t encoder_eof_interrupt(int irq, void *dev_id)
{
	struct encoder_priv *priv = dev_id;
	struct mx6cam_dev *dev = priv->dev;
	struct mx6cam_ctx *ctx = dev->io_ctx;
	struct mx6cam_buffer *frame;
	struct ipuv3_channel *channel;
	enum vb2_buffer_state state;
	struct timeval cur_time;
	unsigned long flags;
	dma_addr_t phys;

	spin_lock_irqsave(&dev->irqlock, flags);

	/* timestamp and return the completed frame */
	frame = priv->active_frame[priv->buf_num];
	if (frame) {
		do_gettimeofday(&cur_time);
		frame->vb.v4l2_buf.timestamp = cur_time;
		state = dev->signal_locked ?
			VB2_BUF_STATE_DONE : VB2_BUF_STATE_ERROR;
		vb2_buffer_done(&frame->vb, state);
	}

	if (priv->last_eof) {
		complete(&priv->last_eof_comp);
		priv->active_frame[priv->buf_num] = NULL;
		priv->last_eof = false;
		goto unlock;
	}

	/* bump the EOF timeout timer */
	mod_timer(&priv->eof_timeout_timer,
		  jiffies + msecs_to_jiffies(MX6CAM_EOF_TIMEOUT));

	if (!list_empty(&ctx->ready_q)) {
		frame = list_entry(ctx->ready_q.next,
				   struct mx6cam_buffer, list);
		phys = vb2_dma_contig_plane_dma_addr(&frame->vb, 0);
		list_del(&frame->list);
		priv->active_frame[priv->buf_num] = frame;
	} else {
		phys = priv->underrun_buf.phys;
		priv->active_frame[priv->buf_num] = NULL;
	}

	channel = (dev->rot_mode >= IPU_ROTATE_90_RIGHT) ?
		priv->enc_rot_out_ch : priv->enc_ch;

	if (ipu_idmac_buffer_is_ready(channel, priv->buf_num))
		ipu_idmac_clear_buffer_ready(channel, priv->buf_num);

	ipu_cpmem_set_buffer(channel, priv->buf_num, phys);
	ipu_idmac_select_buffer(channel, priv->buf_num);

	priv->buf_num ^= 1;

unlock:
	spin_unlock_irqrestore(&dev->irqlock, flags);
	return IRQ_HANDLED;
}

static irqreturn_t encoder_nfb4eof_interrupt(int irq, void *dev_id)
{
	struct encoder_priv *priv = dev_id;
	struct mx6cam_dev *dev = priv->dev;

	v4l2_err(&priv->sd, "NFB4EOF\n");

	/*
	 * It has been discovered that with rotation, encoder disable
	 * creates a single NFB4EOF event which is 100% repeatable. So
	 * scheduling a restart here causes an endless NFB4EOF-->restart
	 * cycle. The error itself seems innocuous, capture is not adversely
	 * affected.
	 *
	 * So don't schedule a restart on NFB4EOF error. If the source
	 * of the NFB4EOF event on disable is ever found, it can
	 * be re-enabled, but is probably not necessary. Detecting the
	 * interrupt (and clearing the irq status in the IPU) seems to
	 * be enough.
	 */
	if (!dev->using_ic)
		v4l2_subdev_notify(&priv->sd, MX6CAM_NFB4EOF_NOTIFY, NULL);

	return IRQ_HANDLED;
}

/*
 * EOF timeout timer function.
 */
static void encoder_eof_timeout(unsigned long data)
{
	struct encoder_priv *priv = (struct encoder_priv *)data;

	v4l2_err(&priv->sd, "encoder EOF timeout\n");

	v4l2_subdev_notify(&priv->sd, MX6CAM_EOF_TIMEOUT_NOTIFY, NULL);
}

static void encoder_free_dma_buf(struct encoder_priv *priv,
				 struct mx6cam_dma_buf *buf)
{
	struct mx6cam_dev *dev = priv->dev;

	if (buf->virt)
		dma_free_coherent(dev->dev, buf->len, buf->virt, buf->phys);

	buf->virt = NULL;
	buf->phys = 0;
}

static int encoder_alloc_dma_buf(struct encoder_priv *priv,
				 struct mx6cam_dma_buf *buf,
				 int size)
{
	struct mx6cam_dev *dev = priv->dev;

	encoder_free_dma_buf(priv, buf);

	buf->len = PAGE_ALIGN(size);
	buf->virt = dma_alloc_coherent(dev->dev, buf->len, &buf->phys,
				       GFP_DMA | GFP_KERNEL);
	if (!buf->virt) {
		v4l2_err(&priv->sd, "failed to alloc dma buffer\n");
		return -ENOMEM;
	}

	return 0;
}

static void encoder_setup_channel(struct encoder_priv *priv,
				  struct ipuv3_channel *channel,
				  struct v4l2_pix_format *f,
				  enum ipu_rotate_mode rot_mode,
				  dma_addr_t addr0, dma_addr_t addr1,
				  bool rot_swap_width_height)
{
	struct mx6cam_dev *dev = priv->dev;
	u32 width, height, stride;
	unsigned int burst_size;
	struct ipu_image image;

	if (dev->using_ic && rot_swap_width_height) {
		width = f->height;
		height = f->width;
	} else {
		width = f->width;
		height = f->height;
	}
	stride = ipu_stride_to_bytes(width, f->pixelformat);

	ipu_cpmem_zero(channel);

	memset(&image, 0, sizeof(image));
	image.pix.width = image.rect.width = width;
	image.pix.height = image.rect.height = height;
	image.pix.bytesperline = stride;
	image.pix.pixelformat = f->pixelformat;
	image.phys0 = addr0;
	image.phys1 = addr1;
	ipu_cpmem_set_image(channel, &image);

	if (channel == priv->enc_rot_in_ch ||
	    channel == priv->enc_rot_out_ch) {
		burst_size = 8;
		ipu_cpmem_set_block_mode(channel);
	} else
		burst_size = (width % 16) ? 8 : 16;

	ipu_cpmem_set_burst_size(channel, burst_size);

	if (!dev->using_ic) {
		int csi_id = ipu_csi_get_num(priv->csi);
		bool passthrough;

		/*
		 * If the sensor uses 16-bit parallel CSI bus, we must handle
		 * the data internally in the IPU as 16-bit generic, aka
		 * passthrough mode.
		 */
		passthrough = (dev->ep->ep.bus_type != V4L2_MBUS_CSI2 &&
			       dev->ep->csi_sig_cfg.data_width ==
			       IPU_CSI_DATA_WIDTH_16);

		if (passthrough)
			ipu_cpmem_set_format_passthrough(channel, 16);

		if (dev->ep->ep.bus_type == V4L2_MBUS_CSI2)
			ipu_smfc_map(priv->smfc, channel, csi_id,
				     dev->ep->ep.id);
		else
			ipu_smfc_map(priv->smfc, channel, csi_id, 0);

		/*
		 * Set the channel for the direct CSI-->memory via SMFC
		 * use-case to very high priority, by enabling the watermark
		 * signal in the SMFC, enabling WM in the channel, and setting
		 * the channel priority to high.
		 *
		 * Refer to the iMx6 rev. D TRM Table 36-8: Calculated priority
		 * value.
		 *
		 * The WM's are set very low by intention here to ensure that
		 * the SMFC FIFOs do not overflow.
		 */
		ipu_smfc_set_wmc(priv->smfc, channel, false, 0x01);
		ipu_smfc_set_wmc(priv->smfc, channel, true, 0x02);
		ipu_cpmem_set_high_priority(channel);
		ipu_idmac_enable_watermark(channel, true);
		ipu_cpmem_set_axi_id(channel, 0);
		ipu_idmac_lock_enable(channel, 8);

		burst_size = ipu_cpmem_get_burst_size(channel);
		ipu_smfc_set_burst_size(priv->smfc, channel,
					burst_size, passthrough);
	}

	if (rot_mode)
		ipu_cpmem_set_rotation(channel, rot_mode);

	if (ipu_csi_is_interlaced(priv->csi) && channel == priv->enc_ch)
		ipu_cpmem_set_interlaced_scan(channel, 0);

	if (dev->using_ic) {
		ipu_ic_task_idma_init(priv->ic_enc, channel, width, height,
				      burst_size, rot_mode);
		ipu_cpmem_set_axi_id(channel, 1);
	}

	ipu_idmac_set_double_buffer(channel, true);
	ipu_idmac_set_triple_buffer(channel, false);
}

static int encoder_setup_rotation(struct encoder_priv *priv,
				  dma_addr_t phys0, dma_addr_t phys1,
				  struct v4l2_mbus_framefmt *inf,
				  struct v4l2_pix_format *outf)
{
	struct mx6cam_dev *dev = priv->dev;
	enum ipu_color_space in_cs, out_cs;
	int out_size = (outf->width * outf->height *
			ipu_bits_per_pixel(outf->pixelformat)) >> 3;
	int err;

	err = encoder_alloc_dma_buf(priv, &priv->underrun_buf, out_size);
	if (err) {
		v4l2_err(&priv->sd, "failed to alloc underrun_buf, %d\n", err);
		return err;
	}

	err = encoder_alloc_dma_buf(priv, &priv->rot_buf[0], out_size);
	if (err) {
		v4l2_err(&priv->sd, "failed to alloc rot_buf[0], %d\n", err);
		goto free_underrun;
	}
	err = encoder_alloc_dma_buf(priv, &priv->rot_buf[1], out_size);
	if (err) {
		v4l2_err(&priv->sd, "failed to alloc rot_buf[1], %d\n", err);
		goto free_rot0;
	}

	in_cs = ipu_mbus_code_to_colorspace(inf->code);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	err = ipu_ic_task_init(priv->ic_enc,
			       inf->width, inf->height,
			       outf->height, outf->width,
			       in_cs, out_cs);
	if (err) {
		v4l2_err(&priv->sd, "ipu_ic_task_init failed, %d\n", err);
		goto free_rot1;
	}

	/* init the IC ENC-->MEM IDMAC channel */
	encoder_setup_channel(priv, priv->enc_ch, outf,
			      IPU_ROTATE_NONE,
			      priv->rot_buf[0].phys,
			      priv->rot_buf[1].phys,
			      true);

	/* init the MEM-->IC ENC ROT IDMAC channel */
	encoder_setup_channel(priv, priv->enc_rot_in_ch, outf,
			      dev->rot_mode,
			      priv->rot_buf[0].phys,
			      priv->rot_buf[1].phys,
			      true);

	/* init the destination IC ENC ROT-->MEM IDMAC channel */
	encoder_setup_channel(priv, priv->enc_rot_out_ch, outf,
			      IPU_ROTATE_NONE,
			      phys0, phys1,
			      false);

	/* now link IC PRP-->MEM to MEM-->IC PRP ROT */
	ipu_link_prp_enc_rot_enc(dev->ipu);

	/* enable the IC and IRT */
	ipu_ic_enable(priv->ic_enc);
	ipu_irt_enable(priv->irt);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->enc_ch, 0);
	ipu_idmac_select_buffer(priv->enc_ch, 1);
	ipu_idmac_select_buffer(priv->enc_rot_out_ch, 0);
	ipu_idmac_select_buffer(priv->enc_rot_out_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->enc_ch);
	ipu_idmac_enable_channel(priv->enc_rot_in_ch);
	ipu_idmac_enable_channel(priv->enc_rot_out_ch);

	/* and finally enable the IC PRPENC task */
	ipu_ic_task_enable(priv->ic_enc);

	return 0;

free_rot1:
	encoder_free_dma_buf(priv, &priv->rot_buf[1]);
free_rot0:
	encoder_free_dma_buf(priv, &priv->rot_buf[0]);
free_underrun:
	encoder_free_dma_buf(priv, &priv->underrun_buf);
	return err;
}

static int encoder_setup_norotation(struct encoder_priv *priv,
				    dma_addr_t phys0, dma_addr_t phys1,
				    struct v4l2_mbus_framefmt *inf,
				    struct v4l2_pix_format *outf)
{
	struct mx6cam_dev *dev = priv->dev;
	enum ipu_color_space in_cs, out_cs;
	int out_size = (outf->width * outf->height *
			ipu_bits_per_pixel(outf->pixelformat)) >> 3;
	int err;

	err = encoder_alloc_dma_buf(priv, &priv->underrun_buf, out_size);
	if (err) {
		v4l2_err(&priv->sd, "failed to alloc underrun_buf, %d\n", err);
		return err;
	}

	in_cs = ipu_mbus_code_to_colorspace(inf->code);
	out_cs = ipu_pixelformat_to_colorspace(outf->pixelformat);

	if (dev->using_ic) {
		err = ipu_ic_task_init(priv->ic_enc,
				       inf->width, inf->height,
				       outf->width, outf->height,
				       in_cs, out_cs);
		if (err) {
			v4l2_err(&priv->sd, "ipu_ic_task_init failed, %d\n",
				 err);
			goto free_underrun;
		}
	}

	/* init the IC PRP-->MEM IDMAC channel */
	encoder_setup_channel(priv, priv->enc_ch, outf,
			      dev->rot_mode, phys0, phys1, false);

	if (dev->using_ic)
		ipu_ic_enable(priv->ic_enc);
	else
		ipu_smfc_enable(priv->smfc);

	/* set buffers ready */
	ipu_idmac_select_buffer(priv->enc_ch, 0);
	ipu_idmac_select_buffer(priv->enc_ch, 1);

	/* enable the channels */
	ipu_idmac_enable_channel(priv->enc_ch);

	if (dev->using_ic) {
		/* enable the IC ENCODE task */
		ipu_ic_task_enable(priv->ic_enc);
	}

	return 0;

free_underrun:
	encoder_free_dma_buf(priv, &priv->underrun_buf);
	return err;
}

static int encoder_start(struct encoder_priv *priv)
{
	struct mx6cam_dev *dev = priv->dev;
	struct mx6cam_ctx *ctx = dev->io_ctx;
	struct v4l2_mbus_framefmt inf;
	struct v4l2_pix_format outf;
	struct mx6cam_buffer *frame, *tmp;
	dma_addr_t phys[2] = {0};
	int i = 0, err;

	err = encoder_get_ipu_resources(priv);
	if (err)
		return err;

	list_for_each_entry_safe(frame, tmp, &ctx->ready_q, list) {
		phys[i] = vb2_dma_contig_plane_dma_addr(&frame->vb, 0);
		list_del(&frame->list);
		priv->active_frame[i++] = frame;
		if (i >= 2)
			break;
	}

	/* if preview is enabled it has already setup the CSI */
	if (!dev->preview_on)
		encoder_setup_csi(priv);

	inf = dev->sensor_fmt;
	inf.width = dev->crop.width;
	inf.height = dev->crop.height;
	outf = dev->user_fmt.fmt.pix;

	priv->buf_num = 0;

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
		err = encoder_setup_rotation(priv, phys[0], phys[1],
					     &inf, &outf);
	else
		err = encoder_setup_norotation(priv, phys[0], phys[1],
					       &inf, &outf);
	if (err)
		goto out_put_ipu;

	priv->nfb4eof_irq = ipu_idmac_channel_irq(dev->ipu,
						 priv->enc_ch,
						 IPU_IRQ_NFB4EOF);
	err = devm_request_irq(dev->dev, priv->nfb4eof_irq,
			       encoder_nfb4eof_interrupt, 0,
			       "mx6cam-enc-nfb4eof", priv);
	if (err) {
		v4l2_err(&priv->sd,
			 "Error registering encode NFB4EOF irq: %d\n", err);
		goto out_put_ipu;
	}

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
		priv->eof_irq = ipu_idmac_channel_irq(
			dev->ipu, priv->enc_rot_out_ch, IPU_IRQ_EOF);
	else
		priv->eof_irq = ipu_idmac_channel_irq(
			dev->ipu, priv->enc_ch, IPU_IRQ_EOF);

	err = devm_request_irq(dev->dev, priv->eof_irq,
			       encoder_eof_interrupt, 0,
			       "mx6cam-enc-eof", priv);
	if (err) {
		v4l2_err(&priv->sd,
			 "Error registering encode eof irq: %d\n", err);
		goto out_free_nfb4eof_irq;
	}

	err = ipu_csi_enable(priv->csi);
	if (err) {
		v4l2_err(&priv->sd, "CSI enable error: %d\n", err);
		goto out_free_eof_irq;
	}

	/* start the EOF timeout timer */
	mod_timer(&priv->eof_timeout_timer,
		  jiffies + msecs_to_jiffies(MX6CAM_EOF_TIMEOUT));

	return 0;

out_free_eof_irq:
	devm_free_irq(dev->dev, priv->eof_irq, priv);
out_free_nfb4eof_irq:
	devm_free_irq(dev->dev, priv->nfb4eof_irq, priv);
out_put_ipu:
	encoder_put_ipu_resources(priv);
	return err;
}

static int encoder_stop(struct encoder_priv *priv)
{
	struct mx6cam_dev *dev = priv->dev;
	struct mx6cam_buffer *frame;
	struct timeval cur_time;
	int i, ret;

	/* stop the EOF timeout timer */
	del_timer_sync(&priv->eof_timeout_timer);

	/*
	 * Mark next EOF interrupt as the last before encoder off,
	 * and then wait for interrupt handler to mark completion.
	 */
	init_completion(&priv->last_eof_comp);
	priv->last_eof = true;
	ret = wait_for_completion_timeout(&priv->last_eof_comp,
					  msecs_to_jiffies(MX6CAM_EOF_TIMEOUT));
	if (ret == 0)
		v4l2_warn(&priv->sd, "wait last encode EOF timeout\n");

	ipu_csi_disable(priv->csi);

	devm_free_irq(dev->dev, priv->eof_irq, priv);
	devm_free_irq(dev->dev, priv->nfb4eof_irq, priv);

	/* disable IC tasks and the channels */
	if (dev->using_ic)
		ipu_ic_task_disable(priv->ic_enc);

	ipu_idmac_disable_channel(priv->enc_ch);
	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT) {
		ipu_idmac_disable_channel(priv->enc_rot_in_ch);
		ipu_idmac_disable_channel(priv->enc_rot_out_ch);
	}

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_unlink_prp_enc_rot_enc(dev->ipu);

	if (dev->using_ic)
		ipu_ic_disable(priv->ic_enc);
	else
		ipu_smfc_disable(priv->smfc);

	if (dev->rot_mode >= IPU_ROTATE_90_RIGHT)
		ipu_irt_disable(priv->irt);

	encoder_free_dma_buf(priv, &priv->rot_buf[0]);
	encoder_free_dma_buf(priv, &priv->rot_buf[1]);
	encoder_free_dma_buf(priv, &priv->underrun_buf);

	encoder_put_ipu_resources(priv);

	/* return any remaining active frames with error */
	for (i = 0; i < 2; i++) {
		frame = priv->active_frame[i];
		if (frame && frame->vb.state == VB2_BUF_STATE_ACTIVE) {
			do_gettimeofday(&cur_time);
			frame->vb.v4l2_buf.timestamp = cur_time;
			vb2_buffer_done(&frame->vb, VB2_BUF_STATE_ERROR);
		}
	}

	return 0;
}

static int encoder_s_stream(struct v4l2_subdev *sd, int enable)
{
	struct encoder_priv *priv = v4l2_get_subdevdata(sd);

	if (enable)
		return encoder_start(priv);
	else
		return encoder_stop(priv);
}

static struct v4l2_subdev_video_ops encoder_video_ops = {
	.s_stream = encoder_s_stream,
};

static struct v4l2_subdev_ops encoder_subdev_ops = {
	.video = &encoder_video_ops,
};

struct v4l2_subdev *mx6cam_encoder_init(struct mx6cam_dev *dev)
{
	struct encoder_priv *priv;

	priv = devm_kzalloc(dev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return ERR_PTR(-ENOMEM);

	init_timer(&priv->eof_timeout_timer);
	priv->eof_timeout_timer.data = (unsigned long)priv;
	priv->eof_timeout_timer.function = encoder_eof_timeout;

	v4l2_subdev_init(&priv->sd, &encoder_subdev_ops);
	strlcpy(priv->sd.name, "mx6-camera-encoder", sizeof(priv->sd.name));
	v4l2_set_subdevdata(&priv->sd, priv);

	priv->dev = dev;
	return &priv->sd;
}
