/*
 * Copyright 2013-2014 Robert Bosch Engineering and Business solutions Ltd.,
 *			Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 *
 */

/*!
 * @file adv7182.c
 *
 * @brief Analog Device adv7182 video decoder functions
 *
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/semaphore.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/of_gpio.h>
#include <linux/wait.h>
#include <linux/videodev2.h>
#include <linux/workqueue.h>
#include <linux/regulator/consumer.h>
#include <media/v4l2-device.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-chip-ident.h>
#include <media/v4l2-subdev.h>
#include <media/v4l2-of.h>
#include <media/v4l2-ctrls.h>
#include <media/imx6.h>
#include "adv7182.h"


/*! Description of video formats supported.
 *
 *  PAL: raw=720x625, active=720x576.
 *  NTSC: raw=720x525, active=720x480.
 */
static struct video_fmt_t video_fmts[] = {
	{ /* NTSC */
		.v4l2_id = V4L2_STD_NTSC,
		.name = "NTSC",
		.raw = {
			.width = 720,
			.height = 525,
		},
		.crop = {
			.width = 720,
			.height = 480,
			.top = 13,
			.left = 0,
		}
	}, { /* (B, G, H, I, N) PAL */
		.v4l2_id = V4L2_STD_PAL,
		.name = "PAL",
		.raw = {
			.width = 720,
			.height = 625,
		},
		.crop = {
			.width = 720,
			.height = 576,
		},
	},
};

/* supported controls */
/* This hasn't been fully implemented yet.
 * This is how it should work, though. */
static struct v4l2_queryctrl adv7182_qctrl[] = {
	{
		.id = V4L2_CID_BRIGHTNESS,
		.type = V4L2_CTRL_TYPE_INTEGER,
		.name = "Brightness",
		.minimum = 0,		/* check this value */
		.maximum = 255,		/* check this value */
		.step = 1,		/* check this value */
		.default_value = 0x00,	/* check this value */
		.flags = 0,
	}, {
		.id = V4L2_CID_SATURATION,
		.type = V4L2_CTRL_TYPE_INTEGER,
		.name = "Saturation",
		.minimum = 0,		/* check this value */
		.maximum = 255,		/* check this value */
		.step = 0x1,		/* check this value */
		.default_value = 128,	/* check this value */
		.flags = 0,
	}, {
		.id = V4L2_CID_CONTRAST,
		.type = V4L2_CTRL_TYPE_INTEGER,
		.name = "Contrast",
		.minimum = 0,
		.maximum = 255,
		.step = 0x1,
		.default_value = 0x80,
		.flags = 0,
	}, {
		.id = V4L2_CID_HUE,
		.type = V4L2_CTRL_TYPE_INTEGER,
		.name = "Hue",
		.minimum = -127,
		.maximum = 128,
		.step = 0x1,
		.default_value = 0,
		.flags = 0,
	}
};
#define ADV7182_NUM_CONTROLS ARRAY_SIZE(adv7182_qctrl)

/* Analog Inputs on 32-Lead LFCSP */
static const struct adv7182_inputs_t adv7182_inputs_32[] = {
	{ .insel = 0x00, .desc = "ADV7182 Composite on Ain1" },
	{ .insel = 0x01, .desc = "ADV7182 Composite on Ain2" },
	{ .insel = 0x02, .desc = "ADV7182 Composite on Ain3" },
	{ .insel = 0x03, .desc = "ADV7182 Composite on Ain4" },
	{ .insel = 0x08, .desc = "ADV7182 Y/C on AIN1/2" },
	{ .insel = 0x09, .desc = "ADV7182 Y/C on AIN3/4" },
	{ .insel = 0x0C, .desc = "ADV7182 YPbPr on AIN1/2/3" },
	{ .insel = 0x0E, .desc = "ADV7182 Differential +/- on AIN1/2" },
	{ .insel = 0x0F, .desc = "ADV7182 Differential +/- on AIN3/4" },
};
#define NUM_INPUTS_32 ARRAY_SIZE(adv7182_inputs_32)

/* Default Register value table */
const u8 adv7182_def_reg_val[ADV7182_MAX_REG][2] = {
/* Register , values */
	{ADV_REG_ADDR_PWR, ADV_PWR_ON},
	{ADV_REG_ADDR_INSEL, ADV_INSEL_DEFAULT},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_RST_CLAMPDEFAULT1},
	{ADV_REG_ADI_RST_CLAMP, ADV_ADI_RST_CLAMPDEFAULT2},
	{ADV_REG_ADI_RST_CLAMP, ADV_ADI_RST_CLAMPDEFAULT3},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
	{ADV_REG_ADDR_ADC_VID_SEL, ADV_ADC_VID_SEL_AUTO},
	{ADV_REG_ADDR_ADC_ADI_CTRL2, ADV_ADC_ADI_CTRL2_VAL},
	{ADV_REG_ADDR_ADC_OUT_CTRL_EXTD, ADV_ADC_OUT_CTRL_EXTD_DEFAULT},
	{ADV_REG_ADDR_ADC_OUT_CTRL, ADV_ADC_OUT_CTRL_DEFAULT},
	{ADV_REG_ADDR_LOCK_CNT, ADV_LOCK_CNT_VAL},
	{ADV_REG_ADDR_DRIVE_STRENGTH, ADV_DRIVE_STRENGTH_VAL},
	{ADV_REG_ADI_IBIAS_AFE, ADV_ADI_IBIAS_AFE_DEFAULT},
	{ADV_REG_ADI_RES_CIR, ADV_ADI_RES_CIR_DEFAULT},
	{ADV_REG_ADI_CLAMP_ADJ, ADV_ADI_CLAMP_ADJ_DEFAULT},
	{ADV_REG_ADI_DIFF_MODE, ADV_ADI_DIFF_MODE_DEFAULT},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_RST_CLAMPDEFAULT1},
	{ADV_REG_ADI_AGC_ADJ1, ADV_ADI_AGC_ADJ1_DEFAULT},
	{ADV_REG_ADI_AGC_ADJ2, ADV_ADI_AGC_ADJ2_DEFAULT},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
	{ADV_REG_ADDR_STATUS1, ADV_STATUS1_DEFAULT},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_USR_SUB_MAP_40},
	{ADV_REG_ADI_ENA_ACE, ADV_ADI_ENA_ACE},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
	{ADV_REG_ADDR_VID_SEL1, ADV_VID_SEL1_DEFAULT},
	{ADV_REG_ADDR_VID_SEL2, ADV_VID_SEL2_DEFAULT},
	{ADV_REG_ADDR_CONTRAST, ADV_CONTRAST_DEFAULT},
	{ADV_REG_ADDR_BRIGHTNESS, ADV_BRIGHTNESS_DEFAULT},
	{ADV_REG_ADDR_HUE, ADV_HUE_DEFAULT},
	{ADV_REG_ADDR_DEF_COLOR_Y, ADV_COLOR_Y_DEFAULT},
	{ADV_REG_ADDR_DEF_COLOR_CR_CB, ADV_COLOR_CR_CB_DEFAULT},
	{ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_CTRL1_DEFAULT},
	{ADV_REG_ADDR_ANA_CLAMP_CTRL, ADV_ANA_CLAMP_CTRL_DEFAULT},
	{ADV_REG_ADDR_DIG_CLAMP_CTRL, ADV_DIG_CLAMP_CTRL_DEFAULT},
	{ADV_REG_ADDR_SHAP_FIL_CTRL1, ADV_SHAP_FIL_CTRL1_VAL},
	{ADV_REG_ADDR_SHAP_FIL_CTRL2, ADV_SHAP_FIL_CTRL2_DEFAULT},
	{ADV_REG_ADDR_COMB_FIL_CTRL, ADV_COMB_FIL_CTRL_DEFAULT},
	{ADV_REG_ADDR_PIX_DELAY, ADV_PIX_DELAY_DEFAULT},
	{ADV_REG_ADDR_MISC_GAIN_CTRL, ADV_MISC_GAIN_CTRL_DEFAULT},
	{ADV_REG_ADDR_AGC_MOD_CTRL, ADV_AGC_MOD_CTRL_DEFAULT},
	{ADV_REG_ADDR_CHR_GAIN_CTRL1, ADV_CHR_GAIN_CTRL1_DEFAULT},
	{ADV_REG_ADDR_CHR_GAIN_CTRL2, ADV_CHR_GAIN_CTRL2_DEFAULT},
	{ADV_REG_ADDR_LUM_GAIN_CTRL1, ADV_LUM_GAIN_CTRL1_DEFAULT},
	{ADV_REG_ADDR_LUM_GAIN_CTRL2, ADV_LUM_GAIN_CTRL2_DEFAULT},
	{ADV_REG_ADDR_VS_CTRL1, ADV_VS_CTRL1_DEFAULT},
	{ADV_REG_ADDR_VS_CTRL2, ADV_VS_CTRL2_DEFAULT},
	{ADV_REG_ADDR_VS_CTRL3, ADV_VS_CTRL3_DEFAULT},
	{ADV_REG_ADDR_HS_CTRL1, ADV_HS_CTRL1_DEFAULT},
	{ADV_REG_ADDR_HS_CTRL2, ADV_HS_CTRL2_DEFAULT},
	{ADV_REG_ADDR_HS_CTRL3, ADV_HS_CTRL3_DEFAULT},
	{ADV_REG_ADDR_POL, ADV_POL_DEFAULT},
	{ADV_REG_ADDR_NTSC_CMB_CTRL, ADV_NTSC_CMB_CTRL_DEFAULT},
	{ADV_REG_ADDR_PAL_CMB_CTRL, ADV_PAL_CMB_CTRL_DEFAULT},
	{ADV_REG_ADDR_WIN_CTRL, ADV_WIN_CTRL_DEFAULT},
	{ADV_REG_ADDR_RESAMPLE_CTRL, ADV_RESAMPLE_CTRL_DEFAULT},
	{ADV_REG_ADDR_CTI_DNR_CTRL1, ADV_CTI_DNR_CTRL1_DEFAULT},
	{ADV_REG_ADDR_CTI_DNR_CTRL2, ADV_CTI_DNR_CTRL2_DEFAULT},
	{ADV_REG_ADDR_DNR_NOISE_THR1, ADV_DNR_NOISE_THR1_DEFAULT},
	{ADV_REG_ADDR_OUT_SYNC_SEL1, ADV_OUT_SYNC_SEL1_DEFAULT},
	{ADV_REG_ADDR_OUT_SYNC_SEL2, ADV_OUT_SYNC_SEL2_DEFAULT},
	{ADV_REG_ADDR_FREE_RUN_LEN1, ADV_FREE_RUN_LEN1_DEFAULT},
	{ADV_REG_ADDR_CCAP1_STATUS, ADV_CCAP1_STATUS_DEFAULT},
	{ADV_REG_ADDR_CCAP2_STATUS, ADV_CCAP2_STATUS_DEFAULT},
	{ADV_REG_ADDR_LETBOX1_STATUS, ADV_LETBOX1_STATUS_DEFAULT},
	{ADV_REG_ADDR_LETBOX2_STATUS, ADV_LETBOX2_STATUS_DEFAULT},
	{ADV_REG_ADDR_LETBOX3_STATUS, ADV_LETBOX3_STATUS_DEFAULT},
	{ADV_REG_ADDR_CRC_ENABLE, ADV_CRC_ENABLE_DEFAULT},
	{ADV_REG_ADDR_LETBOX1_CTRL, ADV_LETBOX1_CTRL_DEFAULT},
	{ADV_REG_ADDR_LETBOX2_CTRL, ADV_LETBOX2_CTRL_DEFAULT},
	{ADV_REG_ADDR_ST_NOISE_STATUS1, ADV_ST_NOISE_STATUS1_DEFAULT},
	{ADV_REG_ADDR_ST_NOISE_STATUS2, ADV_ST_NOISE_STATUS2_DEFAULT},
	{ADV_REG_ADDR_SD_CB_CHN_OFFSET, ADV_SD_CB_CHN_OFFSET_DEFAULT},
	{ADV_REG_ADDR_SD_CR_CHN_OFFSET, ADV_SD_CR_CHN_OFFSET_DEFAULT},
	{ADV_REG_ADDR_SD_CB_CHN_SAT, ADV_SD_CB_CHN_SAT_DEFAULT},
	{ADV_REG_ADDR_SD_CR_CHN_SAT, ADV_SD_CR_CHN_SAT_DEFAULT},
	{ADV_REG_ADDR_NTSC_VSB, ADV_NTSC_VSB_DEFAULT},
	{ADV_REG_ADDR_NTSC_VSE, ADV_NTSC_VSE_VAL},
	{ADV_REG_ADDR_NTSC_FLD_TGLE, ADV_NTSC_FLD_TGLE_DEFAULT},
	{ADV_REG_ADDR_PAL_VSB, ADV_PAL_VSB_DEFAULT},
	{ADV_REG_ADDR_PAL_VSE, ADV_PAL_VSE_VAL},
	{ADV_REG_ADDR_PAL_FLD_TGLE, ADV_PAL_FLD_TGLE_DEFAULT},
	{ADV_REG_ADDR_VBLK_CTRL1, ADV_VBLK_CTRL1_DEFAULT},
	{ADV_REG_ADDR_VBLK_CTRL2, ADV_VBLK_CTRL2_DEFAULT},
	{ADV_REG_ADDR_AFE_CTRL1, ADV_AFE_CTRL1_DEFAULT},
	{ADV_REG_ADDR_IF_CLMP_CTRL1, ADV_IF_CLMP_CTRL1_DEFAULT},
	{ADV_REG_ADDR_VS_MODE_CTRL, ADV_VS_MODE_CTRL_DEFAULT},
	{ADV_REG_ADDR_SHARPNESS, ADV_SHARPNESS_DEFAULT},
	{ADV_REG_ADDR_DNR_NOISE_THR2, ADV_DNR_NOISE_THR2_DEFAULT},
	{ADV_REG_ADDR_CSI_TX_SLAVE, ADV_CSI_TX_SLAVE_DEFAULT}
};

/***********************************************************************
 * I2C transfer.
 ***********************************************************************/

/*! Read one register from a adv7182 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7182_read(struct adv7182_dev *sensor, u8 reg)
{
	int ret;
	ret = i2c_smbus_read_byte_data(sensor->i2c_client, reg);
	if (ret < 0) {
		dev_dbg(sensor->dev,
			"%s:read reg error: reg=%2x\n", __func__, reg);
		dev_err(sensor->dev,
			"ADV7182 I2C communication read failed\n");
	}
	return ret;
}

/*! Write one register of a adv7182 i2c slave device.
 *
 *  @param *reg		register in the device we wish to access.
 *
 *  @return		       0 if success, an error code otherwise.
 */
static int adv7182_write_reg(struct adv7182_dev *sensor, u8 reg, u8 val)
{
	int ret;
	ret = i2c_smbus_write_byte_data(sensor->i2c_client, reg, val);
	if (ret < 0) {
		dev_dbg(sensor->dev,
			"%s:write reg error:reg=%2x,val=%2x\n", __func__,
			reg, val);
		dev_err(sensor->dev,
			"ADV7182 I2C communication write failed\n");
	}
	return ret;
}

/* Update lock status */
static bool adv7182_update_lock_status(struct adv7182_dev *sensor)
{
	int stat1, int_stat1, int_stat3, int_raw_stat3;
	bool ret;

	stat1 = adv7182_read(sensor, ADV_REG_STATUS_1);

	/* Switch to interrupt register map */
	adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1,
			  ADV_ADI_INT_REG_MAP);

	int_stat1 = adv7182_read(sensor, ADV7182_INT_STATUS_1);
	int_stat3 = adv7182_read(sensor, ADV7182_INT_STATUS_3);
	/* clear the interrupts */
	adv7182_write_reg(sensor, ADV7182_INT_CLEAR_1, int_stat1);
	adv7182_write_reg(sensor, ADV7182_INT_CLEAR_3, int_stat3);

	int_raw_stat3 = adv7182_read(sensor, ADV7182_INT_RAW_STATUS_3);

	/* Switch back to normal register map */
	adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1,
			  ADV_ADI_CTRL1_DEFAULT);

	dev_dbg(sensor->dev,
		"stat1=0x%02x, int_stat 1/3/raw3 = 0x%02x/0x%02x/0x%02x\n",
		stat1, int_stat1, int_stat3, int_raw_stat3);

	ret = (((int_stat1 & ADV7182_INT_SD_LOCK) ||
		(int_stat1 & ADV7182_INT_SD_UNLOCK) ||
		(int_stat3 & ADV7182_INT_SD_V_LOCK_CHNG)) != 0);

	sensor->locked = ((stat1 & ADV7182_IN_LOCK) &&
			  (stat1 & ADV7182_FSC_LOCK) &&
			  (int_raw_stat3 & ADV7182_INT_SD_V_LOCK));

	dev_dbg(sensor->dev, "lsc %s: %s\n", ret ? "true" : "false",
		sensor->locked ? "locked" : "unlocked");

	return ret;
}

/* Read AD_RESULT to get the autodetected video standard */
static bool adv7182_get_autodetect_std(struct adv7182_dev *sensor)
{
	int stat1, ad_result, idx = adv7182_PAL;
	v4l2_std_id std = V4L2_STD_PAL;
	bool ret = false;

	/*
	 * When the chip loses lock, it continues to send data at whatever
	 * standard was detected before, so leave the standard at the last
	 * detected standard.
	 */
	if (!sensor->locked) {
		dev_dbg(sensor->dev, "No signal lock\n");
		return false; /* no status change */
	}

	stat1 = adv7182_read(sensor, ADV_REG_STATUS_1);
	ad_result = (stat1 & ADV7182_AD_RESULT_MASK) >> ADV7182_AD_RESULT_BIT;

	switch (ad_result) {
	case ADV7182_AD_PAL:
		std = V4L2_STD_PAL;
		idx = adv7182_PAL;
		dev_dbg(sensor->dev,
			"Detected PAL standard\n");
		break;
	case ADV7182_AD_PAL_M:
		std = V4L2_STD_PAL_M;
		/* PAL M is very similar to NTSC (same lines/field) */
		idx = adv7182_NTSC;
		dev_dbg(sensor->dev,
			"Detected PAL M standard\n");
		break;
	case ADV7182_AD_PAL_N:
		std = V4L2_STD_PAL_N;
		idx = adv7182_PAL;
		dev_dbg(sensor->dev,
			"Detected PAL N standard\n");
		break;
	case ADV7182_AD_PAL_60:
		std = V4L2_STD_PAL_60;
		/* PAL 60 has same lines as NTSC */
		idx = adv7182_NTSC;
		dev_dbg(sensor->dev,
			"Detected PAL 60 standard\n");
		break;
	case ADV7182_AD_NTSC:
		std = V4L2_STD_NTSC;
		idx = adv7182_NTSC;
		dev_dbg(sensor->dev,
			"Detected NTSC standard\n");
		break;
	case ADV7182_AD_NTSC_4_43:
		std = V4L2_STD_NTSC_443;
		idx = adv7182_NTSC;
		dev_dbg(sensor->dev,
			"Detected NTSC 4.43 standard\n");
		break;
	case ADV7182_AD_SECAM:
		std = V4L2_STD_SECAM;
		idx = adv7182_PAL;
		dev_dbg(sensor->dev,
			"Detected SECAM standard\n");
		break;
	case ADV7182_AD_SECAM_525:
		/*
		 * FIXME: could not find any info on "SECAM 525", assume
		 * it is SECAM but with NTSC line standard.
		 */
		std = V4L2_STD_SECAM;
		idx = adv7182_NTSC;
		dev_dbg(sensor->dev,
			"Detected SECAM 525 standard\n");
		break;
	}

	if (std != sensor->std_id) {
		sensor->video_idx = idx;
		sensor->std_id = std;
		sensor->fmt.width = video_fmts[sensor->video_idx].raw.width;
		sensor->fmt.height = video_fmts[sensor->video_idx].raw.height;
		ret = true;

		dev_dbg(sensor->dev, "changing std: 0x%08x --> 0x%08x\n",
			(u32)sensor->std_id, (u32)std);
	}

	return ret;
}

/*!
 * Sets the camera power.
 *
 * sensor -  pointer to the camera device
 * enable if 1, power is to be turned on.  0 means power is to be turned off
 *
 * @sensor: pointer to sensor device structure
 * @enable: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 */
static int adv7182_power(struct adv7182_dev *sensor, bool enable)
{
	if (enable && !sensor->on) {
		if (gpio_is_valid(sensor->gpio_pwrdn))
			gpio_set_value_cansleep(sensor->gpio_pwrdn, 1);

		usleep_range(5000, 5001);
		if (adv7182_write_reg(sensor, ADV_REG_ADDR_PWR,
				      ADV_PWR_ON) != 0)
			return -EIO;
	} else if (!enable && sensor->on) {
		if (adv7182_write_reg(sensor, ADV_REG_ADDR_PWR,
				      ADV_PWR_OFF) != 0)
			return -EIO;

		if (gpio_is_valid(sensor->gpio_pwrdn))
			gpio_set_value_cansleep(sensor->gpio_pwrdn, 0);
	}

	sensor->on = enable;
	return ADV7182_SUCCESS;
}

/*
 * Enable the SD_UNLOCK and SD_AD_CHNG interrupts.
 */
static void adv7182_enable_interrupts(struct adv7182_dev *sensor)
{
	mutex_lock(&sensor->mutex);

	/* Switch to interrupt register map */
	adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1, ADV_ADI_INT_REG_MAP);
	/* INTRQ active low, active until cleared */
	adv7182_write_reg(sensor, ADV7182_INT_CONFIG_1, ADV_INTR_CONFIG_1);
	/* unmask SD_UNLOCK and SD_LOCK */
	adv7182_write_reg(sensor, ADV7182_INT_MASK_1,
			  ADV7182_INT_SD_UNLOCK | ADV7182_INT_SD_LOCK);
	/* unmask SD_AD_CHNG and SD_V_LOCK_CHNG */
	adv7182_write_reg(sensor, ADV7182_INT_MASK_3,
			  ADV7182_INT_SD_AD_CHNG | ADV7182_INT_SD_V_LOCK_CHNG);
	/* Switch back to normal register map */
	adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1,
			  ADV_ADI_CTRL1_DEFAULT);

	mutex_unlock(&sensor->mutex);
}

/* threaded irq handler */
static irqreturn_t adv7182_interrupt(int irq, void *dev_id)
{
	struct adv7182_dev *sensor = dev_id;
	bool std_change, lock_status_change;

	mutex_lock(&sensor->mutex);

	lock_status_change = adv7182_update_lock_status(sensor);
	std_change = adv7182_get_autodetect_std(sensor);

	mutex_unlock(&sensor->mutex);

	if (lock_status_change || std_change)
		v4l2_subdev_notify(&sensor->sd,
				   DECODER_STATUS_CHANGE_NOTIFY, NULL);

	return IRQ_HANDLED;
}

static const struct adv7182_inputs_t *
adv7182_find_input(struct adv7182_dev *sensor, u32 insel)
{
	int i;

	for (i = 0; i < NUM_INPUTS_32; i++) {
		if (insel == adv7182_inputs_32[i].insel)
			return &adv7182_inputs_32[i];
	}

	return NULL;
}

/*!
 * adv7182_gpio_initialise - initialize the GPIO pin with direction set.
 *
 * @param gpio_pin: points to GPIO pin number
 * @param direction: points to direction(IN or OUT)
 * @param high_low: high-1,low-0 is used if direction is set as "OUT".
 * @param gpio_name: Has GPIO name for the provided pin.
 *
 * @return		0 if success, an error code otherwise.
 *
 */

static s32 adv7182_gpio_initialise(struct adv7182_dev *sensor,
				   u32 gpio_pin, u32 direction,
				   u32 high_low, const char *gpio_name)
{
	s32 ret;

	/*ckeck for valid GPIO pin*/
	if (!gpio_is_valid(gpio_pin)) {
		dev_err(sensor->dev,
			"ADV7182: **GPIO invalid pin**\n");
		return ADV7182_GPIO_ERR_INVALID_PIN;
	}

	/*Allocates memory for requested pin*/
	if (gpio_request(gpio_pin, gpio_name)) {
		dev_err(sensor->dev,
			"ADV7182: **GPIO request is failed**\n");
		return ADV7182_GPIO_ERR_REQ;
	}
	switch (direction) {
	case ADV7182_GPIO_DIR_OUT:
	{
		/* set direction as output for the gpio pin and set
		   the value which will be initialised */
		if (gpio_direction_output(gpio_pin, high_low)
		    == ADV7182_SUCCESS) {
			ret = ADV7182_SUCCESS;
		} else {
			dev_err(sensor->dev,
				"ADV7182: gpio Dir set OUT failed\n");
			gpio_free(gpio_pin);
			ret = ADV7182_GPIO_ERR_DIR_OUT;
		}
	}
	break;
	case ADV7182_GPIO_DIR_IN:
	{
		/*set the direction as input for the gpio pin*/
		if (gpio_direction_input(gpio_pin) == ADV7182_SUCCESS) {
			ret = ADV7182_SUCCESS;
		} else {
			dev_err(sensor->dev,
				"ADV7182: gpio Dir set IN failed\n");
			gpio_free(gpio_pin);
			ret = ADV7182_GPIO_ERR_DIR_IN;
		}
	}
	break;
	default:
	{
		/*deallocate the memory for the gpio pin*/
		dev_err(sensor->dev,
			"ADV7182:GPIO dir is not known**\n");
		gpio_free(gpio_pin);
		ret = ADV7182_GPIO_ERR_INVALID_DIR;
	}
	break;
	}

	return ret;
}


/* --------------- Subdev Operations --------------- */

/*!
 * adv7182_querystd - V4L2 sensor interface handler for vidioc_int_querystd_num.
 * Return current video standard. This device autodetects the current
 * standard, so this function also sets the values that need to be changed
 * if the standard changes. There is no set std equivalent function.
 */
static int adv7182_querystd(struct v4l2_subdev *sd, v4l2_std_id *std)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	mutex_lock(&sensor->mutex);

	/*
	 * If we have the ADV7182 irq, we can just return the currently
	 * detected standard. Otherwise we have to poll the AD_RESULT
	 * bits every time adv7182_querystd() is called.
	 */
	if (!sensor->i2c_client->irq) {
		adv7182_update_lock_status(sensor);
		adv7182_get_autodetect_std(sensor);
	}

	*std = sensor->std_id;

	mutex_unlock(&sensor->mutex);

	return ADV7182_SUCCESS;
}

/*!
 * Sets the camera power.
 *
 * s  pointer to the camera device
 * on if 1, power is to be turned on.  0 means power is to be turned off
 *
 * @sd: pointer to standard V4L2 subdevice structure
 * @on: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 */
static int adv7182_s_power(struct v4l2_subdev *sd, int on)
{
	return ADV7182_SUCCESS;
}

/*!
 * adv7182_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int adv7182_g_parm(struct v4l2_subdev *sd, struct v4l2_streamparm *a)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);
	struct v4l2_captureparm *cparm = &a->parm.capture;

	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	memset(a, 0, sizeof(*a));
	a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	cparm->capability = sensor->streamcap.capability;
	cparm->timeperframe = sensor->streamcap.timeperframe;
	cparm->capturemode = sensor->streamcap.capturemode;

	return ADV7182_SUCCESS;
}

/*!
 * adv7182_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 *
 * This driver cannot change these settings.
 */
static int adv7182_s_parm(struct v4l2_subdev *sd, struct v4l2_streamparm *a)
{
	return ADV7182_SUCCESS;
}

static int adv7182_g_mbus_fmt(struct v4l2_subdev *sd,
			      struct v4l2_mbus_framefmt *fmt)

{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	*fmt = sensor->fmt;

	return ADV7182_SUCCESS;
}

/*
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int adv7182_try_mbus_fmt(struct v4l2_subdev *sd,
				struct v4l2_mbus_framefmt *fmt)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	*fmt = sensor->fmt;
	return ADV7182_SUCCESS;
}

/*
 * This driver autodetects a standard video mode, so we don't allow
 * setting a mode, just return the current autodetected mode.
 *
 * Return 0.
 */
static int adv7182_s_mbus_fmt(struct v4l2_subdev *sd,
			      struct v4l2_mbus_framefmt *fmt)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	*fmt = sensor->fmt;
	return ADV7182_SUCCESS;
}

/*!
 * adv7182_s_ctrl - V4L2 sensor interface handler for VIDIOC_S_CTRL ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @vc: standard V4L2 VIDIOC_S_CTRL ioctl structure
 *
 * If the requested control is supported, sets the control's current
 * value in HW (and updates the video_control[] array).  Otherwise,
 * returns -EINVAL if the control is not supported.
 */
static int adv7182_s_ctrl(struct v4l2_ctrl *ctrl)
{
	struct adv7182_dev *sensor = ctrl_to_adv7182_dev(ctrl);
	int retval = ADV7182_SUCCESS;
	u8 tmp;

	switch (ctrl->id) {
	case V4L2_CID_BRIGHTNESS:
		dev_dbg(sensor->dev, "   V4L2_CID_BRIGHTNESS\n");
		tmp = ctrl->val;
		adv7182_write_reg(sensor, ADV_REG_ADDR_BRIGHTNESS, tmp);
		sensor->brightness = ctrl->val;
		break;
	case V4L2_CID_CONTRAST:
		dev_dbg(sensor->dev, "   V4L2_CID_CONTRAST\n");
		tmp = ctrl->val;
		adv7182_write_reg(sensor, ADV_REG_ADDR_CONTRAST, tmp);
		sensor->contrast = ctrl->val;
		break;
	case V4L2_CID_SATURATION:
		dev_dbg(sensor->dev, "   V4L2_CID_SATURATION\n");
		tmp = ctrl->val;
		adv7182_write_reg(sensor, ADV_REG_ADDR_SD_CB_CHN_SAT, tmp);
		adv7182_write_reg(sensor, ADV_REG_ADDR_SD_CR_CHN_SAT, tmp);
		sensor->saturation = ctrl->val;
		break;
	case V4L2_CID_HUE:
		dev_dbg(sensor->dev, "   V4L2_CID_HUE\n");
		tmp = ctrl->val;
		/* Hue is inverted according to HSL chart */
		adv7182_write_reg(sensor, ADV_REG_ADDR_HUE, -tmp);
		sensor->hue = ctrl->val;
		break;
	case V4L2_CID_AUTO_WHITE_BALANCE:
		dev_dbg(sensor->dev, "   V4L2_CID_AUTO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_DO_WHITE_BALANCE:
		dev_dbg(sensor->dev, "   V4L2_CID_DO_WHITE_BALANCE\n");
		break;
	case V4L2_CID_RED_BALANCE:
		dev_dbg(sensor->dev, "   V4L2_CID_RED_BALANCE\n");
		break;
	case V4L2_CID_BLUE_BALANCE:
		dev_dbg(sensor->dev, "   V4L2_CID_BLUE_BALANCE\n");
		break;
	case V4L2_CID_GAMMA:
		dev_dbg(sensor->dev, "   V4L2_CID_GAMMA\n");
		break;
	case V4L2_CID_EXPOSURE:
		dev_dbg(sensor->dev, "   V4L2_CID_EXPOSURE\n");
		break;
	case V4L2_CID_AUTOGAIN:
		dev_dbg(sensor->dev, "   V4L2_CID_AUTOGAIN\n");
		break;
	case V4L2_CID_GAIN:
		dev_dbg(sensor->dev, "   V4L2_CID_GAIN\n");
		break;
	case V4L2_CID_HFLIP:
		dev_dbg(sensor->dev, "   V4L2_CID_HFLIP\n");
		break;
	case V4L2_CID_VFLIP:
		dev_dbg(sensor->dev, "   V4L2_CID_VFLIP\n");
		break;
	default:
		dev_dbg(sensor->dev, "   Default case\n");
		retval = -EPERM;
		break;
	}

	return retval;
}

static const struct v4l2_ctrl_ops adv7182_ctrl_ops = {
	.s_ctrl = adv7182_s_ctrl,
};

static int adv7182_init_controls(struct adv7182_dev *sensor)
{
	struct v4l2_queryctrl *c;
	int i;

	v4l2_ctrl_handler_init(&sensor->ctrl_hdl, ADV7182_NUM_CONTROLS);

	for (i = 0; i < ADV7182_NUM_CONTROLS; i++) {
		c = &adv7182_qctrl[i];

		v4l2_ctrl_new_std(&sensor->ctrl_hdl, &adv7182_ctrl_ops,
				  c->id, c->minimum, c->maximum,
				  c->step, c->default_value);
	}

	sensor->sd.ctrl_handler = &sensor->ctrl_hdl;
	if (sensor->ctrl_hdl.error) {
		int err = sensor->ctrl_hdl.error;

		v4l2_ctrl_handler_free(&sensor->ctrl_hdl);

		dev_err(sensor->dev, "%s: error %d\n", __func__, err);
		return err;
	}
	v4l2_ctrl_handler_setup(&sensor->ctrl_hdl);

	return ADV7182_SUCCESS;
}

/*!
 * adv7182_enum_framesizes - V4L2 sensor interface handler for
 *			   VIDIOC_ENUM_FRAMESIZES ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @fsize: standard V4L2 VIDIOC_ENUM_FRAMESIZES ioctl structure
 *
 * Return 0 if successful, otherwise -EINVAL.
 */
static int adv7182_enum_framesizes(struct v4l2_subdev *sd,
				 struct v4l2_frmsizeenum *fsize)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	if (fsize->index > 0)
		return -EINVAL;

	fsize->type = V4L2_FRMSIZE_TYPE_DISCRETE;
	fsize->discrete.width = video_fmts[sensor->video_idx].crop.width;
	fsize->discrete.height = video_fmts[sensor->video_idx].crop.height;

	dev_dbg(sensor->dev, "Framesize %d %d\n",
		fsize->discrete.width, fsize->discrete.height);

	return ADV7182_SUCCESS;
}

static int adv7182_g_crop(struct v4l2_subdev *sd, struct v4l2_crop *a)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	a->c = video_fmts[sensor->video_idx].crop;

	return ADV7182_SUCCESS;
}

static int adv7182_g_input_status(struct v4l2_subdev *sd, u32 *status)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	mutex_lock(&sensor->mutex);

	*status = 0;

	if (sensor->on) {
		if (!sensor->locked)
			*status = V4L2_IN_ST_NO_SIGNAL | V4L2_IN_ST_NO_SYNC;
	} else
		*status = V4L2_IN_ST_NO_POWER;

	mutex_unlock(&sensor->mutex);

	return ADV7182_SUCCESS;
}

static int adv7182_s_routing(struct v4l2_subdev *sd, u32 input,
			     u32 output, u32 config)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);
	const struct adv7182_inputs_t *advinput;
	int ret = ADV7182_SUCCESS;

	advinput = adv7182_find_input(sensor, input);
	if (!advinput)
		return -EINVAL;

	mutex_lock(&sensor->mutex);
	if (adv7182_write_reg(sensor, ADV_REG_ADDR_INSEL,
			      advinput->insel) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_RST_CLAMPDEFAULT1) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(sensor, ADV_REG_ADI_RST_CLAMP,
			      ADV_ADI_RST_CLAMPDEFAULT2) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(sensor, ADV_REG_ADI_RST_CLAMP,
			      ADV_ADI_RST_CLAMPDEFAULT3) != 0) {
		ret = -EIO;
		goto error;
	}
	if (adv7182_write_reg(sensor, ADV_REG_ADDR_ADI_CTRL1,
			      ADV_ADI_CTRL1_DEFAULT) != 0) {
		ret = -EIO;
		goto error;
	}

	sensor->current_input = input;
error:
	mutex_unlock(&sensor->mutex);
	return ret;
}

/*!
 * adv7182_g_chip_ident - V4L2 sensor interface handler for
 *			VIDIOC_DBG_G_CHIP_IDENT ioctl
 * @sd: pointer to standard V4L2 subdevice structure
 * @chip: pointer to struct v4l2_dbg_chip_ident
 *
 * Return 0.
 */
static int adv7182_g_chip_ident(struct v4l2_subdev *sd,
				struct v4l2_dbg_chip_ident *chip)
{
	struct i2c_client *client = v4l2_get_subdevdata(sd);

	return v4l2_chip_ident_i2c_client(client, chip, V4L2_IDENT_ADV7182, 0);
}

static int adv7182_enum_mbus_fmt(struct v4l2_subdev *sd, unsigned index,
				 enum v4l2_mbus_pixelcode *code)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	if (index != 0)
		return -EINVAL;

	*code = sensor->fmt.code;

	return ADV7182_SUCCESS;
}

static int adv7182_g_mbus_config(struct v4l2_subdev *sd,
				struct v4l2_mbus_config *cfg)
{
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	cfg->type = V4L2_MBUS_BT656;
	cfg->flags = sensor->ep.bus.parallel.flags;

	return ADV7182_SUCCESS;
}

static int adv7182_s_stream(struct v4l2_subdev *sd, int enable)
{
	return ADV7182_SUCCESS;
}

static struct v4l2_subdev_core_ops adv7182_core_ops = {
	.s_power = adv7182_s_power,
	.g_chip_ident = adv7182_g_chip_ident,
	.g_ext_ctrls = v4l2_subdev_g_ext_ctrls,
	.try_ext_ctrls = v4l2_subdev_try_ext_ctrls,
	.s_ext_ctrls = v4l2_subdev_s_ext_ctrls,
	.g_ctrl = v4l2_subdev_g_ctrl,
	.s_ctrl = v4l2_subdev_s_ctrl,
	.queryctrl = v4l2_subdev_queryctrl,
	.querymenu = v4l2_subdev_querymenu,
};

static struct v4l2_subdev_video_ops adv7182_video_ops = {
	.enum_mbus_fmt = adv7182_enum_mbus_fmt,
	.try_mbus_fmt = adv7182_try_mbus_fmt,
	.g_mbus_fmt = adv7182_g_mbus_fmt,
	.s_mbus_fmt = adv7182_s_mbus_fmt,
	.s_parm = adv7182_s_parm,
	.g_parm = adv7182_g_parm,
	.enum_framesizes = adv7182_enum_framesizes,
	.g_crop = adv7182_g_crop,
	.g_input_status = adv7182_g_input_status,
	.s_routing = adv7182_s_routing,
	.querystd = adv7182_querystd,
	.g_mbus_config  = adv7182_g_mbus_config,
	.s_stream = adv7182_s_stream,
};

static struct v4l2_subdev_ops adv7182_subdev_ops = {
	.core = &adv7182_core_ops,
	.video = &adv7182_video_ops,
};

/***********************************************************************
 * I2C client and driver.
 ***********************************************************************/

/*! adv7182 Reset function.
 *
 *  @return		None.
 */
static int adv7182_hard_reset(struct adv7182_dev *sensor)
{
	int count = 0;
	dev_dbg(sensor->dev,
		"In adv7182:adv7182_hard_reset\n");

	while (count < ADV7182_MAX_REG) {
		if (adv7182_write_reg(sensor, adv7182_def_reg_val[count][0],
				      adv7182_def_reg_val[count][1]) != 0)
			return -EIO;
		count++;
	}
	sensor->current_input = ADV7182_INPUT_INDEX_DEF;

	return ADV7182_SUCCESS;
}

static int setup_irq_adv7182(struct adv7182_dev *sensor)
{
	int ret = ADV7182_SUCCESS;
	if (sensor->i2c_client->irq) {
		ret = request_threaded_irq(sensor->i2c_client->irq,
					   NULL, adv7182_interrupt,
					   IRQF_TRIGGER_LOW | IRQF_ONESHOT,
					   IF_NAME, sensor);
		if (ret < 0) {
			dev_err(sensor->dev,
				"Failed to register irq %d\n",
				sensor->i2c_client->irq);
			return ret;
		}

		adv7182_enable_interrupts(sensor);

		dev_info(sensor->dev,
			 "Registered irq %d\n",
			 sensor->i2c_client->irq);
	}
	return ret;
}

/*!
 * adv7182 I2C probe function.
 * Function set in i2c_driver struct.
 * Called by insmod.
 *
 *  @param *client	I2C client descriptor.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7182_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	struct device_node *endpoint;
	struct adv7182_dev *sensor;
	struct device_node *np;
	const char *norm = "pal";
	s32 gpio_rst;
	int ret = 0;

	sensor = devm_kzalloc(&client->dev, sizeof(struct adv7182_dev),
			      GFP_KERNEL);
	if (!sensor)
		return -ENOMEM;

	sensor->dev = &client->dev;
	np = sensor->dev->of_node;

	ret = of_property_read_string(np, "default-std", &norm);
	if (ret < 0 && ret != -EINVAL) {
		dev_err(sensor->dev, "error reading default-std property!\n");
		return ret;
	}
	if (!strcasecmp(norm, "pal")) {
		sensor->std_id = V4L2_STD_PAL;
		sensor->video_idx = adv7182_PAL;
		dev_info(sensor->dev, "defaulting to PAL!\n");
	} else if (!strcasecmp(norm, "ntsc")) {
		sensor->std_id = V4L2_STD_NTSC;
		sensor->video_idx = adv7182_NTSC;
		dev_info(sensor->dev, "defaulting to NTSC!\n");
	} else {
		dev_err(sensor->dev, "invalid default-std value: '%s'!\n",
			norm);
		return -EINVAL;
	}

	sensor->i2c_client = client;
	sensor->streamcap.timeperframe.denominator = 30;
	sensor->streamcap.timeperframe.numerator = 1;
	sensor->fmt.width = video_fmts[sensor->video_idx].raw.width;
	sensor->fmt.height = video_fmts[sensor->video_idx].raw.height;
	sensor->fmt.code = V4L2_MBUS_FMT_UYVY8_2X8;
	sensor->fmt.field = V4L2_FIELD_SEQ_BT;

	mutex_init(&sensor->mutex);

	endpoint = v4l2_of_get_next_endpoint(np, NULL);
	if (!endpoint) {
		dev_err(sensor->dev, "endpoint node not found\n");
		return -EINVAL;
	}

	v4l2_of_parse_endpoint(endpoint, &sensor->ep);
	if (sensor->ep.bus_type != V4L2_MBUS_BT656) {
		dev_err(sensor->dev, "invalid bus type, must be bt.656\n");
		return -EINVAL;
	}
	of_node_put(endpoint);

	dev_dbg(sensor->dev, "%s:adv7182 probe i2c address is 0x%02X\n",
		__func__, sensor->i2c_client->addr);
	gpio_rst = of_get_named_gpio(np, "rst-gpios", 0);
	sensor->gpio_pwrdn = of_get_named_gpio(np, "pwr-gpios", 0);
	if (gpio_rst >= 0) {
		ret = adv7182_gpio_initialise(
			sensor, gpio_rst,
			(u32)ADV7182_GPIO_DIR_OUT,
			(u32)ADV7182_RESET_ACTIVE,
			ADV7182_GPIO_RST_NAME);
		if (ret != ADV7182_SUCCESS) {
			dev_err(sensor->dev,
				"ADV7182:reset pin initialization fail\n");
			goto err;
		}
	} else {
		dev_err(sensor->dev,
			"ADV7182:GPIO reset pin property not exist\n");
		ret = gpio_rst;
		goto err;
	}

	if (sensor->gpio_pwrdn >= 0) {
		ret = adv7182_gpio_initialise(
			sensor, sensor->gpio_pwrdn,
			(u32)ADV7182_GPIO_DIR_OUT,
			(u32)ADV7182_PWRDN_ACTIVE,
			ADV7182_GPIO_PWR_DN_NAME);
		if (ret != ADV7182_SUCCESS) {
			gpio_free(gpio_rst);
			dev_err(sensor->dev,
				"ADV7182:powerdown pin initialization fail\n");
			goto err;
		}

		/*Reset the device ADV7182*/
		gpio_set_value(gpio_rst, ADV7182_RESET_ACTIVE);
		gpio_set_value(sensor->gpio_pwrdn, ADV7182_PWRDN_ACTIVE);
		usleep_range(5000, 10000);
		gpio_set_value(gpio_rst, ADV7182_RESET_DEACTIVE);
		usleep_range(5000, 10000);
		gpio_set_value(sensor->gpio_pwrdn, ADV7182_PWRDN_DEACTIVE);
		gpio_free(gpio_rst);
	} else {
		gpio_free(gpio_rst);
		dev_err(sensor->dev,
			"ADV7182:GPIO powerdown pin property not exist\n");
		ret = sensor->gpio_pwrdn;
		goto err;
	}

	/* Power on the adv7182 chip */
	if ((adv7182_power(sensor, true) != ADV7182_SUCCESS)) {
		dev_err(sensor->dev, "ADV7182:Power ON failed\n");
		ret = -EIO;
		goto cleanup;
	}

	/* Read the revision ID of the ADV7182 chip */
	sensor->rev_id = adv7182_read(sensor, ADV_REG_IDENT);
	if (sensor->rev_id < 0) {
		ret = -ENODEV;
		dev_err(sensor->dev, "ADV7182: Revision ID read failed\n");
		goto cleanup;
	}

	dev_dbg(sensor->dev, "%s:Analog Device adv7%2X0 detected!\n",
		__func__, sensor->rev_id);

	/*! adv7182 initialization. */
	ret = adv7182_hard_reset(sensor);
	if (ret != ADV7182_SUCCESS) {
		dev_err(sensor->dev,
			"ADV7182:chip initialization failed\n");
		goto cleanup;
	}

	v4l2_i2c_subdev_init(&sensor->sd, client, &adv7182_subdev_ops);

	/* see if there is a signal lock already */
	adv7182_update_lock_status(sensor);
	adv7182_get_autodetect_std(sensor);

	ret = setup_irq_adv7182(sensor);
	if (!ret)
		return ret;

	return adv7182_init_controls(sensor);

cleanup:
	gpio_free(sensor->gpio_pwrdn);
err:
	return ret;
}

/*!
 * adv7182 I2C detach function.
 * Called on rmmod.
 *
 *  @param *client	struct i2c_client*.
 *
 *  @return		Error code indicating success or failure.
 */
static int adv7182_detach(struct i2c_client *client)
{
	struct v4l2_subdev *sd = i2c_get_clientdata(client);
	struct adv7182_dev *sensor = to_adv7182_dev(sd);

	dev_dbg(sensor->dev,
		"%s:Removing %s video decoder @ 0x%02X from adapter %s\n",
		__func__, IF_NAME, client->addr << 1, client->adapter->name);

	if (sensor->i2c_client->irq)
		free_irq(sensor->i2c_client->irq, sensor);

	v4l2_ctrl_handler_free(&sensor->ctrl_hdl);

	/* Power off the adv7182 chip */
	if ((adv7182_power(sensor, false) != ADV7182_SUCCESS))
		dev_err(sensor->dev, "ADV7182:Power OFF failed\n");

	if (gpio_is_valid(sensor->gpio_pwrdn))
		gpio_free(sensor->gpio_pwrdn);

	return ADV7182_SUCCESS;
}

static const struct i2c_device_id adv7182_id[] = {
	{ "adv7182", 0 },
	{}
};
MODULE_DEVICE_TABLE(i2c, adv7182_id);

static struct of_device_id adv7182_dt_ids[] = {
	{ .compatible = "adi,adv7182" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, adv7182_dt_ids);

static struct i2c_driver adv7182_driver = {
	.driver = {
		.name	= "adv7182",
		.owner	= THIS_MODULE,
		.of_match_table	= adv7182_dt_ids,
	},
	.id_table	= adv7182_id,
	.probe		= adv7182_probe,
	.remove		= adv7182_detach,
};

module_i2c_driver(adv7182_driver);

MODULE_AUTHOR("Robert Bosch Engineering and Business solutions Ltd");
MODULE_DESCRIPTION("Analog Device adv7182 video decoder driver");
MODULE_LICENSE("GPL");
