/*
 * drivers/input/misc/input-inc.c
 *
 * Input devices over INC
 *
 *   Copyright (c) 2013 Robert Bosch GmbH, Hildesheim
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/slab.h>
#include <linux/input.h>
#include <linux/input/mt.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inc.h>
#include <linux/inc_ports.h>
#include <linux/inc_scc_input_device.h>
#include <linux/dgram_service.h>
#include <linux/kthread.h>

MODULE_DESCRIPTION("INC input device driver");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Matthias Thomae <matthias.thomae@de.bosch.com>");

#define INPUT_INC_DRV_NAME		"input-inc"
#define INPUT_INC_PROTOCOL_VERSION_CUR	2
#define INPUT_INC_PROTOCOL_VERSION_MIN	1
#define INPUT_INC_RX_BUFSIZE		512
#define INPUT_INC_DGRAM_MAX		INPUT_INC_RX_BUFSIZE
#define INPUT_INC_MAX_DEVICES		32
#define INPUT_INC_STATUS_ACTIVE		0x01
#define INPUT_INC_STATUS_INACTIVE	0x02
#define INPUT_INC_PEER_RESPONSE_TIMEOUT	(1*(HZ)) /* 1 second */

struct input_inc {
	struct device		*dev;
	struct input_dev	*indev[INPUT_INC_MAX_DEVICES];
	struct task_struct	*rx_task;
	wait_queue_head_t	waitq;
	struct mutex		lock;
	struct socket		*sock;
	struct sk_dgram		*dgram;
	struct sockaddr_in	local;
	struct sockaddr_in	remote;
	int			is_active;
	int			is_configured;
	int			num_devices;
	int			protocol_version;
};

static unsigned int inet_addr(char *str)
{
	int a, b, c, d;
	char arr[4];
	sscanf(str, "%d.%d.%d.%d", &a, &b, &c, &d);
	arr[0] = a; arr[1] = b; arr[2] = c; arr[3] = d;
	return *(unsigned int *) arr;
}

#ifndef CONFIG_OF
struct platform_device *input_inc_device;
#endif

#ifdef CONFIG_OF
static int input_inc_get_dt_properties(struct input_inc *input_inc)
{
	struct device_node *input_node = input_inc->dev->of_node;
	struct device_node *inc_node;
	const char *prop_str, *inc_node_path;
	const void *prop;

	input_inc->local.sin_family = AF_INET;
	input_inc->remote.sin_family = AF_INET;

	prop_str = "node";
	prop = of_get_property(input_node, prop_str, NULL);
	if (prop)
		inc_node_path = (char *) prop;
	else
		goto err;

	inc_node = of_find_node_by_path(inc_node_path);
	if (!inc_node) {
		dev_err(input_inc->dev, "can't find %s node in device tree\n",
				inc_node_path);
		return -1;
	}

	prop_str = "local-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		input_inc->local.sin_addr.s_addr = inet_addr((char *) prop);
	else
		goto err;

	prop_str = "remote-addr";
	prop = of_get_property(inc_node, prop_str, NULL);
	if (prop)
		input_inc->remote.sin_addr.s_addr = inet_addr((char *) prop);
	else
		goto err;

	return 0;

err:
	dev_err(input_inc->dev, "could not read dt property: %s\n", prop_str);
	return -1;
}
#endif

static int input_inc_sock_open(struct input_inc *input_inc)
{
	struct socket *sock = NULL;
	struct sk_dgram *dgram;
	int ret;

	dev_dbg(input_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
			&input_inc->local.sin_addr.s_addr,
			ntohs(input_inc->local.sin_port),
			&input_inc->remote.sin_addr.s_addr,
			ntohs(input_inc->remote.sin_port));

	ret = sock_create_kern(AF_INC, SOCK_STREAM, 0, &sock);
	if (ret < 0) {
		dev_err(input_inc->dev, "%s: sock_create_kern failed: %d\n",
				__func__, ret);
		goto out;
	}

	dgram = dgram_init(sock, INPUT_INC_DGRAM_MAX, NULL);
	if (dgram == NULL) {
		dev_err(input_inc->dev, "%s: dgram_init failed\n", __func__);
		goto out;
	}

	ret = kernel_bind(sock, (struct sockaddr *) &input_inc->local,
			sizeof(input_inc->local));
	if (ret < 0) {
		dev_err(input_inc->dev, "%s: kernel_bind failed: %d\n",
				__func__, ret);
		goto out_release;
	}

	ret = kernel_connect(sock, (struct sockaddr *) &input_inc->remote,
			sizeof(input_inc->remote), 0);
	if (ret < 0) {
		dev_err(input_inc->dev, "%s: kernel_connect failed: %d\n",
				__func__, ret);
		goto out_release;
	}

	input_inc->sock = sock;
	input_inc->dgram = dgram;

	return 0;

out_release:
	sock_release(sock);
out:
	return ret;
}

static void input_inc_sock_close(struct input_inc *input_inc)
{
	dev_dbg(input_inc->dev, "%s: local %pI4:%d remote %pI4:%d\n", __func__,
			&input_inc->local.sin_addr.s_addr,
			ntohs(input_inc->local.sin_port),
			&input_inc->remote.sin_addr.s_addr,
			ntohs(input_inc->remote.sin_port));

	if (input_inc->dgram) {
		dev_dbg(input_inc->dev, "%s: dgram_exit\n", __func__);
		dgram_exit(input_inc->dgram);
	}

	if (input_inc->sock) {
		dev_dbg(input_inc->dev, "%s: sock_shutdown\n", __func__);
		kernel_sock_shutdown(input_inc->sock, SHUT_RDWR);
	}

	if (input_inc->sock) {
		dev_dbg(input_inc->dev, "%s: sock_release\n", __func__);
		sock_release(input_inc->sock);
	}

	return;
}

static int input_inc_sendmsg(struct input_inc *input_inc, u8 *data, int len)
{
	int ret, i;

	dev_dbg(input_inc->dev, "%s: dgram_send to %pI4:%d\n", __func__,
			&input_inc->remote.sin_addr.s_addr,
			ntohs(input_inc->remote.sin_port));

	ret = dgram_send(input_inc->dgram, data, len);
	if (ret < 0)
		dev_err(input_inc->dev, "%s: could not send msg: %d\n",
				__func__, ret);

	for (i = 0; i < len; i++)
		dev_dbg(input_inc->dev, "%s: data[%d]=0x%02X\n", __func__,
				i, ((u8 *)data)[i]);

	return ret;
}

static int input_inc_recvmsg(struct input_inc *input_inc, void *data, int len)
{
	int ret, i;

	do {
		ret = dgram_recv(input_inc->dgram, data, len);
		if ((ret < 0) && (ret != -EAGAIN))
			dev_alert(input_inc->dev, "%s: dgram_recv failed: %d\n",
					__func__, ret);

		dev_dbg(input_inc->dev, "%s: dgram_recv: ret=%d\n",
				__func__, ret);
		for (i = 0; i < ret; i++)
			dev_dbg(input_inc->dev, "%s: data[%d]=0x%02X\n",
					__func__, i, ((u8 *)data)[i]);
	} while (ret == -EAGAIN);

	return ret;
}

static int input_inc_cmd_status_active(struct input_inc *input_inc)
{
	u8 data[3] = {SCC_INPUT_DEVICE_C_COMPONENT_STATUS_MSGID,
			INPUT_INC_STATUS_ACTIVE,
			INPUT_INC_PROTOCOL_VERSION_CUR};

	return input_inc_sendmsg(input_inc, data, 3);
}

static int input_inc_cmd_get_config(struct input_inc *input_inc)
{
	u8 data[1] = {SCC_INPUT_DEVICE_C_CONFIG_START_MSGID};

	return input_inc_sendmsg(input_inc, data, 1);
}

static int input_inc_config_start(struct input_inc *input_inc, u8 num_devices)
{
	struct input_dev *indev;
	int i;

	dev_dbg(input_inc->dev, "%s\n", __func__);

	for (i = 0; i < num_devices; i++) {
		indev = input_allocate_device();
		if (!indev) {
			dev_err(input_inc->dev,
					"%s: failed to allocate memory\n",
					__func__);
			return -ENOMEM;
		}
		input_inc->indev[i] = indev;
	}

	input_inc->num_devices = num_devices;

	return 0;
}

static int input_inc_config_type(struct input_inc *input_inc,
		u8 *data, int size)
{
	u8 *pos = data + 1;
	u8 devid, ncodes;
	u16 type, code;
	int expected_size, i;
	unsigned long *bit;
	struct input_dev *indev;

	devid = *pos;
	pos += sizeof(devid);

	if (devid >= input_inc->num_devices) {
		dev_alert(input_inc->dev, "%s: invalid device id: %d\n",
				__func__, devid);
		return -EINVAL;
	}
	indev = input_inc->indev[devid];

	type = *(u16 *) pos;
	pos += sizeof(type);

	ncodes = *pos;
	pos += sizeof(ncodes);

	dev_dbg(input_inc->dev, "%s: devid %d evtype 0x%02X ncodes %d\n",
			__func__, devid, type, ncodes);

	expected_size = (pos - data) + ncodes * sizeof(code);
	/* TBD: change this back to (size != expected_size)
	 * once msg size is configurable on peer
	 */
	if (size < expected_size) {
		dev_alert(input_inc->dev, "%s: msg size mismatch: %d < %d\n",
				__func__, size, expected_size);
		return -EINVAL;
	}

	switch (type) {
	case EV_KEY:
		bit = indev->keybit;
		break;
	case EV_REL:
		bit = indev->relbit;
		break;
	case EV_ABS:
		bit = indev->absbit;
		break;
	default:
		dev_alert(input_inc->dev, "%s: invalid type: 0x%02X\n",
				__func__, type);
		return -EINVAL;
	}

	indev->evbit[BIT_WORD(type)] |= BIT_MASK(type);

	for (i = 0; i < ncodes; i++) {
		code = *(u16 *) pos;
		pos += sizeof(code);

		if ((type == EV_KEY && code > KEY_MAX)
			|| (type == EV_REL && code > REL_MAX)
			|| (type == EV_ABS && code > ABS_MAX)) {
			dev_err(input_inc->dev,
					"%s: dev %d type %d: invalid code 0x%x",
					__func__, devid, type, code);
			continue;
		}

		dev_dbg(input_inc->dev, "%s: code: 0x%x", __func__, code);
		bit[BIT_WORD(code)] |= BIT_MASK(code);
	}

	return 0;
}

static int input_inc_config_abs(struct input_inc *input_inc, u8 *data)
{
	u8 *pos = data + 1;
	u8 devid;
	u16 axis;
	s32 min, max, fuzz, flat;

	devid = *pos;
	pos += sizeof(devid);

	if (devid >= input_inc->num_devices) {
		dev_alert(input_inc->dev, "%s: invalid device id: %d\n",
				__func__, devid);
		return -EINVAL;
	}

	axis = *(u16 *) pos;
	pos += sizeof(axis);
	min = *(s32 *) pos;
	pos += sizeof(min);
	max = *(s32 *) pos;
	pos += sizeof(max);
	fuzz = *(s32 *) pos;
	pos += sizeof(fuzz);
	flat = *(s32 *) pos;
	pos += sizeof(flat);

	dev_dbg(input_inc->dev, "%s: axis %d min %d max %d fuzz %d flat %d\n",
			__func__, axis, min, max, fuzz, flat);

	input_set_abs_params(input_inc->indev[devid],
			axis, min, max, fuzz, flat);

	return 0;
}

static int input_inc_config_mt_slots(struct input_inc *input_inc, u8 *data)
{
	u8 *pos = data + 1;
	u8 devid;
	u8 nslots;

	devid = *pos;
	pos += sizeof(devid);

	if (devid >= input_inc->num_devices) {
		dev_alert(input_inc->dev, "%s: invalid device id: %d\n",
				__func__, devid);
		return -EINVAL;
	}

	nslots = *(u8 *) pos;
	pos += sizeof(nslots);

	dev_dbg(input_inc->dev, "%s: nslots %d\n", __func__, nslots);

	return input_mt_init_slots(input_inc->indev[devid], nslots, 0);
}

static int input_inc_config_end(struct input_inc *input_inc)
{
	int i, ret = 0;

	dev_dbg(input_inc->dev, "%s\n", __func__);

	for (i = 0; i < input_inc->num_devices; i++) {
#ifdef CONFIG_OF
		input_inc->indev[i]->name = input_inc->dev->of_node->name;
#else
		input_inc->indev[i]->name = INPUT_INC_DRV_NAME;
#endif
		input_inc->indev[i]->id.bustype = BUS_SPI;
		input_inc->indev[i]->id.version = i;
		input_inc->indev[i]->dev.parent = input_inc->dev;

		ret = input_register_device(input_inc->indev[i]);
		if (ret) {
			dev_alert(input_inc->dev, "%s: register %d failed %d\n",
					__func__, i, ret);
			return -EINVAL;
		}
	}

	input_inc->is_configured = 1;
	wake_up_interruptible(&input_inc->waitq);

	return 0;
}

static int input_inc_report_event(struct input_inc *input_inc,
		u8 *data, int device_id, int size)
{
	u8 *pos = data + 1;
	u8 devid, nevents, slotid, slotstate, tooltype;
	u16 type, code;
	s32 value;
	int expected_size, i;
	struct input_dev *indev;

	if (input_inc->protocol_version == INPUT_INC_PROTOCOL_VERSION_CUR) {
		devid = device_id;
	} else {
		devid = *pos;
		pos += sizeof(devid);
	}

	if (devid >= input_inc->num_devices) {
		dev_err(input_inc->dev, "%s: invalid device id: %d\n",
				__func__, devid);
		return -EINVAL;
	}
	indev = input_inc->indev[devid];

	nevents = *pos;
	pos += sizeof(nevents);

	dev_dbg(input_inc->dev, "%s: devid %d nevents %d\n",
			__func__, devid, nevents);

	expected_size = (pos - data) + nevents *
			(sizeof(type) + sizeof(code) + sizeof(value));
	/* TBD: change this back to (size != expected_size)
	 * once msg size is configurable on peer
	 */
	if (size < expected_size) {
		dev_err(input_inc->dev, "%s: msg size mismatch: %d < %d\n",
				__func__, size, expected_size);
		return -EINVAL;
	}

	for (i = 0; i < nevents; i++) {
		type = *(u16 *) pos;
		pos += sizeof(type);
		code = *(u16 *) pos;
		pos += sizeof(code);
		value = *(s32 *) pos;
		pos += sizeof(value);

		dev_dbg(input_inc->dev, "%s: type %d code %d value %d",
				__func__, type, code, value);

		/* if type/code checking/restriction is not required,
		 * the following could be replaced with:
		 * input_event(indev, type, code, value);
		 */
		switch (type) {
		case EV_SYN:
			switch (code) {
			case SYN_REPORT:
				input_sync(indev);
				break;
			case SYN_MT_REPORT:
				input_mt_sync(indev);
				break;
			default:
				dev_err(input_inc->dev,
						"%s: invalid code: 0x%02X\n",
						__func__, code);
				return -EINVAL;
			}
			break;
		case EV_KEY:
			input_report_key(indev, code, value);
			break;
		case EV_REL:
			input_report_rel(indev, code, value);
			break;
		case EV_ABS:
			switch (code) {
			case ABS_MT_SLOT:
				pos -= sizeof(value);
				slotid = *(u8 *) pos;
				pos += sizeof(slotid);
				slotstate = *(u8 *) pos;
				pos += sizeof(slotstate);
				tooltype = *(u8 *) pos;
				pos += sizeof(tooltype) + sizeof(u8);
				dev_dbg(input_inc->dev,
						"%s: slot %d state %d tool %d",
						__func__, slotid, slotstate,
						tooltype);

				input_mt_slot(indev, slotid);
				input_mt_report_slot_state(indev, tooltype,
						slotstate);
				break;
			default:
				input_report_abs(indev, code, value);
				break;
			}
			break;
		default:
			dev_err(input_inc->dev, "%s: invalid type: 0x%02X\n",
					__func__, type);
			return -EINVAL;
		}
	}

	input_sync(indev);

	return 0;
}

static int input_inc_handle_msg(struct input_inc *input_inc, u8 *data, int size)
{
	int ret = 0;
	struct device *dev = input_inc->dev;
	u8 num_devices;

	if (size <= 0) {
		dev_alert(dev, "%s: invalid msg size: %d\n",
				__func__, size);
		return -EINVAL;
	}

	/* before activation discard all non-activation messages */
	if (data[0] != SCC_INPUT_DEVICE_R_COMPONENT_STATUS_MSGID
			&& !input_inc->is_active) {
		dev_dbg(dev, "%s: discarding msgid: 0x%02X\n",
				__func__, data[0]);
		return 0;
	}

	switch (data[0]) {
	case SCC_INPUT_DEVICE_R_COMPONENT_STATUS_MSGID:
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		if (data[1] != INPUT_INC_STATUS_ACTIVE)
			dev_alert(dev, "%s: invalid status: %d\n",
					__func__, data[1]);
		if (data[2] > INPUT_INC_PROTOCOL_VERSION_CUR ||
				data[2] < INPUT_INC_PROTOCOL_VERSION_MIN) {
			dev_alert(dev, "%s: invalid protocol version: %d\n",
					__func__, data[2]);
			return -EINVAL;
		}
		input_inc->protocol_version = data[2];
		input_inc->is_active = 1;
		wake_up_interruptible(&input_inc->waitq);
		break;
	case SCC_INPUT_DEVICE_R_CONFIG_START_MSGID:
		if (size != 2) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		if (data[1] > INPUT_INC_MAX_DEVICES) {
			dev_alert(dev, "%s: too many num_devices: %d -\n",
					__func__, data[1]);
			dev_alert(dev, "%s: - using max num_devices: %d\n",
					__func__, INPUT_INC_MAX_DEVICES);
			num_devices = INPUT_INC_MAX_DEVICES;
		} else {
			num_devices = data[1];
		}
		input_inc_config_start(input_inc, num_devices);
		break;
	case SCC_INPUT_DEVICE_R_CONFIG_TYPE_MSGID:
		if (size < 5) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		ret = input_inc_config_type(input_inc, data, size);
		break;
	case SCC_INPUT_DEVICE_R_CONFIG_ABS_MSGID:
		if (size != 20) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		ret = input_inc_config_abs(input_inc, data);
		break;
	case SCC_INPUT_DEVICE_R_CONFIG_MT_SLOTS_MSGID:
		if (size != 3) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		ret = input_inc_config_mt_slots(input_inc, data);
		break;
	case SCC_INPUT_DEVICE_R_CONFIG_END_MSGID:
		if (size != 1) {
			dev_alert(dev, "%s: invalid msg size: 0x%02X %d\n",
					__func__, data[0], size);
			return -EINVAL;
		}
		ret = input_inc_config_end(input_inc);
		break;
	case SCC_INPUT_DEVICE_R_EVENT_DEV0_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV1_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV2_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV3_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV4_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV5_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV6_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV7_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV8_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV9_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV10_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV11_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV12_MSGID:
	case SCC_INPUT_DEVICE_R_EVENT_DEV13_MSGID:
		if (!input_inc->is_configured) {
			dev_warn(dev, "%s: cfg ongoing, drop evt: 0x%02X %d\n",
					__func__, data[0], size);
			return 0;
		}
		if (input_inc->protocol_version ==
				INPUT_INC_PROTOCOL_VERSION_CUR) {
			if (size < 10) {
				dev_err(dev, "%s: invalid msgsize: 0x%02X %d\n",
						__func__, data[0], size);
				return 0;
			}
		} else {
			if (data[0] != SCC_INPUT_DEVICE_R_EVENT_DEV0_MSGID) {
				dev_err(dev, "%s: invalid msgid: 0x%02X\n",
						__func__, data[0]);
				return 0;
			}
			if (size < 11) {
				dev_err(dev, "%s: invalid msgsize: 0x%02X %d\n",
						__func__, data[0], size);
				return 0;
			}

		}
		input_inc_report_event(input_inc, data,
				(data[0] - SCC_INPUT_DEVICE_R_EVENT_DEV0_MSGID)
				/ 2, size);
		break;
	default:
		dev_alert(dev, "%s: invalid msgid: 0x%02X\n",
				__func__, data[0]);
		return 0;
	}

	return ret;
}

static int input_inc_rx_task(void *arg)
{
	u8 data[INPUT_INC_RX_BUFSIZE];
	int ret;

	struct input_inc *input_inc = arg;

	dev_dbg(input_inc->dev, "%s: enter\n", __func__);

	do {
		ret = input_inc_recvmsg(input_inc, data, INPUT_INC_RX_BUFSIZE);
		if (ret >= 0) {
			ret = input_inc_handle_msg(input_inc, data, ret);
			if (ret)
				panic("input-inc: handle_msg failed: %i\n",
						ret);
		}
	} while (ret >= 0);

	dev_dbg(input_inc->dev, "%s: exit: %d\n", __func__, ret);

	mutex_lock(&input_inc->lock);
	input_inc->rx_task = NULL;
	mutex_unlock(&input_inc->lock);

	return 0;
}

static int input_inc_remove(struct platform_device *pdev)
{
	struct input_inc *input_inc = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;
	int i, ret = 0;

	if (!input_inc)
		goto out;

	dev_dbg(dev, "%s:\n", __func__);

	mutex_lock(&input_inc->lock);
	if (input_inc->rx_task)
		force_sig(SIGKILL, input_inc->rx_task);
	else
		dev_dbg(dev, "%s: rx_task already terminated\n", __func__);
	mutex_unlock(&input_inc->lock);

	if (input_inc->rx_task)
		kthread_stop(input_inc->rx_task);

	input_inc_sock_close(input_inc);

	for (i = 0; i < input_inc->num_devices; i++) {
		if (input_inc->indev[i]) {
			if (input_inc->is_configured)
				input_unregister_device(input_inc->indev[i]);
			else
				input_free_device(input_inc->indev[i]);
			input_inc->indev[i] = NULL;
		}
	}

	platform_set_drvdata(pdev, NULL);
	kfree(input_inc);

out:
	return ret;
}

#ifdef CONFIG_OF
static struct of_device_id inc_input_match[] = {
	{ .compatible = "rbcm,input-inc", },
	{ /* this can be filled with module_param */ },
	{}
};

module_param_string(of_id, inc_input_match[1].compatible, 128, 0);
MODULE_PARM_DESC(of_id, "compatible devices to be handled additionally");
#endif

static int input_inc_probe(struct platform_device *pdev)
{
	struct input_inc *input_inc;
	struct task_struct *task;
	int ret;
	struct device *dev = &pdev->dev;
#ifdef CONFIG_OF
	const struct of_device_id *id =	of_match_device(inc_input_match, dev);

	if (!id) {
		dev_err(dev, "%s: failed to find input device\n", __func__);
		ret = -EFAULT;
		goto out;
	}

	dev_info(dev, "%s: compatible: %s\n", __func__, id->compatible);
#endif

	input_inc = kzalloc(sizeof(*input_inc), GFP_KERNEL);
	if (!input_inc) {
		dev_err(dev, "%s: failed to allocate memory for device\n",
				__func__);
		ret = -ENOMEM;
		goto out;
	}

	input_inc->dev = dev;
	input_inc->rx_task = NULL;
	input_inc->sock = NULL;
	input_inc->dgram = NULL;
	input_inc->is_active = 0;
	input_inc->is_configured = 0;
	input_inc->num_devices = 0;
	input_inc->protocol_version = 0;

	input_inc->local.sin_port = htons(INPUT_DEVICE_PORT);
	input_inc->remote.sin_port = input_inc->local.sin_port;

	mutex_init(&input_inc->lock);

	init_waitqueue_head(&input_inc->waitq);

	platform_set_drvdata(pdev, input_inc);
#ifdef CONFIG_OF
	ret = input_inc_get_dt_properties(input_inc);
	if (ret < 0)
		goto out_remove;
#else
	input_inc->local.sin_family = AF_INET;
	input_inc->remote.sin_family = AF_INET;
	input_inc->local.sin_addr.s_addr = inet_addr(CONFIG_INPUT_INC_LOCAL_IP);
	input_inc->remote.sin_addr.s_addr =
			inet_addr(CONFIG_INPUT_INC_REMOTE_IP);
#endif

	ret = input_inc_sock_open(input_inc);
	if (ret < 0)
		goto out_remove;

	/* create and start thread */
	task = kthread_run(input_inc_rx_task, input_inc, INPUT_INC_DRV_NAME);
	if (IS_ERR(task)) {
		ret = PTR_ERR(task);
		dev_err(dev, "%s: could not run thread: %d\n", __func__, ret);
		goto out_remove;
	}
	input_inc->rx_task = task;

	ret = input_inc_cmd_status_active(input_inc);
	if (ret < 0)
		goto out_remove;

	ret = wait_event_interruptible_timeout(input_inc->waitq,
			input_inc->is_active,
			INPUT_INC_PEER_RESPONSE_TIMEOUT);
	if (ret < 0)
		goto out_remove;
	if (ret == 0) {
		dev_alert(input_inc->dev,
				"%s: R_COMPONENT_STATUS timeout",
				__func__);
		ret = -ETIME;
		goto out_remove;
	}

	ret = input_inc_cmd_get_config(input_inc);
	if (ret < 0)
		goto out_remove;

	ret = wait_event_interruptible_timeout(input_inc->waitq,
			input_inc->is_configured,
			INPUT_INC_PEER_RESPONSE_TIMEOUT);
	if (ret < 0)
		goto out_remove;
	if (ret == 0) {
		dev_alert(input_inc->dev,
				"%s: R_CONFIG_END timeout",
				__func__);
		ret = -ETIME;
		goto out_remove;
	}

	return 0;

out_remove:
	input_inc_remove(pdev);
out:
	platform_set_drvdata(pdev, NULL);
	return ret;
}

static struct platform_driver input_inc_driver = {
	.driver = {
		.name	= INPUT_INC_DRV_NAME,
		.owner	= THIS_MODULE,
#ifdef CONFIG_OF
		.of_match_table = inc_input_match,
#endif
	},
	.probe		= input_inc_probe,
	.remove		= input_inc_remove,
};

static int __init input_inc_init(void)
{
	int ret =  0;
	pr_debug("%s: enter\n", __func__);
	ret = platform_driver_register(&input_inc_driver);

#ifndef CONFIG_OF
	if (!ret) {
		input_inc_device =
				platform_device_alloc(INPUT_INC_DRV_NAME, -1);
		if (input_inc_device) {
			ret = platform_device_add(input_inc_device);
			if (ret)
				platform_device_put(input_inc_device);
		} else {
			pr_err("Input_inc pf_dev_alloc failed ret=%d\n", ret);
			ret = -1;
		}
		if (ret) {
			platform_driver_unregister(&input_inc_driver);
			pr_err("Input_inc Module init failed ret=%d\n", ret);
		}
	} else
		pr_err("%s: plat_drv_reg failed: %d\n", __func__, ret);
#endif

	pr_debug("%s: exit\n", __func__);
	return ret;

}
late_initcall(input_inc_init);

static void __exit input_inc_exit(void)
{
#ifndef CONFIG_OF
	if (input_inc_device)
		platform_device_unregister(input_inc_device);
#endif
	platform_driver_unregister(&input_inc_driver);
}
module_exit(input_inc_exit);
