/*
 * i.MX drm driver - parallel display implementation
 *
 * Copyright (C) 2012 Sascha Hauer, Pengutronix
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/pinctrl/consumer.h>
#include <drm/drmP.h>
#include <drm/drm_fb_helper.h>
#include <drm/drm_crtc_helper.h>
#include <video/display_timing.h>
#include <video/of_display_timing.h>
#include <video/videomode.h>
#include <linux/videodev2.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include "imx-drm.h"

#define con_to_imxpd(x) container_of(x, struct imx_parallel_display, connector)
#define enc_to_imxpd(x) container_of(x, struct imx_parallel_display, encoder)

struct imx_parallel_display {
	struct drm_connector connector;
	struct imx_drm_connector *imx_drm_connector;
	struct drm_encoder encoder;
	struct imx_drm_encoder *imx_drm_encoder;
	struct display_timings *timings;
	struct device *dev;
	void *edid;
	int edid_len;
	u32 interface_pix_fmt;
	u32 interface_map[6];
	u32 *interface_mapptr;
	int reset_pin;
	enum of_gpio_flags reset_flags;
};

/*
 * Some parallel displays (broken?) require the display interface signals
 * (enable data, clock, HYSNC, VSYNC) to be started with 100us after the
 * reset signal is set inactive. Hence the reset of the display is set
 * inactive just before the display interface is made active in the crtc
 * driver. So this function needs to be called in two places where the
 * crtc is activated: in mode_set() and dpms ON.
 */
static void imx_pd_set_reset_gpio(struct imx_parallel_display *imxpd)
{
	if (gpio_is_valid(imxpd->reset_pin))
		gpio_set_value(imxpd->reset_pin,
			       (imxpd->reset_flags == OF_GPIO_ACTIVE_LOW) ?
			       0 : 1);
}

static enum drm_connector_status imx_pd_connector_detect(
		struct drm_connector *connector, bool force)
{
	return connector_status_connected;
}

static void imx_pd_connector_destroy(struct drm_connector *connector)
{
	/* do not free here */
}

static int imx_pd_connector_get_modes(struct drm_connector *connector)
{
	struct imx_parallel_display *imxpd = con_to_imxpd(connector);
	struct display_timings *timings = imxpd->timings;
	struct drm_display_mode *mode;
	struct videomode vm;
	int i, num_modes = 0;

	if (imxpd->edid) {
		drm_mode_connector_update_edid_property(connector, imxpd->edid);
		num_modes = drm_add_edid_modes(connector, imxpd->edid);
	}

	if (timings) {
		for (i = 0; i < timings->num_timings; i++) {
			if (videomode_from_timings(timings, &vm, i))
				break;

			mode = drm_mode_create(connector->dev);
			drm_display_mode_from_videomode(&vm, mode);

			mode->type = DRM_MODE_TYPE_DRIVER;
			if (i == timings->native_mode)
				mode->type |= DRM_MODE_TYPE_PREFERRED;

			drm_mode_set_name(mode);
			drm_mode_probed_add(connector, mode);
			num_modes++;
		}
	}

	return num_modes;
}

static int imx_pd_connector_mode_valid(struct drm_connector *connector,
			  struct drm_display_mode *mode)
{
	struct imx_parallel_display *imxpd = con_to_imxpd(connector);

	dev_dbg(imxpd->dev, "%s\n", __func__);
	return 0;
}

static struct drm_encoder *imx_pd_connector_best_encoder(
		struct drm_connector *connector)
{
	struct imx_parallel_display *imxpd = con_to_imxpd(connector);

	dev_dbg(imxpd->dev, "%s\n", __func__);
	return &imxpd->encoder;
}

static void imx_pd_encoder_dpms(struct drm_encoder *encoder, int mode)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "%s: %s\n", __func__, mode ? "OFF" : "ON");
}

static bool imx_pd_encoder_mode_fixup(struct drm_encoder *encoder,
			   const struct drm_display_mode *mode,
			   struct drm_display_mode *adjusted_mode)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "%s\n", __func__);
	return true;
}

static void imx_pd_encoder_prepare(struct drm_encoder *encoder)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "%s\n", __func__);

	imx_drm_crtc_panel_format(encoder->crtc, DRM_MODE_ENCODER_NONE,
				  imxpd->interface_pix_fmt,
				  imxpd->interface_mapptr);
}

static void imx_pd_encoder_commit(struct drm_encoder *encoder)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "%s\n", __func__);
}

static void imx_pd_encoder_mode_set(struct drm_encoder *encoder,
			 struct drm_display_mode *mode,
			 struct drm_display_mode *adjusted_mode)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "routing to crtc%d\n",
		imx_drm_crtc_get_id(encoder->crtc));

	imx_pd_set_reset_gpio(imxpd);
}

static void imx_pd_encoder_disable(struct drm_encoder *encoder)
{
	struct imx_parallel_display *imxpd = enc_to_imxpd(encoder);

	dev_dbg(imxpd->dev, "%s\n", __func__);
}

static void imx_pd_encoder_destroy(struct drm_encoder *encoder)
{
	/* do not free here */
}

static void imx_pd_connector_dpms(struct drm_connector *connector, int mode)
{
	struct imx_parallel_display *imxpd = con_to_imxpd(connector);

	dev_dbg(imxpd->dev, "%s\n", __func__);

	/* from off to on */
	if (mode < connector->dpms)
		imx_pd_set_reset_gpio(imxpd);

	drm_helper_connector_dpms(connector, mode);
}

static struct drm_connector_funcs imx_pd_connector_funcs = {
	.dpms = imx_pd_connector_dpms,
	.fill_modes = drm_helper_probe_single_connector_modes,
	.detect = imx_pd_connector_detect,
	.destroy = imx_pd_connector_destroy,
};

static struct drm_connector_helper_funcs imx_pd_connector_helper_funcs = {
	.get_modes = imx_pd_connector_get_modes,
	.best_encoder = imx_pd_connector_best_encoder,
	.mode_valid = imx_pd_connector_mode_valid,
};

static struct drm_encoder_funcs imx_pd_encoder_funcs = {
	.destroy = imx_pd_encoder_destroy,
};

static struct drm_encoder_helper_funcs imx_pd_encoder_helper_funcs = {
	.dpms = imx_pd_encoder_dpms,
	.mode_fixup = imx_pd_encoder_mode_fixup,
	.prepare = imx_pd_encoder_prepare,
	.commit = imx_pd_encoder_commit,
	.mode_set = imx_pd_encoder_mode_set,
	.disable = imx_pd_encoder_disable,
};

static int imx_pd_register(struct imx_parallel_display *imxpd)
{
	int ret;

	imxpd->connector.funcs = &imx_pd_connector_funcs;
	imxpd->encoder.funcs = &imx_pd_encoder_funcs;

	imxpd->encoder.encoder_type = DRM_MODE_ENCODER_NONE;
	imxpd->connector.connector_type = DRM_MODE_CONNECTOR_VGA;

	drm_encoder_helper_add(&imxpd->encoder, &imx_pd_encoder_helper_funcs);
	ret = imx_drm_add_encoder(&imxpd->encoder, &imxpd->imx_drm_encoder,
			THIS_MODULE);
	if (ret) {
		dev_err(imxpd->dev, "adding encoder failed with %d\n", ret);
		return ret;
	}

	drm_connector_helper_add(&imxpd->connector,
			&imx_pd_connector_helper_funcs);

	ret = imx_drm_add_connector(&imxpd->connector,
			&imxpd->imx_drm_connector, THIS_MODULE);
	if (ret) {
		imx_drm_remove_encoder(imxpd->imx_drm_encoder);
		dev_err(imxpd->dev, "adding connector failed with %d\n", ret);
		return ret;
	}

	imxpd->connector.encoder = &imxpd->encoder;

	drm_mode_connector_attach_encoder(&imxpd->connector, &imxpd->encoder);

	return 0;
}

static int imx_pd_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	const u8 *edidp;
	struct imx_parallel_display *imxpd;
	int ret;
	const char *fmt;
	struct pinctrl *pinctrl;

	imxpd = devm_kzalloc(&pdev->dev, sizeof(*imxpd), GFP_KERNEL);
	if (!imxpd)
		return -ENOMEM;

	pinctrl = devm_pinctrl_get_select_default(&pdev->dev);
	if (IS_ERR(pinctrl))
		return PTR_ERR(pinctrl);

	edidp = of_get_property(np, "edid", &imxpd->edid_len);
	if (edidp)
		imxpd->edid = kmemdup(edidp, imxpd->edid_len, GFP_KERNEL);

	imxpd->timings = of_get_display_timings(np);

	ret = of_property_read_string(np, "interface_pix_fmt", &fmt);
	if (!ret) {
		if (!strcmp(fmt, "rgb24"))
			imxpd->interface_pix_fmt = V4L2_PIX_FMT_RGB24;
		else if (!strcmp(fmt, "rgb565"))
			imxpd->interface_pix_fmt = V4L2_PIX_FMT_RGB565;
		else if (!strcmp(fmt, "rgb18"))
			imxpd->interface_pix_fmt = V4L2_PIX_FMT_RGB666;
	}

	ret = of_property_read_u32_array(np, "interface_pix_map",
					 imxpd->interface_map,
					 ARRAY_SIZE(imxpd->interface_map));
	if (!ret)
		imxpd->interface_mapptr = imxpd->interface_map;

	if (!imxpd->interface_pix_fmt && !imxpd->interface_mapptr) {
		dev_err(imxpd->dev, "No interface pix fmt defined!\n");
		return -EINVAL;
	}

	imxpd->dev = &pdev->dev;

	ret = imx_pd_register(imxpd);
	if (ret)
		return ret;

	imxpd->reset_pin = of_get_named_gpio_flags(imxpd->dev->of_node,
						   "reset-gpio", 0,
						   &imxpd->reset_flags);
	if (gpio_is_valid(imxpd->reset_pin)) {
		ret = gpio_request_one(imxpd->reset_pin, GPIOF_DIR_OUT |
			((imxpd->reset_flags == OF_GPIO_ACTIVE_LOW) ?
			GPIOF_INIT_HIGH : GPIOF_INIT_LOW), pdev->name);
		if (ret < 0) {
			dev_err(&pdev->dev, "request gpio failed!\n");
			imxpd->reset_pin = -1;
		}
	}

	ret = imx_drm_encoder_add_possible_crtcs(imxpd->imx_drm_encoder, np);

	platform_set_drvdata(pdev, imxpd);

	return 0;
}

static int imx_pd_remove(struct platform_device *pdev)
{
	struct imx_parallel_display *imxpd = platform_get_drvdata(pdev);
	struct drm_connector *connector = &imxpd->connector;
	struct drm_encoder *encoder = &imxpd->encoder;

	if (gpio_is_valid(imxpd->reset_pin))
		gpio_free(imxpd->reset_pin);

	drm_mode_connector_detach_encoder(connector, encoder);
	imx_drm_remove_connector(imxpd->imx_drm_connector);
	imx_drm_remove_encoder(imxpd->imx_drm_encoder);

	if (imxpd->timings)
		display_timings_release(imxpd->timings);

	kfree(imxpd->edid);

	return 0;
}

static const struct of_device_id imx_pd_dt_ids[] = {
	{ .compatible = "fsl,imx-parallel-display", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_pd_dt_ids);

static struct platform_driver imx_pd_driver = {
	.probe		= imx_pd_probe,
	.remove		= imx_pd_remove,
	.driver		= {
		.of_match_table = imx_pd_dt_ids,
		.name	= IMX_PD_DRIVER,
		.owner	= THIS_MODULE,
	},
};

module_platform_driver(imx_pd_driver);

MODULE_DESCRIPTION("i.MX parallel display driver");
MODULE_AUTHOR("Sascha Hauer, Pengutronix");
MODULE_LICENSE("GPL");
