/*
 * i.MX drm driver - LVDS display bridge
 *
 * Copyright (C) 2012 Mentor Graphics Inc.
 * Copyright (C) 2012 Sascha Hauer, Pengutronix
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/module.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <drm/drmP.h>
#include <drm/drm_fb_helper.h>
#include <drm/drm_crtc_helper.h>
#include <drm/drm_edid.h>
#include <linux/videodev2.h>
#include <linux/gpio.h>
#include <linux/of_i2c.h>
#include <linux/regmap.h>
#include <linux/mfd/syscon.h>
#include <linux/mfd/syscon/imx6q-iomuxc-gpr.h>
#include <linux/of_gpio.h>
#include <video/display_timing.h>
#include <video/of_display_timing.h>
#include <video/videomode.h>
#include "imx-drm.h"

#define MAX_LDB_CHANNELS                2

/* IOMUXC_GPR2 */
#define LDB_CH_MODE_BIT(ch)		(ch ? 2 : 0)
#define LDB_CH_MODE_MASK(ch)		(3 << LDB_CH_MODE_BIT(ch))
#define LDB_SPLIT_MODE_EN		(1 << 4)
#define LDB_DATA_WIDTH_24(ch)		(1 << (5 + 2*ch))
#define LDB_BIT_MAP_JEIDA(ch)		(1 << (6 + 2*ch))
#define LDB_DI_VS_POL_ACT_LOW(di)	(1 << (9 + di))

/* IOMUXC_GPR3 */
#define LVDS_CH_MUX_CTL_OFFS(ch)    (ch ? 8 : 6)

#define con_to_imx_ldb(x) container_of(x, struct imx_ldb, connector)
#define enc_to_imx_ldb(x) container_of(x, struct imx_ldb, encoder)

struct imx_ldb {
	struct device *dev;
	struct regmap *gp_reg;
	struct drm_connector connector;
	struct imx_drm_connector *imx_drm_connector;
	struct drm_encoder encoder;
	struct imx_drm_encoder *imx_drm_encoder;
	struct display_timings *timings;
	struct i2c_adapter *ddc;
	struct edid *edid;
	int edid_len;
	int chno;
	int standby_pin;
	enum of_gpio_flags standby_active;
	u32 interface_pix_fmt;
	struct clk *di_sel;
	struct clk *ldb_clk;
	struct clk *ldb_sel;
	bool ldb_clk_enabled;
	int ipu, di;
};

/* forward references */
static void imx_ldb_setup(struct drm_encoder *encoder,
			  struct drm_display_mode *mode);
static void imx_ldb_encoder_disable(struct drm_encoder *encoder);

static void imx_ldb_get_edid(struct drm_connector *connector)
{
	struct imx_ldb *imx_ldb = con_to_imx_ldb(connector);

	if (imx_ldb->edid)
		return;

	if (imx_ldb->ddc)
		imx_ldb->edid = drm_get_edid(connector, imx_ldb->ddc);
}

/*
 * We can use the result of ddc probe to detect LVDS display
 * presence if a ddc DT node was specified.
 */
static enum drm_connector_status imx_ldb_connector_detect(
	struct drm_connector *connector, bool force)
{
	struct imx_ldb *imx_ldb = con_to_imx_ldb(connector);
	enum drm_connector_status status;

	if (imx_ldb->ddc) {
		if (drm_probe_ddc(imx_ldb->ddc)) {
			status = connector_status_connected;
			dev_info(imx_ldb->dev,
				 "ddc probe success, connected\n");
		} else {
			status = connector_status_disconnected;
			dev_info(imx_ldb->dev,
				 "ddc probe failed, disconnected\n");
		}
	} else {
		status = connector_status_connected;
		dev_info(imx_ldb->dev, "no ddc, assuming connected\n");
	}

	return status;
}

static void imx_ldb_connector_destroy(struct drm_connector *connector)
{
	/* do not free here */
}

static int imx_ldb_connector_get_modes(struct drm_connector *connector)
{
	struct imx_ldb *imx_ldb = con_to_imx_ldb(connector);
	struct display_timings *timings = imx_ldb->timings;
	struct drm_display_mode *mode;
	struct videomode vm;
	int i, num_modes = 0;

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	imx_ldb_get_edid(connector);
	if (imx_ldb->edid) {
		dev_info(imx_ldb->dev, "got edid: width[%d] x height[%d]\n",
			 imx_ldb->edid->width_cm,
			 imx_ldb->edid->height_cm);

		drm_mode_connector_update_edid_property(connector,
							imx_ldb->edid);
		num_modes = drm_add_edid_modes(connector, imx_ldb->edid);
	}

	if (timings) {
		for (i = 0; i < timings->num_timings; i++) {
			if (videomode_from_timings(timings, &vm, i))
				break;

			mode = drm_mode_create(connector->dev);
			drm_display_mode_from_videomode(&vm, mode);

			mode->type = DRM_MODE_TYPE_DRIVER;
			if (i == timings->native_mode)
				mode->type |= DRM_MODE_TYPE_PREFERRED;

			drm_mode_set_name(mode);
			drm_mode_probed_add(connector, mode);
			num_modes++;
		}
	}

	return num_modes;
}

static int imx_ldb_connector_mode_valid(struct drm_connector *connector,
			  struct drm_display_mode *mode)
{
	struct imx_ldb *imx_ldb = con_to_imx_ldb(connector);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	/* TODO */
	return 0;
}

static struct drm_encoder *imx_ldb_connector_best_encoder(
		struct drm_connector *connector)
{
	struct imx_ldb *imx_ldb = con_to_imx_ldb(connector);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	return &imx_ldb->encoder;
}

static void imx_ldb_encoder_dpms(struct drm_encoder *encoder, int mode)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);
	int chno = imx_ldb->chno;

	dev_dbg(imx_ldb->dev, "%s: %s\n", __func__, mode ? "OFF" : "ON");

	if (mode) {
		/*
		 * turn OFF only if we previously were enabled via
		 * prepare and mode_set.
		 */
		if (imx_ldb->ldb_clk_enabled) {
			if (gpio_is_valid(imx_ldb->standby_pin))
				gpio_set_value(imx_ldb->standby_pin,
					       (imx_ldb->standby_active ==
						OF_GPIO_ACTIVE_LOW) ? 0 : 1);

			regmap_update_bits(imx_ldb->gp_reg, IOMUXC_GPR2,
					   LDB_CH_MODE_MASK(chno), 0);

			clk_disable_unprepare(imx_ldb->ldb_clk);
			imx_ldb->ldb_clk_enabled = false;
		}
	} else {
		/*
		 * turn ON only if we previously were turned OFF via DPMS,
		 * i.e. we were previously enabled via prepare and mode_set,
		 * but only the ldb_clk was disabled.
		 */
		if (imx_ldb->ldb_clk && encoder->crtc &&
		    !imx_ldb->ldb_clk_enabled) {
			imx_ldb_setup(encoder, &encoder->crtc->mode);

			clk_prepare_enable(imx_ldb->ldb_clk);
			imx_ldb->ldb_clk_enabled = true;

			if (gpio_is_valid(imx_ldb->standby_pin))
				gpio_set_value(imx_ldb->standby_pin,
					       (imx_ldb->standby_active ==
						OF_GPIO_ACTIVE_LOW) ? 1 : 0);
		}
	}
}

static void imx_ldb_set_ipu_di_mux(struct imx_ldb *imx_ldb,
				   struct drm_encoder *encoder)
{
	int chno = imx_ldb->chno;
	u32 mux, muxbit;

	dev_dbg(imx_ldb->dev, "routing to crtc%d\n",
		imx_drm_crtc_get_id(encoder->crtc));

	/* LVDS channel mode setting */
	mux = 0x1 | (imx_ldb->di << 1);
	regmap_update_bits(imx_ldb->gp_reg, IOMUXC_GPR2,
			   LDB_CH_MODE_MASK(chno),
			   mux << LDB_CH_MODE_BIT(chno));

	/* Now set IOMUX for ipu,di */
	muxbit = LVDS_CH_MUX_CTL_OFFS(chno);
	mux = (imx_ldb->ipu << 1) | imx_ldb->di;
	regmap_update_bits(imx_ldb->gp_reg, IOMUXC_GPR3,
			   0x3 << muxbit,  mux << muxbit);
}

static void imx_ldb_setup(struct drm_encoder *encoder,
			  struct drm_display_mode *mode)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);
	int ret = 0, chno = imx_ldb->chno;
	u32 reg_set = 0, reg_clr;
	unsigned long rate, newrate, newrate_rounded;

	/* only use JEIDA (not SPWG) data mapping */
	reg_set |= LDB_BIT_MAP_JEIDA(chno);
	/* only 24 bpp allowed */
	reg_set |= LDB_DATA_WIDTH_24(chno);

	/* vsync polarity */
	if (mode->flags & DRM_MODE_FLAG_NVSYNC)
		reg_set |= LDB_DI_VS_POL_ACT_LOW(imx_ldb->di);

	reg_clr = LDB_BIT_MAP_JEIDA(chno) | LDB_DATA_WIDTH_24(chno) |
		LDB_DI_VS_POL_ACT_LOW(imx_ldb->di);
	/* split mode not supported */
	reg_clr |= LDB_SPLIT_MODE_EN;

	regmap_update_bits(imx_ldb->gp_reg, IOMUXC_GPR2, reg_clr, reg_set);

	imx_ldb_set_ipu_di_mux(imx_ldb, encoder);

	rate = clk_get_rate(imx_ldb->ldb_sel);
	newrate = mode->clock * 1000 * 7;
	newrate_rounded = clk_round_rate(imx_ldb->ldb_sel, newrate);

	dev_dbg(imx_ldb->dev,
		"serializer rate now %ld, want %ld (rounded %ld)\n",
		rate, newrate, newrate_rounded);

	ret = clk_set_rate(imx_ldb->ldb_sel, newrate_rounded);
	if (ret)
		dev_err(imx_ldb->dev,
			"clk_set_rate failed: %d\n", ret);

	dev_dbg(imx_ldb->dev, "after: %ld\n", clk_get_rate(imx_ldb->ldb_sel));
}

static bool imx_ldb_encoder_mode_fixup(struct drm_encoder *encoder,
			   const struct drm_display_mode *mode,
			   struct drm_display_mode *adjusted_mode)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);
	int chno = imx_ldb->chno;

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	imx_drm_crtc_get_ipu_di(encoder->crtc, &imx_ldb->ipu, &imx_ldb->di);

	/*
	 * This driver does not support dual or split channel mode. So
	 * the only valid h/w config is for LVDS0 to be routed to DI0
	 * (on either IPU) and LVDS1 routed to DI1 (on either IPU).
	 */
	if (imx_ldb->di != chno) {
		dev_err(imx_ldb->dev,
			"assigned crtc is not valid (DI must be %d)\n", chno);
		return false;
	}

	return true;
}

/*
 * (Re)-acquire the clocks in preparation for enabling them.
 */
static void imx_ldb_encoder_prepare(struct drm_encoder *encoder)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);
	char clkname[16];

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	/* clock setup */

	if (imx_ldb->ldb_clk) {
		if (imx_ldb->ldb_clk_enabled) {
			clk_disable_unprepare(imx_ldb->ldb_clk);
			imx_ldb->ldb_clk_enabled = false;
		}
		devm_clk_put(imx_ldb->dev, imx_ldb->ldb_clk);
	}

	sprintf(clkname, "ldb_di%d", imx_ldb->di);
	imx_ldb->ldb_clk = devm_clk_get(imx_ldb->dev, clkname);
	if (IS_ERR(imx_ldb->ldb_clk)) {
		dev_err(imx_ldb->dev, "get clock %s failed\n", clkname);
		imx_ldb->ldb_clk = NULL;
		goto disable;
	}

	if (imx_ldb->di_sel)
		devm_clk_put(imx_ldb->dev, imx_ldb->di_sel);

	sprintf(clkname, "ipu%d_di%d_sel", imx_ldb->ipu + 1, imx_ldb->di);
	imx_ldb->di_sel = devm_clk_get(imx_ldb->dev, clkname);
	if (IS_ERR(imx_ldb->di_sel)) {
		dev_err(imx_ldb->dev, "get clock %s failed\n", clkname);
		imx_ldb->di_sel = NULL;
		goto disable;
	}

	/*
	 * Make sure the DI clock's source is the LDB clock. The DI
	 * clock must be synchronous with the LDB serializer clock.
	 */
	clk_set_parent(imx_ldb->di_sel, imx_ldb->ldb_clk);

	if (imx_ldb->ldb_sel)
		devm_clk_put(imx_ldb->dev, imx_ldb->ldb_sel);

	sprintf(clkname, "ldb_di%d_sel", imx_ldb->di);
	imx_ldb->ldb_sel = devm_clk_get(imx_ldb->dev, clkname);
	if (IS_ERR(imx_ldb->ldb_sel)) {
		dev_err(imx_ldb->dev, "get clock %s failed\n", clkname);
		imx_ldb->ldb_sel = NULL;
		goto disable;
	}

	imx_drm_crtc_panel_format(encoder->crtc, DRM_MODE_ENCODER_LVDS,
				  imx_ldb->interface_pix_fmt, NULL);
	return;
disable:
	imx_ldb_encoder_disable(encoder);
}

static void imx_ldb_encoder_mode_set(struct drm_encoder *encoder,
				     struct drm_display_mode *mode,
				     struct drm_display_mode *adjusted_mode)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	imx_ldb_setup(encoder, mode);
}

static void imx_ldb_encoder_commit(struct drm_encoder *encoder)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	clk_prepare_enable(imx_ldb->ldb_clk);
	imx_ldb->ldb_clk_enabled = true;
}

static void imx_ldb_encoder_disable(struct drm_encoder *encoder)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	regmap_update_bits(imx_ldb->gp_reg, IOMUXC_GPR2,
			   LDB_CH_MODE_MASK(imx_ldb->chno), 0);

	if (imx_ldb->ldb_clk) {
		if (imx_ldb->ldb_clk_enabled) {
			clk_disable_unprepare(imx_ldb->ldb_clk);
			imx_ldb->ldb_clk_enabled = false;
		}
		devm_clk_put(imx_ldb->dev, imx_ldb->ldb_clk);
		imx_ldb->ldb_clk = NULL;
	}

	if (imx_ldb->ldb_sel) {
		devm_clk_put(imx_ldb->dev, imx_ldb->ldb_sel);
		imx_ldb->ldb_sel = NULL;
	}

	if (imx_ldb->di_sel) {
		devm_clk_put(imx_ldb->dev, imx_ldb->di_sel);
		imx_ldb->di_sel = NULL;
	}
}

static void imx_ldb_encoder_destroy(struct drm_encoder *encoder)
{
	struct imx_ldb *imx_ldb = enc_to_imx_ldb(encoder);

	dev_dbg(imx_ldb->dev, "%s\n", __func__);

	/* do not free here */
}

static struct drm_connector_funcs imx_ldb_connector_funcs = {
	.dpms = drm_helper_connector_dpms,
	.fill_modes = drm_helper_probe_single_connector_modes,
	.detect = imx_ldb_connector_detect,
	.destroy = imx_ldb_connector_destroy,
};

static struct drm_connector_helper_funcs imx_ldb_connector_helper_funcs = {
	.get_modes = imx_ldb_connector_get_modes,
	.best_encoder = imx_ldb_connector_best_encoder,
	.mode_valid = imx_ldb_connector_mode_valid,
};

static struct drm_encoder_funcs imx_ldb_encoder_funcs = {
	.destroy = imx_ldb_encoder_destroy,
};

static struct drm_encoder_helper_funcs imx_ldb_encoder_helper_funcs = {
	.dpms = imx_ldb_encoder_dpms,
	.mode_fixup = imx_ldb_encoder_mode_fixup,
	.prepare = imx_ldb_encoder_prepare,
	.commit = imx_ldb_encoder_commit,
	.mode_set = imx_ldb_encoder_mode_set,
	.disable = imx_ldb_encoder_disable,
};

static int imx_ldb_register(struct imx_ldb *imx_ldb)
{
	struct device_node *np = imx_ldb->dev->of_node;
	struct device_node *ddc_node;
	const char *fmt;
	int ret;

	imx_ldb->gp_reg = syscon_regmap_lookup_by_compatible(
		"fsl,imx6q-iomuxc-gpr");
	if (IS_ERR(imx_ldb->gp_reg)) {
		ret = PTR_ERR(imx_ldb->gp_reg);
		dev_err(imx_ldb->dev, "failed to regmap GPR: %d\n", ret);
		return ret;
	}

	ret = of_property_read_string(np, "interface_pix_fmt", &fmt);
	if (!ret) {
		if (!strcmp(fmt, "rgb24"))
			imx_ldb->interface_pix_fmt = V4L2_PIX_FMT_RGB24;
		else if (!strcmp(fmt, "rgb565"))
			imx_ldb->interface_pix_fmt = V4L2_PIX_FMT_RGB565;
		else if (!strcmp(fmt, "rgb18"))
			imx_ldb->interface_pix_fmt = V4L2_PIX_FMT_RGB666;
		else {
			dev_err(imx_ldb->dev, "Unsupported interface pix_fmt!\n");
			return -EINVAL;
		}
	} else if (ret == -EINVAL) { /* property does not exist */
		dev_info(imx_ldb->dev, "No interface_pix_fmt, defaulting to RGB24\n");
		imx_ldb->interface_pix_fmt = V4L2_PIX_FMT_RGB24;
	}

	ddc_node = of_parse_phandle(np, "ddc", 0);
	if (ddc_node) {
		imx_ldb->ddc = of_find_i2c_adapter_by_node(ddc_node);
		of_node_put(ddc_node);
		if (!imx_ldb->ddc)
			return -EPROBE_DEFER;
		ret = i2c_adapter_id(imx_ldb->ddc);
		put_device(&imx_ldb->ddc->dev);
		imx_ldb->ddc = i2c_get_adapter(ret);
		dev_dbg(imx_ldb->dev, "ddc = %p\n", imx_ldb->ddc);
	} else {
		/* if no DDC node specified, fallback to hardcoded EDID */
		const u8 *edidp = of_get_property(np, "edid",
						  &imx_ldb->edid_len);
		if (edidp) {
			imx_ldb->edid = (struct edid *)
				kmemdup(edidp, imx_ldb->edid_len, GFP_KERNEL);
		}
	}

	imx_ldb->timings = of_get_display_timings(np);

	imx_ldb->connector.funcs = &imx_ldb_connector_funcs;
	imx_ldb->encoder.funcs = &imx_ldb_encoder_funcs;

	imx_ldb->encoder.encoder_type = DRM_MODE_ENCODER_LVDS;
	imx_ldb->connector.connector_type = DRM_MODE_CONNECTOR_LVDS;

	drm_encoder_helper_add(&imx_ldb->encoder,
			&imx_ldb_encoder_helper_funcs);
	ret = imx_drm_add_encoder(&imx_ldb->encoder,
			&imx_ldb->imx_drm_encoder, THIS_MODULE);
	if (ret) {
		dev_err(imx_ldb->dev, "adding encoder failed with %d\n", ret);
		return ret;
	}

	drm_connector_helper_add(&imx_ldb->connector,
			&imx_ldb_connector_helper_funcs);

	ret = imx_drm_add_connector(&imx_ldb->connector,
			&imx_ldb->imx_drm_connector, THIS_MODULE);
	if (ret) {
		dev_err(imx_ldb->dev, "adding connector failed with %d\n", ret);
		goto remove;
	}

	imx_ldb->connector.encoder = &imx_ldb->encoder;

	drm_mode_connector_attach_encoder(&imx_ldb->connector,
					  &imx_ldb->encoder);

	imx_drm_encoder_add_possible_crtcs(imx_ldb->imx_drm_encoder, np);

	return 0;

remove:
	imx_drm_remove_encoder(imx_ldb->imx_drm_encoder);
	return ret;
}

static int imx_ldb_probe(struct platform_device *pdev)
{
	struct imx_ldb *imx_ldb;
	int ch, ret = 0;

	imx_ldb = devm_kzalloc(&pdev->dev, sizeof(*imx_ldb), GFP_KERNEL);
	if (!imx_ldb) {
		dev_err(&pdev->dev, "no memory!\n");
		return -ENOMEM;
	}

	imx_ldb->dev = &pdev->dev;

	imx_ldb->standby_pin = of_get_named_gpio_flags(imx_ldb->dev->of_node,
						"standby-gpio", 0,
						&imx_ldb->standby_active);
	if (imx_ldb->standby_pin == -EPROBE_DEFER) {
		ret = -EPROBE_DEFER;
		goto cleanup;
	}

	if (gpio_is_valid(imx_ldb->standby_pin)) {
		ret = gpio_request_one(imx_ldb->standby_pin, GPIOF_DIR_OUT |
			((imx_ldb->standby_active == OF_GPIO_ACTIVE_LOW) ?
			GPIOF_INIT_HIGH : GPIOF_INIT_LOW), pdev->name);
		if (ret < 0) {
			dev_err(&pdev->dev, "request gpio failed!\n");
			goto cleanup;
		}
	}

	ret = of_property_read_u32(imx_ldb->dev->of_node,
			"fsl,ldb-channel", &ch);
	if (ret) {
		dev_err(imx_ldb->dev, "Could not parse ldb channel\n");
		goto cleanup;
	}

	if (ch < 0 || ch >= MAX_LDB_CHANNELS) {
		dev_err(imx_ldb->dev, "Invalid ldb channel %d\n", ch);
		goto cleanup;
	};

	imx_ldb->chno = ch;
	dev_set_name(imx_ldb->dev, "lvds%d", ch);

	ret = imx_ldb_register(imx_ldb);
	if (ret) {
		dev_err(imx_ldb->dev,
			"imx_ldb_register() failed with %d\n", ret);
		goto cleanup;
	}

	platform_set_drvdata(pdev, imx_ldb);

	return 0;

cleanup:
	if (gpio_is_valid(imx_ldb->standby_pin)) {
		gpio_set_value(imx_ldb->standby_pin,
		  (imx_ldb->standby_active == OF_GPIO_ACTIVE_LOW) ? 0 : 1);
		gpio_free(imx_ldb->standby_pin);
	}

	imx_ldb->standby_pin = -1;

	devm_kfree(&pdev->dev, imx_ldb);
	return ret;
}

static int imx_ldb_remove(struct platform_device *pdev)
{
	struct imx_ldb *imx_ldb = platform_get_drvdata(pdev);
	struct drm_connector *connector = &imx_ldb->connector;
	struct drm_encoder *encoder = &imx_ldb->encoder;

	drm_mode_connector_detach_encoder(connector, encoder);
	imx_drm_remove_connector(imx_ldb->imx_drm_connector);
	imx_drm_remove_encoder(imx_ldb->imx_drm_encoder);
	if (imx_ldb->ddc)
		i2c_put_adapter(imx_ldb->ddc);
	kfree(imx_ldb->edid);

	if (imx_ldb->timings)
		display_timings_release(imx_ldb->timings);

	if (gpio_is_valid(imx_ldb->standby_pin)) {
		gpio_set_value(imx_ldb->standby_pin,
		  (imx_ldb->standby_active == OF_GPIO_ACTIVE_LOW) ? 0 : 1);
		gpio_free(imx_ldb->standby_pin);
	}

	imx_ldb->standby_pin = -1;
	devm_kfree(&pdev->dev, imx_ldb);
	return 0;
}

static const struct of_device_id imx_ldb_dt_ids[] = {
	{ .compatible = "fsl,imx-ldb", .data = NULL, },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx_ldb_dt_ids);

static struct platform_driver imx_ldb_driver = {
	.probe		= imx_ldb_probe,
	.remove		= imx_ldb_remove,
	.driver		= {
		.of_match_table = imx_ldb_dt_ids,
		.name	= IMX_LDB_DRIVER,
		.owner	= THIS_MODULE,
	},
};

module_platform_driver(imx_ldb_driver);

MODULE_DESCRIPTION("i.MX LVDS driver");
MODULE_AUTHOR("Sascha Hauer, Pengutronix");
MODULE_LICENSE("GPL");
