/*
 * i.MX IPUv3 Graphics driver
 *
 * Copyright (C) 2011 Sascha Hauer, Pengutronix
 * Copyright (C) 2012 Mengor Graphics
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */
#include <linux/module.h>

#include <drm/drmP.h>
#include <drm/drm_fb_cma_helper.h>
#include <drm/drm_gem_cma_helper.h>
#include <linux/platform_data/imx-ipu-v3.h>

#include "imx-drm.h"

#define DRIVER_DESC		"i.MX IPUv3 Graphics"

struct ipu_plane {
	struct drm_plane	plane;
	struct imx_drm_plane	*imx_plane;
	struct ipu_soc          *ipu; /* our parent IPU */

	unsigned int		src_x;
	unsigned int		src_y;
	unsigned int		crtc_x;
	unsigned int		crtc_y;
	unsigned int		crtc_w;
	unsigned int		crtc_h;

	u8                      global_alpha;
	bool                    global_alpha_en;
	unsigned int		color_key;
	bool                    color_key_en;

	struct device		*dev;
	bool			enabled;

	struct drm_pending_vblank_event *page_flip_event;
	struct drm_framebuffer	*newfb;
	int			irq;

	struct ipuv3_channel	*ipu_ch;
	struct dmfc_channel	*dmfc;
	struct ipu_dp		*dp;
};

#define to_ipu_plane(x) container_of(x, struct ipu_plane, plane)

struct display_channels {
	int idmac;
	int dp;
};

static const struct display_channels sync_fg = {
	.idmac = IPUV3_CHANNEL_MEM_FG_SYNC,
	.dp    = IPU_DP_FLOW_SYNC_FG,
};

/*
 * This driver does not yet support async flows for "smart" displays,
 * but keep this around for future reference.
 */
#if 0
static const struct display_channels async_fg = {
	.idmac = IPUV3_CHANNEL_MEM_FG_ASYNC,
	.dp    = IPU_DP_FLOW_ASYNC0_FG,
};
#endif

static int plane_calc_bandwidth(struct ipu_plane *ipu_plane)
{
	unsigned int vref;

	vref = 60;
/*	vref = mode->clock * 1000 / ipu_plane->crtc_h / ipu_plane->crtc_w; */

	return ipu_plane->crtc_w * ipu_plane->crtc_h * vref;
}

static int ipu_plane_set_base(struct ipu_plane *ipu_plane,
			      struct drm_framebuffer *fb,
			      int bufnum, int x, int y)
{
	struct drm_gem_cma_object *cma_obj;
	unsigned long phys;

	cma_obj = drm_fb_cma_get_gem_obj(fb, 0);
	if (!cma_obj) {
		DRM_LOG_KMS("entry is null.\n");
		return -EFAULT;
	}

	phys = cma_obj->paddr;
	phys += x * (fb->bits_per_pixel >> 3);
	phys += y * fb->pitches[0];

	dev_dbg(ipu_plane->dev, "%s: phys: 0x%lx\n", __func__, phys);
	dev_dbg(ipu_plane->dev, "%s: xy: %dx%d\n", __func__, x, y);

	ipu_cpmem_set_stride(ipu_plane->ipu_ch, fb->pitches[0]);

	/* bufnum < 0 means set both buffer addresses */
	if (bufnum < 0) {
		ipu_cpmem_set_buffer(ipu_plane->ipu_ch, 0, phys);
		ipu_cpmem_set_buffer(ipu_plane->ipu_ch, 1, phys);
	} else
		ipu_cpmem_set_buffer(ipu_plane->ipu_ch, bufnum, phys);

	return 0;
}

static void ipu_plane_enable(struct ipu_plane *ipu_plane)
{
	if (ipu_plane->enabled)
		return;

	ipu_dp_enable(ipu_plane->dp);
	ipu_dmfc_enable_channel(ipu_plane->dmfc);
	ipu_idmac_enable_channel(ipu_plane->ipu_ch);
	ipu_idmac_enable_watermark(ipu_plane->ipu_ch, true);
	ipu_dp_enable_channel(ipu_plane->dp);

	ipu_plane->enabled = true;
}

static void __ipu_plane_disable(struct ipu_plane *ipu_plane)
{
	if (!ipu_plane->enabled)
		return;

	ipu_dp_disable_channel(ipu_plane->dp);
	ipu_idmac_enable_watermark(ipu_plane->ipu_ch, false);
	ipu_idmac_disable_channel(ipu_plane->ipu_ch);
	ipu_dp_uninit_channel(ipu_plane->dp);
	ipu_dmfc_free_bandwidth(ipu_plane->dmfc);
	ipu_dmfc_disable_channel(ipu_plane->dmfc);
	ipu_dp_disable(ipu_plane->dp);

	ipu_plane->enabled = false;
}

static int __ipu_plane_setup(struct ipu_plane *ipu_plane,
			       struct drm_framebuffer *fb)
{
	int ret;
	u32 burstsize, v4l2_fmt;

	dev_dbg(ipu_plane->dev, "%s: crtc_w: %d\n", __func__,
			ipu_plane->crtc_w);
	dev_dbg(ipu_plane->dev, "%s: crtc_h: %d\n", __func__,
			ipu_plane->crtc_h);

	ipu_cpmem_zero(ipu_plane->ipu_ch);

	ret = imx_drm_format_to_v4l(fb->pixel_format, &v4l2_fmt);
	if (ret < 0) {
		dev_err(ipu_plane->dev, "unsupported pixel format 0x%08x\n",
				fb->pixel_format);
		return ret;
	}

	ret = ipu_dp_setup_channel(ipu_plane->dp,
				   ipu_pixelformat_to_colorspace(v4l2_fmt),
				   IPUV3_COLORSPACE_RGB);
	if (ret) {
		dev_err(ipu_plane->dev,
			"initializing display processor failed with %d\n", ret);
		return ret;
	}
	ipu_dp_set_window_pos(ipu_plane->dp,
			      ipu_plane->crtc_x, ipu_plane->crtc_y);
	ipu_dp_set_global_alpha(ipu_plane->dp,
				ipu_plane->global_alpha_en,
				ipu_plane->global_alpha, false);
	ipu_dp_set_chroma_key(ipu_plane->dp,
			      ipu_plane->color_key_en,
			      ipu_plane->color_key);

	ipu_cpmem_set_resolution(ipu_plane->ipu_ch,
			ipu_plane->crtc_w, ipu_plane->crtc_h);
	ipu_cpmem_set_fmt(ipu_plane->ipu_ch, v4l2_fmt);
	ipu_cpmem_set_high_priority(ipu_plane->ipu_ch);
	ipu_cpmem_set_axi_id(ipu_plane->ipu_ch, 0);
	ipu_idmac_lock_enable(ipu_plane->ipu_ch, 8);

	if (ipu_pixelformat_is_planar(v4l2_fmt))
		ipu_cpmem_set_yuv_planar(ipu_plane->ipu_ch, v4l2_fmt,
				fb->pitches[0], ipu_plane->crtc_h);

	burstsize = ipu_cpmem_get_burst_size(ipu_plane->ipu_ch);
	ret = ipu_dmfc_alloc_bandwidth(ipu_plane->dmfc,
				       plane_calc_bandwidth(ipu_plane),
				       ipu_plane->crtc_w, burstsize);
	if (ret) {
		dev_err(ipu_plane->dev,
				"allocating dmfc bandwidth failed with %d\n",
				ret);
		return ret;
	}

	/* enable double-buffering */
	ipu_idmac_set_double_buffer(ipu_plane->ipu_ch, true);
	ipu_idmac_set_triple_buffer(ipu_plane->ipu_ch, false);

	ipu_plane_set_base(ipu_plane, fb, -1,
			   ipu_plane->src_x, ipu_plane->src_y);

	/* set buffers ready */
	ipu_idmac_select_buffer(ipu_plane->ipu_ch, 0);
	ipu_idmac_select_buffer(ipu_plane->ipu_ch, 1);

	ipu_plane_enable(ipu_plane);

	return 0;
}

static int
ipu_plane_update(struct drm_plane *plane, struct drm_crtc *crtc,
		 struct drm_framebuffer *fb, int crtc_x, int crtc_y,
		 unsigned int crtc_w, unsigned int crtc_h,
		 uint32_t src_x, uint32_t src_y,
		 uint32_t src_w, uint32_t src_h)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);
	int crtc_id;

	if (src_w >> 16 != crtc_w || src_h >> 16 != crtc_h) {
		dev_err(ipu_plane->dev, "scaling not supported\n");
		return -EINVAL;
	}

	crtc_id = imx_drm_crtc_get_id(crtc);
	if (!(plane->possible_crtcs & (1 << crtc_id))) {
		dev_err(ipu_plane->dev,
			"crtc%d not supported by this plane\n", crtc_id);
		return -EINVAL;
	}

	ipu_plane->src_x = src_x >> 16;
	ipu_plane->src_y = src_y >> 16;
	ipu_plane->crtc_x = crtc_x;
	ipu_plane->crtc_y = crtc_y;
	ipu_plane->crtc_w = crtc_w;
	ipu_plane->crtc_h = crtc_h;

	return __ipu_plane_setup(ipu_plane, fb);
}


static int ipu_plane_disable(struct drm_plane *plane)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	__ipu_plane_disable(ipu_plane);
	return 0;
}


static void ipu_plane_destroy(struct drm_plane *plane)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	__ipu_plane_disable(ipu_plane);
	imx_drm_remove_plane(ipu_plane->imx_plane);
	kfree(ipu_plane);
}

static int ipu_page_flip(struct drm_plane *plane,
		struct drm_framebuffer *fb,
		struct drm_pending_vblank_event *event)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);
	int ret;

	if (ipu_plane->newfb)
		return -EBUSY;

	if (ipu_plane->crtc_w > fb->width ||
	    ipu_plane->crtc_h > fb->height ||
	    ipu_plane->src_x > fb->width - ipu_plane->crtc_w ||
	    ipu_plane->src_y > fb->height - ipu_plane->crtc_h) {
		DRM_DEBUG_KMS("Invalid fb size %ux%u for plane %ux%u+%d+%d.\n",
			      fb->width, fb->height,
			      ipu_plane->crtc_w, ipu_plane->crtc_h,
			      ipu_plane->src_x, ipu_plane->src_y);
		return -ENOSPC;
	}

	ipu_plane->page_flip_event = event;

	ret = imx_drm_plane_vblank_get(ipu_plane->imx_plane);
	if (ret) {
		dev_dbg(ipu_plane->dev, "failed to acquire vblank counter\n");

		return ret;
	}

	ipu_plane->newfb = fb;

	return 0;
}

static struct drm_plane_funcs ipu_plane_funcs = {
	.update_plane = ipu_plane_update,
	.disable_plane = ipu_plane_disable,
	.destroy = ipu_plane_destroy,
	.page_flip = ipu_page_flip,
};

static void ipu_plane_handle_pageflip(struct ipu_plane *ipu_plane)
{
	struct drm_pending_vblank_event *e;
	struct timeval now;
	unsigned long flags;
	struct drm_device *drm = ipu_plane->plane.dev;

	spin_lock_irqsave(&drm->event_lock, flags);

	e = ipu_plane->page_flip_event;
	if (!e) {
		imx_drm_plane_vblank_put(ipu_plane->imx_plane);
		spin_unlock_irqrestore(&drm->event_lock, flags);
		return;
	}

	do_gettimeofday(&now);
	e->event.sequence = 0;
	e->event.tv_sec = now.tv_sec;
	e->event.tv_usec = now.tv_usec;
	ipu_plane->page_flip_event = NULL;

	imx_drm_plane_vblank_put(ipu_plane->imx_plane);

	list_add_tail(&e->base.link, &e->base.file_priv->event_list);

	wake_up_interruptible(&e->base.file_priv->event_wait);

	spin_unlock_irqrestore(&drm->event_lock, flags);
}

static irqreturn_t ipu_irq_handler(int irq, void *dev_id)
{
	struct ipu_plane *ipu_plane = dev_id;
	int curbuf;

	imx_drm_plane_handle_vblank(ipu_plane->imx_plane);

	if (ipu_plane->newfb) {
		ipu_plane->plane.fb = ipu_plane->newfb;
		ipu_plane->newfb = NULL;

		curbuf = ipu_idmac_current_buffer(ipu_plane->ipu_ch);
		curbuf ^= 1;

		dev_dbg(ipu_plane->dev, "%s: flipping to buffer %d\n",
			__func__, curbuf);

		ipu_plane_set_base(ipu_plane, ipu_plane->plane.fb, curbuf,
				   ipu_plane->src_x, ipu_plane->src_y);

		ipu_idmac_select_buffer(ipu_plane->ipu_ch, curbuf);

		ipu_plane_handle_pageflip(ipu_plane);
	}

	return IRQ_HANDLED;
}

static int ipu_enable_vblank(struct drm_plane *plane)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	enable_irq(ipu_plane->irq);

	return 0;
}

static void ipu_disable_vblank(struct drm_plane *plane)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	disable_irq(ipu_plane->irq);
}

static const uint32_t formats[] = {
	DRM_FORMAT_XRGB8888,
	DRM_FORMAT_ARGB8888,
	DRM_FORMAT_RGB565,
	DRM_FORMAT_RGB888,
	DRM_FORMAT_BGR888,
	DRM_FORMAT_YUYV,
	DRM_FORMAT_UYVY,
	DRM_FORMAT_YUV420,
	DRM_FORMAT_YVU420,
	DRM_FORMAT_YUV422,
	DRM_FORMAT_YVU422,
	DRM_FORMAT_NV12,
};

static void ipu_plane_put_resources(struct ipu_plane *ipu_plane)
{
	devm_free_irq(ipu_plane->dev, ipu_plane->irq, ipu_plane);
	if (!IS_ERR_OR_NULL(ipu_plane->ipu_ch))
		ipu_idmac_put(ipu_plane->ipu_ch);
	if (!IS_ERR_OR_NULL(ipu_plane->dmfc))
		ipu_dmfc_put(ipu_plane->dmfc);
	if (!IS_ERR_OR_NULL(ipu_plane->dp))
		ipu_dp_put(ipu_plane->dp);
}


static int ipu_plane_get_resources(struct ipu_plane *ipu_plane,
				   struct device_node *node)
{
	const struct display_channels *ch = &sync_fg;
	int ret;

	ipu_plane->ipu = dev_get_drvdata(ipu_plane->dev->parent);

	ipu_plane->ipu_ch = ipu_idmac_get(ipu_plane->ipu, ch->idmac, false);
	if (IS_ERR_OR_NULL(ipu_plane->ipu_ch)) {
		ret = PTR_ERR(ipu_plane->ipu_ch);
		goto err_out;
	}

	ipu_plane->dmfc = ipu_dmfc_get(ipu_plane->ipu, ch->idmac);
	if (IS_ERR(ipu_plane->dmfc)) {
		ret = PTR_ERR(ipu_plane->dmfc);
		goto err_out;
	}

	ipu_plane->dp = ipu_dp_get(ipu_plane->ipu, ch->dp);
	if (IS_ERR(ipu_plane->dp)) {
		ret = PTR_ERR(ipu_plane->dp);
		goto err_out;
	}

	return 0;
err_out:
	ipu_plane_put_resources(ipu_plane);

	return ret;
}

static int ipu_plane_get_irqs(struct ipu_plane *ipu_plane)
{
	int ret;

	ipu_plane->irq = ipu_idmac_channel_irq(ipu_plane->ipu,
			ipu_plane->ipu_ch, IPU_IRQ_EOF);
	ret = devm_request_irq(ipu_plane->dev, ipu_plane->irq,
			ipu_irq_handler, 0, "imx_drm", ipu_plane);
	if (ret < 0) {
		dev_err(ipu_plane->dev, "irq request failed with %d.\n", ret);
		return ret;
	}

	disable_irq(ipu_plane->irq);

	return 0;
}

/* helper functions */

static int ipu_plane_get_global_alpha(struct drm_plane *plane,
				      struct drm_imx_global_alpha *ga)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	ga->enable = ipu_plane->global_alpha_en;
	ga->value = ipu_plane->global_alpha;

	return 0;
}

static int ipu_plane_set_global_alpha(struct drm_plane *plane,
				      struct drm_imx_global_alpha *ga)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);
	int ret;

	ret = ipu_dp_set_global_alpha(ipu_plane->dp, ga->enable, ga->value,
				      false);
	if (!ret) {
		ipu_plane->global_alpha_en = ga->enable;
		ipu_plane->global_alpha = ga->value;
	}

	return ret;
}

static int ipu_plane_get_chromakey(struct drm_plane *plane,
				   struct drm_imx_chromakey *ck)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	ck->enable = ipu_plane->color_key_en;
	ck->key = ipu_plane->color_key;

	return 0;
}

static int ipu_plane_set_chromakey(struct drm_plane *plane,
				   struct drm_imx_chromakey *ck)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);
	int ret;

	ret = ipu_dp_set_chroma_key(ipu_plane->dp, ck->enable, ck->key);
	if (!ret) {
		ipu_plane->color_key_en = ck->enable;
		ipu_plane->color_key = ck->key;
	}

	return ret;
}

static void ipu_plane_get_config(struct drm_plane *plane,
				 int *crtc_x, int *crtc_y,
				 uint32_t *crtc_w, uint32_t *crtc_h,
				 uint32_t *src_x,  uint32_t *src_y,
				 uint32_t *src_w,  uint32_t *src_h)
{
	struct ipu_plane *ipu_plane = to_ipu_plane(plane);

	*src_x = ipu_plane->src_x;
	*src_y = ipu_plane->src_y;
	*src_w = ipu_plane->crtc_w << 16;
	*src_h = ipu_plane->crtc_h << 16;
	*crtc_x = ipu_plane->crtc_x;
	*crtc_y = ipu_plane->crtc_y;
	*crtc_w = ipu_plane->crtc_w;
	*crtc_h = ipu_plane->crtc_h;
}

static struct imx_drm_plane_helper_funcs imx_plane_helper_funcs = {
	.get_global_alpha = ipu_plane_get_global_alpha,
	.set_global_alpha = ipu_plane_set_global_alpha,
	.get_chromakey    = ipu_plane_get_chromakey,
	.set_chromakey    = ipu_plane_set_chromakey,
	.enable_vblank    = ipu_enable_vblank,
	.disable_vblank   = ipu_disable_vblank,
	.get_config       = ipu_plane_get_config,
	.plane_funcs      = &ipu_plane_funcs,
};


static int ipu_plane_init(struct ipu_plane *ipu_plane,
			  struct device_node *node)
{
	int ret;

	ret = ipu_plane_get_resources(ipu_plane, node);
	if (ret) {
		dev_err(ipu_plane->dev, "getting resources failed with %d.\n",
				ret);
		return ret;
	}

	ret = imx_drm_add_plane(&ipu_plane->plane, &ipu_plane->imx_plane,
				&imx_plane_helper_funcs,
				THIS_MODULE, formats, ARRAY_SIZE(formats),
				ipu_get_num(ipu_plane->ipu));

	if (ret) {
		dev_err(ipu_plane->dev, "adding plane failed with %d.\n", ret);
		goto err_put_resources;
	}

	ret = ipu_plane_get_irqs(ipu_plane);
	if (ret) {
		dev_err(ipu_plane->dev, "getting irqs failed with %d.\n", ret);
		goto err_put_resources;
	}

	imx_drm_plane_add_possible_crtcs(ipu_plane->imx_plane, node);

	/*
	 * Make sure default global alpha is enabled and completely
	 * opaque.
	 */
	ipu_plane->global_alpha_en = true;
	ipu_plane->global_alpha = 255;

	return 0;

err_put_resources:
	ipu_plane_put_resources(ipu_plane);

	return ret;
}


static int ipu_drm_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct ipu_plane *ipu_plane;
	int ret;

	pdev->dev.coherent_dma_mask = DMA_BIT_MASK(32);

	ipu_plane = devm_kzalloc(&pdev->dev, sizeof(*ipu_plane), GFP_KERNEL);
	if (!ipu_plane)
		return -ENOMEM;

	ipu_plane->dev = &pdev->dev;

	ret = ipu_plane_init(ipu_plane, node);

	platform_set_drvdata(pdev, ipu_plane);

	return 0;
}

static int ipu_drm_remove(struct platform_device *pdev)
{
	struct ipu_plane *ipu_plane = platform_get_drvdata(pdev);

	imx_drm_remove_plane(ipu_plane->imx_plane);

	ipu_plane_put_resources(ipu_plane);

	return 0;
}

static struct of_device_id ipu_drm_dt_ids[] = {
	{ .compatible = "fsl,imx-ipuv3-plane" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, ipu_drm_dt_ids);

static struct platform_driver ipu_drm_driver = {
	.driver = {
		.name = IMX_PLANE_DRIVER,
		.of_match_table	= ipu_drm_dt_ids,
	},
	.probe = ipu_drm_probe,
	.remove = ipu_drm_remove,
};
module_platform_driver(ipu_drm_driver);

MODULE_AUTHOR("Dmitry Eremin-Solenikov <dmitry_eremin@mentor.com>");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
