/*
 * i.MX IPUv3 Graphics driver
 *
 * Copyright (C) 2011 Sascha Hauer, Pengutronix
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */
#include <linux/module.h>
#include <linux/export.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <drm/drmP.h>
#include <drm/drm_fb_helper.h>
#include <drm/drm_crtc_helper.h>
#include <linux/fb.h>
#include <linux/clk.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <drm/drm_gem_cma_helper.h>
#include <drm/drm_fb_cma_helper.h>

#include "imx-drm.h"

#define DRIVER_DESC		"i.MX IPUv3 Graphics"

struct ipu_framebuffer {
	struct drm_framebuffer	base;
	void			*virt;
	dma_addr_t		phys;
	size_t			len;
};

struct ipu_sync_err {
	int			irq;
	bool                    irq_enabled;
	int                     counter;
	struct work_struct      work;
};

struct ipu_crtc {
	struct drm_fb_helper	fb_helper;
	struct ipu_framebuffer	ifb;
	struct device		*dev;
	struct drm_crtc		base;
	struct imx_drm_plane	*active_plane;
	struct imx_drm_crtc	*imx_crtc;
	struct ipu_soc          *ipu; /* our parent IPU */
	struct ipuv3_channel	*ipu_ch;
	struct ipu_dc		*dc;
	struct ipu_dp		*dp;
	struct dmfc_channel	*dmfc;
	struct ipu_di		*di;
	bool			enabled;
	bool			mode_set_base;
	struct completion	mode_set_base_comp;
	struct ipu_priv		*ipu_priv;
	struct drm_pending_vblank_event *page_flip_event;
	struct drm_framebuffer	*newfb;
	int			eof_irq;
	struct ipu_sync_err     sync_err;

	u32			interface_pix_fmt;
	u32			*interface_pix_map;
	unsigned long		di_clkflags;
};

#define to_ipu_crtc(x) container_of(x, struct ipu_crtc, base)

struct display_channels {
	int idmac;
	int dc;
	int dp;
};

#define NO_DP -1

static const struct display_channels sync_bg_dual_plane = {
	.idmac = IPUV3_CHANNEL_MEM_BG_SYNC,
	.dc    = IPU_DC_CHANNEL_DP_SYNC,
	.dp    = IPU_DP_FLOW_SYNC_BG,
};
static const struct display_channels sync_bg_single_plane = {
	.idmac = IPUV3_CHANNEL_MEM_DC_SYNC,
	.dc    = IPU_DC_CHANNEL_SYNC,
	.dp    = NO_DP,
};

/*
 * This driver does not yet support async flows for "smart" displays,
 * but keep this around for future reference.
 */
#if 0
static const struct display_channels async_bg_dual_plane = {
	.idmac = IPUV3_CHANNEL_MEM_BG_ASYNC,
	.dc    = IPU_DC_CHANNEL_DP_ASYNC,
	.dp    = IPU_DP_FLOW_ASYNC0_BG,
};
static const struct display_channels async_bg_single_plane = {
	.idmac = IPUV3_CHANNEL_MEM_DC_ASYNC,
	.dc    = IPU_DC_CHANNEL_ASYNC,
	.dp    = NO_DP,
};
#endif

/*
 * When a new video mode is about to be set, the DI sync error counter
 * is reset and the interrupt is enabled in preparation for the new mode
 * setting. If this new mode causes DI errors, the DI sync error handler
 * will attempt to retry the mode setting after DRM core releases the
 * mode_config.mutex.
 *
 * di_err_retries is the maximum number of mode setting retries before
 * the handler gives up and disables the interrupt. Default is zero (no
 * retries).
 */
static int di_err_retries;
MODULE_PARM_DESC(di_err_retries,
		 "Max mode set retries from DI sync errors (default: 0)");
module_param(di_err_retries, int, 0600);

static void sync_err_irq_enable(struct ipu_sync_err *sync_err, bool enable)
{
	if (enable && !sync_err->irq_enabled) {
		enable_irq(sync_err->irq);
		sync_err->irq_enabled = true;
	} else if (!enable && sync_err->irq_enabled) {
		disable_irq(sync_err->irq);
		sync_err->irq_enabled = false;
	}
}

static int calc_vref(struct drm_display_mode *mode)
{
	unsigned long htotal, vtotal;

	htotal = mode->htotal;
	vtotal = mode->vtotal;

	if (!htotal || !vtotal)
		return 60;

	return mode->clock * 1000 / vtotal / htotal;
}

static int calc_bandwidth(struct drm_display_mode *mode, unsigned int vref)
{
	return mode->hdisplay * mode->vdisplay * vref;
}

static void ipu_fb_enable(struct ipu_crtc *ipu_crtc)
{
	if (ipu_crtc->enabled)
		return;

	dev_dbg(ipu_crtc->dev, "%s: enter\n", __func__);

	if (ipu_crtc->dp)
		ipu_dp_enable(ipu_crtc->dp);
	ipu_dc_enable(ipu_crtc->dc);
	ipu_di_enable(ipu_crtc->di);
	ipu_dmfc_enable_channel(ipu_crtc->dmfc);

	ipu_idmac_enable_channel(ipu_crtc->ipu_ch);
	ipu_idmac_enable_watermark(ipu_crtc->ipu_ch, true);

	if (ipu_crtc->dp)
		ipu_dp_enable_channel(ipu_crtc->dp);
	ipu_dc_enable_channel(ipu_crtc->dc);
	ipu_di_enable_clock(ipu_crtc->di);

	if (ipu_crtc->active_plane) {
		dev_dbg(ipu_crtc->dev, "%s: enabling active plane\n",
			__func__);
		imx_drm_plane_enable(ipu_crtc->active_plane);
	}

	ipu_crtc->enabled = true;

	dev_dbg(ipu_crtc->dev, "%s: exit\n", __func__);
}

static void ipu_fb_disable(struct ipu_crtc *ipu_crtc)
{
	if (!ipu_crtc->enabled)
		return;

	dev_dbg(ipu_crtc->dev, "%s: enter\n", __func__);

	ipu_crtc->active_plane =
		imx_drm_crtc_find_active_plane(ipu_crtc->imx_crtc);
	if (ipu_crtc->active_plane) {
		dev_dbg(ipu_crtc->dev, "%s: disabling active plane\n",
			__func__);
		imx_drm_plane_disable(ipu_crtc->active_plane);
	}

	if (ipu_crtc->dp)
		ipu_dp_disable_channel(ipu_crtc->dp);
	ipu_dc_disable_channel(ipu_crtc->dc);
	ipu_idmac_enable_watermark(ipu_crtc->ipu_ch, false);
	ipu_idmac_disable_channel(ipu_crtc->ipu_ch);
	ipu_di_uninit_sync_panel(ipu_crtc->di);
	if (ipu_crtc->dp)
		ipu_dp_uninit_channel(ipu_crtc->dp);
	ipu_dc_uninit(ipu_crtc->dc);

	ipu_di_disable(ipu_crtc->di);
	ipu_dmfc_free_bandwidth(ipu_crtc->dmfc);
	ipu_dmfc_disable_channel(ipu_crtc->dmfc);
	ipu_dc_disable(ipu_crtc->dc);
	if (ipu_crtc->dp)
		ipu_dp_disable(ipu_crtc->dp);
	ipu_di_disable_clock(ipu_crtc->di);

	ipu_crtc->enabled = false;

	dev_dbg(ipu_crtc->dev, "%s: exit\n", __func__);
}

static int ipu_page_flip(struct drm_crtc *crtc,
		struct drm_framebuffer *fb,
		struct drm_pending_vblank_event *event)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);
	int ret;

	if (ipu_crtc->newfb || ipu_crtc->mode_set_base)
		return -EBUSY;

	ipu_crtc->page_flip_event = event;

	ret = imx_drm_crtc_vblank_get(ipu_crtc->imx_crtc);
	if (ret) {
		dev_dbg(ipu_crtc->dev, "failed to acquire vblank counter\n");

		return ret;
	}

	ipu_crtc->newfb = fb;

	return 0;
}


static int ipu_gamma_set(struct drm_crtc *crtc, bool enable, u32 *m, u32 *b)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	if (!ipu_crtc->dp)
		return -ENODEV;

	return ipu_dp_set_gamma_correction(ipu_crtc->dp, enable, m, b);
}

/*
 * Normally the DRM Gamma API is used to program a color LUT that contains
 * gamma-corrected pixel values for red, green, and blue input pixel values
 * (normally in the range 0 to 255).
 *
 * However the i.MX6 LUT is only 256 entries, so it only supports 8 bpp
 * indexed pixel format. Therefore if the i.MX6 LUT were used to implement
 * gamma correction, th DRM framebuffer would have to use 8 bpp indexed pixel
 * format which is insufficient for most use cases. To support a gamma
 * correcting LUT with full RGB24 or YUV444 pixel formats, there would have
 * to be 3 separate 256-entry LUTs for each color component.
 *
 * But the i.MX6 does support gamma correction via a set of registers that
 * define a piecewise linear approximation to a luminance gamma correction
 * curve. This function uses this approach.
 *
 * The input pixel values to this function must be in a specific format
 * according to the i.MX6 reference manual (see Table 37-28 in the
 * Rev. 1 TRM, dated 04/2013). This info is reprinted here:
 *
 * "The required Gamma correction slope for a specific display should be
 * provided by the display manufacture. This information can be provided
 * in various forms, as graph or formula. The gamma correction input pixel
 * level (Gin) should be normalized to a maximum of 383. The gamma correction
 * output pixel level (Gout) should be normalized to a maximum of 255. Then
 * the following data should be collected:
 *
 * Table 37-28. Gamma correction values
 *   Gin   Gout
 *   ---   -----
 *     0   Gout0
 *     2   Gout1
 *     4   Gout2
 *     8   Gout3
 *    16   Gout4
 *    32   Gout5
 *    64   Gout6
 *    96   Gout7
 *   128   Gout8
 *   160   Gout9
 *   192   Gout10
 *   224   Gout11
 *   256   Gout12
 *   288   Gout13
 *   320   Gout14
 *   352   Gout15"
 *
 *
 * The 16 Gout values must be placed in the input lum[] array. The green
 * and blue input arrays are ignored.
 *
 * The gamma register values are then computed according to Table 37-29
 * in the Rev. 1 TRM.
 */
static void ipu_drm_gamma_set(struct drm_crtc *crtc,
			      u16 *lum, u16 *g_unused, u16 *b_unused,
			      uint32_t start, uint32_t size)
{
	u32 m[DRM_IMX_GAMMA_SIZE], b[DRM_IMX_GAMMA_SIZE];
	int i;

	if (size != DRM_IMX_GAMMA_SIZE)
		return;

	for (i = 0; i < size; i++) {
		if (i == 0) {
			b[0] = lum[0];
			m[0] = 16 * (lum[1] - lum[0]);
		} else if (i == 15) {
			b[15] = lum[15];
			m[15] = 255 - lum[15];
		} else if (i < 5) {
			b[i] = 2 * lum[i] - lum[i+1];
			m[i] = (lum[i+1] - lum[i]) << (5 - i);
		} else {
			b[i] = lum[i];
			m[i] = lum[i+1] - lum[i];
		}
	}

	ipu_gamma_set(crtc, true, m, b);
}

static const struct drm_crtc_funcs ipu_crtc_funcs = {
	.set_config = drm_crtc_helper_set_config,
	.destroy = drm_crtc_cleanup,
	.page_flip = ipu_page_flip,
	.gamma_set = ipu_drm_gamma_set,
};

static int ipu_drm_set_base(struct drm_crtc *crtc, int bufnum, int x, int y)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);
	struct drm_gem_cma_object *cma_obj;
	struct drm_framebuffer *fb = crtc->fb;
	unsigned long phys;

	cma_obj = drm_fb_cma_get_gem_obj(fb, 0);
	if (!cma_obj) {
		DRM_LOG_KMS("entry is null.\n");
		return -EFAULT;
	}

	phys = cma_obj->paddr;
	phys += x * (fb->bits_per_pixel >> 3);
	phys += y * fb->pitches[0];

	dev_dbg(ipu_crtc->dev, "%s: phys: 0x%lx\n", __func__, phys);
	dev_dbg(ipu_crtc->dev, "%s: xy: %dx%d\n", __func__, x, y);

	ipu_cpmem_set_stride(ipu_crtc->ipu_ch, fb->pitches[0]);

	/* bufnum < 0 means set both buffer addresses */
	if (bufnum < 0) {
		ipu_cpmem_set_buffer(ipu_crtc->ipu_ch, 0, phys);
		ipu_cpmem_set_buffer(ipu_crtc->ipu_ch, 1, phys);
	} else
		ipu_cpmem_set_buffer(ipu_crtc->ipu_ch, bufnum, phys);

	return 0;
}

static void ipu_drm_flip_buffer(struct ipu_crtc *ipu_crtc, int x, int y)
{
	int curbuf;

	curbuf = ipu_idmac_current_buffer(ipu_crtc->ipu_ch);
	curbuf ^= 1;

	dev_dbg(ipu_crtc->dev, "%s: flip to %d\n", __func__, curbuf);

	ipu_drm_set_base(&ipu_crtc->base, curbuf, x, y);
	ipu_idmac_select_buffer(ipu_crtc->ipu_ch, curbuf);
}

static int setup_mode(struct ipu_crtc *ipu_crtc,
		      struct drm_display_mode *mode,
		      int x, int y)
{
	struct drm_framebuffer *fb = ipu_crtc->base.fb;
	int ret;
	struct ipu_di_signal_cfg sig_cfg = {};
	u32 out_pixel_fmt, *out_pixel_map;
	u32 burstsize, v4l2_fmt;

	dev_dbg(ipu_crtc->dev, "%s: mode->hdisplay: %d\n", __func__,
			mode->hdisplay);
	dev_dbg(ipu_crtc->dev, "%s: mode->vdisplay: %d\n", __func__,
			mode->vdisplay);

	ipu_cpmem_zero(ipu_crtc->ipu_ch);

	ret = imx_drm_format_to_v4l(fb->pixel_format, &v4l2_fmt);
	if (ret < 0) {
		dev_err(ipu_crtc->dev, "unsupported pixel format 0x%08x\n",
				fb->pixel_format);
		return ret;
	}


	out_pixel_fmt = ipu_crtc->interface_pix_fmt;
	out_pixel_map = ipu_crtc->interface_pix_map;

	if (mode->flags & DRM_MODE_FLAG_INTERLACE)
		sig_cfg.interlaced = 1;
	if (mode->flags & DRM_MODE_FLAG_PHSYNC)
		sig_cfg.hsync_pol = 1;
	if (mode->flags & DRM_MODE_FLAG_PVSYNC)
		sig_cfg.vsync_pol = 1;
	if (mode->flags & DRM_MODE_FLAG_PCLK)
		sig_cfg.clk_pol = 1;

	sig_cfg.enable_pol = 1;

	sig_cfg.width = mode->hdisplay;
	sig_cfg.height = mode->vdisplay;
	sig_cfg.pixel_fmt = out_pixel_fmt;
	sig_cfg.h_back_porch = mode->htotal - mode->hsync_end;
	sig_cfg.h_sync_len = mode->hsync_end - mode->hsync_start;
	sig_cfg.h_front_porch = mode->hsync_start - mode->hdisplay;

	sig_cfg.v_back_porch = mode->vtotal - mode->vsync_end;
	sig_cfg.v_sync_len = mode->vsync_end - mode->vsync_start;
	sig_cfg.v_front_porch = mode->vsync_start - mode->vdisplay;
	sig_cfg.pixelclock = mode->clock * 1000;
	sig_cfg.clkflags = ipu_crtc->di_clkflags;

	sig_cfg.v_to_h_sync = 0;

	if (ipu_crtc->dp) {
		ret = ipu_dp_setup_channel(ipu_crtc->dp,
				ipu_pixelformat_to_colorspace(v4l2_fmt),
				IPUV3_COLORSPACE_RGB);
		if (ret) {
			dev_err(ipu_crtc->dev,
				"initializing display processor failed with %d\n",
				ret);
			return ret;
		}
		ipu_dp_set_global_alpha(ipu_crtc->dp, 1, 0, 1);
	}

	ret = ipu_dc_init_sync(ipu_crtc->dc, ipu_crtc->di, sig_cfg.interlaced,
			       out_pixel_fmt, out_pixel_map, mode->hdisplay);
	if (ret) {
		dev_err(ipu_crtc->dev,
				"initializing display controller failed with %d\n",
				ret);
		return ret;
	}

	ret = ipu_di_init_sync_panel(ipu_crtc->di, &sig_cfg);
	if (ret) {
		dev_err(ipu_crtc->dev,
				"initializing panel failed with %d\n", ret);
		return ret;
	}

	ipu_cpmem_set_resolution(ipu_crtc->ipu_ch,
				 mode->hdisplay, mode->vdisplay);
	ipu_cpmem_set_fmt(ipu_crtc->ipu_ch, v4l2_fmt);
	ipu_cpmem_set_high_priority(ipu_crtc->ipu_ch);
	ipu_cpmem_set_axi_id(ipu_crtc->ipu_ch, 0);
	ipu_idmac_lock_enable(ipu_crtc->ipu_ch, 8);

	if (ipu_pixelformat_is_planar(v4l2_fmt))
		ipu_cpmem_set_yuv_planar(ipu_crtc->ipu_ch, v4l2_fmt,
				fb->pitches[0], mode->vdisplay);

	burstsize = ipu_cpmem_get_burst_size(ipu_crtc->ipu_ch);
	ret = ipu_dmfc_alloc_bandwidth(ipu_crtc->dmfc,
				       calc_bandwidth(mode, calc_vref(mode)),
				       mode->hdisplay, burstsize);
	if (ret) {
		dev_err(ipu_crtc->dev,
				"allocating dmfc bandwidth failed with %d\n",
				ret);
		return ret;
	}

	/* enable double-buffering */
	ipu_idmac_set_double_buffer(ipu_crtc->ipu_ch, true);
	ipu_idmac_set_triple_buffer(ipu_crtc->ipu_ch, false);

	ipu_drm_set_base(&ipu_crtc->base, -1, x, y);

	/* set buffers ready */
	ipu_idmac_select_buffer(ipu_crtc->ipu_ch, 0);
	ipu_idmac_select_buffer(ipu_crtc->ipu_ch, 1);

	return 0;
}

static void ipu_crtc_handle_pageflip(struct ipu_crtc *ipu_crtc)
{
	struct drm_pending_vblank_event *e;
	struct timeval now;
	unsigned long flags;
	struct drm_device *drm = ipu_crtc->base.dev;

	spin_lock_irqsave(&drm->event_lock, flags);

	e = ipu_crtc->page_flip_event;
	if (!e) {
		imx_drm_crtc_vblank_put(ipu_crtc->imx_crtc);
		spin_unlock_irqrestore(&drm->event_lock, flags);
		return;
	}

	do_gettimeofday(&now);
	e->event.sequence = 0;
	e->event.tv_sec = now.tv_sec;
	e->event.tv_usec = now.tv_usec;
	ipu_crtc->page_flip_event = NULL;

	imx_drm_crtc_vblank_put(ipu_crtc->imx_crtc);

	list_add_tail(&e->base.link, &e->base.file_priv->event_list);

	wake_up_interruptible(&e->base.file_priv->event_wait);

	spin_unlock_irqrestore(&drm->event_lock, flags);
}

static irqreturn_t ipu_irq_handler(int irq, void *dev_id)
{
	struct ipu_crtc *ipu_crtc = dev_id;

	imx_drm_crtc_handle_vblank(ipu_crtc->imx_crtc);

	if (ipu_crtc->newfb) {
		/* this is a page flip request */

		ipu_crtc->base.fb = ipu_crtc->newfb;
		ipu_crtc->newfb = NULL;

		ipu_drm_flip_buffer(ipu_crtc, ipu_crtc->base.x,
				    ipu_crtc->base.y);

		ipu_crtc_handle_pageflip(ipu_crtc);
	} else if (ipu_crtc->mode_set_base) {
		/* this is a mode_set_base request */

		ipu_crtc->mode_set_base = false;

		ipu_drm_flip_buffer(ipu_crtc, ipu_crtc->base.x,
				    ipu_crtc->base.y);
		imx_drm_crtc_vblank_put(ipu_crtc->imx_crtc);

		complete(&ipu_crtc->mode_set_base_comp);
	}

	return IRQ_HANDLED;
}

static irqreturn_t ipu_sync_err_irq_handler(int irq, void *dev_id)
{
	struct ipu_crtc *ipu_crtc = dev_id;

	schedule_work(&ipu_crtc->sync_err.work);

	return IRQ_HANDLED;
}

static void sync_err_work_handler(struct work_struct *w)
{
	struct ipu_sync_err *sync_err = container_of(w, struct ipu_sync_err,
						     work);
	struct ipu_crtc *ipu_crtc = container_of(sync_err, struct ipu_crtc,
						 sync_err);
	struct drm_crtc *crtc = &ipu_crtc->base;
	struct drm_device *drm = crtc->dev;
	int di = ipu_di_get_num(ipu_crtc->di);
	int other_crtc;

	mutex_lock(&drm->mode_config.mutex);

	dev_err(ipu_crtc->dev, "di%d sync error\n", di);

	if (++sync_err->counter > di_err_retries) {
		sync_err_irq_enable(sync_err, false);
		if (di_err_retries)
			dev_alert(ipu_crtc->dev,
				"Too many di%d sync errors, giving up\n", di);
	} else {
		if (ipu_crtc->enabled && crtc->enabled) {
			/* disable the other crtc on this IPU */
			other_crtc = (ipu_get_num(ipu_crtc->ipu) << 1) |
				(di ^ 1);
			imx_drm_crtc_dpms(other_crtc, DRM_MODE_DPMS_OFF);

			ipu_fb_disable(ipu_crtc);

			dev_alert(ipu_crtc->dev, "retrying mode on di%d\n", di);
			setup_mode(ipu_crtc, &crtc->mode, crtc->x, crtc->y);
			ipu_fb_enable(ipu_crtc);

			/* re-enable the other crtc on this IPU */
			imx_drm_crtc_dpms(other_crtc, DRM_MODE_DPMS_ON);
		}
	}

	mutex_unlock(&drm->mode_config.mutex);
}

static void ipu_crtc_dpms(struct drm_crtc *crtc, int mode)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	dev_dbg(ipu_crtc->dev, "%s mode: %d\n", __func__, mode);

	switch (mode) {
	case DRM_MODE_DPMS_ON:
		if (!ipu_crtc->enabled) {
			setup_mode(ipu_crtc, &crtc->mode, crtc->x, crtc->y);
			ipu_fb_enable(ipu_crtc);
		}
		break;
	case DRM_MODE_DPMS_STANDBY:
	case DRM_MODE_DPMS_SUSPEND:
	case DRM_MODE_DPMS_OFF:
		if (ipu_crtc->enabled) {
			ipu_crtc->sync_err.counter = 0;
			sync_err_irq_enable(&ipu_crtc->sync_err, true);
			ipu_fb_disable(ipu_crtc);
		}
		break;
	}
}

static int ipu_crtc_mode_set_base(struct drm_crtc *crtc, int x, int y,
				  struct drm_framebuffer *old_fb)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);
	unsigned int timeout;
	int ret;

	dev_dbg(ipu_crtc->dev, "%s\n", __func__);

	if (ipu_crtc->newfb || ipu_crtc->mode_set_base)
		return -EBUSY;

	ipu_crtc->sync_err.counter = 0;
	sync_err_irq_enable(&ipu_crtc->sync_err, true);

	INIT_COMPLETION(ipu_crtc->mode_set_base_comp);

	ret = imx_drm_crtc_vblank_get(ipu_crtc->imx_crtc);
	if (ret) {
		dev_warn(ipu_crtc->dev, "failed to acquire vblank counter\n");

		return -EBUSY;
	}

	ipu_crtc->mode_set_base = true;

	/* wait for EOF */
	timeout = msecs_to_jiffies(100);
	ret = wait_for_completion_timeout(&ipu_crtc->mode_set_base_comp,
					  timeout);


	if (!ret)
		dev_warn(ipu_crtc->dev, "%s: timeout\n", __func__);

	return 0;
}

static bool ipu_crtc_mode_fixup(struct drm_crtc *crtc,
				  const struct drm_display_mode *mode,
				  struct drm_display_mode *adjusted_mode)
{
	return true;
}

static void ipu_crtc_prepare(struct drm_crtc *crtc)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);
	int other_crtc = (ipu_get_num(ipu_crtc->ipu) << 1) |
		(ipu_di_get_num(ipu_crtc->di) ^ 1);

	ipu_crtc->sync_err.counter = 0;
	sync_err_irq_enable(&ipu_crtc->sync_err, true);

	/* disable the other crtc on this IPU */
	imx_drm_crtc_dpms(other_crtc, DRM_MODE_DPMS_OFF);

	ipu_fb_disable(ipu_crtc);
}

static int ipu_crtc_mode_set(struct drm_crtc *crtc,
			       struct drm_display_mode *orig_mode,
			       struct drm_display_mode *mode,
			       int x, int y,
			       struct drm_framebuffer *old_fb)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	return setup_mode(ipu_crtc, mode, x, y);
}

static void ipu_crtc_commit(struct drm_crtc *crtc)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);
	int other_crtc = (ipu_get_num(ipu_crtc->ipu) << 1) |
		(ipu_di_get_num(ipu_crtc->di) ^ 1);

	ipu_fb_enable(ipu_crtc);

	/* re-enable the other crtc on this IPU */
	imx_drm_crtc_dpms(other_crtc, DRM_MODE_DPMS_ON);
}

static void ipu_crtc_load_lut(struct drm_crtc *crtc)
{
}

static struct drm_crtc_helper_funcs ipu_helper_funcs = {
	.dpms = ipu_crtc_dpms,
	.mode_set_base = ipu_crtc_mode_set_base,
	.mode_fixup = ipu_crtc_mode_fixup,
	.prepare = ipu_crtc_prepare,
	.mode_set = ipu_crtc_mode_set,
	.commit = ipu_crtc_commit,
	.load_lut = ipu_crtc_load_lut,
};

static int ipu_enable_vblank(struct drm_crtc *crtc)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	enable_irq(ipu_crtc->eof_irq);

	return 0;
}

static void ipu_disable_vblank(struct drm_crtc *crtc)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	disable_irq(ipu_crtc->eof_irq);
}

static int ipu_set_interface_pix_fmt(struct drm_crtc *crtc, u32 encoder_type,
				     u32 pixfmt, u32 *pix_map)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	ipu_crtc->interface_pix_fmt = pixfmt;
	ipu_crtc->interface_pix_map = pix_map;

	switch (encoder_type) {
	case DRM_MODE_ENCODER_LVDS:
		ipu_crtc->di_clkflags = IPU_DI_CLKMODE_SYNC |
			IPU_DI_CLKMODE_EXT;
		break;
	case DRM_MODE_ENCODER_TMDS:
		ipu_crtc->di_clkflags = IPU_DI_CLKMODE_EXT;
		break;
	case DRM_MODE_ENCODER_NONE:
		ipu_crtc->di_clkflags = 0;
		break;
	}

	return 0;
}

static int ipu_get_ipu_di(struct drm_crtc *crtc, int *ipu, int *di)
{
	struct ipu_crtc *ipu_crtc = to_ipu_crtc(crtc);

	*ipu = ipu_get_num(ipu_crtc->ipu);
	*di = ipu_di_get_num(ipu_crtc->di);

	return 0;
}

static const struct imx_drm_crtc_helper_funcs ipu_crtc_helper_funcs = {
	.enable_vblank = ipu_enable_vblank,
	.disable_vblank = ipu_disable_vblank,
	.set_interface_pix_fmt = ipu_set_interface_pix_fmt,
	.gamma_set = ipu_gamma_set,
	.get_ipu_di = ipu_get_ipu_di,
	.crtc_funcs = &ipu_crtc_funcs,
	.crtc_helper_funcs = &ipu_helper_funcs,
};

static void ipu_put_resources(struct ipu_crtc *ipu_crtc)
{
	if (ipu_crtc->eof_irq)
		devm_free_irq(ipu_crtc->dev, ipu_crtc->eof_irq, ipu_crtc);
	if (ipu_crtc->sync_err.irq)
		devm_free_irq(ipu_crtc->dev, ipu_crtc->sync_err.irq, ipu_crtc);
	if (!IS_ERR_OR_NULL(ipu_crtc->ipu_ch))
		ipu_idmac_put(ipu_crtc->ipu_ch);
	if (!IS_ERR_OR_NULL(ipu_crtc->dc))
		ipu_dc_put(ipu_crtc->dc);
	if (!IS_ERR_OR_NULL(ipu_crtc->dmfc))
		ipu_dmfc_put(ipu_crtc->dmfc);
	if (!IS_ERR_OR_NULL(ipu_crtc->dp))
		ipu_dp_put(ipu_crtc->dp);
	if (!IS_ERR_OR_NULL(ipu_crtc->di))
		ipu_di_put(ipu_crtc->di);
}

/*
 * Locate and return true if there is an overlay plane that is
 * pointing to this crtc as its possible crtc. This is used
 * to configure the crtc for single or dual plane mode.
 */
static bool plane_is_attached(struct ipu_crtc *ipu_crtc,
			      struct device_node *node)
{
	struct device_node *ipu_parent = of_get_parent(node);
	struct device_node *child, *crtc;
	bool ret = false;

	for_each_child_of_node(ipu_parent, child) {
		if (!of_device_is_compatible(child, "fsl,imx-ipuv3-plane")) {
			of_node_put(child);
			continue;
		}

		crtc = of_parse_phandle(child, "crtcs", 0);
		of_node_put(child);
		if (crtc) {
			of_node_put(crtc);
			if (crtc == node) {
				ret = true;
				break;
			}
		}
	}

	of_node_put(ipu_parent);
	return ret;
}

static int ipu_get_resources(struct ipu_crtc *ipu_crtc,
			     struct device_node *node)
{
	const struct display_channels *ch;
	u32 di;
	int ret;

	ipu_crtc->ipu = dev_get_drvdata(ipu_crtc->dev->parent);

	ret = of_property_read_u32(node, "di", &di);
	if (ret < 0)
		goto err_out;

	ipu_crtc->di = ipu_di_get(ipu_crtc->ipu, di);
	if (IS_ERR(ipu_crtc->di)) {
		ret = PTR_ERR(ipu_crtc->di);
		goto err_out;
	}

	if (plane_is_attached(ipu_crtc, node)) {
		dev_info(ipu_crtc->dev, "dual plane mode\n");
		ch = &sync_bg_dual_plane;
	} else {
		dev_info(ipu_crtc->dev, "single plane mode\n");
		ch = &sync_bg_single_plane;
	}

	ipu_crtc->ipu_ch = ipu_idmac_get(ipu_crtc->ipu, ch->idmac, false);
	if (IS_ERR_OR_NULL(ipu_crtc->ipu_ch)) {
		ret = PTR_ERR(ipu_crtc->ipu_ch);
		goto err_out;
	}

	ipu_crtc->dc = ipu_dc_get(ipu_crtc->ipu, ch->dc);
	if (IS_ERR(ipu_crtc->dc)) {
		ret = PTR_ERR(ipu_crtc->dc);
		goto err_out;
	}

	ipu_crtc->dmfc = ipu_dmfc_get(ipu_crtc->ipu, ch->idmac);
	if (IS_ERR(ipu_crtc->dmfc)) {
		ret = PTR_ERR(ipu_crtc->dmfc);
		goto err_out;
	}

	if (ch->dp != NO_DP) {
		ipu_crtc->dp = ipu_dp_get(ipu_crtc->ipu, ch->dp);
		if (IS_ERR(ipu_crtc->dp)) {
			ret = PTR_ERR(ipu_crtc->dp);
			goto err_out;
		}
	}

	return 0;
err_out:
	ipu_put_resources(ipu_crtc);

	return ret;
}

static int ipu_get_irqs(struct ipu_crtc *ipu_crtc)
{
	int di = ipu_di_get_num(ipu_crtc->di);
	int ret;

	ipu_crtc->eof_irq = ipu_idmac_channel_irq(ipu_crtc->ipu,
						  ipu_crtc->ipu_ch,
						  IPU_IRQ_EOF);
	ret = devm_request_irq(ipu_crtc->dev, ipu_crtc->eof_irq,
			       ipu_irq_handler, 0,
			       "imx_drm", ipu_crtc);
	if (ret < 0) {
		dev_err(ipu_crtc->dev,
			"eof irq request failed with %d.\n", ret);
		return ret;
	}

	disable_irq(ipu_crtc->eof_irq);

	INIT_WORK(&ipu_crtc->sync_err.work, sync_err_work_handler);

	ipu_crtc->sync_err.irq = ipu_error_irq(ipu_crtc->ipu,
					       di == 0 ?
					       IPU_IRQ_DI0_SYNC_DISP_ERR :
					       IPU_IRQ_DI1_SYNC_DISP_ERR);
	ret = devm_request_irq(ipu_crtc->dev, ipu_crtc->sync_err.irq,
			       ipu_sync_err_irq_handler, 0,
			       "imx_drm", ipu_crtc);
	if (ret < 0) {
		dev_err(ipu_crtc->dev,
			"sync_err irq request failed with %d.\n", ret);
		return ret;
	}

	disable_irq(ipu_crtc->sync_err.irq);

	return 0;
}

static int ipu_crtc_init(struct ipu_crtc *ipu_crtc,
			 struct device_node *node)
{
	int ret, id;

	init_completion(&ipu_crtc->mode_set_base_comp);

	ret = ipu_get_resources(ipu_crtc, node);
	if (ret) {
		dev_err(ipu_crtc->dev, "getting resources failed with %d.\n",
				ret);
		return ret;
	}

	id = (ipu_get_num(ipu_crtc->ipu) << 1) | ipu_di_get_num(ipu_crtc->di);

	dev_set_name(ipu_crtc->dev, "crtc%d", id);

	ret = imx_drm_add_crtc(&ipu_crtc->base,
			       &ipu_crtc->imx_crtc,
			       &ipu_crtc_helper_funcs, THIS_MODULE,
			       node, id);
	if (ret) {
		dev_err(ipu_crtc->dev, "adding crtc failed with %d.\n", ret);
		goto err_put_resources;
	}

	ret = ipu_get_irqs(ipu_crtc);
	if (ret) {
		dev_err(ipu_crtc->dev, "getting irqs failed with %d.\n", ret);
		goto err_put_resources;
	}

	ret = drm_mode_crtc_set_gamma_size(&ipu_crtc->base,
					   DRM_IMX_GAMMA_SIZE);
	if (ret) {
		dev_err(ipu_crtc->dev, "setting gamma size failed with %d.\n",
			ret);
		goto err_put_resources;
	}

	return 0;

err_put_resources:
	ipu_put_resources(ipu_crtc);

	return ret;
}

static int ipu_drm_probe(struct platform_device *pdev)
{
	struct device_node *node = pdev->dev.of_node;
	struct ipu_crtc *ipu_crtc;
	int ret;

	pdev->dev.coherent_dma_mask = DMA_BIT_MASK(32);

	ipu_crtc = devm_kzalloc(&pdev->dev, sizeof(*ipu_crtc), GFP_KERNEL);
	if (!ipu_crtc)
		return -ENOMEM;

	ipu_crtc->dev = &pdev->dev;

	ret = ipu_crtc_init(ipu_crtc, node);

	platform_set_drvdata(pdev, ipu_crtc);

	return 0;
}

static int ipu_drm_remove(struct platform_device *pdev)
{
	struct ipu_crtc *ipu_crtc = platform_get_drvdata(pdev);

	ipu_fb_disable(ipu_crtc);

	imx_drm_remove_crtc(ipu_crtc->imx_crtc);

	ipu_put_resources(ipu_crtc);

	return 0;
}

static struct of_device_id ipu_drm_dt_ids[] = {
	{ .compatible = "fsl,imx-ipuv3-crtc" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, ipu_drm_dt_ids);

static struct platform_driver ipu_drm_driver = {
	.driver = {
		.name = IMX_CRTC_DRIVER,
		.of_match_table = ipu_drm_dt_ids,
	},
	.probe = ipu_drm_probe,
	.remove = ipu_drm_remove,
};
module_platform_driver(ipu_drm_driver);

MODULE_AUTHOR("Sascha Hauer <s.hauer@pengutronix.de>");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
