/*
 * Author: Adnan Ali<adnan.ali@codethink.co.uk>
 * This file is licensed under the GPL2 license.
 *
 *#############################################################################
 * GPL
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 *#############################################################################
 */
#include <uapi/linux/caam_ee.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <linux/spinlock.h>
#include <linux/dma-mapping.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include "desc_constr.h"
#include "desc.h"
#include "error.h"
#include "jr.h"
#include "sm.h"

#define CAAM_EE "caamee"
static dev_t caam_ee_dev_number;
static struct cdev *caam_ee_object;
struct class *caam_ee_class;

#define CAAM_MAX_KEY_SIZE 128

struct caam_ee_priv {
	struct device *dev;
	uint32_t key_id;
	uint32_t flags;
	uint8_t *rndkey;
#define CAAM_EE_SESSION_ENCRYPTED 0x1
};

struct mutex wraplock;

static u32 caam_ee_cipher_type(u32 type)
{
	u32 cmd = type & CAAM_ENCODE_MASK;
	u32 ret = 0;

	switch (cmd) {
	case CAAM_ENCODE_AES_256:
	case CAAM_ENCODE_AES_192:
	case CAAM_ENCODE_AES_128:
		ret = OP_ALG_ALGSEL_AES;
		break;
	case CAAM_ENCODE_3DES_192:
	case CAAM_ENCODE_3DES_128:
		ret = OP_ALG_ALGSEL_3DES;
		break;
	case CAAM_ENCODE_DES_64:
		ret = OP_ALG_ALGSEL_DES;
		break;
	}

	return ret;
}

static u32 caam_ee_operation_state(u32 state)
{
	u32 cmd = state & CAAM_OP_MASK;
	u32 ret = 0;

	switch (cmd) {
	case CAAM_OP_SINGLE:
		ret = OP_ALG_AS_INITFINAL;
		break;
	case CAAM_OP_MULTI_INIT:
		ret = OP_ALG_AS_INIT;
		break;
	case CAAM_OP_MULTI_UPDATE:
		ret = OP_ALG_AS_UPDATE;
		break;
	case CAAM_OP_MULTI_FINALIZE:
		ret = OP_ALG_AS_FINALIZE;
		break;
	}

	return ret;
}

static u32 wraptype(u32 state)
{
	u32 cmd = state & CAAM_WRAP_MASK;
	u32 rtnval = 0;

	switch (cmd) {
	case CAAM_WRAP_CCM:
	      rtnval = 0;
	      break;
	case CAAM_WRAP_ECB:
	      rtnval = 1;
	      break;
	}

	return rtnval;
}

static u32 caam_ee_hashalgtype(u32 state)
{
	u32 cmd = state & CAAM_HMAC_MASK;
	u32 rtnval = 0;

	switch (cmd) {
	case CAAM_ENCODE_MD5:
		rtnval = OP_ALG_ALGSEL_MD5;
		break;
	case CAAM_ENCODE_SHA_1:
		rtnval = OP_ALG_ALGSEL_SHA1;
		break;
	case CAAM_ENCODE_SHA_224:
		rtnval = OP_ALG_ALGSEL_SHA224;
		break;
	case CAAM_ENCODE_SHA_256:
		rtnval = OP_ALG_ALGSEL_SHA256;
		break;
	case CAAM_ENCODE_SHA_384:
		rtnval = OP_ALG_ALGSEL_SHA384;
		break;
	case CAAM_ENCODE_SHA_512:
		rtnval = OP_ALG_ALGSEL_SHA512;
		break;
	}

	return rtnval;
}
static long caam_ee_set_session_key(struct file *file,
			struct caam_ee_session_key_data *params)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;

	/* FIXME - we need to be able to unset this - requires API change. */
	priv->flags |= CAAM_EE_SESSION_ENCRYPTED;
	priv->key_id = params->key_id;

	return 0;
}

/**
 * Get key from private descriptor (only applies to larg wrap/unwrap)
 */
static u8 *caam_get_priv_key(struct file *file)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	u8 *key;

	key = kzalloc(KEY32_BB_SIZE, GFP_KERNEL);
	if (!key)
		goto fail;

	memcpy(key, priv->rndkey, KEY_SIZE);

	return key;

fail:
	return NULL;
}

/* DMA helper functions */
static inline dma_addr_t caam_dma_in(struct caam_ee_priv *priv,
	char *buf, u32 len)
{
	dma_addr_t ret;

	ret = dma_map_single(priv->dev, buf, len, DMA_TO_DEVICE);
	dma_sync_single_for_device(priv->dev, ret, len, DMA_TO_DEVICE);

	return ret;
}

static inline dma_addr_t caam_dma_out(struct caam_ee_priv *priv,
	char *buf, u32 len)
{
	return dma_map_single(priv->dev, buf, len, DMA_FROM_DEVICE);
}

static inline void caam_dma_sync(struct caam_ee_priv *priv,
	dma_addr_t buf, u32 len)
{
	dma_sync_single_for_cpu(priv->dev, buf, len, DMA_FROM_DEVICE);
}

static long __caam_wrapunwrap_ioctl(struct file *file,
				struct caam_ee_params *data,
				u32 optype, u32 nostore)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	u32 keyid = priv->key_id;
	u32 keylen = 0;
	long ret = 0;
	u8 *key = NULL;

	mutex_lock(&wraplock);
	pr_debug("wrap/unwrap function\n");

	if (keyid && (!nostore)) {
		/* read the key */
		key = caam_kha_readkey(keyid);
		if (!key) {
			ret = -ENOMEM;
			goto out;
		}
		keylen = KEY_SIZE;
	}

	ret = sm_keystore_wrap_unwrap_data(priv->dev,
			SM_DATA_BANK, data->buf, data->auxbuf,
			data->len, key, keylen, optype, wraptype(data->flags));
	if (ret)
		goto failmem;

	goto out;

failmem:
	kfree(key);
out:
	mutex_unlock(&wraplock);
	return ret;
}
/* IOCTL handlers */
static long caam_ee_wrap_ioctl(struct file *file,
				struct caam_ee_params *data, u32 nostore)
{

	return __caam_wrapunwrap_ioctl(file, data,
					OP_TYPE_ENCAP_PROTOCOL, nostore);
}

static long caam_ee_unwrap_ioctl(struct file *file,
				struct caam_ee_params *data, u32 nostore)
{

	return __caam_wrapunwrap_ioctl(file, data,
					OP_TYPE_DECAP_PROTOCOL, nostore);
}

/* encrypt/decrypt desc */
static u32 *caam_ee_create_encdec_desc(dma_addr_t key, u16 keysz,
		dma_addr_t indata, dma_addr_t outdata, u16 sz, u32 cipherdir,
		u32 keymode)
{
	u32 prog_len = 8;
	u32 *desc;

	desc = kmalloc(prog_len * sizeof(u32), GFP_KERNEL);
	if (!desc)
		goto out;

	desc[0] = CMD_DESC_HDR | HDR_ONE | (prog_len & HDR_DESCLEN_MASK);
	desc[1] = CMD_KEY | CLASS_1 | (keysz & KEY_LENGTH_MASK) | keymode;
	desc[2] = (u32)key;
	desc[3] = CMD_OPERATION | OP_TYPE_CLASS1_ALG | OP_ALG_AAI_ECB |
		  cipherdir;
	desc[4] = CMD_FIFO_LOAD | FIFOLD_CLASS_CLASS1 |
		  FIFOLD_TYPE_MSG | FIFOLD_TYPE_LAST1 | sz;
	desc[5] = (u32)indata;
	desc[6] = CMD_FIFO_STORE | FIFOST_TYPE_MESSAGE_DATA | sz;
	desc[7] = (u32)outdata;

out:
	return desc;
}

/* HMAC desc*/
static u32 *caam_ee_create_hmac_desc(dma_addr_t key, u16 keylen,
			dma_addr_t indata, u16 inlen, dma_addr_t outdata,
			u16 outlen, u32 alg)
{
	u32 prog_len = 15;
	u32 *desc = NULL;

	desc = kmalloc(prog_len * sizeof(u32), GFP_KERNEL);
	if (!desc)
		goto out;

	init_job_desc(desc, 0);
	append_key(desc, (dma_addr_t)key, keylen, CLASS_2);
	append_operation(desc, alg);
	append_fifo_load(desc, (dma_addr_t)indata, inlen,
				(CLASS_2+FIFOLD_TYPE_MSG + FIFOLD_TYPE_LAST2));
	append_jump(desc, 1);
	append_store(desc, (dma_addr_t)outdata, outlen,
				(CLASS_2 + LDST_SRCDST_BYTE_CONTEXT));

out:
	return desc;
}

/* RNG desc*/
static u32 *caam_ee_create_rnd_desc(dma_addr_t outdata,	u16 outlen, u32 alg)
{
	u32 prog_len = 8;
	u32 *desc = NULL;

	desc = kmalloc(prog_len * sizeof(u32), GFP_KERNEL);
	if (!desc)
		goto out;

	init_job_desc(desc, 0);
	append_operation(desc, alg);
	append_fifo_store(desc, (dma_addr_t)outdata, outlen,
							FIFOST_TYPE_RNGSTORE);

out:
	return desc;
}

struct exec_res {
	int error;
	struct completion completion;
};

static void caam_exec_finish(struct device *dev, u32 *desc,
					u32 err, void *context)
{
	struct exec_res *res = context;

	if (err) {
		char tmp[CAAM_ERROR_STR_MAX];
		dev_err(dev, "caam_sm_test ERROR %08x: %s\n",
				err, caam_jr_strstatus(tmp, err));
	}

	res->error = err;
	complete(&res->completion);
}

static int caam_exec(struct caam_ee_priv *priv, u32 *jobdesc)
{
	struct exec_res res;
	struct caam_drv_private_sm *kspriv;
	int ret;

	kspriv = dev_get_drvdata(priv->dev);

	init_completion(&res.completion);

	ret = caam_jr_enqueue(kspriv->smringdev, jobdesc, caam_exec_finish,
			      &res);
	if (!ret) {
		wait_for_completion_interruptible(&res.completion);
		ret = res.error;
	}

	return ret;
}

static long __caam_ee_encdecrypt_ioctl(struct file *file,
				struct caam_ee_params *data,
				u32 alg, u32 nostore)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	u32 keyid = priv->key_id;
	u8 *key;
	dma_addr_t enckey_dma, deckey_dma, key_dma;
	u32 __iomem *jdesc, jstat = 0;
	long ret;

	pr_debug("encryption function\n");

	/* Cipher type and operation state select */
	alg |= caam_ee_cipher_type(data->flags) |
	      caam_ee_operation_state(data->flags);

	/* Get key from keystore */
	if (!nostore)
		key = caam_kha_readkey(keyid); /* key from store */
	else
		key = caam_get_priv_key(file); /* priv key */

	if (!key) {
		ret = -ENOMEM;
		goto out;
	}

	/* Setup CAAM pointers */
	enckey_dma = caam_dma_in(priv, data->buf, data->len);
	key_dma    = caam_dma_in(priv, key, KEY_SIZE);
	deckey_dma = caam_dma_out(priv, data->auxbuf, data->len);

	/* Make decrypt job desc*/
	jdesc = caam_ee_create_encdec_desc(key_dma, KEY_SIZE, enckey_dma,
				deckey_dma, data->len, alg, 0);
	if (!jdesc) {
		ret = -ENOMEM;
		goto out_fail_dma;
	}

	/* Now hand job over to CAAM and wait for it to finish */
	jstat = caam_exec(priv, jdesc);
	if (jstat != 0) {
		ret = -EFAULT;
		goto out_fail_dma;
	}

	/* Read the encrypted data back */
	caam_dma_sync(priv, deckey_dma, data->len);

	ret = 0;

out_fail_dma:
	dma_unmap_single(priv->dev, enckey_dma, data->len, DMA_TO_DEVICE);
	dma_unmap_single(priv->dev, key_dma, KEY_SIZE, DMA_TO_DEVICE);
	dma_unmap_single(priv->dev, deckey_dma, data->len, DMA_FROM_DEVICE);

	kfree(key);
	kfree(jdesc);
out:
	return ret;
}

static long caam_ee_encrypt_ioctl(struct file *file,
				struct caam_ee_params *data, u32 nostore)
{
	return __caam_ee_encdecrypt_ioctl(file, data, OP_ALG_ENCRYPT, nostore);
}

static long caam_ee_decrypt_ioctl(struct file *file,
				struct caam_ee_params *data, u32 nostore)
{
	return __caam_ee_encdecrypt_ioctl(file, data, OP_ALG_DECRYPT, nostore);
}

static long caam_ee_verify_ioctl(struct file *file,
	struct caam_ee_params *data)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	u32 keyid = priv->key_id;
	u8 *key = NULL, *tmphash = NULL;
	dma_addr_t data_dma, sig_dma, key_dma;
	u32 __iomem *jdesc = NULL, jstat = 0;
	u32 alg = OP_ALG_AAI_HMAC | OP_TYPE_CLASS2_ALG |
		OP_ALG_DECRYPT;
	long ret;

	pr_debug("verify function\n");

	/* Cipher type and operation state select */
	alg |=  caam_ee_hashalgtype(data->flags) |
		caam_ee_operation_state(data->flags);

	/* create buffer for storing hash */
	tmphash = kmalloc(data->auxlen, GFP_KERNEL);
	if (!tmphash) {
		ret = -ENOMEM;
		goto out;
	}

	/* Get key from keystore */
	key = caam_kha_readkey(keyid);
	if (!key) {
		ret = -ENOMEM;
		goto out;
	}

	/* Setup CAAM pointers */
	data_dma = caam_dma_in(priv, data->buf, data->len);
	key_dma    = caam_dma_in(priv, key, KEY_SIZE);
	sig_dma = caam_dma_out(priv, tmphash, data->auxlen);


	/* Make hmac job desc*/
	jdesc = caam_ee_create_hmac_desc(key_dma, KEY_SIZE, data_dma, data->len,
			sig_dma, data->auxlen, alg);
	if (!jdesc) {
		ret = -ENOMEM;
		goto out_fail_dma;
	}

	/* Now hand job over to CAAM and wait for it to finish */
	jstat = caam_exec(priv, jdesc);
	if (jstat != 0) {
		ret = -EFAULT;
		goto out_fail_dma;
	}

	/* Read the sigature data back */
	caam_dma_sync(priv, sig_dma, data->auxlen);

	/* compare for verification */
	if (memcmp(data->auxbuf, tmphash, data->auxlen)) {
		pr_debug("signature not verified\n");
		ret = -EBADE;
		goto out_fail_dma;
	}
	pr_debug("signature verified\n");

	ret = 0;

out_fail_dma:
	dma_unmap_single(priv->dev, data_dma, data->len, DMA_TO_DEVICE);
	dma_unmap_single(priv->dev, key_dma, KEY_SIZE, DMA_TO_DEVICE);
	dma_unmap_single(priv->dev, sig_dma, data->auxlen, DMA_FROM_DEVICE);
out:
	kfree(key);
	kfree(tmphash);
	kfree(jdesc);

	return ret;
}

/**
 * Generat random key(32 byte long)
 */
static long __caam_rnd_generator(struct file *file, u8 *outdata)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	dma_addr_t rnd_dma;
	u8 *tmprnd;
	u32 *jdesc, jstat, alg = OP_TYPE_CLASS1_ALG + OP_ALG_ALGSEL_RNG;
	long ret = 0;

	jdesc = NULL;

	/* create buffer for storing rnd key */
	tmprnd = kmalloc(KEY_SIZE, GFP_KERNEL);
	if (!tmprnd) {
		ret = -ENOMEM;
		goto fail;
	}

	/* setup dma for rnd job */
	rnd_dma = caam_dma_out(priv, tmprnd, KEY_SIZE);

	/* Create rnd number job descriptor */
	jdesc = caam_ee_create_rnd_desc(rnd_dma, KEY_SIZE, alg);
	if (!jdesc) {
		ret = -ENOMEM;
		goto fail_dma;
	}

	/* Now hand job over to CAAM and wait for it to finish */
	jstat = caam_exec(priv, jdesc);
	if (jstat != 0) {
		ret = -EFAULT;
		goto fail_dma;
	}

	/* Read the rnd data back */
	caam_dma_sync(priv, rnd_dma, KEY_SIZE);

	/* copy data to out buffer */
	memcpy(outdata, tmprnd, KEY_SIZE);

fail_dma:
	dma_unmap_single(priv->dev, rnd_dma, KEY_SIZE, DMA_FROM_DEVICE);
fail:
	kfree(tmprnd);
	kfree(jdesc);

	return ret;
}

/**
 * Wrap/Unwrap of large data
 *
 */
static long caam_wrap_large_ioctl(struct file *file,
				struct caam_ee_params *data)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	struct caam_ee_params tmpdata;
	u32 offset = 0, state = caam_ee_operation_state(data->flags);
	u8 *key, *hash = NULL, *wrapkey = NULL;
	u8 *backup = NULL;
	long ret = 0;

	tmpdata.buf = tmpdata.auxbuf = NULL;
	pr_debug("wrap large data\n");
	if ((data->auxlen < (KEY32_BB_SIZE + KEY_SIZE + 16)) &&
		((state == OP_ALG_AS_INIT) || (state == OP_ALG_AS_INITFINAL))) {
		ret  = -EMSGSIZE;
		goto fail;
	} else if ((data->auxlen < (KEY_SIZE + 16)) &&
					((state == OP_ALG_AS_UPDATE) ||
					(state == OP_ALG_AS_FINALIZE))) {
		ret  = -EMSGSIZE;
		goto fail;
	}

	/* Init the structure */
	tmpdata.buf = kmalloc(data->len, GFP_KERNEL);
	tmpdata.len = data->len;
	tmpdata.auxbuf = kmalloc(data->auxlen, GFP_KERNEL);
	tmpdata.auxlen = data->auxlen;
	tmpdata.flags = (CAAM_WRAP_CCM | CAAM_ENCODE_AES_256 | CAAM_OP_SINGLE);

	if (!tmpdata.buf || !tmpdata.auxbuf) {
		ret = -ENOMEM;
		goto fail;
	}

	if ((state == OP_ALG_AS_INIT) || (state == OP_ALG_AS_INITFINAL)) {
		/* create buffer for storing rnd key */
		key = kmalloc(KEY_SIZE, GFP_KERNEL);
		if (!key) {
			ret = -ENOMEM;
			goto fail;
		}

		ret = __caam_rnd_generator(file, key);

		/* Don't free the key as its assigned to private data*/
		priv->rndkey = key;

		/* Wrap the rnd key*/
		wrapkey  = kmalloc(KEY32_BB_SIZE, GFP_KERNEL);
		if (!wrapkey) {
			ret = -ENOMEM;
			goto fail;
		}

		/* save buffer */
		backup = tmpdata.buf;
		tmpdata.buf = kmalloc(KEY32_BB_SIZE, GFP_KERNEL);
		tmpdata.len = KEY_SIZE;
		tmpdata.auxlen = KEY32_BB_SIZE;
		if (!tmpdata.buf) {
			ret = -ENOMEM;
			goto fail;
		}

		/* Copy the key to be wrapped */
		memcpy(tmpdata.buf, key, KEY_SIZE);

		/* Do the wrap operation */
		ret = caam_ee_wrap_ioctl(file, &tmpdata, 1);
		if (ret)
			goto fail;

		memcpy(wrapkey, tmpdata.auxbuf, KEY32_BB_SIZE);

		/* retrieve buffer */
		kfree(tmpdata.buf);
		tmpdata.buf = backup;
		tmpdata.len = data->len;
		tmpdata.auxlen = data->auxlen;
		backup = NULL;
	}

	/* create hash of red data */
	hash = kmalloc(KEY_SIZE, GFP_KERNEL);
	if (!hash) {
		ret = -ENOMEM;
		goto fail;
	}

	ret = caam_create_hash(data->buf, data->len, hash);
	if (ret)
		goto fail;

	/* Encrypt the data */
	memcpy(tmpdata.buf, data->buf, data->len);
	tmpdata.auxlen =  tmpdata.len;
	tmpdata.len =  data->len;
	ret = caam_ee_encrypt_ioctl(file, &tmpdata, 1);
	if (ret)
		goto fail;

	/* stream for userspace */
	if ((state == OP_ALG_AS_INIT) || (state == OP_ALG_AS_INITFINAL)) {
		memcpy(data->auxbuf + offset, wrapkey, KEY32_BB_SIZE);
		offset += KEY32_BB_SIZE;
	}

	memcpy(data->auxbuf + offset, tmpdata.auxbuf, data->len);
	offset += data->len;
	memcpy(data->auxbuf + offset, hash, KEY_SIZE);

fail:
	kfree(hash);
	kfree(wrapkey);
	kfree(tmpdata.buf);
	kfree(backup);
	kfree(tmpdata.auxbuf);

	return ret;
}

static long caam_unwrap_large_ioctl(struct file *file,
				struct caam_ee_params *data)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	struct caam_ee_params tmpdata;
	u32 offset  = 0, state = caam_ee_operation_state(data->flags);
	u8 *key, *backup = NULL, *hash = NULL;
	long ret = 0;

	/* Init the structure */
	tmpdata.buf = kmalloc(data->len, GFP_KERNEL);
	tmpdata.len = data->len;
	tmpdata.auxbuf = kmalloc(data->auxlen, GFP_KERNEL);
	tmpdata.auxlen = data->auxlen;
	tmpdata.flags = (CAAM_WRAP_CCM | CAAM_ENCODE_AES_256 | CAAM_OP_SINGLE);

	if (!tmpdata.buf || !tmpdata.auxbuf) {
		ret = -ENOMEM;
		goto fail;
	}

	if ((state == OP_ALG_AS_INIT) || (state == OP_ALG_AS_INITFINAL)) {
		/* create buffer for storing rnd key */
		key = kmalloc(KEY_SIZE, GFP_KERNEL);
		if (!key) {
			ret = -ENOMEM;
			goto fail;
		}

		/* Save buffer address */
		backup = tmpdata.auxbuf;
		tmpdata.len = KEY32_BB_SIZE;
		tmpdata.auxlen = KEY_SIZE;
		tmpdata.auxbuf = kmalloc(KEY_SIZE, GFP_KERNEL);
		if (!tmpdata.auxbuf) {
			kfree(key);
			ret = -ENOMEM;
			goto fail;
		}

		/* copy the key blob */
		memcpy(tmpdata.buf, data->buf + offset, KEY32_BB_SIZE);
		offset += KEY32_BB_SIZE;

		/* Do the unwrap operation */
		ret = caam_ee_unwrap_ioctl(file, &tmpdata, 1);
		if (ret) {
			kfree(key);
			goto fail;
		}

		/* copy the key to session */
		memcpy(key, tmpdata.auxbuf, KEY_SIZE);

		/* Don't free the key as its assigned to private data*/
		priv->rndkey = key;

		/* retreive buffer address */
		kfree(tmpdata.auxbuf);
		tmpdata.auxbuf = backup;
		tmpdata.len = data->len;
		tmpdata.auxlen = data->auxlen;
		backup = NULL;
	}

	/* Decrypt the data */
	memcpy(tmpdata.buf, data->buf + offset, data->auxlen);
	tmpdata.auxlen = tmpdata.len =  data->auxlen;
	offset += data->auxlen;
	ret = caam_ee_decrypt_ioctl(file, &tmpdata, 1);
	if (ret)
		goto fail;

	/* create hash of red data */
	hash = kmalloc(KEY_SIZE, GFP_KERNEL);
	if (!hash) {
		ret = -ENOMEM;
		goto fail;
	}

	ret = caam_create_hash(tmpdata.auxbuf, tmpdata.auxlen, hash);
	if (ret)
		goto fail;

	if (memcmp(data->buf + offset, hash, KEY_SIZE) != 0) {
		ret = -EBADMSG;
		goto fail;
	}

	pr_debug("Hash verified ok\n");
	memcpy(data->auxbuf, tmpdata.auxbuf, tmpdata.auxlen);

fail:
	kfree(tmpdata.buf);
	kfree(tmpdata.auxbuf);
	kfree(backup);
	kfree(hash);

	return ret;
}

static long __caam_ee_ioctl(unsigned int cmd, struct file *file,
		struct caam_ee_params *params)
{
	long ret = -EINVAL;
	struct caam_ee_params data;

	/* FIXME Check len and auxlen for sanity here. */

	if (!params->len || !params->auxlen || !params->buf || !params->auxbuf)
		goto out;

	data.buf = kmalloc(params->len, GFP_KERNEL);
	data.len = params->len;
	data.auxbuf = kmalloc(params->auxlen, GFP_KERNEL);
	data.auxlen = params->auxlen;
	data.flags = params->flags;

	if (!data.buf || !data.auxbuf)
		goto out_free;

	ret = copy_from_user(data.buf, params->buf, params->len);
	if (ret)
		goto out_free;

	if (cmd == CAAM_VERIFY_HMAC) {
		ret = copy_from_user(data.auxbuf, params->auxbuf,
								params->auxlen);
		if (ret)
			goto out_free;
	}

	switch (cmd) {
	case CAAM_WRAP_BLOB:
		ret = caam_wrap_large_ioctl(file, &data);
	break;
	case CAAM_UNWRAP_BLOB:
		ret = caam_unwrap_large_ioctl(file, &data);
	break;
	case CAAM_ENCRYPT_BLOCK:
		ret = caam_ee_encrypt_ioctl(file, &data, 0);
	break;
	case CAAM_DECRYPT_BLOCK:
		ret = caam_ee_decrypt_ioctl(file, &data, 0);
	break;
	case CAAM_VERIFY_HMAC:
		ret = caam_ee_verify_ioctl(file, &data);
	break;
	}

	if (ret)
		goto out_free;

	if (cmd != CAAM_VERIFY_HMAC)
		ret = copy_to_user(params->auxbuf, data.auxbuf, params->auxlen);

out_free:
	kfree(data.buf);
	kfree(data.auxbuf);
out:
	return ret;
}

static long caam_ee_ioctl(struct file *file, unsigned int cmd,
		unsigned long arg_)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;
	void __user *arg = (void __user *)arg_;
	int ret = -EINVAL;

	pr_debug("in caam_ee ioctl %d\n", cmd);

	switch (cmd) {
	case CAAM_SET_SESSION_KEY:
	{
		struct caam_ee_session_key_data params;

		ret = copy_from_user(&params, arg, sizeof(params));

		if (ret)
			goto out;

		ret = caam_ee_set_session_key(file, &params);

		break;
	}
	case CAAM_ENCRYPT_BLOCK:
	case CAAM_DECRYPT_BLOCK:
	case CAAM_VERIFY_HMAC:
		/* These functions require a session key to be set */
		if (!(priv->flags & CAAM_EE_SESSION_ENCRYPTED)) {
			ret = -EBADR;
			goto out;
		}

		/* Fall through */
	case CAAM_WRAP_BLOB:
	case CAAM_UNWRAP_BLOB:
	{
		struct caam_ee_params params;

		ret = copy_from_user(&params, arg, sizeof(params));

		if (ret)
			goto out;

		ret = __caam_ee_ioctl(cmd, file, &params);

		break;
	}
	}

out:
	return ret;
}

static int caam_ee_open(struct inode *inode, struct file *file)
{
	struct caam_ee_priv *priv;
	int ret = -ENOMEM;
	struct device *ksdev;

	priv = kzalloc(sizeof(*priv), GFP_KERNEL);

	if (!priv)
		goto out;

	get_jr_device(&ksdev);
	priv->dev = ksdev;
	priv->rndkey = NULL;

	file->private_data = priv;

	return 0;

out:
	return ret;
}

static int caam_ee_release(struct inode *inode, struct file *file)
{
	struct caam_ee_priv *priv = (struct caam_ee_priv *)file->private_data;

	kfree(priv->rndkey);
	kfree(priv);

	return 0;
}

static const struct file_operations caam_ee_fops = {
	.open = caam_ee_open,
	.release = caam_ee_release,
	.unlocked_ioctl = caam_ee_ioctl,
};

static int __init caam_ee_init(void)
{
	struct device *dev;
	int result = alloc_chrdev_region(&caam_ee_dev_number, 0, 1, CAAM_EE);

	if (result < 0) {
		pr_err("Fail to register caam_ee char dev\n");
		return -EIO;
	}

	caam_ee_object = cdev_alloc();
	if (caam_ee_object == NULL)
		goto unregister_chrdev_region;

	caam_ee_object->owner = THIS_MODULE;
	caam_ee_object->ops = &caam_ee_fops;

	if (cdev_add(caam_ee_object, caam_ee_dev_number, 1))
		goto kobject_put;

	caam_ee_class = class_create(THIS_MODULE, CAAM_EE);
	if (IS_ERR(caam_ee_class))
		goto cdev_del;

	dev = device_create(caam_ee_class, NULL, caam_ee_dev_number, NULL,
		      "%s", CAAM_EE);
	if (IS_ERR(dev))
		goto device_destroy;

	mutex_init(&wraplock);

	return result;

device_destroy:
	class_destroy(caam_ee_class);
cdev_del:
	cdev_del(caam_ee_object);
kobject_put:
	kobject_put(&caam_ee_object->kobj);
unregister_chrdev_region:
	unregister_chrdev_region(caam_ee_dev_number, 1);
	return -EIO;
}

static void __exit caam_ee_exit(void)
{
	device_destroy(caam_ee_class, caam_ee_dev_number);
	class_destroy(caam_ee_class);
	cdev_del(caam_ee_object);
	unregister_chrdev_region(caam_ee_dev_number, 1);
}


module_init(caam_ee_init);
module_exit(caam_ee_exit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("CAAM Encryption Engine access for userspace");
MODULE_AUTHOR("Ian Molton <ian.molton@codethink.co.uk>");
