/*
 * Copyright 2011 Freescale Semiconductor, Inc.
 * Copyright 2011 Linaro Ltd.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/debugfs.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/string.h>

#define MMDC_MAARCR             0x400
#define MMDC_MAPSR		0x404
#define MMDC_PHY		0x800
#define BP_MMDC_MAPSR_PSD	0
#define BP_MMDC_MAPSR_PSS	4
#define BP_MMDC_MAARCR_RCH_EN   24

#define MMDC_BLOB_SIZE		0xc4
static struct dentry *mmdcdir;

/*
 * Make the register content of the MMDCx instance from
 * MMDC0: 0x021B0800 - 0x021B08C4
 * MMDC1: 0x021B4800 - 0x021B48C4
 * available via the debugfs in read only mode.
 */
static int imx_mmdc_export(struct platform_device *pdev,
			   void __iomem *mmdc_base)
{
	struct debugfs_blob_wrapper *imxmmdc_blob;
	char *name;
	int ret;

	if (mmdcdir == NULL) {
		mmdcdir = debugfs_create_dir("imx-mmdc", NULL);
		if (IS_ERR(mmdcdir)) {
			ret = PTR_ERR(mmdcdir);
			goto out1;
		}
	}

	imxmmdc_blob = kzalloc(sizeof(struct debugfs_blob_wrapper), GFP_KERNEL);
	if (imxmmdc_blob == NULL) {
		ret = -ENOMEM;
		goto out2;
	}

	imxmmdc_blob->size = MMDC_BLOB_SIZE;
	imxmmdc_blob->data = kmalloc(MMDC_BLOB_SIZE, GFP_KERNEL);
	if (imxmmdc_blob->data == NULL) {
		ret = -ENOMEM;
		goto out3;
	}

	memcpy_fromio(imxmmdc_blob->data, mmdc_base + MMDC_PHY, MMDC_BLOB_SIZE);

	/* Make a MMDC_PHY offset file name string from the MMDC base address */
	name = kstrdup(pdev->name, GFP_KERNEL);
	name[4] = '8';
	name[7] = 0;

	if (IS_ERR(debugfs_create_blob(name, 0444, mmdcdir, imxmmdc_blob))) {
		ret = -ENODEV;
		goto out4;
	}

	kfree(name);

	return 0;

out4:
	kfree(name);
out3:
	kfree(imxmmdc_blob->data);
out2:
	kfree(imxmmdc_blob);
out1:
	debugfs_remove_recursive(mmdcdir);
	return ret;
}

static int imx_mmdc_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	void __iomem *mmdc_base, *reg;
	u32 val;
	int timeout = 0x400;

	mmdc_base = of_iomap(np, 0);
	WARN_ON(!mmdc_base);

	if (imx_mmdc_export(pdev, mmdc_base))
		pr_err("%s: failed export MMDC registers to debugfs\n",
			__func__);

	/* The below has to be done only for MMDC0, even on a Quad/DualLite */
	if (of_device_is_compatible(np, "fsl,imx6q-mmdc1"))
		goto out;

	reg = mmdc_base + MMDC_MAPSR;

	/* Enable automatic power saving */
	val = readl_relaxed(reg);
	val &= ~(1 << BP_MMDC_MAPSR_PSD);
	writel_relaxed(val, reg);

	/* Ensure it's successfully enabled */
	while (!(readl_relaxed(reg) & 1 << BP_MMDC_MAPSR_PSS) && --timeout)
		cpu_relax();

	if (unlikely(!timeout)) {
		pr_warn("%s: failed to enable automatic power saving\n",
			__func__);
		return -EBUSY;
	}

	/* Enable real-time priority requests */
	reg = mmdc_base + MMDC_MAARCR;
	val = readl_relaxed(reg);
	val |= (1 << BP_MMDC_MAARCR_RCH_EN);
	writel_relaxed(val, reg);
	pr_info("MAARCR = 0x%08x\n", readl_relaxed(reg));

out:
	return 0;
}

static struct of_device_id imx_mmdc_dt_ids[] = {
	{ .compatible = "fsl,imx6q-mmdc", },
	{ .compatible = "fsl,imx6q-mmdc1", },
	{ /* sentinel */ }
};

static struct platform_driver imx_mmdc_driver = {
	.driver		= {
		.name	= "imx-mmdc",
		.owner	= THIS_MODULE,
		.of_match_table = imx_mmdc_dt_ids,
	},
	.probe		= imx_mmdc_probe,
};

static int __init imx_mmdc_init(void)
{
	return platform_driver_register(&imx_mmdc_driver);
}
postcore_initcall(imx_mmdc_init);
