/*
 * Copyright 2011-2013 Freescale Semiconductor, Inc.
 * Copyright 2011 Linaro Ltd.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/clk.h>
#include <linux/clkdev.h>
#include <linux/cpu.h>
#include <linux/export.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/opp.h>
#include <linux/pinctrl/machine.h>
#include <linux/phy.h>
#include <linux/regmap.h>
#include <linux/micrel_phy.h>
#include <linux/mfd/syscon.h>
#include <linux/platform_data/imx-iram.h>
#include <linux/platform_data/imx-viv-gpu.h>
#include <linux/mxc_vpu.h>
#include <linux/platform_data/imx-ipu-v3.h>
#include <asm/smp_twd.h>
#include <linux/dma-mapping.h>
#include <linux/memblock.h>
#include <linux/mfd/syscon/imx6q-iomuxc-gpr.h>
#include <asm/hardware/cache-l2x0.h>
#include <asm/hardware/gic.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>
#include <asm/pmu.h>
#include <asm/system_info.h>
#include <asm/system_misc.h>

#include "common.h"
#include "cpuidle.h"
#include "hardware.h"
#include "devices/devices-common.h"

static u32 chip_revision;

int imx6q_revision(void)
{
	return chip_revision;
}

static void __init imx6q_init_revision(void)
{
	u32 rev = imx_anatop_get_digprog();

	switch (rev & 0xff) {
	case 0:
		chip_revision = IMX_CHIP_REVISION_1_0;
		break;
	case 1:
		chip_revision = IMX_CHIP_REVISION_1_1;
		break;
	case 2:
		chip_revision = IMX_CHIP_REVISION_1_2;
		break;
	case 3:
		chip_revision = IMX_CHIP_REVISION_1_3;
		break;
	case 4:
		chip_revision = IMX_CHIP_REVISION_1_4;
		break;
	case 5:
		/*
		 * i.MX6DQ TO1.5 is defined as Rev 1.3 in Data Sheet, marked
		 * as 'D' in Part Number last character.
		 */
		chip_revision = IMX_CHIP_REVISION_1_5;
		break;
	default:
		chip_revision = IMX_CHIP_REVISION_UNKNOWN;
	}

	mxc_set_cpu_type(rev >> 16 & 0xff);

	if (!system_rev)
		system_rev = ((rev >> 4) & 0xff000) | ((rev & 0xff) + 0x10);
}

/* For imx6q sabrelite board: set KSZ9021RN RGMII pad skew */
static int ksz9021rn_phy_fixup(struct phy_device *phydev)
{
	if (IS_BUILTIN(CONFIG_PHYLIB)) {
		/* min rx data delay */
		phy_write(phydev, 0x0b, 0x8105);
		phy_write(phydev, 0x0c, 0x0000);

		/* max rx/tx clock delay, min rx/tx control delay */
		phy_write(phydev, 0x0b, 0x8104);
		phy_write(phydev, 0x0c, 0xf0f0);
		phy_write(phydev, 0x0b, 0x104);
	}

	return 0;
}

static void __init imx6q_sabresd_cko1_setup(void)
{
	struct clk *cko1_sel, *ipg_per, *cko1;
	unsigned long rate;

	cko1_sel = clk_get_sys(NULL, "cko1_sel");
	ipg_per = clk_get_sys(NULL, "ipg_per");
	cko1 = clk_get_sys(NULL, "cko1");
	if (IS_ERR(cko1_sel) || IS_ERR(ipg_per) || IS_ERR(cko1)) {
		pr_err("cko1 setup failed!\n");
		goto put_clk;
	}
	clk_set_parent(cko1_sel, ipg_per);
	rate = clk_round_rate(cko1, 22000000);
	clk_set_rate(cko1, rate);
	clk_register_clkdev(cko1, NULL, "0-000a");

put_clk:
	if (!IS_ERR(cko1_sel))
		clk_put(cko1_sel);
	if (!IS_ERR(ipg_per))
		clk_put(ipg_per);
	if (!IS_ERR(cko1))
		clk_put(cko1);
}

#if defined(CONFIG_FEC) || defined(CONFIG_FEC_MODULE)
/* For imx6q arm2 & sabresd & sabreauto board:
 * set AR803x RGMII output 125MHz clock
 */
static int ar803x_phy_fixup(struct phy_device *phydev)
{
	unsigned short val;

	if (IS_BUILTIN(CONFIG_PHYLIB)) {
		/* disable phy AR8031 SmartEEE function. */
		phy_write(phydev, 0xd, 0x3);
		phy_write(phydev, 0xe, 0x805d);
		phy_write(phydev, 0xd, 0x4003);
		val = phy_read(phydev, 0xe);
		val &= ~(0x1 << 8);
		phy_write(phydev, 0xe, val);

		/* To enable AR8031 ouput a 125MHz clk from CLK_25M */
		phy_write(phydev, 0xd, 0x7);
		phy_write(phydev, 0xe, 0x8016);
		phy_write(phydev, 0xd, 0x4007);
		val = phy_read(phydev, 0xe);

		val &= 0xffe3;
		val |= 0x18;
		phy_write(phydev, 0xe, val);

		/* introduce tx clock delay */
		phy_write(phydev, 0x1d, 0x5);
		val = phy_read(phydev, 0x1e);
		val |= 0x0100;
		phy_write(phydev, 0x1e, val);

		/*check phy power*/
		val = phy_read(phydev, 0x0);
		if (val & BMCR_PDOWN)
			phy_write(phydev, 0x0, (val & ~BMCR_PDOWN));
	}

	return 0;
}

static void __init imx6q_ar803x_phy_fixup(void)
{
	/* The phy layer fixup for AR8031 and AR8033 */
	if (IS_BUILTIN(CONFIG_PHYLIB))
		phy_register_fixup_for_id(PHY_ANY_ID, ar803x_phy_fixup);
}
#else
static void __init imx6q_ar803x_phy_fixup(void) {}
#endif

static void __init imx6q_sabresd_init(void)
{
	imx6q_ar803x_phy_fixup();
	imx6q_sabresd_cko1_setup();
}

static void __init imx6q_sabreauto_init(void)
{
	imx6q_ar803x_phy_fixup();
}

static void __init imx6q_sabrelite_cko1_setup(void)
{
	struct clk *cko1_sel, *ahb, *cko1;
	unsigned long rate;

	cko1_sel = clk_get_sys(NULL, "cko1_sel");
	ahb = clk_get_sys(NULL, "ahb");
	cko1 = clk_get_sys(NULL, "cko1");
	if (IS_ERR(cko1_sel) || IS_ERR(ahb) || IS_ERR(cko1)) {
		pr_err("cko1 setup failed!\n");
		goto put_clk;
	}
	clk_set_parent(cko1_sel, ahb);
	rate = clk_round_rate(cko1, 16000000);
	clk_set_rate(cko1, rate);
put_clk:
	if (!IS_ERR(cko1_sel))
		clk_put(cko1_sel);
	if (!IS_ERR(ahb))
		clk_put(ahb);
	if (!IS_ERR(cko1))
		clk_put(cko1);
}

static void __init imx6q_usb_otg_gpio_1(void)
{
	struct regmap *gpr;

	gpr = syscon_regmap_lookup_by_compatible("fsl,imx6q-iomuxc-gpr");
	if (!IS_ERR(gpr))
		regmap_update_bits(gpr, IOMUXC_GPR1,
				IMX6Q_GPR1_USB_OTG_ID_SEL_MASK,
				IMX6Q_GPR1_USB_OTG_ID_SEL_GPIO_1);
	else
		pr_err("%s: failed to find fsl,imx6q-iomux-gpr regmap\n", __func__);
}

static void __init imx6q_sabrelite_init(void)
{
	if (IS_BUILTIN(CONFIG_PHYLIB))
		phy_register_fixup_for_uid(PHY_ID_KSZ9021, MICREL_PHY_ID_MASK,
				ksz9021rn_phy_fixup);
	imx6q_sabrelite_cko1_setup();
}

static void __init imx6q_config_on_boot(void)
{
	struct device_node *np;
	struct property *pp;
	int cnt, len, i, ret;
	int gpio;
	const __be32 *p;
	u32 u;

	np = of_find_node_by_path("/config-on-boot");
	if (!np)
		return;
	cnt = of_gpio_named_count(np, "output-gpios");
	pp = of_find_property(np, "output-gpio-values", &len);
	if (!pp || cnt != len / sizeof(u32)) {
		pr_err("Invalid config-on-boot gpios!\n");
		of_node_put(np);
		return;
	}
	p = NULL;
	for (i = 0; i < cnt; i++) {
		gpio = of_get_named_gpio(np, "output-gpios", i);
		p = of_prop_next_u32(pp, p, &u);
		if (!p)
			break;
		if (gpio_is_valid(gpio)) {
			ret = gpio_request_one(gpio,
				(u ? GPIOF_OUT_INIT_HIGH : GPIOF_OUT_INIT_LOW) |
					GPIOF_EXPORT,
				"config-on-boot");
			if (ret)
				pr_err("Failed to request output gpio %d: %d\n",
					gpio, ret);
		}
	}

	of_node_put(np);
}

static void __init imx6q_post_populate(void)
{
	imx6q_config_on_boot();
}

#define GPR13			0x34
#define SATA_SPEED_MASK		(1<<15)
#define SATA_SPEED_3GHZ		(1<<15)
#define SATA_MPLL_CK_MASK	(1<<1)
#define SATA_MPLL_CK_OFF	(1<<1)

/* Config IOMUXC_GPR13 settings for  SATA PHY */
static void __init imx6q_sata_init(void)
{
	struct device_node *np;
	void __iomem *iomuxc_base = NULL;
	u32 val;

	np = of_find_compatible_node(NULL, NULL, "fsl,imx6q-ahci");
	if (!np)
		return;

	/* Check DT node status */
	if (!of_device_is_available(np))
		goto out;

	np = of_find_compatible_node(NULL, NULL, "fsl,imx6q-iomuxc");
	if (!np)
		return;

	iomuxc_base = of_iomap(np, 0);
	if (!iomuxc_base)
		goto out;

	/* Set PHY Paremeters, two steps to configure the GPR13,
	 * one write for reset of parameters,
	 * and the other one write for setting the mpll_clk_off_b
	 */
	val = readl_relaxed(iomuxc_base + GPR13);
	writel_relaxed(((val & ~SATA_SPEED_MASK) | SATA_SPEED_3GHZ),
				iomuxc_base + GPR13);

	/* enable SATA_PHY PLL */
	val = readl_relaxed(iomuxc_base + GPR13);
	writel_relaxed(((val & ~SATA_MPLL_CK_MASK) | SATA_MPLL_CK_OFF),
				iomuxc_base + GPR13);

out:
	if (iomuxc_base)
		iounmap(iomuxc_base);

	if (np)
		of_node_put(np);

	return;
}

static void __init imx6q_1588_init(void)
{
	struct regmap *gpr;

	gpr = syscon_regmap_lookup_by_compatible("fsl,imx6q-iomuxc-gpr");
	if (!IS_ERR(gpr))
		regmap_update_bits(gpr, IOMUXC_GPR1,
				IMX6Q_GPR1_ENET_CLK_SEL_MASK,
				IMX6Q_GPR1_ENET_CLK_SEL_ANATOP);
	else
		pr_err("%s: failed to find fsl,imx6q-iomux-gpr regmap\n", __func__);

}

static struct ahci_platform_data ahci_pdata = {
	.init = imx_sata_init,
	.exit = imx_sata_exit,
};

static void mx6q_vpu_reset(void)
{
	imx_reset_vpu();
}

static struct mxc_vpu_platform_data vpu_pdata = {
	.reset = mx6q_vpu_reset,
};

static struct ipuv3_platform_data ipu_pdata = {
	.reset = imx_src_reset_ipu,
};

static struct fec_platform_data fec_pdata = {
	.phy = PHY_INTERFACE_MODE_MII,
};

/*
 * The i.MX6 PMU has a design errata where the interrupts of all cores are
 * wired together into a single irq line. To work around this we have to
 * bounce the interrupt around all cores until we find the one where the PMU
 * counter overflow has happened.
 */
static irqreturn_t imx6q_pmu_handler(int irq, void *dev, irq_handler_t handler)
{
	irqreturn_t ret = handler(irq, dev);
	int next;

	if (ret == IRQ_NONE) {
		/*
		 * Kick the irq over to the next cpu, regardless of it's
		 * online status (it might have gone offline while we were busy
		 * bouncing the irq).
		 */
		next = (smp_processor_id() + 1) % num_present_cpus();
		irq_set_affinity(irq, cpumask_of(next));
	}

	return ret;
}

struct arm_pmu_platdata imx6q_pmu_platdata = {
	.handle_irq		= imx6q_pmu_handler,
};

static const struct of_dev_auxdata imx6q_auxdata_lookup[] __initconst = {
	OF_DEV_AUXDATA("fsl,imx6q-ahci", MX6Q_SATA_BASE_ADDR, "imx6q-ahci",
			&ahci_pdata),
	OF_DEV_AUXDATA("fsl,imx6qdl-vpu", MX6Q_VPU_BASE_ADDR, "mxc_vpu.0",
		       &vpu_pdata),
	OF_DEV_AUXDATA("fsl,imx6dl-ipu", MX6Q_IPU1_BASE_ADDR, NULL, &ipu_pdata),
	OF_DEV_AUXDATA("fsl,imx6q-ipu", MX6Q_IPU1_BASE_ADDR, NULL, &ipu_pdata),
	OF_DEV_AUXDATA("fsl,imx6q-ipu", MX6Q_IPU2_BASE_ADDR, NULL, &ipu_pdata),
	OF_DEV_AUXDATA("fsl,imx6q-fec", MX6Q_ENET_BASE_ADDR, NULL, &fec_pdata),
	OF_DEV_AUXDATA("arm,cortex-a9-pmu", 0, "arm-pmu", &imx6q_pmu_platdata),
};

static void __init imx6q_usb_init(void)
{
	imx_anatop_usb_chrg_detect_disable();
}

#define OCOTP_MAC_ADDRESS_HI	0x620
#define OCOTP_MAC_ADDRESS_LO	0x630
static void __init imx6q_fec_enet_ocotp_mac(void){

	struct device_node *np;
	void __iomem *base;
	u32 val;

	np = of_find_compatible_node(NULL, NULL, "fsl,imx6q-ocotp");
	if (!np) {
		pr_warn("failed to find ocotp node\n");
		return;
	}

	base = of_iomap(np, 0);
	if (!base) {
		pr_warn("failed to map ocotp\n");
		goto put_node;
	}

	val = be16_to_cpu(readl_relaxed(base + OCOTP_MAC_ADDRESS_LO));
	memcpy(fec_pdata.mac, &val, 2);

	val = be32_to_cpu(readl_relaxed(base + OCOTP_MAC_ADDRESS_HI));
	memcpy(&(fec_pdata.mac[2]), &val, 4);

put_node:
	of_node_put(np);

}

#define OCOTP_UNIQUE_ID_LOW  0x410
#define OCOTP_UNIQUE_ID_HIGH 0x420
static void __init imx6q_fetch_ocotp_serial(void){
	struct regmap *ocotp_reg;
	int ret;

	ocotp_reg = syscon_regmap_lookup_by_compatible("fsl,imx6q-ocotp");
	if (IS_ERR(ocotp_reg)) {
		pr_warn("failed to regmap OCOTP: %ld\n", PTR_ERR(ocotp_reg));
		return;
	}

	ret = regmap_read(ocotp_reg, OCOTP_UNIQUE_ID_LOW, &system_serial_low);
	if (ret < 0) {
		pr_warn("failed to read OCOTP UNIQUE_ID[31-0]: %d\n", ret);
		return;
	}
	ret = regmap_read(ocotp_reg, OCOTP_UNIQUE_ID_HIGH, &system_serial_high);
	if (ret < 0) {
		pr_warn("failed to read OCOTP UNIQUE_ID[63-32]: %d\n", ret);
		return;
	}

	system_serial_low  = be32_to_cpu(system_serial_low);
	system_serial_high = be32_to_cpu(system_serial_high);
}

#ifdef CONFIG_RBCM_BL_TIMESTAMP

/* These are relative to MX6Q_IRAM_BASE_ADDR and must be in sync with
 * the corresponding definitions in the bootloader */
#define TT_BASE_OFFSET 0x0001df00
#define TT_END_OFFSET  0x0001e000
#define TT_NENTRIES ((TT_END_OFFSET - TT_BASE_OFFSET)/sizeof(u32))

static u32 *imx6q_timestamp_table;
static size_t imx6q_timestamp_table_size;

u32 *imx6q_get_timestamp_table(size_t *sizep)
{
	if (imx6q_timestamp_table_size == 0)
		return (u32 *) -ENODEV;

	if (sizep == NULL)
		return (u32 *) -EINVAL;

	*sizep = imx6q_timestamp_table_size;
	return imx6q_timestamp_table;
}
EXPORT_SYMBOL(imx6q_get_timestamp_table);

static void get_tt_range(u32 *base_offset, u32 *end_offset)
{
	struct device_node *node;
	u32 values[] = { TT_BASE_OFFSET, TT_END_OFFSET };

	BUG_ON(base_offset == NULL);
	BUG_ON(end_offset == NULL);

	node = of_find_node_by_path("/chosen");
	of_property_read_u32_array(node, "timestamp-iram-offsets", values, 2);

	if (base_offset != NULL)
		*base_offset = values[0];

	if (end_offset != NULL)
		*end_offset = values[1];
}

static void imx6q_timestamp_table_init(void)
{
	unsigned long imx6q_iram_phys_base;
	void __iomem *imx6q_iram_virt_base;
	u32 base_offset;
	u32 end_offset;
	int n_entries;
	u32 *table_base;

	get_tt_range(&base_offset, &end_offset);
	if (base_offset > MX6Q_IRAM_SIZE)
		return;
	if (end_offset > MX6Q_IRAM_SIZE)
		return;
	if (base_offset >= end_offset)
		return;
	n_entries = (end_offset - base_offset)/sizeof(u32);

	imx6q_timestamp_table = kcalloc(n_entries, sizeof(u32), GFP_KERNEL);
	if (imx6q_timestamp_table == NULL)
		return;
	imx6q_timestamp_table_size = n_entries*sizeof(u32);
	imx6q_iram_phys_base = MX6Q_IRAM_BASE_ADDR;
	imx6q_iram_virt_base = ioremap(imx6q_iram_phys_base, MX6Q_IRAM_SIZE);
	if (imx6q_iram_virt_base == NULL)
		return;
	table_base = (u32 *)((u32)imx6q_iram_virt_base +  base_offset);

	memcpy_fromio(imx6q_timestamp_table, table_base,
		      imx6q_timestamp_table_size);

	iounmap(imx6q_iram_virt_base);
}
#else
static void imx6q_timestamp_table_init(void) { }
#endif /* CONFIG_RBCM_BL_TIMESTAMP */

struct coherent_reserve_data {
	const char	*of_node_name;
	const char	*resource_name;
	int		resource_num;
};

static const struct coherent_reserve_data coherent_lookup[] __initconst = {
	{	.of_node_name = "linux,dmacoherent",
		.resource_num = 0,
	},
	{	.of_node_name = "fsl,imx6qdl-vpu",
		.resource_name = "vpu_dma",
	},
	{	.of_node_name = "viv,galcore",
		.resource_name = "contiguous_baseaddr",
	}
};

static void __init imx6q_coherent_reserve(void)
{
	struct device_node *np;
	struct platform_device *pdev;
	struct resource *res;
	int err;
	int i;

	for (i = 0; i < ARRAY_SIZE(coherent_lookup); i++) {
		np = of_find_compatible_node(NULL, NULL,
				coherent_lookup[i].of_node_name);
		if (!np) {
			pr_debug("%s: OF node %s not found", __func__,
					coherent_lookup[i].of_node_name);
			continue;
		}
		pdev = of_find_device_by_node(np);
		if (!pdev) {
			pr_err("%s: failed to find device %s\n", __func__,
					coherent_lookup[i].of_node_name);
			continue;
		}
		if (coherent_lookup[i].resource_name)
			res = platform_get_resource_byname(pdev, IORESOURCE_MEM,
					coherent_lookup[i].resource_name);
		else
			res = platform_get_resource(pdev, IORESOURCE_MEM,
					coherent_lookup[i].resource_num);

		if (!res) {
			dev_dbg(&pdev->dev,
				"No dedicated DMA-coherent region defined\n");
			continue;
		}
		err = dma_declare_coherent_memory(&pdev->dev, res->start,
						  res->start,
						  resource_size(res),
						  DMA_MEMORY_MAP |
						  DMA_MEMORY_EXCLUSIVE);
		if (!err)
			dev_err(&pdev->dev,
				"Unable to declare DMA coherent memory.\n");
	}
}

static void __init imx6q_init_machine(void)
{
	/*
	 * This should be removed when all imx6q boards have pinctrl
	 * states for devices defined in device tree.
	 */
	pinctrl_provide_dummies();

	mxc_arch_reset_init_dt();

	if (of_machine_is_compatible("fsl,imx6q-sabrelite"))
		imx6q_sabrelite_init();
	else if (of_machine_is_compatible("fsl,imx6q-sabresd"))
		imx6q_sabresd_init();
	else if (of_machine_is_compatible("fsl,imx6q-sabreauto") ||
			of_machine_is_compatible("fsl,imx6dl-sabreauto"))
		imx6q_sabreauto_init();

	of_platform_populate(NULL, of_default_bus_match_table,
			     imx6q_auxdata_lookup, NULL);

	imx6q_post_populate();

	imx6q_coherent_reserve();
	imx6q_timestamp_table_init();
	imx_anatop_init();
	imx6q_pm_init();
	imx6q_usb_init();
	if (of_machine_is_compatible("fsl,imx6q-sabrelite"))
		imx6q_usb_otg_gpio_1();
	imx6q_1588_init();
	imx6q_sata_init();
	imx6q_fec_enet_ocotp_mac();
	imx6q_fetch_ocotp_serial();
}

#define OCOTP_CFG3			0x440
#define OCOTP_CFG3_SPEED_SHIFT		16
#define OCOTP_CFG3_SPEED_1P2GHZ		0x3

static void __init imx6q_opp_check_1p2ghz(struct device *cpu_dev)
{
	struct device_node *np;
	void __iomem *base;
	u32 val;

	np = of_find_compatible_node(NULL, NULL, "fsl,imx6q-ocotp");
	if (!np) {
		pr_warn("failed to find ocotp node\n");
		return;
	}

	base = of_iomap(np, 0);
	if (!base) {
		pr_warn("failed to map ocotp\n");
		goto put_node;
	}

	val = readl_relaxed(base + OCOTP_CFG3);
	val >>= OCOTP_CFG3_SPEED_SHIFT;
	if (((val & 0x3) != OCOTP_CFG3_SPEED_1P2GHZ) && cpu_is_imx6q())
		if (opp_disable(cpu_dev, 1200000000))
			pr_warn("failed to disable 1.2 GHz OPP\n");

put_node:
	of_node_put(np);
}

static void __init imx6q_opp_init(struct device *cpu_dev)
{
	struct device_node *np;

	np = of_find_node_by_path("/cpus/cpu@0");
	if (!np) {
		pr_warn("failed to find cpu0 node\n");
		return;
	}

	cpu_dev->of_node = np;
	if (of_init_opp_table(cpu_dev)) {
		pr_warn("failed to init OPP table\n");
		goto put_node;
	}

	imx6q_opp_check_1p2ghz(cpu_dev);

put_node:
	of_node_put(np);
}

struct platform_device imx6q_cpufreq_pdev = {
	.name = "imx6q-cpufreq",
};

static void __init imx6q_init_late(void)
{
	/*
	 * WAIT mode is broken on TO 1.0 and 1.1, so there is no point
	 * to run cpuidle on them.
	 */
	if (imx6q_revision() > IMX_CHIP_REVISION_1_1)
		imx6q_cpuidle_init();

	if (IS_ENABLED(CONFIG_ARM_IMX6Q_CPUFREQ)) {
		imx6q_opp_init(&imx6q_cpufreq_pdev.dev);
		platform_device_register(&imx6q_cpufreq_pdev);
	}
}

static void __init imx6q_map_io(void)
{
	debug_ll_io_init();
	imx_scu_map_io();
}

#ifdef CONFIG_CACHE_L2X0
static void __init imx6q_init_l2cache(void)
{
	void __iomem *l2x0_base;
	struct device_node *np;
	unsigned int val;

	np = of_find_compatible_node(NULL, NULL, "arm,pl310-cache");
	if (!np)
		goto out;

	l2x0_base = of_iomap(np, 0);
	if (!l2x0_base) {
		of_node_put(np);
		goto out;
	}

	/* Configure the L2 PREFETCH and POWER registers */
	val = readl_relaxed(l2x0_base + L2X0_PREFETCH_CTRL);
	val |= 0x70800000;
	/*
	 * The L2 cache controller(PL310) version on the i.MX6D/Q is r3p1-50rel0
	 * The L2 cache controller(PL310) version on the i.MX6DL/SOLO/SL is r3p2
	 * But according to ARM PL310 errata: 752271
	 * ID: 752271: Double linefill feature can cause data corruption
	 * Fault Status: Present in: r3p0, r3p1, r3p1-50rel0. Fixed in r3p2
	 * Workaround: The only workaround to this erratum is to disable the
	 * double linefill feature on the i.MX6D/Q.
	 */
	if (cpu_is_imx6q())
		val &= ~(1 << 30);
	writel_relaxed(val, l2x0_base + L2X0_PREFETCH_CTRL);
	val = L2X0_DYNAMIC_CLK_GATING_EN | L2X0_STNDBY_MODE_EN;
	writel_relaxed(val, l2x0_base + L2X0_POWER_CTRL);

	iounmap(l2x0_base);
	of_node_put(np);

out:
	l2x0_of_init(0, ~0UL);
}
#else
static inline void imx6q_init_l2cache(void) {}
#endif

static const struct of_device_id imx6q_irq_match[] __initconst = {
	{ .compatible = "arm,cortex-a9-gic", .data = gic_of_init, },
	{ /* sentinel */ }
};

static void __init imx6q_init_irq(void)
{
	imx6q_init_revision();
	imx6q_init_l2cache();
	imx_src_init();
	imx_gpc_init();
	of_irq_init(imx6q_irq_match);
}

static void __init imx6q_timer_init(void)
{
	mx6q_clocks_init();
	twd_local_timer_of_register();
	imx_print_silicon_rev(cpu_is_imx6dl() ? "i.MX6DL" : "i.MX6Q",
			      imx6q_revision());
}

static struct sys_timer imx6q_timer = {
	.init = imx6q_timer_init,
};

static const char *imx6q_dt_compat[] __initdata = {
	"fsl,imx6dl",
	"fsl,imx6q",
	"fsl,imx6s",
	NULL,
};

DT_MACHINE_START(IMX6Q, "Freescale i.MX6 Quad/DualLite (Device Tree)")
	/*
	 * i.MX6Q/DL maps system memory at 0x10000000 (offset 256MiB), and
	 * GPU has a limit on physical address that it accesses, which must
	 * be below 2GiB.
	 */
	.dma_zone_size	= (SZ_2G - SZ_256M),
	.smp		= smp_ops(imx_smp_ops),
	.map_io		= imx6q_map_io,
	.init_irq	= imx6q_init_irq,
	.handle_irq	= imx6q_handle_irq,
	.timer		= &imx6q_timer,
	.init_machine	= imx6q_init_machine,
	.init_late      = imx6q_init_late,
	.dt_compat	= imx6q_dt_compat,
	.restart	= mxc_restart,
MACHINE_END
