/*
 * Copyright (c) 2005-2012, Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Module Name:    mfw_gst_v4lsrc.c
 *
 * Description:    Implementation of V4L Source Plugin for Gstreamer
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */

/*
 * Changelog:
 * 1-Feb-2013:     Enhanced to support different RGB and YUV color formats,
 *                 configurable through plugin properties.
 *
 *  19-Dec-2013:   Added check for V4L2_BUF_FLAG_ERROR buffer flag and signal
 *                 will be emitted if flag is set to notify Gstreamer framework.
 *
 *  9-Jan-2014:    Enhanced to support different picture enhancement parameters
 *                 configurable through plugin properties.
 *
 *  20-Jan-2014:   Added SIGNAL_V4L2_BUF_FLAG_OK signal to recover from error
 *                 state and to notify Gstreamer framework.
 *
 *  13-May-2014:   Added cropping feature to configure left, top, width and
 *                 height values through VIDIOC_S_CROP ioctl call
 */



/*=============================================================================
                            INCLUDE FILES
=============================================================================*/
#include <gst/gst.h>
#include <gst/base/gstpushsrc.h>
#include <gst/interfaces/propertyprobe.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdint.h>
#include <linux/videodev2.h>
#include <sys/mman.h>
#include <string.h>
#include <gst/video/video.h>
#include "mfw_gst_v4lsrc.h"
#ifdef ENABLE_DRM
#include "mfw_gst_src_drm_plane.h"
#else
#ifdef MX27
#include "mxcfb.h"
#else
#include "linux/mxcfb.h"
#endif
#endif

#include "mfw_gst_utils.h"
#include "gstbufmeta.h"

//#define GST_DEBUG g_print

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/
/* None */
#define DEFAULT_QUEUE_SIZE 6
#define NO_BUFFER_WAIT_COUNT   200
#define NO_BUFFER_WAIT_TIME_MS 5

/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/

enum
{
  MFW_V4L_SRC_0,
  MFW_V4L_SRC_WIDTH,
  MFW_V4L_SRC_HEIGHT,
  MFW_V4L_SRC_INPUT,
  MFW_V4L_SRC_ROTATE,
  MFW_V4L_SRC_HFLIP,
  MFW_V4L_SRC_VFLIP,
  MFW_V4L_SRC_PREVIEW,
  MFW_V4L_SRC_PREVIEW_WIDTH,
  MFW_V4L_SRC_PREVIEW_TOP,
  MFW_V4L_SRC_PREVIEW_LEFT,
  MFW_V4L_SRC_PREVIEW_HEIGHT,
  MFW_V4L_SRC_CROP_LEFT,   /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_TOP,    /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_WIDTH,  /* Enhanced to support cropping feature */
  MFW_V4L_SRC_CROP_HEIGHT, /* Enhanced to support cropping feature */
  MFW_V4L_SRC_FRAMERATE_NUM,
  MFW_V4L_SRC_FRAMERATE_DEN,
  MFW_V4L_SRC_SENSOR_WIDTH,
  MFW_V4L_SRC_SENSOR_HEIGHT,
  MFW_V4L_SRC_BACKGROUND,
  MFW_V4L_SRC_DEVICE,
  MFW_V4L_SRC_QUEUE_SIZE,
  MFW_V4L_SRC_PIXEL_FMT, /* Enhanced to support diff RGB and YUV color-formats */
#ifdef ENABLE_DRM
  MFW_V4L_SRC_DISP_TYPE,
#endif
  MFW_V4L_SRC_BRIGHTNESS, /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_CONTRAST,   /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_SATURATION, /* Enhanced to support picture enhancement parameters */
  MFW_V4L_SRC_HUE         /* Enhanced to support picture enhancement parameters */
};

enum
{
  SIGNAL_V4L2_BUF_FLAG_ERROR,
  SIGNAL_V4L2_BUF_FLAG_OK,
  SIGNAL_V4L2_BUF_FLAG_NONE
};

static guint gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_NONE] = { 0 };

static GstStaticPadTemplate src_factory =
  GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS(
        GST_VIDEO_CAPS_RGB_16 ";"
        GST_VIDEO_CAPS_BGR ";"
        GST_VIDEO_CAPS_RGB ";"
        GST_VIDEO_CAPS_BGRx ";"
        GST_VIDEO_CAPS_RGBx ";"
        GST_VIDEO_CAPS_YUV ("422P") ";"
        GST_VIDEO_CAPS_YUV ("UYVY") ";"
        GST_VIDEO_CAPS_YUV ("YUY2") ";"
        GST_VIDEO_CAPS_YUV ("I420") ";"
        GST_VIDEO_CAPS_YUV ("NV12") ";"
        )
  );


/*=============================================================================
                              LOCAL MACROS
=============================================================================*/

/* used for debugging */
#define GST_CAT_DEFAULT mfw_gst_v4lsrc_debug

#define ipu_fourcc(a,b,c,d)\
        (((__u32)(a)<<0)|((__u32)(b)<<8)|((__u32)(c)<<16)|((__u32)(d)<<24))

#define IPU_PIX_FMT_RGB332  ipu_fourcc('R','G','B','1') /*!<  8  RGB-3-3-2     */
#define IPU_PIX_FMT_RGB555  ipu_fourcc('R','G','B','O') /*!< 16  RGB-5-5-5     */
#define IPU_PIX_FMT_RGB565  ipu_fourcc('R','G','B','P') /*!< 16  RGB-5-6-5     */
#define IPU_PIX_FMT_RGB666  ipu_fourcc('R','G','B','6') /*!< 18  RGB-6-6-6     */
#define IPU_PIX_FMT_BGR24   ipu_fourcc('B','G','R','3') /*!< 24  BGR-8-8-8     */
#define IPU_PIX_FMT_RGB24   ipu_fourcc('R','G','B','3') /*!< 24  RGB-8-8-8     */
#define IPU_PIX_FMT_BGR32   ipu_fourcc('B','G','R','4') /*!< 32  BGR-8-8-8-8   */
#define IPU_PIX_FMT_BGRA32  ipu_fourcc('B','G','R','A') /*!< 32  BGR-8-8-8-8   */
#define IPU_PIX_FMT_RGB32   ipu_fourcc('R','G','B','4') /*!< 32  RGB-8-8-8-8   */
#define IPU_PIX_FMT_RGBA32  ipu_fourcc('R','G','B','A') /*!< 32  RGB-8-8-8-8   */
#define IPU_PIX_FMT_ABGR32  ipu_fourcc('A','B','G','R') /*!< 32  ABGR-8-8-8-8  */

/* Enhanced to support diff RGB and YUV color-formats */  
#define G_MAX_PIX_FMT	9

/* Enhanced to support picture enhancement parameters */
#define DEFAULT_BRIGHTNESS_VAL	0
#define DEFAULT_CONTRAST_VAL	128
#define DEFAULT_SATURATION_VAL	128
#define DEFAULT_HUE_VAL	0

/*=============================================================================
                             STATIC VARIABLES
=============================================================================*/

/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/
/* None */

/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/

GST_DEBUG_CATEGORY_STATIC (mfw_gst_v4lsrc_debug);
static void mfw_gst_v4lsrc_buffer_class_init (gpointer g_class,
    gpointer class_data);
static void mfw_gst_v4lsrc_buffer_init (GTypeInstance * instance,
    gpointer g_class);
static void mfw_gst_v4lsrc_buffer_finalize (MFWGstV4LSrcBuffer * v4lsrc_buffer);
static void mfw_gst_v4lsrc_fixate (GstPad * pad, GstCaps * caps);
static GstCaps *mfw_gst_v4lsrc_get_caps (GstBaseSrc * src);
static GstFlowReturn mfw_gst_v4lsrc_create (GstPushSrc * src, GstBuffer ** buf);
static GstBuffer *mfw_gst_v4lsrc_buffer_new (MFWGstV4LSrc * v4l_src);
static GstStateChangeReturn mfw_gst_v4lsrc_change_state (GstElement * element,
    GstStateChange transition);
static gboolean unlock (GstBaseSrc *src);
static gboolean mfw_gst_v4lsrc_stop (GstBaseSrc * src);
static gboolean mfw_gst_v4lsrc_start (GstBaseSrc * src);
static gboolean mfw_gst_v4lsrc_set_caps (GstBaseSrc * src, GstCaps * caps);
static void mfw_gst_v4lsrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);
static void mfw_gst_v4lsrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static gint mfw_gst_v4lsrc_capture_setup (MFWGstV4LSrc * v4l_src);
static gint mfw_gst_v4lsrc_stop_capturing (MFWGstV4LSrc * v4l_src);
static gint mfw_gst_v4lsrc_start_capturing (MFWGstV4LSrc * v4l_src);


GST_BOILERPLATE (MFWGstV4LSrc, mfw_gst_v4lsrc, GstPushSrc, GST_TYPE_PUSH_SRC);

static gboolean is_video_std(MFWGstV4LSrc * v4l_src)
{
	return v4l_src->std_id != V4L2_STD_ALL;
}

/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_get_type    

DESCRIPTION:        This funtion registers the  buffer type on to the V4L Source plugin
             
ARGUMENTS PASSED:   void   

RETURN VALUE:       Return the registered buffer type
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
GType
mfw_gst_v4lsrc_buffer_get_type (void)
{
  static GType v4lsrc_buffer_type;

  if (G_UNLIKELY (v4lsrc_buffer_type == 0)) {
    static const GTypeInfo v4lsrc_buffer_info = {
      sizeof (GstBufferClass),
      NULL,
      NULL,
      mfw_gst_v4lsrc_buffer_class_init,
      NULL,
      NULL,
      sizeof (MFWGstV4LSrcBuffer),
      0,
      mfw_gst_v4lsrc_buffer_init,
      NULL
    };
    v4lsrc_buffer_type = g_type_register_static (GST_TYPE_BUFFER,
        "MFWGstV4LSrcBuffer", &v4lsrc_buffer_info, 0);
  }
  return v4lsrc_buffer_type;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_class_init    

DESCRIPTION:   This funtion registers the  funtions used by the 
                buffer class of the V4l source plug-in
             
ARGUMENTS PASSED:
        g_class        -   class from which the mini objext is derived
        class_data     -   global class data

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_class_init (gpointer g_class, gpointer class_data)
{
  GstMiniObjectClass *mini_object_class = GST_MINI_OBJECT_CLASS (g_class);

  mini_object_class->finalize = (GstMiniObjectFinalizeFunction)
      mfw_gst_v4lsrc_buffer_finalize;
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_buffer_init    

DESCRIPTION:   This funtion initialises the buffer class of the V4l source plug-in
             
ARGUMENTS PASSED:
        instance       -   pointer to buffer instance
        g_class        -   global pointer

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_init (GTypeInstance * instance, gpointer g_class)
{

}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_buffer_finalize    

DESCRIPTION:   This function is invoked whenever the buffer object belonging 
               to the V4L Source buffer glass is tried to un-refrenced. Here 
               only the refernce count of the buffer object is increased without 
               freeing the memory allocated.

ARGUMENTS PASSED:
        v4lsrc_buffer -   pointer to V4L sou4rce buffer class

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_buffer_finalize (MFWGstV4LSrcBuffer * v4lsrc_buffer)
{
  MFWGstV4LSrc *v4l_src;
  gint num;
  GstBuffer *buf;
  struct v4l2_buffer v4lbuf;


  v4l_src = v4lsrc_buffer->v4lsrccontext;
  if (v4l_src->start) {
    num = v4lsrc_buffer->num;


    buf = (GstBuffer *) (v4l_src->buffers[num]);
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_LAST);

    g_mutex_lock (v4l_src->pool_lock);
    if (g_list_find (v4l_src->free_pool, (gpointer) (num)))
        GST_WARNING ("something wrong here, v4l buffer index:%d already in queue",
                num);
    else
        GST_LOG ("v4l buffer index:%d will be push in pool", num);
    g_mutex_unlock (v4l_src->pool_lock);

    memset (&v4lbuf, 0, sizeof (v4lbuf));
    v4lbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    v4lbuf.memory = V4L2_MEMORY_MMAP;
    v4lbuf.index = num;

    if (ioctl (v4l_src->fd_v4l, VIDIOC_QBUF, &v4lbuf) < 0) {
      GST_ERROR (">>V4L_SRC: VIDIOC_QBUF failed");
      return;
    }

    g_mutex_lock (v4l_src->pool_lock);
    v4l_src->free_pool = g_list_append (v4l_src->free_pool, (gpointer) num);
    g_mutex_unlock (v4l_src->pool_lock);
    GST_LOG_OBJECT (v4l_src, "freeing buffer %p for frame %d", v4lsrc_buffer,
        num);
    gst_buffer_ref (GST_BUFFER_CAST (v4lsrc_buffer));
  } else {
    GST_LOG ("free buffer %d\n", v4lsrc_buffer->num);
  }
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_start_capturing    
        
DESCRIPTION:   This function triggers the V4L Driver to start Capturing

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_start_capturing (MFWGstV4LSrc * v4l_src)
{
  guint i;
  struct v4l2_buffer *buf;
  MFWGstV4LSrcBuffer *v4lsrc_buf = NULL;
  enum v4l2_buf_type type;

  v4l_src->buffers = g_malloc (v4l_src->queue_size * sizeof (GstBuffer *));
  // query for v4l_src->queue_size number of buffers to store the captured data 
  for (i = 0; i < v4l_src->queue_size; i++) {
    v4lsrc_buf =
        (MFWGstV4LSrcBuffer *) gst_mini_object_new (MFW_GST_TYPE_V4LSRC_BUFFER);
    v4lsrc_buf->num = i;
    v4lsrc_buf->v4lsrccontext = v4l_src;
    /* v4l2_buffer initialization */
    buf = &v4lsrc_buf->v4l2_buf;
    // memset (&buf, 0, sizeof (buf));
    buf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    buf->memory = V4L2_MEMORY_MMAP;
    buf->index = i;
    if (ioctl (v4l_src->fd_v4l, VIDIOC_QUERYBUF, buf) < 0) {
      GST_ERROR (">>V4L_SRC: VIDIOC_QUERYBUF error");
      return -1;
    }
    // v4l_src->buffers[i] = gst_buffer_new ();
    /* GstBuffer initialization */
    v4l_src->buffers[i] = (GstBuffer *) v4lsrc_buf;
    GST_BUFFER_SIZE (v4l_src->buffers[i]) = buf->length;
    GST_BUFFER_OFFSET (v4l_src->buffers[i]) = (size_t) buf->m.offset;
    GST_BUFFER_DATA (v4l_src->buffers[i]) = mmap (NULL,
        GST_BUFFER_SIZE (v4l_src->buffers[i]),
        PROT_READ | PROT_WRITE, MAP_SHARED,
        v4l_src->fd_v4l, GST_BUFFER_OFFSET (v4l_src->buffers[i]));
    memset (GST_BUFFER_DATA (v4l_src->buffers[i]), 0xFF,
        GST_BUFFER_SIZE (v4l_src->buffers[i]));
    {
      gint index;
      GstBufferMeta *meta;
      index = G_N_ELEMENTS (v4l_src->buffers[i]->_gst_reserved) - 1;
      meta = gst_buffer_meta_new ();
      meta->physical_data = (gpointer) (buf->m.offset);
      v4l_src->buffers[i]->_gst_reserved[index] = meta;
    }


    if (ioctl (v4l_src->fd_v4l, VIDIOC_QBUF, buf) < 0) {
      GST_ERROR (">>V4L_SRC: VIDIOC_QBUF error");
      return -1;
    }
    v4l_src->free_pool =
        g_list_append (v4l_src->free_pool, (gpointer) buf->index);

  }

  v4l_src->pool_lock = g_mutex_new ();

  /* Switch ON the capture device */
  type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (ioctl (v4l_src->fd_v4l, VIDIOC_STREAMON, &type) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_STREAMON error");
    return -1;
  }

  v4l_src->start = TRUE;
  v4l_src->time_per_frame =
      gst_util_uint64_scale_int (GST_SECOND, v4l_src->fps_d, v4l_src->fps_n);
  GST_DEBUG (">>V4L_SRC: time per frame %d", (guint32) v4l_src->time_per_frame);
  v4l_src->last_ts = 0;
  return 0;
}


/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_stop_capturing    
        
DESCRIPTION:   This function triggers the V4L Driver to stop Capturing

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_stop_capturing (MFWGstV4LSrc * v4l_src)
{
  enum v4l2_buf_type type;
  guint i;
  gint index;

  type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (ioctl (v4l_src->fd_v4l, VIDIOC_STREAMOFF, &type) < 0) {
    GST_ERROR (">>V4L_SRC: error in VIDIOC_STREAMOFF");
    return -1;
  }
  v4l_src->start = FALSE;
  for (i = 0; i < v4l_src->queue_size; i++) {
    munmap (GST_BUFFER_DATA (v4l_src->buffers[i]),
        GST_BUFFER_SIZE (v4l_src->buffers[i]));

    index = G_N_ELEMENTS (v4l_src->buffers[i]->_gst_reserved) - 1;
    gst_buffer_meta_free (v4l_src->buffers[i]->_gst_reserved[index]);
    gst_buffer_unref (v4l_src->buffers[i]);
  }
  if (v4l_src->buffers)
    g_free (v4l_src->buffers);

  g_list_free (v4l_src->free_pool);
  v4l_src->free_pool = NULL;

  g_mutex_free (v4l_src->pool_lock);
  v4l_src->pool_lock = NULL;

  return 0;
}



/*=============================================================================
FUNCTION:      mfw_gst_v4lsrc_capture_setup    
        
DESCRIPTION:   This function does the necessay initialistions for the V4L capture
               device driver.

ARGUMENTS PASSED:
        v4l_src -   The V4L Souce plug-in context.

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gint
mfw_gst_v4lsrc_capture_setup (MFWGstV4LSrc * v4l_src)
{
  struct v4l2_format fmt = {0};
  struct v4l2_control ctrl = {0};
  struct v4l2_streamparm parm = {0};
  struct v4l2_dbg_chip_ident chip = {0};
  struct v4l2_crop crop = {0};
  gint fd_v4l = 0;
  struct v4l2_mxc_offset off;
  gint in_width = 0; 
  gint in_height = 0;

/* Setting boolean capture_start_flag=TRUE, so that color format cannot be 
 * configured once the capture starts
 */
  v4l_src->capture_start_flag = TRUE;

  g_mutex_lock(&v4l_src->mutex);
  if ((fd_v4l = open (v4l_src->devicename, O_RDWR, 0)) < 0) {
    GST_ERROR (">>V4L_SRC: Unable to open %s", v4l_src->devicename);
    return 0;
  }
  g_mutex_unlock(&v4l_src->mutex);

  if (ioctl (fd_v4l, VIDIOC_DBG_G_CHIP_IDENT, &chip)) {
    GST_ERROR ("VIDIOC_DBG_G_CHIP_IDENT failed.");
  } else
    GST_INFO ("sensor chip is %s", chip.match.name);

  v4l_src->std_id = V4L2_STD_UNKNOWN;
  ioctl (fd_v4l, VIDIOC_G_STD, &v4l_src->std_id);
  GST_INFO ("video standard is %s (0x%llx)\n",
	    v4l_src->std_id == V4L2_STD_NTSC ? "NTSC" :
	    (v4l_src->std_id == V4L2_STD_PAL ? "PAL" :
	     (v4l_src->std_id == V4L2_STD_UNKNOWN ? "unknown" : "other")),
	    v4l_src->std_id);

  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
#ifdef MX51
  fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_NV12;
#else
/* Enhanced to support diff RGB and YUV color-formats */
  #ifdef FSL_COLOR_MODE_SUPPORT
    fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
  #else  
    switch(v4l_src->pix_fmt)
    {
      case PIX_FMT_RGB565:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB565;
        break;

      case PIX_FMT_BGR24:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_BGR24;
        break;
	
      case PIX_FMT_RGB24:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB24;
        break;
	
      case PIX_FMT_BGR32:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_BGR32;
        break;
	
      case PIX_FMT_RGB32:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB32;
        break;
	
      case PIX_FMT_YUV422P:
	      fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV422P;
        break;
	
      case PIX_FMT_UYVY:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_UYVY;
        break;
	
      case PIX_FMT_YUYV:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUYV;
        break;
	
      case PIX_FMT_YUV420: /* YU12 */
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_YUV420;
        break;
	
      case PIX_FMT_NV12:
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_NV12;
        break;
	
      default: /* Unlikely, default color format UYVY is set instead */
        v4l_src->pix_fmt = PIX_FMT_UYVY;           
        fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_UYVY;
        break;	
    }
  #endif
#endif


  in_width = v4l_src->capture_width;
  in_height = v4l_src->capture_height;


  fmt.fmt.pix.width = in_width;
  fmt.fmt.pix.height = in_height;

  fmt.fmt.pix.bytesperline = in_width;
  fmt.fmt.pix.priv = 0;
  fmt.fmt.pix.sizeimage = 0;


  parm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  parm.parm.capture.timeperframe.numerator = v4l_src->fps_n;
  parm.parm.capture.timeperframe.denominator = v4l_src->fps_d;

  if (ioctl (fd_v4l, VIDIOC_S_PARM, &parm) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_S_PARM failed");
    return -1;
  }

  if (ioctl (fd_v4l, VIDIOC_S_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: set format failed");
    return 0;
  }

  if(v4l_src->crop_width && v4l_src->crop_height)
  {
      crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
      crop.c.left = v4l_src->crop_left;
      crop.c.top = v4l_src->crop_top;
      crop.c.width = v4l_src->crop_width;
      crop.c.height = v4l_src->crop_height;
      GST_LOG("set cropping: %dx%d -> %dx%d", v4l_src->crop_left,
              v4l_src->crop_top, v4l_src->crop_width,v4l_src->crop_height);

      if (ioctl(fd_v4l, VIDIOC_S_CROP, &crop) < 0)
      {
          GST_ERROR("set cropping failed\n");
          return 0;
      }

      if (ioctl(fd_v4l, VIDIOC_G_CROP, &crop) < 0)
      {
          GST_ERROR("get cropping failed\n");
          return 0;
      }

      GST_LOG("cropping was set to: %dx%d -> %dx%d",
           crop.c.left, crop.c.top,
           crop.c.width, crop.c.height);
  }

  if (ioctl (fd_v4l, VIDIOC_G_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return 0;
  }



  // Set rotation and horizontal/vertical flips
  ctrl.id = V4L2_CID_ROTATE;
  ctrl.value = v4l_src->rotate;
  if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: rotation set ctrl failed");
    return 0;
  }

  ctrl.id = V4L2_CID_HFLIP;
  ctrl.value = (v4l_src->hflip == TRUE) ? 1 : 0;
  if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: hflip set ctrl failed");
    return 0;
  }

  ctrl.id = V4L2_CID_VFLIP;
  ctrl.value = (v4l_src->vflip == TRUE) ? 1 : 0;
  if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0) {
    GST_ERROR (">>V4L_SRC: vflip set ctrl failed");
    return 0;
  }

  /**
   * Checking if any picture enhancement parameter needs to be set based on the
   * flag set in set_property and set the value through ioctl call
   */
  if (v4l_src->brightness_set_flag)
  {
    ctrl.id = V4L2_CID_BRIGHTNESS;
    ctrl.value = v4l_src->brightness;

    /* Set brightness parameter */
    if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
      GST_ERROR(">>V4LSRC: set brightness control failed");

    v4l_src->brightness_set_flag = FALSE;
  }

  if (v4l_src->contrast_set_flag)
  {
    ctrl.id = V4L2_CID_CONTRAST;
    ctrl.value = v4l_src->contrast;

    /* Set contrast parameter */
    if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
      GST_ERROR(">>V4LSRC: set contrast control failed");

    v4l_src->contrast_set_flag = FALSE;
  }

  if (v4l_src->saturation_set_flag)
  {
    ctrl.id = V4L2_CID_SATURATION;
    ctrl.value = v4l_src->saturation;

    /* Set saturation parameter */
    if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
      GST_ERROR("V4LSRC: set saturtion control failed");

    v4l_src->saturation_set_flag = FALSE;
  }

  if (v4l_src->hue_set_flag)
  {
    ctrl.id = V4L2_CID_HUE;
    ctrl.value = v4l_src->hue;

    /* Set hue parameter */
    if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
      GST_ERROR("V4LSRC: set hue control failed");

    v4l_src->hue_set_flag = FALSE;
  }


  /**
   * Checking for input flag, based on the flag set in set_property and
   * accordingly set the input param value through VIDIOC_S_INPUT ioctl call
   */
  if (v4l_src->input_flag)
  {
    /* Set input parameter */
    if (ioctl(fd_v4l, VIDIOC_S_INPUT, &v4l_src->input) < 0)
      GST_ERROR("V4LSRC: set saturtion control failed");

    v4l_src->input_flag = FALSE;
  }


  if (fmt.fmt.pix.width != v4l_src->capture_width ||
      fmt.fmt.pix.height != v4l_src->capture_height) {
	  GST_WARNING ("capture format changed to %dx%d",
		       fmt.fmt.pix.width, fmt.fmt.pix.height);
	  v4l_src->capture_width = fmt.fmt.pix.width;
	  v4l_src->capture_height = fmt.fmt.pix.height;
  }


  struct v4l2_requestbuffers req;
  memset (&req, 0, sizeof (req));
  req.count = v4l_src->queue_size;
  req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  req.memory = V4L2_MEMORY_MMAP;

  if (ioctl (fd_v4l, VIDIOC_REQBUFS, &req) < 0) {
    GST_ERROR
        (">>V4L_SRC: v4l_mfw_gst_v4lsrc_capture_setup: VIDIOC_REQBUFS failed");
    return 0;
  }

  return fd_v4l;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_set_property   
        
DESCRIPTION:        This function is notified if application changes the values of 
                    a property.            

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (object);
  struct v4l2_control ctrl;
  gint fd_v4l = v4l_src->fd_v4l;

  /* mutex lock, so that device file cannot be closed when setting the values */
  g_mutex_lock(&v4l_src->mutex);

  switch (prop_id) {
    case MFW_V4L_SRC_WIDTH:
      v4l_src->capture_width = g_value_get_int (value);
      GST_DEBUG ("width=%d", v4l_src->capture_width);
      break;
    case MFW_V4L_SRC_HEIGHT:
      v4l_src->capture_height = g_value_get_int (value);
      GST_DEBUG ("height=%d", v4l_src->capture_height);
      break;
    case MFW_V4L_SRC_ROTATE:
      v4l_src->rotate = g_value_get_int (value);
      GST_DEBUG ("rotate=%d", v4l_src->rotate);
      break;
    case MFW_V4L_SRC_HFLIP:
      v4l_src->hflip = g_value_get_boolean (value);
      GST_DEBUG ("hflip=%d", v4l_src->hflip);
      break;
    case MFW_V4L_SRC_VFLIP:
      v4l_src->vflip = g_value_get_boolean (value);
      GST_DEBUG ("vflip=%d", v4l_src->vflip);
      break;

    case MFW_V4L_SRC_PREVIEW:
      v4l_src->preview = g_value_get_boolean (value);
      GST_DEBUG ("preview=%d", v4l_src->preview);
      break;


    case MFW_V4L_SRC_PREVIEW_WIDTH:
      v4l_src->preview_width = g_value_get_int (value);
      GST_DEBUG ("preview_width=%d", v4l_src->preview_width);
      break;

    case MFW_V4L_SRC_PREVIEW_HEIGHT:
      v4l_src->preview_height = g_value_get_int (value);
      GST_DEBUG ("preview_height=%d", v4l_src->preview_height);
      break;

    case MFW_V4L_SRC_PREVIEW_TOP:
      v4l_src->preview_top = g_value_get_int (value);
      GST_DEBUG ("preview_top=%d", v4l_src->preview_top);
      break;

    case MFW_V4L_SRC_PREVIEW_LEFT:
      v4l_src->preview_left = g_value_get_int (value);
      GST_DEBUG ("preview_left=%d", v4l_src->preview_left);
      break;

    case MFW_V4L_SRC_CROP_LEFT:
      v4l_src->crop_left = g_value_get_int (value);
      GST_DEBUG ("crop_left=%d", v4l_src->crop_left);
      break;
    case MFW_V4L_SRC_CROP_TOP:
      v4l_src->crop_top = g_value_get_int (value);
      GST_DEBUG ("crop_top=%d", v4l_src->crop_top);
      break;
    case MFW_V4L_SRC_CROP_WIDTH:
      v4l_src->crop_width = g_value_get_int (value);
      GST_DEBUG ("crop_width=%d", v4l_src->crop_width);
      break;
    case MFW_V4L_SRC_CROP_HEIGHT:
      v4l_src->crop_height = g_value_get_int (value);
      GST_DEBUG ("crop_height=%d", v4l_src->crop_height);
      break;
    case MFW_V4L_SRC_FRAMERATE_NUM:
      v4l_src->fps_n = g_value_get_int (value);
      GST_DEBUG ("framerate numerator =%d", v4l_src->fps_n);
      break;
    case MFW_V4L_SRC_FRAMERATE_DEN:
      v4l_src->fps_d = g_value_get_int (value);
      GST_DEBUG ("framerate denominator=%d", v4l_src->fps_d);
      break;
#if 0
    case MFW_V4L_SRC_SENSOR_WIDTH:
      v4l_src->sensor_width = g_value_get_int (value);
      GST_DEBUG ("sensor width=%d", v4l_src->sensor_width);
      break;
    case MFW_V4L_SRC_SENSOR_HEIGHT:
      v4l_src->sensor_height = g_value_get_int (value);
      GST_DEBUG ("sensor height=%d", v4l_src->sensor_height);
      break;
#endif

    case MFW_V4L_SRC_INPUT:
      v4l_src->input  = g_value_get_int (value);
      GST_DEBUG ("input=%d", v4l_src->input);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->input_flag = TRUE;
      }
      else
      {
        /* this ioctl sets input parameter */
        if (ioctl(fd_v4l, VIDIOC_S_INPUT, &v4l_src->input) < 0)
          GST_ERROR("V4LSRC: set input parameter failed");
      }
      break;

    case MFW_V4L_SRC_BACKGROUND:
      v4l_src->bg = g_value_get_boolean (value);
      GST_DEBUG ("bg value=%d", v4l_src->bg);
      break;
    case MFW_V4L_SRC_DEVICE:
      if (v4l_src->devicename)
        g_free (v4l_src->devicename);
      v4l_src->devicename = g_strdup (g_value_get_string (value));
      break;

    case MFW_V4L_SRC_QUEUE_SIZE:
      v4l_src->queue_size = g_value_get_int (value);
      GST_DEBUG ("queue size=%d", v4l_src->queue_size);
      break;

/* Enhanced to support diff RGB and YUV color-formats */
    case MFW_V4L_SRC_PIXEL_FMT:
      if (TRUE != v4l_src->capture_start_flag)
      {
        v4l_src->pix_fmt = g_value_get_uint(value);
        GST_DEBUG("Pixel format=%u", v4l_src->pix_fmt);
      }
      else
      {
        GST_ERROR("Capturing started, cannot set the color format \n");      
      }                           
      break;
      
#ifdef ENABLE_DRM
    case MFW_V4L_SRC_DISP_TYPE:
      g_free(v4l_src->disp_type);
      v4l_src->disp_type = g_strdup(g_value_get_string(value));
      GST_DEBUG ("disp_type = %s", v4l_src->disp_type);
      break;
#endif

/* Enhanced to support picture enhancement parameters */
    case MFW_V4L_SRC_BRIGHTNESS:
      v4l_src->brightness = g_value_get_int(value);
      GST_DEBUG("Brightness parameter=%d", v4l_src->brightness);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->brightness_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_BRIGHTNESS;
        ctrl.value = v4l_src->brightness;

        /* this ioctl sets brightness parameter */
        if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set brightness control failed");
      }
      break;

    case MFW_V4L_SRC_CONTRAST:
      v4l_src->contrast = g_value_get_int(value);
      GST_DEBUG("Contrast parameter=%d", v4l_src->contrast);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->contrast_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_CONTRAST;
        ctrl.value = v4l_src->contrast;

        /* this ioctl sets contrast parameter */
        if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set contrast control failed");
      }
      break;

    case MFW_V4L_SRC_SATURATION:
      v4l_src->saturation = g_value_get_int(value);
      GST_DEBUG("Saturation parameter=%d", v4l_src->saturation);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->saturation_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_SATURATION;
        ctrl.value = v4l_src->saturation;

        /* this ioctl sets saturation parameter */
        if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set saturtion control failed");
      }
      break;

    case MFW_V4L_SRC_HUE:
      v4l_src->hue = g_value_get_int(value);
      GST_DEBUG("Hue parameter=%d", v4l_src->hue);

      if (fd_v4l < 0)
      {
        GST_INFO("V4LSRC: fd_v4l is not opened");
        v4l_src->hue_set_flag = TRUE;
      }
      else
      {
        ctrl.id = V4L2_CID_HUE;
        ctrl.value = v4l_src->hue;

        /* this ioctl sets hue parameter */
        if (ioctl(fd_v4l, VIDIOC_S_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: set hue control failed");
      }
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_mutex_unlock(&v4l_src->mutex);
}


/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_get_property    
        
DESCRIPTION:    This function is notified if application requests the values of 
                a property.                  

ARGUMENTS PASSED:
        object  -   pointer to GObject   
        prop_id -   id of element
        value   -   pointer to Gvalue
        pspec   -   pointer to GParamSpec

RETURN VALUE:       None
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{

  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (object);
  struct v4l2_control ctrl;
  gint fd_v4l = v4l_src->fd_v4l;

  /* mutex lock, so that device file cannot be closed when getting the values */
  g_mutex_lock(&v4l_src->mutex);

  switch (prop_id) {
    case MFW_V4L_SRC_WIDTH:
      g_value_set_int (value, v4l_src->capture_width);
      break;
    case MFW_V4L_SRC_HEIGHT:
      g_value_set_int (value, v4l_src->capture_height);
      break;
    case MFW_V4L_SRC_ROTATE:
      g_value_set_int (value, v4l_src->rotate);
      break;
    case MFW_V4L_SRC_HFLIP:
      g_value_set_boolean (value, v4l_src->hflip);
      break;
    case MFW_V4L_SRC_VFLIP:
      g_value_set_boolean (value, v4l_src->vflip);
      break;

    case MFW_V4L_SRC_PREVIEW:
      g_value_set_boolean (value, v4l_src->preview);
      break;
    case MFW_V4L_SRC_PREVIEW_WIDTH:
      g_value_set_int (value, v4l_src->preview_width);
      break;
    case MFW_V4L_SRC_PREVIEW_HEIGHT:
      g_value_set_int (value, v4l_src->preview_height);
      break;

    case MFW_V4L_SRC_PREVIEW_TOP:
      g_value_set_int (value, v4l_src->preview_top);
      break;

    case MFW_V4L_SRC_PREVIEW_LEFT:
      g_value_set_int (value, v4l_src->preview_left);
      break;


    case MFW_V4L_SRC_CROP_LEFT:
      g_value_set_int (value, v4l_src->crop_left);
      break;
    case MFW_V4L_SRC_CROP_TOP:
      g_value_set_int (value, v4l_src->crop_top);
      break;
    case MFW_V4L_SRC_CROP_WIDTH:
      g_value_set_int (value, v4l_src->crop_width);
      break;
    case MFW_V4L_SRC_CROP_HEIGHT:
      g_value_set_int (value, v4l_src->crop_height);
      break;
    case MFW_V4L_SRC_FRAMERATE_NUM:
      g_value_set_int (value, v4l_src->fps_n);
      break;
    case MFW_V4L_SRC_FRAMERATE_DEN:
      g_value_set_int (value, v4l_src->fps_d);
      break;

#if 0
    case MFW_V4L_SRC_SENSOR_WIDTH:
      g_value_set_int (value, v4l_src->sensor_width);
      break;
    case MFW_V4L_SRC_SENSOR_HEIGHT:
      g_value_set_int (value, v4l_src->sensor_height);
      break;
#endif

    case MFW_V4L_SRC_INPUT:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened \n");
      else
      {
        /* this ioctl gets input parameter */
        if (ioctl(fd_v4l, VIDIOC_G_INPUT, &v4l_src->input) < 0)
          GST_ERROR("V4LSRC: get input parameter failed");
        else
        {
          GST_DEBUG("Input retreived=%d", v4l_src->input);
        }
      }

      g_value_set_int (value, v4l_src->input);
      break;

    case MFW_V4L_SRC_BACKGROUND:
      g_value_set_boolean (value, v4l_src->bg);
      break;
    case MFW_V4L_SRC_DEVICE:
      g_value_set_string (value, v4l_src->devicename);
      break;

    case MFW_V4L_SRC_QUEUE_SIZE:
      g_value_set_int (value, v4l_src->queue_size);
      break;

/* Enhanced to support diff RGB and YUV color-formats */
    case MFW_V4L_SRC_PIXEL_FMT:
      g_value_set_uint(value, v4l_src->pix_fmt);
      break;
      
#ifdef ENABLE_DRM
    case MFW_V4L_SRC_DISP_TYPE:
      g_value_set_string (value, v4l_src->disp_type);
      break;
#endif

/* Enhanced to support picture enhancement parameters */
    case MFW_V4L_SRC_BRIGHTNESS:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened \n");
      else
      {
        ctrl.id = V4L2_CID_BRIGHTNESS;

        /* this ioctl gets brightness parameter */
        if (ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get brightness control failed");
        else
        {
          GST_DEBUG("Brightness value=%d", ctrl.value);
          v4l_src->brightness = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->brightness);
      break;

    case MFW_V4L_SRC_CONTRAST:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_CONTRAST;

        /* this ioctl gets contrast parameter */
        if (ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get contrast control failed");
        else
        {
          GST_DEBUG("Contrast value=%d", ctrl.value);
          v4l_src->contrast = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->contrast);
      break;

    case MFW_V4L_SRC_SATURATION:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_SATURATION;

        /* this ioctl gets saturation parameter */
        if (ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get saturation control failed");
        else
        {
          GST_DEBUG("Saturation value=%d", ctrl.value);
          v4l_src->saturation = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->saturation);
      break;

    case MFW_V4L_SRC_HUE:
      if (fd_v4l < 0)
        GST_INFO("V4LSRC: fd_v4l is not opened");
      else
      {
        ctrl.id = V4L2_CID_HUE;

        /* this ioctl gets hue parameter */
        if (ioctl(fd_v4l, VIDIOC_G_CTRL, &ctrl) < 0)
          GST_ERROR("V4LSRC: get hue control failed");
        else
        {
          GST_DEBUG("Hue value=%d", ctrl.value);
          v4l_src->hue = ctrl.value;
        }
      }

      g_value_set_int(value, v4l_src->hue);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }

  g_mutex_unlock(&v4l_src->mutex);
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_set_caps
         
DESCRIPTION:         this function does the capability negotiation between adjacent pad

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        caps      -   pointer to GstCaps
        
  
RETURN VALUE:       TRUE or FALSE depending on capability is negotiated or not.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_set_caps (GstBaseSrc * src, GstCaps * caps)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);

  return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_overlay_setup
         
DESCRIPTION:         This function performs the initialisations required for preview

ARGUMENTS PASSED:    
        fd_v4l    -   capture device ID
        fmt       -   pointer to the V4L format structure.
        
  
RETURN VALUE:       TRUE - preview setup initialised successfully
                    FALSE - Error in initialising the preview set up.
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
mfw_gst_v4lsrc_overlay_setup (MFWGstV4LSrc * v4l_src, struct v4l2_format * fmt)
{
  struct v4l2_streamparm parm;
  struct v4l2_control ctl;
  struct v4l2_crop crop = {0};
  int g_sensor_top = 0;
  int g_sensor_left = 0;
  int g_camera_color = 0;
  int fd_v4l = v4l_src->fd_v4l;

  GST_INFO ("display lcd:%d\n", v4l_src->g_display_lcd);
  /* this ioctl sets up the LCD display for preview */
  if (ioctl (fd_v4l, VIDIOC_S_OUTPUT, &v4l_src->g_display_lcd) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_S_OUTPUT failed");
    return FALSE;
  }

  ctl.id = V4L2_CID_PRIVATE_BASE + 2;
  ctl.value = v4l_src->rotate;

  /* this ioctl sets rotation value on the display */
  if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl) < 0) {
    GST_ERROR (">>V4L_SRC: rotation set control failed");
    return FALSE;
  }

  if (ioctl (fd_v4l, VIDIOC_S_FMT, fmt) < 0) {
    GST_ERROR (">>V4L_SRC: set format failed");
    return FALSE;
  }

  if(v4l_src->crop_width && v4l_src->crop_height)
  {
      crop.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
      crop.c.left = v4l_src->crop_left;
      crop.c.top = v4l_src->crop_top;
      crop.c.width = v4l_src->crop_width;
      crop.c.height = v4l_src->crop_height;

      /* this ioctl sets capture rectangle */
      if (ioctl(fd_v4l, VIDIOC_S_CROP, &crop) < 0)
      {
        GST_ERROR(">>V4L_SRC: set capture rectangle for cropping failed");
        return FALSE;
      }
  }

  if (ioctl (fd_v4l, VIDIOC_G_FMT, fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return FALSE;
  }

  return TRUE;
}



/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_start_preview
         
DESCRIPTION:         This function starts the preview of capture

ARGUMENTS PASSED:    
        fd_v4l    -   capture device ID
        
  
RETURN VALUE:        TRUE - preview start initialised successfully
                     FALSE - Error in starting the preview
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
gboolean
mfw_gst_v4lsrc_start_preview (int fd_v4l)
{
  int i;
  int overlay = 1;
  struct v4l2_control ctl;
  int g_camera_color = 0;

  if (ioctl (fd_v4l, VIDIOC_OVERLAY, &overlay) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_OVERLAY start failed");
    return FALSE;
  }

  for (i = 0; i < 3; i++) {
    // flash a frame
    ctl.id = V4L2_CID_PRIVATE_BASE + 1;
    if (ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl) < 0) {
      GST_ERROR (">>V4L_SRC: set ctl failed");
      return FALSE;
    }
    sleep (1);
  }


#if 0
  if (g_camera_color == 1) {
    ctl.id = V4L2_CID_BRIGHTNESS;
    for (i = 0; i < 0xff; i += 0x20) {
      ctl.value = i;
      GST_DEBUG (">>V4L_SRC: change the brightness %d", i);
      ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl);
      sleep (1);
    }
  } else if (g_camera_color == 2) {
    ctl.id = V4L2_CID_SATURATION;
    for (i = 25; i < 150; i += 25) {
      ctl.value = i;
      GST_DEBUG (">>V4L_SRC: change the color saturation %d", i);
      ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl);
      sleep (5);
    }
  } else if (g_camera_color == 3) {
    ctl.id = V4L2_CID_RED_BALANCE;
    for (i = 0; i < 0xff; i += 0x20) {
      ctl.value = i;
      GST_DEBUG (">>V4L_SRC: change the red balance %d", i);
      ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl);
      sleep (1);
    }
  } else if (g_camera_color == 4) {
    ctl.id = V4L2_CID_BLUE_BALANCE;
    for (i = 0; i < 0xff; i += 0x20) {
      ctl.value = i;
      GST_DEBUG (">>V4L_SRC: change the blue balance %d", i);
      ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl);
      sleep (1);
    }
  } else if (g_camera_color == 5) {
    ctl.id = V4L2_CID_BLACK_LEVEL;
    for (i = 0; i < 4; i++) {
      ctl.value = i;
      GST_DEBUG (">>V4L_SRC: change the black balance %d", i);
      ioctl (fd_v4l, VIDIOC_S_CTRL, &ctl);
      sleep (5);
    }
  } else {
    sleep (2);
  }
#endif

  return TRUE;
}

#ifdef ENABLE_DRM
static gboolean
mfw_gst_v4lsrc_setup_preview (MFWGstV4LSrc * v4l_src)
{
  struct v4l2_crop crop;
  struct v4l2_format fmt;

  /* open the frame buffer to display the preview  */
  drm_plane_open(v4l_src);
  crop.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
  crop.c.width = v4l_src->preview_width;
  crop.c.height = v4l_src->preview_height;
  crop.c.top = v4l_src->preview_top;
  crop.c.left = v4l_src->preview_left;
  drm_plane_setup(v4l_src, &crop);

  fmt.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
  fmt.fmt.win.w.top = v4l_src->preview_top;
  fmt.fmt.win.w.left = v4l_src->preview_left;
  fmt.fmt.win.w.width = v4l_src->preview_width;
  fmt.fmt.win.w.height = v4l_src->preview_height;

  /* this function sets up the V4L for preview */
  if (mfw_gst_v4lsrc_overlay_setup (v4l_src, &fmt) == FALSE) {
    GST_ERROR (">>V4L_SRC: Setup overlay failed.");
    return FALSE;
  }

  /* FIXME: color key/alpha */


  return TRUE;
}
#else /* ENABLE_DRM */
static gboolean
mfw_gst_v4lsrc_setup_preview (MFWGstV4LSrc * v4l_src)
{
  struct v4l2_framebuffer fb_v4l2;
  struct v4l2_format fmt;  
  char fb_device[100] = "/dev/fb0";
  int fd_fb = 0;
  struct fb_fix_screeninfo fix;
  struct fb_var_screeninfo var;
  struct mxcfb_color_key color_key;
  struct mxcfb_gbl_alpha alpha;
  unsigned short *fb0;
  unsigned char *cur_fb8;
  unsigned short *cur_fb16;
  unsigned int *cur_fb32;
  int g_display_width = 0;
  int g_display_height = 0;
  int g_display_top = 0;
  int g_display_left = 0;
  __u32 screen_size;
  int h, w;
  int ret = 0;  

	g_display_width = v4l_src->preview_width;
	g_display_height = v4l_src->preview_height;
	g_display_top = v4l_src->preview_top;
	g_display_left = v4l_src->preview_left;
	fmt.type = V4L2_BUF_TYPE_VIDEO_OVERLAY;
	fmt.fmt.win.w.top = g_display_top;
	fmt.fmt.win.w.left = g_display_left;
	fmt.fmt.win.w.width = g_display_width;
	fmt.fmt.win.w.height = g_display_height;

	/* open the frame buffer to display the preview  */
	if ((fd_fb = open (fb_device, O_RDWR)) < 0) {
		GST_ERROR (">>V4L_SRC: Unable to open frame buffer");
		return FALSE;
	}

    if (ioctl (fd_fb, FBIOGET_VSCREENINFO, &var) < 0) {
      close (fd_fb);
      return FALSE;
    }
    if (ioctl (fd_fb, FBIOGET_FSCREENINFO, &fix) < 0) {
      close (fd_fb);
      return FALSE;
    }
    if (strcmp (fix.id, "DISP3 BG - DI1") == 0)
      v4l_src->g_display_lcd = 1;
    else if (strcmp (fix.id, "DISP3 BG") == 0)
      v4l_src->g_display_lcd = 0;

    /* this function sets up the V4L for preview */
    if (mfw_gst_v4lsrc_overlay_setup (v4l_src, &fmt) == FALSE) {
      GST_ERROR (">>V4L_SRC: Setup overlay failed.");
      return FALSE;
    }

    if (!v4l_src->bg) {
      fb_v4l2.fmt.width = var.xres;
      fb_v4l2.fmt.height = var.yres;
      if (var.bits_per_pixel == 32) {
        fb_v4l2.fmt.pixelformat = IPU_PIX_FMT_BGR32;
        fb_v4l2.fmt.bytesperline = 4 * fb_v4l2.fmt.width;
      } else if (var.bits_per_pixel == 24) {
        fb_v4l2.fmt.pixelformat = IPU_PIX_FMT_BGR24;
        fb_v4l2.fmt.bytesperline = 3 * fb_v4l2.fmt.width;
      } else if (var.bits_per_pixel == 16) {
        fb_v4l2.fmt.pixelformat = IPU_PIX_FMT_RGB565;
        fb_v4l2.fmt.bytesperline = 2 * fb_v4l2.fmt.width;
      }

      fb_v4l2.flags = V4L2_FBUF_FLAG_PRIMARY;
      fb_v4l2.base = (void *) fix.smem_start;
    } else {
      /* alpha blending in done in case of display happeing both in the 
         back ground and foreground simultaneously */
      alpha.alpha = 0;
      alpha.enable = 1;
      if (ioctl (fd_fb, MXCFB_SET_GBL_ALPHA, &alpha) < 0) {

        GST_ERROR ("MXCFB_SET_GBL_ALPHA ioctl failed");
        close (fd_fb);
        return FALSE;
      }

      color_key.color_key = 0x00080808;
      color_key.enable = 1;
      if (ioctl (fd_fb, MXCFB_SET_CLR_KEY, &color_key) < 0) {
        GST_ERROR ("MXCFB_SET_CLR_KEY ioctl failed");
        close (fd_fb);
        return FALSE;
      }

      if (ioctl (v4l_src->fd_v4l, VIDIOC_G_FBUF, &fb_v4l2) < 0) {
        GST_ERROR (">>V4L_SRC: Get framebuffer failed");
        return FALSE;
      }
      fb_v4l2.flags = V4L2_FBUF_FLAG_OVERLAY;
    }

    close (fd_fb);

    if (ioctl (v4l_src->fd_v4l, VIDIOC_S_FBUF, &fb_v4l2) < 0) {
      GST_ERROR (">>V4L_SRC: set framebuffer failed");
      return FALSE;
    }

    if (ioctl (v4l_src->fd_v4l, VIDIOC_G_FBUF, &fb_v4l2) < 0) {
      GST_ERROR (">>V4L_SRC: set framebuffer failed");
      return FALSE;
    }

    GST_DEBUG ("frame buffer width %d, height %d, bytesperline %d",
        fb_v4l2.fmt.width, fb_v4l2.fmt.height, fb_v4l2.fmt.bytesperline);

    return TRUE;
}
#endif /* ENABLE_DRM */

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_start
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to start the video capturing process 
                     from this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
RETURN VALUE:        TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_start (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  struct v4l2_format fmt;

  v4l_src->fd_v4l = mfw_gst_v4lsrc_capture_setup (v4l_src);
  if (v4l_src->fd_v4l <= 0) {
    GST_ERROR ("v4lsrc:error in opening the device");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {
    mfw_gst_v4lsrc_setup_preview(v4l_src);
  }

  fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  if (ioctl (v4l_src->fd_v4l, VIDIOC_G_FMT, &fmt) < 0) {
    GST_ERROR (">>V4L_SRC: get format failed");
    return FALSE;
  } else {
    v4l_src->buffer_size = fmt.fmt.pix.sizeimage;
    GST_DEBUG ("Width = %d", fmt.fmt.pix.width);
    GST_DEBUG ("Height = %d", fmt.fmt.pix.height);
    GST_DEBUG ("Image size = %d", fmt.fmt.pix.sizeimage);
    GST_DEBUG ("pixelformat = %d", fmt.fmt.pix.pixelformat);
  }

  if (mfw_gst_v4lsrc_start_capturing (v4l_src) < 0) {
    GST_ERROR ("start_capturing failed");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {
    mfw_gst_v4lsrc_start_preview (v4l_src->fd_v4l);
  }

  v4l_src->offset = 0;
  return TRUE;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_stop
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to stop the video capturing process 
                     by this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
  
RETURN VALUE:        TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_stop (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  gint overlay = 0;

  if (mfw_gst_v4lsrc_stop_capturing (v4l_src) < 0) {
    GST_ERROR (">>V4L_SRC: stop_capturing failed");
    return FALSE;
  }

  if (TRUE == v4l_src->preview) {

    if (ioctl (v4l_src->fd_v4l, VIDIOC_OVERLAY, &overlay) < 0) {
      printf ("VIDIOC_OVERLAY stop failed\n");
      return FALSE;
    }

    drm_plane_close (v4l_src);
  }

  g_mutex_lock(&v4l_src->mutex);
  close (v4l_src->fd_v4l);
  v4l_src->fd_v4l = -1;
  g_mutex_unlock(&v4l_src->mutex);

/* Setting boolean capture_start_flag=FALSE, so that color format can be 
 * configured 
 */
  v4l_src->capture_start_flag = FALSE;

  return TRUE;
}


/*=============================================================================
FUNCTION:           mfw_gst_v4lsrc_buffer_new
         
DESCRIPTION:        This function is used to store the frames captured by the
                    V4L capture driver

ARGUMENTS PASSED:   v4l_src     - 
        
RETURN VALUE:       TRUE or FALSE depending on the sate of capture initiation
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstBuffer *
mfw_gst_v4lsrc_buffer_new (MFWGstV4LSrc * v4l_src)
{
  GstBuffer *buf;
  gint fps_n, fps_d;
  struct v4l2_buffer v4lbuf;
  GstClockTime ts, res;
  gint wait_cnt = 0;

  v4l_src->count++;
  memset (&v4lbuf, 0, sizeof (v4lbuf));
  v4lbuf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
  v4lbuf.memory = V4L2_MEMORY_MMAP;

  while (g_list_length (v4l_src->free_pool) == 0) {
      GST_WARNING ("no buffer available in pool");
      g_usleep(NO_BUFFER_WAIT_TIME_MS*1000);
      wait_cnt ++;
      if (wait_cnt >= NO_BUFFER_WAIT_COUNT) {
          GST_ERROR ("no buffer available in %d ms", NO_BUFFER_WAIT_TIME_MS*NO_BUFFER_WAIT_COUNT);
          return NULL;
      }
  }

  if (v4l_src->stop == TRUE) {
          GST_WARNING("v4l src stopped.");
          return NULL;
  }

  if (ioctl (v4l_src->fd_v4l, VIDIOC_DQBUF, &v4lbuf) < 0) {
    GST_ERROR (">>V4L_SRC: VIDIOC_DQBUF failed.");
    return NULL;
  }

  /**
   * Add check for V4L2_BUF_FLAG_ERROR flag, if set
   * need to discard the captured data buffer and signal is
   * emitted to notify the gstreamer framework captured frame
   * data is error
   */
  if ((V4L2_BUF_FLAG_ERROR == (v4lbuf.flags & V4L2_BUF_FLAG_ERROR)) &&
      (TRUE != v4l_src->v4l2_buf_flag_err))
  {
    v4l_src->v4l2_buf_flag_err = TRUE;

    g_signal_emit(v4l_src, gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_ERROR], 0,
                  GST_BASE_SRC_PAD(v4l_src));
  }

  /**
   * SIGNAL_V4L2_BUF_FLAG_OK signal is emitted to notify the gstreamer
   * framework that captured frame data is correct, if v4l2_buf_flag_err is set
   * to TRUE and if V4L2_BUF_FLAG_ERROR buffer flag is not set
   */
  else if (!(V4L2_BUF_FLAG_ERROR == (v4lbuf.flags & V4L2_BUF_FLAG_ERROR)) &&
           (TRUE == v4l_src->v4l2_buf_flag_err))
  {
    v4l_src->v4l2_buf_flag_err = FALSE;

    g_signal_emit(v4l_src, gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_OK], 0,
                  GST_BASE_SRC_PAD(v4l_src));
  }
  else
  {
    /* Nothing to be done */
  }

  g_mutex_lock (v4l_src->pool_lock);
  if (g_list_find (v4l_src->free_pool, (gpointer) (v4lbuf.index)))
    GST_LOG ("v4l buffer index:%d will be used outside", v4lbuf.index);
  else
    GST_WARNING ("v4l buffer index:%d can not be found in pool", v4lbuf.index);

  v4l_src->free_pool =
      g_list_remove (v4l_src->free_pool, (gpointer) (v4lbuf.index));

  g_mutex_unlock (v4l_src->pool_lock);

  buf = (GstBuffer *) (v4l_src->buffers[v4lbuf.index]);
  GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_LAST);

  GST_LOG ("v4l dequeued buffer index:%d(ref %d), num in pool:%d", v4lbuf.index,
      buf->mini_object.refcount, g_list_length (v4l_src->free_pool));

  GST_BUFFER_SIZE (buf) = v4l_src->buffer_size;

  ts = gst_clock_get_time (GST_ELEMENT (v4l_src)->clock);
  if (ts != GST_CLOCK_TIME_NONE)
    ts -= gst_element_get_base_time (GST_ELEMENT (v4l_src));
  else
    ts = v4l_src->count * v4l_src->time_per_frame;
  GST_BUFFER_TIMESTAMP (buf) = ts;
  GST_BUFFER_DURATION (buf) = v4l_src->time_per_frame;

  if (v4l_src->last_ts) {
    guint num_frame_delay = 0;
    GstClockTimeDiff diff = ts - v4l_src->last_ts;
    if (ts < v4l_src->last_ts)
      diff = v4l_src->last_ts + ts;
    while (diff > v4l_src->time_per_frame) {
      diff -= v4l_src->time_per_frame;
      num_frame_delay++;
    }
    if (num_frame_delay > 1)
      GST_LOG (">>V4L_SRC: Camera ts late by %d frames", num_frame_delay);
  }
  v4l_src->last_ts = ts;

  gst_buffer_set_caps (buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (v4l_src)));
  return buf;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_create
         
DESCRIPTION:         This function is registered with the Base Source Class 
                     This function updates the the buffer to be pushed to the
                     next element with the frame captured.
                     
ARGUMENTS PASSED:    v4l_src     - 
        
  
RETURN VALUE:        
              GST_FLOW_OK       -    buffer create successfull.
              GST_FLOW_ERROR    -    Error in buffer creation.

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstFlowReturn
mfw_gst_v4lsrc_create (GstPushSrc * src, GstBuffer ** buf)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  *buf = mfw_gst_v4lsrc_buffer_new (v4l_src);
  if (*buf == NULL)
    return GST_FLOW_ERROR;
  else
    return GST_FLOW_OK;

}

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_unlock
         
DESCRIPTION:         this function is registered  with the Base Source Class of
                     the gstreamer to unlock any block in mfw_gst_v4lsrc_create
                     by this function

ARGUMENTS PASSED:    
        src       -   pointer to base source 
        
  
RETURN VALUE:        TRUE
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean
mfw_gst_v4lsrc_unlock (GstBaseSrc * src)
{
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  v4l_src->stop = TRUE;

  return TRUE;
}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_change_state

DESCRIPTION: this function keeps track of different states of pipeline.

ARGUMENTS PASSED:
        element     -   pointer to element
        transition  -   state of the pipeline

RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_v4lsrc_change_state(GstElement* element, GstStateChange transition)
{
    
    GstStateChangeReturn retstate = GST_STATE_CHANGE_FAILURE;
    MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (element);
       
    switch (transition)
    {
        case GST_STATE_CHANGE_NULL_TO_READY:
            GST_DEBUG(" in NULL to READY state \n");
               break;
        case GST_STATE_CHANGE_READY_TO_PAUSED:
            GST_DEBUG(" in READY_TO_PAUSED state \n");
            break;
        case  GST_STATE_CHANGE_PAUSED_TO_PLAYING:
            GST_DEBUG(" in  to  PAUSED_TO_PLAYING state \n");
            v4l_src->stop = FALSE;
            break;
        default:
            break;
    }

    retstate = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

    switch (transition)
    {
        case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
            GST_DEBUG(" in  to PLAYING_TO_PAUSED state \n");
            v4l_src->stop = TRUE;
            break;
        case GST_STATE_CHANGE_PAUSED_TO_READY:
            GST_DEBUG(" in  to PAUSED_TO_READY state \n");
            break;
        case GST_STATE_CHANGE_READY_TO_NULL:
            GST_DEBUG(" in  to READY_TO_NULL state \n");
            break;
        default:
            break;
    }
    
    return retstate;
}

/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_get_caps
         
DESCRIPTION:         This function gets the caps to be set on the source pad.
                     

ARGUMENTS PASSED:    
        v4l_src     - 
         
RETURN VALUE:       Returns the caps to be set.
        
PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static GstCaps *
mfw_gst_v4lsrc_get_caps (GstBaseSrc * src)
{
  GstCaps *list;
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (src);
  GstCaps *capslist;
  GstPadTemplate *src_template = NULL;
  gint i;
  guint32 format;

#ifndef MX51
  switch(v4l_src->pix_fmt)
  {
    case PIX_FMT_RGB565:
      format = GST_MAKE_FOURCC('R', 'G', 'B', 'P');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGB_16);
      break;

    case PIX_FMT_BGR24:
      format = GST_MAKE_FOURCC('B', 'G', 'R', '3');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_BGR);
      break;

    case PIX_FMT_RGB24:
      format = GST_MAKE_FOURCC('R', 'G', 'B', '3');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGB);
      break;

    case PIX_FMT_BGR32:
      format = GST_MAKE_FOURCC('B', 'G', 'R', '4');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_BGRx);
      break;

    case PIX_FMT_RGB32:
      format = GST_MAKE_FOURCC('R', 'G', 'B', '4');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_RGBx);
      break;

    case PIX_FMT_YUV422P:
      format = GST_MAKE_FOURCC('4', '2', '2', 'P');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("422P"));
      break;

    case PIX_FMT_UYVY:
      format = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("UYVY"));
      break;

    case PIX_FMT_YUYV:
      format = GST_MAKE_FOURCC('Y', 'U', 'Y', 'V');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("YUY2"));
      break;

    case PIX_FMT_YUV420: /* YU12 */
      format = GST_MAKE_FOURCC('Y', 'U', '1', '2');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("I420"));
      break;

    case PIX_FMT_NV12:
      format = GST_MAKE_FOURCC ('N', 'V', '1', '2');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("NV12"));
      break;

    default: /* Unlikely, default color format UYVY is set instead */
      format = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      capslist = gst_caps_from_string(GST_VIDEO_CAPS_YUV ("UYVY"));
      break;
  }
#else
  format = GST_MAKE_FOURCC ('N', 'V', '1', '2');
#endif

  return capslist;
}


/*=============================================================================
FUNCTION:            mfw_gst_v4lsrc_fixate
         
DESCRIPTION:         Fixes the Caps on the source pad
                     

ARGUMENTS PASSED:    v4l_src     - 
        
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_fixate (GstPad * pad, GstCaps * caps)
{

  gint i = 0;
  GstStructure *structure = NULL;
  MFWGstV4LSrc *v4l_src = MFW_GST_V4LSRC (gst_pad_get_parent (pad));
  guint32 fourcc;

#ifndef MX51
  switch(v4l_src->pix_fmt)
  {
    case PIX_FMT_RGB565:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', 'P'); 
      break;

    case PIX_FMT_BGR24:
      fourcc = GST_MAKE_FOURCC('B', 'G', 'R', '3'); 
      break;

    case PIX_FMT_RGB24:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', '3'); 
      break;

    case PIX_FMT_BGR32:
      fourcc = GST_MAKE_FOURCC('B', 'G', 'R', '4');
      break;

    case PIX_FMT_RGB32:
      fourcc = GST_MAKE_FOURCC('R', 'G', 'B', '4');
      break;

    case PIX_FMT_YUV422P:
      fourcc = GST_MAKE_FOURCC('4', '2', '2', 'P');
      break;

    case PIX_FMT_UYVY:
      fourcc = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      break;

    case PIX_FMT_YUYV:
      fourcc = GST_MAKE_FOURCC('Y', 'U', 'Y', 'V');
      break;

    case PIX_FMT_YUV420: /* YU12 */
      fourcc = GST_MAKE_FOURCC('Y', 'U', '1', '2'); 
      break;

    case PIX_FMT_NV12:
      fourcc = GST_MAKE_FOURCC ('N', 'V', '1', '2');
      break;

    default: /* Unlikely, default color format UYVY is set instead */
      fourcc = GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y');
      break;	
  }   
#else
  fourcc = GST_MAKE_FOURCC ('N', 'V', '1', '2');
#endif

  const GValue *v = NULL;
  for (i = 0; i < gst_caps_get_size (caps); ++i) {
    structure = gst_caps_get_structure (caps, i);
    gst_structure_fixate_field_nearest_int (structure, "width",
        v4l_src->capture_width);
    gst_structure_fixate_field_nearest_int (structure, "height",
        v4l_src->capture_height);
    gst_structure_fixate_field_nearest_fraction (structure, "framerate",
        v4l_src->fps_n, v4l_src->fps_d);
    gst_structure_set (structure, "pixel-aspect-ratio", GST_TYPE_FRACTION,
        1, 1, NULL);

    gst_structure_set (structure, "format", GST_TYPE_FOURCC, fourcc, NULL);
  }
  GST_INFO ("capture: %d, %d, fixrate :%s\n", v4l_src->capture_width,
      v4l_src->capture_height, gst_caps_to_string (caps));
  gst_object_unref (v4l_src);

}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_init   
        
DESCRIPTION:     create the pad template that has been registered with the 
                element class in the _base_init and do library table 
                initialization      

ARGUMENTS PASSED:
        context  -    pointer to v4lsrc element structure      
  
RETURN VALUE:       None
      
PRE-CONDITIONS:     _base_init and _class_init are called 
 
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static void
mfw_gst_v4lsrc_init (MFWGstV4LSrc * v4l_src, MFWGstV4LSrcClass * klass)
{
  v4l_src->capture_width = 640;
  v4l_src->capture_height = 480;
  v4l_src->fps_n = 30;
  v4l_src->fps_d = 1;
  v4l_src->fd_v4l = -1;
  v4l_src->count = 0;
  v4l_src->buffer_size = 0;
  v4l_src->offset = 0;
  v4l_src->rotate = 0;
  v4l_src->hflip = FALSE;
  v4l_src->vflip = FALSE;
  v4l_src->preview = FALSE;
  v4l_src->preview_width = 160;
  v4l_src->preview_height = 128;
  v4l_src->preview_top = 0;
  v4l_src->preview_left = 0;
  v4l_src->sensor_width = 1280;
  v4l_src->sensor_height = 1024;
  v4l_src->input = 0;
  v4l_src->input_flag = FALSE;
  v4l_src->bg = FALSE;
  v4l_src->g_display_lcd = 0;
  v4l_src->queue_size = DEFAULT_QUEUE_SIZE;
  v4l_src->start = FALSE;
  v4l_src->stop = FALSE;
  v4l_src->pix_fmt = PIX_FMT_UYVY; /* Enhanced to support diff RGB and YUV color-formats */
  v4l_src->capture_start_flag = FALSE; /* Enhanced to support diff RGB and YUV color-formats */
  v4l_src->brightness = DEFAULT_BRIGHTNESS_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->contrast = DEFAULT_CONTRAST_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->saturation = DEFAULT_SATURATION_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->hue = DEFAULT_HUE_VAL; /* Enhanced to support picture enhancement parameters */
  v4l_src->brightness_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->contrast_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->saturation_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->hue_set_flag = FALSE; /* Enhanced to support picture enhancement parameters */
  v4l_src->v4l2_buf_flag_err = FALSE; /* Enhanced to support error handling signal */
  v4l_src->crop_left = 0; /* Enhanced to support cropping feature */
  v4l_src->crop_top = 0; /* Enhanced to support cropping feature */
  v4l_src->crop_width = 0; /* Enhanced to support cropping feature */
  v4l_src->crop_height = 0; /* Enhanced to support cropping feature */

#ifdef MX27
  v4l_src->devicename = g_strdup ("/dev/v4l/video0");
#else
  v4l_src->devicename = g_strdup ("/dev/video0");
#endif
  v4l_src->buf_pools = g_malloc (sizeof (GstBuffer *) * v4l_src->queue_size);

#ifdef ENABLE_DRM
  v4l_src->disp_type = NULL;
#endif

  gst_pad_set_fixatecaps_function (GST_BASE_SRC_PAD (v4l_src),
      mfw_gst_v4lsrc_fixate);
  gst_base_src_set_live (GST_BASE_SRC (v4l_src), TRUE);
  drm_plane_init (v4l_src);

#define MFW_GST_V4LSRC_PLUGIN VERSION
  PRINT_PLUGIN_VERSION (MFW_GST_V4LSRC_PLUGIN);
  return;
}

/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_class_init    
        
DESCRIPTION:     Initialise the class only once (specifying what signals,
                arguments and virtual functions the class has and setting up 
                global state)    
     

ARGUMENTS PASSED:
       klass   -   pointer to mp3decoder element class
        
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_class_init (MFWGstV4LSrcClass * klass)
{

  GObjectClass *gobject_class;
  GstBaseSrcClass *basesrc_class;
  GstPushSrcClass *pushsrc_class;
  GstElementClass *element_class;

  gobject_class = (GObjectClass *) klass;
  element_class = (GstElementClass *) klass;
  basesrc_class = (GstBaseSrcClass *) klass;
  pushsrc_class = (GstPushSrcClass *) klass;


  gobject_class->set_property = mfw_gst_v4lsrc_set_property;
  gobject_class->get_property = mfw_gst_v4lsrc_get_property;
  element_class->change_state = mfw_gst_v4lsrc_change_state;

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_WIDTH,
      g_param_spec_int ("capture-width",
		  "capture_width",
		  "gets the width of the image to be captured",
		  16, G_MAXINT, 640, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_HEIGHT,
      g_param_spec_int ("capture-height",
		  "capture_height",
		  "gets the height of the image to be captured",
		  16, G_MAXINT, 480, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_INPUT,
      g_param_spec_int ("input",
          "input",
          "set analog video inputs \n"
          "\t\t\t\tinput=0: Composite on Ain1 \n"
          "\t\t\t\tinput=1: Composite on Ain2 \n"
          "\t\t\t\tinput=2: Composite on Ain3 \n"
          "\t\t\t\tinput=3: Composite on Ain4 \n"
          "\t\t\t\tinput=4: Composite on Ain5 \n"
          "\t\t\t\tinput=5: Composite on Ain6",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_WIDTH,
      g_param_spec_int ("preview-width",
          "preview_width",
          "gets the width of the image to be displayed for preview. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          16, 1920, 176, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_HEIGHT,
      g_param_spec_int ("preview-height",
          "preview_height",
          "gets the height of the image to be displayed for preview. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          16, 1080, 144, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_TOP,
      g_param_spec_int ("preview-top",
          "preview_top",
          "gets the top pixel offset at which the preview should start. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          0, 320, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW_LEFT,
      g_param_spec_int ("preview-left",
          "preview_left",
          "gets the left pixel offset at which the preview should start. \n"
          "\t\t\tNote:property is valid only when preview property is enabled",
          0, 240, 0, G_PARAM_READWRITE));


  /* 
   * FixME: The overlay channel will cause v4l error:
   * v4l2 capture: mxc_v4l_dqueue timeout enc_counter 0 error 
   * disable it 
   */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_PREVIEW,
      g_param_spec_boolean ("preview", "Preview",
          "enable the preview of capture, it will directly pass the data to display",
          FALSE, G_PARAM_READWRITE));


  g_object_class_install_property (gobject_class, MFW_V4L_SRC_ROTATE,
      g_param_spec_int ("rotate",
			"Rotate",
			"gets the values by which the camera rotation "
			"angle can be specified in degrees",
			0, 270, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_HFLIP,
      g_param_spec_boolean ("hflip", "hflip",
          "enable horizontal flip", FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_VFLIP,
      g_param_spec_boolean ("vflip", "vflip",
          "enable vertical flip", FALSE, G_PARAM_READWRITE));

  /* Enhanced to support cropping feature */
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_LEFT,
      g_param_spec_int ("crop-left",
          "crop-left",
          "number of pixel to crop from the left",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_TOP,
      g_param_spec_int ("crop-top",
          "crop-top",
          "number of pixel to crop from the top",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_WIDTH,
      g_param_spec_int ("crop-width",
          "crop-width",
          "image width after cropping (required to be set to enable cropping)",
          0, G_MAXINT, 0, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_CROP_HEIGHT,
      g_param_spec_int ("crop-height",
          "crop-height",
          "image height after cropping (required to be set to enable cropping)",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_FRAMERATE_NUM,
      g_param_spec_int ("fps-n",
          "fps_n",
          "gets the numerator of the framerate at which"
          "the input stream is to be captured",
          0, G_MAXINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_FRAMERATE_DEN,
      g_param_spec_int ("fps-d",
          "fps_d",
          "gets the denominator of the framerate at which"
          "the input stream is to be captured",
          1, G_MAXINT, 1, G_PARAM_READWRITE));
#if 0
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_SENSOR_WIDTH,
      g_param_spec_int ("sensor-width",
          "sensor_width",
          "gets the width of the sensor",
          16, G_MAXINT, 1280, G_PARAM_READWRITE));
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_SENSOR_HEIGHT,
      g_param_spec_int ("sensor-height",
          "sensor_height",
          "gets the height of the sensor",
          16, G_MAXINT, 1024, G_PARAM_READWRITE));
#endif
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_BACKGROUND,
      g_param_spec_boolean ("bg", "BG display",
          "Set BG display or FG display", FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_DEVICE,
      g_param_spec_string ("device", "Device", "Device location",
          NULL, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, MFW_V4L_SRC_QUEUE_SIZE,
      g_param_spec_int ("queue-size",
          "queue-size",
          "v4l2 request buffer number", 0, G_MAXINT, 5, G_PARAM_READWRITE));

/* Enhanced to support diff RGB and YUV color-formats */  
  g_object_class_install_property(gobject_class, MFW_V4L_SRC_PIXEL_FMT, 
      g_param_spec_uint("pix-fmt", "Pixel format", 
          "Support different YUV and RGB pixel/color formats \n"
          "\t\t\t\tpix-fmt=0 (PIX_FMT_RGB565)  : Capture RGB565 image \n"
          "\t\t\t\tpix-fmt=1 (PIX_FMT_BGR24)   : Capture BGR24 image \n"
          "\t\t\t\tpix-fmt=2 (PIX_FMT_RGB24)   : Capture RGB24 image \n"
          "\t\t\t\tpix-fmt=3 (PIX_FMT_BGR32)   : Capture BGR32 image \n"
          "\t\t\t\tpix-fmt=4 (PIX_FMT_RGB32)   : Capture RGB32 image \n"
          "\t\t\t\tpix-fmt=5 (PIX_FMT_YUV422P) : Capture YUV422P image \n"
          "\t\t\t\tpix-fmt=6 (PIX_FMT_UYVY)    : Capture UYVY image \n"
          "\t\t\t\tpix-fmt=7 (PIX_FMT_YUYV)    : Capture YUYV image \n"
          "\t\t\t\tpix-fmt=8 (PIX_FMT_YUV420)  : Capture YUV420 image \n"                    
          "\t\t\t\tpix-fmt=9 (PIX_FMT_NV12)    : Capture NV12 image ",          
          0, G_MAX_PIX_FMT, 0, G_PARAM_READWRITE));
          
#ifdef ENABLE_DRM
  g_object_class_install_property (gobject_class, MFW_V4L_SRC_DISP_TYPE,
      g_param_spec_string ("disp-type",
          "Disp_Type", "sets the DRM display type (hdmia, hdmib, lvds, ...)",
			   NULL, G_PARAM_READWRITE));
#endif

/* Enhanced to support picture enhancement parameters */
  g_object_class_install_property(gobject_class, MFW_V4L_SRC_BRIGHTNESS,
          g_param_spec_int("brightness", "brightness parameter",
              "Brightness parameter configurable support", 0, G_MAXINT,
              DEFAULT_BRIGHTNESS_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_CONTRAST,
          g_param_spec_int("contrast", "contrast parameter",
              "Contrast parameter configurable support", 0, G_MAXINT,
              DEFAULT_CONTRAST_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_SATURATION,
          g_param_spec_int("saturation", "saturation parameter",
              "Saturation parameter configurable support", 0, G_MAXINT,
              DEFAULT_SATURATION_VAL, G_PARAM_READWRITE));

  g_object_class_install_property(gobject_class, MFW_V4L_SRC_HUE,
          g_param_spec_int("hue", "hue parameter",
              "Hue parameter configurable support", 0, G_MAXINT,
              DEFAULT_HUE_VAL, G_PARAM_READWRITE));

  /**
   * Add signal. Check for V4L2_BUF_FLAG_ERROR flag and emit
   * signal if flag is set
   */
  gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_ERROR] =
      g_signal_new ("v4l-buf-flag-error",
          G_TYPE_FROM_CLASS (klass),
          G_SIGNAL_RUN_FIRST,
          G_STRUCT_OFFSET (MFWGstV4LSrcClass, v4l_buf_flag_error),
          NULL, NULL,
          gst_marshal_VOID__OBJECT,
          G_TYPE_NONE, 1,
          GST_TYPE_PAD);

  /**
   * Add SIGNAL_V4L2_BUF_FLAG_OK signal to notify the Gstreamer framework
   */
  gst_v4l_src_signals[SIGNAL_V4L2_BUF_FLAG_OK] =
      g_signal_new ("v4l-buf-flag-ok",
          G_TYPE_FROM_CLASS (klass),
          G_SIGNAL_RUN_FIRST,
          G_STRUCT_OFFSET (MFWGstV4LSrcClass, v4l_buf_flag_ok),
          NULL, NULL,
          gst_marshal_VOID__OBJECT,
          G_TYPE_NONE, 1,
          GST_TYPE_PAD);

  basesrc_class->get_caps = mfw_gst_v4lsrc_get_caps;
  basesrc_class->set_caps = mfw_gst_v4lsrc_set_caps;
  basesrc_class->start = mfw_gst_v4lsrc_start;
  basesrc_class->stop = mfw_gst_v4lsrc_stop;
  basesrc_class->unlock = mfw_gst_v4lsrc_unlock;
  pushsrc_class->create = mfw_gst_v4lsrc_create;
  return;
}


/*=============================================================================
FUNCTION:   mfw_gst_v4lsrc_base_init   
        
DESCRIPTION:     v4l source element details are registered with the plugin during
                _base_init ,This function will initialise the class and child 
                class properties during each new child class creation       

ARGUMENTS PASSED:
        Klass   -   void pointer
  
RETURN VALUE:        None
PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static void
mfw_gst_v4lsrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  FSL_GST_ELEMENT_SET_DETAIL_SIMPLE (element_class, "v4l2 based camera src",
      "Src/Video", "Capture videos by using csi camera");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_factory));

  GST_DEBUG_CATEGORY_INIT (mfw_gst_v4lsrc_debug, "mfw_v4lsrc", 0,
      "V4L2 video src element");

  return;
}

/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or 
                element is loaded and information returned by this function 
                will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded 
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any 
        dependency correctly

PRE-CONDITIONS:      None
POST-CONDITIONS:     None
IMPORTANT NOTES:     None
=============================================================================*/
static gboolean
plugin_init (GstPlugin * plugin)
{
  if (!gst_element_register (plugin, "mfw_v4lsrc", GST_RANK_PRIMARY,
          MFW_GST_TYPE_V4LSRC))
    return FALSE;

  return TRUE;
}

/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/

FSL_GST_PLUGIN_DEFINE ("v4lsrc", "v4l2-based csi camera video src",
    plugin_init);
