/*
 * Copyright (c) 2009-2012, Freescale Semiconductor, Inc. All rights reserved.
 *
 */

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Module Name:    mfw_gst_fb.c
 *
 * Description:    Implementation of FB operations
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */

/*
 * Changelog:
 *
 */

/*=============================================================================
                            INCLUDE FILES
=============================================================================*/

#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
//#if ((!defined (_MX233)) && (!defined (_MX28)) && (!defined (_MX50)))
//#include <linux/mxcfb.h>
//#else
//#include <linux/fb.h>
//#endif

#if ((defined (_MX233)) || (defined (_MX28)) )
#include <linux/fb.h>
#else
#include <linux/fb.h>
#include <linux/mxcfb.h>
#endif

#include "mfw_gst_utils.h"
#include "mfw_gst_fb.h"

#include "mfw_gst_v4lsink.h"

/*=============================================================================
                             MACROS
=============================================================================*/

#define FB_DEVICE "/dev/fb0"

typedef struct mfw_gst_fb_priv {
	gint fd_fb0;
#ifdef ENABLE_LOC_ALPHA
	gint fd_lalpfb;
	struct mxcfb_loc_alpha lalpha;
	void *lalp_buf_vaddr[2];
	gint lalp_buf_cidx;
#endif
#ifdef ENABLE_TVOUT
	gint fd_tvout;
#endif
} mfw_gst_fb_priv_t;

/*=============================================================================
                             GLOBAL VARIABLES
=============================================================================*/

extern guint mfw_gst_v4lsink_signals[SIGNAL_LAST];

/*=============================================================================
                        LOCAL FUNCTION PROTOTYPES
=============================================================================*/


/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/

#define COLORKEY_RED       1 
#define COLORKEY_GREEN     2 
#define COLORKEY_BLUE      3 



#define RGB888(r,g,b)\
    ((((guint32)(r))<<16)|(((guint32)(g))<<8)|(((guint32)(b))))
#define RGB888TORGB565(rgb)\
    ((((rgb)<<8)>>27<<11)|(((rgb)<<18)>>26<<5)|(((rgb)<<27)>>27))

#define RGB565TOCOLORKEY(rgb)                              \
      ( ((rgb & 0xf800)<<8)  |  ((rgb & 0xe000)<<3)  |     \
        ((rgb & 0x07e0)<<5)  |  ((rgb & 0x0600)>>1)  |     \
        ((rgb & 0x001f)<<3)  |  ((rgb & 0x001c)>>2)  )


/*=============================================================================
FUNCTION:           init

DESCRIPTION:        This function allocates and initializes our private data.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean mfw_gst_fb_init(MFW_GST_V4LSINK_INFO_T *v4l)
{
	mfw_gst_fb_priv_t *priv = g_malloc(sizeof(mfw_gst_fb_priv_t));
	if (!priv) {
		GST_ERROR ("Can not allocate fb private data!");
		v4l->fb_priv = NULL;
		return FALSE;
	}

	priv->fd_fb0 = 0;

#ifdef ENABLE_LOC_ALPHA
	priv->fd_lalpfb = 0;
	v4l->alpha_enable = ALPHA_GLOBAL;
	memset (&priv->lalpha, 0, sizeof (struct mxcfb_loc_alpha));
	priv->lalp_buf_vaddr[0] = priv->lalp_buf_vaddr[1] = 0;
	v4l->alpha = 255;
#endif

	v4l->fb_priv = priv;
	return TRUE;
}

static void mfw_gst_fb_release(MFW_GST_V4LSINK_INFO_T *v4l)
{
	mfw_gst_fb_priv_t *priv = v4l->fb_priv;
	g_free(priv);
}

/*=============================================================================
FUNCTION:           fb0_set_global_alpha

DESCRIPTION:        This function Set the global alpha value of fb0.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean fb0_set_global_alpha(MFW_GST_V4LSINK_INFO_T *v4l,
				     gint alphaVal, gboolean enable)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    gboolean ret = TRUE;

#if ((!defined (_MX233)) && (!defined (_MX28)) && (!defined (_MX50)))
    struct mxcfb_gbl_alpha alpha;

    alpha.alpha = alphaVal;
    alpha.enable = enable;
    if (priv->fd_fb0 == 0)
        g_print("no fb0 device\n");
    if (ioctl(priv->fd_fb0, MXCFB_SET_GBL_ALPHA, &alpha) < 0) {
        g_print("set global alpha failed.\n");
        ret = FALSE;
    }
#endif
    return ret;
}

/*=============================================================================
FUNCTION:           fb0_set_colorkey

DESCRIPTION:          This function compute the colorkey and return the color
                    value based on color depth.

ARGUMENTS PASSED:

RETURN VALUE:

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean fb0_set_colorkey(MFW_GST_V4LSINK_INFO_T *v4l,
				 gulong *colorSrc, gboolean enable)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    gboolean ret = TRUE;

#if ((!defined (_MX233)) && (!defined (_MX28)) && (!defined (_MX50)))
    struct mxcfb_color_key colorKey;
    struct fb_var_screeninfo fbVar;

    if (ioctl(priv->fd_fb0, FBIOGET_VSCREENINFO, &fbVar) < 0) {
        g_print("get vscreen info failed.\n");
        ret = FALSE;
    }

    if (fbVar.bits_per_pixel == 16) {
        *colorSrc = RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE));
        GST_DEBUG("%08X:%08X:%8X",RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE),
            RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE)),
            RGB565TOCOLORKEY(RGB888TORGB565(RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE))));
        colorKey.color_key = RGB565TOCOLORKEY(*colorSrc);
    }
    else if ((fbVar.bits_per_pixel == 32) || (fbVar.bits_per_pixel == 24)) {
        *colorSrc = RGB888(COLORKEY_RED, COLORKEY_GREEN, COLORKEY_BLUE);
        colorKey.color_key = *colorSrc;

    }
    GST_DEBUG("fbVar.bits_per_pixel:%d",fbVar.bits_per_pixel);

    GST_INFO("color source:0x%08x,set color key:0x%08x.",*colorSrc,colorKey.color_key);

    colorKey.enable = enable;
    if (ioctl(priv->fd_fb0, MXCFB_SET_CLR_KEY, &colorKey) < 0) {
        g_print("set color key failed.\n");
        ret = FALSE;
    }
#endif
    return ret;
}

/*=============================================================================
FUNCTION:           fb0_open

DESCRIPTION:        This function open the fb0 device return the device pointer.

ARGUMENTS PASSED:

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean fb0_open(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    gboolean retval = TRUE;
    gchar fb_device[100] = FB_DEVICE;

    if ((priv->fd_fb0 = open(fb_device, O_RDWR, 0)) < 0) {
	    g_print("Unable to open %s %d\n", fb_device, priv->fd_fb0);
	    priv->fd_fb0 = 0;
	    retval = FALSE;
    }

    return retval;
}

/*=============================================================================
FUNCTION:           fb0_get_bounds

DESCRIPTION:        This function get the maximum resolution of fb0 device.

ARGUMENTS PASSED:

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean fb0_get_bounds(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct fb_var_screeninfo fb_var;
    gint ret;

    ret = ioctl(priv->fd_fb0, FBIOGET_VSCREENINFO, &fb_var);
    if (ret < 0) {
        g_print("Unable to get resolution value\n");
        v4l->fullscreen_width = 1024;
        v4l->fullscreen_height = 768;
        return FALSE;
    }
    v4l->fullscreen_width = fb_var.xres;
    v4l->fullscreen_height = fb_var.yres;
    g_print("full screen size: %dx%d\n",
	v4l->fullscreen_width, v4l->fullscreen_height);

    return TRUE;

}

/*=============================================================================
FUNCTION:           fb0_close

DESCRIPTION:        This function close the fb0 device.

ARGUMENTS PASSED:
        gstXInfo       -   pointer to GstXInfo

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/
static gboolean fb0_close(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    gboolean retval = TRUE;

    if (priv->fd_fb0) {
        close(priv->fd_fb0);
        priv->fd_fb0 = 0;
    }

    return retval;
}

static gboolean fb0_setup(MFW_GST_V4LSINK_INFO_T *v4l,
			  struct v4l2_crop *crop)
{
	/*
	 * nothing to do, the v4l2 output driver does fb setup
	 */
	return TRUE;
}

#ifdef ENABLE_LOC_ALPHA
/*
 *
 *   The Local alpha will set the alpha value in fb2 which only be
 * supported in MX37 and MX51 platform.
 *
 */
static gboolean local_alpha_open(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    gchar fb_device2[100] = "/dev/fb2";

    if ((priv->fd_lalpfb = open(fb_device2, O_RDWR, 0)) < 0) {
        g_print("Unable to open %s %d\n", fb_device2, priv->fd_lalpfb);
        priv->fd_lalpfb = 0;
        return FALSE;
    }
    return TRUE;
}

static void local_alpha_close(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;

    if (priv->fd_lalpfb) {
        close(priv->fd_lalpfb);
        priv->fd_lalpfb = 0;
    }
}


static void disable_local_alpha(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct mxcfb_loc_alpha * lalp = &priv->lalpha;
    gint lalp_buf_size = v4l->crop.c.width * v4l->crop.c.height;
    gint ret;

    if (lalp->enable) {
        if (priv->lalp_buf_vaddr[0]){
            munmap(priv->lalp_buf_vaddr[0], lalp_buf_size);
            priv->lalp_buf_vaddr[0] = 0;
        }
        if (priv->lalp_buf_vaddr[1]){
            munmap(priv->lalp_buf_vaddr[1], lalp_buf_size);
            priv->lalp_buf_vaddr[1] = 0;
        }


        g_print("emit %p %p\n",
		priv->lalp_buf_vaddr[0], priv->lalp_buf_vaddr[1]);

        g_signal_emit (G_OBJECT (v4l),
        mfw_gst_v4lsink_signals[SIGNAL_LOCALPHA_BUFFER_READY], 0,
		       priv->lalp_buf_vaddr[0]);

        lalp->enable = 0;

        if (priv->fd_lalpfb) {
            ret = ioctl(priv->fd_lalpfb, MXCFB_SET_LOC_ALPHA, lalp);
            if (ret<0) {
                g_print("Error on MXCFB_SET_LOC_ALPHA ret = %d\n", ret);
                return;
            }
        }
    }
}
static void disable_global_alpha(mfw_gst_fb_priv_t *priv)
{
    int ret;
    struct mxcfb_gbl_alpha galpha;

    galpha.alpha = 0;
    galpha.enable = 1;

    ret = ioctl(priv->fd_lalpfb, MXCFB_SET_GBL_ALPHA, &galpha);

    if (ret<0){
        g_print("set global alpha  failed ret=%d.\n");
    }
}


static void enable_local_alpha(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct mxcfb_loc_alpha * lalp = &priv->lalpha;
    struct fb_var_screeninfo scrinfo;
    gint lalp_buf_size = v4l->crop.c.width * v4l->crop.c.height;
    gint ret;

    if ((lalp->enable==0) && (lalp_buf_size)) {

	if (ioctl(priv->fd_lalpfb, FBIOGET_VSCREENINFO,
                  &scrinfo) < 0) {
                printf("Get var of fb2 failed\n");
                return;
        }

        scrinfo.xres = v4l->crop.c.width;
        scrinfo.yres = v4l->crop.c.height;
        scrinfo.xres_virtual = v4l->crop.c.width;
        scrinfo.yres_virtual = v4l->crop.c.height*2;
        if (ioctl(priv->fd_lalpfb, FBIOPUT_VSCREENINFO,
                  &scrinfo) < 0) {
                printf("Put var of fb2 failed\n");
                return;
        }


        lalp->enable = 1;
        lalp->alpha_phy_addr0 = 0;
        lalp->alpha_phy_addr1 = 0;
        if (priv->fd_lalpfb){
            ret = ioctl(priv->fd_lalpfb, MXCFB_SET_LOC_ALPHA, lalp);
            if (ret<0){
                g_print("Error on MXCFB_SET_LOC_ALPHA ret = %d\n", ret);
                return;
            }
            priv->lalp_buf_vaddr[0] = mmap(0, lalp_buf_size, PROT_READ|PROT_WRITE,
                                        MAP_SHARED,priv->fd_lalpfb, lalp->alpha_phy_addr0);



            priv->lalp_buf_vaddr[1] = mmap(0, lalp_buf_size, PROT_READ|PROT_WRITE,
                                        MAP_SHARED,priv->fd_lalpfb, lalp->alpha_phy_addr1);

            if (priv->lalp_buf_vaddr[0]==-1){
                g_print(RED_STR("can not mmap for 0 %p size %d\n", lalp->alpha_phy_addr0, lalp_buf_size));
                priv->lalp_buf_vaddr[0] = 0;
            }

            if (priv->lalp_buf_vaddr[1]==-1){
                priv->lalp_buf_vaddr[1] = 0;
                g_print(RED_STR("can not mmap for 1 %p\n", lalp->alpha_phy_addr1, lalp_buf_size));
            }
        }

        g_print("emit %p %p\n", priv->lalp_buf_vaddr[0], priv->lalp_buf_vaddr[1]);
        //priv->lalp_buf_cidx = 0;
        g_signal_emit (G_OBJECT (v4l),
        mfw_gst_v4lsink_signals[SIGNAL_LOCALPHA_BUFFER_READY], 0, priv->lalp_buf_vaddr[0]);
    }
}

static void set_local_alpha(MFW_GST_V4LSINK_INFO_T *v4l, gint alpha)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct mxcfb_loc_alpha * lalp = &priv->lalpha;
    gint lalp_buf_size = v4l->crop.c.width*v4l->crop.c.height;
    char * alphabuf = priv->lalp_buf_vaddr[priv->lalp_buf_cidx];
    gint ret;

    if (priv->fd_lalpfb==0)
        return;
    g_print("set local alpha %d \n", alpha);

    if (alphabuf){
        if (alpha>=0){
            memset(alphabuf, alpha, lalp_buf_size);
        }
        ret = ioctl(priv->fd_lalpfb, MXCFB_SET_LOC_ALP_BUF,
		    (priv->lalp_buf_cidx ? (&lalp->alpha_phy_addr1) :
		     ((&lalp->alpha_phy_addr0))));
        if (ret<0) {
            g_print("Error on MXCFB_SET_LOC_ALP_BUF ret = %d\n", ret);
        }

        if (priv->lalp_buf_cidx)
            priv->lalp_buf_cidx = 0;
        else
            priv->lalp_buf_cidx = 1;
        g_signal_emit (G_OBJECT (v4l),
		       mfw_gst_v4lsink_signals[SIGNAL_LOCALPHA_BUFFER_READY],
		       0, priv->lalp_buf_vaddr[priv->lalp_buf_cidx]);
    }
}

static void set_global_alpha(MFW_GST_V4LSINK_INFO_T *v4l, gint alpha)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct mxcfb_gbl_alpha galpha;
    int ret;

    galpha.alpha = alpha;
    galpha.enable = 1;

    ret = ioctl(priv->fd_lalpfb, MXCFB_SET_GBL_ALPHA, &galpha);

    if (ret<0){
        g_print("set global alpha  failed ret=%d.\n");
    }
}


static void set_alpha(MFW_GST_V4LSINK_INFO_T *v4l)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;

    if (priv->fd_lalpfb==0)
        return;
    if (v4l->alpha_enable & ALPHA_LOCAL)
        set_local_alpha(v4l, v4l->alpha);
    else if (v4l->alpha_enable & ALPHA_GLOBAL)
        set_global_alpha(v4l, v4l->alpha);
}

static void set_alpha_mask(MFW_GST_V4LSINK_INFO_T *v4l, gint newmask)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    struct mxcfb_loc_alpha * lalp = &priv->lalpha;
    gint ret;

    g_mutex_lock(v4l->flow_lock);

    if (v4l->alpha_enable==newmask){
        g_mutex_unlock(v4l->flow_lock);
        return;
    }

    g_print("switch to mode %d\n", newmask);

    v4l->alpha_enable=newmask;
    if (v4l->stream_on){
        v4l->setpara |= PARAM_SET_V4L;
    }
    g_mutex_unlock(v4l->flow_lock);
}

#endif /* ENABLE_LOC_ALPHA */


#ifdef ENABLE_TVOUT

#if defined (_MX31) || defined (_MX35)

#define MX31_MX35_PAL_MODE      "U:640x480p-50\n"
#define MX31_MX35_NTSC_MODE     "U:640x480p-60\n"
#define MX31_MX35_LCD_MODE      "U:480x640p-67\n"

/*=============================================================================
FUNCTION:           tv_open

DESCRIPTION:        This function open the TV out related device.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint out;
  gchar *mode;
  FILE *pfb0_mode;

  pfb0_mode = fopen ("/sys/class/graphics/fb0/mode", "w");
  if (pfb0_mode < 0) {
    g_print ("No /sys/class/graphics/fb0/mode device to open\n");
    return FALSE;
  }

  GST_DEBUG ("pfb0_mode : %x", pfb0_mode);
  if (v4l_info->tv_mode == PAL) {
    mode = MX31_MX35_PAL_MODE;
    fwrite (mode, 1, strlen (mode), pfb0_mode);
  } else if (v4l_info->tv_mode == NTSC) {
    mode = MX31_MX35_NTSC_MODE;
    fwrite (mode, 1, strlen (mode), pfb0_mode);
  } else {
    GST_ERROR ("Wrong TV mode.");
    fclose (pfb0_mode);
    return FALSE;
  }
  fflush (pfb0_mode);
  fclose (pfb0_mode);

  out = 3;
  ioctl (v4l_info->v4l_id, VIDIOC_S_OUTPUT, &out);

  return TRUE;

}

/*=============================================================================
FUNCTION:           tv_set_lcdmode

DESCRIPTION:        This function set the lcd mode.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_set_lcdmode (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gchar *mode;
  FILE *pfb0_mode;

  pfb0_mode = fopen ("/sys/class/graphics/fb0/mode", "w");
  if (pfb0_mode < 0) {
    g_print ("No /sys/class/graphics/fb0/mode device to open\n");
    return FALSE;
  }
  mode = MX31_MX35_LCD_MODE;
  fwrite (mode, 1, strlen (mode), pfb0_mode);
  fflush (pfb0_mode);
  fclose (pfb0_mode);

  return TRUE;

}

static gboolean tv_set_blank (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return FALSE;
}

/*=============================================================================
FUNCTION:           tv_close

DESCRIPTION:        This function close the TV related device.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint out = 3;
  gchar *mode;
  FILE *pfb0_mode;

  if ((v4l_info->tv_mode == PAL) || (v4l_info->tv_mode == NTSC)) {
    v4l_info->tv_mode = NV_MODE;
    tv_set_lcdmode (v4l_info);
  }

  ioctl (v4l_info->v4l_id, VIDIOC_S_OUTPUT, &out);
  return TRUE;

}

#elif defined (_MX37) || defined (_MX51)

#define MX37_MX51_PAL_MODE      "U:720x576i-50\n"
#define MX37_MX51_NTSC_MODE     "U:720x480i-60\n"
#define MX37_MX51_PAL_MODE      "U:720x576i-50\n"
#define MX37_MX51_720P_MODE      "U:1280x720p-60\n"

/*=============================================================================
FUNCTION:           tv_open

DESCRIPTION:        This function open the TV out related device.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint out;
  gchar *mode;
  FILE *pfb1_mode;

  pfb1_mode = fopen ("/sys/class/graphics/fb1/mode", "w");
  if (pfb1_mode < 0) {
    g_print ("No /sys/class/graphics/fb1/mode device to open\n");
    return FALSE;
  }

  if (v4l_info->tv_mode == PAL) {
    mode = MX37_MX51_PAL_MODE;
    fwrite (mode, 1, strlen (mode), pfb1_mode);
  } else if (v4l_info->tv_mode == NTSC) {
    mode = MX37_MX51_NTSC_MODE;
    fwrite (mode, 1, strlen (mode), pfb1_mode);
  } else if (v4l_info->tv_mode == DISP_720P) {
    mode = MX37_MX51_720P_MODE;
    fwrite (mode, 1, strlen (mode), pfb1_mode);
  } else {
    GST_ERROR ("Wrong TV mode.");
    fclose (pfb1_mode);
    return FALSE;
  }
  fflush (pfb1_mode);
  fclose (pfb1_mode);

  out = 5;
  ioctl (v4l_info->v4l_id, VIDIOC_S_OUTPUT, &out);

  return TRUE;
}

static gboolean tv_set_lcdmode (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return FALSE;
}

/*=============================================================================
FUNCTION:           tv_setblank

DESCRIPTION:        This function set the TV-out to blank.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_set_blank (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  FILE *pfb1_blank;
  gchar *blank = "4\n";

  pfb1_blank = fopen ("/sys/class/graphics/fb1/blank", "w");
  if (pfb1_blank == NULL) {
    GST_DEBUG ("No /sys/class/graphics/fb1/blank device to open\n");
    return FALSE;
  }

  fwrite (blank, 1, strlen (blank), pfb1_blank);
  fflush (pfb1_blank);
  fclose (pfb1_blank);

  return TRUE;

}

/*=============================================================================
FUNCTION:           tv_close

DESCRIPTION:        This function close the TV related device.

ARGUMENTS PASSED:
        v4l_info       -   pointer to MFW_GST_V4LSINK_INFO_T

RETURN VALUE:       TRUE/FALSE (SUCCESS/FAIL)

PRE-CONDITIONS:     None
POST-CONDITIONS:    None
IMPORTANT NOTES:    None
=============================================================================*/

static gboolean tv_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  gint out = 3;

  // tv_setblank (v4l_info);

  v4l_info->tv_mode = NV_MODE;
  ioctl (v4l_info->v4l_id, VIDIOC_S_OUTPUT, &out);

  return TRUE;

}

#elif defined (_MX27)

static gboolean tv_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;

    struct v4l2_output_dev odev = {
	    .disp_num = 2,
	    .id_len = 11,
	    .id = "DISP3 TVOUT"
    };

    priv->fd_tvout = open ("/dev/fb/1", O_RDWR);
    if (priv->fd_tvout < 0) {
	    GST_ERROR ("Unable to open /dev/fb/1");
	    return FALSE;
    }

    if (ioctl (v4l_info->v4l_id, VIDIOC_PUT_OUTPUT, &odev) < 0)
	    GST_ERROR ("TV-OUT ioctl VIDIOC_PUT_OUTPUT failed!");

    return TRUE;

}

static gboolean tv_set_lcdmode (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return FALSE;
}

static gboolean tv_set_blank (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return FALSE;
}

static gboolean tv_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
    mfw_gst_fb_priv_t *priv = v4l->fb_priv;
    if (priv->fd_tvout > 0) {
      struct v4l2_output_dev odev = {
	      .disp_num = 2,
	      .id_len = 11,
	      .id = "DISP3 TVOUT"
      };

      if (ioctl (v4l_info->v4l_id, VIDIOC_GET_OUTPUT, &odev) < 0)
	      GST_ERROR ("TV-OUT ioctl VIDIOC_GET_OUTPUT failed!");

      close (v4l_info->fd_tvout);
      v4l_info->fd_tvout = 0;
    }
    return TRUE;
}

#else

static gboolean tv_open (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return TRUE;
}

static gboolean tv_close (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return TRUE;
}

static gboolean tv_set_lcdmode (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return TRUE;
}

static gboolean tv_set_blank (MFW_GST_V4LSINK_INFO_T * v4l_info)
{
  return TRUE;
}

#endif /* archs */

#endif /* ENABLE_TVOUT */


struct mfw_gst_v4lsink_fb_ops mfw_gst_fbdev_ops = {
	.init                 = mfw_gst_fb_init,
	.release              = mfw_gst_fb_release,
	.open                 = fb0_open,
	.close                = fb0_close,
	.set_global_alpha     = fb0_set_global_alpha,
	.set_colorkey         = fb0_set_colorkey,
	.get_bounds           = fb0_get_bounds,
	.setup                = fb0_setup,

#ifdef ENABLE_LOC_ALPHA
	.local_alpha_open     = local_alpha_open,
	.local_alpha_close    = local_alpha_close,
	.local_alpha_enable   = enable_local_alpha,
	.local_alpha_disable  = disable_local_alpha,
	.set_alpha            = set_alpha,
	.set_alpha_mask       = set_alpha_mask,
#endif
#ifdef ENABLE_TVOUT
	.tv_open              = tv_open,
	.tv_close             = tv_close,
	.tv_set_lcdmode       = tv_set_lcdmode,
	.tv_set_blank         = tv_set_blank,
#endif
};

