/** @file period_align.cpp
 *
 * @brief This demonstrates the basic concepts for writing an external client
 * for period size alignment between server and client.
 *
 * For the sake of example, a port_converter_pair_t is allocated in
 * main(), passed to inprocess() as an argument, then freed
 * in signal_handler().
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <memory.h>
#include <jack/jack.h>
#include <jack/format_converter.h>


typedef struct {
	/* Client private structure/class should have first variable as JackFormatConverterHandle_t
	 * This space will be used by jack format converter
	 */
	JackFormatConverterHandle_t* handle;
	JackBufferConverter_t* buffer_converter;
	JackPortConverter_t* input_port_converter;
	JackPortConverter_t* output_port_converter;
} port_converter_pair_t;

bool is_signaled = false;

static void
signal_handler (int sig)
{
	is_signaled = true;
	fprintf (stderr, "Recieved signal\n");
}

/**
 * Called in the realtime thread on every process cycle.  The entry
 * point name was passed to jack_set_process_callback() from
 * main().
 *
 * @return 0 if successful
 */
int
inprocess (jack_nframes_t nframes, void *arg)
{
	port_converter_pair_t *pp = (port_converter_pair_t *)arg;
	int32_t *in_buffer;
	int32_t *out_buffer;

	in_buffer = (int32_t *) jack_port_convert_get(pp->input_port_converter, nframes);
	out_buffer = (int32_t *) jack_port_convert_get(pp->output_port_converter, nframes);

		/* This memcpy needs to be replaced with the actual processing */
		memcpy (out_buffer, in_buffer, sizeof(int32_t) * nframes);

	jack_port_convert_set(pp->output_port_converter, out_buffer, nframes);

	return 0;
}

int
main (int argc, char *argv[])
{
	const char **ports;
	const char *client_name;
	const char *server_name = NULL;
	jack_options_t options = JackNullOption;
	jack_status_t status;
	jack_port_t *input_port;
	jack_port_t *output_port;
	port_converter_pair_t* pp;
	jack_client_t *client;

	if (argc >= 2) {		/* client name specified? */
		client_name = argv[1];
		if (argc >= 3) {	/* server name specified? */
			server_name = argv[2];
			int my_option = JackNullOption | JackServerName;
			options = (jack_options_t)my_option;
		}
	} else {			/* use basename of argv[0] */
		client_name = strrchr(argv[0], '/');
		if (client_name == 0) {
			client_name = argv[0];
		} else {
			client_name++;
		}
	}

	/* open a client connection to the JACK server */

	client = jack_client_open (client_name, options, &status, server_name);
	if (client == NULL) {
		fprintf (stderr, "jack_client_open() failed, "
			 "status = 0x%2.0x\n", status);
		if (status & JackServerFailed) {
			fprintf (stderr, "Unable to connect to JACK server\n");
		}
		exit (1);
	}
	if (status & JackServerStarted) {
		fprintf (stderr, "JACK server started\n");
	}
	if (status & JackNameNotUnique) {
		client_name = jack_get_client_name(client);
		fprintf (stderr, "unique name `%s' assigned\n", client_name);
	}


	pp = (port_converter_pair_t *) malloc (sizeof (port_converter_pair_t));

	if (pp == NULL)
		return 1;		/* heap exhausted */

	const uint32_t period_time = 16; /* ms */
	pp->buffer_converter = jack_buffer_create_convert(client, inprocess, pp,
												 (period_time * jack_get_sample_rate(client))/1000);


	/* create a pair of ports */
	input_port = jack_port_register (client, "input",
						 JACK_DEFAULT_AUDIO_TYPE,
						 JackPortIsInput, 0);
	output_port = jack_port_register (client, "output",
						  JACK_DEFAULT_AUDIO_TYPE,
						  JackPortIsOutput, 0);

	pp->input_port_converter = jack_port_create_convert(input_port,
									  JACK_PORT_CONV_FMT_INT32,
									  false,
									  pp->buffer_converter);

	pp->output_port_converter = jack_port_create_convert(output_port,
									  JACK_PORT_CONV_FMT_INT32,
									   false,
									   pp->buffer_converter);

	/* join the process() cycle */
	jack_activate (client);

	ports = jack_get_ports (client, NULL, NULL,
				JackPortIsPhysical|JackPortIsOutput);
	if (ports == NULL) {
		fprintf(stderr, "no physical capture ports\n");
		return 1;		/* terminate client */
	}


	if (jack_connect (client, ports[0], jack_port_name (input_port))) {
		fprintf (stderr, "cannot connect input ports\n");
	}

	jack_free (ports);

	ports = jack_get_ports (client, NULL, NULL,
				JackPortIsPhysical|JackPortIsInput);
	if (ports == NULL) {
		fprintf(stderr, "no physical playback ports\n");
		return 1;		/* terminate client */
	}

	if (jack_connect (client, jack_port_name (output_port), ports[0])) {
		fprintf (stderr, "cannot connect output ports\n");
	}
	jack_free (ports);

	/* install a signal handler to properly quits jack client */
#ifdef WIN32
	signal(SIGINT, signal_handler);
	signal(SIGABRT, signal_handler);
	signal(SIGTERM, signal_handler);
#else
	signal(SIGQUIT, signal_handler);
	signal(SIGTERM, signal_handler);
	signal(SIGHUP, signal_handler);
	signal(SIGINT, signal_handler);
#endif

	/* keep running until the Ctrl+C */

	while (1 && !is_signaled) {
	#ifdef WIN32
		Sleep(1000);
	#else
		sleep (1);
	#endif
	}

	/* Free resources only after deactivating to avoid process callback being called */
	jack_deactivate(client);
	jack_port_destroy_convert(pp->input_port_converter);
	jack_port_destroy_convert(pp->output_port_converter);
	jack_buffer_destroy_convert(pp->buffer_converter);
	jack_client_close(client);
	free (pp);
	pp = NULL;

	exit (0);

}
