/*
Copyright (C) 2001 Paul Davis
Copyright (C) 2004 Grame

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef __JackAlsaDriver__
#define __JackAlsaDriver__

#include "JackAudioDriver.h"
#include "JackThreadedDriver.h"
#include "JackTime.h"
#include "alsa_driver.h"
#include "JackAlsaOptionParser.h"

namespace Jack
{

/*!
\brief The ALSA driver.
*/

/**
 * @defgroup AlsaBackendStartupAndRuntimeBehavior Alsa backend startup and runtime behavior
 * \page AlsaBackendStartupAndRuntimeBehavior Alsa backend startup and runtime behavior
 *
 * By default, the JACK daemon will always open and start all configured audio devices.
 * To avoid this, configure the jack startup arguments --playback-state and --capture-state for alsa backend. \n
 * Define the startup and the state of the idle (not connected to any client) devices at runtime using these options.
 * The startup and runtime idle device behavior is : separated and the devices are space separated. \n
 * The states are defined as 0 for keeping the device closed, 1 for opening and preparing it and 2 for starting the device.
 * e.g.
 * @code
 *  -P "deva-media" -o "4" --playback-state "0:0"
 *  -C "deva-btwbrx deva-btnbrx" -i "1 1 10 28 1" --capture-state "0:0 1:1"
 * @endcode
 * \n
 * @startuml
 * title Startup
 * state Closed
 * state Prepared
 * state Running
 * state startup <<choice>>
 * 
 * [*]-->Running : state is not configured
 * 
 * [*]-->startup : state is configured
 * startup-->Closed : state==0-Closed
 * startup-->Prepared : state==1-Prepare
 * startup-->Running : state==2-Running
 * @enduml
 * \n
 * @startuml
 * title Reload backend
 * state Closed
 * state Prepared
 * state Running
 * state preparedReloading <<choice>>
 * state disconnected <<choice>>
 * state configured <<choice>>
 * 
 * [*]-->preparedReloading : reload
 * preparedReloading-->Running : connected
 * preparedReloading-->disconnected : disconnected
 * 
 * disconnected-->Prepared : state is not configured
 * disconnected-->configured : state is configured
 * 
 * configured-->Closed : state==0-Closed
 * configured-->Prepared : state==1-Prepared
 * configured-->Running : state==2-Running
 * @enduml
 * This feature requires specifying input and output channels parameters using -i and -o options
 * because the JACK daemon cannot automatically detect them without opening the device.
 * The order of devices, channels and states per device are important. \n
 *
 * Jack APIs like @ref jack_client_reload_master() or @ref jack_client_audio_backend_command() can be used
 * to stop, close, open and start the devices at runtime on demand. \n
 * @ref jack_client_reload_master() will be deprecated soon as the same result can be obtained by using
 * @ref jack_client_audio_backend_command() with @ref JackBackendReload as the argument. \n
 *
 * But keep in mind that all audio devices used by this JACK daemon instance will be restarted.
 * This is required to start all audio devices synchronously. In case there is an ongoing audio streaming
 * a short interruption could be recognized when triggering the backend command.
 *
 * @ingroup AlsaBackendStartupAndRuntimeBehavior
 * @{
 */

class JackAlsaDriver : public JackAudioDriver
{

    private:
        void UpdateLatencies();

#ifdef UNITTEST
    public:
#endif
        enum DriverMode {
            Init,
            Runtime,
            WaitForRecovery,
            Shutdown,
        };

        jack_driver_t* fDriver;
        bool isAnyDeviceClosedOnStartup();

    public:

        JackAlsaDriver(const char* name, const char* alias, JackLockedEngine* engine, JackSynchro* table)
		: JackAudioDriver(name, alias, engine, table),fDriver(NULL)
        {}
        virtual ~JackAlsaDriver()
        {}

        int Open(alsa_driver_info_t info);

        int Close();
        int Attach();
        int Detach();

        int Start();
        int Stop();
        int ReloadDevices(jack_backend_command_id_t cmd);
        int HandleBackendCommand(jack_backend_command_t cmd);

        int Read();
        int Write();

        // BufferSize can be changed
        bool IsFixedBufferSize()
        {
            return false;
        }

        int SetBufferSize(jack_nframes_t buffer_size);

        void ReadInputAux(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nread);
        void MonitorInputAux();
        void ClearOutputAux();
        void WriteOutputAux(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nwritten);
        void SetTimetAux(jack_time_t time);

        int UpdateDriverTargetState(DriverMode mode, jack_backend_command_id_t cmd);
        void ResetDriverTargetState();
        int TargetState(DriverMode mode, int connections_count, jack_backend_command_id_t cmd, const target_state_t& target_state);

        void ResetDriver(alsa_driver_status_t status);

        // JACK API emulation for the midi driver
        int is_realtime() const;
        int create_thread(pthread_t *thread, int prio, int rt, void *(*start_func)(void*), void *arg);

        jack_port_id_t port_register(const char *port_name, const char *port_type, unsigned long flags, unsigned long buffer_size);
        int port_unregister(jack_port_id_t port_index);
        void* port_get_buffer(int port, jack_nframes_t nframes);
        int port_set_alias(int port, const char* name);

        jack_nframes_t get_sample_rate() const;
        jack_nframes_t frame_time() const;
        jack_nframes_t last_frame_time() const;
};

} // end of namespace

#endif
