 /*
Copyright (C) 2004-2008 Grame

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#include "JackPosixProcessSync.h"
#include "JackError.h"

namespace Jack
{

void JackPosixProcessSync::Signal()
{
    int res = pthread_cond_signal(&fCond);
    if (res != 0) {
        jack_error("JackPosixProcessSync::Signal error err = %s", strerror(res));
    }
}

// TO DO : check thread consistency?
void JackPosixProcessSync::LockedSignal()
{
    int res = pthread_mutex_lock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignal error err = %s", strerror(res));
    }
    res = pthread_cond_signal(&fCond);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignal error err = %s", strerror(res));
    }
    res = pthread_mutex_unlock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignal error err = %s", strerror(res));
    }
}

void JackPosixProcessSync::SignalAll()
{
    int res = pthread_cond_broadcast(&fCond);
    if (res != 0) {
        jack_error("JackPosixProcessSync::SignalAll error err = %s", strerror(res));
    }
}

// TO DO : check thread consistency?
void JackPosixProcessSync::LockedSignalAll()
{
    int res = pthread_mutex_lock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignalAll error err = %s", strerror(res));
    }
    res = pthread_cond_broadcast(&fCond);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignalAll error err = %s", strerror(res));
    }
    res = pthread_mutex_unlock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedSignalAll error err = %s", strerror(res));
    }
}

void JackPosixProcessSync::Wait()
{
    ThrowIf(!pthread_equal(pthread_self(), fOwner), JackException("JackPosixProcessSync::Wait: a thread has to have locked a mutex before it can wait"));
    fOwner = 0;

    int res = pthread_cond_wait(&fCond, &fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::Wait error err = %s", strerror(res));
    } else {
        fOwner = pthread_self();
    }
}

// TO DO : check thread consistency?
void JackPosixProcessSync::LockedWait()
{
    int res;
    res = pthread_mutex_lock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedWait error err = %s", strerror(res));
    }
    if ((res = pthread_cond_wait(&fCond, &fMutex)) != 0) {
        jack_error("JackPosixProcessSync::LockedWait error err = %s", strerror(res));
    }
    res = pthread_mutex_unlock(&fMutex);
    if (res != 0) {
        jack_error("JackPosixProcessSync::LockedWait error err = %s", strerror(res));
    }
}

float JackPosixProcessSync::TimeDiffUs(const struct timespec * const start, const struct timespec * const stop)
{
    struct timespec temp;
    if ((stop->tv_nsec-start->tv_nsec)<0) {
            temp.tv_sec = (stop->tv_sec-start->tv_sec)-1;
            temp.tv_nsec = 1000000000+stop->tv_nsec-start->tv_nsec;
    } else {
            temp.tv_sec = stop->tv_sec-start->tv_sec;
            temp.tv_nsec = stop->tv_nsec-start->tv_nsec;
    }
    return (temp.tv_nsec/1000) + ((int64_t)temp.tv_sec*1000*1000);
}

bool JackPosixProcessSync::TimedWait(long usec)
{
    ThrowIf(!pthread_equal(pthread_self(), fOwner), JackException("JackPosixProcessSync::TimedWait: a thread has to have locked a mutex before it can wait"));
    fOwner = 0;

    struct timespec T0,T1,time;
    int res;

    jack_log("JackPosixProcessSync::TimedWait time out = %ld", usec);

    clock_gettime(CLOCK_MONOTONIC, &T0);

    unsigned long int next_date_nsec = T0.tv_nsec + (usec*1000);
    time.tv_sec = T0.tv_sec + (next_date_nsec / 1000000000);
    time.tv_nsec = (next_date_nsec % 1000000000);

    res = pthread_cond_timedwait(&fCond, &fMutex, &time);

    clock_gettime(CLOCK_MONOTONIC, &T1);
    if (res != 0) {
        jack_error("JackPosixProcessSync::TimedWait error usec = %ld err = %s wait time = %5.1lf us",
                   usec, strerror(res), TimeDiffUs(&T0, &T1));
    } else {
        fOwner = pthread_self();
    }

    jack_log("JackPosixProcessSync::TimedWait finished delta = %5.1lf us", TimeDiffUs(&T0, &T1));

    return (res == 0);
}

// TO DO : check thread consistency?
bool JackPosixProcessSync::LockedTimedWait(long usec)
{
    struct timespec T0,T1,time;
    int res1, res2;

    res1 = pthread_mutex_lock(&fMutex);
    if (res1 != 0) {
        jack_error("JackPosixProcessSync::LockedTimedWait error err = %s", strerror(res1));
    }

    jack_log("JackPosixProcessSync::LockedTimedWait time out = %ld", usec);

    clock_gettime(CLOCK_MONOTONIC, &T0);

    unsigned long int next_date_nsec = T0.tv_nsec + (usec*1000);
    time.tv_sec = T0.tv_sec + (next_date_nsec / 1000000000);
    time.tv_nsec = (next_date_nsec % 1000000000);

    res2 = pthread_cond_timedwait(&fCond, &fMutex, &time);

    clock_gettime(CLOCK_MONOTONIC, &T1);
    if (res2 != 0) {
        jack_error("JackPosixProcessSync::LockedTimedWait error usec = %ld err = %s wait time = %5.1lf us",
                   usec, strerror(res2), TimeDiffUs(&T0, &T1));
    }

    res1 = pthread_mutex_unlock(&fMutex);
    if (res1 != 0) {
        jack_error("JackPosixProcessSync::LockedTimedWait error err = %s", usec, strerror(res1));
    }

    jack_log("JackPosixProcessSync::LockedTimedWait finished delta = %5.1lf us", TimeDiffUs(&T0, &T1));

    return (res2 == 0);
}


} // end of namespace

