/*
Copyright (C) 2001 Paul Davis
Copyright (C) 2004 Grame

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#define __STDC_FORMAT_MACROS   // For inttypes.h to work in C++

#include <iostream>
#include <math.h>
#include <stdio.h>
#include <memory.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <string.h>
#include <sstream>
#include <fstream>
#include <algorithm>
#include <cctype>
#include <vector>

#include "JackAlsaDriver.h"
#include "JackEngineControl.h"
#include "JackClientControl.h"
#include "JackPort.h"
#include "JackGraphManager.h"
#include "JackLockedEngine.h"
#ifdef __ANDROID__
#include "JackAndroidThread.h"
#else
#include "JackPosixThread.h"
#endif
#include "JackCompilerDeps.h"
#include "JackServerGlobals.h"

#ifndef ARRAY_SIZE
#define ARRAY_SIZE(a) (sizeof(a) / sizeof(a[0]))
#endif

static struct jack_constraint_enum_str_descriptor midi_constraint_descr_array[] =
{
    { "none", "no MIDI driver" },
    { "seq", "ALSA Sequencer driver" },
    { "raw", "ALSA RawMIDI driver" },
    { 0 }
};

static struct jack_constraint_enum_char_descriptor dither_constraint_descr_array[] =
{
    { 'n', "none" },
    { 'r', "rectangular" },
    { 's', "shaped" },
    { 't', "triangular" },
    { 0 }
};

namespace Jack
{

int JackAlsaDriver::SetBufferSize(jack_nframes_t buffer_size)
{
    jack_log("JackAlsaDriver::SetBufferSize %u", buffer_size);
    alsa_driver_status_t res = alsa_driver_reset_parameters((alsa_driver_t *)fDriver, buffer_size,
                                           ((alsa_driver_t *)fDriver)->user_nperiods,
                                           ((alsa_driver_t *)fDriver)->frame_rate);

    if (res == ALSA_DRIVER_OK) { // update fEngineControl and fGraphManager
        JackAudioDriver::SetBufferSize(buffer_size);  // Generic change, never fails
        // ALSA specific
        UpdateLatencies();
    } else {
        // Restore old values
        alsa_driver_reset_parameters((alsa_driver_t *)fDriver, fEngineControl->fBufferSize,
                                     ((alsa_driver_t *)fDriver)->user_nperiods,
                                     ((alsa_driver_t *)fDriver)->frame_rate);
    }

    return AUDIO_DRIVER_OK;
}

void JackAlsaDriver::UpdateLatencies()
{
    jack_latency_range_t range;
    alsa_driver_t* alsa_driver = (alsa_driver_t*)fDriver;

    for (int i = 0; i < fCaptureChannels; i++) {
        range.min = range.max = alsa_driver->frames_per_cycle + alsa_driver->capture_frame_latency;
        fGraphManager->GetPort(fCapturePortList[i])->SetLatencyRange(JackCaptureLatency, &range);
    }

    for (int i = 0; i < fPlaybackChannels; i++) {
        // Add one buffer more latency if "async" mode is used...
        range.min = range.max = (alsa_driver->frames_per_cycle * (alsa_driver->user_nperiods - 1)) +
                         ((fEngineControl->fSyncMode) ? 0 : fEngineControl->fBufferSize) + alsa_driver->playback_frame_latency;
        fGraphManager->GetPort(fPlaybackPortList[i])->SetLatencyRange(JackPlaybackLatency, &range);
        // Monitor port
        if (fWithMonitorPorts) {
            range.min = range.max = alsa_driver->frames_per_cycle;
            fGraphManager->GetPort(fMonitorPortList[i])->SetLatencyRange(JackCaptureLatency, &range);
        }
    }
}

int JackAlsaDriver::Attach()
{
    JackPort* port;
    jack_port_id_t port_id;
    unsigned long port_flags = (unsigned long)CaptureDriverFlags;
    char name[REAL_JACK_PORT_NAME_SIZE+1];
    char alias[REAL_JACK_PORT_NAME_SIZE+1];

    assert(fCaptureChannels < DRIVER_PORT_NUM);
    assert(fPlaybackChannels < DRIVER_PORT_NUM);

    alsa_driver_t* alsa_driver = (alsa_driver_t*)fDriver;

    if (alsa_driver->has_hw_monitoring)
        port_flags |= JackPortCanMonitor;

    // ALSA driver may have changed the values
    JackAudioDriver::SetBufferSize(alsa_driver->frames_per_cycle);
    JackAudioDriver::SetSampleRate(alsa_driver->frame_rate);

    jack_log("JackAlsaDriver::Attach fBufferSize %ld fSampleRate %ld", fEngineControl->fBufferSize, fEngineControl->fSampleRate);

    for (int i = 0, port_list_index = 0; i < alsa_driver->devices_c_count; ++i) {
        alsa_device_t *device = &alsa_driver->devices[i];
        for (int j = 0; j < device->capture_nchannels; ++j, ++port_list_index) {
            snprintf(name, sizeof(name), "%s:capture_%d", fClientControl.fName, port_list_index + 1);
            snprintf(alias, sizeof(alias), "%s:%s:capture_%d", fAliasName, device->capture_name, j + 1);
            if (fEngine->PortRegister(fClientControl.fRefNum, name, JACK_DEFAULT_AUDIO_TYPE, (JackPortFlags)port_flags, fEngineControl->fBufferSize, &port_id) < 0) {
            jack_error("driver: cannot register port for %s", name);
                return AUDIO_DRIVER_FAILED;
            }
            port = fGraphManager->GetPort(port_id);
            port->SetAlias(alias);
            fCapturePortList[port_list_index] = port_id;
            jack_log("JackAlsaDriver::Attach fCapturePortList[i] %ld ", port_id);
        }
    }

    port_flags = (unsigned long)PlaybackDriverFlags;

    for (int i = 0, port_list_index = 0; i < alsa_driver->devices_p_count; ++i) {
        alsa_device_t *device = &alsa_driver->devices[i];
        for (int j = 0; j < device->playback_nchannels; ++j, ++port_list_index) {
            snprintf(name, sizeof(name), "%s:playback_%d", fClientControl.fName, port_list_index + 1);
            snprintf(alias, sizeof(alias), "%s:%s:playback_%d", fAliasName, device->playback_name, j + 1);
            if (fEngine->PortRegister(fClientControl.fRefNum, name, JACK_DEFAULT_AUDIO_TYPE, (JackPortFlags)port_flags, fEngineControl->fBufferSize, &port_id) < 0) {
                jack_error("driver: cannot register port for %s", name);
                return AUDIO_DRIVER_FAILED;
            }
            port = fGraphManager->GetPort(port_id);
            port->SetAlias(alias);
            fPlaybackPortList[port_list_index] = port_id;
            jack_log("JackAlsaDriver::Attach fPlaybackPortList[i] %ld ", port_id);

            // Monitor ports
            if (fWithMonitorPorts) {
                jack_log("Create monitor port");
                snprintf(name, sizeof(name), "%s:monitor_%d", fClientControl.fName, port_list_index + 1);
                if (fEngine->PortRegister(fClientControl.fRefNum, name, JACK_DEFAULT_AUDIO_TYPE, MonitorDriverFlags, fEngineControl->fBufferSize, &port_id) < 0) {
                    jack_error("ALSA: cannot register monitor port for %s", name);
                } else {
                    fMonitorPortList[port_list_index] = port_id;
                }
            }
        }
    }

    UpdateLatencies();

    if (alsa_driver->midi) {
        int err = (alsa_driver->midi->attach)(alsa_driver->midi);
        if (err)
            jack_error ("ALSA: cannot attach MIDI: %d", err);
    }

    return AUDIO_DRIVER_OK;
}

int JackAlsaDriver::Detach()
{
    alsa_driver_t* alsa_driver = (alsa_driver_t*)fDriver;
    if (alsa_driver->midi)
        (alsa_driver->midi->detach)(alsa_driver->midi);

    return JackAudioDriver::Detach();
}

#ifndef __QNXNTO__
extern "C" char* get_control_device_name(const char * device_name)
{
    char * ctl_name;
    const char * comma;

    /* the user wants a hw or plughw device, the ctl name
     * should be hw:x where x is the card identification.
     * We skip the subdevice suffix that starts with comma */

    if (strncasecmp(device_name, "plughw:", 7) == 0) {
        /* skip the "plug" prefix" */
        device_name += 4;
    }

    comma = strchr(device_name, ',');
    if (comma == NULL) {
        ctl_name = strdup(device_name);
        if (ctl_name == NULL) {
            jack_error("strdup(\"%s\") failed.", device_name);
        }
    } else {
        ctl_name = strndup(device_name, comma - device_name);
        if (ctl_name == NULL) {
            jack_error("strndup(\"%s\", %u) failed.", device_name, (unsigned int)(comma - device_name));
        }
    }

    return ctl_name;
}
#endif

#ifndef __QNXNTO__
static int card_to_num(const char* device)
{
    int err;
    char* ctl_name;
    snd_ctl_card_info_t *card_info;
    snd_ctl_t* ctl_handle;
    int i = -1;

    snd_ctl_card_info_alloca (&card_info);

    ctl_name = get_control_device_name(device);
    if (ctl_name == NULL) {
        jack_error("get_control_device_name() failed.");
        goto fail;
    }

    if ((err = snd_ctl_open (&ctl_handle, ctl_name, 0)) < 0) {
        jack_error ("control open \"%s\" (%s)", ctl_name,
                    snd_strerror(err));
        goto free;
    }

    if ((err = snd_ctl_card_info(ctl_handle, card_info)) < 0) {
        jack_error ("control hardware info \"%s\" (%s)",
                    device, snd_strerror (err));
        goto close;
    }

    i = snd_ctl_card_info_get_card(card_info);

close:
    snd_ctl_close(ctl_handle);

free:
    free(ctl_name);

fail:
    return i;
}
#endif

bool JackAlsaDriver::isAnyDeviceClosedOnStartup()
{
    const alsa_driver_t& driver = *reinterpret_cast<alsa_driver_t*>(fDriver);
    for (int i=0; i<driver.devices_count; ++i) {
        const alsa_device_t& device = driver.devices[i];
        if (device.capture_useroption_state.startup == SND_PCM_STATE_CLOSED ||
            device.playback_useroption_state.startup == SND_PCM_STATE_CLOSED) {
            return true;
        }
    }
    return false;
}

int JackAlsaDriver::Open(alsa_driver_info_t info)
{
    alsa_driver_status_t err = ALSA_DRIVER_OK;
    // Generic JackAudioDriver Open
    if (JackAudioDriver::Open(
            info.frames_per_period,
            info.frame_rate,
            info.devices_capture_size > 0,
            info.devices_playback_size > 0,
            0,
            0,
            info.monitor,
            info.devices_capture_size > 0 ? info.devices[0].capture_name : "-",
            info.devices_playback_size > 0 ? info.devices[0].playback_name : "-",
            info.capture_latency,
            info.playback_latency) != 0) {
        return AUDIO_DRIVER_FAILED;
    }

    jack_log("JackAlsaDriver::Open capture_driver_name = %s", info.devices_capture_size > 0 ? info.devices[0].capture_name : "-");
    jack_log("JackAlsaDriver::Open playback_driver_name = %s", info.devices_playback_size > 0 ? info.devices[0].playback_name : "-");

#ifndef __QNXNTO__
#ifndef __ANDROID__
#ifndef UNITTEST
    if (strcmp(info.midi_name, "seq") == 0)
        info.midi_driver = alsa_seqmidi_new((jack_client_t*)this, 0);
    else if (strcmp(info.midi_name, "raw") == 0)
        info.midi_driver = alsa_rawmidi_new((jack_client_t*)this);
#endif
#endif

    // FIXME: needs adaptation for multiple drivers
    if (JackServerGlobals::on_device_acquire != NULL) {
        int capture_card = card_to_num(info.devices_capture_size > 0 ? info.devices[0].capture_name : "-");
        int playback_card = card_to_num(info.devices_playback_size > 0 ? info.devices[0].playback_name : "-");
        char audio_name[32];

        if (capture_card >= 0) {
            snprintf(audio_name, sizeof(audio_name), "Audio%d", capture_card);
            if (!JackServerGlobals::on_device_acquire(audio_name)) {
                jack_error("Audio device %s cannot be acquired...", info.devices_capture_size > 0 ? info.devices[0].capture_name : "-");
                return AUDIO_DRIVER_FAILED;
            }
        }

        if (playback_card >= 0 && playback_card != capture_card) {
            snprintf(audio_name, sizeof(audio_name), "Audio%d", playback_card);
            if (!JackServerGlobals::on_device_acquire(audio_name)) {
                jack_error("Audio device %s cannot be acquired...",info.devices_playback_size > 0 ? info.devices[0].playback_name : "-" );
                if (capture_card >= 0) {
                    snprintf(audio_name, sizeof(audio_name), "Audio%d", capture_card);
                    JackServerGlobals::on_device_release(audio_name);
                }
                return AUDIO_DRIVER_FAILED;
            }
        }
    }
#endif

    fDriver = alsa_driver_new ((char*)"alsa_pcm", info, NULL);
    if (!fDriver) {
        /*
         * As devices are not opened yet, no need to call close()
         * Freeing of memory is ensured to be done before
         * exit of alsa_driver_new()
         */
        return AUDIO_DRIVER_FAILED;
    }

    /* we need to initialize variables for all devices, mainly channels count since this is required by Jack to setup ports */
    UpdateDriverTargetState(DriverMode::Init, JackBackendCmdNone);

    if ((err = alsa_driver_open((alsa_driver_t *)fDriver)) != ALSA_DRIVER_OK) {
        ResetDriver(err);
        return AUDIO_DRIVER_RESET;
    }

    /* we are starting with all alsa devices closed, therfore populate jack channels based on user hint */
    if (isAnyDeviceClosedOnStartup()) {
        for (size_t i = 0; i < std::max(info.devices_capture_size, info.devices_playback_size); ++i) {
            if (i < info.devices_capture_size && info.devices[i].capture_channels < 1) {
                jack_error ("invalid or missing channels parameter with device option capture-state: '%s'", info.devices[i].capture_name);
                Close();
                return AUDIO_DRIVER_FAILED;
            }

            if (i < info.devices_playback_size && info.devices[i].playback_channels < 1) {
                jack_error ("invalid or missing channels parameter with device option playback-state: '%s'", info.devices[i].playback_name);
                Close();
                return AUDIO_DRIVER_FAILED;
            }

            fCaptureChannels += info.devices[i].capture_channels;
            fPlaybackChannels += info.devices[i].playback_channels;
        }
    /* in case we really opened alsa devices, channel information is generated by driver */
    } else {
        fCaptureChannels = ((alsa_driver_t *)fDriver)->capture_nchannels;
        fPlaybackChannels = ((alsa_driver_t *)fDriver)->playback_nchannels;
    }

    return AUDIO_DRIVER_OK;
}

int JackAlsaDriver::Close()
{
    // Generic audio driver close
    int res = JackAudioDriver::Close();

    UpdateDriverTargetState(DriverMode::Shutdown, JackBackendCmdNone);
    alsa_driver_close((alsa_driver_t *)fDriver);

    if (fDriver) {
        alsa_driver_delete((alsa_driver_t*)fDriver);
    }

#ifndef __QNXNTO__
    // FIXME: needs adaptation for multiple drivers
    if (JackServerGlobals::on_device_release != NULL)
    {
        char audio_name[32];
        int capture_card = card_to_num(fCaptureDriverName);
        if (capture_card >= 0) {
            snprintf(audio_name, sizeof(audio_name), "Audio%d", capture_card);
            JackServerGlobals::on_device_release(audio_name);
        }

        int playback_card = card_to_num(fPlaybackDriverName);
        if (playback_card >= 0 && playback_card != capture_card) {
            snprintf(audio_name, sizeof(audio_name), "Audio%d", playback_card);
            JackServerGlobals::on_device_release(audio_name);
        }
    }
#endif

    return res;
}

int JackAlsaDriver::Start()
{
    alsa_driver_status_t err = ALSA_DRIVER_OK;
    int res = JackAudioDriver::Start();

    if(res < 0 && alsa_check_fatal((alsa_driver_t *)fDriver))
        jack_fatal("JackAudioDriver::Start() failed");
    if (res >= 0) {
        err = alsa_driver_start((alsa_driver_t *)fDriver);
        if (err < 0) {
            ResetDriver(err);
            /* Return 0 as this is directly called by ThreadDriver and it only checks for < 0 */
            res = AUDIO_DRIVER_OK;
        }
    }
    return res;
}

int JackAlsaDriver::Stop()
{
    int res = AUDIO_DRIVER_OK;
    alsa_driver_status_t err = alsa_driver_stop((alsa_driver_t *)fDriver);
    if (err < 0) {
        ResetDriver(err);
       /* Return AUDIO_DRIVER_RESET because the error needs to be handled in called functions */
        res = AUDIO_DRIVER_RESET;
    }
    if (JackAudioDriver::Stop() < 0) {
        if(alsa_check_fatal((alsa_driver_t *)fDriver))
            jack_fatal("JackAudioDriver::Stop() failed");
        else
            jack_error("JackAudioDriver::Stop() failed");
        if(res == ALSA_DRIVER_OK)
            res = AUDIO_DRIVER_FAILED;
    }
    return res;
}

int JackAlsaDriver::ReloadDevices(jack_backend_command_id_t cmd)
{
    int ret = 0;
    alsa_driver_status_t driver_status = ALSA_DRIVER_OK;

#ifdef __QNXNTO__
    constexpr int DEFAULT_SCHED_POL = SCHED_FIFO;
    struct sched_param current_sched_param = {0};
    int saved_sched_pol;
    int saved_sched_priority;
    bool saved_sched_valid = false;
    int err;

    err = pthread_getschedparam(pthread_self(), &saved_sched_pol, &current_sched_param);
    if (err != EOK) {
        jack_error("pthread_getschedparam failed with error %d", err);
    } else {
        saved_sched_priority = current_sched_param.sched_priority;
        saved_sched_valid = true;
        jack_log("Saved priority %d and sched policy %d", saved_sched_priority, saved_sched_pol);

        if ((saved_sched_priority < fEngineControl->fServerPriority) ||
                (saved_sched_pol != DEFAULT_SCHED_POL)) {
            current_sched_param.sched_priority = fEngineControl->fServerPriority;
            err = pthread_setschedparam(pthread_self(), DEFAULT_SCHED_POL, &current_sched_param);
            if (err != EOK) {
                jack_error("pthread_setschedparam for priority %d and "
                           "sched policy %d failed with error %d.",
                           fEngineControl->fServerPriority, DEFAULT_SCHED_POL, err);
                saved_sched_valid = false;
            } else {
                jack_log("Priority set to %d and sched policy to %d",
                         fEngineControl->fServerPriority, DEFAULT_SCHED_POL);
            }
        }
    }
#endif

    UpdateDriverTargetState(DriverMode::Runtime, cmd);

    alsa_driver_t* driver = (alsa_driver_t*) fDriver;
    if (alsa_driver_close (driver) != ALSA_DRIVER_OK) {
        jack_error("JackAlsaDriver::ReloadDevices close failed");
        ret = AUDIO_DRIVER_FAILED;
        goto end;
    }
    if ((driver_status = alsa_driver_open (driver)) != ALSA_DRIVER_OK) {
        jack_error("JackAlsaDriver::ReloadDevices open failed");
        ResetDriver(driver_status);
        /* Return AUDIO_DRIVER_OK so that the ThreadedDriver will Start the backend thread for dummy cycle*/
        return AUDIO_DRIVER_OK;
    }

end:
#ifdef __QNXNTO__
    if (saved_sched_valid &&
            ((saved_sched_priority < fEngineControl->fServerPriority) ||
             (saved_sched_pol != DEFAULT_SCHED_POL))) {
        current_sched_param.sched_priority = saved_sched_priority;
        err = pthread_setschedparam(pthread_self(), saved_sched_pol, &current_sched_param);
        if (err != EOK) {
            jack_error("pthread_setschedparam to previous priority %d and "
                       "sched policy %d failed with error %d.",
                       saved_sched_priority, saved_sched_pol, err);
        } else {
            jack_log("Priority set back to %d and sched policy to %d",
                     saved_sched_priority, saved_sched_pol);
        }
    }
#endif

    return ret;
}

int JackAlsaDriver::HandleBackendCommand(jack_backend_command_t cmd)
{
    int ret = 0;

    jack_log("JackAlsaDriver::HandleBackendCommand cmd = %d value = %d", cmd.cmd, cmd.arg);

    switch(cmd.cmd) {
        case JackBackendCmdNone: {
            /*Do Nothing*/
            break;
        }
        case JackBackendReload:
        case JackBackendStopAndCloseIdleDevices:
        case JackBackendOpenAndStartAllDevices: {
            ret = ReloadDevices(cmd.cmd);
            break;
        }
        default: {
            jack_error("JackAlsaDriver::HandleBackendCommand: command not supported %d", cmd);
            ret = -1;
        }
    }

    return ret;
}

/**
 * \brief Execute read on ALSA device
 * \return AUDIO_DRIVER_OK: OK
 *         AUDIO_DRIVER_RECOVERED: Backend ran into xruns but xrun recovery was done. OK.
 *         AUDIO_DRIVER_FAILED: Backend ran into errors or max xrun recovery count reached. NOT OK.
 */
int JackAlsaDriver::Read()
{
    /* Taken from alsa_driver_run_cycle */
    jack_nframes_t nframes;
    fDelayedUsecs = 0.f;
    alsa_driver_status_t err = ALSA_DRIVER_OK;

retry_wait:
    err = alsa_driver_wait((alsa_driver_t *)fDriver, -1, &nframes, &fDelayedUsecs);

    if (err != ALSA_DRIVER_OK) {
        jack_log("JackAlsaDriver::Read wait failed");
        goto retry;
    }

    if (nframes == 0)
    {
        goto retry_wait;
    }

    if (nframes != fEngineControl->fBufferSize)
        jack_log("JackAlsaDriver::Read warning fBufferSize = %ld nframes = %ld", fEngineControl->fBufferSize, nframes);

    // Has to be done before read
    JackDriver::CycleIncTime();

    if ((err = alsa_driver_read((alsa_driver_t *)fDriver, fEngineControl->fBufferSize)) != 0) {
        jack_error("JackAlsaDriver::Read read failed");
        goto retry;
    }

    return AUDIO_DRIVER_OK;

retry:
    if(err != ALSA_DRIVER_RESET) {
        /* we detected an xrun and restarted: notify
         * clients about the delay.
         */
        jack_log("JackAlsaDriver::Read failed, xrun recovery");
        err = alsa_driver_xrun_recovery((alsa_driver_t *)fDriver, &fDelayedUsecs);
        NotifyXRun(fBeginDateUst, fDelayedUsecs);
        if(err != ALSA_DRIVER_OK) {
            ResetDriver(err);
        }
    } else {
        jack_log("JackAlsaDriver::Read failed, driver in reset state");
        ResetDriver(err);
    }
    return (err == ALSA_DRIVER_OK) ? AUDIO_DRIVER_RECOVERED : AUDIO_DRIVER_RESET;
}

/**
 * \brief Execute write on ALSA device
 * \return AUDIO_DRIVER_OK: OK
 *         AUDIO_DRIVER_RECOVERED: Backend ran into xruns but xrun recovery was done. OK
 *         AUDIO_DRIVER_FAILED: Backend ran into errors or max xrun recovery count reached. NOT OK.
 *         AUDIO_DRIVER_RESET: Backend is in a reset state. OK
 */

int JackAlsaDriver::Write()
{
    alsa_driver_status_t err = ALSA_DRIVER_OK;
    JackAudioDriverStatus ret = AUDIO_DRIVER_OK;

    err = alsa_driver_write((alsa_driver_t *)fDriver, fEngineControl->fBufferSize);
    if (err != ALSA_DRIVER_OK) {
        if(err != ALSA_DRIVER_RESET) {
            jack_log("JackAlsaDriver::Write failed, xrun recovery");
            err = alsa_driver_xrun_recovery((alsa_driver_t *)fDriver, &fDelayedUsecs);
            NotifyXRun(fBeginDateUst, fDelayedUsecs);
            if(err != ALSA_DRIVER_OK) {
                ResetDriver(err);
            }
            ret = (err == ALSA_DRIVER_OK) ? AUDIO_DRIVER_RECOVERED : AUDIO_DRIVER_RESET;
        } else {
            jack_log("JackAlsaDriver::Write failed, driver in reset state");
            ResetDriver(err);
            ret = AUDIO_DRIVER_RESET;
        }
    }
    return (err == ALSA_DRIVER_OK) ? AUDIO_DRIVER_OK : ret;
}

void JackAlsaDriver::ReadInputAux(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nread)
{
    /* global channel offset to fCapturePortList of this capture alsa device */
    channel_t port_n = device->capture_channel_offset;

    for (channel_t chn = 0; chn < device->capture_nchannels; ++chn, ++port_n) {
        if (fGraphManager->GetConnectionsNum(fCapturePortList[port_n]) > 0) {
            jack_default_audio_sample_t* buf = (jack_default_audio_sample_t*)fGraphManager->GetBuffer(fCapturePortList[port_n], orig_nframes);
            alsa_driver_read_from_channel((alsa_driver_t *)fDriver, device, chn, buf + nread, contiguous);
        }
    }
}

void JackAlsaDriver::MonitorInputAux()
{
    for (int chn = 0; chn < fCaptureChannels; chn++) {
        JackPort* port = fGraphManager->GetPort(fCapturePortList[chn]);
        if (port->MonitoringInput()) {
            ((alsa_driver_t *)fDriver)->input_monitor_mask |= (1 << chn);
        }
    }
}

void JackAlsaDriver::ClearOutputAux()
{
    for (int chn = 0; chn < fPlaybackChannels; chn++) {
        jack_default_audio_sample_t* buf =
            (jack_default_audio_sample_t*)fGraphManager->GetBuffer(fPlaybackPortList[chn], fEngineControl->fBufferSize);
        memset(buf, 0, sizeof (jack_default_audio_sample_t) * fEngineControl->fBufferSize);
    }
}

void JackAlsaDriver::SetTimetAux(jack_time_t time)
{
    fBeginDateUst = time;
}

int JackAlsaDriver::UpdateDriverTargetState(DriverMode mode, jack_backend_command_id_t cmd )
{
    int c_list_index = 0, p_list_index = 0;
    alsa_driver_t* driver = (alsa_driver_t*) fDriver;

    for (int i = 0; i < driver->devices_count; ++i) {
        alsa_device_t *device = &driver->devices[i];

        int capture_connections_count = 0;
        for (int j = 0; j < device->capture_nchannels; ++j) {
            capture_connections_count += fGraphManager->GetConnectionsNum(fCapturePortList[c_list_index]);
            c_list_index++;
        }
        device->capture_target_state = TargetState(mode, capture_connections_count, cmd, device->capture_useroption_state);

        int playback_connections_count = 0;
        for (int j = 0; j < device->playback_nchannels; ++j) {
            playback_connections_count += fGraphManager->GetConnectionsNum(fPlaybackPortList[p_list_index]);
            p_list_index++;
        }
        device->playback_target_state = TargetState(mode, playback_connections_count, cmd, device->playback_useroption_state);
    }

    return AUDIO_DRIVER_OK;
}

void JackAlsaDriver::ResetDriverTargetState()
{
    alsa_driver_t* driver = (alsa_driver_t*) fDriver;

    for (int i = 0; i < driver->devices_count; ++i) {
        alsa_device_t *device = &driver->devices[i];
        device->capture_target_state = SND_PCM_STATE_CLOSED;
        device->playback_target_state = SND_PCM_STATE_CLOSED;
    }
}

int JackAlsaDriver::TargetState(DriverMode mode, int connections_count, jack_backend_command_id_t cmd, const target_state_t& target_state)
{
    if (mode == DriverMode::Init)
        return target_state.startup;
    else if (mode == DriverMode::Shutdown) {
        return SND_PCM_STATE_CLOSED;
    }

    switch (cmd) {
        case JackBackendOpenAndStartAllDevices:
            return SND_PCM_STATE_RUNNING;
        case JackBackendStopAndCloseIdleDevices:
            return (connections_count > 0) ? SND_PCM_STATE_RUNNING : SND_PCM_STATE_CLOSED;
        default:
            jack_error("Invalid backend cmd %d. So go with reload behavior", cmd);
        case JackBackendReload:
            if (connections_count > 0)
                return SND_PCM_STATE_RUNNING;
            else
                return target_state.idle_at_runtime;
    }
}

void JackAlsaDriver::ResetDriver(alsa_driver_status_t status)
{
        ResetDriverTargetState();
        jack_error("Resetting the driver");

        alsa_driver_stop((alsa_driver_t *)fDriver);
        alsa_driver_close ((alsa_driver_t *)fDriver);

        if(status == ALSA_DRIVER_RESET) {
            NotifyDriverError(JackDriverInHardwareResetState);
        } else {
            NotifyDriverError(JackDriverInResetState);
        }
}

void JackAlsaDriver::WriteOutputAux(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nwritten)
{
    /* global channel offset to fPlaybackPortList of this playback alsa device */
    channel_t port_n = device->playback_channel_offset;

    for (channel_t chn = 0; chn < device->playback_nchannels; ++chn, ++port_n) {
        // Output ports
        if (fGraphManager->GetConnectionsNum(fPlaybackPortList[port_n]) > 0) {
            jack_default_audio_sample_t* buf = (jack_default_audio_sample_t*)fGraphManager->GetBuffer(fPlaybackPortList[port_n], orig_nframes);
            alsa_driver_write_to_channel(((alsa_driver_t *)fDriver), device, chn, buf + nwritten, contiguous);
            // Monitor ports
            if (fWithMonitorPorts && fGraphManager->GetConnectionsNum(fMonitorPortList[port_n]) > 0) {
                jack_default_audio_sample_t* monbuf = (jack_default_audio_sample_t*)fGraphManager->GetBuffer(fMonitorPortList[port_n], orig_nframes);
                memcpy(monbuf + nwritten, buf + nwritten, contiguous * sizeof(jack_default_audio_sample_t));
            }
        }
    }
}

int JackAlsaDriver::is_realtime() const
{
    return fEngineControl->fRealTime;
}

int JackAlsaDriver::create_thread(pthread_t *thread, int priority, int realtime, void *(*start_routine)(void*), void *arg)
{
#ifdef __ANDROID__
    return JackAndroidThread::StartImp(thread, priority, realtime, start_routine, arg);
#else
    return JackPosixThread::StartImp(thread, priority, realtime, start_routine, arg);
#endif
}

jack_port_id_t JackAlsaDriver::port_register(const char *port_name, const char *port_type, unsigned long flags, unsigned long buffer_size)
{
    jack_port_id_t port_index;
    int res = fEngine->PortRegister(fClientControl.fRefNum, port_name, port_type, flags, buffer_size, &port_index);
    return (res == 0) ? port_index : 0;
}

int JackAlsaDriver::port_unregister(jack_port_id_t port_index)
{
    return fEngine->PortUnRegister(fClientControl.fRefNum, port_index);
}

void* JackAlsaDriver::port_get_buffer(int port, jack_nframes_t nframes)
{
    return fGraphManager->GetBuffer(port, nframes);
}

int  JackAlsaDriver::port_set_alias(int port, const char* name)
{
    return fGraphManager->GetPort(port)->SetAlias(name);
}

jack_nframes_t JackAlsaDriver::get_sample_rate() const
{
    return fEngineControl->fSampleRate;
}

jack_nframes_t JackAlsaDriver::frame_time() const
{
    JackTimer timer;
    fEngineControl->ReadFrameTime(&timer);
    return timer.Time2Frames(GetMicroSeconds(), fEngineControl->fBufferSize);
}

jack_nframes_t JackAlsaDriver::last_frame_time() const
{
    JackTimer timer;
    fEngineControl->ReadFrameTime(&timer);
    return timer.CurFrame();
}

} // end of namespace


#ifdef __cplusplus
extern "C"
{
#endif

#ifndef __QNXNTO__
static
jack_driver_param_constraint_desc_t *
enum_alsa_devices()
{
    snd_ctl_t * handle;
    snd_ctl_card_info_t * info;
    snd_pcm_info_t * pcminfo_capture;
    snd_pcm_info_t * pcminfo_playback;
    int card_no = -1;
    jack_driver_param_value_t card_id;
    jack_driver_param_value_t device_id;
    char description[64];
    int device_no;
    bool has_capture;
    bool has_playback;
    jack_driver_param_constraint_desc_t * constraint_ptr;
    uint32_t array_size = 0;

    snd_ctl_card_info_alloca(&info);
    snd_pcm_info_alloca(&pcminfo_capture);
    snd_pcm_info_alloca(&pcminfo_playback);

    constraint_ptr = NULL;

    while(snd_card_next(&card_no) >= 0 && card_no >= 0)
    {
        snprintf(card_id.str, sizeof(card_id.str), "hw:%d", card_no);

        if (snd_ctl_open(&handle, card_id.str, 0) >= 0 &&
            snd_ctl_card_info(handle, info) >= 0)
        {
            snprintf(card_id.str, sizeof(card_id.str), "hw:%s", snd_ctl_card_info_get_id(info));
            if (!jack_constraint_add_enum(
                    &constraint_ptr,
                    &array_size,
                    &card_id,
                    snd_ctl_card_info_get_name(info)))
                goto fail;

            device_no = -1;

            while (snd_ctl_pcm_next_device(handle, &device_no) >= 0 && device_no != -1)
            {
                snprintf(device_id.str, sizeof(device_id.str), "%s,%d", card_id.str, device_no);

                snd_pcm_info_set_device(pcminfo_capture, device_no);
                snd_pcm_info_set_subdevice(pcminfo_capture, 0);
                snd_pcm_info_set_stream(pcminfo_capture, SND_PCM_STREAM_CAPTURE);
                has_capture = snd_ctl_pcm_info(handle, pcminfo_capture) >= 0;

                snd_pcm_info_set_device(pcminfo_playback, device_no);
                snd_pcm_info_set_subdevice(pcminfo_playback, 0);
                snd_pcm_info_set_stream(pcminfo_playback, SND_PCM_STREAM_PLAYBACK);
                has_playback = snd_ctl_pcm_info(handle, pcminfo_playback) >= 0;

                if (has_capture && has_playback)
                {
                    snprintf(description, sizeof(description),"%s (duplex)", snd_pcm_info_get_name(pcminfo_capture));
                }
                else if (has_capture)
                {
                    snprintf(description, sizeof(description),"%s (capture)", snd_pcm_info_get_name(pcminfo_capture));
                }
                else if (has_playback)
                {
                    snprintf(description, sizeof(description),"%s (playback)", snd_pcm_info_get_name(pcminfo_playback));
                }
                else
                {
                    continue;
                }

                if (!jack_constraint_add_enum(
                        &constraint_ptr,
                        &array_size,
                        &device_id,
                        description))
                    goto fail;
            }

            snd_ctl_close(handle);
        }
    }

    return constraint_ptr;
fail:
    jack_constraint_free(constraint_ptr);
    return NULL;
}
#endif

static int
dither_opt (char c, DitherAlgorithm* dither)
{
    switch (c) {
        case '-':
        case 'n':
            *dither = None;
            break;

        case 'r':
            *dither = Rectangular;
            break;

        case 's':
            *dither = Shaped;
            break;

        case 't':
            *dither = Triangular;
            break;

        default:
            fprintf (stderr, "ALSA driver: illegal dithering mode %c\n", c);
            return ALSA_DRIVER_ERROR;
    }
    return ALSA_DRIVER_OK;
}

SERVER_EXPORT const jack_driver_desc_t* driver_get_descriptor ()
{
    jack_driver_desc_t * desc;
    jack_driver_desc_filler_t filler;
    jack_driver_param_value_t value;

    desc = jack_driver_descriptor_construct("alsa", JackDriverMaster, "Linux ALSA API based audio backend", &filler);

    strcpy(value.str, "hw:0");
#ifndef __QNXNTO__
#ifdef __ANDROID__
    jack_driver_descriptor_add_parameter(desc, &filler, "device", 'd', JackDriverParamString, &value, NULL, "ALSA device name", NULL);
#else
    jack_driver_descriptor_add_parameter(desc, &filler, "device", 'd', JackDriverParamString, &value, enum_alsa_devices(), "ALSA device name", NULL);
#endif
#endif


    strcpy(value.str, "none");
    jack_driver_descriptor_add_parameter(desc, &filler, "capture", 'C', JackDriverParamString, &value, NULL, "Provide capture ports.  Optionally set device", NULL);
    jack_driver_descriptor_add_parameter(desc, &filler, "playback", 'P', JackDriverParamString, &value, NULL, "Provide playback ports.  Optionally set device", NULL);

    value.ui = 48000U;
    jack_driver_descriptor_add_parameter(desc, &filler, "rate", 'r', JackDriverParamUInt, &value, NULL, "Sample rate", NULL);

    value.ui = 1024U;
    jack_driver_descriptor_add_parameter(desc, &filler, "period", 'p', JackDriverParamUInt, &value, NULL, "Frames per period", NULL);

    value.ui = 2U;
    jack_driver_descriptor_add_parameter(desc, &filler, "nperiods", 'n', JackDriverParamUInt, &value, NULL, "Number of periods of playback latency", NULL);

    value.ui  = 2U;
    jack_driver_descriptor_add_parameter(desc, &filler, "nprefill", 'f', JackDriverParamUInt, &value, NULL, "Number of periods of playback prefill", NULL);

    value.i = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "hwmon", 'H', JackDriverParamBool, &value, NULL, "Hardware monitoring, if available", NULL);

    value.i = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "hwmeter", 'M', JackDriverParamBool, &value, NULL, "Hardware metering, if available", NULL);

    value.i = 1;
    jack_driver_descriptor_add_parameter(desc, &filler, "duplex", 'D', JackDriverParamBool, &value, NULL, "Provide both capture and playback ports", NULL);

    value.i = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "softmode", 's', JackDriverParamBool, &value, NULL, "Soft-mode, no xrun handling", NULL);

    value.i = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "monitor", 'm', JackDriverParamBool, &value, NULL, "Provide monitor ports for the output", NULL);

    value.c = 'n';
    jack_driver_descriptor_add_parameter(
        desc,
        &filler,
        "dither",
        'z',
        JackDriverParamChar,
        &value,
        jack_constraint_compose_enum_char(
            JACK_CONSTRAINT_FLAG_STRICT | JACK_CONSTRAINT_FLAG_FAKE_VALUE,
            dither_constraint_descr_array),
        "Dithering mode",
        NULL);

    strcpy(value.str, "none");
    jack_driver_descriptor_add_parameter(desc, &filler, "inchannels", 'i', JackDriverParamString, &value, NULL, "List of device capture channels (defaults to hw max)", NULL);
    jack_driver_descriptor_add_parameter(desc, &filler, "outchannels", 'o', JackDriverParamString, &value, NULL, "List of device playback channels (defaults to hw max)", NULL);

    value.i = FALSE;
    jack_driver_descriptor_add_parameter(desc, &filler, "shorts", 'S', JackDriverParamBool, &value, NULL, "Try 16-bit samples before 32-bit", NULL);

    value.ui = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "input-latency", 'I', JackDriverParamUInt, &value, NULL, "Extra input latency (frames)", NULL);
    jack_driver_descriptor_add_parameter(desc, &filler, "output-latency", 'O', JackDriverParamUInt, &value, NULL, "Extra output latency (frames)", NULL);

    strcpy(value.str, "none");
    jack_driver_descriptor_add_parameter(
        desc,
        &filler,
        "midi-driver",
        'X',
        JackDriverParamString,
        &value,
        jack_constraint_compose_enum_str(
            JACK_CONSTRAINT_FLAG_STRICT | JACK_CONSTRAINT_FLAG_FAKE_VALUE,
            midi_constraint_descr_array),
        "ALSA MIDI driver",
        NULL);

    strncpy(value.str, "startup(stream/2):idle_at_runtime(prepare/1)", sizeof(value.str));
    value.str[sizeof(value.str) - 1] = '\0';
    jack_driver_descriptor_add_parameter(desc, &filler, "capture-state", 'l', JackDriverParamString, &value, NULL, "Capture device state options,0-close,1-prepare,2-stream", NULL);
    jack_driver_descriptor_add_parameter(desc, &filler, "playback-state", 'q', JackDriverParamString, &value, NULL, "Playback device state options,0-close,1-prepare,2-stream", NULL);

    value.i = 0;
    jack_driver_descriptor_add_parameter(desc, &filler, "unlinked-devs", 'u', JackDriverParamBool, &value, NULL, "Do not link devices", NULL);

    return desc;
}

void array_string_free(struct array_string_t *obj)
{
    if (obj == NULL) {
        return;
    }
    for (size_t i = 0; i < obj->data.size(); ++i) {
        free(obj->data[i]);
    }
}

struct array_string_t array_string_split(const char *str, const char sep, array_string_t::flags flags)
{
    struct array_string_t result;

    std::stringstream stream;
    stream << std::string(str);
    if (stream.str().find(sep) == std::string::npos) {
        result.data.push_back((char*) calloc(JACK_CLIENT_NAME_SIZE + 1, sizeof(char)));
        strncpy(result.data[0], str, JACK_CLIENT_NAME_SIZE);
        result.data[0][JACK_CLIENT_NAME_SIZE] = '\0';
        return result;
    }

    std::string driver;
    while (std::getline(stream, driver, sep)) {
        driver.erase(std::remove_if(driver.begin(), driver.end(), isspace), driver.end());
        if (std::find(result.data.begin(), result.data.end(), driver) != result.data.end() && (flags & array_string_t::discard_duplicate))
            continue;
        char *str = (char*) calloc(JACK_CLIENT_NAME_SIZE + 1, sizeof(char));
        strncpy(str, driver.c_str(), JACK_CLIENT_NAME_SIZE);
        str[JACK_CLIENT_NAME_SIZE] = '\0';
        result.data.push_back(str);
    }

    return result;
}

snd_pcm_state_t GetTargetStateFromOption(unsigned int device_option)
{
    static const snd_pcm_state_t PARAM2SND_PCM_STATE[] = {
        SND_PCM_STATE_CLOSED,
        SND_PCM_STATE_PREPARED,
        SND_PCM_STATE_RUNNING,
    };
    if (device_option >= ARRAY_SIZE(PARAM2SND_PCM_STATE)) {
        jack_error("Invalid option [%d], so go to the default option (state:stream)", device_option);
        return SND_PCM_STATE_RUNNING;
     }
    return PARAM2SND_PCM_STATE[device_option];
}

void SetUserOptionState(const struct array_string_t& target_state_param, size_t device_index, target_state_t& target_state)
{
    target_state.startup = SND_PCM_STATE_RUNNING;
    target_state.idle_at_runtime = SND_PCM_STATE_PREPARED;

    if (device_index < target_state_param.data.size()) {
        unsigned int device_option = 0;
        char *endptr = nullptr;

        //Startup options
        const char* startup_string = strtok(target_state_param.data[device_index],":");
        if (startup_string == nullptr) {
            jack_error("Strtok failed to get the split from the user option. So go with the default");
            return;
        }

        device_option = strtol(startup_string, &endptr, 10);
        if (startup_string == endptr) {
            jack_error("Strtol failed to get the digit value. So go with the default");
            return;
        }

        target_state.startup = GetTargetStateFromOption(device_option);

        //idle_at_runtime options
        const char* idle_at_runtime_string = strtok(NULL,":");
        if (idle_at_runtime_string == nullptr) {
            jack_error("Strtok failed to get the split from the user option. So go with the default");
            return;
        }

        device_option = strtol(idle_at_runtime_string, &endptr, 10);
        if (idle_at_runtime_string == endptr) {
            jack_error("Strtol failed to get the digit value. So go with the default");
            return;
        }

        target_state.idle_at_runtime = GetTargetStateFromOption(device_option);
    }

}

static Jack::JackAlsaDriver* g_alsa_driver;

SERVER_EXPORT Jack::JackDriverClientInterface* driver_initialize(Jack::JackLockedEngine* engine, Jack::JackSynchro* table, const JSList* params)
{
    const JSList * node;
    const jack_driver_param_t * param;

    alsa_driver_info_t info = {};
    info.devices = NULL;
    info.midi_name = strdup("none");
    info.hw_monitoring = FALSE;
    info.hw_metering = FALSE;
    info.monitor = FALSE;
    info.soft_mode = FALSE;
    info.frame_rate = 48000;
    info.frames_per_period = 1024;
    info.periods_n = 2;
    /* Period prefill default value is -1 to notice if this value was configured.
       If it is not set it will be later set to periods_n.*/
    info.playback_prefill_periods_n = -1;
    info.dither = None;
    info.shorts_first = FALSE;
    info.capture_latency = 0;
    info.playback_latency = 0;

    char *capture_names_param = NULL;
    char *playback_names_param = NULL;

    char *capture_channels_param = NULL;
    char *playback_channels_param = NULL;

    char *capture_state_params = NULL;
    char *playback_state_params = NULL;

    int duplex = FALSE;

    for (node = params; node; node = jack_slist_next (node)) {
        param = (const jack_driver_param_t *) node->data;

        switch (param->character) {

            case 'C':
                if (strcmp (param->value.str, "none") != 0) {
                    free(capture_names_param);
                    capture_names_param = strdup (param->value.str);
                    jack_log("capture device %s", capture_names_param);
                }
                break;

            case 'P':
                if (strcmp (param->value.str, "none") != 0) {
                    free(playback_names_param);
                    playback_names_param = strdup (param->value.str);
                    jack_log("playback device %s", playback_names_param);
                }
                break;

            case 'D':
                duplex = TRUE;
                break;

            case 'd':
                if (strcmp (param->value.str, "none") != 0) {
                    free(playback_names_param);
                    free(capture_names_param);
                    playback_names_param = strdup (param->value.str);
                    capture_names_param = strdup (param->value.str);
                    jack_log("playback device %s", playback_names_param);
                    jack_log("capture device %s", capture_names_param);
                }
                break;

            case 'f':
                info.playback_prefill_periods_n = param->value.ui;
                /* Final amount of prefill periods is logged later. */
                break;

            case 'H':
                info.hw_monitoring = param->value.i;
                break;

            case 'm':
                info.monitor = param->value.i;
                break;

            case 'M':
                info.hw_metering = param->value.i;
                break;

            case 'r':
                info.frame_rate = param->value.ui;
                jack_log("apparent rate = %d", info.frame_rate);
                break;

            case 'p':
                info.frames_per_period = param->value.ui;
                jack_log("frames per period = %d", info.frames_per_period);
                break;

            case 'n':
                info.periods_n = param->value.ui;
                if (info.periods_n < 2) {    /* enforce minimum value */
                    info.periods_n = 2;
                }
                break;

            case 's':
                info.soft_mode = param->value.i;
                break;

            case 'z':
                if (dither_opt (param->value.c, &info.dither)) {
                    free(info.midi_name);
                    free(capture_channels_param);
                    return NULL;
                }
                break;

            case 'i':
                free(capture_channels_param);
                capture_channels_param = strdup(param->value.str);
                break;

            case 'o':
                free(playback_channels_param);
                playback_channels_param = strdup(param->value.str);
                break;

            case 'S':
                info.shorts_first = param->value.i;
                break;

            case 'I':
                info.capture_latency = param->value.ui;
                break;

            case 'O':
                info.playback_latency = param->value.ui;
                break;

            case 'X':
                free(info.midi_name);
                info.midi_name = strdup(param->value.str);
                break;

            case 'u':
                info.features |= param->value.i ? ALSA_DRIVER_FEAT_UNLINKED_DEVS : 0;
                break;
            case 'l':
                free(capture_state_params);
                capture_state_params = strdup(param->value.str);
                break;

            case 'q':
                free(playback_state_params);
                playback_state_params = strdup(param->value.str);
                break;
        }
    }

    /* duplex is the default */
    if (!capture_names_param && !playback_names_param) {
        duplex = TRUE;
    }

    if (duplex) {
        if (!capture_names_param) {
            capture_names_param = strdup("hw:0");
        }
        if (!playback_names_param) {
            playback_names_param = strdup("hw:0");
        }
    }

    struct array_string_t capture_names = {};
    if (capture_names_param) {
        capture_names = array_string_split(capture_names_param, ' ', array_string_t::discard_duplicate);
        free(capture_names_param);
    }

    struct array_string_t playback_names = {};
    if (playback_names_param) {
        playback_names = array_string_split(playback_names_param, ' ', array_string_t::discard_duplicate);
        free(playback_names_param);
    }

    struct array_string_t capture_channels = {};
    if (capture_channels_param) {
        capture_channels = array_string_split(capture_channels_param, ' ');
        free(capture_channels_param);
    }

    struct array_string_t playback_channels = {};
    if (playback_channels_param) {
        playback_channels = array_string_split(playback_channels_param, ' ');
        free(playback_channels_param);
    }

    struct array_string_t capture_target_state_param = {};
    if (capture_state_params) {
        capture_target_state_param = array_string_split(capture_state_params, ' ');
        free(capture_state_params);
    }

    struct array_string_t playback_target_state_param = {};
    if (playback_state_params) {
        playback_target_state_param = array_string_split(playback_state_params, ' ');
        free(playback_state_params);
    }

    info.devices_capture_size = capture_names.data.size();
    info.devices_playback_size = playback_names.data.size();
    info.devices = (alsa_device_info_t*) calloc(std::max(info.devices_capture_size, info.devices_playback_size), sizeof(alsa_device_info_t));
    for (size_t i = 0; i < std::max(info.devices_capture_size, info.devices_playback_size); ++i) {
        if (i < capture_names.data.size()) {
            info.devices[i].capture_name = strdup(capture_names.data[i]);
        }
        if (i < capture_channels.data.size()) {
            info.devices[i].capture_channels = atoi(capture_channels.data[i]);
        }

        SetUserOptionState(capture_target_state_param, i, info.devices[i].capture_useroption_state);

        if (i < playback_names.data.size()) {
            info.devices[i].playback_name = strdup(playback_names.data[i]);
        }
        if (i < playback_channels.data.size()) {
            info.devices[i].playback_channels = atoi(playback_channels.data[i]);
        }

        SetUserOptionState(playback_target_state_param, i, info.devices[i].playback_useroption_state);
    }

    array_string_free(&capture_names);
    array_string_free(&playback_names);
    array_string_free(&capture_channels);
    array_string_free(&playback_channels);
    array_string_free(&capture_target_state_param);
    array_string_free(&playback_target_state_param);

    g_alsa_driver = new Jack::JackAlsaDriver("system", "alsa_pcm", engine, table);
    Jack::JackDriverClientInterface* threaded_driver = new Jack::JackThreadedDriver(g_alsa_driver);
    // Special open for ALSA driver...
    if (g_alsa_driver->Open(info) != AUDIO_DRIVER_FAILED) {
        return threaded_driver;
    } else {
        delete threaded_driver; // Delete the decorated driver
        return NULL;
    }
}

// Code to be used in alsa_driver.c
#ifndef UNITTEST

void ReadInput(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nread)
{
    g_alsa_driver->ReadInputAux(device, orig_nframes, contiguous, nread);
}
void MonitorInput()
{
    g_alsa_driver->MonitorInputAux();
}
void ClearOutput()
{
    g_alsa_driver->ClearOutputAux();
}
void WriteOutput(alsa_device_t *device, jack_nframes_t orig_nframes, snd_pcm_sframes_t contiguous, snd_pcm_sframes_t nwritten)
{
    g_alsa_driver->WriteOutputAux(device, orig_nframes, contiguous, nwritten);
}
void SetTime(jack_time_t time)
{
    g_alsa_driver->SetTimetAux(time);
}

/*
This function is called during XRUN recovery by alsa_driver_restart().
Hence return alsa_driver_status_t.
*/

alsa_driver_status_t Restart()
{
    int res = g_alsa_driver->Stop();
    if (res != AUDIO_DRIVER_OK && res != AUDIO_DRIVER_RESET) {
        jack_error("restart: stop driver failed");
        return ALSA_DRIVER_ERROR;
    }
    /* ResetDriverTargetState() is called during Stop() failure due to SSR
       Hence its okay to call Start() */
    if (g_alsa_driver->Start() != AUDIO_DRIVER_OK) {
        jack_error("restart: start driver failed");
        return ALSA_DRIVER_ERROR;
    }
    return (res == AUDIO_DRIVER_RESET ? ALSA_DRIVER_RESET : ALSA_DRIVER_OK);
}
#endif

#ifdef __cplusplus
}
#endif


