/*!*************************************************************************
 * FILE :			ArabicStringUtil.h
 *
 * DESCRIPTION:	This module provides methods for processing arabic text 
 *                to make it usable in applications which don't support Arabic.
 *
 *                Parts of the core encoder functionality (ReshapeText) was 
 *                adapted from Mohammad (ohammadhi@gmail.com). His
 *                c# code is under MIT and GPL license (Version: 1.0.4)
 *
 *                I took parts of the implementation from the modules 
 *                Arabix.cs and TextUtils.cs and adapted this for the 
 *                ReshapeText method.
 *
 *                You will find the original c# source code here:
 *                  http://sourceforge.net/projects/tadween/files/
 *
 * AUTHOR:			Helmut.Frericks@de.bosch.com
 *
 * COPYRIGHT:     (c) 2005 Robert Bosch GmbH, Hildesheim
 *
 * HISTROY:
 *
 *  28.02.11	CM-AI/PJ-VW37 Helmut Frericks (fr83hi)  Initial Version
 *  09.05.11	CM-AI/PJ-VW37 Helmut Frericks (fr83hi)  Memory leak,maintenance 
 *  22.05.11	CM-AI/PJ-VW37 Helmut Frericks (fr83hi)  rework of MirrorNonArabic
 *  26.06.11	CM-AI/PJ-VW37 Helmut Frericks (fr83hi)  corrections MirrorNonArabic
 *  14.09.11    CM-AI/PJ-VW37 Helmut Frericks (fr83hi) performance update
 *  21.09.11    CM-AI/PJ-VW37 Helmut Frericks (fr83hi) Arabic-Indic digits (U+660 - U+669) 
                implemented. These numbers are widely used in Arabic countries and it's 
                very essential to be supported.
 *  19.10.11    alloca crashes with compressed audio ( multithreaded ?? )
                repleced with slower function OSAL_NEW
 ***************************************************************************/
/***************************************************************************
 *                                                                      
 *  Version: 1.0.4 - 03.07.2009                                         
 *  Copyright (c) 2009 Mohammad Arabix (mohammadhi at gmail d0t com)    
 *  Dual licensed under the MIT and GPL licenses.                       
 *                                                                     
 ***************************************************************************/


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
/*
#if ((OSAL_OS!=OSAL_TENGINE) || (ARM))
	#if (OSAL_OS==OSAL_WINNT || OSAL_OS==OSAL_WINCE) 
		#include <malloc.h>
	#else
		#include <alloca.h>
	#endif
#endif
*/

/*
#ifdef PROJECTBUILD_NISSAN_LCN2
# define VARIANT_S_FTR_ENABLE_CONVERT_ARABICDIGITS
#endif
*/
#include "cUtf8String.h"
#include "ArabicStringUtil.h"

#ifdef __TIME_MESSURE_ARABIC

#include <windows.h>

class cTicker
{   
  public:
   cTicker() { Reset(); }
  public:
   void Reset()
   {  
		::QueryPerformanceFrequency((LARGE_INTEGER*)&cTicker::m_i64Frequency);
      m_i64TickStart= 0; 
      m_i64TickSum  = 0; 
   }
   void Start() 
   {  ::QueryPerformanceCounter((LARGE_INTEGER*)&m_i64TickStart); 
   }
   void Stop()  
   {  __int64 i64TickStop; ::QueryPerformanceCounter((LARGE_INTEGER*)&i64TickStop);
      m_i64TickSum += (i64TickStop-m_i64TickStart);
   }
   DWORD DiffUsec()
   {	// Gesamtzeit in usec seid Start()  (maximal 70 Minuten)
      __int64 i64diff = (m_i64TickSum * 1000000 / m_i64Frequency);
      return DWORD(i64diff);
   }
    DWORD   DiffMsec() { return (DiffUsec()/1000); }
 // private:
	  static __int64 m_i64Frequency; 
	__int64 m_i64TickReset;
	__int64 m_i64TickStart;
	__int64 m_i64TickSum;
};

__int64 cTicker::m_i64Frequency;

static int nStrings;
static cTicker tStep1_NEW1;
static cTicker tStep1_IsArabic;
static cTicker tStep2_Reshape;
static cTicker tStep3_MirrorNonArabic;
static cTicker tStep4_UTF16_8;

#endif


class cArabicTextCoder
{
  public:
    cArabicTextCoder ();
    /// Reshapes the characters of the specified text into the appropriate ligature 
    /// according to their context.
    int ReshapeText (tUTF16char * textout, int utf16CharLen) const;
  private:
    /// Checks if the specified character is the arabic ALEF letter.
    inline bool IsAlefChar ( tUTF16char ch ) const;
    /// Checks if the specified character is the arabic LAM letter. 
    inline bool IsLamChar ( tUTF16char ch ) const;
    /// Checks if the specified character is one of the arabic LAMALEF letters (لا، لأ، لإ، لآ).
    inline bool IsLamAlefChar ( tUTF16char ch ) const;
    /// Checks if the specified character doesn't affect the ligature of the preceeding and succeeding letters.
    inline bool IsTransparentChar ( tUTF16char ch ) const;
    /// Checks if the specified character should be reshaped.
    inline bool InLinkRange ( tUTF16char ch ) const;
    /// Checks if the specified character should be reshaped.
    inline bool InUnlinkRange ( tUTF16char ch );
    /// Checks if the specified character can have a link with the preceeding letter.
    inline bool IsLinkableBefore ( tUTF16char ch ) const;
    /// Checks if the specified character can have a link with the suceeding letter.
    inline bool IsLinkableAfter ( tUTF16char ch ) const;

    /// Checks if the specified character is an Arabic Indic digit (U+660-669)
    inline bool IsArabicIndicDigit ( tUTF16char ch ) const;
    /// Checks if the specified character is an extended Arabic Indic digit (U+6F0-6F9)
    inline bool IsPersianArabicIndicDigit ( tUTF16char ch ) const;

    /// Checks if the specified character is a arabic Seperator ( localization Europe= ',', NAR='.' )
    inline bool IsArabicSeparator ( tUTF16char ch ) const { return (ch == _UNICODE_(0x066B)); }

    /// Gets the link type of the specified character.
    unsigned char GetCharLinkType ( tUTF16char ch ) const;
    /// Reshapes the specified character accourding to the specified context.
    tUTF16char LinkChar ( tUTF16char ch, int context ) const;
    /// Return the specified character to its original form.
    tUTF16char LinkLamAlef ( tUTF16char chAlef, int context ) const;
};


// 'Zero-Width No-Break Space' character, not used
// const tUTF16char  ZWNBSP = _UNICODE_(0xFEFF);

// Letter contexts
const int ISOLATED = 0;  // there is no letter precedes or succeeds this litter (one-letter word!)
const int FINAL	 = 1;	 // the last letter within a word
const int INITIAL  = 2;  // the first letter within a word
const int MEDIAL	 = 3;	 // there is a letter precedes this litter and a letter succeeds it


typedef unsigned char tLinkType; 

const tLinkType NONE	= 0;	   // like HAMZA	(ء)
const tLinkType BEFORE	= 1;	// like DAL		(د)
const tLinkType DUAL	= 2;	   // like JEEM	(ج)
const tLinkType CAUSING = 3;	// like TATWEEL (ـ)


// The link types of arabic characters
const tLinkType aArabicCharacterTypeTable[] =
{  // 0621 - 064A
   NONE, BEFORE, BEFORE, BEFORE, BEFORE, DUAL, BEFORE, DUAL, BEFORE, DUAL, DUAL, DUAL, DUAL, DUAL, BEFORE, BEFORE,
   BEFORE, BEFORE, DUAL, DUAL, DUAL, DUAL, DUAL, DUAL, DUAL, DUAL, NONE, NONE, NONE, NONE, NONE, CAUSING, DUAL, DUAL,
   DUAL, DUAL, DUAL, DUAL, DUAL, BEFORE, DUAL, DUAL
};

// Letter mapping
#define LINK_MAP_RANGE_0 _UNICODE_(0x0621)
#define LINK_MAP_RANGE_1 _UNICODE_(0x064A)
const tUTF16char  aLetterMapLinkTable[] =
{   // 0621 - 064A
   _UNICODE_(0xFE80), _UNICODE_(0xFE81), _UNICODE_(0xFE83), _UNICODE_(0xFE85), _UNICODE_(0xFE87), _UNICODE_(0xFE89), _UNICODE_(0xFE8D), _UNICODE_(0xFE8F),
   _UNICODE_(0xFE93), _UNICODE_(0xFE95), _UNICODE_(0xFE99), _UNICODE_(0xFE9D), _UNICODE_(0xFEA1), _UNICODE_(0xFEA5), _UNICODE_(0xFEA9), _UNICODE_(0xFEAB),
   _UNICODE_(0xFEAD), _UNICODE_(0xFEAF), _UNICODE_(0xFEB1), _UNICODE_(0xFEB5), _UNICODE_(0xFEB9), _UNICODE_(0xFEBD), _UNICODE_(0xFEC1), _UNICODE_(0xFEC5),
   _UNICODE_(0xFEC9), _UNICODE_(0xFECD), _UNICODE_(0x063B), _UNICODE_(0x063C), _UNICODE_(0x063D), _UNICODE_(0x063E), _UNICODE_(0x063F), _UNICODE_(0x0640),
   _UNICODE_(0xFED1), _UNICODE_(0xFED5), _UNICODE_(0xFED9), _UNICODE_(0xFEDD), _UNICODE_(0xFEE1), _UNICODE_(0xFEE5), _UNICODE_(0xFEE9), _UNICODE_(0xFEED),
   _UNICODE_(0xFEEF), _UNICODE_(0xFEF1)
};

// Geht dem Alif ein Lam voran, wird die Ligatur (Lam-Alif) verwendet.
#define LAMALEF_LINK_MAP_RANGE_0 _UNICODE_(0x0622)
// #define LAMALEF_LINK_MAP_RANGE_1 _UNICODE_(0x0627) 
const tUTF16char  aLamAlefMapLinkTable[] =
{   // 0622 - 0627
    _UNICODE_(0xFEF5), _UNICODE_(0xFEF7), _UNICODE_(0x0624), _UNICODE_(0xFEF9), _UNICODE_(0x0626), _UNICODE_(0xFEFB)
};


//#define LAMALEF_UNLINK_MAP_RANGE_0  _UNICODE_(0xFEF5)
//#define LAMALEF_UNLINK_MAP_RANGE_1  _UNICODE_(0xFEFC) 
//const tUTF16char aLamAlefMapUnLinkTable[] =
//{
//    _UNICODE_(0x0622), _UNICODE_(0x0622), _UNICODE_(0x0623), _UNICODE_(0x0623), _UNICODE_(0x0625), _UNICODE_(0x0625), _UNICODE_(0x0627), _UNICODE_(0x0627)
//};

/// returns the length (count bytes, not characters) of a UTF8 string
#define nUTF8strlen(_a_) strlen((const char *)(_a_))

/// returns the length (in characters) of a UTF16 string
static int nUTF16CharLen( const tUTF16char* text)
{
   int count = 0;
   while(*text++ != 0)
      count++;
   return count;
}

// ctor.
cArabicTextCoder::cArabicTextCoder () 
{ 
}

/// Checks if the specified character is the arabic ALEF letter.
bool cArabicTextCoder::IsAlefChar ( tUTF16char ch ) const
{
    return (ch == _UNICODE_(0x0622) || ch == _UNICODE_(0x0623) || ch == _UNICODE_(0x0625) || ch == _UNICODE_(0x0627));
}

/// Checks if the specified character is the arabic LAM letter. 
bool cArabicTextCoder::IsLamChar ( tUTF16char ch ) const
{
    return (ch == _UNICODE_(0x0644));
}

/// Checks if the specified character is one of the arabic LAMALEF letters (لا، لأ، لإ، لآ).
bool cArabicTextCoder::IsLamAlefChar ( tUTF16char ch ) const
{
    return (ch >= _UNICODE_(0xFEF5) && ch <= _UNICODE_(0xFEFC));
}

/// Checks if the specified character doesn't affect the ligature of the preceeding 
/// and succeeding letters.
bool cArabicTextCoder::IsTransparentChar ( tUTF16char ch ) const
{
    return (ch >= _UNICODE_(0x064B) && ch <= _UNICODE_(0x065E));
}

/// Checks if the specified character is an Arabic Indic digit (U+660-669)
bool cArabicTextCoder::IsArabicIndicDigit ( tUTF16char ch ) const
{
    return (ch >= _UNICODE_(0x0660) && ch <= _UNICODE_(0x0669));
}

 /// Checks if the specified character is an extended Arabic Indic digit (U+6F0-6F9)
bool cArabicTextCoder::IsPersianArabicIndicDigit ( tUTF16char ch ) const
{
    return (ch >= _UNICODE_(0x06F0) && ch <= _UNICODE_(0x06F9));
}

/// Checks if the specified character should be reshaped.
bool cArabicTextCoder::InLinkRange ( tUTF16char ch ) const
{
    return (ch >= LINK_MAP_RANGE_0 && ch <= LINK_MAP_RANGE_1);
}

/// Checks if the specified character can have a link with the preceeding letter.
bool cArabicTextCoder::IsLinkableBefore ( tUTF16char ch ) const
{
    if (!InLinkRange(ch))
        return false;

    int     ii = (ch - LINK_MAP_RANGE_0);
    return (aArabicCharacterTypeTable[ii] != NONE);
}

/// Checks if the specified character can have a link with the suceeding letter.
bool cArabicTextCoder::IsLinkableAfter ( tUTF16char ch ) const
{
   if (!InLinkRange(ch))
      return false;

   int ii = (ch - LINK_MAP_RANGE_0);
   int linkType = aArabicCharacterTypeTable[ii];
   return (linkType == DUAL || linkType == CAUSING);
}

/// Gets the link type of the specified character.
tLinkType cArabicTextCoder::GetCharLinkType ( tUTF16char ch ) const
{
   if (!InLinkRange(ch))
      return NONE;

   int ii = (ch - LINK_MAP_RANGE_0);
   return aArabicCharacterTypeTable[ii];
}

/// Reshapes the specified character accourding to the specified context.
tUTF16char cArabicTextCoder::LinkChar ( tUTF16char ch, int context ) const
{
   if (!InLinkRange(ch))
      return ch;

   int ii = (ch - LINK_MAP_RANGE_0);
   switch (aArabicCharacterTypeTable[ii])
   {
    case NONE:
        return (tUTF16char) (aLetterMapLinkTable[ii]); // return the isolated form

    case BEFORE:
        return (tUTF16char) (aLetterMapLinkTable[ii] + (context % 2)); // return the proper form (isolated or final)

    case DUAL:
        return (tUTF16char) (aLetterMapLinkTable[ii] + context); // return the proper form (all forms applicable)

    case CAUSING:
    default:
        return ch;
   }
}

/// Reshapes the specified LAMALEF character accourding to the specified context.
tUTF16char cArabicTextCoder::LinkLamAlef ( tUTF16char chAlef, int context ) const
{
   if (!IsAlefChar(chAlef))
      return chAlef;

   int ii = chAlef - LAMALEF_LINK_MAP_RANGE_0;
   return (tUTF16char) ((int) aLamAlefMapLinkTable[ii] + (context % 2));
}

/// Reshapes the characters of the specified text into the appropriate ligature 
/// according to their context.
int cArabicTextCoder::ReshapeText (tUTF16char * textout, int utf16CharLen) const
{
   if(textout == NULL)
       return 0;

   tUTF16char * newText = textout;

   int nTextinLength = utf16CharLen;

#ifdef WIN32
   // do some additional check for simulation 
   nTextinLength = nUTF16CharLen(textout);
   FATAL_M_ASSERT(nTextinLength == utf16CharLen);
#endif

   int lastContext = ISOLATED;
   int decrease = 0;
   int charIndex = 0;

   while(charIndex < utf16CharLen)
   {
      tUTF16char ch = textout[charIndex];
      
      if (ch == 0)
         break;

      if (GetCharLinkType(ch) == CAUSING) 
      {
         // the character is the TATWEEL letter (ـ)
         newText[charIndex - decrease] = ch; // copy the character to the output 
         lastContext = MEDIAL;
         charIndex++;
         continue;
      }
      if (IsTransparentChar(ch)) 
      {
         // the character is transparent (like FATHA, DAMMA, KASRA..) // 0x064B-0x0652, und evtl folgende
         // vowel after base consonant
         newText[charIndex - decrease] = ch; // just copy the character to the output 
         charIndex++;
         continue;
      }
      int nextCharIndex = charIndex + 1; // determine the next non transparent character

      while (nextCharIndex < nTextinLength - 1 && IsTransparentChar(textout[nextCharIndex]))
         nextCharIndex++;

      int context = (lastContext == INITIAL || lastContext == MEDIAL) ? FINAL : ISOLATED;


      if (nextCharIndex < nTextinLength)
      {
         // Special Case: Lam-Alef
         if (IsLamChar(ch) && IsAlefChar(textout[nextCharIndex]))
         {
            newText[charIndex - decrease] = LinkLamAlef(textout[nextCharIndex], context);
            lastContext = context;
            // bShrink : determines whether the resulted gaps in the text should be 
            // removed or just filled with ZWNBSP characher
            memcpy(&newText[charIndex + 1 - decrease], &textout[charIndex + 1], nextCharIndex - charIndex - 1);
            decrease++;
            charIndex = nextCharIndex;
            charIndex++;
            continue;
         }

         if (IsLinkableAfter(ch) && IsLinkableBefore(textout[nextCharIndex]))
            context |= INITIAL;
      }
      newText[charIndex - decrease] = LinkChar(ch, context);
      lastContext = context;
      charIndex++;
   }
   utf16CharLen = (charIndex-decrease); 
   FATAL_M_ASSERT(utf16CharLen >= 0);
   textout[utf16CharLen] = '\0';
   return utf16CharLen;
}


///////////////////////////////////////////////////////////////////////////////////////////

/// ctor
cArabicStringUtil::cArabicStringUtil()
{
   _strResultConst   = (const tUTF8char *)"\0";
   _strResultAlloc   = NULL;
   _cursorGlyph      = 0;
   _szUTF16WorkSize  = 0;
   _szUTF16MaxChars  = 0;
   _szUTF16Work      = NULL; 
}

cArabicStringUtil::cArabicStringUtil(const tUTF8char *utf8Text, bool bMirrorBuffer,tUTF16char cursorGlyph)
{
   _strResultConst   = (const tUTF8char *)"\0";
   _strResultAlloc   = NULL;
   _cursorGlyph      = cursorGlyph;
   _szUTF16WorkSize  = 0;
   _szUTF16MaxChars  = 0;
   _szUTF16Work      = NULL; 
   ProcessArabic(utf8Text, bMirrorBuffer);
}

/// dtor. This one deletes all contained elements.
cArabicStringUtil::~cArabicStringUtil()
{
   Clear();
   if (_szUTF16Work != NULL)
   {
      OSAL_DELETE [] _szUTF16Work;
   }
   _szUTF16WorkSize  = 0;
   _szUTF16MaxChars  = 0;
   _szUTF16Work      = NULL; 
}

/// deletes all allocated memory and resets all members
tVoid cArabicStringUtil::Clear() 
{
   if (_strResultAlloc != NULL)
   {
      OSAL_DELETE [] _strResultAlloc;
   }
   _strResultConst = (const tUTF8char *)"\0";
   _strResultAlloc = NULL;
}

/// result string operator
cArabicStringUtil::operator const tUTF8char * () const      
{ 
   return get(); 
}

/// returns the length of the result string    
int cArabicStringUtil::length () const      
{ 
   if (_strResultAlloc != NULL )
      return nUTF8strlen(_strResultAlloc);
   if (_strResultConst != NULL )
      return nUTF8strlen(_strResultConst);
   return 0;
}



/// Main Method to convert Arabic Strings
/// Param
///   utf8Text      : Input string 
///   bMirrorBuffer : For singleline strings; if true, string will be mirrored  
tBool cArabicStringUtil::ProcessArabic( const tUTF8char * utf8Text, bool bMirrorBuffer ) 
{
   Clear();

   int utf8len  = nUTF8strlen(utf8Text);

   // Step 0: shortcut, if text has no content
   if(utf8len == 0)
   {
      _strResultConst = utf8Text;
      return FALSE;
   }

#ifdef __TIME_MESSURE_ARABIC
   tStep1_NEW1.Start();
#endif
   // 
   // Calculate and get memory for utf16 working string 
   // memory worst case is a pure ascii string 
   // for speed we allocate the working string once, 
   //  so the calling application should keep the cArabicStringUtil object 
   // 
   int utf16MemSize  = utf8len*2+2;
   if (_szUTF16WorkSize < utf16MemSize)
   {
      _szUTF16WorkSize = utf16MemSize;
      if (_szUTF16WorkSize < 500)
         _szUTF16WorkSize = 500;
      if (_szUTF16Work != NULL)
      {
         OSAL_DELETE [] _szUTF16Work;
      }
      _szUTF16MaxChars  = _szUTF16WorkSize/2;
      _szUTF16Work = OSAL_NEW tUTF16char[_szUTF16MaxChars];
      if(_szUTF16Work == NULL)
      {
         #ifdef __TIME_MESSURE_ARABIC
         tStep1_NEW1.Stop();
         #endif
         _strResultConst = (const tUTF8char *)("##Internal arabic coder error1");
         return FALSE;
      }
   }

#ifdef __TIME_MESSURE_ARABIC
   tStep1_NEW1.Stop();
   nStrings++;
   tStep1_IsArabic.Start();
#endif
   // Step 1: UTF8 -> UTf16 for better processing and
   // check for the availability of arabic characters
   bool bContainsArabic = false;
   bool bContainsNumeric = false;

   bContainsNumeric = bContainsNumeric; //to make Lint happy

   cUtf8StringBase s(utf8Text);
   int utf16CharLen=0;
   for(cUtf8StringIter it = s.itBegin(); it<s.itEnd() && utf16CharLen < _szUTF16MaxChars; it=s.itNext(it))
   {
      tUTF16char ch = (tUTF16char) s.GetCharCode(it);
      if(TextUtils::IsArabicChar(ch))
         bContainsArabic=true;
      #ifdef VARIANT_S_FTR_ENABLE_CONVERT_ARABICDIGITS
      else if(TextUtils::IsLatinNumeric(ch))
         bContainsNumeric=bContainsArabic=true;
      #ifdef PROJECTBUILD_NISSAN_LCN2
      else if(TextUtils::IsNumericMonoSpaced(ch))
         bContainsNumeric=bContainsArabic=true;
      #endif
      #endif
      _szUTF16Work[utf16CharLen] = ch;
      utf16CharLen++;
   }
   _szUTF16Work[utf16CharLen] = 0;
#ifdef __TIME_MESSURE_ARABIC
   tStep1_IsArabic.Stop();
#endif
   // return immediatelly if no arabic characters found or there is only 1 character
#ifdef VARIANT_S_FTR_ENABLE_CONVERT_ARABICDIGITS
   if((!bContainsArabic) || ((utf16CharLen < 2) && !bContainsNumeric))
   {
      _strResultConst = utf8Text;
      if (!bContainsNumeric)
         return FALSE;
   }
#else
   if((!bContainsArabic) || (utf16CharLen < 2))
   {
      _strResultConst = utf8Text;
      return FALSE;
   }
#endif

   // Step 2: reshape the text
   // wszText: out = in to avoid memory allocation for each operation
#ifdef __TIME_MESSURE_ARABIC
   tStep2_Reshape.Start();
#endif
   cArabicTextCoder coder;
   utf16CharLen = coder.ReshapeText (_szUTF16Work, utf16CharLen);
#ifdef __TIME_MESSURE_ARABIC
   tStep2_Reshape.Stop();
#endif
   // For mixed mode we add a cursor glyph to show the cursor position
   // inside the string e.g. " (CALC)|إعادة حساب الطريق "
   if(_cursorGlyph != 0)
   {
      _szUTF16Work[utf16CharLen++] = _cursorGlyph;
      _szUTF16Work[utf16CharLen  ] = 0;
   }

   // Step 3: Mirror the Non Arrabic chars
#ifdef __TIME_MESSURE_ARABIC
   tStep3_MirrorNonArabic.Start();
#endif
   utf16CharLen = MirrorNonArabic (_szUTF16Work, utf16CharLen);
#ifdef VARIANT_S_FTR_ENABLE_CONVERT_ARABICDIGITS
   utf16CharLen = ConvertArabicDigits (_szUTF16Work, utf16CharLen);
#endif

#ifdef __TIME_MESSURE_ARABIC
   tStep3_MirrorNonArabic.Stop();
#endif
   // Step 4:
   // Mirror all chars to display "hello" as "olleh"; only useful if you have 
   // no multiline text; otherwise call ProcessArabic without Mirror option an 
   // call for each substring the MirrorUTF8String() of this class.
   if(bMirrorBuffer == true)
   {
      TextUtils::MirrorUTF16String(_szUTF16Work);
   }
   // Step 4: UTF16 -> UTF8
#ifdef __TIME_MESSURE_ARABIC
   tStep4_UTF16_8.Start();
#endif
   cUtf8String str;
   _strResultAlloc = OSAL_NEW tUTF8char[utf16CharLen*4];
   str.vInit(_strResultAlloc, utf16CharLen*4);
   str.vSet16bitStr(_szUTF16Work);
   //int slen = str.wGetLength();
#ifdef __TIME_MESSURE_ARABIC
   tStep4_UTF16_8.Stop();
#endif
   return TRUE;
}

/// Convert the arabic numeric chars, numeric characters should be shown with Arabic Indic digit (U+660- U+669)
int cArabicStringUtil::ConvertArabicDigits (tUTF16char * szUTF16Work, unsigned utf16CharLen ) const
{
   for (unsigned i = 0; i <= utf16CharLen && szUTF16Work[i]; i++)
   {
      if (TextUtils::IsLatinNumeric(szUTF16Work[i]))
      {  // 0x30 -> 0x660
         szUTF16Work[i] += 0x630;
      }
      #ifdef PROJECTBUILD_NISSAN_LCN2
      else if(TextUtils::IsNumericMonoSpaced(szUTF16Work[i]))
      {  // 0xF830 -> 0x660
         szUTF16Work[i] -= 0xF1D0;
      }
      #endif
   }
   return utf16CharLen;
}

/// Mirror the non arabic chars like latin text and also arabic numbers 
int cArabicStringUtil::MirrorNonArabic (tUTF16char * szUTF16Work, unsigned utf16CharLen ) const
{

   if(szUTF16Work == NULL)
       return 0;

    //tUTF16char * result = (tUTF16char *)alloca((utf16CharLen+1)*2);
   tUTF16char * szUTF16Temp = OSAL_NEW  tUTF16char[utf16CharLen+1];

   if(szUTF16Temp == NULL)
       return 0;


#ifdef WIN32
   // for debug session to have always a terminated string
   memset(szUTF16Temp,0,((utf16CharLen+1)*2));
#endif

   unsigned resultIndex = 0;

   bool bIsArabic  = false;
   bool bInArabic  = true;

   bool bIsLatin   = false;
   bool bInLatin  = false;

   bool bIsSymbol = false;
   bool bIsBracketOpen = false;
   bool bIsBracketClose = false;
   bool bInBracketArabic = false;
   bool bIsSpecialSeperatorNearArabic = false;
   bool bIsNearArabic = false;

   bool bIsLineBreak = false;
   bool bIsNextArabic = false;
   bool bIsPrevArabic = false;


   int iLast = 0;

   tUTF16char ch = 0xFFFF;

   for (unsigned i = 0; i <= utf16CharLen && ch != 0; i++)
   {
      ch = szUTF16Work[i];

      bIsNextArabic = ((i < utf16CharLen) && TextUtils::IsArabicChar(szUTF16Work[i+1]))? true:false;
      bIsPrevArabic = ((i > 0) && TextUtils::IsArabicChar(szUTF16Work[i-1]))? true:false;
      bIsNearArabic = bIsNextArabic || bIsPrevArabic || bIsArabic;
      bIsSpecialSeperatorNearArabic  = (ch == ' ' || ch == '/') && bIsNearArabic;
      bIsBracketOpen = ch == '(' || ch == '[' || ch == '{';
      bIsBracketClose = ch == ')' || ch == ']' || ch == '}';

      /*Exclude brackets from symbols, they are handled separate*/
      bIsSymbol = (bIsBracketOpen || bIsBracketClose)? false:TextUtils::IsSymbol(ch);
      bIsLineBreak  = (ch == 0x0A);

      /*If last character is of Arabic type then consider the Brackets Arabic*/
      if((bIsNearArabic && bIsBracketOpen) || (bInBracketArabic && bIsBracketClose))
      {
          bIsArabic = true;
          if(bIsBracketOpen)
              bInBracketArabic = true;
          else if(bIsBracketClose)
              bInBracketArabic = false;
      }
      else
          bIsArabic = TextUtils::IsArabicChar(ch);

      bIsLatin  = !(bIsSymbol || bIsArabic || bIsLineBreak || bIsSpecialSeperatorNearArabic) && ch != 0;
      
      if (bIsLineBreak || bIsSpecialSeperatorNearArabic || bIsArabic) 
      {  // break off reason: Latin Char   
         bInArabic = true;
      }
      else if(bIsLatin)
      {  // break off reason: Arabic Char 
         bInLatin = true;
      } 

      // copy routines
      if ((ch == 0 || bIsLatin) && bInArabic)
      {  // Ende der arabischen Zeichen 
         // alles kopieren, aber zahlenfelder drehen
         bInArabic = false;
         // if the tUTF16char is Arabic or symbol, then just copy it to output.
         // numeric values has to be shown from left to right
         int numChars=0;
         unsigned j;
         
         for (j = iLast; j < i; j++) 
         {
            if (TextUtils::IsLatinNumeric(szUTF16Work[j]) || (numChars>0 && (j+1<i) && TextUtils::IsSeperator(szUTF16Work[j]) && TextUtils::IsLatinNumeric(szUTF16Work[j+1]) )) 
            {
               numChars++;
            }
            else 
            {  // at first flip the numeric values, if captured
               if (numChars)       
               {  for (unsigned k = j-1; numChars>0; numChars--) 
                     szUTF16Temp[resultIndex++] = szUTF16Work[k--];//lint !e661 !e662 PQM_authorized_multi_235 reason: Array bounds checked by passed parameter
               } 
               // copy the arabic and symbol stuff
               szUTF16Temp[resultIndex++] = TextUtils::MirrorBrackets(szUTF16Work[j]);
            }
         }
         // numeric chars captured ?
         if (numChars) 
         {  for (int k = j-1; numChars>0 && k>=0 && resultIndex<=utf16CharLen; numChars--) 
               szUTF16Temp[resultIndex++] = szUTF16Work[k--];
         }
         iLast = i;
      }
      else if ((ch == 0 || bIsArabic || bIsLineBreak || bIsSpecialSeperatorNearArabic) && bInLatin) 
      {  // flip all chars
         bInLatin = false;
         // Lateinische Zeichen kopieren
         for (int j = i-1; j >= iLast; j--) 
            szUTF16Temp[resultIndex++] = szUTF16Work[j];//lint !e661 !e662 PQM_authorized_multi_235 reason: Array bounds checked by passed parameter
         iLast = i;
      }
    }
    FATAL_M_ASSERT(resultIndex <= utf16CharLen);
    szUTF16Temp[resultIndex++] = 0; // terminate
    memcpy(szUTF16Work,szUTF16Temp,resultIndex*2);
    OSAL_DELETE [] szUTF16Temp;
    return utf16CharLen;
}


/// check for arabic characters inside the string
tBool cArabicStringUtil::TextUtils::ContainsArabic(const tUTF8char *utf8Text)
{
   cUtf8StringBase s(utf8Text);
   for(cUtf8StringIter it = s.itBegin();it<s.itEnd();it=s.itNext(it))
   {
      tUTF16char ch = (tUTF16char)s.GetCharCode(it);
      if(TextUtils::IsArabicChar(ch))
      {
         return TRUE;
      }
#ifdef VARIANT_S_FTR_ENABLE_CONVERT_ARABICDIGITS
      if(TextUtils::IsLatinNumeric(ch))
      {
         return TRUE;
      }
#endif
   }
   return FALSE;
}

/// swaps all characters inside a UTF8 string
tVoid cArabicStringUtil::TextUtils::MirrorUTF8String(tUTF8char * data) 
{

    if(data == NULL)
       return;
   // reverse the string in case of right2left direction
   cUtf8String sDst,sSrc;
   sSrc.vAssign(data);

   int nDstSize = nUTF8strlen(data) +1;

   //tUTF8char *pTemp = (tUTF8char *)alloca(nDstSize);
   tUTF8char *pTemp = OSAL_NEW  tUTF8char[nDstSize];

    if(pTemp == NULL)
       return;

   sDst.vInit(pTemp, nDstSize);

   cUtf8StringIter it = sSrc.itEnd();
   // reverse the string first
   while(it > sSrc.itBegin())
   {
      it = sSrc.itPrevious(it);
      sDst.vAppendChar(sSrc.GetCharCode(it));
   }
   memcpy(data, pTemp, nDstSize);
   OSAL_DELETE [] pTemp;
}

/// swaps all characters inside a UTF16 string
tVoid cArabicStringUtil::TextUtils::MirrorUTF16String(tUTF16char * data) 
{  // flip all the words
   int len = nUTF16CharLen(data);
   int i=0;
   if (len > 1)
   {  tUTF16char temp;
      for (int j = len-1; j > i; j--, i++) 
      {
          temp = data[i];  
          data[i] = data[j];
          data[j] = temp;  
      }
   }
}

/// Returns the mirror symbol of the specified character if it exists.
tUTF16char cArabicStringUtil::TextUtils::MirrorBrackets ( tUTF16char ch )
{
    switch (ch)
    {
    case '(':
        return ')';
    case ')':
        return '(';
    case '{':
        return '}';
    case '}':
        return '{';
    case '[':
        return ']';
    case ']':
        return '[';
    case '<':
        return '>';
    case '>':
        return '<';
    default:
        return ch;
    }
}


#ifdef __TIME_MESSURE_ARABIC
void cArabicStringUtil::PrintMeasurements ()
{     char buf[100];
      sprintf(buf,"Step1 New1     %10ld, %d Strings converted\n",tStep1_NEW1.DiffUsec(),nStrings); OutputDebugString(buf);
      sprintf(buf,"Step1 IsArabic %10ld\n",tStep1_IsArabic.DiffUsec()); OutputDebugString(buf);
      sprintf(buf,"Step2 Reshape  %10ld\n",tStep2_Reshape.DiffUsec()); OutputDebugString(buf);
      sprintf(buf,"Step3 Mirror   %10ld\n",tStep3_MirrorNonArabic.DiffUsec()); OutputDebugString(buf);
      sprintf(buf,"Step4 16->8    %10ld\n",tStep4_UTF16_8.DiffUsec()); OutputDebugString(buf);
}
#endif

