/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/include/osgUtil/Optimizer
List of changes:
1. Support to replace a single texture in an atlas.
*/

#ifndef TEXTURE_BATCHING_STRATEGY_ATLAS_HEADER
#define TEXTURE_BATCHING_STRATEGY_ATLAS_HEADER

#include "oss/textures/Optimizer.h"
#include "TextureBatchingStrategyBase.h"
#include "oss/Export.h"

namespace mapengine {
namespace oss {

class MAPENGINE_OSS_EXPORT me_TextureBatchingStrategyAtlas : public me_TextureBatchingStrategyBase, public mapengine::oss::Optimizer::TextureAtlasBuilder 
{
public:
   /**
   * @brief Add texture to the batch
   * 
   * Add the textures to the batch. Generate and store the texture co-ordinates.
   * 
   * @param[in] imageId the image ID
   * @param[in] imageData vector containing an image's data
   * @param[in] textureParam The texture parameters
   * @param[in] frameNumber The frame number at the time of function call
   *
   * @return Returns false if the image could not be added
   */
   virtual bool addImage(unsigned int imageId, const std::vector<unsigned char>& imageData, const TextureParameters& textureParam, unsigned int frameNumber);

   /**
   * @brief Texture query function
   * 
   * Checks whether the texture is present in the atlas
   *
   * @param[in] textureID The texture to be looked for
   * @param[in] batchID The preferred batchID
   * @param[out] batchedTexture The batched texture 
   * @param[out] texCoords The texture coords of the texture referenced by textureID
   *
   * @return Return true if the texture is present, else false
   */
   virtual bool isTexturePresent(unsigned int textureID, unsigned int atlasID,
      osg::ref_ptr<osg::Texture>& batchedTexture, osg::ref_ptr<osg::Array>& texCoords);

   virtual bool isTexturePresent(unsigned int textureID, unsigned int atlasID,
      osg::ref_ptr<osg::Texture>& batchedTexture, osg::ref_ptr<osg::Array>& texCoords,
      unsigned int frameNumber);

   /**
   * @brief Get the atlas texture and the texture co-ordinates for the specified texture
   * 
   * Looks through the atlas list and the source list for the specified texture.
   * If the texture is not present, then the texture is added and the atlas texture and the texture co-ordinates are returned.
   * The frame number of the corresponding source is also updated
   *
   * @param[in] frameNumber FrameNumber at the time of function call
   * @param[in] inputTexture Input texture for which the texture and texture co-ordinates is to be fetched/generated
   * @param[in] textureID TextureID for the icon
   * @param[in] textureParam The texture parameters         
   * @param[out] texCoords The calculated texture co-ordinates for the input texture

   * @return The wallpaper/atlas texture to which the specified texture belongs is returned

   * @remarks 
   * The parameter size is for the creation of new atlas. If the atlas already exists, making a call to the function with a new size value
   * will not resize the atlas. 
   * The texture co-ordinates are in the order - Top Right, Top Left, Bottom Left, Bottom Right
   */
   virtual osg::Texture* addTextureObject(unsigned int textureID, osg::Texture2D* inputTexture, unsigned int frameNumber,
      const TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords);

   virtual osg::Texture* addTextureObject(unsigned int textureID, osg::Image* inputImage, unsigned int frameNumber,
      const me_TextureBatchingStrategyBase::TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords);

   virtual osg::Texture* replaceTextureObject(unsigned int textureID, osg::Texture2D* inputTexture, unsigned int frameNumber,
      const TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords);

   virtual osg::Texture* replaceTextureObject(unsigned int textureID, osg::Image* inputImage, unsigned int frameNumber,
      const me_TextureBatchingStrategyBase::TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords);

   virtual osg::Array* createTexCoordArray() const;
   virtual void addToTexCoordArray(osg::Array* tcArray, const osg::ref_ptr<osg::Array>& texCoordsToAdd) const;
   virtual void addToTexCoordArray(std::vector<osg::Vec3>& tcArray, const osg::ref_ptr<osg::Array>& texCoordsToAdd) const;
   virtual void addDefaultsToTexCoordArray(osg::Array* tcArray) const;
   virtual void replaceEndWithDefaultsForTexCoordArray(osg::Array* tcArray) const;
   virtual void clearTexCoordArray(osg::Array* tcArray) const;
   virtual void setTexCoordPointer(osg::State& state, osg::Array* tcArray, unsigned int vertOffset);
   virtual osg::Texture* createSingleTexture(osg::Image* image) const;
   virtual void removeAllTextures(unsigned int batchID);
   virtual OpenThreads::ReentrantMutex& getMutex() { return _mutex; }

   /**
   * We parse through all images present in the atlasmap and dump them in this function
   */
   virtual void dumpBatches(const std::string& dumpPath);

   virtual bool getSourceTextureDimensions(unsigned int textureID, unsigned int atlasID, unsigned int& width, unsigned int& height);

   void updateAllDirtyContentFlags();

   OpenThreads::ReentrantMutex _mutex;

protected:
   osg::Texture* addSource(osg::Texture2D* texture,
      unsigned int textureID,
      unsigned int frameNumber,
      const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
      osg::ref_ptr<osg::Array>& texCoords);

   class MAPENGINE_OSS_EXPORT AtlasSource : public mapengine::oss::Optimizer::TextureAtlasBuilder::Source
   {
   public:
      /** Constructor */
      AtlasSource() : _id(0), _textureCoords(NULL), _frameNumber(0) {}

      /**
      * Constructor
      * @param image The image to the source
      * @param textureID The preferred/generated textureID
      */
      AtlasSource(const osg::Image* image,unsigned int textureID) : Source(image), _id(textureID), _textureCoords(NULL), _frameNumber(0)
      {
      }

      /** 
      * Constructor
      * @param texture The 2D texture to the source
      * @param textureID The preferred/generated textureID
      */
      AtlasSource(const osg::Texture2D* texture, unsigned int tID) : Source(texture), _id(tID), _textureCoords(NULL), _frameNumber(0)
      {
      }

      /** 
      * @brief Set the source id 
      * @param id The ID for the source
      */
      inline void setID(unsigned int id);

      /** 
      * @brief Set the texture co-ordinates 
      * @param texCoords The texture co-ordinates of the source
      */
      inline void setTextureCoords(osg::Vec2Array* texCoords);

      /** 
      * @brief Set the frame number 
      * @param frameNumber Framestamp for the source
      */
      inline void setFrameNumber(unsigned int frameNumber);

      /**
      * @brief Get the source id 
      * @return Returns the source ID
      */
      inline unsigned int getID() const;

      /** 
      * @brief Get the texture co-ordinates 
      * @return Returns the texture co-ordinates of the source
      */
      inline osg::ref_ptr<osg::Vec2Array> getTextureCoords() const;

      /** 
      * @brief Get the frame number 
      * @return Returns the frame number of the source
      */
      inline unsigned int getFrameNumber() const;

   protected:
      virtual ~AtlasSource() 
      {
         _textureCoords = NULL;
      }

   private:
      unsigned int _id; // id for this source
      osg::ref_ptr<osg::Vec2Array>_textureCoords; // texture co-ordinates of this source
      unsigned int _frameNumber; // frame number when the source was last used
   };

   class MAPENGINE_OSS_EXPORT WallpaperAtlas : public mapengine::oss::Optimizer::TextureAtlasBuilder::Atlas
   {
   public:
      /** 
      * Constructor
      * @param width Width of the wallpaper 
      * @param height Height of the wallpaper
      * @param margin Margin for each source in the wallpaper
      */
      WallpaperAtlas(int width, int height, int margin) : Atlas(width, height, margin)
      {
         _atlasFull = false;
         _id = 0;
         _subID = 0;
         _isDirtyUpdatePending = false;
      }

      /** 
      * @brief Copy only a single source 
      * Copies the source image into the atlas image 
      * @param source The source whose image has to be copied
      */
      virtual void copyOnlyThisSource(Source* source, bool dirtyImmediately);

      /** 
      * @brief Set the wallpaper atlas id  
      * @param id The ID for the atlas
      */
      inline void setID(unsigned int id);

      /** 
      * @brief Get the wallpaper atlas id 
      * @return Returns the atlas ID
      */
      inline unsigned int getID() const;

      /** @brief Set the atlas as full */
      inline void setAtlasFull(bool atlasFull);

      /** 
      * @brief Query if atlas is full 
      * @return Returns the status of the atlas
      */
      inline bool isAtlasFull() const;

      /**
      * @brief Set the wallpaper sub ID
      * @param subID The subID for the atlas
      */
      void setSubID(unsigned int subID)
      {
         _subID = subID;
      }

      /**
      * @brief Get the Wallpaper subID
      * @return Returns the wallpaper subID
      */
      unsigned int getSubID() const
      {
         return _subID;
      }

      AtlasSource* getExistingSource(unsigned int textureID);

      bool getDirtyUpdatePending() const { return _isDirtyUpdatePending; }
      void setDirtyUpdatePending(bool val) { _isDirtyUpdatePending = val; }

      void updateContentDirtyFlag();
   protected:
      virtual ~WallpaperAtlas() {}

      bool _atlasFull;     // flag to indicate if the atlas can take any more sources or not

      bool _isDirtyUpdatePending;

   private:
      unsigned int _id;    // id for the atlas
      unsigned int _subID; // sub ID for the atlas
   };

   struct CompareFrame
   {
      bool operator()(Source* src1, Source* src2) const
      {
         return ( static_cast<AtlasSource*>(src1)->getFrameNumber() < static_cast<AtlasSource*>(src2)->getFrameNumber());  //lint  !e1774
      }
   };

   // predicate for erasing source
   struct SourceErase
   {
      SourceErase(WallpaperAtlas* atlas) : _atlas(atlas) {}
      bool operator()(Source* source) const
      {
         return (source->_atlas == _atlas) ? true : false;
      }
      WallpaperAtlas* _atlas;
   };

   /** 
   * @brief Add a new source with the texture and textureID 
   * @param texture The texture to be added to the source
   * @param textureID The ID for the source 
   */
   void addSourceToList(me_TextureBatchingStrategyAtlas::AtlasSource* source);

   bool rebuildAtlas(osg::ref_ptr<AtlasSource> source, osg::ref_ptr<WallpaperAtlas> atlas, unsigned int frameNumber);

   void storeTextureCoordinates(osg::ref_ptr<AtlasSource> source);

   virtual bool replaceSource( Source* oldSrc , Source* newSrc, bool dirtyImmediately);

   osg::Texture2D* createTextureWrapper(osg::Image* inputImage, const me_TextureBatchingStrategyBase::TextureParameters& textureParam) const;

   osg::ref_ptr<WallpaperAtlas> createAtlasAndAddSource(osg::ref_ptr<AtlasSource> atlasSource,
      const me_TextureBatchingStrategyBase::TextureParameters& textureParam, unsigned int subId, unsigned int frameNumber);

   mapengine::oss::me_TextureBatchingStrategyAtlas::WallpaperAtlas* createAtlas(const me_TextureBatchingStrategyBase::TextureParameters& textureParam, int firstTextureWidth, int firstTextureHeight);

   me_TextureBatchingStrategyAtlas::AtlasSource* findSource(unsigned int textureID, unsigned int atlasID);

   typedef std::map<unsigned int, std::vector<WallpaperAtlas*> > AtlasMap;
   AtlasMap _atlasMap;
};

inline void me_TextureBatchingStrategyAtlas::AtlasSource::setID(unsigned int id)
{
   _id = id;
}

inline unsigned int me_TextureBatchingStrategyAtlas::AtlasSource::getID() const
{
   return _id;
}

inline void me_TextureBatchingStrategyAtlas::AtlasSource::setTextureCoords(osg::Vec2Array* texCoords)
{
   _textureCoords = texCoords;
}

inline void me_TextureBatchingStrategyAtlas::AtlasSource::setFrameNumber(unsigned int frameNumber)
{
   _frameNumber = frameNumber;

}

inline osg::ref_ptr<osg::Vec2Array> me_TextureBatchingStrategyAtlas::AtlasSource::getTextureCoords() const
{
   return _textureCoords;
}

inline unsigned int me_TextureBatchingStrategyAtlas::AtlasSource::getFrameNumber() const
{
   return _frameNumber;
}

inline void me_TextureBatchingStrategyAtlas::WallpaperAtlas::setID(unsigned int id)
{
   _id = id;
}

inline unsigned int me_TextureBatchingStrategyAtlas::WallpaperAtlas::getID() const
{
   return _id;
}

inline void me_TextureBatchingStrategyAtlas::WallpaperAtlas::setAtlasFull(bool atlasFull)
{
   _atlasFull = atlasFull;
}

inline bool me_TextureBatchingStrategyAtlas::WallpaperAtlas::isAtlasFull() const
{
   return _atlasFull;
}

}
}

#endif //TEXTURE_BATCHING_STRATEGY_ATLAS_HEADER
