/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/include/osgUtil/Optimizer
List of changes:
1. Support to replace a single texture in an atlas.
*/

#ifndef MAPENGINE_OSS_OPTIMIZER_
#define MAPENGINE_OSS_OPTIMIZER_

#include <set>
#include <osg/Matrix>
#include <osg/Texture2D>
#include "oss/Export.h"

namespace mapengine {
namespace oss {

class MAPENGINE_OSS_EXPORT Optimizer
{
public:
   Optimizer() {}
   virtual ~Optimizer() {}      

public:
   class MAPENGINE_OSS_EXPORT TextureAtlasBuilder
   {
   public:
      TextureAtlasBuilder();
      virtual ~TextureAtlasBuilder();

      void reset();

      void setMaximumAtlasSize(int width, int height);

      int getMaximumAtlasWidth() const { return _maximumAtlasWidth; }
      int getMaximumAtlasHeight() const { return _maximumAtlasHeight; }

      void setMargin(int margin);
      int getMargin() const { return _margin; }

      virtual void addSource(const osg::Image* image);
      virtual void addSource(const osg::Texture2D* texture);

      unsigned int getNumSources() const { return static_cast<unsigned int>(_sourceList.size()); }
      const osg::Image* getSourceImage(unsigned int i) { return _sourceList[i]->_image.get(); }
      const osg::Texture2D* getSourceTexture(unsigned int i) { return _sourceList[i]->_texture.get(); }

      void buildAtlas();
      osg::Image* getImageAtlas(unsigned int i);
      osg::Texture2D* getTextureAtlas(unsigned int i);
      osg::Matrix getTextureMatrix(unsigned int i);

      osg::Image* getImageAtlas(const osg::Image* image);
      osg::Texture2D* getTextureAtlas(const osg::Image* image);
      osg::Matrix getTextureMatrix(const osg::Image* image);

      osg::Image* getImageAtlas(const osg::Texture2D* textue);
      osg::Texture2D* getTextureAtlas(const osg::Texture2D* texture);
      osg::Matrix getTextureMatrix(const osg::Texture2D* texture);

   protected:

      int _maximumAtlasWidth;
      int _maximumAtlasHeight;
      int _margin;

      // forward declare
      class Atlas;

      class MAPENGINE_OSS_EXPORT Source : public osg::Referenced
      {
      public:
         Source():
            _x(0),_y(0),_atlas(0) {}

            Source(const osg::Image* image):
            _x(0),_y(0),_atlas(0),_image(image) {}

            Source(const osg::Texture2D* texture):
            _x(0),_y(0),_atlas(0),_texture(texture) { if (texture) _image = texture->getImage(); }

            int _x;
            int _y;
            Atlas* _atlas;

            osg::ref_ptr<const osg::Image> _image;
            osg::ref_ptr<const osg::Texture2D> _texture;

            bool suitableForAtlas(int maximumAtlasWidth, int maximumAtlasHeight, int margin) const;
            osg::Matrix computeTextureMatrix() const;

      protected:
         virtual ~Source() 
         {
            _atlas = NULL; 
         }
      };

      typedef std::vector< osg::ref_ptr<Source> > SourceList;

      class MAPENGINE_OSS_EXPORT Atlas : public osg::Referenced
      {
      public:
         Atlas(int width, int height, int margin):
            _maximumAtlasWidth(width),
               _maximumAtlasHeight(height),
               _margin(margin),
               _x(0),
               _y(0),
               _width(0),
               _height(0),
               _indexFirstOfRow(0){}

            int _maximumAtlasWidth;
            int _maximumAtlasHeight;
            int _margin;

            osg::ref_ptr<osg::Texture2D> _texture;
            osg::ref_ptr<osg::Image> _image;

            SourceList _sourceList;

            int _x;
            int _y;
            int _width;
            int _height;
            unsigned int _indexFirstOfRow; ///< Contain the index of the first element of the last row.
            enum FitsIn
            {
               DOES_NOT_FIT_IN_ANY_ROW,       
               FITS_IN_CURRENT_ROW,
               IN_NEXT_ROW
            };
            FitsIn doesSourceFit(Source* source) const;
            bool addSource(Source* source);
            void clampToNearestPowerOfTwoSize();
            void copySources();
            virtual void copyOnlyThisSource(Source* source, bool dirtyImmediately);

      protected:
         virtual ~Atlas() {}
      };

      typedef std::vector< osg::ref_ptr<Atlas> > AtlasList;

      Source* getSource(const osg::Image* image);
      Source* getSource(const osg::Texture2D* texture);

      SourceList _sourceList;
      AtlasList _atlasList;
      void completeRow(unsigned int indexAtlas);
   private:
      struct CompareSrc
      {
         bool operator()(Source* src1, Source* src2) const
         {
            return src1->_image->t() > src2->_image->t();
         }
      };
   };
};

}
}

#endif
