/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
- Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgViewer/StatsHandler.cpp
- Implementation of some APIs to enable display of dynamically changing strings as stats in addition to integers and floating point values for debugging purposes.
- Implementation of some additional APIs to group stats based on whether they change or remain consistent across multiple views.
- Implementation of some additional APIs to show frame number and frame reference time as stats
*/

#include "oss/stats/CustomStatsHandlerBase.h"

using namespace osgGA;
using namespace osgViewer;
using namespace mapengine::oss;

me_CustomStatsHandlerBase::me_CustomStatsHandlerBase()
: _enumToStringMaps(USER_STATS_COUNT)
, _updateStatsFlag(false)
, _customStatsWidth(1100.0f)
, _customStatsHeight(625.0f)
, _customCharacterSize(13.0f)
, _backgroundColor(0.0f, 0.0f, 0.0f, 0.3f)
, _backgroundMargin(5.0f)
, _backgroundSpacing(3.0f)
, _staticTextColor(1.0f, 1.0f, 0.0f, 1.0f)
, _dynamicTextColor(1.0f, 1.0f, 1.0f, 1.0f)
, _frameStatsVisible(false)
, _userStatsVisibility(VIEW_INDEPENDENT_STATS_VISIBLE)
, _userPrintStatsToTrace(VIEW_INDEP_STATS_TRACE)
{
   _customCamera = new osg::Camera;

   if(_customCamera.valid())
   {
      _customCamera->setRenderer(new Renderer(_customCamera.get()));
      _customCamera->setProjectionResizePolicy(osg::Camera::FIXED);
   }
}

bool me_CustomStatsHandlerBase::handle(const osgGA::GUIEventAdapter& ea, osgGA::GUIActionAdapter& aa)
{
   osgViewer::View* myview = dynamic_cast<osgViewer::View*>(&aa);
   if (!myview) return false;
   osgViewer::ViewerBase* viewer = myview->getViewerBase();

   if (ea.getHandled()) return false;

   switch(ea.getEventType())
   {
   case(osgGA::GUIEventAdapter::KEYUP):
      {
         if ((ea.getKey() == KEY_CUSTOM_STATS) && (NULL != viewer))
         {
            if (viewer->getViewerStats())
            {
               if (!_initialized)
               {
                  customSetUpHUDCamera(myview);
                  setUpStatsGrouping(myview);
                  customSetUpScene(myview, 0, false);
               }
               else
               {
                  customSetUpScene(myview, 0, true);
               }

               std::vector<osgViewer::View*> views;
               viewer->getViews(views);
               std::vector<osgViewer::View*>::iterator it;
               handleStatsGrouping();

               // displays stats on pressing 'u'
               switch (_userStatsVisibility)
               {

               case(NO_USER_STATS) :
               {
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", false);
                  }

                  bool flag = false;

                  for (it = views.begin(); it != views.end(); ++it)
                  {
                     if ((*it)->getStats()->collectStats("user_viewDep_stats")) flag = true;
                  }

                  if (!flag)
                  {
                     if (!viewer->getViewerStats()->collectStats("user_frameNumTime_stats"))
                     {
                        _customCamera->setNodeMask(0x0);
                        _customSwitch->setAllChildrenOff();
                        _updateStatsFlag = false;
                     }
                  }

                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, false);
                  _userStatsVisibility = VIEW_INDEPENDENT_STATS_VISIBLE;
                  break;
               }

               case(VIEW_INDEPENDENT_STATS_VISIBLE) :
               {
                  _updateStatsFlag = true;
                  viewer->getViewerStats()->collectStats("user_viewInd_stats", true);
                  _customCamera->setNodeMask(0xffffffff);
                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, true);
                  _userStatsVisibility = ALL_MAP_STATS_VISIBLE;
                  break;
               }

               case(ALL_MAP_STATS_VISIBLE) :
               {
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", true);
                  }
                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, true);
                  _userStatsVisibility = VIEW_DEPENDENT_STATS_VISIBLE;
                  break;
               }

               case(VIEW_DEPENDENT_STATS_VISIBLE) :
               {
                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, false);
                  _userStatsVisibility = NO_USER_STATS;
                  break;
               }

               default:
                  _userStatsVisibility = VIEW_INDEPENDENT_STATS_VISIBLE;
                  break;
               }
               _userPrintStatsToTrace = VIEW_INDEP_STATS_TRACE;
               aa.requestRedraw();
            }
         }
         else if ((ea.getKey() == KEY_FRAME_STATS) && (NULL != viewer))
         {
            if (viewer->getViewerStats())
            {
               if (!_initialized)
               {
                  customSetUpHUDCamera(myview);
                  customSetUpScene(myview, 0, false);
               }
               else
               {
                  customSetUpScene(myview, 0, true);
               }

               if (_frameStatsVisible)
               {
                  _customSwitch->setValue(USER_FRAME_NUM_TIME_STATS, false);
                  _frameStatsVisible = false;

                  std::vector<osgViewer::View*> views;
                  viewer->getViews(views);
                  std::vector<osgViewer::View*>::iterator it = views.begin();
                  bool flag = false;
                  for (; it != views.end(); ++it)
                  {
                     if ((*it)->getStats()->collectStats("user_viewDep_stats")) flag = true;
                  }
                  if (!flag && !viewer->getViewerStats()->collectStats("user_viewInd_stats"))
                  {
                     _customCamera->setNodeMask(0x0);
                     _customSwitch->setAllChildrenOff();
                     _updateStatsFlag = false;
                  }
               }
               else
               {
                  _frameStatsVisible = true;
                  _updateStatsFlag = true;
                  viewer->getViewerStats()->collectStats("user_frameNumTime_stats", true);
                  _customCamera->setNodeMask(0xffffffff);
                  _customSwitch->setValue(USER_FRAME_NUM_TIME_STATS, true);
               }
               aa.requestRedraw();
            }
         }
         //displays and prints view independent and then view dependent stats on pressing 'U'.
         else if ((ea.getKey() == KEY_CUSTOM_STATS_TRACE) && (NULL != viewer))
         {
            if (viewer->getViewerStats())
            {
               if (!_initialized)
               {
                  customSetUpHUDCamera(myview);
                  setUpStatsGrouping(myview);
                  customSetUpScene(myview, KEY_CUSTOM_STATS_TRACE, false);
               }
               else
               {
                  customSetUpScene(myview, KEY_CUSTOM_STATS_TRACE, true);
               }

               std::vector<osgViewer::View*> views;
               viewer->getViews(views);
               std::vector<osgViewer::View*>::iterator it;
               handleStatsGrouping();

               switch (_userPrintStatsToTrace)
               {
               case(VIEW_NO_STATS_TRACE) :
               {
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", false);
                  }

                  bool flag = false;

                  for (it = views.begin(); it != views.end(); ++it)
                  {
                     if ((*it)->getStats()->collectStats("user_viewDep_stats")) flag = true;
                  }

                  if (!flag)
                  {
                     viewer->getViewerStats()->collectStats("user_viewInd_stats", true);
                     _updateStatsFlag = false;
                  }

                  _customCamera->setNodeMask(0x0);
                  _customSwitch->setAllChildrenOff();
                  _userStatsVisibility = VIEW_INDEPENDENT_STATS_VISIBLE;
                  _userPrintStatsToTrace = VIEW_INDEP_STATS_TRACE;
                  break;
               }

               case(VIEW_INDEP_STATS_TRACE) :
               {
                  _updateStatsFlag = true;
                  viewer->getViewerStats()->collectStats("user_viewInd_stats", true);

                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, true);
                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, false);
                  _customCamera->setNodeMask(0xffffffff);
                  _userPrintStatsToTrace = VIEW_DEP_STATS_TRACE;
                  _userStatsVisibility = ALL_MAP_STATS_VISIBLE;
                  break;
               }

               case(VIEW_DEP_STATS_TRACE) :
               {
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", true);
                  }
                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, false);
                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, true);
                  _customCamera->setNodeMask(0xffffffff);
                  _userPrintStatsToTrace = VIEW_NO_STATS_TRACE;
                  _userStatsVisibility = ALL_MAP_STATS_VISIBLE;
                  break;
               }

               default:
               {
                  _userPrintStatsToTrace = VIEW_INDEP_STATS_TRACE;
                  _userStatsVisibility = VIEW_INDEPENDENT_STATS_VISIBLE;
                  break;
               }
               }
               aa.requestRedraw();
            }
         }
      return true;
      }
      break;

   case(osgGA::GUIEventAdapter::RESIZE) :
      customSetWindowSize(ea.getWindowWidth(), ea.getWindowHeight());
      break;

   default: break;
   }
   return false;
}

void me_CustomStatsHandlerBase::customSetWindowSize(int width, int height) const
{
   if ((width > 0) && (height > 0))
   {
      _customCamera->setViewport(0, 0, width, height);
      if (fabsf(static_cast<float>(height) * _customStatsWidth) <= fabsf(static_cast<float>(width) * _customStatsHeight))
      {
         _customCamera->setProjectionMatrix(osg::Matrix::ortho2D(0.0, (double)(static_cast<float>(width) * _customStatsHeight/static_cast<float>(height)), 0.0, (double)(_customStatsHeight)));
      }
      else
      {
         _customCamera->setProjectionMatrix(osg::Matrix::ortho2D(0.0, (double)(_customStatsWidth), (double)(_customStatsHeight - static_cast<float>(height) * _customStatsWidth/static_cast<float>(width)), (double)(_customStatsHeight)));
      }
   }
}

void me_CustomStatsHandlerBase::customSetUpHUDCamera(osgViewer::View* viewer)
{
   viewer->getCamera()->asGroup()->addChild(_customCamera);
   _customCamera->setGraphicsContext(viewer->getCamera()->getGraphicsContext());
   _customCamera->setRenderOrder(osg::Camera::POST_RENDER, 10);
   _customCamera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
   _customCamera->setViewMatrix(osg::Matrix::identity());

   customSetWindowSize(viewer->getCamera()->getGraphicsContext()->getTraits()->width, viewer->getCamera()->getGraphicsContext()->getTraits()->height);

   _customCamera->setClearMask(0);
   _customCamera->setAllowEventFocus(false);
   _customCamera->setRenderer(new Renderer(_customCamera.get()));

   _initialized = true;
}
