// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_GALTHREAD_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_GALTHREAD_H

#ifdef WIN32
#include <windows.h>
#else
#include <pthread.h>
#include <sched.h>
#endif

/**
 * A class that wraps a thread. Extend this class to create threads. The default
 * implementation uses pthreads, you must customize this to work on your platform.
 */
class GalThread {
public:
    GalThread() : mRunning(false) {
#ifndef WIN32
        mThread = static_cast<pthread_t>(0);
#endif
    }
    virtual ~GalThread();
    /**
     * Call this method to start running the thread.
     */
    bool start();
    /**
     * Call this method to wait for the thread represented by this object to terminate.
     * You are responsible for actually making the thread shutdown somehow.
     */
    bool join();
    /**
     * Return an identifier for this thread.
     */
    unsigned long id();
    /**
     * Yield the cpu to any other runnable threads.
     */
    void yield();
    /**
     * Set the thread priority.
     * @param priority The priority value to set. You should check the range of priority values
     *        that your system supports.
     * @return True on success, false otherwise.
     */
    bool setPriority(int priority);
    /**
     * Set the name of the thread.
     * @param name The name of this thread.
     * @return True on success, false otherwise.
     */
    bool setName(const char* name);
    /**
     * Changed by ADIT.
     * Set the thread scheduling policy and the thread priority.
     * @param policy   The scheduling policy value to set.
     * @param priority The priority value to set. You should check the range of priority values
     *        that your system supports.
     * @return True on success, false otherwise.
     *
     * Note: The input value for policy should be 0 to 2
     *       0 means SCHED_OTHER. SCHED_OTHER can only be used at static priority 0
     *       1 means SCHED_FIFO. SCHED_FIFO have a priority value in the range 1 (low) to 99 (high)
     *       2 means SCHED_RR. SCHED_RR have a priority value in the range 1 (low) to 99 (high)
     */
    bool setSchedParam(int policy, int priority);
protected:
    /**
     * @internal
     * The actual method sent to pthreads to run. This calls the overloaded run
     * method internally. The this pointer is sent across while calling pthread_create
     * so we can differentiate objects.
     * @param arg The address of the thread object that needs to be run.
     */
#ifdef WIN32
    // Windows requires stdcall calling convention.
    static DWORD WINAPI callback(void* arg);
#else
    static void* callback(void* arg);
#endif

    /**
     * Override this method with your actual thread function. The idea is that since
     * you have to extend this class, you can add any necessary methods to set up the
     * state that your threads need.
     */
    virtual void run() = 0;
private:
#ifdef WIN32
    HANDLE mThread;
#else
    pthread_t mThread;
#endif
    bool mRunning;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_GALTHREAD_H
